/**
 * Copyright (C) 2009-2012 enStratus Networks Inc
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */

package org.dasein.cloud.aws.network;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;

import org.apache.log4j.Logger;
import org.dasein.cloud.CloudException;
import org.dasein.cloud.InternalException;
import org.dasein.cloud.ProviderContext;
import org.dasein.cloud.aws.AWSCloud;
import org.dasein.cloud.aws.compute.EC2Exception;
import org.dasein.cloud.aws.compute.EC2Method;
import org.dasein.cloud.compute.ComputeServices;
import org.dasein.cloud.compute.VirtualMachineSupport;
import org.dasein.cloud.identity.ServiceAction;
import org.dasein.cloud.network.Direction;
import org.dasein.cloud.network.Firewall;
import org.dasein.cloud.network.FirewallRule;
import org.dasein.cloud.network.FirewallSupport;
import org.dasein.cloud.network.Protocol;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class SecurityGroup implements FirewallSupport {
	static private final Logger logger = AWSCloud.getLogger(SecurityGroup.class);
	
	private AWSCloud provider = null;
	
	SecurityGroup(AWSCloud provider) {
		this.provider = provider;
	}
	
	@Override
	public @Nonnull String authorize(@Nonnull String securityGroupId, @Nonnull String cidr, @Nonnull Protocol protocol, int startPort, int endPort) throws CloudException, InternalException {
        return authorize(securityGroupId, Direction.INGRESS, cidr, protocol, startPort, endPort);
    }

    @Override
    public @Nonnull String authorize(@Nonnull String firewallId, @Nonnull Direction direction, @Nonnull String cidr, @Nonnull Protocol protocol, int beginPort, int endPort) throws CloudException, InternalException {
        String action = (direction.equals(Direction.INGRESS) ? EC2Method.AUTHORIZE_SECURITY_GROUP_INGRESS : EC2Method.AUTHORIZE_SECURITY_GROUP_EGRESS);
        Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), action);
        EC2Method method;
        NodeList blocks;
        Document doc;

        if( cidr.indexOf('/') == -1 ) {
            cidr = cidr + "/32";
        }
        if( provider.getEC2Provider().isEucalyptus() ) {
            parameters.put("GroupName", firewallId);
            parameters.put("IpProtocol", protocol.name().toLowerCase());
            parameters.put("FromPort", String.valueOf(beginPort));
            parameters.put("ToPort", endPort == -1 ? String.valueOf(beginPort) : String.valueOf(endPort));
            parameters.put("CidrIp", cidr);
        }
        else {
            parameters.put("GroupId", firewallId);
            parameters.put("IpPermissions.1.IpProtocol", protocol.name().toLowerCase());
            parameters.put("IpPermissions.1.FromPort", String.valueOf(beginPort));
            parameters.put("IpPermissions.1.ToPort", endPort == -1 ? String.valueOf(beginPort) : String.valueOf(endPort));
            parameters.put("IpPermissions.1.IpRanges.1.CidrIp", cidr);
        }
        method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
            doc = method.invoke();
        }
        catch( EC2Exception e ) {
            String code = e.getCode();

            if( code != null && code.equals("InvalidPermission.Duplicate") ) {
                return FirewallRule.getRuleId(firewallId, cidr, direction, protocol, beginPort, endPort);
            }
            logger.error(e.getSummary());
            throw new CloudException(e);
        }
        blocks = doc.getElementsByTagName("return");
        if( blocks.getLength() > 0 ) {
            if( !blocks.item(0).getFirstChild().getNodeValue().equalsIgnoreCase("true") ) {
                throw new CloudException("Failed to authorize security group rule without explanation.");
            }
        }
        return FirewallRule.getRuleId(firewallId, cidr, direction, protocol, beginPort, endPort);
    }

    @Override
	public @Nonnull String create(@Nonnull String name, @Nonnull String description) throws InternalException, CloudException {
		Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), EC2Method.CREATE_SECURITY_GROUP);
		EC2Method method;
        NodeList blocks;
		Document doc;

		name = getUniqueName(name);
		parameters.put("GroupName", name);
		parameters.put("GroupDescription", description);
		method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
        	doc = method.invoke();
        }
        catch( EC2Exception e ) {
        	logger.error(e.getSummary());
        	throw new CloudException(e);
        }
        if( provider.getEC2Provider().isEucalyptus() ) {
            return name;
        }
        else {
            blocks = doc.getElementsByTagName("groupId");
            if( blocks.getLength() > 0 ) {
                return blocks.item(0).getFirstChild().getNodeValue().trim();
            }
            throw new CloudException("Failed to create security group without explanation.");
        }
	}
 	
   @Override
    public @Nonnull String createInVLAN(@Nonnull String name, @Nonnull String description, @Nonnull String providerVlanId) throws InternalException, CloudException {
        Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), EC2Method.CREATE_SECURITY_GROUP);
        EC2Method method;
        NodeList blocks;
        Document doc;

        name = getUniqueName(name);
        parameters.put("GroupName", name);
        parameters.put("GroupDescription", description);
        parameters.put("VpcId", providerVlanId);
        method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
            doc = method.invoke();
        }
        catch( EC2Exception e ) {
            logger.error(e.getSummary());
            throw new CloudException(e);
        }
        blocks = doc.getElementsByTagName("groupId");
        if( blocks.getLength() > 0 ) {
            return blocks.item(0).getFirstChild().getNodeValue().trim();
        }
        throw new CloudException("Failed to create security group without explanation.");
    }
	   
	@Override
	public void delete(@Nonnull String securityGroupId) throws InternalException, CloudException {
		Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), EC2Method.DELETE_SECURITY_GROUP);
		EC2Method method;
        NodeList blocks;
		Document doc;

        if( provider.getEC2Provider().isEucalyptus() ) {
            parameters.put("GroupName", securityGroupId);
        }
        else {
            parameters.put("GroupId", securityGroupId);
        }
		method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
        	doc = method.invoke();
        }
        catch( EC2Exception e ) {
        	logger.error(e.getSummary());
        	throw new CloudException(e);
        }
        blocks = doc.getElementsByTagName("return");
        if( blocks.getLength() > 0 ) {
        	if( !blocks.item(0).getFirstChild().getNodeValue().equalsIgnoreCase("true") ) {
        		throw new CloudException("Failed to delete security group without explanation.");
        	}
        }
	}

	@Override
	public @Nullable Firewall getFirewall(@Nonnull String securityGroupId) throws InternalException, CloudException {
        ProviderContext ctx = provider.getContext();
        
        if( ctx == null ) {
            throw new CloudException("No context has been established for this request");
        }
		Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), EC2Method.DESCRIBE_SECURITY_GROUPS);
		EC2Method method;
        NodeList blocks;
		Document doc;

        if( provider.getEC2Provider().isEucalyptus() ) {
            parameters.put("GroupName.1", securityGroupId);
        }
        else {
            parameters.put("GroupId.1", securityGroupId);
        }
		method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
        	doc = method.invoke();
        }
        catch( EC2Exception e ) {
        	String code = e.getCode();
        	
        	if( code != null && code.startsWith("InvalidGroup") ) {
        		return null;
        	}
        	logger.error(e.getSummary());
        	throw new CloudException(e);
        }
        blocks = doc.getElementsByTagName("securityGroupInfo");
        for( int i=0; i<blocks.getLength(); i++ ) {
        	NodeList items = blocks.item(i).getChildNodes();
        	
            for( int j=0; j<items.getLength(); j++ ) {
            	Node item = items.item(j);
            	
            	if( item.getNodeName().equals("item") ) {
            		Firewall firewall = toFirewall(ctx, item);
            		
                    if( firewall != null && securityGroupId.equals(firewall.getProviderFirewallId()) ) {
            			return firewall;
            		}
            	}
            }
        }
        return null;
	}

	@Override
	public @Nonnull String getProviderTermForFirewall(@Nonnull Locale locale) {
		return "security group";
	}

	@Override
	public @Nonnull Collection<FirewallRule> getRules(@Nonnull String securityGroupId) throws InternalException, CloudException {
		Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), EC2Method.DESCRIBE_SECURITY_GROUPS);
		ArrayList<FirewallRule> list = new ArrayList<FirewallRule>();
		EC2Method method;
        NodeList blocks;
		Document doc;

        if( provider.getEC2Provider().isEucalyptus() ) {
            parameters.put("GroupName.1", securityGroupId);
        }
        else {
            parameters.put("GroupId.1", securityGroupId);
        }
		method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
        	doc = method.invoke();
        }
        catch( EC2Exception e ) {
        	String code = e.getCode();
        	
        	if( code != null && code.startsWith("InvalidGroup") ) {
                return Collections.emptyList();
        	}
        	logger.error(e.getSummary());
        	throw new CloudException(e);
        }
        blocks = doc.getElementsByTagName("securityGroupInfo");
        for( int i=0; i<blocks.getLength(); i++ ) {
        	NodeList items = blocks.item(i).getChildNodes();
        	
            for( int j=0; j<items.getLength(); j++ ) {
            	Node item = items.item(j);
            	
            	if( item.getNodeName().equals("item") ) {
            		NodeList attrs = item.getChildNodes();
            		
            		for( int k=0; k<attrs.getLength(); k++ ) {
            			Node attr = attrs.item(k);
            			
            			if( attr.getNodeName().equals("ipPermissions") ) {
            				NodeList subList = attr.getChildNodes();
            				
            				for( int l=0; l<subList.getLength(); l++ ) {
            					Node sub = subList.item(l);
            					
            					if( sub.getNodeName().equals("item") ) {
            						list.addAll(toFirewallRules(securityGroupId, sub));
            					}
            				}
            			}
            		}
            	}
            }
        }
        return list;
	}

	private @Nonnull String getUniqueName(@Nonnull String name) throws InternalException, CloudException {
        StringBuilder str = new StringBuilder();
        
        for( int i=0; i<name.length(); i++ ) {
            char c = name.charAt(i);

            if( c == '-' || c == '_'  ) {
                str.append(c);
            }
            else if( i == 0 && Character.isDigit(c) ) {
                str.append("e-");
                str.append(c);
            }
            else if( i == 0 && Character.isLetter(c) ) {
                str.append(c);
            }
            else if( i > 0 && Character.isLetterOrDigit(c) ) {
                str.append(c);
            }
        }
        if( str.length() < 1 ) {
            return "new-group";
        }
        String baseName = str.toString();
        String withName = baseName;
        int count = 1;
        boolean found;
        char c = 'a';
        
        do {
            found = false;
            for( Firewall fw : list() ) {
                String id = fw.getProviderFirewallId();
                
                if( id == null ) {
                    continue;
                }
                if( id.equals(withName) ) {
                    found = true;
                    if( count == 1 ) {
                        withName = baseName + "-" + String.valueOf(c);
                    }
                    else {
                        withName = baseName + String.valueOf(c);
                    }
                    if( c == 'z' ) {
                        if( count == 1 ) {
                            baseName = baseName + "-a";
                        }
                        else {
                            baseName = baseName + "a";
                        }
                        c = 'a';
                        count++;
                        if( count > 10 ) {
                            throw new CloudException("Could not generate a unique firewall name from " + baseName);
                        }
                    }
                    else {
                        c++;
                    }
                    break;
                }                        
            }                        
        } while( found ); 		
        return withName;
	}

    @Override
    public boolean isSubscribed() throws CloudException, InternalException {
        ComputeServices svc = provider.getComputeServices();
        
        if( svc == null ) {
            return false;
        }
        VirtualMachineSupport support = svc.getVirtualMachineSupport();
        
        return (support != null && support.isSubscribed());
    }

	@Override
	public @Nonnull Collection<Firewall> list() throws InternalException, CloudException {
        ProviderContext ctx = provider.getContext();

        if( ctx == null ) {
            throw new CloudException("No context has been established for this request");
        }
		Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), EC2Method.DESCRIBE_SECURITY_GROUPS);
		ArrayList<Firewall> list = new ArrayList<Firewall>();
		EC2Method method;
        NodeList blocks;
		Document doc;

		method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
        	doc = method.invoke();
        }
        catch( EC2Exception e ) {
        	logger.error(e.getSummary());
        	throw new CloudException(e);
        }
        blocks = doc.getElementsByTagName("securityGroupInfo");
        for( int i=0; i<blocks.getLength(); i++ ) {
        	NodeList items = blocks.item(i).getChildNodes();
        	
            for( int j=0; j<items.getLength(); j++ ) {
            	Node item = items.item(j);
            	
            	if( item.getNodeName().equals("item") ) {
            		Firewall firewall = toFirewall(ctx, item);
            		
            		if( firewall != null ) {
            			list.add(firewall);
            		}
            	}
            }
        }
        return list;
	}

    @Override
    public @Nonnull String[] mapServiceAction(@Nonnull ServiceAction action) {
        if( action.equals(FirewallSupport.ANY) ) {
            return new String[] { EC2Method.EC2_PREFIX + "*" };
        }
        else if( action.equals(FirewallSupport.AUTHORIZE) ) {
            return new String[] { EC2Method.EC2_PREFIX + EC2Method.AUTHORIZE_SECURITY_GROUP_INGRESS, EC2Method.EC2_PREFIX + EC2Method.AUTHORIZE_SECURITY_GROUP_EGRESS };
        }
        else if( action.equals(FirewallSupport.CREATE_FIREWALL) ) {
            return new String[] { EC2Method.EC2_PREFIX + EC2Method.CREATE_SECURITY_GROUP };
        }
        else if( action.equals(FirewallSupport.GET_FIREWALL) || action.equals(FirewallSupport.LIST_FIREWALL) ) {
            return new String[] { EC2Method.EC2_PREFIX + EC2Method.DESCRIBE_SECURITY_GROUPS };
        }
        else if( action.equals(FirewallSupport.REMOVE_FIREWALL) ) {
            return new String[] { EC2Method.EC2_PREFIX + EC2Method.DELETE_SECURITY_GROUP };
        }
        else if( action.equals(FirewallSupport.REVOKE) ) {
            return new String[] { EC2Method.EC2_PREFIX + EC2Method.REVOKE_SECURITY_GROUP_INGRESS, EC2Method.EC2_PREFIX + EC2Method.REVOKE_SECURITY_GROUP_EGRESS };
        }
        return new String[0];
    }

	@Override
	public void revoke(@Nonnull String securityGroupId, @Nonnull String cidr, @Nonnull Protocol protocol, int startPort, int endPort) throws CloudException, InternalException {
        revoke(securityGroupId, Direction.INGRESS, cidr, protocol, startPort, endPort);
    }

    @Override
    public void revoke(@Nonnull String firewallId, @Nonnull Direction direction, @Nonnull String cidr, @Nonnull Protocol protocol, int beginPort, int endPort) throws CloudException, InternalException {
        String action = (direction.equals(Direction.INGRESS) ? EC2Method.REVOKE_SECURITY_GROUP_INGRESS : EC2Method.REVOKE_SECURITY_GROUP_EGRESS);
        Map<String,String> parameters = provider.getStandardParameters(provider.getContext(), action);
        EC2Method method;
        Document doc;

        if( provider.getEC2Provider().isEucalyptus() ) {
            parameters.put("GroupName", firewallId);
        }
        else {
            parameters.put("GroupId", firewallId);
        }
        parameters.put("IpProtocol", protocol.name().toLowerCase());
        parameters.put("FromPort", String.valueOf(beginPort));
        parameters.put("ToPort", endPort == -1 ? String.valueOf(beginPort) : String.valueOf(endPort));
        parameters.put("CidrIp", cidr);
        method = new EC2Method(provider, provider.getEc2Url(), parameters);
        try {
            doc = method.invoke();
        }
        catch( EC2Exception e ) {
            logger.error(e.getSummary());
            throw new CloudException(e);
        }
        method.checkSuccess(doc.getElementsByTagName("return"));
    }

    @Override
    public boolean supportsRules(@Nonnull Direction direction, boolean inVlan) throws CloudException, InternalException {
        return !(inVlan && provider.getEC2Provider().isEucalyptus()) && (inVlan || direction.equals(Direction.INGRESS));
    }

    private @Nullable Firewall toFirewall(@Nonnull ProviderContext ctx, @Nullable Node node) {
        if( node == null ) {
            return null;
        }
        String fwName = null, fwId = null, fwDesc = null;
		NodeList attrs = node.getChildNodes();
		Firewall firewall = new Firewall();
        String regionId = ctx.getRegionId();
		String vpcId = null;

        if( regionId == null ) {
            return null;
        }
        firewall.setRegionId(regionId);
        firewall.setAvailable(true);
        firewall.setActive(true);
		for( int i=0; i<attrs.getLength(); i++ ) {
			Node attr = attrs.item(i);
			String name;
			
			name = attr.getNodeName();
			if( name.equals("groupName") ) {
                fwName = attr.getFirstChild().getNodeValue().trim();
			}
			else if( name.equals("groupDescription") ) {
				fwDesc = attr.getFirstChild().getNodeValue().trim();
			}
			else if( name.equals("groupId") ) {
                fwId = attr.getFirstChild().getNodeValue().trim();			    
			}
			else if( name.equals("vpcId") ) {
			    if( attr.hasChildNodes() ) {
			        vpcId = attr.getFirstChild().getNodeValue();
			        if( vpcId != null ) {
			            vpcId = vpcId.trim();
			        }
			    }
			}
		}
        if( fwId == null ) {
            if( fwName == null ) {
                return null;
            }
            fwId = fwName;
        }
		if( fwName == null ) {
		    fwName = fwId;
		}
        firewall.setProviderFirewallId(fwId);
        firewall.setName(fwName);
        if( fwDesc == null ) {
            fwDesc = fwName;
        }
        firewall.setDescription(fwDesc);
		if( vpcId != null ) {
		    firewall.setName(firewall.getName() + " (VPC " + vpcId + ")");
		    firewall.setProviderVlanId(vpcId);
		}
		return firewall;
	}
	
	private @Nonnull Collection<FirewallRule> toFirewallRules(@Nonnull String securityGroupId, @Nullable Node node) {
	    ArrayList<FirewallRule> rules = new ArrayList<FirewallRule>();
        
        if( node == null ) {
            return rules;
        }
	    ArrayList<String> cidrs = new ArrayList<String>();
		NodeList attrs = node.getChildNodes();
        int startPort = -2, endPort = -2;
		Protocol protocol = Protocol.TCP;
		
		for( int i=0; i<attrs.getLength(); i++ ) {
			Node attr = attrs.item(i);
			String name;
			
			name = attr.getNodeName();
			if( name.equals("ipProtocol") ) {
			    String val = attr.getFirstChild().getNodeValue().trim();
			    
			    if( !val.equals("") && !val.equals("-1") ) {
			        protocol = Protocol.valueOf(attr.getFirstChild().getNodeValue().trim().toUpperCase());
			    }
			    else {
			        protocol = Protocol.ICMP;
			    }
			}
			else if( name.equals("fromPort") ) {
				startPort = Integer.parseInt(attr.getFirstChild().getNodeValue().trim());
			}
			else if( name.equals("toPort") ) {
				endPort = Integer.parseInt(attr.getFirstChild().getNodeValue().trim());
			}
			else if( name.equals("ipRanges") ) {
				if( attr.hasChildNodes() ) {
					NodeList children = attr.getChildNodes();
				
					for( int j=0; j<children.getLength(); j++ ) {
						Node child = children.item(j);
					
						if( child.getNodeName().equals("item") ) {
						    if( child.hasChildNodes() ) { 
						        NodeList targets = child.getChildNodes();
						        
						        for( int k=0; k<targets.getLength(); k++ ) {
						            Node cidr = targets.item(k);

						            if( cidr.getNodeName().equals("cidrIp") ) {
						                cidrs.add(cidr.getFirstChild().getNodeValue());
						            }
						        }
						    }
						}
					}
				}
			}
		}
		for( String cidr : cidrs ) {
		    FirewallRule rule = new FirewallRule();

		    rule.setFirewallId(securityGroupId);
		    rule.setCidr(cidr);
		    rule.setEndPort(endPort);
		    rule.setStartPort(startPort);
		    rule.setProtocol(protocol);
		    rules.add(rule);
		}
		return rules;		
	}
}
