/**
 * Copyright (C) 2010-2011 enStratusNetworks LLC
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */

package org.dasein.cloud.vsphere;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import org.dasein.cloud.CloudException;
import org.dasein.cloud.InternalException;
import org.dasein.cloud.dc.DataCenter;
import org.dasein.cloud.dc.DataCenterServices;
import org.dasein.cloud.dc.Region;

import com.vmware.vim25.InvalidProperty;
import com.vmware.vim25.RuntimeFault;
import com.vmware.vim25.mo.Datacenter;
import com.vmware.vim25.mo.Folder;
import com.vmware.vim25.mo.InventoryNavigator;
import com.vmware.vim25.mo.ManagedEntity;
import com.vmware.vim25.mo.ServiceInstance;

public class Dc implements DataCenterServices {

    private PrivateCloud provider;
    
    Dc(PrivateCloud cloud) {
        provider = cloud;
    }
    
    @Override
    public DataCenter getDataCenter(String dcId) throws InternalException, CloudException {
        ServiceInstance service = provider.getServiceInstance();
        
        try {
            return toDataCenter(getVmwareDatacenter(service, dcId));
        }
        finally {
            if( service != null ) {
                service.getServerConnection().logout();
            }
        }
    }

    @Override
    public String getProviderTermForDataCenter(Locale locale) {
        return "data center";
    }

    @Override
    public String getProviderTermForRegion(Locale locale) {
        return "region";
    }

    @Override
    public Region getRegion(String regionId) throws InternalException, CloudException {
        for( Region region : listRegions() ) {
            if( region.getProviderRegionId().equals(regionId) ) {
                return region;
            }
        }
        return null;
    }

    
    public Datacenter getVmwareDatacenter(ServiceInstance service, String dcId) throws CloudException, InternalException {
        Folder rootFolder = service.getRootFolder();
        
        try {
            return (Datacenter)(new InventoryNavigator(rootFolder).searchManagedEntity("Datacenter", dcId));
        }
        catch( InvalidProperty e ) {
            throw new InternalException("Invalid DC property: " + e.getMessage());
        }
        catch( RuntimeFault e ) {
            throw new InternalException("Error talking to the cluster: " + e.getMessage());
        }
        catch( RemoteException e ) {
            throw new CloudException("Error in processing the request in the cluster: " + e.getMessage());
        }
    }
    
    @Override
    public Collection<DataCenter> listDataCenters(String regionId) throws InternalException, CloudException {
        ArrayList<DataCenter> dataCenters = new ArrayList<DataCenter>();
        ServiceInstance instance = provider.getServiceInstance();
            
        try {
            Folder rootFolder = instance.getRootFolder();
            ManagedEntity[] mes;
            
            try {
                mes = new InventoryNavigator(rootFolder).searchManagedEntities("Datacenter");
            }
            catch( InvalidProperty e ) {
                throw new CloudException("No virtual machine support in cluster: " + e.getMessage());
            }
            catch( RuntimeFault e ) {
                throw new CloudException("Error in processing request to cluster: " + e.getMessage());
            }
            catch( RemoteException e ) {
                throw new CloudException("Error in cluster processing request: " + e.getMessage());
            }
            
            if( mes == null || mes.length < 1 ) {
                return dataCenters;
            }
            for( ManagedEntity entity : mes ) {
                DataCenter dc = toDataCenter((Datacenter)entity);
                
                if( dc != null ) {
                    dataCenters.add(dc);
                }
            }
            return dataCenters;
        }
        finally {
            if( instance != null ) {
                instance.getServerConnection().logout();
            }
        }            
    }

    @Override
    public Collection<Region> listRegions() throws InternalException, CloudException {
        ArrayList<Region> regions = new ArrayList<Region>();
        
        for( String endpoint : provider.getEndpoints() ) {
            Region region = toRegion(provider.getRegionId(endpoint));
            
            if( region != null ) {
                regions.add(region);
            }
        }
        return regions;
    }

    private DataCenter toDataCenter(Datacenter dc) {
        if( dc != null ) {
            DataCenter dsDc = new DataCenter();
            
            dsDc.setActive(true);
            dsDc.setAvailable(true);
            dsDc.setName(dc.getName());
            dsDc.setProviderDataCenterId(dc.getName());
            dsDc.setRegionId("home");
            return dsDc;
        }
        return null;
    }
    
    private Region toRegion(String regionId) {
        Region region = new Region();
        
        region.setActive(true);
        region.setAvailable(true);
        region.setName(regionId);
        region.setProviderRegionId(regionId);
        return region;        
    }
}
