/**
 * Copyright (C) 2010-2011 enStratusNetworks LLC
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */

package org.dasein.cloud.vsphere;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;

import org.apache.log4j.Logger;
import org.dasein.cloud.AbstractCloud;
import org.dasein.cloud.CloudException;
import org.dasein.cloud.InternalException;
import org.dasein.cloud.storage.StorageServices;
import org.dasein.cloud.vsphere.network.VMwareNetworkServices;

import com.vmware.vim25.mo.ServiceInstance;

public class PrivateCloud extends AbstractCloud {
    static private final Logger logger = Logger.getLogger(PrivateCloud.class);
    
    public PrivateCloud() { }
    
    @Override
    public String getCloudName() {
        return getContext().getCloudName();
    }

    @Override
    public VMwareNetworkServices getNetworkServices() {
        return new VMwareNetworkServices(this);
    }
    
    @Override
    public String getProviderName() {
        return getContext().getProviderName();
    }
    
    @Override
    public Dc getDataCenterServices() {
        return new Dc(this);
    }
    
    private String getEndpoint(String regionId) {
        for( String endpoint : getEndpoints() ) {
            String rid = getRegionId(endpoint);
            
            if( rid.equals(regionId) ) {
                return endpoint;
            }
        }
        return null;
    }
    
    String[] getEndpoints() {
        String[] parts = getContext().getEndpoint().split(",");
        
        if( parts == null || parts.length < 1 ) {
            return new String[] { getContext().getEndpoint() };
        }
        return parts;
    }
    
    String getRegionId(String endpoint) {
        String[] parts = endpoint.split("/");
        
        if( parts == null || parts.length < 1 ) {
            return endpoint;
        }
        for( int i=0; i<parts.length; i++ ) {
            String part = parts[i];
            
            if( part == null ) {
                continue;
            }
            part = part.trim();
            if( part.length() < 1 ) {
                continue;
            }
            if( i == 0 && part.startsWith("http") ) {
                continue;
            }
            return part;
        }
        return endpoint;        
    }
    
    public ServiceInstance getServiceInstance() throws CloudException, InternalException {
        try {
            String endpoint = getEndpoint(getContext().getRegionId());
            
            return new ServiceInstance(new URL(endpoint), new String(getContext().getAccessPublic(), "utf-8"), new String(getContext().getAccessPrivate(), "utf-8"), true);
        }
        catch( RemoteException e ) {
            throw new CloudException("Error creating service instance: " + e.getMessage());
        }
        catch( MalformedURLException e ) {
            throw new InternalException("Failed to generate endpoint URL for " + getContext().getEndpoint() + ": " + e.getMessage());
        }
        catch( UnsupportedEncodingException e ) {
            throw new InternalException("Encoding UTF-8 not supported: " + e.getMessage());
        }
    }
    
    @Override
    public String testContext() {
        try {
            if( !getComputeServices().getVirtualMachineSupport().isSubscribed() ) {
                return null;
            }
            if( hasStorageServices() ) {
                // test the storage cloud if connected to one
                StorageServices services = getStorageServices();
                
                if( services.hasBlobStoreSupport() ) {
                    services.getBlobStoreSupport().listFiles(null).iterator().next();
                }
            }
            return getContext().getAccountNumber();
        }
        catch( Throwable t ) {
            logger.warn("Failed to test vSphere context: " + t.getMessage());
            if( logger.isDebugEnabled() ) {
                t.printStackTrace();
            }
            return null;
        }
    }
}
