/**
 * Copyright (C) 2010-2011 enStratusNetworks LLC
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */

package org.dasein.cloud.vsphere.compute;

import java.io.InputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Locale;

import org.dasein.cloud.AsynchronousTask;
import org.dasein.cloud.CloudException;
import org.dasein.cloud.CloudProvider;
import org.dasein.cloud.InternalException;
import org.dasein.cloud.OperationNotSupportedException;
import org.dasein.cloud.compute.Architecture;
import org.dasein.cloud.compute.MachineImage;
import org.dasein.cloud.compute.MachineImageFormat;
import org.dasein.cloud.compute.MachineImageSupport;
import org.dasein.cloud.compute.MachineImageType;
import org.dasein.cloud.compute.Platform;
import org.dasein.cloud.vsphere.PrivateCloud;
import org.dasein.util.Jiterator;
import org.dasein.util.JiteratorPopulator;
import org.dasein.util.PopulatorThread;

import com.vmware.vim25.InvalidProperty;
import com.vmware.vim25.RuntimeFault;
import com.vmware.vim25.VirtualMachineConfigInfo;
import com.vmware.vim25.VirtualMachineGuestOsIdentifier;
import com.vmware.vim25.mo.Folder;
import com.vmware.vim25.mo.InventoryNavigator;
import com.vmware.vim25.mo.ManagedEntity;
import com.vmware.vim25.mo.ServiceInstance;
import com.vmware.vim25.mo.VirtualMachine;

public class Template implements MachineImageSupport {

    private PrivateCloud provider;
    
    Template(PrivateCloud cloud) {
        provider = cloud;
    }
    
    @Override
    public void remove(String templateId) throws InternalException, CloudException {
        // TODO Auto-generated method stub
        
    }

    @Override
    public MachineImage getMachineImage(String templateId) throws InternalException, CloudException {
        for( MachineImage image : listMachineImages() ) {
            if( image.getProviderMachineImageId().equals(templateId) ) {
                return image;
            }
        }
        return null;
    }

    @Override
    public String getProviderTermForImage(Locale locale) {
        return "template";
    }

    @Override
    public Collection<String> listShares(String templateId) throws CloudException, InternalException {
        return Collections.emptyList();
    }


    @Override
    public Collection<MachineImage> listMachineImages() throws InternalException, CloudException {
        PopulatorThread<MachineImage> populator;
        
        populator = new PopulatorThread<MachineImage>(new JiteratorPopulator<MachineImage>() {
            public void populate(Jiterator<MachineImage> iterator) throws CloudException, InternalException {
                populateImageList(iterator);
            }
        });
        populator.populate();
        return populator.getResult();
    }

    private void populateImageList(Jiterator<MachineImage> iterator) throws CloudException, InternalException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            Folder rootFolder = instance.getRootFolder();
            ManagedEntity[] mes;
            
            try {
                mes = new InventoryNavigator(rootFolder).searchManagedEntities("VirtualMachine");
            }
            catch( InvalidProperty e ) {
                throw new CloudException("No virtual machine support in cluster: " + e.getMessage());
            }
            catch( RuntimeFault e ) {
                throw new CloudException("Error in processing request to cluster: " + e.getMessage());
            }
            catch( RemoteException e ) {
                throw new CloudException("Error in cluster processing request: " + e.getMessage());
            }
            
            if( mes != null && mes.length > 0 ) {
                for( ManagedEntity entity : mes ) {
                    VirtualMachine template = (VirtualMachine)entity;

                    if( template != null ) {
                        VirtualMachineConfigInfo cfg = template.getConfig();
                        
                        if( cfg != null && cfg.isTemplate() ) {
                            MachineImage image = toMachineImage(template);
                
                            if( image != null ) {
                                iterator.push(image);
                            }
                        }
                    }
                }
            }
        }
        finally {
            if( instance != null ) {
                instance.getServerConnection().logout();
            }
        }
    }
    
    @Override
    public void shareMachineImage(String templateId, String withAccountId, boolean grant) throws InternalException, CloudException {
        throw new OperationNotSupportedException("Sharing not supported.");
    }
    
    private MachineImage toMachineImage(VirtualMachine template) throws InternalException, CloudException {
        if( template != null ) {
            VirtualMachineConfigInfo vminfo = template.getConfig();
            VirtualMachineGuestOsIdentifier os = VirtualMachineGuestOsIdentifier.valueOf(vminfo.getGuestId());
            MachineImage image = new MachineImage();
            
            image.setArchitecture(((Vm)(provider.getComputeServices().getVirtualMachineSupport())).getArchitecture(os));
            image.setDescription(template.getName());
            image.setName(template.getName());
            image.setProviderOwnerId(provider.getContext().getAccountNumber());
            image.setPlatform(Platform.guess(vminfo.getGuestFullName()));
            image.setProviderMachineImageId(vminfo.getUuid());
            image.setType(MachineImageType.VOLUME);
            return image;
        }
        return null;
    }

    @Override
    public void downloadImage(String machineImageId, OutputStream toOutput) throws CloudException, InternalException {
        throw new OperationNotSupportedException("Not supported");
    }

    @Override
    public boolean hasPublicLibrary() {
        return false;
    }

    @Override
    public AsynchronousTask<String> imageVirtualMachine(String vmId, String name, String description) throws CloudException, InternalException {
        final AsynchronousTask<String> task = new AsynchronousTask<String>();
        final String fvmId = vmId;
        final String fname = name;
        
        Thread t = new Thread() {
            public void run() {
                try {
                    ServiceInstance service = provider.getServiceInstance();
                    
                    try {
                        com.vmware.vim25.mo.VirtualMachine vm = ((Vm)provider.getComputeServices().getVirtualMachineSupport()).getVirtualMachine(service, fvmId);
                    
                        task.completeWithResult(toMachineImage(((Vm)provider.getComputeServices().getVirtualMachineSupport()).clone(service, vm, fname, true, false)).getProviderMachineImageId());
                    }
                    finally {
                        service.getServerConnection().logout();                        
                    }
                }
                catch( Throwable t ) {
                    task.complete(t);
                }
            }
        };
        
        t.setName("Image VM " + vmId);
        t.setDaemon(false);
        t.start();
        return task;
    }

    @Override
    public AsynchronousTask<String> imageVirtualMachineToStorage(String vmId, String name, String description, String directory) throws CloudException, InternalException {
        throw new OperationNotSupportedException("Not supported");
    }

    @Override
    public String installImageFromUpload(MachineImageFormat format, InputStream imageStream) throws CloudException, InternalException {
        throw new OperationNotSupportedException("Not supported");
    }

    @Override
    public boolean isImageSharedWithPublic(String machineImageId) throws CloudException, InternalException {
        return false;
    }

    @Override
    public boolean isSubscribed() throws CloudException, InternalException {
        return true;
    }

    @Override
    public Iterable<MachineImage> listMachineImagesOwnedBy(String accountId) throws CloudException, InternalException {
        if( accountId == null ) {
            return Collections.emptyList();
        }
        else if( accountId.equals(provider.getContext().getAccountNumber()) ) {
            return listMachineImages();
        }
        else {
            return Collections.emptyList();
        }
    }

    @Override
    public Iterable<MachineImageFormat> listSupportedFormats() throws CloudException, InternalException {
        return Collections.singletonList(MachineImageFormat.VMDK);
    }

    @Override
    public String registerMachineImage(String atStorageLocation) throws CloudException, InternalException {
        throw new OperationNotSupportedException("Not supported");
    }

    @Override
    public Iterable<MachineImage> searchMachineImages(String keyword, Platform platform, Architecture architecture) throws CloudException, InternalException {
        ArrayList<MachineImage> matches = new ArrayList<MachineImage>();
        
        for( MachineImage image : listMachineImages() ) {
            if( architecture != null && !architecture.equals(image.getArchitecture()) ) {
                continue;
            }
            if( platform != null && !platform.equals(Platform.UNKNOWN) ) {
                Platform mine = image.getPlatform();
                
                if( platform.isWindows() && !mine.isWindows() ) {
                    continue;
                }
                if( platform.isUnix() && !mine.isUnix() ) {
                    continue;
                }
                if( platform.isBsd() && !mine.isBsd() ) {
                    continue;
                }
                if( platform.isLinux() && !mine.isLinux() ) {
                    continue;
                }
                if( platform.equals(Platform.UNIX) ) {
                    if( !mine.isUnix() ) {
                        continue;
                    }
                }
                else if( !platform.equals(mine) ) {
                    continue;
                }
            }
            if( keyword != null && !keyword.equals("") ) {
                keyword = keyword.toLowerCase();
                if( !image.getProviderMachineImageId().toLowerCase().contains(keyword) ) {
                    if( !image.getName().toLowerCase().contains(keyword) ) {
                        if( !image.getDescription().toLowerCase().contains(keyword) ) {
                            continue;
                        }
                    }
                }
            }
            matches.add(image);
        }
        return matches;
    }

    @Override
    public boolean supportsCustomImages() {
        return true;
    }

    @Override
    public boolean supportsImageSharing() {
        return false;
    }

    @Override
    public boolean supportsImageSharingWithPublic() {
        return false;
    }

    @Override
    public String transfer(CloudProvider fromCloud, String machineImageId) throws CloudException, InternalException {
        throw new OperationNotSupportedException("Not supported");
    }

}
