/**
 * Copyright (C) 2010-2011 enStratusNetworks LLC
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */

package org.dasein.cloud.vsphere.compute;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Random;

import org.dasein.cloud.CloudException;
import org.dasein.cloud.InternalException;
import org.dasein.cloud.Tag;
import org.dasein.cloud.compute.Architecture;
import org.dasein.cloud.compute.MachineImage;
import org.dasein.cloud.compute.Platform;
import org.dasein.cloud.compute.VirtualMachine;
import org.dasein.cloud.compute.VirtualMachineProduct;
import org.dasein.cloud.compute.VirtualMachineSupport;
import org.dasein.cloud.compute.VmState;
import org.dasein.cloud.compute.VmStatistics;
import org.dasein.cloud.dc.DataCenter;
import org.dasein.cloud.vsphere.Dc;
import org.dasein.cloud.vsphere.PrivateCloud;

import com.vmware.vim25.GuestInfo;
import com.vmware.vim25.InvalidProperty;
import com.vmware.vim25.InvalidState;
import com.vmware.vim25.ManagedEntityStatus;
import com.vmware.vim25.RuntimeFault;
import com.vmware.vim25.TaskInProgress;
import com.vmware.vim25.VirtualHardware;
import com.vmware.vim25.VirtualMachineCloneSpec;
import com.vmware.vim25.VirtualMachineConfigInfo;
import com.vmware.vim25.VirtualMachineConfigSpec;
import com.vmware.vim25.VirtualMachineGuestOsIdentifier;
import com.vmware.vim25.VirtualMachinePowerState;
import com.vmware.vim25.VirtualMachineRelocateSpec;
import com.vmware.vim25.VirtualMachineRuntimeInfo;
import com.vmware.vim25.mo.ClusterComputeResource;
import com.vmware.vim25.mo.Datacenter;
import com.vmware.vim25.mo.Folder;
import com.vmware.vim25.mo.HostSystem;
import com.vmware.vim25.mo.InventoryNavigator;
import com.vmware.vim25.mo.ManagedEntity;
import com.vmware.vim25.mo.ResourcePool;
import com.vmware.vim25.mo.ServiceInstance;
import com.vmware.vim25.mo.Task;

public class Vm implements VirtualMachineSupport {

    private PrivateCloud provider;
    
    Vm(PrivateCloud provider) {
        this.provider = provider;
    }
    
    @Override
    public void boot(String serverId) throws InternalException, CloudException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            com.vmware.vim25.mo.VirtualMachine vm = getVirtualMachine(instance, serverId);
        
            if( vm != null ) {
                try {
                    Datacenter dc = getVmwareDatacenter(instance, vm);

                    if( dc == null ) {
                        throw new CloudException("Could not identify a deployment host.");
                    }
                    HostSystem host = getBestHost(dc);
                    
                    if( host == null ) {
                        throw new CloudException("Could not identify a deployment host.");
                    }
                    vm.powerOnVM_Task(host);
                }
                catch( TaskInProgress e ) {
                    throw new CloudException(e);
                }
                catch( InvalidState e ) {
                    throw new CloudException(e);
                }
                catch( RuntimeFault e ) {
                    throw new InternalException(e);
                }
                catch( RemoteException e ) {
                    throw new CloudException(e);
                }
            }
        }
        finally {
            if( instance != null ) {
                instance.getServerConnection().logout();
            }
        } 
    }

    com.vmware.vim25.mo.VirtualMachine clone(ServiceInstance service, com.vmware.vim25.mo.VirtualMachine vm, String name,  boolean asTemplate, boolean powerOn) throws InternalException, CloudException {
        try {
            name = validateName(name);
            
            Datacenter dc = provider.getDataCenterServices().getVmwareDatacenter(service, getDataCenter(vm));

            Folder vmFolder = dc.getVmFolder();
            
            VirtualMachineConfigSpec config = new VirtualMachineConfigSpec();
            VirtualMachineProduct product = getProduct(vm.getConfig().getHardware());
            String[] sizeInfo = product.getProductId().split(":");
            int cpuCount = Integer.parseInt(sizeInfo[0]);
            long memory = Long.parseLong(sizeInfo[1]);
            
            config.setName(name);
            config.setAnnotation(vm.getConfig().getAnnotation());
            config.setMemoryMB(memory);
            config.setNumCPUs(cpuCount);

            VirtualMachineCloneSpec spec = new VirtualMachineCloneSpec();
            VirtualMachineRelocateSpec location = new VirtualMachineRelocateSpec();
            ResourcePool pool = (ResourcePool) new InventoryNavigator(dc).searchManagedEntities("ResourcePool")[0];
            
            location.setHost(getBestHost(dc).getConfig().getHost());
            location.setPool(pool.getConfig().getEntity());
            spec.setLocation(location);
            spec.setPowerOn(false);
            spec.setTemplate(asTemplate);
            spec.setConfig(config);

            Task task = vm.cloneVM_Task(vmFolder, name, spec);                
            String status = task.waitForTask();
            
            if( status.equals(Task.SUCCESS) ) {
                return (com.vmware.vim25.mo.VirtualMachine)(new InventoryNavigator(vmFolder).searchManagedEntity("VirtualMachine", name));
            }
            else {
                throw new CloudException("Failed to create VM: " + task.getTaskInfo().getError().getLocalizedMessage());
            }
        }
        catch( InvalidProperty e ) {
            throw new CloudException(e);
        }
        catch( RuntimeFault e ) {
            throw new InternalException(e);
        }
        catch( RemoteException e ) {
            throw new CloudException(e);
        }
        catch( InterruptedException e ) {
            throw new InternalException(e);
        }
    }
    
    @Override
    public VirtualMachine clone(String serverId, String intoDcId, String name, String description, boolean powerOn, String ... firewallIds) throws InternalException, CloudException {
        ServiceInstance service = provider.getServiceInstance();
        
        try {
            com.vmware.vim25.mo.VirtualMachine vm = getVirtualMachine(service, serverId);
            
            if( vm != null ) {
                return toServer(clone(service, vm, name, false, powerOn), description);
            }
            throw new CloudException("No virtual machine " + serverId + ".");
        }
        finally {
            service.getServerConnection().logout();
        }
    }

    private Random random = new Random();
    
    private VirtualMachine define(String templateId, VirtualMachineProduct size, String dataCenterId, String name, boolean withAnalytics, String ... firewalls) throws InternalException, CloudException {
        ServiceInstance service = provider.getServiceInstance();

        try {
            try {
                com.vmware.vim25.mo.VirtualMachine template = getTemplate(service, templateId);
            
                name = validateName(name);
                
                Datacenter dc;
                
                if( dataCenterId == null ) {
                    for( DataCenter dsdc : provider.getDataCenterServices().listDataCenters(provider.getContext().getRegionId()) ) {
                        dataCenterId = dsdc.getProviderDataCenterId();
                        if( random.nextInt()%3 == 0 ) {
                            break;
                        }
                    }
                }
                dc = ((Dc)provider.getDataCenterServices()).getVmwareDatacenter(service, dataCenterId);

                Folder vmFolder = dc.getVmFolder();
                
                VirtualMachineConfigSpec config = new VirtualMachineConfigSpec();
                String[] vmInfo = size.getProductId().split(":");
                int cpuCount = Integer.parseInt(vmInfo[0]);
                long memory = Long.parseLong(vmInfo[1]);
                
                config.setName(name);
                config.setAnnotation(templateId);
                config.setMemoryMB(memory);
                config.setNumCPUs(cpuCount);

                VirtualMachineCloneSpec spec = new VirtualMachineCloneSpec();
                VirtualMachineRelocateSpec location = new VirtualMachineRelocateSpec();
                ResourcePool pool = (ResourcePool) new InventoryNavigator(dc).searchManagedEntities("ResourcePool")[0];
                
                location.setHost(getBestHost(dc).getConfig().getHost());
                location.setPool(pool.getConfig().getEntity());
                spec.setLocation(location);
                spec.setPowerOn(false);
                spec.setTemplate(false);
                spec.setConfig(config);

                Task task = template.cloneVM_Task(vmFolder, name, spec);                
                String status = task.waitForTask();
                
                if( status.equals(Task.SUCCESS) ) {
                    for( VirtualMachine server : listVirtualMachines() ) {
                        if( server.getName().equals(name) ) {
                            return server;
                        }
                    }
                    throw new CloudException("Unable to identify newly created server.");
                }
                else {
                    throw new CloudException("Failed to create VM: " + task.getTaskInfo().getError().getLocalizedMessage());
                }
            }
            catch( InvalidProperty e ) {
                throw new CloudException(e);
            }
            catch( RuntimeFault e ) {
                throw new InternalException(e);
            }
            catch( RemoteException e ) {
                throw new CloudException(e);
            }
            catch( InterruptedException e ) {
                throw new InternalException(e);
            }
        }
        finally {
            service.getServerConnection().logout();
        }
    }

    Architecture getArchitecture(VirtualMachineGuestOsIdentifier os) {
        if( os.name().contains("64") ) {
            return Architecture.I64;
        }
        else {
            return Architecture.I32;
        }
    }
    
    HostSystem getBestHost(Datacenter forDatacenter) throws InvalidProperty, RuntimeFault, RemoteException {
        HostSystem ohWell = null;
        
        for( ManagedEntity me : forDatacenter.getHostFolder().getChildEntity() ) {
            ClusterComputeResource cluster = (ClusterComputeResource)me;

            for( HostSystem host : cluster.getHosts() ) {
                if( host.getConfigStatus().equals(ManagedEntityStatus.green) ) {
                    return host;
                }
                if( ohWell == null || host.getConfigStatus().equals(ManagedEntityStatus.yellow) ) {
                    ohWell = host;
                }
            }
        }     
        return ohWell;
    }
    
    @Override
    public String getConsoleOutput(String serverId) throws InternalException, CloudException {
        return "";
    }

    private String getDataCenter(com.vmware.vim25.mo.VirtualMachine vm) throws InternalException, CloudException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            ManagedEntity parent = vm.getParent();
            
            while( parent != null ) {
                if( parent instanceof Datacenter ) {
                    return ((Datacenter)parent).getName();
                }
                parent = parent.getParent();
            }
            return null;
        }
        finally {
            if( instance != null ) {
                instance.getServerConnection().logout();
            }
        } 
    }
    
    @Override
    public Collection<String> listFirewalls(String serverId) throws InternalException, CloudException {
        return Collections.emptyList();
    }

    private HostSystem getHost(com.vmware.vim25.mo.VirtualMachine vm) throws InternalException, CloudException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            ManagedEntity parent = vm.getParent();
            
            while( parent != null ) {
                if( parent instanceof HostSystem ) {
                    return ((HostSystem)parent);
                }
                parent = parent.getParent();
            }
            return null;
        }
        finally {
            if( instance != null ) {
                instance.getServerConnection().logout();
            }
        } 
    }
    
    private com.vmware.vim25.mo.VirtualMachine getTemplate(ServiceInstance service, String templateId) throws InvalidProperty, RuntimeFault, RemoteException {
        Folder rootFolder = service.getRootFolder();
        ManagedEntity[] mes;
        
        mes = new InventoryNavigator(rootFolder).searchManagedEntities("VirtualMachine");
        if( mes != null && mes.length > 0 ) {
            for( ManagedEntity entity : mes ) {
                com.vmware.vim25.mo.VirtualMachine template = (com.vmware.vim25.mo.VirtualMachine)entity;
                
                if( template != null ) {
                    VirtualMachineConfigInfo vminfo = template.getConfig();
                    
                    if( vminfo.isTemplate() && vminfo.getUuid().equals(templateId) ) {
                        return template;
                    }
                }
            }
        }
        return null;
    }
    
    @Override 
    public VirtualMachineProduct getProduct(String productId) throws InternalException, CloudException {
        for( VirtualMachineProduct product : listProducts(Architecture.I64) ) {
            if( product.getProductId().equals(productId) ) {
                return product;
            }
        }
        for( VirtualMachineProduct product : listProducts(Architecture.I32) ) {
            if( product.getProductId().equals(productId) ) {
                return product;
            }
        }
        return null;
    }
    
    @Override
    public String getProviderTermForServer(Locale locale) {
        return "server";
    }

    @Override
    public VirtualMachine getVirtualMachine(String serverId) throws InternalException, CloudException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            com.vmware.vim25.mo.VirtualMachine vm = getVirtualMachine(instance, serverId);
            
            if( vm == null ) {
                System.out.println("No such VM: " + serverId);
                return null;
            }
            return toServer(vm, null);
        }
        finally {
            instance.getServerConnection().logout();
        }
    }

    @Override
    public VmStatistics getVMStatistics(String serverId, long start, long end) throws InternalException, CloudException {
        return new VmStatistics();
    }

    @Override
    public Collection<VmStatistics> getVMStatisticsForPeriod(String serverId, long start, long end) throws InternalException, CloudException {
        return new ArrayList<VmStatistics>();
    }

    private VirtualMachineProduct getProduct(VirtualHardware hardware) throws InternalException, CloudException {
        VirtualMachineProduct product = getProduct(hardware.getNumCPU() + ":" + hardware.getMemoryMB());
        
        if( product == null ) {
            int cpu = hardware.getNumCPU();
            int ram = hardware.getMemoryMB();
            int disk = 1;

            product = new VirtualMachineProduct();
            product.setCpuCount(cpu);
            product.setDescription("Custom product " + cpu + " CPU, " + ram + " RAM");
            product.setName(cpu + " CPU/" + ram + " MB RAM");
            product.setDiskSizeInGb(disk);
            product.setProductId(cpu + ":" + ram);
        }
        return product;
    }
    
    @Override
    public Collection<VirtualMachineProduct> listProducts(Architecture architecture) throws InternalException, CloudException {
        ArrayList<VirtualMachineProduct> sizes = new ArrayList<VirtualMachineProduct>();
        
        switch( architecture ) {
            case I32:
                for( int ram : new int[] { 1, 2 } ) {
                    for( int cpu : new int[] { 512, 1024, 2048 } ) {
                        VirtualMachineProduct product = new VirtualMachineProduct();
                        
                        product.setCpuCount(cpu);
                        product.setDescription("Custom product " + cpu + " CPU, " + ram + " RAM");
                        product.setName(cpu + " CPU/" + ram + " MB RAM");
                        product.setDiskSizeInGb(1);
                        product.setProductId(cpu + ":" + ram);
                        sizes.add(product);
                    }
                }
                break;
            case I64:
                for( int ram : new int[] { 1, 2, 4, 8 } ) {
                    for( int cpu : new int[] { 1024, 2048, 4096, 10240, 20480 } ) {
                        VirtualMachineProduct product = new VirtualMachineProduct();
                        
                        product.setCpuCount(cpu);
                        product.setDescription("Custom product " + cpu + " CPU, " + ram + " RAM");
                        product.setName(cpu + " CPU/" + ram + " MB RAM");
                        product.setDiskSizeInGb(1);
                        product.setProductId(cpu + ":" + ram);
                        sizes.add(product);
                    }
                }                
                break;
        }
        return sizes;
    }

    com.vmware.vim25.mo.VirtualMachine getVirtualMachine(ServiceInstance instance, String vmId) throws CloudException, InternalException { 
        Folder rootFolder = instance.getRootFolder();
        ManagedEntity[] mes;
        
        StringBuilder debug = new StringBuilder();
        
        debug.append("Looking up " + vmId + " /" );
        try {
            mes = new InventoryNavigator(rootFolder).searchManagedEntities("VirtualMachine");
            debug.append("Folder=" + mes + " / ");
        }
        catch( InvalidProperty e ) {
            throw new CloudException("No virtual machine support in cluster: " + e.getMessage());
        }
        catch( RuntimeFault e ) {
            throw new CloudException("Error in processing request to cluster: " + e.getMessage());
        }
        catch( RemoteException e ) {
            throw new CloudException("Error in cluster processing request: " + e.getMessage());
        }
        
        if( mes != null && mes.length > 0 ) {
            for( ManagedEntity entity : mes ) {
                com.vmware.vim25.mo.VirtualMachine vm = (com.vmware.vim25.mo.VirtualMachine)entity;

                debug.append("VM=" + vm + " / ");
                if( vm != null ) {
                    debug.append(vm.getName() + ": " + vm.getConfig().getInstanceUuid() + " / ");
                }
                if( vm != null && vm.getConfig().getInstanceUuid().equals(vmId) ) {
                    return vm;
                }
            }
        }
        System.out.println("DEBUG=" + debug.toString());
        return null;
    }
    
    private Datacenter getVmwareDatacenter(ServiceInstance service, com.vmware.vim25.mo.VirtualMachine vm) throws CloudException {
        ManagedEntity parent = vm.getParent();
        
        while( parent != null ) {
            if( parent instanceof Datacenter ) {
                return ((Datacenter)parent);
            }
            parent = parent.getParent();
        }
        return null;
    }

    @Override
    public VirtualMachine launch(String templateId, VirtualMachineProduct product, String dataCenterId, String serverName, String description, String withKey, String inVlanId, boolean withMonitoring, boolean forImaging, String... firewalls) throws InternalException, CloudException {
        return launch(templateId, product, dataCenterId, serverName, description, withKey, inVlanId, withMonitoring, forImaging, firewalls, new Tag[0]);
    }
    
    @Override
    public VirtualMachine launch(String templateId, VirtualMachineProduct product, String dataCenterId, String serverName, String description, String withKeypairId, String inVlanId, boolean withMonitoring, boolean asSandbox, String[] firewalls, Tag ... tags) throws InternalException, CloudException {        
        VirtualMachine server = define(templateId, product, dataCenterId, serverName, withMonitoring, firewalls);
        
        boot(server.getProviderVirtualMachineId());
        return server;
    }

    @Override
    public Collection<VirtualMachine> listVirtualMachines() throws InternalException, CloudException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            ArrayList<VirtualMachine> servers = new ArrayList<VirtualMachine>();
            Folder rootFolder = instance.getRootFolder();
            ManagedEntity[] mes;
            
            try {
                mes = new InventoryNavigator(rootFolder).searchManagedEntities("VirtualMachine");
            }
            catch( InvalidProperty e ) {
                throw new CloudException("No virtual machine support in cluster: " + e.getMessage());
            }
            catch( RuntimeFault e ) {
                throw new CloudException("Error in processing request to cluster: " + e.getMessage());
            }
            catch( RemoteException e ) {
                throw new CloudException("Error in cluster processing request: " + e.getMessage());
            }
            
            if( mes != null && mes.length > 0 ) {
                for( ManagedEntity entity : mes ) {
                    VirtualMachine server = toServer((com.vmware.vim25.mo.VirtualMachine)entity, null);
                
                    if( server != null ) {
                        servers.add(server);
                    }
                }
            }
            return servers;
        }
        finally {
            if( instance != null ) {
                instance.getServerConnection().logout();
            }
        }
    }

    @Override
    public void enableAnalytics(String serverId) throws InternalException, CloudException {
        // NO-OP
    }

    @Override
    public void pause(String serverId) throws InternalException, CloudException {
        ServiceInstance instance = provider.getServiceInstance();
        
        try {
            com.vmware.vim25.mo.VirtualMachine vm = getVirtualMachine(instance, serverId);
            
            if( vm != null ) {
                try {
                    vm.powerOffVM_Task();
                }
                catch( TaskInProgress e ) {
                    throw new CloudException(e);
                }
                catch( InvalidState e ) {
                    throw new CloudException(e);
                }
                catch( RuntimeFault e ) {
                    throw new InternalException(e);
                }
                catch( RemoteException e ) {
                    throw new CloudException(e);
                }
            }
        }
        finally {
            if( instance != null) {
                instance.getServerConnection().logout();
            }
        }
    }

    @Override
    public void reboot(String serverId) throws CloudException, InternalException {
        final String id = serverId;
        
        Thread t = new Thread() {
            public void run() {
                powerOnAndOff(id);
            }
        };
        
        t.setName("Reboot " + serverId);
        t.setDaemon(true);
        t.start();
    }
    
    private void powerOnAndOff(String serverId) {
        try {
            ServiceInstance service = provider.getServiceInstance();
            
            try {
                com.vmware.vim25.mo.VirtualMachine vm = getVirtualMachine(service, serverId);
                HostSystem host = getHost(vm);

                if( vm != null ) {
                    Task task = vm.powerOffVM_Task();
                    String status = task.waitForTask();
    
                    if( !status.equals(Task.SUCCESS) ) {
                        System.err.println("Reboot failed: " + status);
                    }
                    else {
                        try { Thread.sleep(15000L); }
                        catch( InterruptedException e ) { }
                        vm = getVirtualMachine(service, serverId);
                        vm.powerOnVM_Task(host);
                    }
                }
            }
            finally {
                service.getServerConnection().logout();
            }
        }
        catch( Throwable t ) {
            t.printStackTrace();
        }
    }

    @Override
    public void terminate(String serverId) throws InternalException, CloudException {
        final String id = serverId;
        
        Thread t = new Thread() {
            public void run() {
                terminateVm(id);
            }
        };
        
        t.setName("Reboot " + serverId);
        t.setDaemon(true);
        t.start();
    }

    private void terminateVm(String serverId) {
        try {
            ServiceInstance service = provider.getServiceInstance();
            
            try {
                com.vmware.vim25.mo.VirtualMachine vm = getVirtualMachine(service, serverId);

                if( vm != null ) {
                    Task task = vm.powerOffVM_Task();
                    String status = task.waitForTask();
    
                    if( !status.equals(Task.SUCCESS) ) {
                        System.err.println("Termination failed: " + status);
                    }
                    else {
                        try { Thread.sleep(15000L); }
                        catch( InterruptedException e ) { }
                        vm = getVirtualMachine(service, serverId);
                        vm.destroy_Task();
                    }
                }
            }
            finally {
                service.getServerConnection().logout();
            }
        }
        catch( Throwable t ) {
            t.printStackTrace();
        }
    }
    
    static private HashMap<String,Integer> randomIps = new HashMap<String,Integer>();
    
    static private int ip = 1;
    
    private VirtualMachine toServer(com.vmware.vim25.mo.VirtualMachine vm, String description) throws InternalException, CloudException {
        if( vm != null ) {
            VirtualMachineConfigInfo vminfo = vm.getConfig();
            
            if( vminfo.isTemplate() ) {
                return null;
            }
            VirtualMachineGuestOsIdentifier os = VirtualMachineGuestOsIdentifier.valueOf(vminfo.getGuestId());
            VirtualMachine server = new VirtualMachine();
            Integer myIp = randomIps.get(vminfo.getInstanceUuid());
            
            if( myIp == null ) {
                myIp = ip++;
                randomIps.put(vminfo.getInstanceUuid(), myIp);
            }
            server.setPrivateIpAddresses(new String[] {"10.200.0." + myIp });
            server.setName(vm.getName());
            server.setPlatform(Platform.guess(vminfo.getGuestFullName()));
            server.setProviderVirtualMachineId(vm.getConfig().getInstanceUuid());
            server.setPersistent(true);
            server.setArchitecture(getArchitecture(os));
            if( description != null ) {
                server.setDescription(description);
            }
            else {
                server.setDescription(vm.getName() + " (" + vminfo.getGuestFullName() + ")");
            }
            server.setProduct(getProduct(vminfo.getHardware()));
            String imageId = vminfo.getAnnotation();
            MachineImage img = null;
            

            if( imageId != null ) {
                img = provider.getComputeServices().getImageSupport().getMachineImage(imageId);
            }
            if( img != null ) {
                server.setProviderMachineImageId(vminfo.getAnnotation());
            }
            else {  
                server.setProviderMachineImageId(provider.getContext().getAccountNumber() + "-unknown");
            }
            server.setProviderRegionId(provider.getContext().getRegionId());
            server.setProviderDataCenterId(getDataCenter(vm));

            GuestInfo guestInfo = vm.getGuest();
            
            if( guestInfo != null ) {
                String ipAddress = guestInfo.getIpAddress();
                
                if( ipAddress != null ) {
                    server.setPrivateIpAddresses(new String[] { guestInfo.getIpAddress() });
                    server.setPrivateDnsAddress(guestInfo.getIpAddress());
                }
            }

            VirtualMachineRuntimeInfo runtime = vm.getRuntime();
            
            if( runtime != null ) {
                VirtualMachinePowerState state = runtime.getPowerState();
                
                if( server.getCurrentState() == null ) {
                    switch( state ) {
                        case poweredOff: case suspended:
                            server.setCurrentState(VmState.PAUSED);
                            break;
                        case poweredOn:
                            boolean pending = false;
                            
                            /*
                            for( Task task : vm.getRecentTasks() ) {
                                // TODO: review task
                                // if power on or power off, set pending = true
                            }
                            */
                            server.setCurrentState(pending ? VmState.PENDING : VmState.RUNNING);
                            break;
                    }
                }
                Calendar suspend = runtime.getSuspendTime();                
                Calendar time = runtime.getBootTime();

                if( suspend == null || suspend.getTimeInMillis() < 1L ) {
                    server.setLastPauseTimestamp(-1L);
                }
                else {
                    server.setLastPauseTimestamp(suspend.getTimeInMillis());
                    server.setCreationTimestamp(server.getLastPauseTimestamp());
                }
                if( time == null || time.getTimeInMillis() < 1L ) {
                    server.setLastBootTimestamp(0L);
                }
                else {
                    server.setLastBootTimestamp(time.getTimeInMillis());
                    server.setCreationTimestamp(server.getLastBootTimestamp());
                }
            }
            server.setProviderOwnerId(provider.getContext().getAccountNumber());
            return server;
        }
        return null;
    }
    
    @Override
    public void disableAnalytics(String serverId) throws InternalException, CloudException {
        // NO-OP
    }

    private String validateName(String name) {
        name = name.toLowerCase().replaceAll("_", "-");
        if( name.length() <= 30 ) {
            return name;
        }
        return name.substring(0, 30);
    }

    @Override
    public boolean isSubscribed() throws CloudException, InternalException {
        return true;
    }

    @Override
    public boolean supportsAnalytics() throws CloudException, InternalException {
        return false;
    }
}
