/**
 * Copyright (C) 2010-2011 enStratusNetworks LLC
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */

package org.dasein.cloud.vsphere.network;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.Untainted;

import org.dasein.cloud.CloudException;
import org.dasein.cloud.InternalException;
import org.dasein.cloud.network.AddressType;
import org.dasein.cloud.network.IpAddress;
import org.dasein.cloud.network.IpAddressSupport;
import org.dasein.cloud.network.IpForwardingRule;
import org.dasein.cloud.network.Protocol;
import org.dasein.cloud.vsphere.PrivateCloud;

@Untainted
public class StaticIp implements IpAddressSupport {
    static private volatile Map<String,Collection<IpAddress>> ipAddresses;
    
    static {
        ipAddresses = new HashMap<String,Collection<IpAddress>>();
        load();
    }
    
    static private void addIp(Map<String,Collection<IpAddress>> current, String ipAddress, String regionId) {
        IpAddress addr = new IpAddress();
        
        addr.setAddress(ipAddress);
        addr.setIpAddressId(ipAddress);
        addr.setRegionId(regionId);
        addr.setServerId(null);  
        addr.setProviderLoadBalancerId(null);
        addr.setAddressType(AddressType.PUBLIC);
        Collection<IpAddress> addresses = current.get(regionId);
        
        if( addresses == null ) {
            addresses = new ArrayList<IpAddress>();
            current.put(regionId, addresses);
        }
        addresses.add(addr);
    }
    
    static private void load() {
        try {
            InputStream input = StaticIp.class.getResourceAsStream("/es-vsphere-ip.cfg");
            
            if( input != null ) {
                BufferedReader reader = new BufferedReader(new InputStreamReader(input));
                HashMap<String,Collection<IpAddress>> tmp = new HashMap<String,Collection<IpAddress>>();
                String line;
                
                while( (line = reader.readLine()) != null ) {
                    line = line.trim();
                    if( !line.startsWith("#") && !line.startsWith("//") ) {
                        String[] parts = line.split("=");
                        String cidr = parts[0];
                        String regionId = parts[1];
                        int idx;
                        
                        idx = cidr.indexOf('/');
                        if( idx == -1 ) {
                            addIp(tmp, cidr, regionId);
                        }
                        else {
                            int mask;
                            
                            parts = cidr.split("/");
                            cidr = parts[0];
                            mask = Integer.parseInt(parts[1]);
                            if( mask == 32 ) {
                                addIp(tmp, cidr, regionId);
                            }
                            else {
                                // TODO: calculate netmask
                            }
                        }
                    }
                }
                HashMap<String,Collection<IpAddress>> clean = new HashMap<String,Collection<IpAddress>>();

                for( Map.Entry<String,Collection<IpAddress>> entry : tmp.entrySet() ) {
                    clean.put(entry.getKey(), Collections.unmodifiableCollection(entry.getValue()));
                }
                ipAddresses = clean;
            }
        }
        catch( Throwable t ) {
            t.printStackTrace();
        }
    }
    
    private PrivateCloud provider;
    
    StaticIp(PrivateCloud cloud) {
        provider = cloud;
    }
    
    @Override
    public void assign(@Nonnull String addressId, @Nonnull String toServerId) throws InternalException, CloudException {
        // TODO Auto-generated method stub
        
    }

    @Override
    public @Nonnull String forward(@Nonnull String addressId, int publicPort, @Nonnull Protocol protocol, int privatePort, @Nonnull String onServerId) throws InternalException, CloudException {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public @Nullable IpAddress getIpAddress(@Nonnull String addressId) throws InternalException, CloudException {
        for( IpAddress address : listPublicIpPool(false) ) {
            if( address.getProviderIpAddressId().equals(addressId) ) {
                return address;
            }
        }
        for( IpAddress address : listPrivateIpPool(false) ) {
            if( address.getProviderIpAddressId().equals(addressId) ) {
                return address;
            }
        }
        return null;
    }

    @Override
    public @Nonnull String getProviderTermForIpAddress(@Nonnull Locale locale) {
        return "IP address";
    }

    @Override
    public boolean isAssigned(@Nonnull AddressType type) {
        return false;
    }

    @Override
    public boolean isForwarding() {
        return false;
    }

    @Override
    public boolean isRequestable(@Nonnull AddressType type) {
        return false;
    }
    
    @Override
    public boolean isSubscribed() throws CloudException, InternalException {
        return true;
    }
    
    @Override
    public @Nonnull Iterable<IpAddress> listPrivateIpPool(boolean unassignedOnly) throws InternalException, CloudException {
        return Collections.emptyList();
    }
    
    @Override
    public @Nonnull Iterable<IpAddress> listPublicIpPool(boolean unassignedOnly) throws InternalException, CloudException {
        Collection<IpAddress> addresses = ipAddresses.get(provider.getContext().getRegionId());
        
        if( addresses == null ) {
            return Collections.emptyList();
        }
        else {
            return addresses;
        }
    }

    @Override
    public @Nonnull Iterable<IpForwardingRule> listRules(@Nonnull String addressId) throws InternalException, CloudException {
        return Collections.emptyList();
    }

    @Override
    public @Nonnull String request(@Nonnull AddressType type) throws InternalException, CloudException {
        throw new CloudException("Unable to allocate new IP addresses.");
    }

    @Override
    public void releaseFromPool(@Nonnull String addressId) throws InternalException, CloudException {
        throw new CloudException("Unable to delete the specified address.");
    }
    
    @Override
    public void releaseFromServer(@Nonnull String addressId) throws InternalException, CloudException {
        // TODO Auto-generated method stub        
    }

    @Override
    public void stopForward(@Nonnull String addressId) throws InternalException, CloudException {
        // TODO Auto-generated method stub
    }
}
