/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.excel;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import javax.transaction.xa.XAResource;

import org.apache.poi.ss.usermodel.Workbook;
import org.datanucleus.OMFContext;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.store.connection.AbstractConnectionFactory;
import org.datanucleus.store.connection.AbstractManagedConnection;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.connection.ManagedConnectionResourceListener;

/**
 * Implementation of a ConnectionFactory for Excel.
 */
public class ConnectionFactoryImpl extends AbstractConnectionFactory
{
    boolean ooxml = false;

    /**
     * Constructor.
     * @param omfContext The OMF context
     * @param resourceType Type of resource (tx, nontx)
     */
    public ConnectionFactoryImpl(OMFContext omfContext, String resourceType)
    {
        super(omfContext, resourceType);
        this.omfContext = omfContext;
    }

    /**
     * Obtain a connection from the Factory. The connection will be enlisted within the {@link org.datanucleus.Transaction} 
     * associated to the <code>poolKey</code> if "enlist" is set to true.
     * @param poolKey the pool that is bound the connection during its lifecycle (or null)
     * @param options Any options for then creating the connection
     * @return the {@link org.datanucleus.store.connection.ManagedConnection}
     */
    public ManagedConnection createManagedConnection(Object poolKey, Map transactionOptions)
    {
        return new ManagedConnectionImpl(omfContext, transactionOptions);
    }

    public class ManagedConnectionImpl extends AbstractManagedConnection
    {
        OMFContext omf;
        Map options;

        /** The Excel XLS/OOXML file. */
        File file;
        
        public ManagedConnectionImpl(OMFContext omf, Map options)
        {
            this.omf = omf;
            this.options = options;
        }

        public void close()
        {
            if (conn == null)
            {
                return;
            }
            for (int i=0; i<listeners.size(); i++)
            {
                ((ManagedConnectionResourceListener)listeners.get(i)).managedConnectionPreClose();
            }
            try
            {
                //close
                FileOutputStream os = new FileOutputStream(file);
                //this is a kind of commit.... do it somewhere else
                ((Workbook)conn).write(new FileOutputStream(file));
                os.close();
                conn = null;
            }
            catch (IOException e)
            {
                throw new NucleusException(e.getMessage(),e);
            }
            finally
            {
                for (int i=0; i<listeners.size(); i++)
                {
                    ((ManagedConnectionResourceListener)listeners.get(i)).managedConnectionPostClose();
                }
            }
        }

        public Object getConnection()
        {
            if (conn == null)
            {
                // "excel:file:{filename}"/"ooxml:file:{filename}"
                String url = omf.getStoreManager().getConnectionURL();
                if (url == null)
                {
                    throw new NucleusException("You haven't specified persistence property 'datanucleus.ConnectionURL' (or alias)");
                }

                if (url.startsWith("excel:"))
                {
                    ooxml = false;
                }
                else if (url.startsWith("ooxml:"))
                {
                    ooxml = true;
                }
                else
                {
                    throw new NucleusException("invalid URL: "+url);
                }

                // Split the URL into filename
                String str = url.substring(6); // Omit "excel:"/"ooxml:"
                if (str.indexOf("file:") != 0)
                {
                    throw new NucleusException("invalid URL: "+url);
                }
                try
                {
                    String filename = str.substring("file:".length()); // Omit "file:"
                    file = new File(filename);
                    if (!file.exists())
                    {
                        // Excel document doesn't exist, so create
                        Workbook wb = null;
                        if (ooxml)
                        {
                            wb = new org.apache.poi.xssf.usermodel.XSSFWorkbook();
                        }
                        else
                        {
                            wb = new org.apache.poi.hssf.usermodel.HSSFWorkbook();
                        }
                        FileOutputStream fileOut = new FileOutputStream(file);
                        wb.write(fileOut);
                        fileOut.close();
                    }

                    InputStream is = new FileInputStream(file);
                    if (ooxml)
                    {
                        conn = new org.apache.poi.xssf.usermodel.XSSFWorkbook(is);
                    }
                    else
                    {
                        conn = new org.apache.poi.hssf.usermodel.HSSFWorkbook(is);
                    }
                }
                catch (IOException e)
                {
                    throw new NucleusException(e.getMessage(),e);
                }
            }
            return conn;
        }

        public XAResource getXAResource()
        {
            return null;
        }
    }
}