/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors :
2008 Andy Jefferson - app id dups check, row number finder, factor much code into ExcelUtils
 ...
***********************************************************************/
package org.datanucleus.store.excel;

import java.sql.Timestamp;
import java.util.Date;
import java.util.Iterator;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.excel.fieldmanager.FetchFieldManager;
import org.datanucleus.store.excel.fieldmanager.StoreFieldManager;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;
import org.datanucleus.exceptions.NucleusOptimisticException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.identity.OID;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.metadata.VersionMetaData;
import org.datanucleus.metadata.VersionStrategy;
import org.datanucleus.store.AbstractPersistenceHandler;
import org.datanucleus.store.StoreManager;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Persistence Handler for Excel datastores. 
 * Handles the insert/update/delete/fetch/locate operations by using Apache POI.
 * Some notes about Apache POI utilisation :-
 * <ul>
 * <li>We have a Workbook, composed of a set of named Sheet objects.</li>
 * <li>Each class is persisted to its own sheet.</li>
 * <li>Insert of an object requires creation of a Row. Unless we are on the last row
 *     in the sheet in which case we have a row and just need to add cells. See "delete"</li>
 * <li>Delete of an object will involve removal of the row, EXCEPT in the case of the final row
 *     in the sheet in which case we have to delete all cells since Apache POI doesn't seem to
 *     allow removal of the last row.</li>
 * </ul>
 */
public class ExcelPersistenceHandler extends AbstractPersistenceHandler
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.store.excel.Localisation", ExcelStoreManager.class.getClassLoader());

    protected final ExcelStoreManager storeMgr;

    /**
     * Constructor.
     * @param storeMgr Manager for the datastore
     */
    public ExcelPersistenceHandler(StoreManager storeMgr)
    {
        this.storeMgr = (ExcelStoreManager)storeMgr;
    }

    public void close()
    {
        // Nothing to do since we maintain no resources
    }

    /**
     * Method to insert the object into the datastore.
     * @param sm StateManager of the object
     */
    public void insertObject(final StateManager sm)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        AbstractClassMetaData cmd = sm.getClassMetaData();
        if (cmd.getIdentityType() == IdentityType.APPLICATION || cmd.getIdentityType() == IdentityType.DATASTORE)
        {
            // Enforce uniqueness of datastore rows
            try
            {
                locateObject(sm);
                throw new NucleusUserException(LOCALISER.msg("Excel.Insert.ObjectWithIdAlreadyExists",
                    StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId()));
            }
            catch (NucleusObjectNotFoundException onfe)
            {
                // Do nothing since object with this id doesn't exist
            }
        }

        ObjectManager om = sm.getObjectManager();
        ManagedConnection mconn = storeMgr.getConnection(om);
        try
        {
            long startTime = System.currentTimeMillis();
            if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.Insert.Start", 
                    StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId()));
            }

            Workbook wb = (Workbook) mconn.getConnection();
            int[] fieldNumbers = cmd.getAllMemberPositions();
            String sheetName = ExcelUtils.getSheetNameForClass(cmd);
            Sheet sheet = wb.getSheet(sheetName);
            int rowNum = 0;
            if (sheet == null)
            {
                // Sheet doesn't exist so create it
                sheet = wb.createSheet(sheetName);
                if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
                {
                    NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.Insert.SheetCreated",
                        StringUtils.toJVMIDString(sm.getObject()), sheetName));
                }
            }
            else
            {
                // Find number of active rows in this sheet
                rowNum += ExcelUtils.getNumberOfRowsInSheetOfWorkbook(sm, wb);
            }

            // Create the object in the datastore
            Row row = sheet.getRow(rowNum);
            if (row == null)
            {
                // No row present so create holder for the cells
                row = sheet.createRow(rowNum);
            }
            else
            {
                // Row already exists but with no cells
            }
            sm.provideFields(fieldNumbers, new StoreFieldManager(sm, row));

            if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.ExecutionTime", 
                    (System.currentTimeMillis() - startTime)));
            }
            if (storeMgr.getRuntimeManager() != null)
            {
                storeMgr.getRuntimeManager().incrementInsertCount();
            }

            if (cmd.getIdentityType() == IdentityType.DATASTORE)
            {
                // Set the datastore identity column value
                int idCellNum = (int)ExcelUtils.getColumnIndexForFieldOfClass(cmd, -1);
                Object key = ((OID)sm.getInternalObjectId()).getKeyValue();
                if (key instanceof String)
                {
                    Cell idCell = row.getCell(idCellNum);
                    if (idCell == null)
                    {
                        idCell = row.createCell(idCellNum);
                    }
                    String idValue = (String)key;
                    CreationHelper createHelper = wb.getCreationHelper();
                    idCell.setCellValue(createHelper.createRichTextString(idValue));
                }
                else
                {
                    Cell idCell = row.getCell(idCellNum);
                    if (idCell == null)
                    {
                        idCell = row.createCell(idCellNum);
                    }
                    long idValue = ((Long)key).longValue();
                    idCell.setCellValue(idValue);
                }
            }
            if (cmd.hasVersionStrategy())
            {
                // versioned object so set its version
                int verCellNum = (int)ExcelUtils.getColumnIndexForFieldOfClass(cmd, -2);
                Cell verCell = row.getCell(verCellNum);
                if (verCell == null)
                {
                    verCell = row.createCell(verCellNum);
                }
                if (cmd.getVersionMetaData().getVersionStrategy() == VersionStrategy.VERSION_NUMBER)
                {
                    long versionNumber = 1;
                    sm.setTransactionalVersion(Long.valueOf(versionNumber));
                    if (NucleusLogger.DATASTORE.isDebugEnabled())
                    {
                        NucleusLogger.DATASTORE.debug(LOCALISER.msg("Excel.Insert.ObjectPersistedWithVersion",
                            StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId(), "" + versionNumber));
                    }
                    verCell.setCellValue(versionNumber);
                }
                else if (cmd.getVersionMetaData().getVersionStrategy() == VersionStrategy.DATE_TIME)
                {
                    Date date = new Date();
                    Timestamp ts = new Timestamp(date.getTime());
                    sm.setTransactionalVersion(ts);
                    if (NucleusLogger.DATASTORE.isDebugEnabled())
                    {
                        NucleusLogger.DATASTORE.debug(LOCALISER.msg("Excel.Insert.ObjectPersistedWithVersion",
                            StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId(), "" + ts));
                    }
                    verCell.setCellValue(date);
                }
            }
            else
            {
                if (NucleusLogger.DATASTORE.isDebugEnabled())
                {
                    NucleusLogger.DATASTORE.debug(LOCALISER.msg("Excel.Insert.ObjectPersisted",
                        StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId()));
                }
            }
        }
        finally
        {
            mconn.release();
        }
    }

    /**
     * Method to handle the update of fields of an object in the datastore.
     * @param sm StateManager for the object
     * @param fieldNumbers Absolute numbers of fields to be updated
     */
    public void updateObject(final StateManager sm, int[] fieldNumbers)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        ManagedConnection mconn = storeMgr.getConnection(sm.getObjectManager());
        try
        {
            Workbook wb = (Workbook) mconn.getConnection();
            final Sheet sheet = ExcelUtils.getSheetForClass(sm, wb);

            // TODO Add optimistic checks
            Object currentVersion = sm.getTransactionalVersion(sm.getObject());
            Object nextVersion = null;
            AbstractClassMetaData acmd = sm.getClassMetaData();
            if (acmd.hasVersionStrategy())
            {
                // Version object so calculate version to store with
                VersionMetaData vermd = acmd.getVersionMetaData();
                if (acmd.getVersionMetaData().getFieldName() != null)
                {
                    // Version field
                    AbstractMemberMetaData verfmd = acmd.getMetaDataForMember(vermd.getFieldName());
                    if (currentVersion instanceof Integer)
                    {
                        // Cater for Integer-based versions TODO Generalise this
                        currentVersion = Long.valueOf(((Integer)currentVersion).longValue());
                    }

                    nextVersion = acmd.getVersionMetaData().getNextVersion(currentVersion);
                    if (verfmd.getType() == Integer.class || verfmd.getType() == int.class)
                    {
                        // Cater for Integer-based versions TODO Generalise this
                        nextVersion = Integer.valueOf(((Long)nextVersion).intValue());
                    }
                    sm.replaceField(verfmd.getAbsoluteFieldNumber(), nextVersion, false);
                }
                else
                {
                    // Surrogate version column
                    nextVersion = vermd.getNextVersion(currentVersion);
                }
                sm.setTransactionalVersion(nextVersion);
            }

            long startTime = System.currentTimeMillis();
            if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
            {
                StringBuffer fieldStr = new StringBuffer();
                for (int i=0;i<fieldNumbers.length;i++)
                {
                    if (i > 0)
                    {
                        fieldStr.append(",");
                    }
                    fieldStr.append(acmd.getMetaDataForManagedMemberAtAbsolutePosition(fieldNumbers[i]).getName());
                }
                NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.Update.Start", 
                    StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId(), fieldStr.toString()));
            }

            // Update the row in the worksheet
            final Row row = sheet.getRow(ExcelUtils.getRowNumberForObjectInWorkbook(sm, wb));
            if (row == null)
            {
                String sheetName = ExcelUtils.getSheetNameForClass(sm.getClassMetaData());
                throw new NucleusDataStoreException(LOCALISER.msg("Excel.RowNotFoundForSheetForWorkbook",
                    sheetName, StringUtils.toJVMIDString(sm.getInternalObjectId())));
            }
            sm.provideFields(fieldNumbers, new StoreFieldManager(sm, row));

            if (acmd.hasVersionStrategy())
            {
                // Versioned object so set version cell in spreadsheet
                int verCellNum = (int)ExcelUtils.getColumnIndexForFieldOfClass(acmd, -2);
                Cell verCell = row.getCell(verCellNum);
                if (acmd.getVersionMetaData().getVersionStrategy() == VersionStrategy.VERSION_NUMBER)
                {
                    sm.setTransactionalVersion(nextVersion);
                    if (NucleusLogger.DATASTORE.isDebugEnabled())
                    {
                        NucleusLogger.DATASTORE.debug(LOCALISER.msg("Excel.Insert.ObjectPersistedWithVersion",
                            StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId(), 
                            "" + nextVersion));
                    }
                    verCell.setCellValue(((Long)nextVersion).longValue());
                }
                else if (acmd.getVersionMetaData().getVersionStrategy() == VersionStrategy.DATE_TIME)
                {
                    sm.setTransactionalVersion(nextVersion);
                    if (NucleusLogger.DATASTORE.isDebugEnabled())
                    {
                        NucleusLogger.DATASTORE.debug(LOCALISER.msg("Excel.Insert.ObjectPersistedWithVersion",
                            StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId(), "" + nextVersion));
                    }
                    Timestamp ts = (Timestamp)nextVersion;
                    Date date = new Date();
                    date.setTime(ts.getTime());
                    verCell.setCellValue(ts);
                }
            }

            if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.ExecutionTime", 
                    (System.currentTimeMillis() - startTime)));
            }
            if (storeMgr.getRuntimeManager() != null)
            {
                storeMgr.getRuntimeManager().incrementUpdateCount();
            }
        }
        finally
        {
            mconn.release();
        }
    }

    /**
     * Deletes a persistent object from the database.
     * @param sm The state manager of the object to be deleted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     * @throws NucleusOptimisticException thrown if version checking fails on an optimistic transaction for this object
     */
    public void deleteObject(StateManager sm)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        ManagedConnection mconn = storeMgr.getConnection(sm.getObjectManager());
        try
        {
            Workbook wb = (Workbook) mconn.getConnection();
            final Sheet sheet = ExcelUtils.getSheetForClass(sm, wb);

            // TODO Add optimistic checks

            long startTime = System.currentTimeMillis();
            if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.Delete.Start", 
                    StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId()));
            }

            int rowId = ExcelUtils.getRowNumberForObjectInWorkbook(sm, wb);
            if (rowId < 0)
            {
                throw new NucleusObjectNotFoundException("object not found", sm.getObject());
            }
            else if (rowId > 0)
            {
                // Isn't the only row of the sheet so safe to remove it
                sheet.removeRow(sheet.getRow(rowId));
                if (sheet.getLastRowNum() > rowId)
                {
                    sheet.shiftRows(rowId+1, sheet.getLastRowNum(), -1);
                }
            }
            else
            {
                if (storeMgr instanceof XLSStoreManager && sheet.getLastRowNum() == rowId)
                {
                    // Deleting top row which is last row so just remove all cells and leave row
                    // otherwise Apache POI throws an ArrayIndexOutOfBoundsException
                    Row rrow = sheet.getRow(rowId);
                    Iterator it = rrow.cellIterator();
                    while(it.hasNext())
                    {
                        Cell cell = (Cell) it.next();
                        rrow.removeCell(cell);
                    }
                }
                else
                {
                    // Deleting top row so remove it
                    sheet.removeRow(sheet.getRow(rowId));
                    if (sheet.getLastRowNum()>rowId)
                    {
                        sheet.shiftRows(rowId+1, sheet.getLastRowNum(),-1);
                    }
                }
            }

            if (NucleusLogger.DATASTORE_PERSIST.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_PERSIST.debug(LOCALISER.msg("Excel.ExecutionTime", 
                    (System.currentTimeMillis() - startTime)));
            }
            if (storeMgr.getRuntimeManager() != null)
            {
                storeMgr.getRuntimeManager().incrementDeleteCount();
            }
        }
        finally
        {
            mconn.release();
        }
    }

    /**
     * Fetches fields of a persistent object from the database.
     * @param sm The state manager of the object to be fetched.
     * @param fieldNumbers The numbers of the fields to be fetched.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void fetchObject(final StateManager sm, int[] fieldNumbers)
    {
        AbstractClassMetaData acmd = sm.getClassMetaData();
        if (NucleusLogger.PERSISTENCE.isDebugEnabled())
        {
            // Debug information about what we are retrieving
            StringBuffer str = new StringBuffer("Fetching object \"");
            str.append(StringUtils.toJVMIDString(sm.getObject())).append("\" (id=");
            str.append(sm.getObjectManager().getApiAdapter().getObjectId(sm)).append(")").append(" fields [");
            for (int i=0;i<fieldNumbers.length;i++)
            {
                if (i > 0)
                {
                    str.append(",");
                }
                str.append(acmd.getMetaDataForManagedMemberAtAbsolutePosition(fieldNumbers[i]).getName());
            }
            str.append("]");
            NucleusLogger.PERSISTENCE.debug(str);
        }

        ManagedConnection mconn = storeMgr.getConnection(sm.getObjectManager());
        try
        {
            Workbook wb = (Workbook) mconn.getConnection();
            final Sheet sheet = ExcelUtils.getSheetForClass(sm, wb);

            long startTime = System.currentTimeMillis();
            if (NucleusLogger.DATASTORE_RETRIEVE.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_RETRIEVE.debug(LOCALISER.msg("Excel.Fetch.Start", 
                    StringUtils.toJVMIDString(sm.getObject()), sm.getInternalObjectId()));
            }

            int rowNumber = ExcelUtils.getRowNumberForObjectInWorkbook(sm, wb);
            if (rowNumber < 0)
            {
                throw new NucleusObjectNotFoundException("object not found", sm.getObject());
            }
            sm.replaceFields(fieldNumbers, new FetchFieldManager(sm, sheet, rowNumber, 0));

            if (NucleusLogger.DATASTORE_RETRIEVE.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_RETRIEVE.debug(LOCALISER.msg("Excel.ExecutionTime", 
                    (System.currentTimeMillis() - startTime)));
            }
            if (storeMgr.getRuntimeManager() != null)
            {
                storeMgr.getRuntimeManager().incrementFetchCount();
            }

            if (acmd.getVersionMetaData() != null && sm.getTransactionalVersion(sm.getObject()) == null)
            {
                // Object has no version set so update it from this fetch
                long verColNo = -1;
                if (acmd.getVersionMetaData().getFieldName() == null)
                {
                    // Surrogate version
                    verColNo = ExcelUtils.getColumnIndexForFieldOfClass(acmd, -2);
                }
                else
                {
                    // Field-based version
                    verColNo = ExcelUtils.getColumnIndexForFieldOfClass(acmd, 
                        acmd.getAbsolutePositionOfMember(acmd.getVersionMetaData().getFieldName()));
                }

                Row row = sheet.getRow(rowNumber);
                Cell cell = row.getCell((int)verColNo);
                if (acmd.getVersionMetaData().getVersionStrategy() == VersionStrategy.VERSION_NUMBER)
                {
                    long val = (long)cell.getNumericCellValue();
                    sm.setVersion(Long.valueOf(val));
                }
                else if (acmd.getVersionMetaData().getVersionStrategy() == VersionStrategy.DATE_TIME)
                {
                    Date val = cell.getDateCellValue();
                    sm.setVersion(val);
                }
            }
        }
        finally
        {
            mconn.release();
        }
    }

    /**
     * Accessor for the object with the specified identity (if present).
     * Since we don't manage the memory instantiation of objects this just returns null.
     * @param om ObjectManager in use
     * @param id Identity of the object
     * @return The object
     */
    public Object findObject(ObjectManager om, Object id)
    {
        return null;
    }

    /**
     * Method to locate if an object exists in the datastore.
     * Goes through the rows in the worksheet and finds a row with the required identity.
     */
    public void locateObject(StateManager sm)
    {
        final AbstractClassMetaData acmd = sm.getClassMetaData();
        if (acmd.getIdentityType() == IdentityType.APPLICATION || 
            acmd.getIdentityType() == IdentityType.DATASTORE)
        {
            ManagedConnection mconn = storeMgr.getConnection(sm.getObjectManager());
            try
            {
                Workbook wb = (Workbook) mconn.getConnection();
                int rownum = ExcelUtils.getRowNumberForObjectInWorkbook(sm, wb);
                if (rownum >= 0)
                {
                    return;
                }
            }
            finally
            {
                mconn.release();
            }
        }

        throw new NucleusObjectNotFoundException("Object not found",sm.getExternalObjectId(sm.getObject()));
    }
}