/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.excel;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.identity.OID;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.IdentityMetaData;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.metadata.VersionMetaData;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.StringUtils;

/**
 * Class providing convenience methods for handling Excel datastores.
 * Please refer to Apache POI http://poi.apache.org
 */
public class ExcelUtils
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.store.excel.Localisation", ExcelStoreManager.class.getClassLoader());

    /**
     * Convenience method to get the name of the worksheet where we store instances of this class.
     * Takes the metadata extension "sheet" if provided, otherwise returns the "table", else returns 
     * the simple class name
     * @param cmd MetaData for the class
     * @return name of the sheet
     */
    public static String getSheetNameForClass(AbstractClassMetaData cmd)
    {
        String sheetName = cmd.getValueForExtension("sheet");
        if (sheetName == null)
        {
            if (cmd.getTable() != null)
            {
                sheetName = cmd.getTable();
            }
            else
            {
                sheetName = cmd.getName();
            }
        }
        return sheetName;
    }

    /**
     * Convenience method to get the index number where a field of a class is persisted.
     * Uses the extension tag "index" and if not provided uses "column" expecting an integer value.
     * The field number is the absolute number (0 or higher); a value of -1 implies surrogate identity column,
     * and -2 implies surrogate version column
     * @param acmd MetaData for the class
     * @param fieldNumber Absolute field number that we are interested in.
     */
    public static long getColumnIndexForFieldOfClass(AbstractClassMetaData acmd, int fieldNumber)
    {
        if (fieldNumber >= 0)
        {
            // Field of the class
            AbstractMemberMetaData ammd = acmd.getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
            String index = ammd.getValueForExtension("index");
            if (index == null)
            {
                if (ammd.getColumn() != null)
                {
                    String colName = ammd.getColumn();
                    try
                    {
                        return Long.valueOf(colName).longValue();
                    }
                    catch (NumberFormatException nfe)
                    {
                        return (long)fieldNumber;
                    }
                }
                else
                {
                    return (long)fieldNumber;
                }
            }
            else
            {
                try
                {
                    return Long.valueOf(index).longValue();
                }
                catch (NumberFormatException nfe)
                {
                    return (long)fieldNumber;
                }
            }
        }
        else if (fieldNumber == -1)
        {
            // Surrogate datastore identity column
            IdentityMetaData imd = acmd.getIdentityMetaData();
            if (imd != null)
            {
                String index = imd.getValueForExtension("index");
                if (index == null)
                {
                    if (imd.getColumnMetaData() != null && imd.getColumnMetaData().length > 0)
                    {
                        String colName = imd.getColumnMetaData()[0].getName();
                        try
                        {
                            return Long.valueOf(colName).longValue();
                        }
                        catch (NumberFormatException nfe)
                        {
                        }
                    }
                }
                else
                {
                    try
                    {
                        return Long.valueOf(index).longValue();
                    }
                    catch (NumberFormatException nfe)
                    {
                    }
                }
            }
            int fallbackPosition = acmd.getNoOfInheritedManagedMembers() + acmd.getNoOfManagedMembers();
            return fallbackPosition;
        }
        else if (fieldNumber == -2)
        {
            // Surrogate version column
            VersionMetaData vmd = acmd.getVersionMetaData();
            if (vmd != null)
            {
                String index = vmd.getValueForExtension("index");
                if (index == null)
                {
                    if (vmd.getColumnMetaData() != null && vmd.getColumnMetaData().length > 0)
                    {
                        String colName = vmd.getColumnMetaData()[0].getName();
                        try
                        {
                            return Long.valueOf(colName).longValue();
                        }
                        catch (NumberFormatException nfe)
                        {
                        }
                    }
                }
                else
                {
                    try
                    {
                        return Long.valueOf(index).longValue();
                    }
                    catch (NumberFormatException nfe)
                    {
                    }
                }
            }
            int fallbackPosition = acmd.getNoOfInheritedManagedMembers() + acmd.getNoOfManagedMembers() + 1;
            return fallbackPosition;
        }
        else
        {
            throw new NucleusException("Unsupported field number " + fieldNumber);
        }
    }

    /**
     * Convenience method to return the worksheet used for storing the specified object.
     * @param sm StateManager for the object
     * @param wb Workbook
     * @return The Work Sheet
     * @throws NucleusDataStoreException if the work sheet doesn't exist in this workbook
     */
    public static Sheet getSheetForClass(StateManager sm, Workbook wb)
    {
        String sheetName = getSheetNameForClass(sm.getClassMetaData());
        final Sheet sheet = wb.getSheet(sheetName);
        if (sheet == null)
        {
            throw new NucleusDataStoreException(LOCALISER.msg("Excel.SheetNotFoundForWorkbook",
                sheetName, StringUtils.toJVMIDString(sm.getObject())));
        }
        return sheet;
    }

    /**
     * Convenience method to find the row number of an object in the provided workbook.
     * For application-identity does a search for a row with the specified PK field values.
     * For datastore-identity does a search for the row with the datastore column having the specified value
     * @param sm StateManager for the object
     * @param wb Workbook
     * @return The row number (or -1 if not found)
     */
    public static int getRowNumberForObjectInWorkbook(StateManager sm, Workbook wb)
    {
        final AbstractClassMetaData cmd = sm.getClassMetaData();
        if (cmd.getIdentityType() == IdentityType.APPLICATION)
        {
            int[] pkFieldNumbers = cmd.getPKMemberPositions();
            Object[] pkFieldValues = new Object[pkFieldNumbers.length];
            for (int i=0;i<pkFieldNumbers.length;i++)
            {
                pkFieldValues[i] = sm.provideField(pkFieldNumbers[i]);
            }

            String sheetName = getSheetNameForClass(cmd);
            final Sheet sheet = wb.getSheet(sheetName);
            if (sheet != null && sheet.getPhysicalNumberOfRows() > 0)
            {
                for (int i=sheet.getFirstRowNum(); i<sheet.getLastRowNum()+1; i++)
                {
                    Row row = sheet.getRow(i);
                    if (row != null)
                    {
                        boolean pkMatches = true;
                        for (int j=0;j<pkFieldNumbers.length;j++)
                        {
                            AbstractMemberMetaData mmd = cmd.getMetaDataForManagedMemberAtAbsolutePosition(pkFieldNumbers[j]);
                            int colNumber = (int)ExcelUtils.getColumnIndexForFieldOfClass(cmd, pkFieldNumbers[j]);
                            Cell cell = row.getCell(colNumber);
                            if (cell != null)
                            {
                                if (String.class.isAssignableFrom(mmd.getType()) && 
                                        !cell.getRichStringCellValue().getString().equals(pkFieldValues[j]))
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == int.class || mmd.getType() == Integer.class) && 
                                        ((Integer)pkFieldValues[j]).intValue() != (int)cell.getNumericCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == long.class || mmd.getType() == Long.class) && 
                                        ((Long)pkFieldValues[j]).longValue() != (long)cell.getNumericCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == short.class || mmd.getType() == Short.class) && 
                                        ((Short)pkFieldValues[j]).shortValue() != (short)cell.getNumericCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == float.class || mmd.getType() == Float.class) &&
                                        ((Float)pkFieldValues[j]).floatValue() != (float)cell.getNumericCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == double.class || mmd.getType() == Double.class) && 
                                        ((Double)pkFieldValues[j]).doubleValue() != (double)cell.getNumericCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == boolean.class || mmd.getType() == Boolean.class) && 
                                        ((Boolean)pkFieldValues[j]).booleanValue() != cell.getBooleanCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == byte.class || mmd.getType() == Byte.class) && 
                                        ((Byte)pkFieldValues[j]).byteValue() != (byte)cell.getNumericCellValue())
                                {
                                    pkMatches = false;
                                    break;
                                }
                                else if ((mmd.getType() == char.class || mmd.getType() == Character.class) && 
                                        ((Character)pkFieldValues[j]).charValue() != cell.getRichStringCellValue().getString().charAt(0))
                                {
                                    pkMatches = false;
                                    break;
                                }
                            }
                            else
                            {
                                // Row has a null cell so must be inactive row, so ignore it
                                pkMatches = false;
                            }
                        }
                        if (pkMatches)
                        {
                            // Found the object with the correct PK values so return
                            return row.getRowNum();
                        }
                    }
                }
            }
        }
        else if (cmd.getIdentityType() == IdentityType.DATASTORE)
        {
            String sheetName = getSheetNameForClass(cmd);
            final Sheet sheet = wb.getSheet(sheetName);
            int datastoreIdColNumber = (int)ExcelUtils.getColumnIndexForFieldOfClass(cmd, -1);
            Object key = ((OID)sm.getInternalObjectId()).getKeyValue();
            if (sheet != null)
            {
                for (int i=0; i<sheet.getLastRowNum()+1; i++)
                {
                    Row row = sheet.getRow(i);
                    if (row != null)
                    {
                        Cell cell = row.getCell(datastoreIdColNumber);
                        if (cell != null)
                        {
                            if (key instanceof Long)
                            {
                                double cellVal = cell.getNumericCellValue();
                                if ((long)cellVal == ((Long)key).longValue())
                                {
                                    return row.getRowNum();
                                }
                            }
                            else if (key instanceof String)
                            {
                                String cellVal = cell.getRichStringCellValue().getString();
                                if (cellVal.equals((String)key))
                                {
                                    return row.getRowNum();
                                }
                            }
                        }
                    }
                }
            }
        }
        return -1;
    }

    /**
     * Convenience method to find the number of rows in a workbook.
     * This takes into account the fact that it seems to be impossible (with Apache POI 3.0.2)
     * to delete rows from a sheet. Consequently what we do is leave the row but delete
     * all cells. When returning the number of rows this ignores rows that have no cells.
     * @param sm StateManager for the object
     * @param wb Workbook
     * @return Number of (active) rows (or 0 if no active rows)
     */
    public static int getNumberOfRowsInSheetOfWorkbook(StateManager sm, Workbook wb)
    {
        int numRows = 0;

        final AbstractClassMetaData cmd = sm.getClassMetaData();
        if (cmd.getIdentityType() == IdentityType.APPLICATION)
        {
            int[] pkFieldNumbers = cmd.getPKMemberPositions();
            Object[] pkFieldValues = new Object[pkFieldNumbers.length];
            for (int i=0;i<pkFieldNumbers.length;i++)
            {
                pkFieldValues[i] = sm.provideField(pkFieldNumbers[i]);
            }

            String sheetName = getSheetNameForClass(cmd);
            final Sheet sheet = wb.getSheet(sheetName);
            if (sheet != null && sheet.getPhysicalNumberOfRows() > 0)
            {
                for (int i=sheet.getFirstRowNum(); i<sheet.getLastRowNum()+1; i++)
                {
                    Row row = sheet.getRow(i);
                    if (row != null)
                    {
                        for (int j=0;j<pkFieldNumbers.length;j++)
                        {
                            int colNumber = (int)ExcelUtils.getColumnIndexForFieldOfClass(cmd, pkFieldNumbers[j]);
                            Cell cell = row.getCell(colNumber);
                            if (cell != null)
                            {
                                // Valid row. Apache POI would return cell as null if not active
                                numRows++;
                            }
                        }
                    }
                }
            }
        }
        else if (cmd.getIdentityType() == IdentityType.DATASTORE)
        {
            String sheetName = getSheetNameForClass(cmd);
            final Sheet sheet = wb.getSheet(sheetName);
            if (sheet != null && sheet.getPhysicalNumberOfRows() > 0)
            {
                int datastoreIdColNumber = (int)ExcelUtils.getColumnIndexForFieldOfClass(cmd, -1);
                for (int i=sheet.getFirstRowNum(); i<sheet.getLastRowNum()+1; i++)
                {
                    Row rrow = sheet.getRow(i);
                    Cell cell = rrow.getCell(datastoreIdColNumber);
                    if (cell != null)
                    {
                        // Valid row. Apache POI would return cell as null if not active
                        numRows++;
                    }
                }
            }
        }

        return numRows;
    }
}