/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors :
2008 Andy Jefferson - change to use ExcelUtils
 ...
***********************************************************************/
package org.datanucleus.store.excel.fieldmanager;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Map;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.metadata.MetaDataUtils;
import org.datanucleus.metadata.Relation;
import org.datanucleus.sco.SCOUtils;
import org.datanucleus.store.excel.ExcelUtils;
import org.datanucleus.store.fieldmanager.FieldManager;
import org.datanucleus.store.types.ObjectLongConverter;
import org.datanucleus.store.types.ObjectStringConverter;
import org.datanucleus.util.NucleusLogger;

/**
 * FieldManager to handle the retrieval of information from an Excel worksheet row/column into
 * a field of an object.
 */
public class FetchFieldManager implements FieldManager
{
    StateManager sm;
    Sheet sheet;
    int row;
    int col;

    public FetchFieldManager(StateManager sm, Sheet sheet, int row, int col)
    {
        this.sm = sm;
        this.row= row;
        this.col = col;
        this.sheet = sheet;
    }

    public String fetchStringField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(),fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return null;
        }
        return cell.getRichStringCellValue().getString();
    }

    public Object fetchObjectField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return null;
        }

        ObjectManager om = sm.getObjectManager();
        ClassLoaderResolver clr = om.getClassLoaderResolver();
        AbstractMemberMetaData mmd = 
            sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
        int relationType = mmd.getRelationType(clr);
        if (relationType == Relation.NONE)
        {
            if (Date.class.isAssignableFrom(mmd.getType()))
            {
                Date date = cell.getDateCellValue();
                if (date == null)
                {
                    return null;
                }

                Object value = date;
                if (mmd.getType() == java.sql.Date.class)
                {
                    value = new java.sql.Date(date.getTime());
                }
                else if (mmd.getType() == java.sql.Time.class)
                {
                    value = new java.sql.Time(date.getTime());
                }
                else if (mmd.getType() == java.sql.Timestamp.class)
                {
                    value = new java.sql.Timestamp(date.getTime());
                }

                value = sm.wrapSCOField(fieldNumber, value, false, false, true);

                return value;
            }
            else if (Calendar.class.isAssignableFrom(mmd.getType()))
            {
                Date date = cell.getDateCellValue();
                if (date == null)
                {
                    return null;
                }

                Calendar cal = Calendar.getInstance();
                cal.setTime(date);
                Object value = sm.wrapSCOField(fieldNumber, cal, false, false, true);
                return value;
            }
            else if (Number.class.isAssignableFrom(mmd.getType()))
            {
                double val = cell.getNumericCellValue();
                if (Double.class.isAssignableFrom(mmd.getType()))
                {
                    return Double.valueOf(val);
                }
                else if (Float.class.isAssignableFrom(mmd.getType()))
                {
                    return Float.valueOf((float)val);
                }
                else if (Integer.class.isAssignableFrom(mmd.getType()))
                {
                    return Integer.valueOf((int)val);
                }
                else if (Long.class.isAssignableFrom(mmd.getType()))
                {
                    return Long.valueOf((long)val);
                }
                else if (Short.class.isAssignableFrom(mmd.getType()))
                {
                    return Short.valueOf((short)val);
                }
                else if (BigDecimal.class.isAssignableFrom(mmd.getType()))
                {
                    return new BigDecimal(val);
                }
                else if (BigInteger.class.isAssignableFrom(mmd.getType()))
                {
                    return new BigInteger("" + val);
                }
            }
            else if (Enum.class.isAssignableFrom(mmd.getType()))
            {
                String value = cell.getRichStringCellValue().getString();
                if (value != null && value.length() > 0)
                {
                    return Enum.valueOf(mmd.getType(), value);
                }
                else
                {
                    return null;
                }
            }

            boolean useLong = false;
            ColumnMetaData[] colmds = mmd.getColumnMetaData();
            if (colmds != null && colmds.length == 1)
            {
                String jdbc = colmds[0].getJdbcType();
                if (jdbc != null && (jdbc.equalsIgnoreCase("int") || jdbc.equalsIgnoreCase("integer")))
                {
                    useLong = true;
                }
            }

            // See if we can persist it using object converters
            ObjectStringConverter strConv = 
                sm.getObjectManager().getOMFContext().getTypeManager().getStringConverter(mmd.getType());
            ObjectLongConverter longConv =
                sm.getObjectManager().getOMFContext().getTypeManager().getLongConverter(mmd.getType());
            Object value = null;
            if (useLong && longConv != null)
            {
                value = longConv.toObject((long)cell.getNumericCellValue());
            }
            else if (!useLong && strConv != null)
            {
                String cellValue = (cell.getRichStringCellValue() != null ? cell.getRichStringCellValue().getString() : null);
                if (cellValue != null && cellValue.length() > 0)
                {
                    value = strConv.toObject(cell.getRichStringCellValue().getString());
                }
                else
                {
                    return null;
                }
            }
            else if (!useLong && longConv != null)
            {
                value = longConv.toObject((long)cell.getNumericCellValue());
            }
            else
            {
                // Not supported as String so just set to null
                NucleusLogger.PERSISTENCE.warn("Field " + mmd.getFullFieldName() + 
                    " could not be set in the object since it is not persistable to Excel");
                return null;
            }

            // Wrap the field if it is SCO
            value = sm.wrapSCOField(fieldNumber, value, false, false, true);

            return value;
        }
        else if (relationType == Relation.ONE_TO_ONE_BI || relationType == Relation.ONE_TO_ONE_UNI ||
            relationType == Relation.MANY_TO_ONE_BI)
        {
            // Persistable object - retrieve the string form of the identity, and find the object
            String idStr = cell.getRichStringCellValue().getString();
            if (idStr == null)
            {
                return null;
            }

            if (idStr.startsWith("[") && idStr.endsWith("]"))
            {
                idStr = idStr.substring(1, idStr.length()-1);
                AbstractClassMetaData relatedCmd = om.getMetaDataManager().getMetaDataForClass(mmd.getType(), clr);
                return getObjectFromIdString(idStr, relatedCmd);
            }
            else
            {
                return null;
            }
        }
        else if (relationType == Relation.MANY_TO_MANY_BI || relationType == Relation.ONE_TO_MANY_BI ||
            relationType == Relation.ONE_TO_MANY_UNI)
        {
            // Collection/Map/Array
            String cellStr = cell.getRichStringCellValue().getString();
            if (cellStr == null)
            {
                return null;
            }

            if (cellStr.startsWith("[") && cellStr.endsWith("]"))
            {
                cellStr = cellStr.substring(1, cellStr.length()-1);
                String[] components = MetaDataUtils.getInstance().getValuesForCommaSeparatedAttribute(cellStr);
                if (Collection.class.isAssignableFrom(mmd.getType()))
                {
                    Collection<Object> coll;
                    try
                    {
                        Class instanceType = SCOUtils.getContainerInstanceType(mmd.getType(), mmd.getOrderMetaData() != null);
                        coll = (Collection<Object>) instanceType.newInstance();
                    }
                    catch (Exception e)
                    {
                        throw new NucleusDataStoreException(e.getMessage(), e);
                    }

                    if (components != null)
                    {
                        for (int i=0;i<components.length;i++)
                        {
                            AbstractClassMetaData elementCmd = mmd.getCollection().getElementClassMetaData(
                                om.getClassLoaderResolver(), om.getMetaDataManager());
                            Object element = getObjectFromIdString(components[i], elementCmd);
                            coll.add(element);
                        }
                    }
                    return sm.wrapSCOField(fieldNumber, coll, false, false, true);
                }
                else if (Map.class.isAssignableFrom(mmd.getType()))
                {
                    // TODO Implement map retrieval
                }
                else if (mmd.getType().isArray())
                {
                    Object array = null;
                    if (components != null)
                    {
                        array = Array.newInstance(mmd.getType().getComponentType(), components.length);
                        for (int i=0;i<components.length;i++)
                        {
                            AbstractClassMetaData elementCmd = mmd.getCollection().getElementClassMetaData(
                                om.getClassLoaderResolver(), om.getMetaDataManager());
                            Object element = getObjectFromIdString(components[i], elementCmd);
                            Array.set(array, i, element);
                        }
                    }
                    else
                    {
                        array = Array.newInstance(mmd.getType().getComponentType(), 0);
                    }
                    return sm.wrapSCOField(fieldNumber, array, false, false, true);
                }
            }
        }
        throw new NucleusException("Dont currently support retrieval of type " + mmd.getTypeName());
    }

    public boolean fetchBooleanField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return false;
        }
        return cell.getBooleanCellValue();
    }

    public byte fetchByteField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return (byte) cell.getNumericCellValue();
    }

    public char fetchCharField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return cell.getRichStringCellValue().getString().charAt(0);
    }

    public double fetchDoubleField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return cell.getNumericCellValue();
    }

    public float fetchFloatField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return (float) cell.getNumericCellValue();
    }

    public int fetchIntField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return (int) cell.getNumericCellValue();
    }

    public long fetchLongField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return (long) cell.getNumericCellValue();
    }

    public short fetchShortField(int fieldNumber)
    {
        int index = (int)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Row rrow = sheet.getRow(row);
        Cell cell = rrow.getCell(index);
        if (cell == null)
        {
            return 0;
        }
        return (short) cell.getNumericCellValue();
    }

    public void storeStringField(int fieldNumber, String value)
    {
    }

    public void storeShortField(int fieldNumber, short value)
    {
    }

    public void storeObjectField(int fieldNumber, Object value)
    {
    }

    public void storeLongField(int fieldNumber, long value)
    {
    }

    public void storeIntField(int fieldNumber, int value)
    {
    }

    public void storeFloatField(int fieldNumber, float value)
    {
    }

    public void storeDoubleField(int fieldNumber, double value)
    {
    }

    public void storeCharField(int fieldNumber, char value)
    {
    }

    public void storeByteField(int fieldNumber, byte value)
    {
    }

    public void storeBooleanField(int fieldNumber, boolean value)
    {
    }

    /**
     * Convenience method to find an object given a string form of its identity, and the metadata for the
     * class (or a superclass).
     * @param idStr The id string
     * @param cmd Metadata for the class
     * @return The object
     */
    protected Object getObjectFromIdString(String idStr, AbstractClassMetaData cmd)
    {
        ObjectManager om = sm.getObjectManager();
        ClassLoaderResolver clr = om.getClassLoaderResolver();
        Object id = null;
        if (cmd.usesSingleFieldIdentityClass())
        {
            id = om.getApiAdapter().getNewApplicationIdentityObjectId(clr, cmd, idStr);
        }
        else
        {
            Class cls = clr.classForName(cmd.getFullClassName());
            id = om.newObjectId(cls, idStr);
        }
        return om.findObject(id, true, true, null);
    }
}