/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors :
2008 Andy Jefferson - change to use ExcelUtils
2008 Andy Jefferson - mechanism for persisting types as String
 ...
***********************************************************************/
package org.datanucleus.store.excel.fieldmanager;

import java.lang.reflect.Array;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Row;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.metadata.Relation;
import org.datanucleus.store.excel.ExcelUtils;
import org.datanucleus.store.fieldmanager.FieldManager;
import org.datanucleus.store.types.ObjectLongConverter;
import org.datanucleus.store.types.ObjectStringConverter;
import org.datanucleus.util.NucleusLogger;

/**
 * FieldManager to handle the store information into an Excel worksheet row using an object.
 */
public class StoreFieldManager implements FieldManager
{
    private final StateManager sm;

    private final Row row;

    public StoreFieldManager(StateManager sm, Row row)
    {
        this.sm = sm;
        this.row = row;
    }

    public void storeStringField(int fieldNumber, String value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        if (value == null)
        {
            row.removeCell(cell);
        }
        else
        {
            CreationHelper createHelper = row.getSheet().getWorkbook().getCreationHelper();
            cell.setCellValue(createHelper.createRichTextString(value));
        }
    }

    public void storeObjectField(int fieldNumber, Object value)
    {
        AbstractMemberMetaData mmd = 
            sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
        int relationType = mmd.getRelationType(sm.getObjectManager().getClassLoaderResolver());

        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        if (value == null)
        {
            if (Number.class.isAssignableFrom(mmd.getType()))
            {
                cell.setCellValue(0.0);
            }
            else if (Date.class.isAssignableFrom(mmd.getType()))
            {
                // Do nothing
            }
            else if (Calendar.class.isAssignableFrom(mmd.getType()))
            {
                // Do nothing
            }
            else
            {
                // Do nothing
            }
        }
        else if (relationType == Relation.NONE)
        {
            if (value instanceof Date)
            {
                cell.setCellValue((Date)value);
            }
            else if (value instanceof Calendar)
            {
                cell.setCellValue((Calendar)value);
            }
            else if (value instanceof Enum)
            {
                // TODO Allow persistence as numeric
                CreationHelper createHelper = row.getSheet().getWorkbook().getCreationHelper();
                cell.setCellValue(createHelper.createRichTextString(((Enum)value).name()));
            }
            else
            {
                // Try to persist using converters
                boolean useLong = false;
                ColumnMetaData[] colmds = mmd.getColumnMetaData();
                if (colmds != null && colmds.length == 1)
                {
                    String jdbc = colmds[0].getJdbcType();
                    if (jdbc != null && (jdbc.equalsIgnoreCase("int") || jdbc.equalsIgnoreCase("integer")))
                    {
                        useLong = true;
                    }
                }

                // See if we can persist it using object converters
                ObjectStringConverter strConv = 
                    sm.getObjectManager().getOMFContext().getTypeManager().getStringConverter(value.getClass());
                ObjectLongConverter longConv =
                    sm.getObjectManager().getOMFContext().getTypeManager().getLongConverter(value.getClass());
                if (useLong)
                {
                    if (longConv != null)
                    {
                        cell.setCellValue(longConv.toLong(value));
                        return;
                    }
                }
                else
                {
                    if (strConv != null)
                    {
                        CreationHelper createHelper = row.getSheet().getWorkbook().getCreationHelper();
                        cell.setCellValue(createHelper.createRichTextString(strConv.toString(value)));
                        return;
                    }
                    else if (longConv != null)
                    {
                        cell.setCellValue(longConv.toLong(value));
                        return;
                    }
                }

                NucleusLogger.PERSISTENCE.warn(
                    "DataNucleus doesnt currently support persistence of field " + mmd.getFullFieldName() + 
                    " type=" + value.getClass().getName() + " - ignoring");
            }
        }
        else if (relationType == Relation.ONE_TO_ONE_BI || relationType == Relation.ONE_TO_ONE_UNI ||
                relationType == Relation.MANY_TO_ONE_BI)
        {
            // Persistable object - persist the related object and store the identity in the cell
            Object valuePC = sm.getObjectManager().persistObjectInternal(value, null, sm, fieldNumber, -1);
            Object valueId = sm.getObjectManager().getApiAdapter().getIdForObject(valuePC);
            CreationHelper createHelper = row.getSheet().getWorkbook().getCreationHelper();
            cell.setCellValue(createHelper.createRichTextString("[" + valueId.toString() + "]"));
        }
        else if (relationType == Relation.MANY_TO_MANY_BI || relationType == Relation.ONE_TO_MANY_BI ||
                relationType == Relation.ONE_TO_MANY_UNI)
        {
            // Collection/Map/Array
            if (mmd.hasCollection())
            {
                StringBuffer cellValue = new StringBuffer("[");
                Collection coll = (Collection)value;
                Iterator collIter = coll.iterator();
                while (collIter.hasNext())
                {
                    Object element = collIter.next();
                    Object elementPC = sm.getObjectManager().persistObjectInternal(element, null, sm, fieldNumber, -1);
                    Object elementID = sm.getObjectManager().getApiAdapter().getIdForObject(elementPC);
                    cellValue.append(elementID.toString());
                    if (collIter.hasNext())
                    {
                        cellValue.append(",");
                    }
                }
                cellValue.append("]");
                CreationHelper createHelper = row.getSheet().getWorkbook().getCreationHelper();
                cell.setCellValue(createHelper.createRichTextString(cellValue.toString()));
            }
            else if (mmd.hasMap())
            {
                // TODO Implement map persistence - what to do if key or value is non-PC
                throw new NucleusException("Dont currently support persistence of map types to Excel");
            }
            else if (mmd.hasArray())
            {
                StringBuffer cellValue = new StringBuffer("[");
                for (int i=0;i<Array.getLength(value);i++)
                {
                    Object element = Array.get(value, i);
                    Object elementPC = sm.getObjectManager().persistObjectInternal(element, null, sm, fieldNumber, -1);
                    Object elementID = sm.getObjectManager().getApiAdapter().getIdForObject(elementPC);
                    cellValue.append(elementID.toString());
                    if (i < (Array.getLength(value)-1))
                    {
                        cellValue.append(",");
                    }
                }
                cellValue.append("]");
                CreationHelper createHelper = row.getSheet().getWorkbook().getCreationHelper();
                cell.setCellValue(createHelper.createRichTextString(cellValue.toString()));
            }
        }
    }

    public void storeBooleanField(int fieldNumber, boolean value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeByteField(int fieldNumber, byte value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeCharField(int fieldNumber, char value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeDoubleField(int fieldNumber, double value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeFloatField(int fieldNumber, float value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeIntField(int fieldNumber, int value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeLongField(int fieldNumber, long value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public void storeShortField(int fieldNumber, short value)
    {
        short index = (short)ExcelUtils.getColumnIndexForFieldOfClass(sm.getClassMetaData(), fieldNumber);
        Cell cell = row.createCell((int)index);
        cell.setCellValue(value);
    }

    public String fetchStringField(int fieldNumber)
    {
        return null;
    }

    public short fetchShortField(int fieldNumber)
    {
        return 0;
    }

    public Object fetchObjectField(int fieldNumber)
    {
        return null;
    }

    public long fetchLongField(int fieldNumber)
    {
        return 0;
    }

    public int fetchIntField(int fieldNumber)
    {
        return 0;
    }

    public float fetchFloatField(int fieldNumber)
    {
        return 0;
    }

    public double fetchDoubleField(int fieldNumber)
    {
        return 0;
    }

    public char fetchCharField(int fieldNumber)
    {
        return 0;
    }

    public byte fetchByteField(int fieldNumber)
    {
        return 0;
    }

    public boolean fetchBooleanField(int fieldNumber)
    {
        return false;
    }
}