package org.dbtools.android.room.sqliteorg

import android.arch.persistence.db.SupportSQLiteDatabase
import android.arch.persistence.db.SupportSQLiteOpenHelper
import android.content.Context
import org.dbtools.android.room.sqliteorg.SqliteOrgSQLiteOpenHelperFactory.Companion.DEFAULT_SQLITEX_LIBRARY_LOADER
import org.sqlite.database.sqlite.SQLiteDatabase
import org.sqlite.database.sqlite.SQLiteOpenHelper
import java.io.File

open class SqliteOrgSQLiteOpenHelper(
        context: Context,
        path: String,
        name: String?,
        callback: SupportSQLiteOpenHelper.Callback,
        password: String,
        libraryLoader: () -> Unit = DEFAULT_SQLITEX_LIBRARY_LOADER
) : SupportSQLiteOpenHelper {

    private val delegate: OpenHelper

    init {
        val databaseFilepath = if (path.isEmpty()) {
            context.getDatabasePath(name).absolutePath
        } else {
            path + "/" + name
        }

        val databaseFile = File(databaseFilepath)
        databaseFile.parentFile.mkdirs()

        delegate = SqliteOrgSQLiteOpenHelper.OpenHelper(context, libraryLoader, databaseFilepath, callback, password)
    }

    override fun getDatabaseName(): String? {
        return delegate.databaseName
    }

    override fun setWriteAheadLoggingEnabled(enabled: Boolean) {
        delegate.setWriteAheadLoggingEnabled(enabled)
    }

    override fun getWritableDatabase(): SupportSQLiteDatabase {
        return delegate.getWritableSupportDatabase()
    }

    override fun getReadableDatabase(): SupportSQLiteDatabase {
        return delegate.getReadableSupportDatabase()
    }

    override fun close() {
        delegate.close()
    }

    class OpenHelper(
            context: Context,
            libraryLoader: () -> Unit = {},
            private val name: String?,
            private val callback: SupportSQLiteOpenHelper.Callback,
            private val password: String) : SQLiteOpenHelper(context, name, null, callback.version
    ) {

        var wrappedDb: SqliteOrgDatabase? = null

        init {
            libraryLoader()
        }

        override fun onCreate(sqLiteDatabase: SQLiteDatabase) {
            wrappedDb = SqliteOrgDatabase(sqLiteDatabase)
            callback.onCreate(wrappedDb)
        }

        override fun onUpgrade(sqLiteDatabase: SQLiteDatabase, oldVersion: Int, newVersion: Int) {
            callback.onUpgrade(getWrappedDb(sqLiteDatabase), oldVersion, newVersion)
        }

        override fun onConfigure(db: SQLiteDatabase) {
            callback.onConfigure(getWrappedDb(db))
        }

        override fun onDowngrade(db: SQLiteDatabase, oldVersion: Int, newVersion: Int) {
            callback.onDowngrade(getWrappedDb(db), oldVersion, newVersion)
        }

        override fun onOpen(db: SQLiteDatabase) {
            callback.onOpen(getWrappedDb(db))
        }

        fun getWritableSupportDatabase(): SupportSQLiteDatabase {
            val db = super.getWritableDatabase()
            return getWrappedDb(db)
        }

        fun getReadableSupportDatabase(): SupportSQLiteDatabase {
            val db = super.getReadableDatabase()
            return getWrappedDb(db)
        }

        fun getWrappedDb(sqLiteDatabase: SQLiteDatabase): SqliteOrgDatabase {
            if (wrappedDb == null) {
                val database = SqliteOrgDatabase(sqLiteDatabase)
                if (password.isNotBlank()) {
                    database.execSQL("PRAGMA key = '$password'")
                }

                wrappedDb = database
            }

            return wrappedDb!!
        }

        @Synchronized
        override fun close() {
            super.close()
            wrappedDb = null
        }

        override fun getDatabaseName(): String? {
            return name
        }
    }
}