/*******************************************************************************
 * Copyright 2015 Efe Kahraman
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/

package org.dynamise.shared.annotation;

import org.dynamise.shared.bundle.BundleType;

import java.lang.annotation.*;

/**
 * Indicates that annotated class is a <i>bundle</i>. Every <i>bundle</i> must be bound to a <i>service</i>.
 * <p>This annotation represents a general form of bundle. Instead of this, specific bundle annotations can be
 * used as well:</p>
 * <ul>
 *     <li>{@link BusinessLogic Business Logic}</li>
 *     <li>{@link Security Security}</li>
 *     <li>{@link Lifecycle Lifecycle}</li>
 * </ul>
 * <p>Each bundle is identified by its name, type and service. A bundle may have several versions, whereas
 * the latest version is always favored during deployment.</p>
 *
 * <p>Bundles may be started and stopped by Dynamise server under certain circumstances. When a bundle is
 * started, <code>&#64;Create</code> annotated method is called if present. On the other hand,
 * <code>&#64;Destroy</code> annotated method is called before being stopped in the same fashion.</p>
 *
 * <p><code>&#64;Bundle</code> can not be applied to <code>abstract</code> or <code>interface</code> types. A Bundle
 * class must provide a <code>public</code> default constructor.</p>
 *
 * <p>Builtin properties infrastructure is provided for bundles.
 * See {@link org.dynamise.shared.bundle.BundleProperties BundleProperties} for details.</p>
 *
 * @author Efe Kahraman
 * @since 1.0.0
 * @see Service
 * @see Create
 * @see Destroy
 */

@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface Bundle {

	/**
	 * Name of the bundle. Name can include letters, numbers or underscore.
	 * @return bundle name
	 */
	String value();

	/**
	 * Version of the bundle.
	 * @return bundle version
	 */
	int version() default 0;

	/**
	 * Type of the bundle.
	 * @return bundle type
	 */
	BundleType type() default BundleType.BUSINESS_LOGIC;
}
