/*******************************************************************************
 * Copyright 2015 Efe Kahraman
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/

package org.dynamise.shared.annotation;

import org.dynamise.shared.bundle.BundleType;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Indicates that annotated class is a Business Logic bundle. Business Logic bundles are invoked due to service
 * requests. Once invoked, bundle can process, discard, or even reject the request.
 *
 * <p>A Business Logic bundle registers itself to incoming requests via one or more <code>&#64;Invoke</code>
 * annotated methods.</p>
 *
 * <p>Usually, services contain at least one Business Logic bundles. It's a recommended practice to
 * design Business Logic bundles according to the single responsibility principle.</p>
 *
 * <p><code>&#64;BusinessLogic</code> can not be applied to <code>abstract</code> or <code>interface</code> types.
 * A Business Logic bundle class must provide a <code>public</code> default constructor.</p>
 *
 * @author Efe Kahraman
 * @since 1.0.0
 * @see Bundle
 * @see org.dynamise.shared.bundle.annotation.Invoke
 */

@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface BusinessLogic {

	/**
	 * Name of the bundle. Name can include letters, numbers or underscore.
	 * @return bundle name
	 */
	String value();

	/**
	 * Version of the bundle.
	 * @return bundle version
	 */
	int version() default 0;

}
