/*******************************************************************************
 * Copyright 2015 Efe Kahraman
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/

package org.dynamise.shared.bundle.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.dynamise.shared.bundle.MethodType;

/**
 * Indicates that annotated method is an entry point for incoming service requests. Business logic bundles may have one
 * or more <code>&#64;Invoke</code> annotated methods to register and process different requests.
 * <p>Incoming service request consists of two parts:</p>
 * <ul>
 *     <li>Method</li>
 *     <li>Entity</li>
 * </ul>
 * <p><code>&#64;Invoke</code> can be used to register for specific methods and entities or any of them as well.</p>
 *
 * <p>If a method is annotated with <code>&#64;Invoke</code> it may return a value as result by using {@link Response
 * &#64;Response} annotation in method return. Besides, it may take following parameters in method
 * signature:</p>
 * <ul>
 *     <li>Request body (Should be annotated with {@link Request &#64;Request})</li>
 *     <li>{@link org.dynamise.shared.context.RequestContext Request context}</li>
 *     <li>{@link org.dynamise.shared.context.ServiceContext Service context}</li>
 *     <li>{@link org.dynamise.shared.context.SecurityContext Security context}</li>
 * </ul>
 *
 * <p>If <code>&#64;Invoke</code> annotated method throws {@link org.dynamise.shared.bundle.BundleException
 * BundleException}, service executions stops immediately and client receives an appropriate error.</p>
 *
 * @author Efe Kahraman
 * @since 1.0.0
 * @see org.dynamise.shared.annotation.BusinessLogic
 * @see MethodType
 * @see Request
 * @see Response
 */

@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface Invoke {

	/**
	 * Constant representing wildcard value for Entity.
	 */
	String ENTITY_DEFAULT = "";

	/**
	 * Method type of incoming service request
	 * @return method type
	 */
	MethodType value() default MethodType.ALL;

	/**
	 * Entity parameter of incoming service request
	 * @return entity
	 */
	String entity() default ENTITY_DEFAULT;

}
