/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.ditto.internal.utils.persistence.mongo.indices;

import akka.Done;
import akka.NotUsed;
import akka.japi.function.Function;
import akka.stream.Graph;
import akka.stream.Materializer;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import com.mongodb.reactivestreams.client.MongoDatabase;
import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;
import javax.annotation.concurrent.Immutable;
import org.eclipse.ditto.internal.utils.persistence.mongo.indices.Index;
import org.eclipse.ditto.internal.utils.persistence.mongo.indices.IndexOperations;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Immutable
public final class IndexInitializer {
    private static final Logger LOGGER = LoggerFactory.getLogger(IndexInitializer.class);
    private final Materializer materializer;
    private final IndexOperations indexOperations;

    private IndexInitializer(MongoDatabase db, Materializer materializer) {
        this.materializer = materializer;
        this.indexOperations = IndexOperations.of(db);
    }

    public static IndexInitializer of(MongoDatabase db, Materializer materializer) {
        Objects.requireNonNull(db);
        Objects.requireNonNull(materializer);
        return new IndexInitializer(db, materializer);
    }

    public CompletionStage<Void> initialize(String collectionName, List<Index> indices) {
        Objects.requireNonNull(collectionName);
        Objects.requireNonNull(indices);
        LOGGER.info("Starting index-initialization with defined indices: {}", indices);
        return this.createNonExistingIndices(collectionName, indices).thenCompose(done -> this.dropUndefinedIndices(collectionName, indices)).thenApply(unused -> {
            LOGGER.info("Index-Initialization was successful.");
            return null;
        });
    }

    public CompletionStage<Done> createNonExistingIndices(String collectionName, List<Index> indices) {
        if (indices.isEmpty()) {
            LOGGER.warn("No indices are defined, thus no indices are created.");
            return CompletableFuture.completedFuture(Done.getInstance());
        }
        return (CompletionStage)this.indexOperations.getIndicesExceptDefaultIndex(collectionName).flatMapConcat((Function & Serializable)existingIndices -> {
            LOGGER.info("Create non-existing indices: Existing indices are: {}", existingIndices);
            List<Index> indicesToCreate = IndexInitializer.excludeIndices(indices, existingIndices);
            LOGGER.info("Indices to create are: {}", indicesToCreate);
            return this.createIndices(collectionName, indicesToCreate);
        }).runWith((Graph)Sink.ignore(), this.materializer);
    }

    private Source<Done, NotUsed> createIndices(String collectionName, List<Index> indices) {
        if (indices.isEmpty()) {
            return Source.empty();
        }
        return Source.from(indices).flatMapConcat((Function & Serializable)index -> this.createIndex(collectionName, (Index)index));
    }

    private Source<Done, NotUsed> createIndex(String collectionName, Index index) {
        LOGGER.info("Creating index: {}", (Object)index);
        return this.indexOperations.createIndex(collectionName, index);
    }

    private CompletionStage<Done> dropUndefinedIndices(String collectionName, List<Index> definedIndices) {
        return (CompletionStage)this.getIndicesExceptDefaultIndex(collectionName).flatMapConcat((Function & Serializable)existingIndices -> {
            LOGGER.info("Drop undefined indices - Existing indices are: {}", existingIndices);
            List<String> indicesToDrop = IndexInitializer.getUndefinedIndexNames(existingIndices, definedIndices);
            LOGGER.info("Dropping undefined indices: {}", indicesToDrop);
            return this.dropIndices(collectionName, indicesToDrop);
        }).runWith((Graph)Sink.ignore(), this.materializer);
    }

    private static List<String> getUndefinedIndexNames(Collection<Index> allIndices, Collection<Index> definedIndices) {
        return IndexInitializer.excludeIndices(allIndices, definedIndices).stream().map(Index::getName).collect(Collectors.toList());
    }

    private Source<Done, NotUsed> dropIndices(String collectionName, List<String> indices) {
        if (indices.isEmpty()) {
            return Source.empty();
        }
        return Source.from(indices).flatMapConcat((Function & Serializable)index -> this.dropIndex(collectionName, (String)index));
    }

    private Source<Done, NotUsed> dropIndex(String collectionName, String indexName) {
        LOGGER.info("Dropping index: {}", (Object)indexName);
        return this.indexOperations.dropIndex(collectionName, indexName);
    }

    private Source<List<Index>, NotUsed> getIndicesExceptDefaultIndex(String collectionName) {
        return this.indexOperations.getIndicesExceptDefaultIndex(collectionName);
    }

    private static Set<String> extractIndexNames(Collection<Index> indices) {
        return indices.stream().map(Index::getName).collect(Collectors.toSet());
    }

    private static List<Index> excludeIndices(Collection<Index> allIndices, Collection<Index> indicesToExclude) {
        Set<String> excludedIndexNames = IndexInitializer.extractIndexNames(indicesToExclude);
        return allIndices.stream().filter(indexModel -> !excludedIndexNames.contains(indexModel.getName())).collect(Collectors.toList());
    }
}

