/*
 * Copyright (c) 2024 Robert Bosch Manufacturing Solutions GmbH
 *
 * See the AUTHORS file(s) distributed with this work for additional
 * information regarding authorship.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * SPDX-License-Identifier: MPL-2.0
 */

package org.eclipse.esmf.metamodel;

import java.util.Arrays;
import java.util.Optional;

import org.eclipse.esmf.aspectmodel.urn.AspectModelUrn;
import org.eclipse.esmf.metamodel.QuantityKind;
import org.eclipse.esmf.metamodel.visitor.AspectVisitor;
import org.eclipse.esmf.samm.KnownVersion;

/**
 * Enumeration of Quantity Kinds as defined in <a href="http://tfig.unece.org/contents/recommendation-20.htm">Recommendation 20</a>
 * by the <a href="http://www.unece.org/info/ece-homepage.html">UNECE</a> (United Nations Economic Commission for Europe).
 */
public enum QuantityKinds implements QuantityKind {
SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME( "specificHeatCapacityAtConstantVolume", "specific heat capacity at constant volume" ),
TOTAL_CROSS_SECTION( "totalCrossSection", "total cross-section" ),
FIRST_RADIATION_CONSTANT( "firstRadiationConstant", "first radiation constant" ),
MOMENT_OF_A_COUPLE( "momentOfACouple", "moment of a couple" ),
AVOGADRO_CONSTANT( "avogadroConstant", "Avogadro constant" ),
MIGRATION_LENGTH( "migrationLength", "migration length" ),
PHOTON_FLUX( "photonFlux", "photon flux" ),
TENSION( "tension", "tension" ),
PRESSURE( "pressure", "pressure" ),
MASSIEU_FUNCTION( "massieuFunction", "massieu function" ),
VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES( "volumicNumberOfMoleculesOrParticles", "volumic number of molecules (or particles)" ),
LUMINOUS_EFFICIENCY_AT_A_SPECIFIED_WAVELENGTH( "luminousEfficiencyAtASpecifiedWavelength", "luminous efficiency at a specified wavelength" ),
NUCLEAR_MAGNETON_ORNUCLEUS( "nuclearMagnetonOrnucleus", "nuclear magneton ornucleus" ),
STANTON_NUMBER_FOR_MASS_TRANSFER( "stantonNumberForMassTransfer", "Stanton number for mass transfer" ),
CHEMICAL_POTENTIAL( "chemicalPotential", "chemical potential" ),
THERMAL_CONDUCTANCE( "thermalConductance", "thermal conductance" ),
NEUTRONFLUX_DENSITY( "neutronfluxDensity", "neutronflux density" ),
TORSIONAL_STIFFNESS( "torsionalStiffness", "torsional stiffness" ),
ENERGY_IMPARTED( "energyImparted", "energy imparted" ),
ACTIVITY_CONCENTRATION( "activityConcentration", "activity concentration" ),
NUSSELT_NUMBER( "nusseltNumber", "Nusselt number" ),
GRAVITATIONAL_CONSTANT( "gravitationalConstant", "gravitational constant" ),
ACTIVE_POWER( "activePower", "active power" ),
LONDON_PENETRATION_DEPTH( "londonPenetrationDepth", "London penetration depth" ),
CHARGE_DENSITY( "chargeDensity", "charge density" ),
MASSIC_ENERGY( "massicEnergy", "massic energy" ),
REFLECTANCE( "reflectance", "reflectance" ),
INFORMATION_ENTROPY( "informationEntropy", "Information entropy" ),
LEWIS_NUMBER( "lewisNumber", "Lewis number" ),
ELECTRIC_POLARIZABILITY_OF_A_MOLECULE( "electricPolarizabilityOfAMolecule", "electric polarizability of a molecule" ),
FRICTION_FACTOR( "frictionFactor", "friction factor" ),
AREIC_MASS( "areicMass", "areic mass" ),
MAGNETIC_VECTOR_POTENTIAL( "magneticVectorPotential", "magnetic vector potential" ),
PELTIER_COEFFICIENT_FOR_SUBSTANCES_A_AND_B( "peltierCoefficientForSubstancesAAndB", "Peltier coefficient for substances a and b" ),
LETHARGY( "lethargy", "lethargy" ),
SELF_INDUCTANCE( "selfInductance", "self inductance" ),
PHOTON_RADIANCE( "photonRadiance", "photon radiance" ),
ALFVEN_NUMBER( "alfvenNumber", "Alfven number" ),
PHASE_COEFFICIENT( "phaseCoefficient", "phase coefficient" ),
VISCOSITY_DYNAMIC_VISCOSITY( "viscosityDynamicViscosity", "viscosity (dynamic viscosity)" ),
PERIOD( "period", "period" ),
MAGNETIC_MOMENT_OF_PARTICLE( "magneticMomentOfParticle", "magnetic moment of particle" ),
PULSATANCE( "pulsatance", "pulsatance" ),
ELECTRIC_SUSCEPTIBILITY( "electricSusceptibility", "electric susceptibility" ),
LATTICE_PLANE_SPACING( "latticePlaneSpacing", "lattice plane spacing" ),
DISSIPATION_FACTOR( "dissipationFactor", "dissipation factor" ),
DIMENSIONLESS( "dimensionless", "dimensionless" ),
GRASHOF_NUMBER( "grashofNumber", "Grashof number" ),
SPECTRAL_REFLECTANCE( "spectralReflectance", "spectral reflectance" ),
SPECTRAL_EMISSIVITY( "spectralEmissivity", "spectral emissivity" ),
MOLALITY_OF_SOLUTE_B( "molalityOfSoluteB", "molality of solute B" ),
RADIANT_ENERGY_FLUENCE( "radiantEnergyFluence", "radiant energy fluence" ),
CARRIER_LIFE_TIME( "carrierLifeTime", "carrier life time" ),
POROSITY( "porosity", "porosity" ),
ACOUSTIC_IMPEDANCE( "acousticImpedance", "acoustic impedance" ),
BINDING_FRACTION( "bindingFraction", "binding fraction" ),
DIAMETER( "diameter", "diameter" ),
REACTIVE_POWER( "reactivePower", "reactive power" ),
RELATIVE_MASS_EXCESS( "relativeMassExcess", "relative mass excess" ),
MODULUS_OF_COMPRESSION( "modulusOfCompression", "modulus of compression" ),
MACROSCOPIC_TOTAL_CROSS_SECTION( "macroscopicTotalCrossSection", "macroscopic total cross-section" ),
MOBILITY_RATIO( "mobilityRatio", "mobility ratio" ),
THERMAL_DIFFUSION_FACTOR( "thermalDiffusionFactor", "thermal diffusion factor" ),
RELAXATION_TIME( "relaxationTime", "relaxation time" ),
ATTENUATION_COEFFICIENT( "attenuationCoefficient", "attenuation coefficient" ),
BOHR_RADIUS( "bohrRadius", "Bohr radius" ),
SECOND_AXIAL_MOMENT_OF_AREA( "secondAxialMomentOfArea", "second axial moment of area" ),
FARADAY_CONSTANT( "faradayConstant", "Faraday constant" ),
SOUND_POWER( "soundPower", "sound power" ),
CURVATURE( "curvature", "curvature" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION( "standardAbsoluteActivityOfSolventAEspeciallyInADiluteSolution", "standard absolute activity of solvent A (especially in a dilute solution)" ),
MAGNETIC_QUANTUM_NUMBER( "magneticQuantumNumber", "magnetic quantum number" ),
ENERGY_FLUENCE( "energyFluence", "energy fluence" ),
ELECTRIC_DIPOLE_MOMENT_OF_MOLECULE( "electricDipoleMomentOfMolecule", "electric dipole moment of molecule" ),
LOUDNESS( "loudness", "loudness" ),
PLANCK_FUNCTION( "planckFunction", "planck function" ),
COWLING_NUMBER( "cowlingNumber", "Cowling number" ),
MODULUS_OF_ADMITTANCE( "modulusOfAdmittance", "modulus of admittance" ),
COMPTON_WAVELENGTH( "comptonWavelength", "Compton wavelength" ),
ELECTRIC_POTENTIAL( "electricPotential", "electric potential" ),
DIFFUSION_LENGTH( "diffusionLength", "diffusion length" ),
CURRENT_LINKAGE( "currentLinkage", "current linkage" ),
RADIANT_ENERGY_DENSITY( "radiantEnergyDensity", "radiant energy density" ),
RADIUS_OF_CURVATURE( "radiusOfCurvature", "radius of curvature" ),
SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE( "specificHeatCapacityAtConstantPressure", "specific heat capacity at constant pressure" ),
IONIC_STRENGTH( "ionicStrength", "ionic strength" ),
NUCLIDIC_MASS( "nuclidicMass", "nuclidic mass" ),
MEAN_LIFE( "meanLife", "mean life" ),
MAGNETIC_POLARIZATION( "magneticPolarization", "magnetic polarization" ),
ACCELERATION( "acceleration", "acceleration" ),
SECOND_MOMENT_OF_AREA( "secondMomentOfArea", "second moment of area" ),
VOLUME_OR_BULK_STRAIN( "volumeOrBulkStrain", "volume or bulk strain" ),
RELATIVE_PERMEABILITY( "relativePermeability", "relative permeability" ),
SLOWING_DOWN_AREA( "slowingDownArea", "slowing down area" ),
VOLUMIC_CROSS_SECTION( "volumicCrossSection", "volumic cross-section" ),
POTENTIAL_ENERGY( "potentialEnergy", "potential energy" ),
REST_MASS_OF_PROTON( "restMassOfProton", "(rest) mass of proton" ),
LINEAR_EXPANSION_COEFFICIENT( "linearExpansionCoefficient", "linear expansion coefficient" ),
POISSON_NUMBER( "poissonNumber", "poisson number" ),
MOLECULAR_PARTITION_FUNCTION( "molecularPartitionFunction", "molecular partition function" ),
POWER_FOR_DIRECT_CURRENT( "powerForDirectCurrent", "power (for direct current)" ),
SEEBECK_COEFFICIENT_FOR_SUBSTANCES_A_AND_B( "seebeckCoefficientForSubstancesAAndB", "Seebeck coefficient for substances a and b" ),
COEFFICIENT_OF_THERMAL_INSULATION( "coefficientOfThermalInsulation", "coefficient of thermal insulation" ),
MOLAR_HEAT_CAPACITY( "molarHeatCapacity", "molar heat capacity" ),
SPECTRAL_ABSORPTION_FACTOR( "spectralAbsorptionFactor", "spectral absorption factor" ),
GAP_ENERGY( "gapEnergy", "gap energy" ),
PHOTON_EXPOSURE( "photonExposure", "photon exposure" ),
IRRADIANCE( "irradiance", "irradiance" ),
CHARACTERISTIC_IMPEDANCE_OF_A_MEDIUM( "characteristicImpedanceOfAMedium", "characteristic impedance of a medium" ),
RELATIVE_PRESSURE_COEFFICIENT( "relativePressureCoefficient", "relative pressure coefficient" ),
LEVEL_OF_A_POWER_QUANTITY( "levelOfAPowerQuantity", "level of a power quantity" ),
SPECIFIC_VOLUME( "specificVolume", "specific volume" ),
INTERNAL_CONVERSION_FACTOR( "internalConversionFactor", "internal conversion factor" ),
ANGULAR_WAVE_NUMBER( "angularWaveNumber", "angular wave number" ),
SURFACE_TENSION( "surfaceTension", "surface tension" ),
COEFFICIENT_OF_FRICTION( "coefficientOfFriction", "coefficient of friction" ),
VELOCITY_SPEED_ON_PROPAGATION_OF_ELECTROMAGNETIC_WAVES_IN_VACUO( "velocitySpeedOnPropagationOfElectromagneticWavesInVacuo", "velocity (speed) on propagation of electromagnetic waves in vacuo" ),
LEAKAGE_COEFFICIENT( "leakageCoefficient", "leakage coefficient" ),
CURRENT_DENSITY_OF_PARTICLES( "currentDensityOfParticles", "current density of particles" ),
MUTUAL_INDUCTANCE( "mutualInductance", "mutual inductance" ),
PECLET_NUMBER_FOR_MASS_TRANSFER( "pecletNumberForMassTransfer", "Peclet number for mass transfer" ),
MASS_ENERGY_TRANSFER_COEFFICIENT( "massEnergyTransferCoefficient", "mass energy transfer coefficient" ),
RELATIVE_MOLECULAR_MASS( "relativeMolecularMass", "relative molecular mass" ),
SPECTRAL_ANGULAR_CROSS_SECTION( "spectralAngularCrossSection", "spectral angular cross-section" ),
NUMBER_OF_MOLECULES_OR_OTHER_ELEMENTARY_ENTITIES( "numberOfMoleculesOrOtherElementaryEntities", "number of molecules or other elementary entities" ),
SPECTRAL_CROSS_SECTION( "spectralCrossSection", "spectral cross-section" ),
ACTIVITY_OF_SOLVENT_A( "activityOfSolventA", "activity of solvent A" ),
MEAN_FREE_PATH( "meanFreePath", "mean free path" ),
PERMITTIVITY( "permittivity", "permittivity" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_LIQUID_OR_A_SOLID_MIXTURE( "standardAbsoluteActivityOfBInALiquidOrASolidMixture", "standard absolute activity of B (in a liquid or a solid mixture)" ),
ANGULAR_MOMENTUM( "angularMomentum", "angular momentum" ),
PERMEANCE( "permeance", "permeance" ),
PHOTON_INTENSITY( "photonIntensity", "photon intensity" ),
GRUENEISEN_PARAMETER( "grueneisenParameter", "Grüneisen parameter" ),
MOLAR_ENTROPY( "molarEntropy", "molar entropy" ),
VERGENCE( "vergence", "vergence" ),
MAGNETOMOTIVE_FORCE( "magnetomotiveForce", "magnetomotive force" ),
REACTANCE( "reactance", "reactance" ),
REFRACTIVE_INDEX( "refractiveIndex", "refractive index" ),
SPECTRAL_CONCENTRATION_OF_VIBRATIONAL_MODES_IN_TERMS_OF_ANGULAR_FREQUENCY( "spectralConcentrationOfVibrationalModesInTermsOfAngularFrequency", "spectral concentration of vibrational modes (in terms of angular frequency)" ),
DENSITY_OF_STATES( "densityOfStates", "density of states" ),
MECHANICAL_IMPEDANCE( "mechanicalImpedance", "mechanical impedance" ),
APPARENT_POWER( "apparentPower", "apparent power" ),
DEBYE_ANGULAR_REPETENCY( "debyeAngularRepetency", "Debye angular repetency" ),
REST_MASS_OF_NEUTRON( "restMassOfNeutron", "(rest) mass of neutron" ),
ANGULAR_RECIPROCAL_LATTICE_VECTOR( "angularReciprocalLatticeVector", "angular reciprocal lattice vector" ),
REPETENCY( "repetency", "repetency" ),
LUMINOUS_EFFICACY_AT_A_SPECIFIED_WAVELENGTH( "luminousEfficacyAtASpecifiedWavelength", "luminous efficacy at a specified wavelength" ),
SPECTRAL_CONCENTRATION_OF_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVELENGTH( "spectralConcentrationOfRadiantEnergyDensityInTermsOfWavelength", "spectral concentration of radiant energy density (in terms of wavelength)" ),
THERMODYNAMIC( "thermodynamic", "thermodynamic" ),
MOLE_FRACTION_OF_B( "moleFractionOfB", "mole fraction of B" ),
RATIO_OF_THE_SPECIFIC_HEAT_CAPACITIES( "ratioOfTheSpecificHeatCapacities", "ratio of the specific heat capacities" ),
RESISTANCE_TO_DIRECT_CURRENT( "resistanceToDirectCurrent", "resistance (to direct current)" ),
OBJECT_DISTANCE( "objectDistance", "object distance" ),
INSTANTANEOUS_VOLUME_FLOW_RATE( "instantaneousVolumeFlowRate", "(instantaneous) volume flow rate" ),
STEFAN_BOLTZMANN_CONSTANT( "stefanBoltzmannConstant", "Stefan-Boltzmann constant" ),
FERMI_ANGULAR_REPETENCY( "fermiAngularRepetency", "Fermi angular repetency" ),
REACTION_ENERGY( "reactionEnergy", "reaction energy" ),
NUMBER_OF_PHASES( "numberOfPhases", "number of phases" ),
HEAT_CAPACITY( "heatCapacity", "heat capacity" ),
NUCLEON_NUMBER( "nucleonNumber", "nucleon number" ),
AREA( "area", "area" ),
SPECIFIC_OPTICAL_ROTATORY_POWER( "specificOpticalRotatoryPower", "specific optical rotatory power" ),
LENGTH_OF_PATH( "lengthOfPath", "length of path" ),
SUPERCONDUCTOR_ENERGY_GAP( "superconductorEnergyGap", "superconductor energy gap" ),
ACCELERATION_DUE_TO_GRAVITY( "accelerationDueToGravity", "acceleration due to gravity" ),
COEFFICIENT( "coefficient", "coefficient" ),
COMPLEX_ADMITTANCE( "complexAdmittance", "complex admittance" ),
TRANSMITTANCE( "transmittance", "transmittance" ),
ABSORBED_DOSE_RATE( "absorbedDoseRate", "absorbed dose rate" ),
INSTANTANEOUS_SOUND_PARTICLE_VELOCITY( "instantaneousSoundParticleVelocity", "(instantaneous) sound particle velocity" ),
FERMI_TEMPERATURE( "fermiTemperature", "Fermi temperature" ),
INTRINSIC_NUMBER_DENSITY( "intrinsicNumberDensity", "intrinsic number density" ),
ELEMENTARY_CHARGE( "elementaryCharge", "elementary charge" ),
PARTIAL_PRESSURE_OF_B_IN_A_GASEOUS_MIXTURE( "partialPressureOfBInAGaseousMixture", "partial pressure of B (in a gaseous mixture)" ),
FAHRENHEIT_TEMPERATURE( "fahrenheitTemperature", "fahrenheit temperature" ),
SPECIFIC_ENERGY_IMPARTED( "specificEnergyImparted", "specific energy imparted" ),
REACTOR_TIME_CONSTANT( "reactorTimeConstant", "reactor time constant" ),
PARTICLE_FLUENCE( "particleFluence", "particle fluence" ),
THERMAL_DIFFUSIVITY( "thermalDiffusivity", "thermal diffusivity" ),
MASSIC_THERMODYNAMIC_ENERGY( "massicThermodynamicEnergy", "massic thermodynamic energy" ),
VOLUME( "volume", "volume" ),
MADELUNG_CONSTANT( "madelungConstant", "Madelung constant" ),
ELECTRIC_FLUX_DENSITY( "electricFluxDensity", "electric flux density" ),
SPECTRAL_TRANSMITTANCE( "spectralTransmittance", "spectral transmittance" ),
RELATIVE_MASS_DENSITY( "relativeMassDensity", "relative mass density" ),
ELECTRIC_FIELD_STRENGTH( "electricFieldStrength", "electric field strength" ),
TOTAL_IONIZATION_BY_A_PARTICLE( "totalIonizationByAParticle", "total ionization by a particle" ),
COUPLING_COEFFICIENT( "couplingCoefficient", "coupling coefficient" ),
ATOMIC_ATTENUATION_COEFFICIENT( "atomicAttenuationCoefficient", "atomic attenuation coefficient" ),
DOSE_EQUIVALENT( "doseEquivalent", "dose equivalent" ),
SOUND_PRESSURE_LEVEL( "soundPressureLevel", "sound pressure level" ),
ELECTRIC_CONSTANT( "electricConstant", "electric constant" ),
LINEIC_CHARGE( "lineicCharge", "lineic charge" ),
VOLUME_PER_TEMPERATURE( "volumePerTemperature", "volume per temperature" ),
HEAT_FLOW_RATE( "heatFlowRate", "heat flow rate" ),
THOMPSON_COEFFICIENT( "thompsonCoefficient", "Thompson coefficient" ),
LOUDNESS_LEVEL( "loudnessLevel", "loudness level" ),
RESONANCE_ESCAPE_PROBABILITY( "resonanceEscapeProbability", "resonance escape probability" ),
SPECTRAL_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVE_LENGTH( "spectralRadiantEnergyDensityInTermsOfWaveLength", "spectral radiant energy density (in terms of wave length)" ),
ADMITTANCE( "admittance", "admittance" ),
FLUENCE_RATE( "fluenceRate", "fluence rate" ),
EMISSIVITY_AT_A_SPECIFIED_WAVELENGTH( "emissivityAtASpecifiedWavelength", "emissivity at a specified wavelength" ),
BRAGG_ANGLE( "braggAngle", "Bragg angle" ),
SOUND_POWER_LEVEL( "soundPowerLevel", "sound power level" ),
ANGULAR_REPETENCY( "angularRepetency", "angular repetency" ),
PHOTON_EXITANCE( "photonExitance", "photon exitance" ),
ANGULAR_CROSS_SECTION( "angularCrossSection", "angular cross-section" ),
PHASE_DISPLACEMENT( "phaseDisplacement", "phase displacement" ),
SOUND_REDUCTION_INDEX( "soundReductionIndex", "sound reduction index" ),
DIFFUSION_COEFFICIENT_FOR_NEUTRON_NUMBER_DENSITY( "diffusionCoefficientForNeutronNumberDensity", "diffusion coefficient for neutron number density" ),
ELECTROMOTIVE_FORCE( "electromotiveForce", "electromotive force" ),
THERMODYNAMIC_CRITICAL_MAGNETIC_FLUX_DENSITY( "thermodynamicCriticalMagneticFluxDensity", "thermodynamic critical magnetic flux density" ),
PARTICLE_FLUENCE_RATE( "particleFluenceRate", "particle fluence rate" ),
DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUENCE_RATE( "diffusionCoefficientForNeutronFluenceRate", "diffusion coefficient for neutron fluence rate" ),
LATTICE_VECTOR( "latticeVector", "lattice vector" ),
DISPLACEMENT_VECTOR_OF_ION_OR_ATOM( "displacementVectorOfIonOrAtom", "displacement vector of ion or atom" ),
LARMOR_ANGULAR_FREQUENCY( "larmorAngularFrequency", "Larmor angular frequency" ),
NEEL_TEMPERATURE( "neelTemperature", "Néel temperature" ),
MASS_DEFECT( "massDefect", "mass defect" ),
ACCEPTOR_IONIZATION_ENERGY( "acceptorIonizationEnergy", "acceptor ionization energy" ),
ELECTRIC_DIPOLE_MOMENT( "electricDipoleMoment", "electric dipole moment" ),
RELATIVE_ATOMIC_MASS( "relativeAtomicMass", "relative atomic mass" ),
STANDARD_EQUILIBRIUM_CONSTANT( "standardEquilibriumConstant", "standard equilibrium constant" ),
IMAGE_DISTANCE( "imageDistance", "image distance" ),
SPECTRAL_ABSORPTANCE( "spectralAbsorptance", "spectral absorptance" ),
QUANTITY_OF_HEAT( "quantityOfHeat", "quantity of heat" ),
LUMINOUS_EFFICIENCY( "luminousEfficiency", "luminous efficiency" ),
ACTIVITY_COEFFICIENT_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION( "activityCoefficientOfSoluteBEspeciallyInADiluteSolution", "activity coefficient of solute B (especially in a dilute solution)" ),
MAXIMUM_BETA_PARTICLE_ENERGY( "maximumBetaParticleEnergy", "maximum beta particle energy" ),
LENGTH( "length", "length" ),
HELMHOLTZ_FUNCTION( "helmholtzFunction", "Helmholtz function" ),
MOLAR_ATTENUATION_COEFFICIENT( "molarAttenuationCoefficient", "molar attenuation coefficient" ),
SURFACE_DENSITY( "surfaceDensity", "surface density" ),
LOWER_CRITICAL_MAGNETIC_FLUX_DENSITY( "lowerCriticalMagneticFluxDensity", "lower critical magnetic flux density" ),
MOLAR_FLUX( "molarFlux", "molar flux" ),
PLANCK_CONSTANT( "planckConstant", "Planck constant" ),
SOUND_EXPOSURE( "soundExposure", "sound exposure" ),
VOLUMIC_ELECTRON_NUMBER( "volumicElectronNumber", "volumic electron number" ),
MODULUS_OF_ELASTICITY( "modulusOfElasticity", "modulus of elasticity" ),
MOLAR_VOLUME( "molarVolume", "molar volume" ),
MODULUS_OF_IMPEDANCE( "modulusOfImpedance", "modulus of impedance" ),
ION_NUMBER_DENSITY( "ionNumberDensity", "ion number density" ),
GYROMAGNETIC_COEFFICIENT( "gyromagneticCoefficient", "gyromagnetic coefficient" ),
HEAT( "heat", "heat" ),
CONDUCTANCE_FOR_DIRECT_CURRENT( "conductanceForDirectCurrent", "conductance (for direct current)" ),
VOLTAGE( "voltage", "voltage" ),
NEUTRON_FLUENCE_RATE( "neutronFluenceRate", "neutron fluence rate" ),
FUGACITY_OF_B_IN_A_GASEOUS_MIXTURE( "fugacityOfBInAGaseousMixture", "fugacity of B (in a gaseous mixture)" ),
ANGULAR_IMPULSE( "angularImpulse", "angular impulse" ),
RADIANT_EXITANCE( "radiantExitance", "radiant exitance" ),
MASS_ATTENUATION_COEFFICIENT( "massAttenuationCoefficient", "mass attenuation coefficient" ),
KNUDSEN_NUMBER( "knudsenNumber", "Knudsen number" ),
SECTION_MODULUS( "sectionModulus", "section modulus" ),
GRASHOF_NUMBER_FOR_MASS_TRANSFER( "grashofNumberForMassTransfer", "Grashof number for mass transfer" ),
REVERBERATION_TIME( "reverberationTime", "reverberation time" ),
POWER( "power", "power" ),
POTENTIAL_DIFFERENCE( "potentialDifference", "potential difference" ),
TIME( "time", "time" ),
SPECIFIC_ENERGY( "specificEnergy", "specific energy" ),
WORK( "work", "work" ),
NEUTRON_NUMBER( "neutronNumber", "neutron number" ),
PACKING_FRACTION( "packingFraction", "packing fraction" ),
WAVELENGTH( "wavelength", "wavelength" ),
VOLUMIC_INTRINSIS_NUMBER( "volumicIntrinsisNumber", "volumic intrinsis number" ),
MOMENTUM( "momentum", "momentum" ),
KINEMATIC_VISCOSITY( "kinematicViscosity", "kinematic viscosity" ),
HARTREE_ENERGY( "hartreeEnergy", "Hartree energy" ),
DISSIPANCE( "dissipance", "dissipance" ),
ISENTROPIC_COMPRESSIBILITY( "isentropicCompressibility", "isentropic compressibility" ),
ANGULAR_ACCELERATION( "angularAcceleration", "angular acceleration" ),
MOLECULAR_CONCENTRATION_OF_B( "molecularConcentrationOfB", "molecular concentration of B" ),
OSMOTIC_COEFFICIENT_OF_THE_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION( "osmoticCoefficientOfTheSolventAEspeciallyInADiluteSolution", "osmotic coefficient of the solvent A (especially in a dilute solution)" ),
SECOND_POLAR_MOMENT_OF_AREA( "secondPolarMomentOfArea", "second polar moment of area" ),
ROTATORY_POWER( "rotatoryPower", "rotatory power" ),
GRAND_CANONICAL_PARTITION_FUNCTION( "grandCanonicalPartitionFunction", "grand-canonical partition function" ),
RELATIVE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION( "relativeActivityOfSolventAEspeciallyInADiluteSolution", "relative activity of solvent A (especially in a dilute solution)" ),
PECLET_NUMBER( "pecletNumber", "Peclet number" ),
TORQUE( "torque", "torque" ),
RAYLEIGH_NUMBER( "rayleighNumber", "Rayleigh number" ),
GROUP_VELOCITY( "groupVelocity", "group velocity" ),
IMPEDANCE( "impedance", "impedance" ),
LOSS_ANGLE( "lossAngle", "loss angle" ),
SPECTRAL_RADIANCE_FACTOR( "spectralRadianceFactor", "spectral radiance factor" ),
SECOND_RADIATION_CONSTANT( "secondRadiationConstant", "second radiation constant" ),
DEBYE_ANGULAR_WAVE_NUMBER( "debyeAngularWaveNumber", "Debye angular wave number" ),
MASS_NUMBER( "massNumber", "mass number" ),
AVERAGE_LOGARITHMIC_ENERGY_DECREMENT( "averageLogarithmicEnergyDecrement", "average logarithmic energy decrement" ),
VOLUMIC_DONOR_NUMBER( "volumicDonorNumber", "volumic donor number" ),
LEVEL_OF_A_FIELD_QUANTITY( "levelOfAFieldQuantity", "level of a field quantity" ),
CIRCULAR_FREQUENCY( "circularFrequency", "circular frequency" ),
SHEAR_STRESS( "shearStress", "shear stress" ),
NUCLEAR_QUADRUPOLE_MOMENT( "nuclearQuadrupoleMoment", "nuclear quadrupole moment" ),
RESISTIVITY( "resistivity", "resistivity" ),
LINEAR_IONIZATION_BY_A_PARTICLE( "linearIonizationByAParticle", "linear ionization by a particle" ),
FAST_FISSION_FACTOR( "fastFissionFactor", "fast fission factor" ),
PRESSURE_RATIO( "pressureRatio", "pressure ratio" ),
ORBITAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER( "orbitalAngularMomentumQuantumNumber", "orbital angular momentum quantum number" ),
ENTROPY( "entropy", "entropy" ),
TRANSPORT_NUMBER_OF_ION_B( "transportNumberOfIonB", "transport number of ion B" ),
LINEAR_MASS( "linearMass", "linear mass" ),
PARTITION_FUNCTION_OF_A_MOLECULE( "partitionFunctionOfAMolecule", "partition function of a molecule" ),
FREQUENCY_INTERVAL( "frequencyInterval", "frequency interval" ),
MASSIC_HELMHOLTZ_FREE_ENERGY( "massicHelmholtzFreeEnergy", "massic Helmholtz free energy" ),
HALF_LIFE( "halfLife", "half life" ),
ORDER_OF_REFLEXION( "orderOfReflexion", "order of reflexion" ),
CURIE_TEMPERATURE( "curieTemperature", "Curie temperature" ),
DISTANCE( "distance", "distance" ),
LENS_POWER( "lensPower", "lens power" ),
MOLAR_GAS_CONSTANT( "molarGasConstant", "molar gas constant" ),
SLOWING_DOWN_LENGTH( "slowingDownLength", "slowing-down length" ),
PARTICLE_POSITION_VECTOR( "particlePositionVector", "particle position vector" ),
REFLECTION_FACTOR( "reflectionFactor", "reflection factor" ),
WEBER_NUMBER( "weberNumber", "Weber number" ),
MAGNETIC_REYNOLDS_NUMBER( "magneticReynoldsNumber", "magnetic Reynolds number" ),
RADIANT_ENERGY( "radiantEnergy", "radiant energy" ),
STOICHIOMETRIC_NUMBER_OF_B( "stoichiometricNumberOfB", "stoichiometric number of B" ),
SPECIFIC_ENTHALPY( "specificEnthalpy", "specific enthalpy" ),
MASS( "mass", "mass" ),
BOHR_MAGNETON( "bohrMagneton", "Bohr magneton" ),
PERMEABILITY_OF_VACUUM( "permeabilityOfVacuum", "permeability of vacuum" ),
MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA( "momentOfInertiaDynamicMomentOfInertia", "moment of inertia (dynamic moment of inertia)" ),
LINEAR_EXTINCTION_COEFFICIENT( "linearExtinctionCoefficient", "linear extinction coefficient" ),
VOLUME_RATIO( "volumeRatio", "volume ratio" ),
DIFFUSION_COEFFICIENT( "diffusionCoefficient", "diffusion coefficient" ),
THERMAL_UTILIZATION_FACTOR( "thermalUtilizationFactor", "thermal utilization factor" ),
G_FACTOR_OF_NUCLEUS( "gFactorOfNucleus", "g-factor of nucleus" ),
CYCLOTRON_ANGULAR_FREQUENCY( "cyclotronAngularFrequency", "cyclotron angular frequency" ),
FINE_STRUCTURE_CONSTANT( "fineStructureConstant", "fine structure constant" ),
MASS_OF_MOLECULE( "massOfMolecule", "mass of molecule" ),
IMPULSE( "impulse", "impulse" ),
PROPAGATION_COEFFICIENT( "propagationCoefficient", "propagation coefficient" ),
TEMPERATURE( "temperature", "temperature" ),
INSTANTANEOUS_SOUND_PARTICLE_ACCELERATION( "instantaneousSoundParticleAcceleration", "(instantaneous) sound particle acceleration" ),
PHASE_VELOCITY_OF_ELECTROMAGNETIC_WAVES( "phaseVelocityOfElectromagneticWaves", "phase velocity of electromagnetic waves" ),
ANGULAR_FREQUENCY( "angularFrequency", "angular frequency" ),
MOMENT_OF_FORCE( "momentOfForce", "moment of force" ),
MOLAR_OPTICAL_ROTATORY_POWER( "molarOpticalRotatoryPower", "molar optical rotatory power" ),
NEUTRON_YIELD_PER_ABSORPTION( "neutronYieldPerAbsorption", "neutron yield per absorption" ),
COEFFICIENT_OF_HEAT_TRANSFER( "coefficientOfHeatTransfer", "coefficient of heat transfer" ),
DENSITY_OF_HEAT_FLOW_RATE( "densityOfHeatFlowRate", "density of heat flow rate" ),
FREQUENCY( "frequency", "frequency" ),
LINEAR_ELECTRIC_CURRENT_DENSITY( "linearElectricCurrentDensity", "linear electric current density" ),
EMISSIVITY( "emissivity", "emissivity" ),
VOLUMIC_DOSE( "volumicDose", "volumic dose" ),
MASS_OF_ATOM_OF_A_NUCLIDE_X( "massOfAtomOfANuclideX", "mass of atom (of a nuclide x)" ),
THERMAL_CONDUCTIVITY( "thermalConductivity", "thermal conductivity" ),
AREA_RELATED_TORSIONAL_MOMENT( "areaRelatedTorsionalMoment", "area-related torsional moment" ),
LINEAR_STRAIN( "linearStrain", "linear strain" ),
SLOWING_DOWN_DENSITY( "slowingDownDensity", "slowing down density" ),
ISOTHERMAL_COMPRESSIBILITY( "isothermalCompressibility", "isothermal compressibility" ),
MOLAR_CONDUCTIVITY( "molarConductivity", "molar conductivity" ),
EXPOSURE( "exposure", "exposure" ),
MASSIC_ENTHALPY( "massicEnthalpy", "massic enthalpy" ),
NEUTRON_NUMBER_DENSITY( "neutronNumberDensity", "neutron number density" ),
ISENTROPIC_EXPONENT( "isentropicExponent", "isentropic exponent" ),
THERMOELECTROMOTIVE_FORCE_BETWEEN_SUBSTANCES_A_AND_B( "thermoelectromotiveForceBetweenSubstancesAAndB", "thermoelectromotive force between substances a and b" ),
PHASE_DIFFERENCE( "phaseDifference", "phase difference" ),
LINEAR_ABSORPTION_COEFFICIENT( "linearAbsorptionCoefficient", "linear absorption coefficient" ),
BURGERS_VECTOR( "burgersVector", "Burgers vector" ),
CAPACITANCE( "capacitance", "capacitance" ),
THERMAL_DIFFUSION_RATIO( "thermalDiffusionRatio", "thermal diffusion ratio" ),
VOLUMIC_MASS( "volumicMass", "volumic mass" ),
OPTICAL_DENSITY( "opticalDensity", "optical density" ),
TEMPERATURE_VARIATION_OVER_TIME( "temperatureVariationOverTime", "temperature variation over time" ),
VOLUMIC_ACTIVITY( "volumicActivity", "volumic activity" ),
VOLUMIC_HOLE_NUMBER( "volumicHoleNumber", "volumic hole number" ),
EQUIVALENCE_DOSE_OUTPUT( "equivalenceDoseOutput", "equivalence dose output" ),
HARDNESS_INDEX( "hardnessIndex", "hardness index" ),
HARTMANN_NUMBER( "hartmannNumber", "Hartmann number" ),
RADIUS( "radius", "radius" ),
MOLAR_THERMODYNAMIC_ENERGY( "molarThermodynamicEnergy", "molar thermodynamic energy" ),
COMPLEX_IMPEDANCES( "complexImpedances", "complex impedances" ),
MASSIC_VOLUME( "massicVolume", "massic volume" ),
DEBYE_WALLE_FACTOR( "debyeWalleFactor", "Debye-Walle factor" ),
EFFECTIVE_MULTIPLICATION_FACTOR( "effectiveMultiplicationFactor", "effective multiplication factor" ),
AFFINITY_OF_A_CHEMICAL_REACTION( "affinityOfAChemicalReaction", "affinity (of a chemical reaction)" ),
SOUND_ENERGY_DENSITY( "soundEnergyDensity", "sound energy density" ),
FOURIER_NUMBER_FOR_MASS_TRANSFER( "fourierNumberForMassTransfer", "Fourier number for mass transfer" ),
VOLUMIC_CHARGE( "volumicCharge", "volumic charge" ),
THERMAL_RESISTANCE( "thermalResistance", "thermal resistance" ),
COORDINATES_TRICHROMATIC( "coordinatesTrichromatic", "coordinates trichromatic" ),
EQUIVALENT_ABSORPTION_AREA_OF_A_SURFACE_OR_OBJECT( "equivalentAbsorptionAreaOfASurfaceOrObject", "equivalent absorption area of a surface or object" ),
COHERENCE_LENGTH( "coherenceLength", "coherence length" ),
RESONANCE_ENERGY( "resonanceEnergy", "resonance energy" ),
RELATIVE_MASS_DEFECT( "relativeMassDefect", "relative mass defect" ),
DISPLACEMENT_ELECTRIC_POLARIZATION( "displacementElectricPolarization", "displacement electric polarization" ),
TIME_CONSTANT( "timeConstant", "time constant" ),
KINETIC_ENERGY( "kineticEnergy", "kinetic energy" ),
CONDUCTANCE_FOR_ALTERNATING_CURRENT( "conductanceForAlternatingCurrent", "conductance (for alternating current)" ),
FORCE_DIVIDED_BY_LENGTH( "forceDividedByLength", "force divided by length" ),
RADIANT_POWER( "radiantPower", "radiant power" ),
ELECTRIC_CURRENT( "electricCurrent", "electric current" ),
TRANSMISSION_FACTOR( "transmissionFactor", "transmission factor" ),
DAMPING_COEFFICIENT( "dampingCoefficient", "damping coefficient" ),
BETA_DISINTEGRATION_ENERGY( "betaDisintegrationEnergy", "beta disintegration energy" ),
QUANTITY_OF_ELECTRICITY( "quantityOfElectricity", "quantity of electricity" ),
RELUCTANCE( "reluctance", "reluctance" ),
MAGNETIZATION( "magnetization", "magnetization" ),
BOLTZMANN_CONSTANT( "boltzmannConstant", "Boltzmann constant" ),
PERMITTIVITY_OF_VACUUM( "permittivityOfVacuum", "permittivity of vacuum" ),
DEBYE_TEMPERATURE( "debyeTemperature", "Debye temperature" ),
SPECIFIC_ACTIVITY_IN_A_SAMPLE( "specificActivityInASample", "specific activity in a sample" ),
LANDAU_GINZBURG_NUMBER( "landauGinzburgNumber", "Landau-Ginzburg number" ),
NORMAL_STRESS( "normalStress", "normal stress" ),
ELECTRON_RADIUS( "electronRadius", "electron radius" ),
LINEAR_DENSITY( "linearDensity", "linear density" ),
WEIGHT( "weight", "weight" ),
DONOR_NUMBER_DENSITY( "donorNumberDensity", "donor number density" ),
FORCE( "force", "force" ),
EXCHANGE_INTERGRAL( "exchangeIntergral", "exchange intergral" ),
DISINTEGRATION_CONSTANT( "disintegrationConstant", "disintegration constant" ),
ACTIVITY_COEFFICIENT_OF_B_IN_A_LIQUID_AS_A_SOLID_MIXTURE( "activityCoefficientOfBInALiquidAsASolidMixture", "activity coefficient of B (in a liquid as a solid mixture)" ),
INSTANTANEOUS_SOUND_PARTICLE_DISPLACEMENT( "instantaneousSoundParticleDisplacement", "(instantaneous) sound particle displacement" ),
WAVE_NUMBER( "waveNumber", "wave number" ),
PRINCIPLE_QUANTUM_NUMBER( "principleQuantumNumber", "principle quantum number" ),
HALL_COEFFICIENT( "hallCoefficient", "Hall coefficient" ),
MIGRATION_AREA( "migrationArea", "migration area" ),
ATOMIC_NUMBER( "atomicNumber", "atomic number" ),
PROTON_NUMBER( "protonNumber", "proton number" ),
MAGNETIC_POTENTIAL_DIFFERENCE( "magneticPotentialDifference", "magnetic potential difference" ),
PHASE_VELOCITY( "phaseVelocity", "phase velocity" ),
SCHMIDT_NUMBER( "schmidtNumber", "Schmidt number" ),
GRAND_PARTITION_FUNCTION( "grandPartitionFunction", "grand partition function" ),
ACTIVE_ENERGY( "activeEnergy", "active energy" ),
MACH_NUMBER( "machNumber", "Mach number" ),
UNIFIED_ATOMIC_MASS_CONSTANT( "unifiedAtomicMassConstant", "unified atomic mass constant" ),
STATISTICAL_WEIGHT( "statisticalWeight", "statistical weight" ),
EULER_NUMBER( "eulerNumber", "Euler number" ),
MEAN_MASS_RANGE( "meanMassRange", "mean mass range" ),
VOLUME_DENSITY_OF_CHARGE( "volumeDensityOfCharge", "volume density of charge" ),
DONOR_IONIZATION_ENERGY( "donorIonizationEnergy", "donor ionization energy" ),
RADIANT_INTENSITY( "radiantIntensity", "radiant intensity" ),
ELECTRON_AFFINITY( "electronAffinity", "electron affinity" ),
ABSOLUTE_ACTIVITY( "absoluteActivity", "absolute activity" ),
FUNDAMENTAL_LATTICE_VECTOR( "fundamentalLatticeVector", "fundamental lattice vector" ),
RELATIVE_ELONGATION( "relativeElongation", "relative elongation" ),
TOTAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER( "totalAngularMomentumQuantumNumber", "total angular momentum quantum number" ),
ROTATIONAL_FREQUENCY( "rotationalFrequency", "rotational frequency" ),
RECOMBINATION_COEFFICIENT( "recombinationCoefficient", "recombination coefficient" ),
WORK_PER_UNIT_WEIGHT( "workPerUnitWeight", "work per unit weight" ),
TOTAL_LINEAR_STOPPING_POWER( "totalLinearStoppingPower", "total linear stopping power" ),
LEAKAGE_RATE_OF_GAS( "leakageRateOfGas", "leakage rate of gas" ),
POYNTING_VECTOR( "poyntingVector", "Poynting vector" ),
MEAN_ENERGY_IMPARTED( "meanEnergyImparted", "mean energy imparted" ),
MAGNETIC_FLUX( "magneticFlux", "magnetic flux" ),
ELECTROMAGNETIC_ENERGY_DENSITY( "electromagneticEnergyDensity", "electromagnetic energy density" ),
SOUND_INTENSITY( "soundIntensity", "sound intensity" ),
RELATIVE_PERMITTIVITY( "relativePermittivity", "relative permittivity" ),
MEAN_FREE_PATH_OF_PHONONS_OR_ELECTRONS( "meanFreePathOfPhononsOrElectrons", "mean free path of phonons or electrons" ),
UPPER_CRITICAL_MAGNETIC_FLUX_DENSITY( "upperCriticalMagneticFluxDensity", "upper critical magnetic flux density" ),
EQUILIBRIUM_POSITION_VECTOR_OF_ION_OR_ATOM( "equilibriumPositionVectorOfIonOrAtom", "equilibrium position vector of ion or atom" ),
VOLUMIC( "volumic", "volumic" ),
HELMHOLTZ_FREE_ENERGY( "helmholtzFreeEnergy", "Helmholtz free energy" ),
HEIGHT( "height", "height" ),
CHARGE_NUMBER_OF_ION( "chargeNumberOfIon", "charge number of ion" ),
WORK_FUNCTION( "workFunction", "work function" ),
RADIANT_ENERGYFLUX( "radiantEnergyflux", "radiant energyflux" ),
CROSS_SECTION( "crossSection", "cross-section" ),
MULTIPLICATION_FACTOR( "multiplicationFactor", "multiplication factor" ),
FERMI_ENERGY( "fermiEnergy", "Fermi energy" ),
SPECTRAL_REFLECTIONFACTOR( "spectralReflectionfactor", "spectral reflectionfactor" ),
SPIN_ANGULAR_MOMENTUM_QUANTUM_NUMBER( "spinAngularMomentumQuantumNumber", "spin angular momentum quantum number" ),
VOLUMIC_TOTAL_CROSS_SECTION( "volumicTotalCrossSection", "volumic total cross-section" ),
ENERGY_FLUX_DENSITY( "energyFluxDensity", "energy flux density" ),
NUCLEAR_PRECESSION( "nuclearPrecession", "nuclear precession" ),
DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUX_DENSITY( "diffusionCoefficientForNeutronFluxDensity", "diffusion coefficient for neutron flux density" ),
PHASE_SPEED_OF_ELECTROMAGNETIC_WAVES( "phaseSpeedOfElectromagneticWaves", "phase speed of electromagnetic waves" ),
PERFORMANCE_CHARACTERISTIC( "performanceCharacteristic", "performance characteristic" ),
MICROCANONICAL_PARTITION_FUNCTION( "microcanonicalPartitionFunction", "microcanonical partition function" ),
REACTIVITY( "reactivity", "reactivity" ),
ABSORBANCE( "absorbance", "absorbance" ),
RESISTANCE_TO_ALTERNATING_CURRENT( "resistanceToAlternatingCurrent", "resistance (to alternating current)" ),
ELECTRON_NUMBER_DENSITY( "electronNumberDensity", "electron number density" ),
COMPRESSIBILITY( "compressibility", "compressibility" ),
G_FACTOR_OF_ATOM_OR_ELECTRON( "gFactorOfAtomOrElectron", "g-factor of atom or electron" ),
HYPERFINE_STRUCTURE_QUANTUM_NUMBER( "hyperfineStructureQuantumNumber", "hyperfine structure quantum number" ),
CUBIC_EXPANSION_COEFFICIENT( "cubicExpansionCoefficient", "cubic expansion coefficient" ),
MASS_CONCENTRATION_OF_B( "massConcentrationOfB", "mass concentration of B" ),
MACROSCOPIC_CROSS_SECTION( "macroscopicCrossSection", "macroscopic cross-section" ),
DATA_RATE( "dataRate", "Data rate" ),
ELECTROLYTIC_CONDUCTIVITY( "electrolyticConductivity", "electrolytic conductivity" ),
TOTAL_ATOMIC_STOPPING_POWER( "totalAtomicStoppingPower", "total atomic stopping power" ),
MASS_RATIO( "massRatio", "mass ratio" ),
MAGNETIC_FLUX_DENSITY( "magneticFluxDensity", "magnetic flux density" ),
MASSIC_OPTICAL( "massicOptical", "massic optical" ),
DIRECTIONAL_SPECTRAL_EMISSIVITY( "directionalSpectralEmissivity", "directional spectral emissivity" ),
CONCENTRATION_OF_B( "concentrationOfB", "concentration of B" ),
ELECTRIC_CHARGE( "electricCharge", "electric charge" ),
PRANDTL_NUMBER( "prandtlNumber", "Prandtl number" ),
RYDBERG_CONSTANT( "rydbergConstant", "Rydberg constant" ),
THERMAL_INSULANCE( "thermalInsulance", "thermal insulance" ),
MOLAR_ABSORPTION_COEFFICIENT( "molarAbsorptionCoefficient", "molar absorption coefficient" ),
FERMI_ANGULAR_WAVE_NUMBER( "fermiAngularWaveNumber", "Fermi angular wave number" ),
FOURIER_NUMBER( "fourierNumber", "Fourier number" ),
CIE_COLORIMETRIC_FUNCTIONS( "cieColorimetricFunctions", "CIE colorimetric functions" ),
NON_LEAKAGE_PROBABILITY( "nonLeakageProbability", "non leakage probability" ),
WAVENUMBER( "wavenumber", "wavenumber" ),
INSTANTANEOUS_SOUND_PRESSURE( "instantaneousSoundPressure", "(instantaneous) sound pressure" ),
NEUTRON_SPEED( "neutronSpeed", "neutron speed" ),
NUMBER_OF_TURNS_IN_A_WINDING( "numberOfTurnsInAWinding", "number of turns in a winding" ),
ALPHA_DISINTEGRATION_ENERGY( "alphaDisintegrationEnergy", "alpha disintegration energy" ),
BULK_COMPRESSIBILITY( "bulkCompressibility", "bulk compressibility" ),
MODULUS_OF_RIGIDITY( "modulusOfRigidity", "modulus of rigidity" ),
NUMBER_OF_PAIRS_OF_POLES( "numberOfPairsOfPoles", "number of pairs of poles" ),
VOLUMIC_ACCEPTOR_NUMBER( "volumicAcceptorNumber", "volumic acceptor number" ),
INFINITE_MEDIUM_MULTIPLICATION_FACTOR( "infiniteMediumMultiplicationFactor", "infinite medium multiplication factor" ),
SPECTRAL_TRANSMISSION_FACTOR( "spectralTransmissionFactor", "spectral transmission factor" ),
CURRENT_DENSITY( "currentDensity", "current density" ),
DENSITY( "density", "density" ),
FOCAL_DISTANCE( "focalDistance", "focal distance" ),
NUCLEAR_SPIN_QUANTUM_NUMBER( "nuclearSpinQuantumNumber", "nuclear spin quantum number" ),
RESISTANCE_LOAD_PER_UNIT_LENGTH( "resistanceLoadPerUnitLength", "resistance load per unit length" ),
NUCLEAR_RADIUS( "nuclearRadius", "nuclear radius" ),
ENERGY_DENSITY( "energyDensity", "energy density" ),
STROUHAL_NUMBER( "strouhalNumber", "Strouhal number" ),
CARTESIAN_COORDINATES( "cartesianCoordinates", "cartesian coordinates" ),
LUMINANCE( "luminance", "luminance" ),
MOLAR_MASS( "molarMass", "molar mass" ),
BREADTH( "breadth", "breadth" ),
BURST_INDEX( "burstIndex", "burst index" ),
SHORT_RANGE_ORDER_PARAMETER( "shortRangeOrderParameter", "short-range order parameter" ),
LUMINOUS_EXITANCE( "luminousExitance", "luminous exitance" ),
REST_MASS_OF_ELECTRON( "restMassOfElectron", "(rest) mass of electron" ),
MASSIC_ENERGY_IMPARTED( "massicEnergyImparted", "massic energy imparted" ),
EFFECTIVE_MASS( "effectiveMass", "effective mass" ),
ENERGY_FLUENCE_RATE( "energyFluenceRate", "energy fluence rate" ),
ILLUMINANCE( "illuminance", "illuminance" ),
MEAN_LINEAR_RANGE( "meanLinearRange", "mean linear range" ),
LEVEL_WIDTH( "levelWidth", "level width" ),
MAGNETIC_CONSTANT( "magneticConstant", "magnetic constant" ),
RESIDUAL_RESISTIVITY( "residualResistivity", "residual resistivity" ),
SHEAR_MODULUS( "shearModulus", "shear modulus" ),
VELOCITY( "velocity", "velocity" ),
MAGNETIC_FLUX_QUANTUM( "magneticFluxQuantum", "magnetic flux quantum" ),
HALF_THICKNESS( "halfThickness", "half-thickness" ),
BULK_MODULUS( "bulkModulus", "bulk modulus" ),
MAGNETIC_SUSCEPTIBILITY( "magneticSusceptibility", "magnetic susceptibility" ),
PRESSURE_COEFFICIENT( "pressureCoefficient", "pressure coefficient" ),
RICHARDSON_CONSTANT( "richardsonConstant", "Richardson constant" ),
STATIC_PRESSURE( "staticPressure", "static pressure" ),
OSMOTIC_PRESSURE( "osmoticPressure", "osmotic pressure" ),
ANGULAR_WAVENUMBER( "angularWavenumber", "angular wavenumber" ),
MOMENT_OF_MOMENTUM( "momentOfMomentum", "moment of momentum" ),
SUPER_CONDUCTOR_TRANSITION_TEMPERATURE( "superConductorTransitionTemperature", "Super conductor transition temperature" ),
SOUND_ENERGY( "soundEnergy", "sound energy" ),
GYROMAGNETIC_RATIO( "gyromagneticRatio", "gyromagnetic ratio" ),
ENERGY( "energy", "energy" ),
ANGLE_PLANE( "anglePlane", "angle (plane)" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_GASEOUS_MIXTURE( "standardAbsoluteActivityOfBInAGaseousMixture", "standard absolute activity of B (in a gaseous mixture)" ),
ELECTROMAGNETIC_MOMENT( "electromagneticMoment", "electromagnetic moment" ),
EXPOSURE_RATE( "exposureRate", "exposure rate" ),
FROUDE_NUMBER( "froudeNumber", "Froude number" ),
MAGNETIC_AREA_MOMENT( "magneticAreaMoment", "magnetic area moment" ),
CURRENT_FRACTION_OF_ION_B( "currentFractionOfIonB", "current fraction of ion B" ),
SPECIFIC_HEAT_CAPACITY_AT_SATURATION( "specificHeatCapacityAtSaturation", "specific heat capacity at saturation" ),
CATALYTIC_ACTIVITY( "catalyticActivity", "catalytic activity" ),
ENTHALPY( "enthalpy", "enthalpy" ),
SPECTRAL_LUMINOUS_EFFICACY( "spectralLuminousEfficacy", "spectral luminous efficacy" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION( "standardAbsoluteActivityOfSoluteBEspeciallyInADiluteSolution", "standard absolute activity of solute B (especially in a dilute solution)" ),
MAXIMUM_SPECTRAL_LUMINOUS_EFFICACY( "maximumSpectralLuminousEfficacy", "maximum spectral luminous efficacy" ),
LINEAR_ENERGY_TRANSFER( "linearEnergyTransfer", "linear energy transfer" ),
LUMINOUS_INTENSITY( "luminousIntensity", "luminous intensity" ),
MOLE_RATIO_OF_SOLUTE_B( "moleRatioOfSoluteB", "mole ratio of solute B" ),
MAGNETIC_DIPOLE_MOMENT( "magneticDipoleMoment", "magnetic dipole moment" ),
DEGREE_OF_DISSOCIATION( "degreeOfDissociation", "degree of dissociation" ),
MAGNETIC_INDUCTION( "magneticInduction", "magnetic induction" ),
NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES( "numberDensityOfMoleculesOrParticles", "number density of molecules  (or particles)" ),
ION_DENSITY( "ionDensity", "ion density" ),
SURFACE_DENSITY_OF_CHARGE( "surfaceDensityOfCharge", "surface density of charge" ),
POISSON_RATIO( "poissonRatio", "poisson ratio" ),
ABSORPTION_FACTOR( "absorptionFactor", "absorption factor" ),
LINEIC_RESISTANCE( "lineicResistance", "lineic resistance" ),
THERMODYNAMIC_ENERGY( "thermodynamicEnergy", "thermodynamic energy" ),
FUNDAMENTAL_RECIPROCAL_LATTICE_VECTOR( "fundamentalReciprocalLatticeVector", "fundamental reciprocal lattice vector" ),
VELOCITY_OF_SOUND_PHASE_VELOCITY( "velocityOfSoundPhaseVelocity", "velocity of sound (phase velocity)" ),
ANGULAR_VELOCITY( "angularVelocity", "angular velocity" ),
LUMINIOUS_EFFICACY( "luminiousEfficacy", "luminious efficacy" ),
SPECTRAL_LUMINOUS_EFFICIENCY( "spectralLuminousEfficiency", "spectral luminous efficiency" ),
RADIANCE_EXPOSURE( "radianceExposure", "radiance exposure" ),
SOLID_ANGLE( "solidAngle", "solid angle" ),
CONDUCTIVITY( "conductivity", "conductivity" ),
HALF_VALUE_THICKNESS( "halfValueThickness", "half-value thickness" ),
SPECIFIC_THERMODYNAMIC_ENERGY( "specificThermodynamicEnergy", "specific thermodynamic energy" ),
MASS_EXCESS( "massExcess", "mass excess" ),
DIFFUSION_AREA( "diffusionArea", "diffusion area" ),
PHOTON_LUMINANCE( "photonLuminance", "photon luminance" ),
LOGARITHMIC_DECREMENT( "logarithmicDecrement", "logarithmic decrement" ),
VOLUMIC_ELECTROMAGNETIC_ENERGY( "volumicElectromagneticEnergy", "volumic electromagnetic energy" ),
QUANTITY_OF_LIGHT( "quantityOfLight", "quantity of light" ),
RATIO_OF_THE_MASSIC_HEAT_CAPACITY( "ratioOfTheMassicHeatCapacity", "ratio of the massic heat capacity" ),
LIGHT_EXPOSURE( "lightExposure", "light exposure" ),
PERIODIC_TIME( "periodicTime", "periodic time" ),
SHEAR_STRAIN( "shearStrain", "shear strain" ),
MASS_FLOW_RATE( "massFlowRate", "mass flow rate" ),
VOLUME_FLOW_RATE( "volumeFlowRate", "volume flow rate" ),
NUSSELT_NUMBER_FOR_MASS_TRANSFER( "nusseltNumberForMassTransfer", "Nusselt number for mass transfer" ),
ACTIVITY( "activity", "activity" ),
MAGNETIC_MOMENT( "magneticMoment", "magnetic moment" ),
ANGLE_OF_OPTICAL_ROTATION( "angleOfOpticalRotation", "angle of optical rotation" ),
LINEAR_ATTENUATION_COEFFICIENT( "linearAttenuationCoefficient", "linear attenuation coefficient" ),
LINEIC_ELECTRIC_CURRENT( "lineicElectricCurrent", "lineic electric current" ),
CANONICAL_PARTITION_FUNCTION( "canonicalPartitionFunction", "canonical partition function" ),
MOBILITY( "mobility", "mobility" ),
HOLE_NUMBER_DENSITY( "holeNumberDensity", "hole number density" ),
AMOUNT_OF_SUBSTANCE( "amountOfSubstance", "amount of substance" ),
PARTICAL_FLUX_DENSITY( "particalFluxDensity", "partical flux density" ),
TOTAL_MASS_STOPPING_POWER( "totalMassStoppingPower", "total mass stopping power" ),
LUMINOUS_FLUX( "luminousFlux", "luminous flux" ),
THICKNESS( "thickness", "thickness" ),
NEUTRON_YIELD_PER_FISSION( "neutronYieldPerFission", "neutron yield per fission" ),
DEBYE_ANGULAR_FREQUENCY( "debyeAngularFrequency", "Debye angular frequency" ),
SURFACE_COEFFICIENT_OF_HEAT_TRANSFER( "surfaceCoefficientOfHeatTransfer", "surface coefficient of heat transfer" ),
ACCELERATION_OF_FREE_FALL( "accelerationOfFreeFall", "acceleration of free fall" ),
ABSORBED_DOSE( "absorbedDose", "absorbed dose" ),
LONG_RANGE_ORDER_PARAMETER( "longRangeOrderParameter", "long-range order parameter" ),
MASS_DENSITY( "massDensity", "mass density" ),
MAGNETIC_FIELD_STRENGTH( "magneticFieldStrength", "magnetic field strength" ),
PERMEABILITY( "permeability", "permeability" ),
REYNOLDS_NUMBER( "reynoldsNumber", "Reynolds number" ),
RELATIVE_DENSITY( "relativeDensity", "relative density" ),
THERMAL_DIFFUSION_COEFFICIENT( "thermalDiffusionCoefficient", "thermal diffusion coefficient" ),
STANTON_NUMBER( "stantonNumber", "Stanton number" ),
RADIANCE( "radiance", "radiance" ),
ACCEPTOR_NUMBER_DENSITY( "acceptorNumberDensity", "acceptor number density" ),
ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT( "electricFluxFluxOfDisplacement", "electric flux (flux of displacement)" ),
DECAY_CONSTANT( "decayConstant", "decay constant" );

   private final String name;
   private final String label;

   QuantityKinds( final String name, final String label ) {
      this.name = name;
      this.label = label;
   }

   /**
    * Returns the quantity kind's unique name
    */
   @Override
   public String getName() {
      return name;
   }

   /**
    * Returns the quantity kind's human-readable name
    */
   @Override
   public String getLabel() {
      return label;
   }

   /**
    * Returns the quantity kind's human-readable name
    */
   @Override
   public String toString() {
      return getLabel();
   }

   @Override
   public Optional<AspectModelUrn> getAspectModelUrn() {
      return Optional.of( AspectModelUrn.fromUrn(
            String.format( "urn:samm:org.eclipse.esmf.samm:unit:%s#%s", KnownVersion.getLatest().toVersionString(), name ) ) );
   }

   @Override
   public KnownVersion getMetaModelVersion() {
      return KnownVersion.getLatest();
   }

   @Override
   public <T, C> T accept( final AspectVisitor<T, C> visitor, final C context ) {
      return visitor.visitQuantityKind( this, context );
   }

   /**
    * Finds the quantity kind with a given name
    */
   public static Optional<QuantityKind> fromName( final String name ) {
      return Arrays.stream( QuantityKinds.values() )
            .filter( quantityKind -> quantityKind.getName().equals( name ) )
            .map( QuantityKind.class::cast )
            .findAny();
   }
}
