/*
 * Copyright (c) 2024 Robert Bosch Manufacturing Solutions GmbH
 *
 * See the AUTHORS file(s) distributed with this work for additional
 * information regarding authorship.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * SPDX-License-Identifier: MPL-2.0
 */

package org.eclipse.esmf.metamodel;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.HashSet;
import java.util.Collections;
import java.util.stream.Collectors;

import org.eclipse.esmf.aspectmodel.urn.AspectModelUrn;
import org.eclipse.esmf.metamodel.QuantityKind;
import org.eclipse.esmf.metamodel.Unit;
import org.eclipse.esmf.metamodel.impl.DefaultUnit;
import org.eclipse.esmf.samm.KnownVersion;
import org.eclipse.esmf.metamodel.loader.MetaModelBaseAttributes;
import org.eclipse.esmf.metamodel.datatypes.LangString;

/**
 * Enumeration of Units as defined in <a href="https://tfig.unece.org/contents/recommendation-20.htm">Recommendation 20</a> by
 * the <a href="https://www.unece.org/info/ece-homepage.html">UNECE</a> (United Nations Economic Commission for Europe).
 */
public class Units {
   private static final Map<String, Unit> UNITS_BY_NAME = new HashMap<>();

   private Units() {
   }

private static void init0() {
UNITS_BY_NAME.put( "accessLine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#accessLine" ), "accessLine", Set.of(new LangString( "access line", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "accountingUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#accountingUnit" ), "accountingUnit", Set.of(new LangString( "accounting unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E50"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "acre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#acre" ), "acre", Set.of(new LangString( "acre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("acre"), Optional.of("ACR"), Optional.of("squareMetre"), Optional.of("4046.873 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "acreFootBasedOnUsSurveyFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#acreFootBasedOnUsSurveyFoot" ), "acreFootBasedOnUsSurveyFoot", Set.of(new LangString( "acre-foot (based on U.S. survey foot)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("acre-ft (US survey)"), Optional.of("M67"), Optional.of("cubicMetre"), Optional.of("1.233489 \u00D7 10\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "activeUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#activeUnit" ), "activeUnit", Set.of(new LangString( "active unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E25"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "activityUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#activityUnit" ), "activityUnit", Set.of(new LangString( "activity", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ACT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "actualPer360", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#actualPer360" ), "actualPer360", Set.of(new LangString( "actual/360", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("y (360 days)"), Optional.of("M37"), Optional.of("secondUnitOfTime"), Optional.of("3.1104000 \u00D7 10\u2077 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "additionalMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#additionalMinute" ), "additionalMinute", Set.of(new LangString( "additional minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "airDryMetricTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#airDryMetricTon" ), "airDryMetricTon", Set.of(new LangString( "air dry metric ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "airDryTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#airDryTon" ), "airDryTon", Set.of(new LangString( "air dry ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E28"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "alcoholicStrengthByMass", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#alcoholicStrengthByMass" ), "alcoholicStrengthByMass", Set.of(new LangString( "alcoholic strength by mass", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ASM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "alcoholicStrengthByVolume", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#alcoholicStrengthByVolume" ), "alcoholicStrengthByVolume", Set.of(new LangString( "alcoholic strength by volume", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ASU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampere" ), "ampere", Set.of(new LangString( "ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A"), Optional.of("AMP"), Optional.empty(), Optional.of("A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "ampereHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampereHour" ), "ampereHour", Set.of(new LangString( "ampere hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A\u00B7h"), Optional.of("AMH"), Optional.of("coulomb"), Optional.of("3.6 \u00D7 10\u00B3 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "ampereMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampereMinute" ), "ampereMinute", Set.of(new LangString( "ampere minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A\u00B7min"), Optional.of("N95"), Optional.of("coulomb"), Optional.of("60 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "amperePerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerCentimetre" ), "amperePerCentimetre", Set.of(new LangString( "ampere per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/cm"), Optional.of("A2"), Optional.of("amperePerMetre"), Optional.of("10\u00B2 A/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerKilogram" ), "amperePerKilogram", Set.of(new LangString( "ampere per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/kg"), Optional.of("H31"), Optional.empty(), Optional.of("A \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerMetre" ), "amperePerMetre", Set.of(new LangString( "ampere per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/m"), Optional.of("AE"), Optional.empty(), Optional.of("A/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIZATION, QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerMillimetre" ), "amperePerMillimetre", Set.of(new LangString( "ampere per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/mm"), Optional.of("A3"), Optional.of("amperePerMetre"), Optional.of("10\u00B3 A/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerPascal" ), "amperePerPascal", Set.of(new LangString( "ampere per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/Pa"), Optional.of("N93"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerSquareCentimetre" ), "amperePerSquareCentimetre", Set.of(new LangString( "ampere per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/cm\u00B2"), Optional.of("A4"), Optional.of("amperePerSquareMetre"), Optional.of("10\u2074 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerSquareMetre" ), "amperePerSquareMetre", Set.of(new LangString( "ampere per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/m\u00B2"), Optional.of("A41"), Optional.empty(), Optional.of("A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerSquareMetreKelvinSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerSquareMetreKelvinSquared" ), "amperePerSquareMetreKelvinSquared", Set.of(new LangString( "ampere per square metre kelvin squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/(m\u00B2\u00B7K\u00B2)"), Optional.of("A6"), Optional.empty(), Optional.of("A/(m\u00B2 \u00D7 K\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.RICHARDSON_CONSTANT)) ) );
UNITS_BY_NAME.put( "amperePerSquareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#amperePerSquareMillimetre" ), "amperePerSquareMillimetre", Set.of(new LangString( "ampere per square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A/mm\u00B2"), Optional.of("A7"), Optional.of("amperePerSquareMetre"), Optional.of("10\u2076 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "ampereSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampereSecond" ), "ampereSecond", Set.of(new LangString( "ampere second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A\u00B7s"), Optional.of("A8"), Optional.of("coulomb"), Optional.of("C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.QUANTITY_OF_ELECTRICITY)) ) );
UNITS_BY_NAME.put( "ampereSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampereSquareMetre" ), "ampereSquareMetre", Set.of(new LangString( "ampere square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A\u00B7m\u00B2"), Optional.of("A5"), Optional.empty(), Optional.of("A \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_MOMENT_OF_PARTICLE, QuantityKinds.NUCLEAR_MAGNETON_ORNUCLEUS, QuantityKinds.MAGNETIC_MOMENT, QuantityKinds.BOHR_MAGNETON, QuantityKinds.ELECTROMAGNETIC_MOMENT, QuantityKinds.MAGNETIC_AREA_MOMENT)) ) );
UNITS_BY_NAME.put( "ampereSquareMetrePerJouleSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampereSquareMetrePerJouleSecond" ), "ampereSquareMetrePerJouleSecond", Set.of(new LangString( "ampere square metre per joule second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A\u00B7m\u00B2/(J\u00B7s)"), Optional.of("A10"), Optional.empty(), Optional.of("(A \u00D7 s)/kg"), new HashSet<>(Arrays.asList(QuantityKinds.GYROMAGNETIC_RATIO, QuantityKinds.GYROMAGNETIC_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "ampereSquaredSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ampereSquaredSecond" ), "ampereSquaredSecond", Set.of(new LangString( "ampere squared second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("A\u00B2\u00B7s"), Optional.of("H32"), Optional.empty(), Optional.of("A\u00B2 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "angstrom", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#angstrom" ), "angstrom", Set.of(new LangString( "angstrom", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00C5"), Optional.of("A11"), Optional.of("metre"), Optional.of("10\u207B\u00B9\u2070 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LATTICE_PLANE_SPACING, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.WAVELENGTH, QuantityKinds.BURGERS_VECTOR, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.BOHR_RADIUS, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "antiHemophilicFactorAhfUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#antiHemophilicFactorAhfUnit" ), "antiHemophilicFactorAhfUnit", Set.of(new LangString( "anti-hemophilic factor (AHF) unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "assembly", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#assembly" ), "assembly", Set.of(new LangString( "assembly", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AY"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "assortment", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#assortment" ), "assortment", Set.of(new LangString( "assortment", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "astronomicalUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#astronomicalUnit" ), "astronomicalUnit", Set.of(new LangString( "astronomical unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ua"), Optional.of("A12"), Optional.of("metre"), Optional.of("1.49597870 \u00D7 10\u00B9\u00B9 m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "attofarad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#attofarad" ), "attofarad", Set.of(new LangString( "attofarad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("aF"), Optional.of("H48"), Optional.empty(), Optional.of("10\u207B\u00B9\u2078 m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u2074 \u00D7 A\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "attojoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#attojoule" ), "attojoule", Set.of(new LangString( "attojoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("aJ"), Optional.of("A13"), Optional.of("joule"), Optional.of("10\u207B\u00B9\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRON_AFFINITY, QuantityKinds.ACCEPTOR_IONIZATION_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.SUPERCONDUCTOR_ENERGY_GAP, QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.EXCHANGE_INTERGRAL, QuantityKinds.KINETIC_ENERGY, QuantityKinds.DONOR_IONIZATION_ENERGY)) ) );
UNITS_BY_NAME.put( "averageMinutePerCall", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#averageMinutePerCall" ), "averageMinutePerCall", Set.of(new LangString( "average minute per call", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ball", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ball" ), "ball", Set.of(new LangString( "ball", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("AA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "barCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barCubicMetrePerSecond" ), "barCubicMetrePerSecond", Set.of(new LangString( "bar cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bar\u00B7m\u00B3/s"), Optional.of("F92"), Optional.empty(), Optional.of("10\u2075 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "barLitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barLitrePerSecond" ), "barLitrePerSecond", Set.of(new LangString( "bar litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bar\u00B7l/s"), Optional.of("F91"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "barPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barPerBar" ), "barPerBar", Set.of(new LangString( "bar per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bar/bar"), Optional.of("J56"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "barPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barPerKelvin" ), "barPerKelvin", Set.of(new LangString( "bar per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bar/K"), Optional.of("F81"), Optional.empty(), Optional.of("10\u2075 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "barUnitOfPressure", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barUnitOfPressure" ), "barUnitOfPressure", Set.of(new LangString( "bar [unit of pressure]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bar"), Optional.of("BAR"), Optional.of("pascal"), Optional.of("10\u2075 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.STATIC_PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.INSTANTANEOUS_SOUND_PRESSURE, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "barn", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barn" ), "barn", Set.of(new LangString( "barn", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("b"), Optional.of("A14"), Optional.of("squareMetre"), Optional.of("10\u207B\u00B2\u2078 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barnPerElectronvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barnPerElectronvolt" ), "barnPerElectronvolt", Set.of(new LangString( "barn per electronvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("b/eV"), Optional.of("A15"), Optional.of("squareMetrePerJoule"), Optional.of("6.241 51 \u00D7 10\u207B\u00B9\u2070 m\u00B2/J"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barnPerSteradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barnPerSteradian" ), "barnPerSteradian", Set.of(new LangString( "barn per steradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("b/sr"), Optional.of("A17"), Optional.of("squareMetrePerSteradian"), Optional.of("1 \u00D7 10\u207B\u00B2\u2078 m\u00B2/sr"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barnPerSteradianElectronvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barnPerSteradianElectronvolt" ), "barnPerSteradianElectronvolt", Set.of(new LangString( "barn per steradian electronvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("b/(sr\u00B7eV)"), Optional.of("A16"), Optional.empty(), Optional.of("6.241 51 \u00D7 10\u207B\u00B9\u2070 m\u00B2/(sr \u00D7J)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barrelImperial", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelImperial" ), "barrelImperial", Set.of(new LangString( "barrel, imperial", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("B4"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "barrelUkPetroleum", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUkPetroleum" ), "barrelUkPetroleum", Set.of(new LangString( "barrel (UK petroleum)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (UK liq.)"), Optional.of("J57"), Optional.of("cubicMetre"), Optional.of("0.15911315 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUkPetroleumPerDay" ), "barrelUkPetroleumPerDay", Set.of(new LangString( "barrel (UK petroleum) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (UK liq.)/d"), Optional.of("J59"), Optional.of("cubicMetrePerSecond"), Optional.of("1.8415874 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUkPetroleumPerHour" ), "barrelUkPetroleumPerHour", Set.of(new LangString( "barrel (UK petroleum) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (UK liq.)/h"), Optional.of("J60"), Optional.of("cubicMetrePerSecond"), Optional.of("4.419810 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUkPetroleumPerMinute" ), "barrelUkPetroleumPerMinute", Set.of(new LangString( "barrel (UK petroleum) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (UK liq.)/min"), Optional.of("J58"), Optional.of("cubicMetrePerSecond"), Optional.of("2.651886 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUkPetroleumPerSecond" ), "barrelUkPetroleumPerSecond", Set.of(new LangString( "barrel (UK petroleum) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (UK liq.)/s"), Optional.of("J61"), Optional.of("cubicMetrePerSecond"), Optional.of("0.15911315 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUs" ), "barrelUs", Set.of(new LangString( "barrel (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("barrel (US)"), Optional.of("BLL"), Optional.of("cubicMetre"), Optional.of("158.9873 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "barrelUsPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUsPerDay" ), "barrelUsPerDay", Set.of(new LangString( "barrel (US) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("barrel\u00A0(US)/d"), Optional.of("B1"), Optional.of("cubicMetrePerSecond"), Optional.of("1.84013 \u00D7 10\u207B\u2076 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "barrelUsPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUsPerMinute" ), "barrelUsPerMinute", Set.of(new LangString( "barrel (US) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("barrel (US)/min"), Optional.of("5A"), Optional.of("cubicMetrePerSecond"), Optional.of("2.64979 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUsPetroleumPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUsPetroleumPerHour" ), "barrelUsPetroleumPerHour", Set.of(new LangString( "barrel (US petroleum) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (US)/h"), Optional.of("J62"), Optional.of("cubicMetrePerSecond"), Optional.of("4.416314 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUsPetroleumPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#barrelUsPetroleumPerSecond" ), "barrelUsPetroleumPerSecond", Set.of(new LangString( "barrel (US petroleum) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (US)/s"), Optional.of("J63"), Optional.of("cubicMetrePerSecond"), Optional.of("0.1589873 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "baseBox", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#baseBox" ), "baseBox", Set.of(new LangString( "base box", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("BB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "batch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#batch" ), "batch", Set.of(new LangString( "batch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("5B"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "battingPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#battingPound" ), "battingPound", Set.of(new LangString( "batting pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("B3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "baud", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#baud" ), "baud", Set.of(new LangString( "baud", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Bd"), Optional.of("J38"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "beaufort", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#beaufort" ), "beaufort", Set.of(new LangString( "Beaufort", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Bft"), Optional.of("M19"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "becquerel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#becquerel" ), "becquerel", Set.of(new LangString( "becquerel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Bq"), Optional.of("BQL"), Optional.of("curie"), Optional.of("27.027 \u00D7 10\u207B\u00B9\u00B2 Ci"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "becquerelPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#becquerelPerCubicMetre" ), "becquerelPerCubicMetre", Set.of(new LangString( "becquerel per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Bq/m\u00B3"), Optional.of("A19"), Optional.empty(), Optional.of("Bq/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_ACTIVITY, QuantityKinds.ACTIVITY_CONCENTRATION)) ) );
UNITS_BY_NAME.put( "becquerelPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#becquerelPerKilogram" ), "becquerelPerKilogram", Set.of(new LangString( "becquerel per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Bq/kg"), Optional.of("A18"), Optional.of("curiePerKilogram"), Optional.of("27.027 \u00D7 10\u207B\u00B9\u00B2 Ci/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "bel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bel" ), "bel", Set.of(new LangString( "bel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("B"), Optional.of("M72"), Optional.empty(), Optional.of("B"), new HashSet<>(Arrays.asList(QuantityKinds.LEVEL_OF_A_POWER_QUANTITY, QuantityKinds.LEVEL_OF_A_FIELD_QUANTITY)) ) );
UNITS_BY_NAME.put( "belPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#belPerMetre" ), "belPerMetre", Set.of(new LangString( "bel per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("B/m"), Optional.of("P43"), Optional.empty(), Optional.of("B/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_POWER_LEVEL)) ) );
UNITS_BY_NAME.put( "bigPoint", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bigPoint" ), "bigPoint", Set.of(new LangString( "big point", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bp"), Optional.of("H82"), Optional.of("metre"), Optional.of("0.3527778 \u00D7 10\u207B\u00B3 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "billionEur", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#billionEur" ), "billionEur", Set.of(new LangString( "billion (EUR)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("BIL"), Optional.empty(), Optional.of("10\u00B9\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "biot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#biot" ), "biot", Set.of(new LangString( "biot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Bi"), Optional.of("N96"), Optional.of("ampere"), Optional.of("10\u00B9 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_LINKAGE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE)) ) );
UNITS_BY_NAME.put( "bit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bit" ), "bit", Set.of(new LangString( "bit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("b"), Optional.of("A99"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "bitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bitPerCubicMetre" ), "bitPerCubicMetre", Set.of(new LangString( "bit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bit/m\u00B3"), Optional.of("F01"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bitPerMetre" ), "bitPerMetre", Set.of(new LangString( "bit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bit/m"), Optional.of("E88"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bitPerSecond" ), "bitPerSecond", Set.of(new LangString( "bit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bit/s"), Optional.of("B10"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bitPerSquareMetre" ), "bitPerSquareMetre", Set.of(new LangString( "bit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bit/m\u00B2"), Optional.of("E89"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "blank", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#blank" ), "blank", Set.of(new LangString( "blank", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("H21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "boardFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#boardFoot" ), "boardFoot", Set.of(new LangString( "board foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fbm"), Optional.of("BFT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "book", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#book" ), "book", Set.of(new LangString( "book", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("D63"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "brakeHorsePower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#brakeHorsePower" ), "brakeHorsePower", Set.of(new LangString( "brake horse power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BHP"), Optional.of("BHP"), Optional.of("watt"), Optional.of("7.457 \u00D7 10\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "britishThermalUnit39Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnit39Degreesf" ), "britishThermalUnit39Degreesf", Set.of(new LangString( "British thermal unit (39 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btu (39 \u00B0F)"), Optional.of("N66"), Optional.of("joule"), Optional.of("1.05967 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.HEAT, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnit59Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnit59Degreesf" ), "britishThermalUnit59Degreesf", Set.of(new LangString( "British thermal unit (59 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btu (59 \u00B0F)"), Optional.of("N67"), Optional.of("joule"), Optional.of("1.05480 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENERGY, QuantityKinds.HEAT, QuantityKinds.ENTHALPY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnit60Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnit60Degreesf" ), "britishThermalUnit60Degreesf", Set.of(new LangString( "British thermal unit (60 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btu (60 \u00B0F)"), Optional.of("N68"), Optional.of("joule"), Optional.of("1.05468 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTable", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTable" ), "britishThermalUnitInternationalTable", Set.of(new LangString( "British thermal unit (international table)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT"), Optional.of("BTU"), Optional.of("joule"), Optional.of("1.055056 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENTHALPY, QuantityKinds.ENERGY, QuantityKinds.HEAT, QuantityKinds.THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTableFootPerHourSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTableFootPerHourSquareFootDegreeFahrenheit" ), "britishThermalUnitInternationalTableFootPerHourSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) foot per hour\u00A0square foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT\u00B7ft/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J40"), Optional.of("wattPerMetreKelvin"), Optional.of("1.730735 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTableInchPerHourSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTableInchPerHourSquareFootDegreeFahrenheit" ), "britishThermalUnitInternationalTableInchPerHourSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) inch per hour square\u00A0foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT\u00B7in/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J41"), Optional.of("wattPerMetreKelvin"), Optional.of("0.1442279 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTableInchPerSecondSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTableInchPerSecondSquareFootDegreeFahrenheit" ), "britishThermalUnitInternationalTableInchPerSecondSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) inch per second square\u00A0foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT\u00B7in/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J42"), Optional.of("wattPerMetreKelvin"), Optional.of("5.192204 \u00D7 10\u00B2 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerCubicFoot" ), "britishThermalUnitInternationalTablePerCubicFoot", Set.of(new LangString( "British thermal unit (international table) per cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/ft\u00B3"), Optional.of("N58"), Optional.of("joulePerCubicMetre"), Optional.of("3.725895 \u00D710\u2074 J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerDegreeFahrenheit" ), "britishThermalUnitInternationalTablePerDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/\u00B0F"), Optional.of("N60"), Optional.of("joulePerKelvin"), Optional.of("1.899101 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerDegreeRankine" ), "britishThermalUnitInternationalTablePerDegreeRankine", Set.of(new LangString( "British thermal unit (international table) per degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/\u00B0R"), Optional.of("N62"), Optional.of("joulePerKelvin"), Optional.of("1.899101 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerHour" ), "britishThermalUnitInternationalTablePerHour", Set.of(new LangString( "British thermal unit (international table) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/h"), Optional.of("2I"), Optional.of("watt"), Optional.of("2.930711\u00D7 10\u207B\u00B9 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerHourSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerHourSquareFootDegreeFahrenheit" ), "britishThermalUnitInternationalTablePerHourSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) per hour square foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N74"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("5.678263 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerHourSquareFootDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerHourSquareFootDegreeRankine" ), "britishThermalUnitInternationalTablePerHourSquareFootDegreeRankine", Set.of(new LangString( "British thermal unit (international table) per hour square foot degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(h\u00B7ft\u00B2\u00B7\u00B0R)"), Optional.of("A23"), Optional.empty(), Optional.of("5.67826 W/ (m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerMinute" ), "britishThermalUnitInternationalTablePerMinute", Set.of(new LangString( "British thermal unit (international table) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/min"), Optional.of("J44"), Optional.of("watt"), Optional.of("17.584266 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerPound" ), "britishThermalUnitInternationalTablePerPound", Set.of(new LangString( "British thermal unit (international table) per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/lb"), Optional.of("AZ"), Optional.of("joulePerKilogram"), Optional.of("2326 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASSIC_HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerPoundDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerPoundDegreeFahrenheit" ), "britishThermalUnitInternationalTablePerPoundDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) per pound degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(lb\u00B7\u00B0F)"), Optional.of("J43"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.1868 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerPoundDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerPoundDegreeRankine" ), "britishThermalUnitInternationalTablePerPoundDegreeRankine", Set.of(new LangString( "British thermal unit (international table) per pound degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btu/IT(lb\u00B7\u00B0R)"), Optional.of("A21"), Optional.of("joulePerKilogramKelvin"), Optional.of("4186.8 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_SATURATION)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSecond" ), "britishThermalUnitInternationalTablePerSecond", Set.of(new LangString( "British thermal unit (international table) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/s"), Optional.of("J45"), Optional.of("watt"), Optional.of("1.055056 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecondFootDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSecondFootDegreeRankine" ), "britishThermalUnitInternationalTablePerSecondFootDegreeRankine", Set.of(new LangString( "British thermal unit (international table) per second foot degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(s\u00B7ft\u00B7\u00B0R)"), Optional.of("A22"), Optional.of("wattPerMetreKelvin"), Optional.of("6230.64 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecondSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSecondSquareFootDegreeFahrenheit" ), "britishThermalUnitInternationalTablePerSecondSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (international table) per second square foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N76"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("2.044175 \u00D7 10\u2074 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecondSquareFootDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSecondSquareFootDegreeRankine" ), "britishThermalUnitInternationalTablePerSecondSquareFootDegreeRankine", Set.of(new LangString( "British thermal unit (international table) per second square foot degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(s\u00B7ft\u00B2\u00B7\u00B0R)"), Optional.of("A20"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("20441.7 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
}

private static void init1() {
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSquareFoot" ), "britishThermalUnitInternationalTablePerSquareFoot", Set.of(new LangString( "British thermal unit (international table) per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/ft\u00B2"), Optional.of("P37"), Optional.of("joulePerSquareMetre"), Optional.of("1.135653 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareFootHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSquareFootHour" ), "britishThermalUnitInternationalTablePerSquareFootHour", Set.of(new LangString( "British thermal unit (international table) per square foot hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(ft\u00B2\u00B7h)"), Optional.of("N50"), Optional.of("wattPerSquareMetre"), Optional.of("3.154591 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareFootSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSquareFootSecond" ), "britishThermalUnitInternationalTablePerSquareFootSecond", Set.of(new LangString( "British thermal unit (international table) per square foot second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(ft\u00B2\u00B7s)"), Optional.of("N53"), Optional.of("wattPerSquareMetre"), Optional.of("1.135653 \u00D7 10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareInchSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitInternationalTablePerSquareInchSecond" ), "britishThermalUnitInternationalTablePerSquareInchSecond", Set.of(new LangString( "British thermal unit (international table) per square inch second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/(in\u00B2\u00B7s)"), Optional.of("N55"), Optional.of("wattPerSquareMetre"), Optional.of("1.634246 \u00D7 10\u2076 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitMean", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitMean" ), "britishThermalUnitMean", Set.of(new LangString( "British thermal unit (mean)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btu"), Optional.of("J39"), Optional.of("joule"), Optional.of("1.05587 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.HEAT)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalFootPerHourSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalFootPerHourSquareFootDegreeFahrenheit" ), "britishThermalUnitThermochemicalFootPerHourSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) foot per hour square\u00A0foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth\u00B7ft/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J46"), Optional.of("wattPerMetreKelvin"), Optional.of("1.729577 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalInchPerHourSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalInchPerHourSquareFootDegreeFahrenheit" ), "britishThermalUnitThermochemicalInchPerHourSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) inch per hour square\u00A0foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth\u00B7in/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J48"), Optional.of("wattPerMetreKelvin"), Optional.of("0.1441314 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalInchPerSecondSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalInchPerSecondSquareFootDegreeFahrenheit" ), "britishThermalUnitThermochemicalInchPerSecondSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) inch per second\u00A0square foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth\u00B7in/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J49"), Optional.of("wattPerMetreKelvin"), Optional.of("5.188732 \u00D7 10\u00B2 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerCubicFoot" ), "britishThermalUnitThermochemicalPerCubicFoot", Set.of(new LangString( "British thermal unit (thermochemical) per cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/ft\u00B3"), Optional.of("N59"), Optional.of("joulePerCubicMetre"), Optional.of("3.723403 \u00D710\u2074 J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerDegreeFahrenheit" ), "britishThermalUnitThermochemicalPerDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/\u00B0F"), Optional.of("N61"), Optional.of("joulePerKelvin"), Optional.of("1.897830 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerDegreeRankine" ), "britishThermalUnitThermochemicalPerDegreeRankine", Set.of(new LangString( "British thermal unit (thermochemical) per degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/\u00B0R"), Optional.of("N63"), Optional.of("joulePerKelvin"), Optional.of("1.897830 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerHour" ), "britishThermalUnitThermochemicalPerHour", Set.of(new LangString( "British thermal unit (thermochemical) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/h"), Optional.of("J47"), Optional.of("watt"), Optional.of("0.2928751 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerHourSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerHourSquareFootDegreeFahrenheit" ), "britishThermalUnitThermochemicalPerHourSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) per hour square foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N75"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("5.674466 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerMinute" ), "britishThermalUnitThermochemicalPerMinute", Set.of(new LangString( "British thermal unit (thermochemical) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/min"), Optional.of("J51"), Optional.of("watt"), Optional.of("17.57250 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerPound" ), "britishThermalUnitThermochemicalPerPound", Set.of(new LangString( "British thermal unit (thermochemical) per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/lb"), Optional.of("N73"), Optional.of("joulePerKilogram"), Optional.of("2.324444 \u00D7 10\u00B3 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASSIC_HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerPoundDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerPoundDegreeFahrenheit" ), "britishThermalUnitThermochemicalPerPoundDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) per pound degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/(lb\u00B7\u00B0F)"), Optional.of("J50"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.184 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerPoundDegreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerPoundDegreeRankine" ), "britishThermalUnitThermochemicalPerPoundDegreeRankine", Set.of(new LangString( "British thermal unit (thermochemical) per pound degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(Btuth/\u00B0R)/lb"), Optional.of("N64"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.184 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerSecond" ), "britishThermalUnitThermochemicalPerSecond", Set.of(new LangString( "British thermal unit (thermochemical) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/s"), Optional.of("J52"), Optional.of("watt"), Optional.of("1.054350 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSecondSquareFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerSecondSquareFootDegreeFahrenheit" ), "britishThermalUnitThermochemicalPerSecondSquareFootDegreeFahrenheit", Set.of(new LangString( "British thermal unit (thermochemical) per second square foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N77"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("2.042808 \u00D7 10\u2074 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerSquareFoot" ), "britishThermalUnitThermochemicalPerSquareFoot", Set.of(new LangString( "British thermal unit (thermochemical) per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/ft\u00B2"), Optional.of("P38"), Optional.of("joulePerSquareMetre"), Optional.of("1.134893 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFootHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerSquareFootHour" ), "britishThermalUnitThermochemicalPerSquareFootHour", Set.of(new LangString( "British thermal unit (thermochemical) per square foot hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/(ft\u00B2\u00B7h)"), Optional.of("N51"), Optional.of("wattPerSquareMetre"), Optional.of("3.152481 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFootMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerSquareFootMinute" ), "britishThermalUnitThermochemicalPerSquareFootMinute", Set.of(new LangString( "British thermal unit (thermochemical) per square foot minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/(ft\u00B2\u00B7min)"), Optional.of("N52"), Optional.of("wattPerSquareMetre"), Optional.of("1.891489 \u00D7 10\u00B2 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFootSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#britishThermalUnitThermochemicalPerSquareFootSecond" ), "britishThermalUnitThermochemicalPerSquareFootSecond", Set.of(new LangString( "British thermal unit (thermochemical) per square foot second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Btuth/(ft\u00B2\u00B7s)"), Optional.of("N54"), Optional.of("wattPerSquareMetre"), Optional.of("1.134893 \u00D7 10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bulkPack", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bulkPack" ), "bulkPack", Set.of(new LangString( "bulk pack", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk"), Optional.of("AB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bushelUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUk" ), "bushelUk", Set.of(new LangString( "bushel (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bushel (UK)"), Optional.of("BUI"), Optional.of("cubicMetre"), Optional.of("3.636872 \u00D7 10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "bushelUkPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUkPerDay" ), "bushelUkPerDay", Set.of(new LangString( "bushel (UK) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (UK)/d"), Optional.of("J64"), Optional.of("cubicMetrePerSecond"), Optional.of("4.209343 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUkPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUkPerHour" ), "bushelUkPerHour", Set.of(new LangString( "bushel (UK) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (UK)/h"), Optional.of("J65"), Optional.of("cubicMetrePerSecond"), Optional.of("1.010242 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUkPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUkPerMinute" ), "bushelUkPerMinute", Set.of(new LangString( "bushel (UK) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (UK)/min"), Optional.of("J66"), Optional.of("cubicMetrePerSecond"), Optional.of("6.061453 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUkPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUkPerSecond" ), "bushelUkPerSecond", Set.of(new LangString( "bushel (UK) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (UK)/s"), Optional.of("J67"), Optional.of("cubicMetrePerSecond"), Optional.of("3.636872 \u00D7 10\u207B\u00B2 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUs" ), "bushelUs", Set.of(new LangString( "bushel (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (US)"), Optional.of("BUA"), Optional.of("cubicMetre"), Optional.of("3.523907 \u00D7 10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUsDryPerDay" ), "bushelUsDryPerDay", Set.of(new LangString( "bushel (US dry) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (US dry)/d"), Optional.of("J68"), Optional.of("cubicMetrePerSecond"), Optional.of("4.078596 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUsDryPerHour" ), "bushelUsDryPerHour", Set.of(new LangString( "bushel (US dry) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (US dry)/h"), Optional.of("J69"), Optional.of("cubicMetrePerSecond"), Optional.of("9.788631 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUsDryPerMinute" ), "bushelUsDryPerMinute", Set.of(new LangString( "bushel (US dry) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (US dry)/min"), Optional.of("J70"), Optional.of("cubicMetrePerSecond"), Optional.of("5.873178 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bushelUsDryPerSecond" ), "bushelUsDryPerSecond", Set.of(new LangString( "bushel (US dry) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bu (US dry)/s"), Optional.of("J71"), Optional.of("cubicMetrePerSecond"), Optional.of("3.523907 \u00D7 10\u207B\u00B2 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "byte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#byte" ), "byte", Set.of(new LangString( "byte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("B"), Optional.of("AD"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "bytePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#bytePerSecond" ), "bytePerSecond", Set.of(new LangString( "byte per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("byte/s"), Optional.of("P93"), Optional.empty(), Optional.of("byte/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cake", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cake" ), "cake", Set.of(new LangString( "cake", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "call", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#call" ), "call", Set.of(new LangString( "call", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("C0"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "calorie20Degreesc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorie20Degreesc" ), "calorie20Degreesc", Set.of(new LangString( "calorie (20 \u00B0C)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cal\u2082\u2080"), Optional.of("N69"), Optional.of("joule"), Optional.of("4.18190"), new HashSet<>(Arrays.asList(QuantityKinds.ENTHALPY, QuantityKinds.ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT)) ) );
UNITS_BY_NAME.put( "calorieInternationalTablePerGramDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieInternationalTablePerGramDegreeCelsius" ), "calorieInternationalTablePerGramDegreeCelsius", Set.of(new LangString( "calorie (international table) per gram degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calIT/(g\u00B7\u00B0C)"), Optional.of("J76"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.1868 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "calorieMean", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieMean" ), "calorieMean", Set.of(new LangString( "calorie (mean)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cal"), Optional.of("J75"), Optional.of("joule"), Optional.of("4.19002 J"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerCentimetreSecondDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerCentimetreSecondDegreeCelsius" ), "calorieThermochemicalPerCentimetreSecondDegreeCelsius", Set.of(new LangString( "calorie (thermochemical) per centimetre second degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/(cm\u00B7s\u00B7\u00B0C)"), Optional.of("J78"), Optional.of("wattPerMetreKelvin"), Optional.of("4.184 \u00D7 10\u00B2 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerGramDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerGramDegreeCelsius" ), "calorieThermochemicalPerGramDegreeCelsius", Set.of(new LangString( "calorie (thermochemical) per gram degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/(g\u00B7\u00B0C)"), Optional.of("J79"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.184 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerMinute" ), "calorieThermochemicalPerMinute", Set.of(new LangString( "calorie (thermochemical) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/min"), Optional.of("J81"), Optional.of("watt"), Optional.of("6.973333 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerSecond" ), "calorieThermochemicalPerSecond", Set.of(new LangString( "calorie (thermochemical) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/s"), Optional.of("J82"), Optional.of("watt"), Optional.of("4.184 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerSquareCentimetre" ), "calorieThermochemicalPerSquareCentimetre", Set.of(new LangString( "calorie (thermochemical) per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/cm\u00B2"), Optional.of("P39"), Optional.of("joulePerSquareMetre"), Optional.of("4.184 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSquareCentimetreMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerSquareCentimetreMinute" ), "calorieThermochemicalPerSquareCentimetreMinute", Set.of(new LangString( "calorie (thermochemical) per square centimetre minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/(cm\u00B2\u00B7min)"), Optional.of("N56"), Optional.of("wattPerSquareMetre"), Optional.of("6.973333 \u00D7 10\u00B2 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSquareCentimetreSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#calorieThermochemicalPerSquareCentimetreSecond" ), "calorieThermochemicalPerSquareCentimetreSecond", Set.of(new LangString( "calorie (thermochemical) per square centimetre second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("calth/(cm\u00B2\u00B7s)"), Optional.of("N57"), Optional.of("wattPerSquareMetre"), Optional.of("4.184 \u00D7 10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "candela", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#candela" ), "candela", Set.of(new LangString( "candela", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cd"), Optional.of("CDL"), Optional.empty(), Optional.of("cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "candelaPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#candelaPerSquareFoot" ), "candelaPerSquareFoot", Set.of(new LangString( "candela per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cd/ft\u00B2"), Optional.of("P32"), Optional.of("candelaPerSquareMetre"), Optional.of("1.076391 \u00D7 10 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "candelaPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#candelaPerSquareInch" ), "candelaPerSquareInch", Set.of(new LangString( "candela per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cd/in\u00B2"), Optional.of("P28"), Optional.of("candelaPerSquareMetre"), Optional.of("1.550003 \u00D7 10\u00B3 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "candelaPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#candelaPerSquareMetre" ), "candelaPerSquareMetre", Set.of(new LangString( "candela per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cd/m\u00B2"), Optional.of("A24"), Optional.empty(), Optional.of("cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "card", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#card" ), "card", Set.of(new LangString( "card", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "carryingCapacityInMetricTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#carryingCapacityInMetricTon" ), "carryingCapacityInMetricTon", Set.of(new LangString( "carrying capacity in metric ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CCT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "centalUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centalUk" ), "centalUk", Set.of(new LangString( "cental (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CNT"), Optional.of("kilogram"), Optional.of("45.359237 kg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "centigram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centigram" ), "centigram", Set.of(new LangString( "centigram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cg"), Optional.of("CGM"), Optional.of("kilogram"), Optional.of("10\u207B\u2075 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "centilitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centilitre" ), "centilitre", Set.of(new LangString( "centilitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cl"), Optional.of("CLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "centimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetre" ), "centimetre", Set.of(new LangString( "centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm"), Optional.of("CMT"), Optional.of("metre"), Optional.of("10\u207B\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "centimetreOfMercury0Degreesc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetreOfMercury0Degreesc" ), "centimetreOfMercury0Degreesc", Set.of(new LangString( "centimetre of mercury (0 \u00B0C)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cmHg (0 \u00B0C)"), Optional.of("N13"), Optional.of("pascal"), Optional.of("1.33322 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "centimetreOfWater4Degreesc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetreOfWater4Degreesc" ), "centimetreOfWater4Degreesc", Set.of(new LangString( "centimetre of water (4 \u00B0C)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cmH\u2082O (4 \u00B0C)"), Optional.of("N14"), Optional.of("pascal"), Optional.of("9.80638 \u00D7 10 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "centimetrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerBar" ), "centimetrePerBar", Set.of(new LangString( "centimetre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm/bar"), Optional.of("G04"), Optional.empty(), Optional.of("10\u207B\u2077 kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "centimetrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerHour" ), "centimetrePerHour", Set.of(new LangString( "centimetre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm/h"), Optional.of("H49"), Optional.empty(), Optional.of("0.277777778 \u00D7 10\u207B\u2076 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerKelvin" ), "centimetrePerKelvin", Set.of(new LangString( "centimetre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm/K"), Optional.of("F51"), Optional.empty(), Optional.of("10\u207B\u00B2 m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.LENGTH)) ) );
UNITS_BY_NAME.put( "centimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerSecond" ), "centimetrePerSecond", Set.of(new LangString( "centimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm/s"), Optional.of("2M"), Optional.of("metrePerSecond"), Optional.of("10\u207B\u00B2 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerSecondBar" ), "centimetrePerSecondBar", Set.of(new LangString( "centimetre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(cm/s)/bar"), Optional.of("J85"), Optional.of("metrePerSecondPascal"), Optional.of("10\u207B\u2077 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerSecondKelvin" ), "centimetrePerSecondKelvin", Set.of(new LangString( "centimetre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(cm/s)/K"), Optional.of("J84"), Optional.of("metrePerSecondKelvin"), Optional.of("10\u207B\u00B2 (m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centimetrePerSecondSquared" ), "centimetrePerSecondSquared", Set.of(new LangString( "centimetre per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm/s\u00B2"), Optional.of("M39"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u00B2 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "centinewtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centinewtonMetre" ), "centinewtonMetre", Set.of(new LangString( "centinewton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cN\u00B7m"), Optional.of("J72"), Optional.of("newtonMetre"), Optional.of("10\u207B\u00B2 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "centipoise", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centipoise" ), "centipoise", Set.of(new LangString( "centipoise", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cP"), Optional.of("C7"), Optional.of("pascalSecond"), Optional.of("10\u207B\u00B3 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "centipoisePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centipoisePerBar" ), "centipoisePerBar", Set.of(new LangString( "centipoise per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cP/bar"), Optional.of("J74"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2078 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "centipoisePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centipoisePerKelvin" ), "centipoisePerKelvin", Set.of(new LangString( "centipoise per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cP/K"), Optional.of("J73"), Optional.empty(), Optional.of("10\u207B\u00B3 Pa \u00D7 s/K"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "centistokes", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#centistokes" ), "centistokes", Set.of(new LangString( "centistokes", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cSt"), Optional.of("4C"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "chainBasedOnUsSurveyFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#chainBasedOnUsSurveyFoot" ), "chainBasedOnUsSurveyFoot", Set.of(new LangString( "chain (based on U.S. survey foot)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ch (US survey)"), Optional.of("M49"), Optional.of("metre"), Optional.of("2.011684 \u00D7 10 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "circularMil", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#circularMil" ), "circularMil", Set.of(new LangString( "circular mil", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cmil"), Optional.of("M47"), Optional.of("squareMetre"), Optional.of("5.067075 \u00D7 10\u207B\u00B9\u2070 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "clo", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#clo" ), "clo", Set.of(new LangString( "clo", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("clo"), Optional.of("J83"), Optional.of("squareMetreKelvinPerWatt"), Optional.of("0.155 m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "coilGroup", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coilGroup" ), "coilGroup", Set.of(new LangString( "coil group", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("C9"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "commonYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#commonYear" ), "commonYear", Set.of(new LangString( "common year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("y (365 days)"), Optional.of("L95"), Optional.of("secondUnitOfTime"), Optional.of("3.1536 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "contentGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#contentGram" ), "contentGram", Set.of(new LangString( "content gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CTG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "contentTonMetric", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#contentTonMetric" ), "contentTonMetric", Set.of(new LangString( "content ton (metric)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CTN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "conventionalMetreOfWater", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#conventionalMetreOfWater" ), "conventionalMetreOfWater", Set.of(new LangString( "conventional metre of water", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mH\u2082O"), Optional.of("N23"), Optional.of("pascal"), Optional.of("9.80665 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "cord", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cord" ), "cord", Set.of(new LangString( "cord", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("WCD"), Optional.of("cubicMetre"), Optional.of("3.63 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cord128Ft3", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cord128Ft3" ), "cord128Ft3", Set.of(new LangString( "cord (128 ft3)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cord"), Optional.of("M68"), Optional.of("cubicMetre"), Optional.of("3.624556 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "coulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulomb" ), "coulomb", Set.of(new LangString( "coulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C"), Optional.of("COU"), Optional.of("ampereSecond"), Optional.of("A \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELEMENTARY_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY)) ) );
UNITS_BY_NAME.put( "coulombMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombMetre" ), "coulombMetre", Set.of(new LangString( "coulomb metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C\u00B7m"), Optional.of("A26"), Optional.empty(), Optional.of("A \u00D7 s \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_DIPOLE_MOMENT_OF_MOLECULE, QuantityKinds.ELECTRIC_DIPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "coulombMetreSquaredPerVolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombMetreSquaredPerVolt" ), "coulombMetreSquaredPerVolt", Set.of(new LangString( "coulomb metre squared per volt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C\u00B7m\u00B2/V"), Optional.of("A27"), Optional.empty(), Optional.of("A\u00B2 \u00D7 s\u2074/kg"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_POLARIZABILITY_OF_A_MOLECULE)) ) );
UNITS_BY_NAME.put( "coulombPerCubicCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerCubicCentimetre" ), "coulombPerCubicCentimetre", Set.of(new LangString( "coulomb per cubic centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/cm\u00B3"), Optional.of("A28"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2076 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "coulombPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerCubicMetre" ), "coulombPerCubicMetre", Set.of(new LangString( "coulomb per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/m\u00B3"), Optional.of("A29"), Optional.empty(), Optional.of("C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "coulombPerCubicMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerCubicMillimetre" ), "coulombPerCubicMillimetre", Set.of(new LangString( "coulomb per cubic millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/mm\u00B3"), Optional.of("A30"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2079 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE)) ) );
UNITS_BY_NAME.put( "coulombPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerKilogram" ), "coulombPerKilogram", Set.of(new LangString( "coulomb per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/kg"), Optional.of("CKG"), Optional.empty(), Optional.of("A \u00D7 s/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "coulombPerKilogramSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerKilogramSecond" ), "coulombPerKilogramSecond", Set.of(new LangString( "coulomb per kilogram second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/(kg\u00B7s)"), Optional.of("A31"), Optional.of("amperePerKilogram"), Optional.of("A/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE_RATE)) ) );
UNITS_BY_NAME.put( "coulombPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerMetre" ), "coulombPerMetre", Set.of(new LangString( "coulomb per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/m"), Optional.of("P10"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 s \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_CHARGE)) ) );
UNITS_BY_NAME.put( "coulombPerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerMole" ), "coulombPerMole", Set.of(new LangString( "coulomb per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/mol"), Optional.of("A32"), Optional.empty(), Optional.of("A \u00D7 s/mol"), new HashSet<>(Arrays.asList(QuantityKinds.FARADAY_CONSTANT)) ) );
UNITS_BY_NAME.put( "coulombPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerSquareCentimetre" ), "coulombPerSquareCentimetre", Set.of(new LangString( "coulomb per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/cm\u00B2"), Optional.of("A33"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u2074 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "coulombPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerSquareMetre" ), "coulombPerSquareMetre", Set.of(new LangString( "coulomb per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/m\u00B2"), Optional.of("A34"), Optional.empty(), Optional.of("C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "coulombPerSquareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombPerSquareMillimetre" ), "coulombPerSquareMillimetre", Set.of(new LangString( "coulomb per square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C/mm\u00B2"), Optional.of("A35"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u2076 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "coulombSquareMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#coulombSquareMetrePerKilogram" ), "coulombSquareMetrePerKilogram", Set.of(new LangString( "coulomb square metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("C\u00B7m\u00B2/kg"), Optional.of("J53"), Optional.empty(), Optional.of("C \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "credit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#credit" ), "credit", Set.of(new LangString( "credit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("B17"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cubicCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetre" ), "cubicCentimetre", Set.of(new LangString( "cubic centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3"), Optional.of("CMQ"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerBar" ), "cubicCentimetrePerBar", Set.of(new LangString( "cubic centimetre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/bar"), Optional.of("G94"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerCubicMetre" ), "cubicCentimetrePerCubicMetre", Set.of(new LangString( "cubic centimetre per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/m\u00B3"), Optional.of("J87"), Optional.empty(), Optional.of("10\u207B\u2076"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
}

private static void init2() {
UNITS_BY_NAME.put( "cubicCentimetrePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerDay" ), "cubicCentimetrePerDay", Set.of(new LangString( "cubic centimetre per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/d"), Optional.of("G47"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerDayBar" ), "cubicCentimetrePerDayBar", Set.of(new LangString( "cubic centimetre per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(d\u00B7bar)"), Optional.of("G78"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2076 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerDayKelvin" ), "cubicCentimetrePerDayKelvin", Set.of(new LangString( "cubic centimetre per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(d\u00B7K)"), Optional.of("G61"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerHour" ), "cubicCentimetrePerHour", Set.of(new LangString( "cubic centimetre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/h"), Optional.of("G48"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerHourBar" ), "cubicCentimetrePerHourBar", Set.of(new LangString( "cubic centimetre per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(h\u00B7bar)"), Optional.of("G79"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2075 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerHourKelvin" ), "cubicCentimetrePerHourKelvin", Set.of(new LangString( "cubic centimetre per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(h\u00B7K)"), Optional.of("G62"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerKelvin" ), "cubicCentimetrePerKelvin", Set.of(new LangString( "cubic centimetre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/K"), Optional.of("G27"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerMinute" ), "cubicCentimetrePerMinute", Set.of(new LangString( "cubic centimetre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/min"), Optional.of("G49"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerMinuteBar" ), "cubicCentimetrePerMinuteBar", Set.of(new LangString( "cubic centimetre per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(min\u00B7bar)"), Optional.of("G80"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B3 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerMinuteKelvin" ), "cubicCentimetrePerMinuteKelvin", Set.of(new LangString( "cubic centimetre per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(min\u00B7K)"), Optional.of("G63"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerMole" ), "cubicCentimetrePerMole", Set.of(new LangString( "cubic centimetre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/mol"), Optional.of("A36"), Optional.of("cubicMetrePerMole"), Optional.of("10\u207B\u2076 m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerSecond" ), "cubicCentimetrePerSecond", Set.of(new LangString( "cubic centimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/s"), Optional.of("2J"), Optional.of("cubicMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerSecondBar" ), "cubicCentimetrePerSecondBar", Set.of(new LangString( "cubic centimetre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(s\u00B7bar)"), Optional.of("G81"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicCentimetrePerSecondKelvin" ), "cubicCentimetrePerSecondKelvin", Set.of(new LangString( "cubic centimetre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B3/(s\u00B7K)"), Optional.of("G64"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecametre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecametre" ), "cubicDecametre", Set.of(new LangString( "cubic decametre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dam\u00B3"), Optional.of("DMA"), Optional.of("cubicMetre"), Optional.of("10\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetre" ), "cubicDecimetre", Set.of(new LangString( "cubic decimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3"), Optional.of("DMQ"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerCubicMetre" ), "cubicDecimetrePerCubicMetre", Set.of(new LangString( "cubic decimetre per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/m\u00B3"), Optional.of("J91"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerDay" ), "cubicDecimetrePerDay", Set.of(new LangString( "cubic decimetre per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/d"), Optional.of("J90"), Optional.of("cubicMetrePerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerHour" ), "cubicDecimetrePerHour", Set.of(new LangString( "cubic decimetre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/h"), Optional.of("E92"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerKilogram" ), "cubicDecimetrePerKilogram", Set.of(new LangString( "cubic decimetre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/kg"), Optional.of("N28"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerMinute" ), "cubicDecimetrePerMinute", Set.of(new LangString( "cubic decimetre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/min"), Optional.of("J92"), Optional.of("cubicMetrePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerMole" ), "cubicDecimetrePerMole", Set.of(new LangString( "cubic decimetre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/mol"), Optional.of("A37"), Optional.of("cubicMetrePerMole"), Optional.of("10\u207B\u00B3 m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicDecimetrePerSecond" ), "cubicDecimetrePerSecond", Set.of(new LangString( "cubic decimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B3/s"), Optional.of("J93"), Optional.of("cubicMetrePerSecond"), Optional.of("10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFoot" ), "cubicFoot", Set.of(new LangString( "cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3"), Optional.of("FTQ"), Optional.of("cubicMetre"), Optional.of("2.831685 \u00D7 10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerDay" ), "cubicFootPerDay", Set.of(new LangString( "cubic foot per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/d"), Optional.of("K22"), Optional.of("cubicMetrePerSecond"), Optional.of("3.277413 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFootPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerDegreeFahrenheit" ), "cubicFootPerDegreeFahrenheit", Set.of(new LangString( "cubic foot per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/\u00B0F"), Optional.of("K21"), Optional.of("cubicMetrePerKelvin"), Optional.of("5.097033 \u00D7 10\u207B\u00B2 m\u00B3/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerHour" ), "cubicFootPerHour", Set.of(new LangString( "cubic foot per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/h"), Optional.of("2K"), Optional.of("cubicMetrePerSecond"), Optional.of("7.86579 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFootPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerMinute" ), "cubicFootPerMinute", Set.of(new LangString( "cubic foot per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/min"), Optional.of("2L"), Optional.of("cubicMetrePerSecond"), Optional.of("4.719474 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFootPerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerPound" ), "cubicFootPerPound", Set.of(new LangString( "cubic foot per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/lb"), Optional.of("N29"), Optional.of("cubicMetrePerKilogram"), Optional.of("6.242796 \u00D7 10\u207B\u00B2 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerPsi" ), "cubicFootPerPsi", Set.of(new LangString( "cubic foot per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/psi"), Optional.of("K23"), Optional.of("cubicMetrePerPascal"), Optional.of("4.107012 \u00D7 10\u207B\u2076 m\u00B3/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicFootPerSecond" ), "cubicFootPerSecond", Set.of(new LangString( "cubic foot per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B3/s"), Optional.of("E17"), Optional.of("cubicMetrePerSecond"), Optional.of("2.831685 \u00D7 10\u207B\u00B2 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cubicHectometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicHectometre" ), "cubicHectometre", Set.of(new LangString( "cubic hectometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hm\u00B3"), Optional.of("H19"), Optional.of("cubicMetre"), Optional.of("10\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicInch" ), "cubicInch", Set.of(new LangString( "cubic inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B3"), Optional.of("INQ"), Optional.of("cubicMetre"), Optional.of("16.387064 \u00D7 10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME, QuantityKinds.SECTION_MODULUS)) ) );
UNITS_BY_NAME.put( "cubicInchPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicInchPerHour" ), "cubicInchPerHour", Set.of(new LangString( "cubic inch per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B3/h"), Optional.of("G56"), Optional.empty(), Optional.of("4.55196 \u00D7 10\u207B\u2079 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicInchPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicInchPerMinute" ), "cubicInchPerMinute", Set.of(new LangString( "cubic inch per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B3/min"), Optional.of("G57"), Optional.empty(), Optional.of("2.73118 \u00D7 10\u207B\u2077 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicInchPerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicInchPerPound" ), "cubicInchPerPound", Set.of(new LangString( "cubic inch per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B3/lb"), Optional.of("N30"), Optional.of("cubicMetrePerKilogram"), Optional.of("3.612728 \u00D7 10\u207B\u2075 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicInchPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicInchPerSecond" ), "cubicInchPerSecond", Set.of(new LangString( "cubic inch per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B3/s"), Optional.of("G58"), Optional.empty(), Optional.of("1.63871 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicKilometre" ), "cubicKilometre", Set.of(new LangString( "cubic kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km\u00B3"), Optional.of("H20"), Optional.of("cubicMetre"), Optional.of("10\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetre" ), "cubicMetre", Set.of(new LangString( "cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3"), Optional.of("MTQ"), Optional.empty(), Optional.of("m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME, QuantityKinds.SECTION_MODULUS)) ) );
UNITS_BY_NAME.put( "cubicMetrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerBar" ), "cubicMetrePerBar", Set.of(new LangString( "cubic metre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/bar"), Optional.of("G96"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerCoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerCoulomb" ), "cubicMetrePerCoulomb", Set.of(new LangString( "cubic metre per coulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/C"), Optional.of("A38"), Optional.empty(), Optional.of("m\u00B3/A \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.HALL_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "cubicMetrePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerCubicMetre" ), "cubicMetrePerCubicMetre", Set.of(new LangString( "cubic metre per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/m\u00B3"), Optional.of("H60"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "cubicMetrePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerDay" ), "cubicMetrePerDay", Set.of(new LangString( "cubic metre per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/d"), Optional.of("G52"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerDayBar" ), "cubicMetrePerDayBar", Set.of(new LangString( "cubic metre per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(d\u00B7bar)"), Optional.of("G86"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2070 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerDayKelvin" ), "cubicMetrePerDayKelvin", Set.of(new LangString( "cubic metre per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(d\u00B7K)"), Optional.of("G69"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerHour" ), "cubicMetrePerHour", Set.of(new LangString( "cubic metre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/h"), Optional.of("MQH"), Optional.of("cubicMetrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerHourBar" ), "cubicMetrePerHourBar", Set.of(new LangString( "cubic metre per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(h\u00B7bar)"), Optional.of("G87"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2079 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerHourKelvin" ), "cubicMetrePerHourKelvin", Set.of(new LangString( "cubic metre per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(h\u00B7K)"), Optional.of("G70"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerKelvin" ), "cubicMetrePerKelvin", Set.of(new LangString( "cubic metre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/K"), Optional.of("G29"), Optional.empty(), Optional.of("m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerKilogram" ), "cubicMetrePerKilogram", Set.of(new LangString( "cubic metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/kg"), Optional.of("A39"), Optional.empty(), Optional.of("m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerMinute" ), "cubicMetrePerMinute", Set.of(new LangString( "cubic metre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/min"), Optional.of("G53"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerMinuteBar" ), "cubicMetrePerMinuteBar", Set.of(new LangString( "cubic metre per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(min\u00B7bar)"), Optional.of("G88"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2077 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerMinuteKelvin" ), "cubicMetrePerMinuteKelvin", Set.of(new LangString( "cubic metre per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(min\u00B7K)"), Optional.of("G71"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerMole" ), "cubicMetrePerMole", Set.of(new LangString( "cubic metre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/mol"), Optional.of("A40"), Optional.empty(), Optional.of("m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerPascal" ), "cubicMetrePerPascal", Set.of(new LangString( "cubic metre per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/Pa"), Optional.of("M71"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerSecond" ), "cubicMetrePerSecond", Set.of(new LangString( "cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/s"), Optional.of("MQS"), Optional.empty(), Optional.of("m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE, QuantityKinds.RECOMBINATION_COEFFICIENT, QuantityKinds.INSTANTANEOUS_VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerSecondBar" ), "cubicMetrePerSecondBar", Set.of(new LangString( "cubic metre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(s\u00B7bar)"), Optional.of("G89"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerSecondKelvin" ), "cubicMetrePerSecondKelvin", Set.of(new LangString( "cubic metre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B3/(s\u00B7K)"), Optional.of("G72"), Optional.empty(), Optional.of("m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerSecondPascal" ), "cubicMetrePerSecondPascal", Set.of(new LangString( "cubic metre per second pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(m\u00B3/s)/Pa"), Optional.of("N45"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMetrePerSecondSquareMetre" ), "cubicMetrePerSecondSquareMetre", Set.of(new LangString( "cubic metre per second square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(m\u00B3/s)/m\u00B2"), Optional.of("P87"), Optional.of("metrePerSecond"), Optional.of("m/s"), new HashSet<>(Arrays.asList(QuantityKinds.POROSITY)) ) );
UNITS_BY_NAME.put( "cubicMicrometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMicrometre" ), "cubicMicrometre", Set.of(new LangString( "cubic micrometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5m\u00B3"), Optional.empty(), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B9\u2078 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMileUkStatute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMileUkStatute" ), "cubicMileUkStatute", Set.of(new LangString( "cubic mile (UK statute)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi\u00B3"), Optional.of("M69"), Optional.of("cubicMetre"), Optional.of("4.168182 \u00D7 10\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMillimetre" ), "cubicMillimetre", Set.of(new LangString( "cubic millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm\u00B3"), Optional.of("MMQ"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMillimetrePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicMillimetrePerCubicMetre" ), "cubicMillimetrePerCubicMetre", Set.of(new LangString( "cubic millimetre per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm\u00B3/m\u00B3"), Optional.of("L21"), Optional.empty(), Optional.of("10\u2079"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "cubicYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYard" ), "cubicYard", Set.of(new LangString( "cubic yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3"), Optional.of("YDQ"), Optional.of("cubicMetre"), Optional.of("0.764555 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicYardPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYardPerDay" ), "cubicYardPerDay", Set.of(new LangString( "cubic yard per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3/d"), Optional.of("M12"), Optional.of("cubicMetrePerSecond"), Optional.of("8.849015 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicYardPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYardPerDegreeFahrenheit" ), "cubicYardPerDegreeFahrenheit", Set.of(new LangString( "cubic yard per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3/\u00B0F"), Optional.of("M11"), Optional.of("cubicMetrePerKelvin"), Optional.of("1.376199 m\u00B3/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicYardPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYardPerHour" ), "cubicYardPerHour", Set.of(new LangString( "cubic yard per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3/h"), Optional.of("M13"), Optional.of("cubicMetrePerSecond"), Optional.of("2.123764 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicYardPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYardPerMinute" ), "cubicYardPerMinute", Set.of(new LangString( "cubic yard per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3/min"), Optional.of("M15"), Optional.of("cubicMetrePerSecond"), Optional.of("1.274258 \u00D7 10\u207B\u00B2 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicYardPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYardPerPsi" ), "cubicYardPerPsi", Set.of(new LangString( "cubic yard per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3/psi"), Optional.of("M14"), Optional.of("cubicMetrePerPascal"), Optional.of("1.108893 \u00D7 10\u207B\u2074 m\u00B3/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicYardPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cubicYardPerSecond" ), "cubicYardPerSecond", Set.of(new LangString( "cubic yard per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B3/s"), Optional.of("M16"), Optional.of("cubicMetrePerSecond"), Optional.of("0.7645549 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cupUnitOfVolume", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cupUnitOfVolume" ), "cupUnitOfVolume", Set.of(new LangString( "cup [unit of volume]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cup (US)"), Optional.of("G21"), Optional.of("cubicMetre"), Optional.of("2.365882 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "curie", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#curie" ), "curie", Set.of(new LangString( "curie", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Ci"), Optional.of("CUR"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u00B9\u2070 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "curiePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#curiePerKilogram" ), "curiePerKilogram", Set.of(new LangString( "curie per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Ci/kg"), Optional.of("A42"), Optional.of("becquerelPerKilogram"), Optional.of("3.7 \u00D7 10\u00B9\u2070 Bq/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "cycle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#cycle" ), "cycle", Set.of(new LangString( "cycle", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("B7"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "day", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#day" ), "day", Set.of(new LangString( "day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("d"), Optional.of("DAY"), Optional.of("secondUnitOfTime"), Optional.of("86400 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "deadweightTonnage", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#deadweightTonnage" ), "deadweightTonnage", Set.of(new LangString( "deadweight tonnage", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dwt"), Optional.of("A43"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "decade", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decade" ), "decade", Set.of(new LangString( "decade", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DEC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "decadeLogarithmic", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decadeLogarithmic" ), "decadeLogarithmic", Set.of(new LangString( "decade (logarithmic)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dec"), Optional.of("P41"), Optional.empty(), Optional.of("dec"), new HashSet<>(Arrays.asList(QuantityKinds.LOGARITHMIC_DECREMENT)) ) );
UNITS_BY_NAME.put( "decagram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decagram" ), "decagram", Set.of(new LangString( "decagram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dag"), Optional.of("DJ"), Optional.of("kilogram"), Optional.of("10\u207B\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "decalitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decalitre" ), "decalitre", Set.of(new LangString( "decalitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dal"), Optional.of("A44"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "decametre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decametre" ), "decametre", Set.of(new LangString( "decametre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dam"), Optional.of("A45"), Optional.of("metre"), Optional.of("10 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES)) ) );
UNITS_BY_NAME.put( "decapascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decapascal" ), "decapascal", Set.of(new LangString( "decapascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("daPa"), Optional.of("H75"), Optional.of("pascal"), Optional.of("10\u00B9 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "decare", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decare" ), "decare", Set.of(new LangString( "decare", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("daa"), Optional.of("DAA"), Optional.of("squareMetre"), Optional.of("10\u00B3 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "decibel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decibel" ), "decibel", Set.of(new LangString( "decibel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dB"), Optional.of("2N"), Optional.of("neper"), Optional.of("0.1151293 Np"), new HashSet<>(Arrays.asList(QuantityKinds.LEVEL_OF_A_POWER_QUANTITY, QuantityKinds.SOUND_POWER_LEVEL, QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_REDUCTION_INDEX, QuantityKinds.LEVEL_OF_A_FIELD_QUANTITY)) ) );
UNITS_BY_NAME.put( "decibelPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decibelPerKilometre" ), "decibelPerKilometre", Set.of(new LangString( "decibel per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dB/km"), Optional.of("H51"), Optional.of("belPerMetre"), Optional.of("10\u207B\u2074 B/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_POWER_LEVEL)) ) );
UNITS_BY_NAME.put( "decibelPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decibelPerMetre" ), "decibelPerMetre", Set.of(new LangString( "decibel per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dB/m"), Optional.of("H52"), Optional.of("belPerMetre"), Optional.of("10\u207B\u00B9 B/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_POWER_LEVEL)) ) );
UNITS_BY_NAME.put( "decigram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decigram" ), "decigram", Set.of(new LangString( "decigram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dg"), Optional.of("DG"), Optional.of("kilogram"), Optional.of("10\u207B\u2074 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "decilitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decilitre" ), "decilitre", Set.of(new LangString( "decilitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dl"), Optional.of("DLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "decilitrePerGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decilitrePerGram" ), "decilitrePerGram", Set.of(new LangString( "decilitre per gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dl/g"), Optional.of("22"), Optional.of("cubicMetrePerKilogram"), Optional.of("10\u207B\u00B9 \u00D7 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "decimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decimetre" ), "decimetre", Set.of(new LangString( "decimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm"), Optional.of("DMT"), Optional.of("metre"), Optional.of("10\u207B\u00B9 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "decinewtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decinewtonMetre" ), "decinewtonMetre", Set.of(new LangString( "decinewton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dN\u00B7m"), Optional.of("DN"), Optional.of("newtonMetre"), Optional.of("10\u207B\u00B9 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "decitex", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decitex" ), "decitex", Set.of(new LangString( "decitex", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dtex (g/10km)"), Optional.of("A47"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "decitonne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#decitonne" ), "decitonne", Set.of(new LangString( "decitonne", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dt or dtn"), Optional.of("DTN"), Optional.of("kilogram"), Optional.of("10\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "degreeApi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeApi" ), "degreeApi", Set.of(new LangString( "degree API", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0API"), Optional.of("J13"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBalling", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeBalling" ), "degreeBalling", Set.of(new LangString( "degree Balling", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0Balling"), Optional.of("J17"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBaumeOriginScale", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeBaumeOriginScale" ), "degreeBaumeOriginScale", Set.of(new LangString( "degree Baume (origin scale)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0B\u00E9"), Optional.of("J14"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBaumeUsHeavy", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeBaumeUsHeavy" ), "degreeBaumeUsHeavy", Set.of(new LangString( "degree Baume (US heavy)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0B\u00E9 (US heavy)"), Optional.of("J15"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBaumeUsLight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeBaumeUsLight" ), "degreeBaumeUsLight", Set.of(new LangString( "degree Baume (US light)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0B\u00E9 (US light)"), Optional.of("J16"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBrix", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeBrix" ), "degreeBrix", Set.of(new LangString( "degree Brix", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0Bx"), Optional.of("J18"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}

private static void init3() {
UNITS_BY_NAME.put( "degreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeCelsius" ), "degreeCelsius", Set.of(new LangString( "degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0C"), Optional.of("CEL"), Optional.of("kelvin"), Optional.of("1 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeCelsiusPerBar" ), "degreeCelsiusPerBar", Set.of(new LangString( "degree Celsius per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0C/bar"), Optional.of("F60"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeCelsiusPerHour" ), "degreeCelsiusPerHour", Set.of(new LangString( "degree Celsius per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0C/h"), Optional.of("H12"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 s\u207B\u00B9 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeCelsiusPerKelvin" ), "degreeCelsiusPerKelvin", Set.of(new LangString( "degree Celsius per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0C/K"), Optional.of("E98"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeCelsiusPerMinute" ), "degreeCelsiusPerMinute", Set.of(new LangString( "degree Celsius per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0C/min"), Optional.of("H13"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 s\u207B\u00B9 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeCelsiusPerSecond" ), "degreeCelsiusPerSecond", Set.of(new LangString( "degree Celsius per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0C/s"), Optional.of("H14"), Optional.empty(), Optional.of("s\u207B\u00B9 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeDay" ), "degreeDay", Set.of(new LangString( "degree day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("deg da"), Optional.of("E10"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheit" ), "degreeFahrenheit", Set.of(new LangString( "degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F"), Optional.of("FAH"), Optional.empty(), Optional.of("5/9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.FAHRENHEIT_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourPerBritishThermalUnitInternationalTable", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitHourPerBritishThermalUnitInternationalTable" ), "degreeFahrenheitHourPerBritishThermalUnitInternationalTable", Set.of(new LangString( "degree Fahrenheit hour per British thermal unit (international table)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/(BtuIT/h)"), Optional.of("N84"), Optional.of("kelvinPerWatt"), Optional.of("1.895634 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourPerBritishThermalUnitThermochemical", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitHourPerBritishThermalUnitThermochemical" ), "degreeFahrenheitHourPerBritishThermalUnitThermochemical", Set.of(new LangString( "degree Fahrenheit hour per British thermal unit (thermochemical)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/(Btuth/h)"), Optional.of("N85"), Optional.of("kelvinPerWatt"), Optional.of("1.896903 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTable", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTable" ), "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTable", Set.of(new LangString( "degree Fahrenheit hour square foot per British thermal unit (international table)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/BtuIT"), Optional.of("J22"), Optional.of("squareMetreKelvinPerWatt"), Optional.of("0.1761102 m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTableInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTableInch" ), "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTableInch", Set.of(new LangString( "degree Fahrenheit hour square foot per British thermal unit (international table) inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/(BtuIT\u00B7in)"), Optional.of("N88"), Optional.of("kelvinMetrePerWatt"), Optional.of("6.933472 K \u00D7 m/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemical", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemical" ), "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemical", Set.of(new LangString( "degree Fahrenheit hour square foot per British thermal unit (thermochemical)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/Btuth"), Optional.of("J19"), Optional.of("squareMetreKelvinPerWatt"), Optional.of("0.176228 m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemicalInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemicalInch" ), "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemicalInch", Set.of(new LangString( "degree Fahrenheit hour square foot per British thermal unit (thermochemical) inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/(Btuth\u00B7in)"), Optional.of("N89"), Optional.of("kelvinMetrePerWatt"), Optional.of("6.938112 K \u00D7 m/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitPerBar" ), "degreeFahrenheitPerBar", Set.of(new LangString( "degree Fahrenheit per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/bar"), Optional.of("J21"), Optional.of("kelvinPerPascal"), Optional.of("0.5555556 \u00D7 10\u207B\u2075 K/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitPerHour" ), "degreeFahrenheitPerHour", Set.of(new LangString( "degree Fahrenheit per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/h"), Optional.of("J23"), Optional.of("kelvinPerSecond"), Optional.of("1.543210 \u00D7 10\u207B\u2074 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitPerKelvin" ), "degreeFahrenheitPerKelvin", Set.of(new LangString( "degree Fahrenheit per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/K"), Optional.of("J20"), Optional.empty(), Optional.of("0.5555556"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitPerMinute" ), "degreeFahrenheitPerMinute", Set.of(new LangString( "degree Fahrenheit per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/min"), Optional.of("J24"), Optional.of("kelvinPerSecond"), Optional.of("9.259259 \u00D7 10\u207B\u00B3 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitPerSecond" ), "degreeFahrenheitPerSecond", Set.of(new LangString( "degree Fahrenheit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/s"), Optional.of("J25"), Optional.of("kelvinPerSecond"), Optional.of("0.5555556 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitSecondPerBritishThermalUnitInternationalTable", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitSecondPerBritishThermalUnitInternationalTable" ), "degreeFahrenheitSecondPerBritishThermalUnitInternationalTable", Set.of(new LangString( "degree Fahrenheit second per British thermal unit (international table)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/(BtuIT/s)"), Optional.of("N86"), Optional.of("kelvinPerWatt"), Optional.of("5.265651 \u00D7 10\u207B\u2074 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitSecondPerBritishThermalUnitThermochemical", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeFahrenheitSecondPerBritishThermalUnitThermochemical" ), "degreeFahrenheitSecondPerBritishThermalUnitThermochemical", Set.of(new LangString( "degree Fahrenheit second per British thermal unit (thermochemical)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0F/(Btuth/s)"), Optional.of("N87"), Optional.of("kelvinPerWatt"), Optional.of("5.269175 \u00D7 10\u207B\u2074 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeOechsle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeOechsle" ), "degreeOechsle", Set.of(new LangString( "degree Oechsle", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0Oechsle"), Optional.of("J27"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreePerMetre" ), "degreePerMetre", Set.of(new LangString( "degree per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0/m"), Optional.of("H27"), Optional.of("radianPerMetre"), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "degreePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreePerSecond" ), "degreePerSecond", Set.of(new LangString( "degree per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0/s"), Optional.of("E96"), Optional.empty(), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "degreePlato", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreePlato" ), "degreePlato", Set.of(new LangString( "degree Plato", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0P"), Optional.of("PLA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeRankine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeRankine" ), "degreeRankine", Set.of(new LangString( "degree Rankine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0R"), Optional.of("A48"), Optional.empty(), Optional.of("5/9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeRankinePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeRankinePerHour" ), "degreeRankinePerHour", Set.of(new LangString( "degree Rankine per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0R/h"), Optional.of("J28"), Optional.of("kelvinPerSecond"), Optional.of("1.543210 \u00D7 10\u207B\u2074 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeRankinePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeRankinePerMinute" ), "degreeRankinePerMinute", Set.of(new LangString( "degree Rankine per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0R/min"), Optional.of("J29"), Optional.of("kelvinPerSecond"), Optional.of("9.259259 \u00D7 10\u207B\u00B3 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeRankinePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeRankinePerSecond" ), "degreeRankinePerSecond", Set.of(new LangString( "degree Rankine per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0R/s"), Optional.of("J30"), Optional.of("kelvinPerSecond"), Optional.of("0.5555556 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeTwaddell", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeTwaddell" ), "degreeTwaddell", Set.of(new LangString( "degree Twaddell", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0Tw"), Optional.of("J31"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeUnitOfAngle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeUnitOfAngle" ), "degreeUnitOfAngle", Set.of(new LangString( "degree [unit of angle]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0"), Optional.of("DD"), Optional.of("radian"), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad"), new HashSet<>(Arrays.asList(QuantityKinds.BRAGG_ANGLE, QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "degreeUnitOfAnglePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#degreeUnitOfAnglePerSecondSquared" ), "degreeUnitOfAnglePerSecondSquared", Set.of(new LangString( "degree [unit of angle] per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B0/s\u00B2"), Optional.of("M45"), Optional.empty(), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad / s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_ACCELERATION)) ) );
UNITS_BY_NAME.put( "denier", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#denier" ), "denier", Set.of(new LangString( "denier", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("den (g/9 km)"), Optional.of("M83"), Optional.of("kilogramPerMetre"), Optional.of("1.111111 \u00D7 10\u207B\u2077 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "digit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#digit" ), "digit", Set.of(new LangString( "digit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("B19"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dioptre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dioptre" ), "dioptre", Set.of(new LangString( "dioptre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dpt"), Optional.of("Q25"), Optional.of("reciprocalMetre"), Optional.of("m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "displacementTonnage", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#displacementTonnage" ), "displacementTonnage", Set.of(new LangString( "displacement tonnage", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DPT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dose", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dose" ), "dose", Set.of(new LangString( "dose", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E27"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dotsPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dotsPerInch" ), "dotsPerInch", Set.of(new LangString( "dots per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dpi"), Optional.of("E39"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dozen" ), "dozen", Set.of(new LangString( "dozen", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("DOZ"), Optional.of("DZN"), Optional.empty(), Optional.of("12"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenPack", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dozenPack" ), "dozenPack", Set.of(new LangString( "dozen pack", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DZP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenPair", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dozenPair" ), "dozenPair", Set.of(new LangString( "dozen pair", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DPR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenPiece", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dozenPiece" ), "dozenPiece", Set.of(new LangString( "dozen piece", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DPC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenRoll", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dozenRoll" ), "dozenRoll", Set.of(new LangString( "dozen roll", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DRL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dramUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dramUk" ), "dramUk", Set.of(new LangString( "dram (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DRI"), Optional.of("gram"), Optional.of("1.771745 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dramUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dramUs" ), "dramUs", Set.of(new LangString( "dram (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DRA"), Optional.of("gram"), Optional.of("3.887935 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dryBarrelUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dryBarrelUs" ), "dryBarrelUs", Set.of(new LangString( "dry barrel (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("bbl (US)"), Optional.of("BLD"), Optional.of("cubicMetre"), Optional.of("1.15627 \u00D7 10\u207B\u00B9 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryGallonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dryGallonUs" ), "dryGallonUs", Set.of(new LangString( "dry gallon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dry gal (US)"), Optional.of("GLD"), Optional.of("cubicMetre"), Optional.of("4.404884 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryPintUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dryPintUs" ), "dryPintUs", Set.of(new LangString( "dry pint (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dry pt (US)"), Optional.of("PTD"), Optional.of("cubicMetre"), Optional.of("5.506105 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dryPound" ), "dryPound", Set.of(new LangString( "dry pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dryQuartUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dryQuartUs" ), "dryQuartUs", Set.of(new LangString( "dry quart (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dry qt (US)"), Optional.of("QTD"), Optional.of("cubicMetre"), Optional.of("1.101221 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dryTon" ), "dryTon", Set.of(new LangString( "dry ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dyneMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#dyneMetre" ), "dyneMetre", Set.of(new LangString( "dyne metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dyn\u00B7m"), Optional.of("M97"), Optional.of("newtonMetre"), Optional.of("10\u207B\u2075 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "each", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#each" ), "each", Set.of(new LangString( "each", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("EA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "eightPartCloudCover", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#eightPartCloudCover" ), "eightPartCloudCover", Set.of(new LangString( "8-part cloud cover", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("A59"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "electronicMailBox", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#electronicMailBox" ), "electronicMailBox", Set.of(new LangString( "electronic mail box", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("EB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "electronvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#electronvolt" ), "electronvolt", Set.of(new LangString( "electronvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("eV"), Optional.of("A53"), Optional.of("joule"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.BETA_DISINTEGRATION_ENERGY, QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.RESONANCE_ENERGY, QuantityKinds.FERMI_ENERGY, QuantityKinds.ENERGY, QuantityKinds.MAXIMUM_BETA_PARTICLE_ENERGY)) ) );
UNITS_BY_NAME.put( "electronvoltPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#electronvoltPerMetre" ), "electronvoltPerMetre", Set.of(new LangString( "electronvolt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("eV/m"), Optional.of("A54"), Optional.of("joulePerMetre"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J/m"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_LINEAR_STOPPING_POWER, QuantityKinds.LINEAR_ENERGY_TRANSFER)) ) );
UNITS_BY_NAME.put( "electronvoltSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#electronvoltSquareMetre" ), "electronvoltSquareMetre", Set.of(new LangString( "electronvolt square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("eV\u00B7m\u00B2"), Optional.of("A55"), Optional.of("jouleSquareMetre"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_ATOMIC_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "electronvoltSquareMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#electronvoltSquareMetrePerKilogram" ), "electronvoltSquareMetrePerKilogram", Set.of(new LangString( "electronvolt square metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("eV\u00B7m\u00B2/kg"), Optional.of("A56"), Optional.of("jouleSquareMetrePerKilogram"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_MASS_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "equivalentGallon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#equivalentGallon" ), "equivalentGallon", Set.of(new LangString( "equivalent gallon", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("EQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "erlang", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#erlang" ), "erlang", Set.of(new LangString( "erlang", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("E"), Optional.of("Q11"), Optional.empty(), Optional.of("1 E"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exabitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exabitPerSecond" ), "exabitPerSecond", Set.of(new LangString( "exabit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Ebit/s"), Optional.of("E58"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exabyte" ), "exabyte", Set.of(new LangString( "Exabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("EB"), Optional.empty(), Optional.of("byte"), Optional.of("10\u00B9\u2078 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "exajoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exajoule" ), "exajoule", Set.of(new LangString( "exajoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("EJ"), Optional.of("A68"), Optional.of("joule"), Optional.of("10\u00B9\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "exbibitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exbibitPerCubicMetre" ), "exbibitPerCubicMetre", Set.of(new LangString( "exbibit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Eibit/m\u00B3"), Optional.of("E67"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exbibitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exbibitPerMetre" ), "exbibitPerMetre", Set.of(new LangString( "exbibit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Eibit/m"), Optional.of("E65"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exbibitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exbibitPerSquareMetre" ), "exbibitPerSquareMetre", Set.of(new LangString( "exbibit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Eibit/m\u00B2"), Optional.of("E66"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exbibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#exbibyte" ), "exbibyte", Set.of(new LangString( "exbibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Eibyte"), Optional.of("E59"), Optional.of("byte"), Optional.of("2\u2076\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "failuresInTime", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#failuresInTime" ), "failuresInTime", Set.of(new LangString( "failures in time", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("FIT"), Optional.of("FIT"), Optional.of("reciprocalSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B3 s\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "farad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#farad" ), "farad", Set.of(new LangString( "farad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("F"), Optional.of("FAR"), Optional.empty(), Optional.of("F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "faradPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#faradPerKilometre" ), "faradPerKilometre", Set.of(new LangString( "farad per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("F/km"), Optional.of("H33"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u00B3 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CONSTANT, QuantityKinds.PERMITTIVITY, QuantityKinds.PERMITTIVITY_OF_VACUUM)) ) );
UNITS_BY_NAME.put( "faradPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#faradPerMetre" ), "faradPerMetre", Set.of(new LangString( "farad per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("F/m"), Optional.of("A69"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u207B\u00B3 \u00D7 s\u2074 \u00D7 A\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.PERMITTIVITY, QuantityKinds.ELECTRIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "fathom", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fathom" ), "fathom", Set.of(new LangString( "fathom", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fth"), Optional.of("AK"), Optional.of("metre"), Optional.of("1.8288 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "femtojoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#femtojoule" ), "femtojoule", Set.of(new LangString( "femtojoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fJ"), Optional.of("A70"), Optional.of("joule"), Optional.of("10\u207B\u00B9\u2075 J"), new HashSet<>(Arrays.asList(QuantityKinds.GAP_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "femtometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#femtometre" ), "femtometre", Set.of(new LangString( "femtometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fm"), Optional.of("A71"), Optional.of("metre"), Optional.of("10\u207B\u00B9\u2075 m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.DIAMETER)) ) );
UNITS_BY_NAME.put( "fibreMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fibreMetre" ), "fibreMetre", Set.of(new LangString( "fibre metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("FBM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "fivePack", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fivePack" ), "fivePack", Set.of(new LangString( "five pack", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("P5"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "fixedRate", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fixedRate" ), "fixedRate", Set.of(new LangString( "fixed rate", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("1I"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "flakeTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#flakeTon" ), "flakeTon", Set.of(new LangString( "flake ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("FL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "fluidOunceUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fluidOunceUk" ), "fluidOunceUk", Set.of(new LangString( "fluid ounce (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (UK)"), Optional.of("OZI"), Optional.of("cubicMetre"), Optional.of("2.841306 \u00D7 10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "fluidOunceUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fluidOunceUs" ), "fluidOunceUs", Set.of(new LangString( "fluid ounce (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (US)"), Optional.of("OZA"), Optional.of("cubicMetre"), Optional.of("2.957353 \u00D7 10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "foot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#foot" ), "foot", Set.of(new LangString( "foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft"), Optional.of("FOT"), Optional.of("metre"), Optional.of("0.3048 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES)) ) );
UNITS_BY_NAME.put( "footOfWater392Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footOfWater392Degreesf" ), "footOfWater392Degreesf", Set.of(new LangString( "foot of water (39.2 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ftH\u2082O (39,2 \u00B0F)"), Optional.of("N15"), Optional.of("pascal"), Optional.of("2.98898 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "footPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerDegreeFahrenheit" ), "footPerDegreeFahrenheit", Set.of(new LangString( "foot per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft/\u00B0F"), Optional.of("K13"), Optional.of("metrePerKelvin"), Optional.of("0.54864 m/K"), new HashSet<>(Arrays.asList(QuantityKinds.BREADTH, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "footPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerHour" ), "footPerHour", Set.of(new LangString( "foot per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft/h"), Optional.of("K14"), Optional.of("metrePerSecond"), Optional.of("8.466667 \u00D7 10\u207B\u2075m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerMinute" ), "footPerMinute", Set.of(new LangString( "foot per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft/min"), Optional.of("FR"), Optional.of("metrePerSecond"), Optional.of("5.08 \u00D7 10\u207B\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerPsi" ), "footPerPsi", Set.of(new LangString( "foot per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft/psi"), Optional.of("K17"), Optional.of("metrePerPascal"), Optional.of("4.420750 \u00D7 10\u207B\u2075 m/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES)) ) );
UNITS_BY_NAME.put( "footPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerSecond" ), "footPerSecond", Set.of(new LangString( "foot per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft/s"), Optional.of("FS"), Optional.of("metrePerSecond"), Optional.of("0.3048 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerSecondDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerSecondDegreeFahrenheit" ), "footPerSecondDegreeFahrenheit", Set.of(new LangString( "foot per second degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(ft/s)/\u00B0F"), Optional.of("K18"), Optional.of("metrePerSecondKelvin"), Optional.of("0.54864 (m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerSecondPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerSecondPsi" ), "footPerSecondPsi", Set.of(new LangString( "foot per second psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(ft/s)/psi"), Optional.of("K19"), Optional.of("metrePerSecondPascal"), Optional.of("4.420750 \u00D7 10\u207B\u2075 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerSecondSquared" ), "footPerSecondSquared", Set.of(new LangString( "foot per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft/s\u00B2"), Optional.of("A73"), Optional.of("metrePerSecondSquared"), Optional.of("0.3048 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "footPerThousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPerThousand" ), "footPerThousand", Set.of(new LangString( "foot per thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E33"), Optional.of("metre"), Optional.of("3.048 \u00D7 10\u207B\u2074 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "footPoundForce", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPoundForce" ), "footPoundForce", Set.of(new LangString( "foot pound-force", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B7lbf"), Optional.of("85"), Optional.of("joule"), Optional.of("1.355818 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "footPoundForcePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPoundForcePerHour" ), "footPoundForcePerHour", Set.of(new LangString( "foot pound-force per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B7lbf/h"), Optional.of("K15"), Optional.of("watt"), Optional.of("3.766161 \u00D7 10\u207B\u2074 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "footPoundForcePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPoundForcePerMinute" ), "footPoundForcePerMinute", Set.of(new LangString( "foot pound-force per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B7lbf/min"), Optional.of("K16"), Optional.of("watt"), Optional.of("2.259697 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "footPoundForcePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPoundForcePerSecond" ), "footPoundForcePerSecond", Set.of(new LangString( "foot pound-force per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B7lbf/s"), Optional.of("A74"), Optional.of("watt"), Optional.of("1.355818 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "footPoundal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footPoundal" ), "footPoundal", Set.of(new LangString( "foot poundal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B7pdl"), Optional.of("N46"), Optional.of("joule"), Optional.of("4.214011 \u00D7 10\u207B\u00B2 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "footToTheFourthPower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footToTheFourthPower" ), "footToTheFourthPower", Set.of(new LangString( "foot to the fourth power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u2074"), Optional.of("N27"), Optional.of("metreToTheFourthPower"), Optional.of("8.630975 \u00D7 10\u207B\u00B3 m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_POLAR_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "footUsSurvey", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footUsSurvey" ), "footUsSurvey", Set.of(new LangString( "foot (U.S. survey)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft (US survey)"), Optional.of("M51"), Optional.of("metre"), Optional.of("3.048006 \u00D7 10\u207B\u00B9 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "footcandle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footcandle" ), "footcandle", Set.of(new LangString( "footcandle", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ftc"), Optional.of("P27"), Optional.empty(), Optional.of("1.076391 \u00D7 10\u00B9 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
}

private static void init4() {
UNITS_BY_NAME.put( "footlambert", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#footlambert" ), "footlambert", Set.of(new LangString( "footlambert", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ftL"), Optional.of("P29"), Optional.of("candelaPerSquareMetre"), Optional.of("3.426259 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "fortyFootContainer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#fortyFootContainer" ), "fortyFootContainer", Set.of(new LangString( "forty foot container", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "franklin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#franklin" ), "franklin", Set.of(new LangString( "franklin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Fr"), Optional.of("N94"), Optional.of("coulomb"), Optional.of("3.335641 \u00D7 10\u207B\u00B9\u2070 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "freightTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#freightTon" ), "freightTon", Set.of(new LangString( "freight ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("A75"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "frenchGauge", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#frenchGauge" ), "frenchGauge", Set.of(new LangString( "French gauge", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Fg"), Optional.of("H79"), Optional.of("metre"), Optional.of("0.333333333 \u00D7 10\u207B\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "furlong", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#furlong" ), "furlong", Set.of(new LangString( "furlong", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fur"), Optional.of("M50"), Optional.of("metre"), Optional.of("2.01168 \u00D7 10\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.LENGTH)) ) );
UNITS_BY_NAME.put( "gal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gal" ), "gal", Set.of(new LangString( "gal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gal"), Optional.of("A76"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u00B2 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "gallonUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUk" ), "gallonUk", Set.of(new LangString( "gallon (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (UK)"), Optional.of("GLI"), Optional.of("cubicMetre"), Optional.of("4.546092 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "gallonUkPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUkPerDay" ), "gallonUkPerDay", Set.of(new LangString( "gallon (UK) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (UK)/d"), Optional.of("K26"), Optional.of("cubicMetrePerSecond"), Optional.of("5.261678 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUkPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUkPerHour" ), "gallonUkPerHour", Set.of(new LangString( "gallon (UK) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (UK)/h"), Optional.of("K27"), Optional.of("cubicMetrePerSecond"), Optional.of("1.262803 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUkPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUkPerSecond" ), "gallonUkPerSecond", Set.of(new LangString( "gallon (UK) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (UK)/s"), Optional.of("K28"), Optional.of("cubicMetrePerSecond"), Optional.of("4.54609 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUs" ), "gallonUs", Set.of(new LangString( "gallon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (US)"), Optional.of("GLL"), Optional.of("cubicMetre"), Optional.of("3.785412 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "gallonUsLiquidPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUsLiquidPerSecond" ), "gallonUsLiquidPerSecond", Set.of(new LangString( "gallon (US liquid) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (US liq.)/s"), Optional.of("K30"), Optional.of("cubicMetrePerSecond"), Optional.of("3.785412 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUsPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUsPerDay" ), "gallonUsPerDay", Set.of(new LangString( "gallon (US) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (US)/d"), Optional.of("GB"), Optional.of("cubicMetrePerSecond"), Optional.of("4.381264 \u00D7 10\u207B\u2078 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gallonUsPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gallonUsPerHour" ), "gallonUsPerHour", Set.of(new LangString( "gallon (US) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal/h"), Optional.of("G50"), Optional.empty(), Optional.of("1.0515 \u00D7 10\u207B\u2076 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gamma", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gamma" ), "gamma", Set.of(new LangString( "gamma", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u03B3"), Optional.of("P12"), Optional.of("tesla"), Optional.of("10\u207B\u2079 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "gibibit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gibibit" ), "gibibit", Set.of(new LangString( "gibibit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gibit"), Optional.of("B30"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gibibitPerCubicMetre" ), "gibibitPerCubicMetre", Set.of(new LangString( "gibibit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gibit/m\u00B3"), Optional.of("E71"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gibibitPerMetre" ), "gibibitPerMetre", Set.of(new LangString( "gibibit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gibit/m"), Optional.of("E69"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gibibitPerSquareMetre" ), "gibibitPerSquareMetre", Set.of(new LangString( "gibibit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gibit/m\u00B2"), Optional.of("E70"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gibibyte" ), "gibibyte", Set.of(new LangString( "Gibibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gibyte"), Optional.of("E62"), Optional.of("byte"), Optional.of("2\u00B3\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "gigabecquerel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigabecquerel" ), "gigabecquerel", Set.of(new LangString( "gigabecquerel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GBq"), Optional.of("GBQ"), Optional.of("becquerel"), Optional.of("10\u2079 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "gigabit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigabit" ), "gigabit", Set.of(new LangString( "gigabit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gbit"), Optional.of("B68"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gigabitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigabitPerSecond" ), "gigabitPerSecond", Set.of(new LangString( "gigabit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gbit/s"), Optional.of("B80"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gigabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigabyte" ), "gigabyte", Set.of(new LangString( "Gigabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GB"), Optional.of("E34"), Optional.of("byte"), Optional.of("10\u2079 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "gigabytePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigabytePerSecond" ), "gigabytePerSecond", Set.of(new LangString( "gigabyte per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gbyte/s"), Optional.of("E68"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gigacoulombPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigacoulombPerCubicMetre" ), "gigacoulombPerCubicMetre", Set.of(new LangString( "gigacoulomb per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GC/m\u00B3"), Optional.of("A84"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2079 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "gigaelectronvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigaelectronvolt" ), "gigaelectronvolt", Set.of(new LangString( "gigaelectronvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GeV"), Optional.of("A85"), Optional.of("electronvolt"), Optional.of("10\u2079 eV"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.POTENTIAL_ENERGY)) ) );
UNITS_BY_NAME.put( "gigahertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigahertz" ), "gigahertz", Set.of(new LangString( "gigahertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GHz"), Optional.of("A86"), Optional.of("hertz"), Optional.of("10\u2079 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "gigahertzMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigahertzMetre" ), "gigahertzMetre", Set.of(new LangString( "gigahertz metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GHz\u00B7m"), Optional.of("M18"), Optional.of("hertzMetre"), Optional.of("10\u2079 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "gigajoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigajoule" ), "gigajoule", Set.of(new LangString( "gigajoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GJ"), Optional.of("GV"), Optional.of("joule"), Optional.of("10\u2079 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "gigaohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigaohm" ), "gigaohm", Set.of(new LangString( "gigaohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("G\u2126"), Optional.of("A87"), Optional.of("ohm"), Optional.of("10\u2079 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.REACTANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.IMPEDANCE)) ) );
UNITS_BY_NAME.put( "gigaohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigaohmMetre" ), "gigaohmMetre", Set.of(new LangString( "gigaohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("G\u2126\u00B7m"), Optional.of("A88"), Optional.of("ohmMetre"), Optional.of("10\u2079 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "gigaohmPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigaohmPerMetre" ), "gigaohmPerMetre", Set.of(new LangString( "gigaohm per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("G\u2126/m"), Optional.of("M26"), Optional.of("ohmPerMetre"), Optional.of("10\u2079 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_LOAD_PER_UNIT_LENGTH)) ) );
UNITS_BY_NAME.put( "gigapascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigapascal" ), "gigapascal", Set.of(new LangString( "gigapascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GPa"), Optional.of("A89"), Optional.of("pascal"), Optional.of("10\u2079 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "gigawatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigawatt" ), "gigawatt", Set.of(new LangString( "gigawatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GW"), Optional.of("A90"), Optional.of("watt"), Optional.of("10\u2079 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER, QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "gigawattHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gigawattHour" ), "gigawattHour", Set.of(new LangString( "gigawatt hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("GW\u00B7h"), Optional.of("GWH"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u00B9\u00B2 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "gilbert", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gilbert" ), "gilbert", Set.of(new LangString( "gilbert", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gi"), Optional.of("N97"), Optional.of("ampere"), Optional.of("7.957747 \u00D7 10\u207B\u00B9 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_LINKAGE, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE)) ) );
UNITS_BY_NAME.put( "gillUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUk" ), "gillUk", Set.of(new LangString( "gill (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (UK)"), Optional.of("GII"), Optional.of("cubicMetre"), Optional.of("1.420653 \u00D7 10\u207B\u2074 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gillUkPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUkPerDay" ), "gillUkPerDay", Set.of(new LangString( "gill (UK) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (UK)/d"), Optional.of("K32"), Optional.of("cubicMetrePerSecond"), Optional.of("1.644274 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUkPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUkPerHour" ), "gillUkPerHour", Set.of(new LangString( "gill (UK) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (UK)/h"), Optional.of("K33"), Optional.of("cubicMetrePerSecond"), Optional.of("3.946258 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUkPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUkPerMinute" ), "gillUkPerMinute", Set.of(new LangString( "gill (UK) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (UK)/min"), Optional.of("K34"), Optional.of("cubicMetrePerSecond"), Optional.of("0.02367755 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUkPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUkPerSecond" ), "gillUkPerSecond", Set.of(new LangString( "gill (UK) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (UK)/s"), Optional.of("K35"), Optional.of("cubicMetrePerSecond"), Optional.of("1.420653 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUs" ), "gillUs", Set.of(new LangString( "gill (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (US)"), Optional.of("GIA"), Optional.of("cubicMetre"), Optional.of("1.182941 \u00D7 10\u207B\u2074 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gillUsPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUsPerDay" ), "gillUsPerDay", Set.of(new LangString( "gill (US) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (US)/d"), Optional.of("K36"), Optional.of("cubicMetrePerSecond"), Optional.of("1.369145 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUsPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUsPerHour" ), "gillUsPerHour", Set.of(new LangString( "gill (US) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (US)/h"), Optional.of("K37"), Optional.of("cubicMetrePerSecond"), Optional.of("3.285947 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUsPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUsPerMinute" ), "gillUsPerMinute", Set.of(new LangString( "gill (US) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (US)/min"), Optional.of("K38"), Optional.of("cubicMetrePerSecond"), Optional.of("1.971568 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUsPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gillUsPerSecond" ), "gillUsPerSecond", Set.of(new LangString( "gill (US) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi (US)/s"), Optional.of("K39"), Optional.of("cubicMetrePerSecond"), Optional.of("1.182941 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gon" ), "gon", Set.of(new LangString( "gon", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gon"), Optional.of("A91"), Optional.of("radian"), Optional.of("1.570796 \u00D7 10\u207B\u00B2 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "grain", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#grain" ), "grain", Set.of(new LangString( "grain", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gr"), Optional.of("GRN"), Optional.of("kilogram"), Optional.of("64.79891 \u00D7 10\u207B\u2076 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "grainPerGallonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#grainPerGallonUs" ), "grainPerGallonUs", Set.of(new LangString( "grain per gallon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gr/gal (US)"), Optional.of("K41"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.711806 \u00D7 10\u207B\u00B2 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gram" ), "gram", Set.of(new LangString( "gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g"), Optional.of("GRM"), Optional.of("kilogram"), Optional.of("10\u207B\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "gramCentimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramCentimetrePerSecond" ), "gramCentimetrePerSecond", Set.of(new LangString( "gram centimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g\u00B7(cm/s)"), Optional.of("M99"), Optional.of("kilogramMetrePerSecond"), Optional.of("10\u207B\u2075 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "gramDryWeight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramDryWeight" ), "gramDryWeight", Set.of(new LangString( "gram, dry weight", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("GDW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramForcePerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramForcePerSquareCentimetre" ), "gramForcePerSquareCentimetre", Set.of(new LangString( "gram-force per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gf/cm\u00B2"), Optional.of("K31"), Optional.of("pascal"), Optional.of("98.0665 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "gramIncludingContainer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramIncludingContainer" ), "gramIncludingContainer", Set.of(new LangString( "gram, including container", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("GIC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramIncludingInnerPackaging", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramIncludingInnerPackaging" ), "gramIncludingInnerPackaging", Set.of(new LangString( "gram, including inner packaging", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("GIP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramMillimetre" ), "gramMillimetre", Set.of(new LangString( "gram millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g\u00B7mm"), Optional.of("H84"), Optional.of("kilogramMetre"), Optional.of("10\u207B\u2076 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "gramOfFissileIsotope", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramOfFissileIsotope" ), "gramOfFissileIsotope", Set.of(new LangString( "gram of fissile isotope", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gi F/S"), Optional.of("GFI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerBar" ), "gramPerBar", Set.of(new LangString( "gram per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/bar"), Optional.of("F74"), Optional.empty(), Optional.of("10\u207B\u2078 m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCentimetreSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCentimetreSecond" ), "gramPerCentimetreSecond", Set.of(new LangString( "gram per centimetre second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(cm\u00B7s)"), Optional.of("N41"), Optional.of("pascalSecond"), Optional.of("0.1 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicCentimetre" ), "gramPerCubicCentimetre", Set.of(new LangString( "gram per cubic centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/cm\u00B3"), Optional.of("23"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "gramPerCubicCentimetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicCentimetreBar" ), "gramPerCubicCentimetreBar", Set.of(new LangString( "gram per cubic centimetre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(cm\u00B3\u00B7bar)"), Optional.of("G11"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicCentimetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicCentimetreKelvin" ), "gramPerCubicCentimetreKelvin", Set.of(new LangString( "gram per cubic centimetre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(cm\u00B3\u00B7K)"), Optional.of("G33"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicDecimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicDecimetre" ), "gramPerCubicDecimetre", Set.of(new LangString( "gram per cubic decimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/dm\u00B3"), Optional.of("F23"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicDecimetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicDecimetreBar" ), "gramPerCubicDecimetreBar", Set.of(new LangString( "gram per cubic decimetre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(dm\u00B3\u00B7bar)"), Optional.of("G12"), Optional.empty(), Optional.of("10\u207B\u2075 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicDecimetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicDecimetreKelvin" ), "gramPerCubicDecimetreKelvin", Set.of(new LangString( "gram per cubic decimetre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(dm\u00B3\u00B7K)"), Optional.of("G34"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicMetre" ), "gramPerCubicMetre", Set.of(new LangString( "gram per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/m\u00B3"), Optional.of("A93"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicMetreBar" ), "gramPerCubicMetreBar", Set.of(new LangString( "gram per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(m\u00B3\u00B7bar)"), Optional.of("G14"), Optional.empty(), Optional.of("10\u207B\u2078 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerCubicMetreKelvin" ), "gramPerCubicMetreKelvin", Set.of(new LangString( "gram per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(m\u00B3\u00B7K)"), Optional.of("G36"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerDay" ), "gramPerDay", Set.of(new LangString( "gram per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/d"), Optional.of("F26"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerDayBar" ), "gramPerDayBar", Set.of(new LangString( "gram per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(d\u00B7bar)"), Optional.of("F62"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B3 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerDayKelvin" ), "gramPerDayKelvin", Set.of(new LangString( "gram per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(d\u00B7K)"), Optional.of("F35"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerHertz" ), "gramPerHertz", Set.of(new LangString( "gram per hertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/Hz"), Optional.of("F25"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerHour" ), "gramPerHour", Set.of(new LangString( "gram per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/h"), Optional.of("F27"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerHourBar" ), "gramPerHourBar", Set.of(new LangString( "gram per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(h\u00B7bar)"), Optional.of("F63"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B2 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerHourKelvin" ), "gramPerHourKelvin", Set.of(new LangString( "gram per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(h\u00B7K)"), Optional.of("F36"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerKelvin" ), "gramPerKelvin", Set.of(new LangString( "gram per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/K"), Optional.of("F14"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerLitre" ), "gramPerLitre", Set.of(new LangString( "gram per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/l"), Optional.of("GL"), Optional.of("kilogramPerCubicMetre"), Optional.of("kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerLitreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerLitreBar" ), "gramPerLitreBar", Set.of(new LangString( "gram per litre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(l\u00B7bar)"), Optional.of("G13"), Optional.empty(), Optional.of("10\u207B\u2075 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerLitreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerLitreKelvin" ), "gramPerLitreKelvin", Set.of(new LangString( "gram per litre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(l\u00B7K)"), Optional.of("G35"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMetreGramPer100Centimetres", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMetreGramPer100Centimetres" ), "gramPerMetreGramPer100Centimetres", Set.of(new LangString( "gram per metre (gram per 100 centimetres)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/m"), Optional.of("GF"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u00B3 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillilitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMillilitre" ), "gramPerMillilitre", Set.of(new LangString( "gram per millilitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/ml"), Optional.of("GJ"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillilitreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMillilitreBar" ), "gramPerMillilitreBar", Set.of(new LangString( "gram per millilitre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(ml\u00B7bar)"), Optional.of("G15"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillilitreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMillilitreKelvin" ), "gramPerMillilitreKelvin", Set.of(new LangString( "gram per millilitre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(ml\u00B7K)"), Optional.of("G37"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMillimetre" ), "gramPerMillimetre", Set.of(new LangString( "gram per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/mm"), Optional.of("H76"), Optional.empty(), Optional.of("10\u00B9 kg \u00D7 m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMinute" ), "gramPerMinute", Set.of(new LangString( "gram per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/min"), Optional.of("F28"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMinuteBar" ), "gramPerMinuteBar", Set.of(new LangString( "gram per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(min\u00B7bar)"), Optional.of("F64"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u2070 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMinuteKelvin" ), "gramPerMinuteKelvin", Set.of(new LangString( "gram per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(min\u00B7K)"), Optional.of("F37"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerMole" ), "gramPerMole", Set.of(new LangString( "gram per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/mol"), Optional.of("A94"), Optional.of("kilogramPerMole"), Optional.of("10\u207B\u00B3 kg/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_MASS)) ) );
UNITS_BY_NAME.put( "gramPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerSecond" ), "gramPerSecond", Set.of(new LangString( "gram per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/s"), Optional.of("F29"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerSecondBar" ), "gramPerSecondBar", Set.of(new LangString( "gram per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(s\u00B7bar)"), Optional.of("F65"), Optional.empty(), Optional.of("10\u207B\u2078 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerSecondKelvin" ), "gramPerSecondKelvin", Set.of(new LangString( "gram per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/(s\u00B7K)"), Optional.of("F38"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerSquareCentimetre" ), "gramPerSquareCentimetre", Set.of(new LangString( "gram per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/cm\u00B2"), Optional.of("25"), Optional.of("kilogramPerSquareMetre"), Optional.of("10 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "gramPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerSquareMetre" ), "gramPerSquareMetre", Set.of(new LangString( "gram per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/m\u00B2"), Optional.of("GM"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u207B\u00B3 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "gramPerSquareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gramPerSquareMillimetre" ), "gramPerSquareMillimetre", Set.of(new LangString( "gram per square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("g/mm\u00B2"), Optional.of("N24"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u00B3 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "gray", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gray" ), "gray", Set.of(new LangString( "gray", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gy"), Optional.of("A95"), Optional.empty(), Optional.of("m\u00B2/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENERGY_IMPARTED, QuantityKinds.MASSIC_ENERGY_IMPARTED)) ) );
UNITS_BY_NAME.put( "grayPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#grayPerHour" ), "grayPerHour", Set.of(new LangString( "gray per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gy/h"), Optional.of("P61"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "grayPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#grayPerMinute" ), "grayPerMinute", Set.of(new LangString( "gray per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gy/min"), Optional.of("P57"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "grayPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#grayPerSecond" ), "grayPerSecond", Set.of(new LangString( "gray per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Gy/s"), Optional.of("A96"), Optional.empty(), Optional.of("m\u00B2/s\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
}

private static void init5() {
UNITS_BY_NAME.put( "greatGross", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#greatGross" ), "greatGross", Set.of(new LangString( "great gross", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("GGR"), Optional.empty(), Optional.of("1728"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gross", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#gross" ), "gross", Set.of(new LangString( "gross", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gr"), Optional.of("GRO"), Optional.empty(), Optional.of("144"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "grossKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#grossKilogram" ), "grossKilogram", Set.of(new LangString( "gross kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E4"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "group", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#group" ), "group", Set.of(new LangString( "group", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("10"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "guntersChain", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#guntersChain" ), "guntersChain", Set.of(new LangString( "Gunter's chain", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ch (UK)"), Optional.of("X1"), Optional.of("metre"), Optional.of("20.1168 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "halfYear6Months", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#halfYear6Months" ), "halfYear6Months", Set.of(new LangString( "half year (6 months)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SAN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hangingContainer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hangingContainer" ), "hangingContainer", Set.of(new LangString( "hanging container", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("Z11"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hank", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hank" ), "hank", Set.of(new LangString( "hank", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hartley", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hartley" ), "hartley", Set.of(new LangString( "hartley", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Hart"), Optional.of("Q15"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hartleyPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hartleyPerSecond" ), "hartleyPerSecond", Set.of(new LangString( "hartley per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Hart/s"), Optional.of("Q18"), Optional.empty(), Optional.of("Hart/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "head", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#head" ), "head", Set.of(new LangString( "head", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HEA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hectobar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectobar" ), "hectobar", Set.of(new LangString( "hectobar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hbar"), Optional.of("HBA"), Optional.of("pascal"), Optional.of("10\u2077 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "hectogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectogram" ), "hectogram", Set.of(new LangString( "hectogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hg"), Optional.of("HGM"), Optional.of("kilogram"), Optional.of("10\u207B\u00B9 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "hectolitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectolitre" ), "hectolitre", Set.of(new LangString( "hectolitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hl"), Optional.of("HLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B9 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "hectolitreOfPureAlcohol", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectolitreOfPureAlcohol" ), "hectolitreOfPureAlcohol", Set.of(new LangString( "hectolitre of pure alcohol", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HPA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hectometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectometre" ), "hectometre", Set.of(new LangString( "hectometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hm"), Optional.of("HMT"), Optional.of("metre"), Optional.of("10\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.THICKNESS, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "hectopascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectopascal" ), "hectopascal", Set.of(new LangString( "hectopascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hPa"), Optional.of("A97"), Optional.of("pascal"), Optional.of("10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "hectopascalCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectopascalCubicMetrePerSecond" ), "hectopascalCubicMetrePerSecond", Set.of(new LangString( "hectopascal cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hPa\u00B7m\u00B3/s"), Optional.of("F94"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "hectopascalLitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectopascalLitrePerSecond" ), "hectopascalLitrePerSecond", Set.of(new LangString( "hectopascal litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hPa\u00B7l/s"), Optional.of("F93"), Optional.empty(), Optional.of("10\u207B\u00B9 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "hectopascalPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectopascalPerBar" ), "hectopascalPerBar", Set.of(new LangString( "hectopascal per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hPa/bar"), Optional.of("E99"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "hectopascalPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectopascalPerKelvin" ), "hectopascalPerKelvin", Set.of(new LangString( "hectopascal per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hPa/K"), Optional.of("F82"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "hectopascalPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hectopascalPerMetre" ), "hectopascalPerMetre", Set.of(new LangString( "hectopascal per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hPa/m"), Optional.of("P82"), Optional.empty(), Optional.of("10\u00B2 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "hefnerKerze", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hefnerKerze" ), "hefnerKerze", Set.of(new LangString( "Hefner-Kerze", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("HK"), Optional.of("P35"), Optional.of("candela"), Optional.of("0.903 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "henry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#henry" ), "henry", Set.of(new LangString( "henry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("H"), Optional.of("81"), Optional.empty(), Optional.of("H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "henryPerKiloohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#henryPerKiloohm" ), "henryPerKiloohm", Set.of(new LangString( "henry per kiloohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("H/k\u2126"), Optional.of("H03"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE)) ) );
UNITS_BY_NAME.put( "henryPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#henryPerMetre" ), "henryPerMetre", Set.of(new LangString( "henry per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("H/m"), Optional.of("A98"), Optional.empty(), Optional.of("H/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEABILITY_OF_VACUUM, QuantityKinds.PERMEABILITY, QuantityKinds.MAGNETIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "henryPerOhm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#henryPerOhm" ), "henryPerOhm", Set.of(new LangString( "henry per ohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("H/\u2126"), Optional.of("H04"), Optional.of("secondUnitOfTime"), Optional.of("s"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE)) ) );
UNITS_BY_NAME.put( "hertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hertz" ), "hertz", Set.of(new LangString( "hertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Hz"), Optional.of("HTZ"), Optional.empty(), Optional.of("Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "hertzMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hertzMetre" ), "hertzMetre", Set.of(new LangString( "hertz metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Hz\u00B7m"), Optional.of("H34"), Optional.empty(), Optional.of("Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "horsepowerBoiler", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#horsepowerBoiler" ), "horsepowerBoiler", Set.of(new LangString( "horsepower (boiler)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("boiler hp"), Optional.of("K42"), Optional.of("watt"), Optional.of("9.80950 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "horsepowerElectric", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#horsepowerElectric" ), "horsepowerElectric", Set.of(new LangString( "horsepower (electric)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("electric hp"), Optional.of("K43"), Optional.of("watt"), Optional.of("746 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "hour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hour" ), "hour", Set.of(new LangString( "hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("h"), Optional.of("HUR"), Optional.of("secondUnitOfTime"), Optional.of("3600 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "hundred", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundred" ), "hundred", Set.of(new LangString( "hundred", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CEN"), Optional.empty(), Optional.of("100"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredBoardFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredBoardFoot" ), "hundredBoardFoot", Set.of(new LangString( "hundred board foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("BP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredBoxes", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredBoxes" ), "hundredBoxes", Set.of(new LangString( "hundred boxes", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HBX"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredCount", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredCount" ), "hundredCount", Set.of(new LangString( "hundred count", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredCubicFoot" ), "hundredCubicFoot", Set.of(new LangString( "hundred cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredCubicMetre" ), "hundredCubicMetre", Set.of(new LangString( "hundred cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("FF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredInternationalUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredInternationalUnit" ), "hundredInternationalUnit", Set.of(new LangString( "hundred international unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HIU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredKilogramDryWeight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredKilogramDryWeight" ), "hundredKilogramDryWeight", Set.of(new LangString( "hundred kilogram, dry weight", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HDW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredKilogramNetMass", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredKilogramNetMass" ), "hundredKilogramNetMass", Set.of(new LangString( "hundred kilogram, net mass", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("HKM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredLeave", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredLeave" ), "hundredLeave", Set.of(new LangString( "hundred leave", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CLF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredMetre" ), "hundredMetre", Set.of(new LangString( "hundred metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("JPS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredPack", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredPack" ), "hundredPack", Set.of(new LangString( "hundred pack", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CNP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredPoundCwtOrHundredWeightUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredPoundCwtOrHundredWeightUs" ), "hundredPoundCwtOrHundredWeightUs", Set.of(new LangString( "hundred pound (cwt) / hundred weight (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cwt (US)"), Optional.of("CWA"), Optional.of("kilogram"), Optional.of("45.3592 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "hundredWeightUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hundredWeightUk" ), "hundredWeightUk", Set.of(new LangString( "hundred weight (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cwt (UK)"), Optional.of("CWI"), Optional.of("kilogram"), Optional.of("50.80235 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "hydraulicHorsePower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#hydraulicHorsePower" ), "hydraulicHorsePower", Set.of(new LangString( "hydraulic horse power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("5J"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "imperialGallonPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#imperialGallonPerMinute" ), "imperialGallonPerMinute", Set.of(new LangString( "Imperial gallon per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (UK) /min"), Optional.of("G3"), Optional.of("cubicMetrePerSecond"), Optional.of("7.57682 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "inch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inch" ), "inch", Set.of(new LangString( "inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in"), Optional.of("INH"), Optional.of("metre"), Optional.of("25.4 \u00D7 10\u207B\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "inchOfMercury", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchOfMercury" ), "inchOfMercury", Set.of(new LangString( "inch of mercury", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("inHg"), Optional.of("F79"), Optional.empty(), Optional.of("3.38639 \u00D7 10\u00B3 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "inchOfMercury32Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchOfMercury32Degreesf" ), "inchOfMercury32Degreesf", Set.of(new LangString( "inch of mercury (32 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("inHG (32 \u00B0F)"), Optional.of("N16"), Optional.of("pascal"), Optional.of("3.38638 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "inchOfMercury60Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchOfMercury60Degreesf" ), "inchOfMercury60Degreesf", Set.of(new LangString( "inch of mercury (60 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("inHg (60 \u00B0F)"), Optional.of("N17"), Optional.of("pascal"), Optional.of("3.37685 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "inchOfWater", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchOfWater" ), "inchOfWater", Set.of(new LangString( "inch of water", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("inH\u2082O"), Optional.of("F78"), Optional.empty(), Optional.of("2.49089 \u00D7 10\u00B2 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "inchOfWater392Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchOfWater392Degreesf" ), "inchOfWater392Degreesf", Set.of(new LangString( "inch of water (39.2 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("inH\u2082O (39,2 \u00B0F)"), Optional.of("N18"), Optional.of("pascal"), Optional.of("2.49082 \u00D7 10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "inchOfWater60Degreesf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchOfWater60Degreesf" ), "inchOfWater60Degreesf", Set.of(new LangString( "inch of water (60 \u00B0F)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("inH\u2082O (60 \u00B0F)"), Optional.of("N19"), Optional.of("pascal"), Optional.of("2.4884 \u00D7 10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "inchPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerDegreeFahrenheit" ), "inchPerDegreeFahrenheit", Set.of(new LangString( "inch per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/\u00B0F"), Optional.of("K45"), Optional.of("metrePerKelvin"), Optional.of("4.572 \u00D7 10\u207B\u00B2 m/K"), new HashSet<>(Arrays.asList(QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS, QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "inchPerLinearFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerLinearFoot" ), "inchPerLinearFoot", Set.of(new LangString( "inch per linear foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("B82"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "inchPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerMinute" ), "inchPerMinute", Set.of(new LangString( "inch per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/min"), Optional.of("M63"), Optional.of("metrePerSecond"), Optional.of("4.233333 \u00D7 10\u207B\u2074 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerPsi" ), "inchPerPsi", Set.of(new LangString( "inch per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/psi"), Optional.of("K46"), Optional.of("metrePerPascal"), Optional.of("3.683959 \u00D7 10\u207B\u2076 m/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "inchPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerSecond" ), "inchPerSecond", Set.of(new LangString( "inch per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/s"), Optional.of("IU"), Optional.of("metrePerSecond"), Optional.of("0.0254 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerSecondDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerSecondDegreeFahrenheit" ), "inchPerSecondDegreeFahrenheit", Set.of(new LangString( "inch per second degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(in/s)/\u00B0F"), Optional.of("K47"), Optional.of("metrePerSecondKelvin"), Optional.of("4.572 \u00D7 10\u207B\u00B2 (m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerSecondPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerSecondPsi" ), "inchPerSecondPsi", Set.of(new LangString( "inch per second psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(in/s)/psi"), Optional.of("K48"), Optional.of("metrePerSecondPascal"), Optional.of("3.683959 \u00D7 10\u207B\u2076 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerSecondSquared" ), "inchPerSecondSquared", Set.of(new LangString( "inch per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/s\u00B2"), Optional.of("IV"), Optional.of("metrePerSecondSquared"), Optional.of("0.0254 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "inchPerTwoPiRadiant", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerTwoPiRadiant" ), "inchPerTwoPiRadiant", Set.of(new LangString( "inch per two pi radiant", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/revolution"), Optional.of("H57"), Optional.empty(), Optional.of("2.54 \u00D7 10\u207B\u00B2 m/(2 \u00D7 \u03C0 \u00D7 rad)"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "inchPerYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPerYear" ), "inchPerYear", Set.of(new LangString( "inch per year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in/y"), Optional.of("M61"), Optional.of("metrePerSecond"), Optional.of("8.048774 \u00D7 10\u207B\u00B9\u2070 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPoundPoundInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPoundPoundInch" ), "inchPoundPoundInch", Set.of(new LangString( "inch pound (pound inch)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B7lb"), Optional.of("IA"), Optional.of("kilogramMetre"), Optional.of("1.15212 \u00D7 10\u207B\u00B2 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "inchPoundal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchPoundal" ), "inchPoundal", Set.of(new LangString( "inch poundal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B7pdl"), Optional.of("N47"), Optional.of("joule"), Optional.of("3.511677 \u00D7 10\u207B\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "inchToTheFourthPower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#inchToTheFourthPower" ), "inchToTheFourthPower", Set.of(new LangString( "inch to the fourth power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u2074"), Optional.of("D69"), Optional.of("metreToTheFourthPower"), Optional.of("41.62314 \u00D7 10\u207B\u2078 m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_POLAR_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "internationalCandle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#internationalCandle" ), "internationalCandle", Set.of(new LangString( "international candle", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("IK"), Optional.of("P36"), Optional.of("candela"), Optional.of("1.019 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "internationalSugarDegree", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#internationalSugarDegree" ), "internationalSugarDegree", Set.of(new LangString( "international sugar degree", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ISD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "job", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#job" ), "job", Set.of(new LangString( "job", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E51"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "joule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joule" ), "joule", Set.of(new LangString( "joule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J"), Optional.of("JOU"), Optional.empty(), Optional.of("J"), new HashSet<>(Arrays.asList(QuantityKinds.MEAN_ENERGY_IMPARTED, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.WORK_FUNCTION, QuantityKinds.ALPHA_DISINTEGRATION_ENERGY, QuantityKinds.HARTREE_ENERGY, QuantityKinds.HEAT, QuantityKinds.LEVEL_WIDTH, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.REACTION_ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.RADIANT_ENERGY, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENERGY_IMPARTED, QuantityKinds.ACTIVE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT)) ) );
UNITS_BY_NAME.put( "joulePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerCubicMetre" ), "joulePerCubicMetre", Set.of(new LangString( "joule per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/m\u00B3"), Optional.of("B8"), Optional.empty(), Optional.of("J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_DENSITY, QuantityKinds.ELECTROMAGNETIC_ENERGY_DENSITY, QuantityKinds.VOLUMIC_ELECTROMAGNETIC_ENERGY, QuantityKinds.SOUND_ENERGY_DENSITY, QuantityKinds.SOUND_ENERGY, QuantityKinds.VOLUMIC)) ) );
UNITS_BY_NAME.put( "joulePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerDay" ), "joulePerDay", Set.of(new LangString( "joule per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/d"), Optional.of("P17"), Optional.of("watt"), Optional.of("1.15741 \u00D7 10\u207B\u2075 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerGram" ), "joulePerGram", Set.of(new LangString( "joule per gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/g"), Optional.of("D95"), Optional.empty(), Optional.of("J/(10\u207B\u00B3 \u00D7 kg)"), new HashSet<>(Arrays.asList(QuantityKinds.MASSIC_THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "joulePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerHour" ), "joulePerHour", Set.of(new LangString( "joule per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/h"), Optional.of("P16"), Optional.of("watt"), Optional.of("2.77778 \u00D7 10\u207B\u2074 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerKelvin" ), "joulePerKelvin", Set.of(new LangString( "joule per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/K"), Optional.of("JE"), Optional.empty(), Optional.of("J/K"), new HashSet<>(Arrays.asList(QuantityKinds.PLANCK_FUNCTION, QuantityKinds.BOLTZMANN_CONSTANT, QuantityKinds.ENTROPY, QuantityKinds.HEAT_CAPACITY, QuantityKinds.MASSIEU_FUNCTION)) ) );
UNITS_BY_NAME.put( "joulePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerKilogram" ), "joulePerKilogram", Set.of(new LangString( "joule per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/kg"), Optional.of("J2"), Optional.empty(), Optional.of("J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENERGY, QuantityKinds.MASSIC_ENERGY)) ) );
UNITS_BY_NAME.put( "joulePerKilogramKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerKilogramKelvin" ), "joulePerKilogramKelvin", Set.of(new LangString( "joule per kilogram kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/(kg\u00B7K)"), Optional.of("B11"), Optional.empty(), Optional.of("J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_SATURATION, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE)) ) );
UNITS_BY_NAME.put( "joulePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerMetre" ), "joulePerMetre", Set.of(new LangString( "joule per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/m"), Optional.of("B12"), Optional.empty(), Optional.of("J/m"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_LINEAR_STOPPING_POWER, QuantityKinds.LINEAR_ENERGY_TRANSFER)) ) );
UNITS_BY_NAME.put( "joulePerMetreToTheFourthPower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerMetreToTheFourthPower" ), "joulePerMetreToTheFourthPower", Set.of(new LangString( "joule per metre to the fourth power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/m\u2074"), Optional.of("B14"), Optional.empty(), Optional.of("J/m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVE_LENGTH, QuantityKinds.SPECTRAL_CONCENTRATION_OF_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVELENGTH)) ) );
UNITS_BY_NAME.put( "joulePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerMinute" ), "joulePerMinute", Set.of(new LangString( "joule per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/min"), Optional.of("P15"), Optional.of("watt"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerMole" ), "joulePerMole", Set.of(new LangString( "joule per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/mol"), Optional.of("B15"), Optional.empty(), Optional.of("J/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_THERMODYNAMIC_ENERGY, QuantityKinds.CHEMICAL_POTENTIAL, QuantityKinds.AFFINITY_OF_A_CHEMICAL_REACTION)) ) );
UNITS_BY_NAME.put( "joulePerMoleKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerMoleKelvin" ), "joulePerMoleKelvin", Set.of(new LangString( "joule per mole kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/(mol\u00B7K)"), Optional.of("B16"), Optional.empty(), Optional.of("J/(mol \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_HEAT_CAPACITY, QuantityKinds.MOLAR_GAS_CONSTANT, QuantityKinds.MOLAR_ENTROPY)) ) );
UNITS_BY_NAME.put( "joulePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerSecond" ), "joulePerSecond", Set.of(new LangString( "joule per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/s"), Optional.of("P14"), Optional.of("watt"), Optional.of("W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerSquareCentimetre" ), "joulePerSquareCentimetre", Set.of(new LangString( "joule per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/cm\u00B2"), Optional.of("E43"), Optional.of("joulePerSquareMetre"), Optional.of("10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "joulePerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerSquareMetre" ), "joulePerSquareMetre", Set.of(new LangString( "joule per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/m\u00B2"), Optional.of("B13"), Optional.empty(), Optional.of("J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE, QuantityKinds.ENERGY_FLUENCE)) ) );
UNITS_BY_NAME.put( "joulePerTesla", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#joulePerTesla" ), "joulePerTesla", Set.of(new LangString( "joule per tesla", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J/T"), Optional.of("Q10"), Optional.empty(), Optional.of("m\u00B2 \u00D7 A"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "jouleSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#jouleSecond" ), "jouleSecond", Set.of(new LangString( "joule second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J\u00B7s"), Optional.of("B18"), Optional.empty(), Optional.of("J \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.PLANCK_CONSTANT)) ) );
UNITS_BY_NAME.put( "jouleSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#jouleSquareMetre" ), "jouleSquareMetre", Set.of(new LangString( "joule square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J\u00B7m\u00B2"), Optional.of("D73"), Optional.empty(), Optional.of("J \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_ATOMIC_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "jouleSquareMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#jouleSquareMetrePerKilogram" ), "jouleSquareMetrePerKilogram", Set.of(new LangString( "joule square metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J\u00B7m\u00B2/kg"), Optional.of("B20"), Optional.empty(), Optional.of("J \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_MASS_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "katal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#katal" ), "katal", Set.of(new LangString( "katal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kat"), Optional.of("KAT"), Optional.empty(), Optional.of("s\u207B\u00B9 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "kelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvin" ), "kelvin", Set.of(new LangString( "kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K"), Optional.of("KEL"), Optional.empty(), Optional.of("K"), new HashSet<>(Arrays.asList(QuantityKinds.FERMI_TEMPERATURE, QuantityKinds.CURIE_TEMPERATURE, QuantityKinds.THERMODYNAMIC, QuantityKinds.DEBYE_TEMPERATURE, QuantityKinds.SUPER_CONDUCTOR_TRANSITION_TEMPERATURE, QuantityKinds.NEEL_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinMetrePerWatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinMetrePerWatt" ), "kelvinMetrePerWatt", Set.of(new LangString( "kelvin metre per watt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K\u00B7m/W"), Optional.of("H35"), Optional.empty(), Optional.of("K \u00D7 m\u207B\u00B9 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "kelvinPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerBar" ), "kelvinPerBar", Set.of(new LangString( "kelvin per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/bar"), Optional.of("F61"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerHour" ), "kelvinPerHour", Set.of(new LangString( "kelvin per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/h"), Optional.of("F10"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 s\u207B\u00B9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerKelvin" ), "kelvinPerKelvin", Set.of(new LangString( "kelvin per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/K"), Optional.of("F02"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerMinute" ), "kelvinPerMinute", Set.of(new LangString( "kelvin per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/min"), Optional.of("F11"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 s\u207B\u00B9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerPascal" ), "kelvinPerPascal", Set.of(new LangString( "kelvin per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/Pa"), Optional.of("N79"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerSecond" ), "kelvinPerSecond", Set.of(new LangString( "kelvin per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/s"), Optional.of("F12"), Optional.empty(), Optional.of("s\u207B\u00B9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
}

private static void init6() {
UNITS_BY_NAME.put( "kelvinPerWatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kelvinPerWatt" ), "kelvinPerWatt", Set.of(new LangString( "kelvin per watt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K/W"), Optional.of("B21"), Optional.empty(), Optional.of("K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "kibibit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kibibit" ), "kibibit", Set.of(new LangString( "kibibit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Kibit"), Optional.of("C21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kibibitPerCubicMetre" ), "kibibitPerCubicMetre", Set.of(new LangString( "kibibit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Kibit/m\u00B3"), Optional.of("E74"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kibibitPerMetre" ), "kibibitPerMetre", Set.of(new LangString( "kibibit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Kibit/m"), Optional.of("E72"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kibibitPerSquareMetre" ), "kibibitPerSquareMetre", Set.of(new LangString( "kibibit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Kibit/m\u00B2"), Optional.of("E73"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kibibyte" ), "kibibyte", Set.of(new LangString( "kibibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("KiB"), Optional.of("E64"), Optional.of("byte"), Optional.of("2\u00B9\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "kiloampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloampere" ), "kiloampere", Set.of(new LangString( "kiloampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kA"), Optional.of("B22"), Optional.of("ampere"), Optional.of("10\u00B3 A"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "kiloampereHourThousandAmpereHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloampereHourThousandAmpereHour" ), "kiloampereHourThousandAmpereHour", Set.of(new LangString( "kiloampere hour (thousand ampere hour)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kA\u00B7h"), Optional.of("TAH"), Optional.of("coulomb"), Optional.of("3.6 \u00D7 10\u2076 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "kiloamperePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloamperePerMetre" ), "kiloamperePerMetre", Set.of(new LangString( "kiloampere per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kA/m"), Optional.of("B24"), Optional.of("amperePerMetre"), Optional.of("10\u00B3 A/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "kiloamperePerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloamperePerSquareMetre" ), "kiloamperePerSquareMetre", Set.of(new LangString( "kiloampere per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kA/m\u00B2"), Optional.of("B23"), Optional.of("amperePerSquareMetre"), Optional.of("10\u00B3 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "kilobar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobar" ), "kilobar", Set.of(new LangString( "kilobar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kbar"), Optional.of("KBA"), Optional.of("pascal"), Optional.of("10\u2078 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "kilobaud", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobaud" ), "kilobaud", Set.of(new LangString( "kilobaud", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kBd"), Optional.of("K50"), Optional.of("baud"), Optional.of("10\u00B3 Bd"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilobecquerel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobecquerel" ), "kilobecquerel", Set.of(new LangString( "kilobecquerel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kBq"), Optional.of("2Q"), Optional.of("becquerel"), Optional.of("10\u00B3 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "kilobecquerelPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobecquerelPerKilogram" ), "kilobecquerelPerKilogram", Set.of(new LangString( "kilobecquerel per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kBq/kg"), Optional.of("B25"), Optional.of("becquerelPerKilogram"), Optional.of("10\u00B3 Bq/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "kilobit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobit" ), "kilobit", Set.of(new LangString( "kilobit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kbit"), Optional.of("C37"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilobitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobitPerSecond" ), "kilobitPerSecond", Set.of(new LangString( "kilobit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kbit/s"), Optional.of("C74"), Optional.of("bitPerSecond"), Optional.of("10\u00B3 bit/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilobyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobyte" ), "kilobyte", Set.of(new LangString( "Kilobyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kB"), Optional.of("2P"), Optional.of("byte"), Optional.of("10\u00B3 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "kilobytePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilobytePerSecond" ), "kilobytePerSecond", Set.of(new LangString( "kilobyte per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kbyte/s"), Optional.of("P94"), Optional.of("bytePerSecond"), Optional.of("10\u00B3 byte/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilocalorieInternationalTable", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieInternationalTable" ), "kilocalorieInternationalTable", Set.of(new LangString( "kilocalorie (international table)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcalIT"), Optional.of("E14"), Optional.of("joule"), Optional.of("4.1868 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.HEAT, QuantityKinds.ENTHALPY)) ) );
UNITS_BY_NAME.put( "kilocalorieInternationalTablePerGramKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieInternationalTablePerGramKelvin" ), "kilocalorieInternationalTablePerGramKelvin", Set.of(new LangString( "kilocalorie (international table) per gram kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kcalIT/K)/g"), Optional.of("N65"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.1868 \u00D7 10\u2076 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "kilocalorieInternationalTablePerHourMetreDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieInternationalTablePerHourMetreDegreeCelsius" ), "kilocalorieInternationalTablePerHourMetreDegreeCelsius", Set.of(new LangString( "kilocalorie (international table) per hour metre degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcal/(m\u00B7h\u00B7\u00B0C)"), Optional.of("K52"), Optional.empty(), Optional.of("1.163 J/(m \u00D7 s \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilocalorieMean", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieMean" ), "kilocalorieMean", Set.of(new LangString( "kilocalorie (mean)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcal"), Optional.of("K51"), Optional.of("joule"), Optional.of("4.19002 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemical", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieThermochemical" ), "kilocalorieThermochemical", Set.of(new LangString( "kilocalorie (thermochemical)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcalth"), Optional.of("K53"), Optional.of("joule"), Optional.of("4.184 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemicalPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieThermochemicalPerHour" ), "kilocalorieThermochemicalPerHour", Set.of(new LangString( "kilocalorie (thermochemical) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcalth/h"), Optional.of("E15"), Optional.of("watt"), Optional.of("1.16222 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemicalPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieThermochemicalPerMinute" ), "kilocalorieThermochemicalPerMinute", Set.of(new LangString( "kilocalorie (thermochemical) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcalth/min"), Optional.of("K54"), Optional.of("watt"), Optional.of("69.73333 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemicalPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocalorieThermochemicalPerSecond" ), "kilocalorieThermochemicalPerSecond", Set.of(new LangString( "kilocalorie (thermochemical) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcalth/s"), Optional.of("K55"), Optional.of("watt"), Optional.of("4.184 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilocandela", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocandela" ), "kilocandela", Set.of(new LangString( "kilocandela", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kcd"), Optional.of("P33"), Optional.of("candela"), Optional.of("10\u00B3 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "kilocharacter", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocharacter" ), "kilocharacter", Set.of(new LangString( "kilocharacter", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilocoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocoulomb" ), "kilocoulomb", Set.of(new LangString( "kilocoulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kC"), Optional.of("B26"), Optional.of("coulomb"), Optional.of("10\u00B3 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "kilocoulombPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocoulombPerCubicMetre" ), "kilocoulombPerCubicMetre", Set.of(new LangString( "kilocoulomb per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kC/m\u00B3"), Optional.of("B27"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u00B3 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.VOLUME_DENSITY_OF_CHARGE)) ) );
UNITS_BY_NAME.put( "kilocoulombPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocoulombPerSquareMetre" ), "kilocoulombPerSquareMetre", Set.of(new LangString( "kilocoulomb per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kC/m\u00B2"), Optional.of("B28"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u00B3 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.SURFACE_DENSITY_OF_CHARGE)) ) );
UNITS_BY_NAME.put( "kilocurie", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilocurie" ), "kilocurie", Set.of(new LangString( "kilocurie", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kCi"), Optional.of("2R"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u00B9\u00B3 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "kiloelectronvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloelectronvolt" ), "kiloelectronvolt", Set.of(new LangString( "kiloelectronvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("keV"), Optional.of("B29"), Optional.of("electronvolt"), Optional.of("10\u00B3 eV"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.POTENTIAL_ENERGY)) ) );
UNITS_BY_NAME.put( "kilofarad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilofarad" ), "kilofarad", Set.of(new LangString( "kilofarad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kF"), Optional.of("N90"), Optional.of("farad"), Optional.of("10\u00B3 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "kilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogram" ), "kilogram", Set.of(new LangString( "kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg"), Optional.of("KGM"), Optional.empty(), Optional.of("kg"), new HashSet<>(Arrays.asList(QuantityKinds.NUCLIDIC_MASS, QuantityKinds.MASS, QuantityKinds.EFFECTIVE_MASS, QuantityKinds.MASS_OF_ATOM_OF_A_NUCLIDE_X, QuantityKinds.MASS_EXCESS, QuantityKinds.MASS_OF_MOLECULE)) ) );
UNITS_BY_NAME.put( "kilogramCentimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramCentimetrePerSecond" ), "kilogramCentimetrePerSecond", Set.of(new LangString( "kilogram centimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7(cm/s)"), Optional.of("M98"), Optional.of("kilogramMetrePerSecond"), Optional.of("10\u207B\u00B2 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "kilogramDrainedNetWeight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramDrainedNetWeight" ), "kilogramDrainedNetWeight", Set.of(new LangString( "kilogram drained net weight", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/net eda"), Optional.of("KDW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramDryWeight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramDryWeight" ), "kilogramDryWeight", Set.of(new LangString( "kilogram, dry weight", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MND"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramForceMetrePerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramForceMetrePerSquareCentimetre" ), "kilogramForceMetrePerSquareCentimetre", Set.of(new LangString( "kilogram-force metre per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kgf\u00B7m/cm\u00B2"), Optional.of("E44"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramForcePerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramForcePerSquareCentimetre" ), "kilogramForcePerSquareCentimetre", Set.of(new LangString( "kilogram-force per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kgf/cm\u00B2"), Optional.of("E42"), Optional.of("pascal"), Optional.of("9.80665 \u00D7 10\u2074 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "kilogramForcePerSquareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramForcePerSquareMillimetre" ), "kilogramForcePerSquareMillimetre", Set.of(new LangString( "kilogram-force per square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kgf/mm\u00B2"), Optional.of("E41"), Optional.of("pascal"), Optional.of("9.80665 \u00D7 10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "kilogramIncludingContainer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramIncludingContainer" ), "kilogramIncludingContainer", Set.of(new LangString( "kilogram, including container", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KIC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramIncludingInnerPackaging", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramIncludingInnerPackaging" ), "kilogramIncludingInnerPackaging", Set.of(new LangString( "kilogram, including inner packaging", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KIP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramMetre" ), "kilogramMetre", Set.of(new LangString( "kilogram metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7m"), Optional.of("M94"), Optional.empty(), Optional.of("kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "kilogramMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramMetrePerSecond" ), "kilogramMetrePerSecond", Set.of(new LangString( "kilogram metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7m/s"), Optional.of("B31"), Optional.empty(), Optional.of("kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "kilogramMetrePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramMetrePerSecondSquared" ), "kilogramMetrePerSecondSquared", Set.of(new LangString( "kilogram metre per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7m/s\u00B2"), Optional.of("M77"), Optional.empty(), Optional.of("(kg \u00D7 m)/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "kilogramMetreSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramMetreSquared" ), "kilogramMetreSquared", Set.of(new LangString( "kilogram metre squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7m\u00B2"), Optional.of("B32"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA)) ) );
UNITS_BY_NAME.put( "kilogramMetreSquaredPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramMetreSquaredPerSecond" ), "kilogramMetreSquaredPerSecond", Set.of(new LangString( "kilogram metre squared per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7m\u00B2/s"), Optional.of("B33"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_MOMENTUM, QuantityKinds.ANGULAR_MOMENTUM)) ) );
UNITS_BY_NAME.put( "kilogramNamedSubstance", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramNamedSubstance" ), "kilogramNamedSubstance", Set.of(new LangString( "kilogram named substance", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KNS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfCholineChloride", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfCholineChloride" ), "kilogramOfCholineChloride", Set.of(new LangString( "kilogram of choline chloride", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg C\u2085 H\u2081\u2084ClNO"), Optional.of("KCC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfHydrogenPeroxide", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfHydrogenPeroxide" ), "kilogramOfHydrogenPeroxide", Set.of(new LangString( "kilogram of hydrogen peroxide", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg H\u2082O\u2082"), Optional.of("KHY"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfImportedMeatLessOffal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfImportedMeatLessOffal" ), "kilogramOfImportedMeatLessOffal", Set.of(new LangString( "kilogram of imported meat, less offal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TMS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfMethylamine", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfMethylamine" ), "kilogramOfMethylamine", Set.of(new LangString( "kilogram of methylamine", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg met.am."), Optional.of("KMA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfNitrogen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfNitrogen" ), "kilogramOfNitrogen", Set.of(new LangString( "kilogram of nitrogen", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg N"), Optional.of("KNI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfPhosphorusPentoxidePhosphoricAnhydride", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfPhosphorusPentoxidePhosphoricAnhydride" ), "kilogramOfPhosphorusPentoxidePhosphoricAnhydride", Set.of(new LangString( "kilogram of phosphorus pentoxide (phosphoric anhydride)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KPP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfPotassiumHydroxideCausticPotash", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfPotassiumHydroxideCausticPotash" ), "kilogramOfPotassiumHydroxideCausticPotash", Set.of(new LangString( "kilogram of potassium hydroxide (caustic potash)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg KOH"), Optional.of("KPH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfPotassiumOxide", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfPotassiumOxide" ), "kilogramOfPotassiumOxide", Set.of(new LangString( "kilogram of potassium oxide", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg K\u2082O"), Optional.of("KPO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfSodiumHydroxideCausticSoda", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfSodiumHydroxideCausticSoda" ), "kilogramOfSodiumHydroxideCausticSoda", Set.of(new LangString( "kilogram of sodium hydroxide (caustic soda)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg NaOH"), Optional.of("KSH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfSubstance90PercentDry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfSubstance90PercentDry" ), "kilogramOfSubstance90PercentDry", Set.of(new LangString( "kilogram of substance 90 % dry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg 90 % sdt"), Optional.of("KSD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfTungstenTrioxide", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfTungstenTrioxide" ), "kilogramOfTungstenTrioxide", Set.of(new LangString( "kilogram of tungsten trioxide", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg WO\u2083"), Optional.of("KWO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfUranium", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramOfUranium" ), "kilogramOfUranium", Set.of(new LangString( "kilogram of uranium", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg U"), Optional.of("KUR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerBar" ), "kilogramPerBar", Set.of(new LangString( "kilogram per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/bar"), Optional.of("H53"), Optional.empty(), Optional.of("10\u207B\u2075 m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicCentimetre" ), "kilogramPerCubicCentimetre", Set.of(new LangString( "kilogram per cubic centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/cm\u00B3"), Optional.of("G31"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicCentimetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicCentimetreBar" ), "kilogramPerCubicCentimetreBar", Set.of(new LangString( "kilogram per cubic centimetre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(cm\u00B3\u00B7bar)"), Optional.of("G16"), Optional.empty(), Optional.of("10\u00B9 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicCentimetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicCentimetreKelvin" ), "kilogramPerCubicCentimetreKelvin", Set.of(new LangString( "kilogram per cubic centimetre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(cm\u00B3\u00B7K)"), Optional.of("G38"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicDecimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicDecimetre" ), "kilogramPerCubicDecimetre", Set.of(new LangString( "kilogram per cubic decimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/dm\u00B3"), Optional.of("B34"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicDecimetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicDecimetreBar" ), "kilogramPerCubicDecimetreBar", Set.of(new LangString( "kilogram per cubic decimetre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kg/dm\u00B3)/bar"), Optional.of("H55"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicDecimetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicDecimetreKelvin" ), "kilogramPerCubicDecimetreKelvin", Set.of(new LangString( "kilogram per cubic decimetre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kg/dm\u00B3)/K"), Optional.of("H54"), Optional.empty(), Optional.of("10\u00B3 m\u207B\u00B3 \u00D7 kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicMetre" ), "kilogramPerCubicMetre", Set.of(new LangString( "kilogram per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/m\u00B3"), Optional.of("KMQ"), Optional.empty(), Optional.of("kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicMetreBar" ), "kilogramPerCubicMetreBar", Set.of(new LangString( "kilogram per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B3\u00B7bar)"), Optional.of("G18"), Optional.empty(), Optional.of("10\u207B\u2075 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicMetreKelvin" ), "kilogramPerCubicMetreKelvin", Set.of(new LangString( "kilogram per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B3\u00B7K)"), Optional.of("G40"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetrePascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerCubicMetrePascal" ), "kilogramPerCubicMetrePascal", Set.of(new LangString( "kilogram per cubic metre pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kg/m\u00B3)/Pa"), Optional.of("M73"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerDay" ), "kilogramPerDay", Set.of(new LangString( "kilogram per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/d"), Optional.of("F30"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerDayBar" ), "kilogramPerDayBar", Set.of(new LangString( "kilogram per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(d\u00B7bar)"), Optional.of("F66"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2070 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerDayKelvin" ), "kilogramPerDayKelvin", Set.of(new LangString( "kilogram per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(d\u00B7K)"), Optional.of("F39"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerHour" ), "kilogramPerHour", Set.of(new LangString( "kilogram per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/h"), Optional.of("E93"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerHourBar" ), "kilogramPerHourBar", Set.of(new LangString( "kilogram per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(h\u00B7bar)"), Optional.of("F67"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2079 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerHourKelvin" ), "kilogramPerHourKelvin", Set.of(new LangString( "kilogram per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(h\u00B7K)"), Optional.of("F40"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerKelvin" ), "kilogramPerKelvin", Set.of(new LangString( "kilogram per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/K"), Optional.of("F15"), Optional.empty(), Optional.of("kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerKilogram" ), "kilogramPerKilogram", Set.of(new LangString( "kilogram per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/kg"), Optional.of("M29"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "kilogramPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerKilometre" ), "kilogramPerKilometre", Set.of(new LangString( "kilogram per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/km"), Optional.of("M31"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u00B3 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerKilomol", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerKilomol" ), "kilogramPerKilomol", Set.of(new LangString( "kilogram per kilomol", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/kmol"), Optional.of("F24"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 mol\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerLitre" ), "kilogramPerLitre", Set.of(new LangString( "kilogram per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/l or kg/L"), Optional.of("B35"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "kilogramPerLitreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerLitreBar" ), "kilogramPerLitreBar", Set.of(new LangString( "kilogram per litre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(l\u00B7bar)"), Optional.of("G17"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerLitreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerLitreKelvin" ), "kilogramPerLitreKelvin", Set.of(new LangString( "kilogram per litre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(l\u00B7K)"), Optional.of("G39"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMetre" ), "kilogramPerMetre", Set.of(new LangString( "kilogram per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/m"), Optional.of("KL"), Optional.empty(), Optional.of("kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMetreDay" ), "kilogramPerMetreDay", Set.of(new LangString( "kilogram per metre day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B7d)"), Optional.of("N39"), Optional.of("pascalSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2075 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMetreHour" ), "kilogramPerMetreHour", Set.of(new LangString( "kilogram per metre hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B7h)"), Optional.of("N40"), Optional.of("pascalSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMetreMinute" ), "kilogramPerMetreMinute", Set.of(new LangString( "kilogram per metre minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B7min)"), Optional.of("N38"), Optional.of("pascalSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMetreSecond" ), "kilogramPerMetreSecond", Set.of(new LangString( "kilogram per metre second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B7s)"), Optional.of("N37"), Optional.of("pascalSecond"), Optional.of("Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMillimetre" ), "kilogramPerMillimetre", Set.of(new LangString( "kilogram per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/mm"), Optional.of("KW"), Optional.of("kilogramPerMetre"), Optional.of("10\u00B3 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMillimetreWidth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMillimetreWidth" ), "kilogramPerMillimetreWidth", Set.of(new LangString( "kilogram per millimetre width", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KI"), Optional.of("kilogramPerMetre"), Optional.of("10\u00B3 kg/m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMinute" ), "kilogramPerMinute", Set.of(new LangString( "kilogram per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/min"), Optional.of("F31"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMinuteBar" ), "kilogramPerMinuteBar", Set.of(new LangString( "kilogram per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(min\u00B7bar)"), Optional.of("F68"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2077 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMinuteKelvin" ), "kilogramPerMinuteKelvin", Set.of(new LangString( "kilogram per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(min\u00B7K)"), Optional.of("F41"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerMole" ), "kilogramPerMole", Set.of(new LangString( "kilogram per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/mol"), Optional.of("D74"), Optional.empty(), Optional.of("kg/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_MASS)) ) );
UNITS_BY_NAME.put( "kilogramPerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerPascal" ), "kilogramPerPascal", Set.of(new LangString( "kilogram per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/Pa"), Optional.of("M74"), Optional.empty(), Optional.of("m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSecond" ), "kilogramPerSecond", Set.of(new LangString( "kilogram per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/s"), Optional.of("KGS"), Optional.empty(), Optional.of("kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSecondBar" ), "kilogramPerSecondBar", Set.of(new LangString( "kilogram per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(s\u00B7bar)"), Optional.of("F69"), Optional.empty(), Optional.of("10\u207B\u2075 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSecondKelvin" ), "kilogramPerSecondKelvin", Set.of(new LangString( "kilogram per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(s\u00B7K)"), Optional.of("F42"), Optional.empty(), Optional.of("kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
}

private static void init7() {
UNITS_BY_NAME.put( "kilogramPerSecondPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSecondPascal" ), "kilogramPerSecondPascal", Set.of(new LangString( "kilogram per second pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kg/s)/Pa"), Optional.of("M87"), Optional.empty(), Optional.of("m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSquareCentimetre" ), "kilogramPerSquareCentimetre", Set.of(new LangString( "kilogram per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/cm\u00B2"), Optional.of("D5"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u2074 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "kilogramPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSquareMetre" ), "kilogramPerSquareMetre", Set.of(new LangString( "kilogram per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/m\u00B2"), Optional.of("28"), Optional.empty(), Optional.of("kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.MEAN_MASS_RANGE, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "kilogramPerSquareMetrePascalSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSquareMetrePascalSecond" ), "kilogramPerSquareMetrePascalSecond", Set.of(new LangString( "kilogram per square metre pascal second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B2\u00B7Pa\u00B7s)"), Optional.of("Q28"), Optional.empty(), Optional.of("kg/(m\u00B2 \u00D7 Pa \u00D7 s)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramPerSquareMetreSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramPerSquareMetreSecond" ), "kilogramPerSquareMetreSecond", Set.of(new LangString( "kilogram per square metre second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg/(m\u00B2\u00B7s)"), Optional.of("H56"), Optional.empty(), Optional.of("kg m\u207B\u00B2 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramSquareCentimetre" ), "kilogramSquareCentimetre", Set.of(new LangString( "kilogram square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7cm\u00B2"), Optional.of("F18"), Optional.empty(), Optional.of("10\u207B\u2074 kg m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramSquareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilogramSquareMillimetre" ), "kilogramSquareMillimetre", Set.of(new LangString( "kilogram square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kg\u00B7mm\u00B2"), Optional.of("F19"), Optional.empty(), Optional.of("10\u207B\u2076 kg m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilohenry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilohenry" ), "kilohenry", Set.of(new LangString( "kilohenry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kH"), Optional.of("P24"), Optional.of("henry"), Optional.of("10\u00B3 H"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "kilohertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilohertz" ), "kilohertz", Set.of(new LangString( "kilohertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kHz"), Optional.of("KHZ"), Optional.of("hertz"), Optional.of("10\u00B3 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "kilohertzMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilohertzMetre" ), "kilohertzMetre", Set.of(new LangString( "kilohertz metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kHz\u00B7m"), Optional.of("M17"), Optional.of("hertzMetre"), Optional.of("10\u00B3 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "kilojoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoule" ), "kilojoule", Set.of(new LangString( "kilojoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ"), Optional.of("KJO"), Optional.of("joule"), Optional.of("10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "kilojoulePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerDay" ), "kilojoulePerDay", Set.of(new LangString( "kilojoule per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/d"), Optional.of("P21"), Optional.of("watt"), Optional.of("1.15741 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilojoulePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerHour" ), "kilojoulePerHour", Set.of(new LangString( "kilojoule per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/h"), Optional.of("P20"), Optional.of("watt"), Optional.of("2.77778 \u00D7 10\u207B\u00B9 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilojoulePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerKelvin" ), "kilojoulePerKelvin", Set.of(new LangString( "kilojoule per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/K"), Optional.of("B41"), Optional.of("joulePerKelvin"), Optional.of("10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "kilojoulePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerKilogram" ), "kilojoulePerKilogram", Set.of(new LangString( "kilojoule per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/kg"), Optional.of("B42"), Optional.of("joulePerKilogram"), Optional.of("10\u00B3 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENTHALPY, QuantityKinds.MASSIC_ENTHALPY)) ) );
UNITS_BY_NAME.put( "kilojoulePerKilogramKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerKilogramKelvin" ), "kilojoulePerKilogramKelvin", Set.of(new LangString( "kilojoule per kilogram kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/(kg\u00B7K)"), Optional.of("B43"), Optional.of("joulePerKilogramKelvin"), Optional.of("10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_SATURATION, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE)) ) );
UNITS_BY_NAME.put( "kilojoulePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerMinute" ), "kilojoulePerMinute", Set.of(new LangString( "kilojoule per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/min"), Optional.of("P19"), Optional.of("watt"), Optional.of("1.66667 \u00D7 10 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilojoulePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerMole" ), "kilojoulePerMole", Set.of(new LangString( "kilojoule per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/mol"), Optional.of("B44"), Optional.of("joulePerMole"), Optional.of("10\u00B3 J/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "kilojoulePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilojoulePerSecond" ), "kilojoulePerSecond", Set.of(new LangString( "kilojoule per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kJ/s"), Optional.of("P18"), Optional.of("watt"), Optional.of("10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilolitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilolitre" ), "kilolitre", Set.of(new LangString( "kilolitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kl"), Optional.of("K6"), Optional.of("cubicMetre"), Optional.of("m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "kilolitrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilolitrePerHour" ), "kilolitrePerHour", Set.of(new LangString( "kilolitre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kl/h"), Optional.of("4X"), Optional.of("cubicMetrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilolux", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilolux" ), "kilolux", Set.of(new LangString( "kilolux", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("klx"), Optional.of("KLX"), Optional.empty(), Optional.of("10\u00B3 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "kilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilometre" ), "kilometre", Set.of(new LangString( "kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km"), Optional.of("KMT"), Optional.of("metre"), Optional.of("10\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "kilometrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilometrePerHour" ), "kilometrePerHour", Set.of(new LangString( "kilometre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km/h"), Optional.of("KMH"), Optional.of("metrePerSecond"), Optional.of("0.277778 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "kilometrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilometrePerSecond" ), "kilometrePerSecond", Set.of(new LangString( "kilometre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km/s"), Optional.of("M62"), Optional.of("metrePerSecond"), Optional.of("10\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "kilometrePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilometrePerSecondSquared" ), "kilometrePerSecondSquared", Set.of(new LangString( "kilometre per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km/s\u00B2"), Optional.of("M38"), Optional.of("metrePerSecondSquared"), Optional.of("10\u00B3 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "kilomole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomole" ), "kilomole", Set.of(new LangString( "kilomole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kmol"), Optional.of("B45"), Optional.of("mole"), Optional.of("10\u00B3 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "kilomolePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerCubicMetre" ), "kilomolePerCubicMetre", Set.of(new LangString( "kilomole per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kmol/m\u00B3"), Optional.of("B46"), Optional.of("molePerCubicMetre"), Optional.of("10\u00B3 mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "kilomolePerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerCubicMetreBar" ), "kilomolePerCubicMetreBar", Set.of(new LangString( "kilomole per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kmol/m\u00B3)/bar"), Optional.of("K60"), Optional.of("molPerCubicMetrePascal"), Optional.of("10\u207B\u00B2 (mol/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilomolePerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerCubicMetreKelvin" ), "kilomolePerCubicMetreKelvin", Set.of(new LangString( "kilomole per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(kmol/m\u00B3)/K"), Optional.of("K59"), Optional.of("molePerCubicMetreKelvin"), Optional.of("10\u00B3 (mol/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilomolePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerHour" ), "kilomolePerHour", Set.of(new LangString( "kilomole per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kmol/h"), Optional.of("K58"), Optional.of("molePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
UNITS_BY_NAME.put( "kilomolePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerKilogram" ), "kilomolePerKilogram", Set.of(new LangString( "kilomole per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kmol/kg"), Optional.of("P47"), Optional.of("molePerKilogram"), Optional.of("10\u00B3 mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "kilomolePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerMinute" ), "kilomolePerMinute", Set.of(new LangString( "kilomole per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kmol/min"), Optional.of("K61"), Optional.of("molePerSecond"), Optional.of("16.6667 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
UNITS_BY_NAME.put( "kilomolePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilomolePerSecond" ), "kilomolePerSecond", Set.of(new LangString( "kilomole per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kmol/s"), Optional.of("E94"), Optional.empty(), Optional.of("10\u00B3 s\u207B\u00B9 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "kilonewton", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilonewton" ), "kilonewton", Set.of(new LangString( "kilonewton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kN"), Optional.of("B47"), Optional.of("newton"), Optional.of("10\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "kilonewtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilonewtonMetre" ), "kilonewtonMetre", Set.of(new LangString( "kilonewton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kN\u00B7m"), Optional.of("B48"), Optional.of("newtonMetre"), Optional.of("10\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "kilonewtonPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilonewtonPerMetre" ), "kilonewtonPerMetre", Set.of(new LangString( "kilonewton per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kN/m"), Optional.of("N31"), Optional.of("newtonPerMetre"), Optional.of("10\u00B3 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "kiloohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloohm" ), "kiloohm", Set.of(new LangString( "kiloohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("k\u2126"), Optional.of("B49"), Optional.of("ohm"), Optional.of("10\u00B3 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.IMPEDANCE, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.REACTANCE, QuantityKinds.COMPLEX_IMPEDANCES)) ) );
UNITS_BY_NAME.put( "kiloohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloohmMetre" ), "kiloohmMetre", Set.of(new LangString( "kiloohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("k\u2126\u00B7m"), Optional.of("B50"), Optional.of("ohmMetre"), Optional.of("10\u00B3 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "kilopascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopascal" ), "kilopascal", Set.of(new LangString( "kilopascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kPa"), Optional.of("KPA"), Optional.of("pascal"), Optional.of("10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "kilopascalPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopascalPerBar" ), "kilopascalPerBar", Set.of(new LangString( "kilopascal per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kPa/bar"), Optional.of("F03"), Optional.empty(), Optional.of("10\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "kilopascalPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopascalPerKelvin" ), "kilopascalPerKelvin", Set.of(new LangString( "kilopascal per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kPa/K"), Optional.of("F83"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "kilopascalPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopascalPerMetre" ), "kilopascalPerMetre", Set.of(new LangString( "kilopascal per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kPa/m"), Optional.of("P81"), Optional.empty(), Optional.of("10\u00B3 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "kilopascalPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopascalPerMillimetre" ), "kilopascalPerMillimetre", Set.of(new LangString( "kilopascal per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kPa/mm"), Optional.of("34"), Optional.empty(), Optional.of("10\u2076 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "kilopascalSquareMetrePerGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopascalSquareMetrePerGram" ), "kilopascalSquareMetrePerGram", Set.of(new LangString( "kilopascal square metre per gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kPa\u00B7m\u00B2/g"), Optional.of("33"), Optional.of("metrePerSecondSquared"), Optional.of("10\u2076 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.BURST_INDEX)) ) );
UNITS_BY_NAME.put( "kilopoundForce", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopoundForce" ), "kilopoundForce", Set.of(new LangString( "kilopound-force", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kip"), Optional.of("M75"), Optional.of("newton"), Optional.of("4.448222 \u00D7 10\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "kilopoundPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilopoundPerHour" ), "kilopoundPerHour", Set.of(new LangString( "kilopound per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("klb/h"), Optional.of("M90"), Optional.of("kilogramPerSecond"), Optional.of("0.125997889 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kiloroentgen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloroentgen" ), "kiloroentgen", Set.of(new LangString( "kiloroentgen", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kR"), Optional.of("KR"), Optional.of("coulombPerKilogram"), Optional.of("2.58 \u00D7 10\u207B\u00B9 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "kilosecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilosecond" ), "kilosecond", Set.of(new LangString( "kilosecond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ks"), Optional.of("B52"), Optional.of("secondUnitOfTime"), Optional.of("10\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "kilosegment", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilosegment" ), "kilosegment", Set.of(new LangString( "kilosegment", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KJ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilosiemens", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilosiemens" ), "kilosiemens", Set.of(new LangString( "kilosiemens", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kS"), Optional.of("B53"), Optional.of("siemens"), Optional.of("10\u00B3 S"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.ADMITTANCE)) ) );
UNITS_BY_NAME.put( "kilosiemensPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilosiemensPerMetre" ), "kilosiemensPerMetre", Set.of(new LangString( "kilosiemens per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kS/m"), Optional.of("B54"), Optional.of("siemensPerMetre"), Optional.of("10\u00B3 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilotesla", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilotesla" ), "kilotesla", Set.of(new LangString( "kilotesla", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kT"), Optional.of("P13"), Optional.of("tesla"), Optional.of("10\u00B3 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_INDUCTION, QuantityKinds.MAGNETIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "kilotonne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilotonne" ), "kilotonne", Set.of(new LangString( "kilotonne", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kt"), Optional.of("KTN"), Optional.of("kilogram"), Optional.of("10\u2076 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "kilovar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovar" ), "kilovar", Set.of(new LangString( "kilovar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kvar"), Optional.of("KVR"), Optional.of("voltAmpere"), Optional.of("10\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.REACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilovolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovolt" ), "kilovolt", Set.of(new LangString( "kilovolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kV"), Optional.of("KVT"), Optional.of("voltDc"), Optional.of("10\u00B3 V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.TENSION, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.VOLTAGE)) ) );
UNITS_BY_NAME.put( "kilovoltAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovoltAmpere" ), "kilovoltAmpere", Set.of(new LangString( "kilovolt - ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kV\u00B7A"), Optional.of("KVA"), Optional.of("voltAmpere"), Optional.of("10\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "kilovoltAmpereHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovoltAmpereHour" ), "kilovoltAmpereHour", Set.of(new LangString( "kilovolt ampere hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kVAh"), Optional.of("C79"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilovoltAmpereReactiveDemand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovoltAmpereReactiveDemand" ), "kilovoltAmpereReactiveDemand", Set.of(new LangString( "kilovolt ampere reactive demand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("K2"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilovoltAmpereReactiveHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovoltAmpereReactiveHour" ), "kilovoltAmpereReactiveHour", Set.of(new LangString( "kilovolt ampere reactive hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kvar\u00B7h"), Optional.of("K3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilovoltPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilovoltPerMetre" ), "kilovoltPerMetre", Set.of(new LangString( "kilovolt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kV/m"), Optional.of("B55"), Optional.of("voltPerMetre"), Optional.of("10\u00B3 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "kilowatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowatt" ), "kilowatt", Set.of(new LangString( "kilowatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW"), Optional.of("KWT"), Optional.of("watt"), Optional.of("10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE, QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "kilowattDemand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattDemand" ), "kilowattDemand", Set.of(new LangString( "kilowatt demand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("K1"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattHour" ), "kilowattHour", Set.of(new LangString( "kilowatt hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW\u00B7h"), Optional.of("KWH"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u2076 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "kilowattHourPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattHourPerCubicMetre" ), "kilowattHourPerCubicMetre", Set.of(new LangString( "kilowatt hour per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW\u00B7h/m\u00B3"), Optional.of("E46"), Optional.of("joulePerCubicMetre"), Optional.of("3.6 \u00D7 10\u2076 J/m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattHourPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattHourPerHour" ), "kilowattHourPerHour", Set.of(new LangString( "kilowatt hour per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW\u00B7h/h"), Optional.of("D03"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattHourPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattHourPerKelvin" ), "kilowattHourPerKelvin", Set.of(new LangString( "kilowatt hour per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW\u00B7h/K"), Optional.of("E47"), Optional.of("joulePerKelvin"), Optional.of("3.6 \u00D7 10\u2076 J/K"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattPerMetreDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattPerMetreDegreeCelsius" ), "kilowattPerMetreDegreeCelsius", Set.of(new LangString( "kilowatt per metre degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW/(m\u00B7\u00B0C)"), Optional.of("N82"), Optional.of("wattPerMetreKelvin"), Optional.of("10\u00B3 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilowattPerMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattPerMetreKelvin" ), "kilowattPerMetreKelvin", Set.of(new LangString( "kilowatt per metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW/(m\u00B7K)"), Optional.of("N81"), Optional.of("wattPerMetreKelvin"), Optional.of("10\u00B3 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilowattPerSquareMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kilowattPerSquareMetreKelvin" ), "kilowattPerSquareMetreKelvin", Set.of(new LangString( "kilowatt per square metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kW/(m\u00B2\u00B7K)"), Optional.of("N78"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("10\u00B3 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "kiloweber", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloweber" ), "kiloweber", Set.of(new LangString( "kiloweber", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kWb"), Optional.of("P11"), Optional.of("weber"), Optional.of("10\u00B3 Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX)) ) );
UNITS_BY_NAME.put( "kiloweberPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kiloweberPerMetre" ), "kiloweberPerMetre", Set.of(new LangString( "kiloweber per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kWb/m"), Optional.of("B56"), Optional.of("weberPerMetre"), Optional.of("10\u00B3 Wb/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_VECTOR_POTENTIAL)) ) );
UNITS_BY_NAME.put( "kipPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kipPerSquareInch" ), "kipPerSquareInch", Set.of(new LangString( "kip per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ksi"), Optional.of("N20"), Optional.of("pascal"), Optional.of("6.894757 \u00D7 10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "kit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#kit" ), "kit", Set.of(new LangString( "kit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "knot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#knot" ), "knot", Set.of(new LangString( "knot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kn"), Optional.of("KNT"), Optional.of("metrePerSecond"), Optional.of("0.514444 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "labourHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#labourHour" ), "labourHour", Set.of(new LangString( "labour hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lacticDryMaterialPercentage", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lacticDryMaterialPercentage" ), "lacticDryMaterialPercentage", Set.of(new LangString( "lactic dry material percentage", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KLK"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lactoseExcessPercentage", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lactoseExcessPercentage" ), "lactoseExcessPercentage", Set.of(new LangString( "lactose excess percentage", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LAC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lambert", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lambert" ), "lambert", Set.of(new LangString( "lambert", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Lb"), Optional.of("P30"), Optional.of("candelaPerSquareMetre"), Optional.of("3.183099 \u00D7 10\u00B3 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "langley", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#langley" ), "langley", Set.of(new LangString( "langley", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Ly"), Optional.of("P40"), Optional.of("joulePerSquareMetre"), Optional.of("4.184 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "layer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#layer" ), "layer", Set.of(new LangString( "layer", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "leaf", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#leaf" ), "leaf", Set.of(new LangString( "leaf", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LEF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lengthUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lengthUnit" ), "lengthUnit", Set.of(new LangString( "length", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lightYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lightYear" ), "lightYear", Set.of(new LangString( "light year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ly"), Optional.of("B57"), Optional.of("metre"), Optional.of("9.46073 \u00D7 10\u00B9\u2075 m"), new HashSet<>(Arrays.asList(QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "linearFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#linearFoot" ), "linearFoot", Set.of(new LangString( "linear foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "linearMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#linearMetre" ), "linearMetre", Set.of(new LangString( "linear metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "linearYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#linearYard" ), "linearYard", Set.of(new LangString( "linear yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LY"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "link", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#link" ), "link", Set.of(new LangString( "link", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LK"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "liquidPintUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#liquidPintUs" ), "liquidPintUs", Set.of(new LangString( "liquid pint (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("liq pt (US)"), Optional.of("PTL"), Optional.of("cubicMetre"), Optional.of("4. 731765 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "liquidPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#liquidPound" ), "liquidPound", Set.of(new LangString( "liquid pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "liquidQuartUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#liquidQuartUs" ), "liquidQuartUs", Set.of(new LangString( "liquid quart (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("liq qt (US)"), Optional.of("QTL"), Optional.of("cubicMetre"), Optional.of("9.463529 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "litre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litre" ), "litre", Set.of(new LangString( "litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l"), Optional.of("LTR"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "litreOfPureAlcohol", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litreOfPureAlcohol" ), "litreOfPureAlcohol", Set.of(new LangString( "litre of pure alcohol", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LPA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "litrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerBar" ), "litrePerBar", Set.of(new LangString( "litre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/bar"), Optional.of("G95"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "litrePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerDay" ), "litrePerDay", Set.of(new LangString( "litre per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/d"), Optional.of("LD"), Optional.of("cubicMetrePerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerDayBar" ), "litrePerDayBar", Set.of(new LangString( "litre per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(d\u00B7bar)"), Optional.of("G82"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B3 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerDayKelvin" ), "litrePerDayKelvin", Set.of(new LangString( "litre per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(d\u00B7K)"), Optional.of("G65"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerHour" ), "litrePerHour", Set.of(new LangString( "litre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/h"), Optional.of("E32"), Optional.of("cubicMetrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2077 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "litrePerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerHourBar" ), "litrePerHourBar", Set.of(new LangString( "litre per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(h\u00B7bar)"), Optional.of("G83"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B2 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerHourKelvin" ), "litrePerHourKelvin", Set.of(new LangString( "litre per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(h\u00B7K)"), Optional.of("G66"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
}

private static void init8() {
UNITS_BY_NAME.put( "litrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerKelvin" ), "litrePerKelvin", Set.of(new LangString( "litre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/K"), Optional.of("G28"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "litrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerKilogram" ), "litrePerKilogram", Set.of(new LangString( "litre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/kg"), Optional.of("H83"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "litrePerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerLitre" ), "litrePerLitre", Set.of(new LangString( "litre per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/l"), Optional.of("K62"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "litrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerMinute" ), "litrePerMinute", Set.of(new LangString( "litre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/min"), Optional.of("L2"), Optional.of("cubicMetrePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerMinuteBar" ), "litrePerMinuteBar", Set.of(new LangString( "litre per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(min\u00B7bar)"), Optional.of("G84"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u2070 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerMinuteKelvin" ), "litrePerMinuteKelvin", Set.of(new LangString( "litre per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(min\u00B7K)"), Optional.of("G67"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerMole" ), "litrePerMole", Set.of(new LangString( "litre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/mol"), Optional.of("B58"), Optional.of("cubicMetrePerMole"), Optional.of("10\u207B\u00B3 m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "litrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerSecond" ), "litrePerSecond", Set.of(new LangString( "litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/s"), Optional.of("G51"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerSecondBar" ), "litrePerSecondBar", Set.of(new LangString( "litre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(s\u00B7bar)"), Optional.of("G85"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#litrePerSecondKelvin" ), "litrePerSecondKelvin", Set.of(new LangString( "litre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("l/(s\u00B7K)"), Optional.of("G68"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "load", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#load" ), "load", Set.of(new LangString( "load", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lotUnitOfProcurement", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lotUnitOfProcurement" ), "lotUnitOfProcurement", Set.of(new LangString( "lot  [unit of procurement]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lotUnitOfWeight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lotUnitOfWeight" ), "lotUnitOfWeight", Set.of(new LangString( "lot  [unit of weight]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("D04"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lumen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumen" ), "lumen", Set.of(new LangString( "lumen", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lm"), Optional.of("LUM"), Optional.empty(), Optional.of("cd \u00D7 sr"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_FLUX)) ) );
UNITS_BY_NAME.put( "lumenHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumenHour" ), "lumenHour", Set.of(new LangString( "lumen hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lm\u00B7h"), Optional.of("B59"), Optional.empty(), Optional.of("3.6 \u00D7 10\u00B3 s \u00D7 cd \u00D7 sr"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_LIGHT)) ) );
UNITS_BY_NAME.put( "lumenPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumenPerSquareFoot" ), "lumenPerSquareFoot", Set.of(new LangString( "lumen per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lm/ft\u00B2"), Optional.of("P25"), Optional.empty(), Optional.of("1.076391 \u00D7 10\u00B9 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "lumenPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumenPerSquareMetre" ), "lumenPerSquareMetre", Set.of(new LangString( "lumen per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lm/m\u00B2"), Optional.of("B60"), Optional.empty(), Optional.of("cd \u00D7 sr/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_EXITANCE)) ) );
UNITS_BY_NAME.put( "lumenPerWatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumenPerWatt" ), "lumenPerWatt", Set.of(new LangString( "lumen per watt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lm/W"), Optional.of("B61"), Optional.empty(), Optional.of("cd \u00D7 sr/W"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_LUMINOUS_EFFICACY, QuantityKinds.MAXIMUM_SPECTRAL_LUMINOUS_EFFICACY, QuantityKinds.LUMINOUS_EFFICACY_AT_A_SPECIFIED_WAVELENGTH, QuantityKinds.LUMINIOUS_EFFICACY)) ) );
UNITS_BY_NAME.put( "lumenSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumenSecond" ), "lumenSecond", Set.of(new LangString( "lumen second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lm\u00B7s"), Optional.of("B62"), Optional.empty(), Optional.of("s \u00D7 cd \u00D7 sr"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_LIGHT)) ) );
UNITS_BY_NAME.put( "lumpSum", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lumpSum" ), "lumpSum", Set.of(new LangString( "lump sum", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lux", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#lux" ), "lux", Set.of(new LangString( "lux", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lx"), Optional.of("LUX"), Optional.empty(), Optional.of("cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "luxHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#luxHour" ), "luxHour", Set.of(new LangString( "lux hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lx\u00B7h"), Optional.of("B63"), Optional.empty(), Optional.of("3.6 \u00D7 10\u00B3 s \u00D7 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LIGHT_EXPOSURE)) ) );
UNITS_BY_NAME.put( "luxSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#luxSecond" ), "luxSecond", Set.of(new LangString( "lux second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lx\u00B7s"), Optional.of("B64"), Optional.empty(), Optional.of("s \u00D7 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LIGHT_EXPOSURE)) ) );
UNITS_BY_NAME.put( "manmonth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#manmonth" ), "manmonth", Set.of(new LangString( "manmonth", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("3C"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "meal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#meal" ), "meal", Set.of(new LangString( "meal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("Q3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mebibit" ), "mebibit", Set.of(new LangString( "mebibit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mibit"), Optional.of("D11"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mebibitPerCubicMetre" ), "mebibitPerCubicMetre", Set.of(new LangString( "mebibit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mibit/m\u00B3"), Optional.of("E77"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mebibitPerMetre" ), "mebibitPerMetre", Set.of(new LangString( "mebibit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mibit/m"), Optional.of("E75"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mebibitPerSquareMetre" ), "mebibitPerSquareMetre", Set.of(new LangString( "mebibit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mibit/m\u00B2"), Optional.of("E76"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mebibyte" ), "mebibyte", Set.of(new LangString( "mebibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mibyte"), Optional.of("E63"), Optional.of("byte"), Optional.of("2\u00B2\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "megaampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaampere" ), "megaampere", Set.of(new LangString( "megaampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MA"), Optional.of("H38"), Optional.of("ampere"), Optional.of("10\u2076 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT)) ) );
UNITS_BY_NAME.put( "megaamperePerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaamperePerSquareMetre" ), "megaamperePerSquareMetre", Set.of(new LangString( "megaampere per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MA/m\u00B2"), Optional.of("B66"), Optional.of("amperePerSquareMetre"), Optional.of("10\u2076 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "megabaud", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabaud" ), "megabaud", Set.of(new LangString( "megabaud", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MBd"), Optional.of("J54"), Optional.of("baud"), Optional.of("10\u2076 Bd"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megabecquerel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabecquerel" ), "megabecquerel", Set.of(new LangString( "megabecquerel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MBq"), Optional.of("4N"), Optional.of("becquerel"), Optional.of("10\u2076 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "megabecquerelPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabecquerelPerKilogram" ), "megabecquerelPerKilogram", Set.of(new LangString( "megabecquerel per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MBq/kg"), Optional.of("B67"), Optional.of("becquerelPerKilogram"), Optional.of("10\u2076 Bq/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "megabit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabit" ), "megabit", Set.of(new LangString( "megabit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mbit"), Optional.of("D36"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megabitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabitPerSecond" ), "megabitPerSecond", Set.of(new LangString( "megabit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mbit/s"), Optional.of("E20"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabyte" ), "megabyte", Set.of(new LangString( "Megabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MB"), Optional.of("4L"), Optional.of("byte"), Optional.of("10\u2076 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "megabytePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megabytePerSecond" ), "megabytePerSecond", Set.of(new LangString( "megabyte per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mbyte/s"), Optional.of("P95"), Optional.of("bytePerSecond"), Optional.of("10\u2076 byte/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megacoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megacoulomb" ), "megacoulomb", Set.of(new LangString( "megacoulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MC"), Optional.of("D77"), Optional.of("coulomb"), Optional.of("10\u2076 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "megacoulombPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megacoulombPerCubicMetre" ), "megacoulombPerCubicMetre", Set.of(new LangString( "megacoulomb per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MC/m\u00B3"), Optional.of("B69"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2076 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "megacoulombPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megacoulombPerSquareMetre" ), "megacoulombPerSquareMetre", Set.of(new LangString( "megacoulomb per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MC/m\u00B2"), Optional.of("B70"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u2076 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "megaelectronvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaelectronvolt" ), "megaelectronvolt", Set.of(new LangString( "megaelectronvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MeV"), Optional.of("B71"), Optional.of("electronvolt"), Optional.of("10\u2076 eV"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "megagram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megagram" ), "megagram", Set.of(new LangString( "megagram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mg"), Optional.of("2U"), Optional.of("kilogram"), Optional.of("10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "megagramPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megagramPerCubicMetre" ), "megagramPerCubicMetre", Set.of(new LangString( "megagram per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mg/m\u00B3"), Optional.of("B72"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "megahertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megahertz" ), "megahertz", Set.of(new LangString( "megahertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MHz"), Optional.of("MHZ"), Optional.of("hertz"), Optional.of("10\u2076 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "megahertzKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megahertzKilometre" ), "megahertzKilometre", Set.of(new LangString( "megahertz kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MHz\u00B7km"), Optional.of("H39"), Optional.of("hertzMetre"), Optional.of("10\u2079 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "megahertzMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megahertzMetre" ), "megahertzMetre", Set.of(new LangString( "megahertz metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MHz\u00B7m"), Optional.of("M27"), Optional.of("hertzMetre"), Optional.of("10\u2076 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "megajoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megajoule" ), "megajoule", Set.of(new LangString( "megajoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MJ"), Optional.of("3B"), Optional.of("joule"), Optional.of("10\u2076 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "megajoulePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megajoulePerCubicMetre" ), "megajoulePerCubicMetre", Set.of(new LangString( "megajoule per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MJ/m\u00B3"), Optional.of("JM"), Optional.of("joulePerCubicMetre"), Optional.of("10\u2076 J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "megajoulePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megajoulePerKilogram" ), "megajoulePerKilogram", Set.of(new LangString( "megajoule per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MJ/kg"), Optional.of("JK"), Optional.of("joulePerKilogram"), Optional.of("10\u2076 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "megajoulePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megajoulePerSecond" ), "megajoulePerSecond", Set.of(new LangString( "megajoule per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MJ/s"), Optional.of("D78"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megalitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megalitre" ), "megalitre", Set.of(new LangString( "megalitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Ml"), Optional.of("MAL"), Optional.of("cubicMetre"), Optional.of("10\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "megametre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megametre" ), "megametre", Set.of(new LangString( "megametre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mm"), Optional.of("MAM"), Optional.of("metre"), Optional.of("10\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "meganewton", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#meganewton" ), "meganewton", Set.of(new LangString( "meganewton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MN"), Optional.of("B73"), Optional.of("newton"), Optional.of("10\u2076 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "meganewtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#meganewtonMetre" ), "meganewtonMetre", Set.of(new LangString( "meganewton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MN\u00B7m"), Optional.of("B74"), Optional.of("newtonMetre"), Optional.of("10\u2076 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "megaohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaohm" ), "megaohm", Set.of(new LangString( "megaohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("M\u2126"), Optional.of("B75"), Optional.of("ohm"), Optional.of("10\u2076 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.IMPEDANCE, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.REACTANCE)) ) );
UNITS_BY_NAME.put( "megaohmKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaohmKilometre" ), "megaohmKilometre", Set.of(new LangString( "megaohm kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("M\u2126\u00B7km"), Optional.of("H88"), Optional.of("ohmMetre"), Optional.of("10\u2079 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "megaohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaohmMetre" ), "megaohmMetre", Set.of(new LangString( "megaohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("M\u2126\u00B7m"), Optional.of("B76"), Optional.of("ohmMetre"), Optional.of("10\u2076 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "megaohmPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaohmPerKilometre" ), "megaohmPerKilometre", Set.of(new LangString( "megaohm per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("M\u2126/km"), Optional.of("H36"), Optional.of("ohmPerMetre"), Optional.of("10\u00B3 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "megaohmPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megaohmPerMetre" ), "megaohmPerMetre", Set.of(new LangString( "megaohm per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("M\u2126/m"), Optional.of("H37"), Optional.of("ohmPerMetre"), Optional.of("10\u2076 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "megapascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megapascal" ), "megapascal", Set.of(new LangString( "megapascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MPa"), Optional.of("MPA"), Optional.of("pascal"), Optional.of("10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "megapascalCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megapascalCubicMetrePerSecond" ), "megapascalCubicMetrePerSecond", Set.of(new LangString( "megapascal cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MPa\u00B7m\u00B3/s"), Optional.of("F98"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "megapascalLitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megapascalLitrePerSecond" ), "megapascalLitrePerSecond", Set.of(new LangString( "megapascal litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MPa\u00B7l/s"), Optional.of("F97"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "megapascalPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megapascalPerBar" ), "megapascalPerBar", Set.of(new LangString( "megapascal per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MPa/bar"), Optional.of("F05"), Optional.empty(), Optional.of("10\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "megapascalPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megapascalPerKelvin" ), "megapascalPerKelvin", Set.of(new LangString( "megapascal per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MPa/K"), Optional.of("F85"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "megapixel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megapixel" ), "megapixel", Set.of(new LangString( "megapixel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E38"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megasiemensPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megasiemensPerMetre" ), "megasiemensPerMetre", Set.of(new LangString( "megasiemens per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MS/m"), Optional.of("B77"), Optional.of("siemensPerMetre"), Optional.of("10\u2076 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "megavar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megavar" ), "megavar", Set.of(new LangString( "megavar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mvar"), Optional.of("MAR"), Optional.of("voltAmpere"), Optional.of("10\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.REACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "megavolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megavolt" ), "megavolt", Set.of(new LangString( "megavolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MV"), Optional.of("B78"), Optional.of("voltDc"), Optional.of("10\u2076 V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.TENSION, QuantityKinds.VOLTAGE)) ) );
UNITS_BY_NAME.put( "megavoltAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megavoltAmpere" ), "megavoltAmpere", Set.of(new LangString( "megavolt - ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MV\u00B7A"), Optional.of("MVA"), Optional.of("voltAmpere"), Optional.of("10\u2076 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "megavoltAmpereReactiveHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megavoltAmpereReactiveHour" ), "megavoltAmpereReactiveHour", Set.of(new LangString( "megavolt ampere reactive hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mvar\u00B7h"), Optional.of("MAH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megavoltPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megavoltPerMetre" ), "megavoltPerMetre", Set.of(new LangString( "megavolt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MV/m"), Optional.of("B79"), Optional.of("voltPerMetre"), Optional.of("10\u2076 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "megawatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megawatt" ), "megawatt", Set.of(new LangString( "megawatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MW"), Optional.of("MAW"), Optional.of("watt"), Optional.of("10\u2076 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "megawattHour1000Kwh", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megawattHour1000Kwh" ), "megawattHour1000Kwh", Set.of(new LangString( "megawatt hour (1000\u00A0kW.h)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MW\u00B7h"), Optional.of("MWH"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u2079 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.POTENTIAL_ENERGY)) ) );
UNITS_BY_NAME.put( "megawattHourPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megawattHourPerHour" ), "megawattHourPerHour", Set.of(new LangString( "megawatt hour per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MW\u00B7h/h"), Optional.of("E07"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megawattPerHertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#megawattPerHertz" ), "megawattPerHertz", Set.of(new LangString( "megawatt per hertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MW/Hz"), Optional.of("E08"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mesh", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mesh" ), "mesh", Set.of(new LangString( "mesh", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("57"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "message", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#message" ), "message", Set.of(new LangString( "message", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metre" ), "metre", Set.of(new LangString( "metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m"), Optional.of("MTR"), Optional.empty(), Optional.of("m"), new HashSet<>(Arrays.asList(QuantityKinds.DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUENCE_RATE, QuantityKinds.NUCLEAR_RADIUS, QuantityKinds.BREADTH, QuantityKinds.MIGRATION_LENGTH, QuantityKinds.HEIGHT, QuantityKinds.HALF_THICKNESS, QuantityKinds.HALF_VALUE_THICKNESS, QuantityKinds.INSTANTANEOUS_SOUND_PARTICLE_DISPLACEMENT, QuantityKinds.DIFFUSION_LENGTH, QuantityKinds.RADIUS, QuantityKinds.DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUX_DENSITY, QuantityKinds.DIAMETER, QuantityKinds.FOCAL_DISTANCE, QuantityKinds.MEAN_LINEAR_RANGE, QuantityKinds.COHERENCE_LENGTH, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.LONDON_PENETRATION_DEPTH, QuantityKinds.PARTICLE_POSITION_VECTOR, QuantityKinds.DISPLACEMENT_VECTOR_OF_ION_OR_ATOM, QuantityKinds.MEAN_FREE_PATH_OF_PHONONS_OR_ELECTRONS, QuantityKinds.BOHR_RADIUS, QuantityKinds.WAVELENGTH, QuantityKinds.LATTICE_VECTOR, QuantityKinds.COMPTON_WAVELENGTH, QuantityKinds.IMAGE_DISTANCE, QuantityKinds.MEAN_FREE_PATH, QuantityKinds.SLOWING_DOWN_LENGTH, QuantityKinds.OBJECT_DISTANCE, QuantityKinds.FUNDAMENTAL_LATTICE_VECTOR, QuantityKinds.ELECTRON_RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.EQUILIBRIUM_POSITION_VECTOR_OF_ION_OR_ATOM, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "metreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metreKelvin" ), "metreKelvin", Set.of(new LangString( "metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B7K"), Optional.of("D18"), Optional.empty(), Optional.of("m \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_RADIATION_CONSTANT)) ) );
UNITS_BY_NAME.put( "metrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerBar" ), "metrePerBar", Set.of(new LangString( "metre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/bar"), Optional.of("G05"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER)) ) );
UNITS_BY_NAME.put( "metrePerDegreeCelsiusMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerDegreeCelsiusMetre" ), "metrePerDegreeCelsiusMetre", Set.of(new LangString( "metre per degree Celsius metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/(\u00B0C\u00B7m)"), Optional.of("N83"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "metrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerHour" ), "metrePerHour", Set.of(new LangString( "metre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/h"), Optional.of("M60"), Optional.of("metrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerKelvin" ), "metrePerKelvin", Set.of(new LangString( "metre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/K"), Optional.of("F52"), Optional.empty(), Optional.of("m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THICKNESS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES)) ) );
UNITS_BY_NAME.put( "metrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerMinute" ), "metrePerMinute", Set.of(new LangString( "metre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/min"), Optional.of("2X"), Optional.of("metrePerSecond"), Optional.of("0.016666 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerPascal" ), "metrePerPascal", Set.of(new LangString( "metre per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/Pa"), Optional.of("M53"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "metrePerRadiant", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerRadiant" ), "metrePerRadiant", Set.of(new LangString( "metre per radiant", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/rad"), Optional.of("M55"), Optional.empty(), Optional.of("m/rad"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "metrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerSecond" ), "metrePerSecond", Set.of(new LangString( "metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/s"), Optional.of("MTS"), Optional.empty(), Optional.of("m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_SPEED_OF_ELECTROMAGNETIC_WAVES, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY_OF_ELECTROMAGNETIC_WAVES, QuantityKinds.PHASE_VELOCITY, QuantityKinds.NEUTRON_SPEED, QuantityKinds.VELOCITY_OF_SOUND_PHASE_VELOCITY, QuantityKinds.INSTANTANEOUS_SOUND_PARTICLE_VELOCITY, QuantityKinds.VELOCITY_SPEED_ON_PROPAGATION_OF_ELECTROMAGNETIC_WAVES_IN_VACUO)) ) );
UNITS_BY_NAME.put( "metrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerSecondBar" ), "metrePerSecondBar", Set.of(new LangString( "metre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(m/s)/bar"), Optional.of("L13"), Optional.of("metrePerSecondPascal"), Optional.of("10\u207B\u2075 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerSecondKelvin" ), "metrePerSecondKelvin", Set.of(new LangString( "metre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(m/s)/K"), Optional.of("L12"), Optional.empty(), Optional.of("(m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerSecondPascal" ), "metrePerSecondPascal", Set.of(new LangString( "metre per second pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(m/s)/Pa"), Optional.of("M59"), Optional.empty(), Optional.of("m\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerSecondSquared" ), "metrePerSecondSquared", Set.of(new LangString( "metre per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/s\u00B2"), Optional.of("MSK"), Optional.empty(), Optional.of("m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.INSTANTANEOUS_SOUND_PARTICLE_ACCELERATION, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "metrePerVoltSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metrePerVoltSecond" ), "metrePerVoltSecond", Set.of(new LangString( "metre per volt second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m/(V\u00B7s)"), Optional.of("H58"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.MOBILITY)) ) );
UNITS_BY_NAME.put( "metreToTheFourthPower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metreToTheFourthPower" ), "metreToTheFourthPower", Set.of(new LangString( "metre to the fourth power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u2074"), Optional.of("B83"), Optional.empty(), Optional.of("m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_MOMENT_OF_AREA, QuantityKinds.SECOND_AXIAL_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "metricCarat", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metricCarat" ), "metricCarat", Set.of(new LangString( "metric carat", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("CTM"), Optional.of("milligram"), Optional.of("200 mg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metricTonIncludingContainer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metricTonIncludingContainer" ), "metricTonIncludingContainer", Set.of(new LangString( "metric ton, including container", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TIC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metricTonIncludingInnerPackaging", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metricTonIncludingInnerPackaging" ), "metricTonIncludingInnerPackaging", Set.of(new LangString( "metric ton, including inner packaging", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TIP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metricTonLubricatingOil", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#metricTonLubricatingOil" ), "metricTonLubricatingOil", Set.of(new LangString( "metric ton, lubricating oil", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LUB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "microInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microInch" ), "microInch", Set.of(new LangString( "micro-inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5in"), Optional.of("M7"), Optional.of("metre"), Optional.of("25.4 \u00D7 10\u207B\u2079 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT)) ) );
}

private static void init9() {
UNITS_BY_NAME.put( "microampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microampere" ), "microampere", Set.of(new LangString( "microampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5A"), Optional.of("B84"), Optional.of("ampere"), Optional.of("10\u207B\u2076 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "microbar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microbar" ), "microbar", Set.of(new LangString( "microbar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5bar"), Optional.of("B85"), Optional.of("pascal"), Optional.of("10\u207B\u00B9 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "microbecquerel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microbecquerel" ), "microbecquerel", Set.of(new LangString( "microbecquerel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Bq"), Optional.of("H08"), Optional.of("becquerel"), Optional.of("10\u207B\u2076 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "microcoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microcoulomb" ), "microcoulomb", Set.of(new LangString( "microcoulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5C"), Optional.of("B86"), Optional.of("coulomb"), Optional.of("10\u207B\u2076 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "microcoulombPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microcoulombPerCubicMetre" ), "microcoulombPerCubicMetre", Set.of(new LangString( "microcoulomb per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5C/m\u00B3"), Optional.of("B87"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u207B\u2076 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUMIC_CHARGE)) ) );
UNITS_BY_NAME.put( "microcoulombPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microcoulombPerSquareMetre" ), "microcoulombPerSquareMetre", Set.of(new LangString( "microcoulomb per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5C/m\u00B2"), Optional.of("B88"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u207B\u2076 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "microcurie", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microcurie" ), "microcurie", Set.of(new LangString( "microcurie", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Ci"), Optional.of("M5"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u2074 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "microfarad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microfarad" ), "microfarad", Set.of(new LangString( "microfarad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5F"), Optional.of("4O"), Optional.of("farad"), Optional.of("10\u207B\u2076 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "microfaradPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microfaradPerKilometre" ), "microfaradPerKilometre", Set.of(new LangString( "microfarad per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5F/km"), Optional.of("H28"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u2079 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.ELECTRIC_CONSTANT, QuantityKinds.PERMITTIVITY)) ) );
UNITS_BY_NAME.put( "microfaradPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microfaradPerMetre" ), "microfaradPerMetre", Set.of(new LangString( "microfarad per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5F/m"), Optional.of("B89"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u2076 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY, QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.ELECTRIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "microgram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microgram" ), "microgram", Set.of(new LangString( "microgram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5g"), Optional.of("MC"), Optional.of("kilogram"), Optional.of("10\u207B\u2079 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "microgramPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microgramPerCubicMetre" ), "microgramPerCubicMetre", Set.of(new LangString( "microgram per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5g/m\u00B3"), Optional.of("GQ"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u2079 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "microgramPerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microgramPerCubicMetreBar" ), "microgramPerCubicMetreBar", Set.of(new LangString( "microgram per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(\u00B5g/m\u00B3)/bar"), Optional.of("J35"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("10\u207B\u00B9\u2074 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "microgramPerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microgramPerCubicMetreKelvin" ), "microgramPerCubicMetreKelvin", Set.of(new LangString( "microgram per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(\u00B5g/m\u00B3)/K"), Optional.of("J34"), Optional.empty(), Optional.of("10\u207B\u2079 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "microgramPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microgramPerKilogram" ), "microgramPerKilogram", Set.of(new LangString( "microgram per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5g/kg"), Optional.of("J33"), Optional.empty(), Optional.of("10\u207B\u2079"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "microgramPerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microgramPerLitre" ), "microgramPerLitre", Set.of(new LangString( "microgram per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5g/l"), Optional.of("H29"), Optional.empty(), Optional.of("10\u207B\u2076 m\u207B\u00B3 \u00D7 kg"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "micrograyPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micrograyPerHour" ), "micrograyPerHour", Set.of(new LangString( "microgray per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Gy/h"), Optional.of("P63"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "micrograyPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micrograyPerMinute" ), "micrograyPerMinute", Set.of(new LangString( "microgray per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Gy/min"), Optional.of("P59"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2078 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "micrograyPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micrograyPerSecond" ), "micrograyPerSecond", Set.of(new LangString( "microgray per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Gy/s"), Optional.of("P55"), Optional.of("grayPerSecond"), Optional.of("10\u207B\u2076 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "microhenry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microhenry" ), "microhenry", Set.of(new LangString( "microhenry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5H"), Optional.of("B90"), Optional.of("henry"), Optional.of("10\u207B\u2076 H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "microhenryPerKiloohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microhenryPerKiloohm" ), "microhenryPerKiloohm", Set.of(new LangString( "microhenry per kiloohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5H/k\u2126"), Optional.of("G98"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2079 s"), new HashSet<>(Arrays.asList(QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE)) ) );
UNITS_BY_NAME.put( "microhenryPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microhenryPerMetre" ), "microhenryPerMetre", Set.of(new LangString( "microhenry per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5H/m"), Optional.of("B91"), Optional.of("henryPerMetre"), Optional.of("10\u207B\u2076 H/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEABILITY, QuantityKinds.PERMEABILITY_OF_VACUUM, QuantityKinds.MAGNETIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "microhenryPerOhm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microhenryPerOhm" ), "microhenryPerOhm", Set.of(new LangString( "microhenry per ohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5H/\u2126"), Optional.of("G99"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "microlitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microlitre" ), "microlitre", Set.of(new LangString( "microlitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5l"), Optional.of("4G"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "microlitrePerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microlitrePerLitre" ), "microlitrePerLitre", Set.of(new LangString( "microlitre per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5l/l"), Optional.of("J36"), Optional.empty(), Optional.of("10\u207B\u2076"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "micrometreMicron", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micrometreMicron" ), "micrometreMicron", Set.of(new LangString( "micrometre (micron)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5m"), Optional.of("4H"), Optional.of("metre"), Optional.of("10\u207B\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "micrometrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micrometrePerKelvin" ), "micrometrePerKelvin", Set.of(new LangString( "micrometre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5m/K"), Optional.of("F50"), Optional.empty(), Optional.of("10\u207B\u2076 m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "micromole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micromole" ), "micromole", Set.of(new LangString( "micromole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5mol"), Optional.of("FH"), Optional.of("mole"), Optional.of("10\u207B\u2076 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "micronewton", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micronewton" ), "micronewton", Set.of(new LangString( "micronewton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5N"), Optional.of("B92"), Optional.of("newton"), Optional.of("10\u207B\u2076 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "micronewtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micronewtonMetre" ), "micronewtonMetre", Set.of(new LangString( "micronewton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5N\u00B7m"), Optional.of("B93"), Optional.of("newtonMetre"), Optional.of("10\u207B\u2076 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "microohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microohm" ), "microohm", Set.of(new LangString( "microohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5\u2126"), Optional.of("B94"), Optional.of("ohm"), Optional.of("10\u207B\u2076 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.REACTANCE)) ) );
UNITS_BY_NAME.put( "microohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microohmMetre" ), "microohmMetre", Set.of(new LangString( "microohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5\u2126\u00B7m"), Optional.of("B95"), Optional.of("ohmMetre"), Optional.of("10\u207B\u2076 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "micropascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micropascal" ), "micropascal", Set.of(new LangString( "micropascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Pa"), Optional.of("B96"), Optional.of("pascal"), Optional.of("10\u207B\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "micropoise", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#micropoise" ), "micropoise", Set.of(new LangString( "micropoise", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5P"), Optional.of("J32"), Optional.of("pascalSecond"), Optional.of("10\u207B\u2076 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "microradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microradian" ), "microradian", Set.of(new LangString( "microradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5rad"), Optional.of("B97"), Optional.of("radian"), Optional.of("10\u207B\u2076 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "microsecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsecond" ), "microsecond", Set.of(new LangString( "microsecond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5s"), Optional.of("B98"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "microsiemens", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsiemens" ), "microsiemens", Set.of(new LangString( "microsiemens", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5S"), Optional.of("B99"), Optional.of("siemens"), Optional.of("10\u207B\u2076 S"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "microsiemensPerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsiemensPerCentimetre" ), "microsiemensPerCentimetre", Set.of(new LangString( "microsiemens per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5S/cm"), Optional.of("G42"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2074 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.ADMITTANCE, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE)) ) );
UNITS_BY_NAME.put( "microsiemensPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsiemensPerMetre" ), "microsiemensPerMetre", Set.of(new LangString( "microsiemens per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5S/m"), Optional.of("G43"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2076 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT)) ) );
UNITS_BY_NAME.put( "microsievertPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsievertPerHour" ), "microsievertPerHour", Set.of(new LangString( "microsievert per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Sv/h"), Optional.of("P72"), Optional.of("sievertPerSecond"), Optional.of("0.277777778 \u00D7 10\u207B\u00B9\u2070 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "microsievertPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsievertPerMinute" ), "microsievertPerMinute", Set.of(new LangString( "microsievert per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Sv/min"), Optional.of("P76"), Optional.of("sievertPerSecond"), Optional.of("1.666666667 \u00D7 10\u207B\u2078 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "microsievertPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microsievertPerSecond" ), "microsievertPerSecond", Set.of(new LangString( "microsievert per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5Sv/s"), Optional.of("P67"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u2076 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "microtesla", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microtesla" ), "microtesla", Set.of(new LangString( "microtesla", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5T"), Optional.of("D81"), Optional.of("tesla"), Optional.of("10\u207B\u2076 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_INDUCTION, QuantityKinds.MAGNETIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "microvolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microvolt" ), "microvolt", Set.of(new LangString( "microvolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5V"), Optional.of("D82"), Optional.of("voltDc"), Optional.of("10\u207B\u2076 V"), new HashSet<>(Arrays.asList(QuantityKinds.VOLTAGE, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.TENSION)) ) );
UNITS_BY_NAME.put( "microvoltPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microvoltPerMetre" ), "microvoltPerMetre", Set.of(new LangString( "microvolt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5V/m"), Optional.of("C3"), Optional.of("voltPerMetre"), Optional.of("10\u207B\u2076 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "microwatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microwatt" ), "microwatt", Set.of(new LangString( "microwatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5W"), Optional.of("D80"), Optional.of("watt"), Optional.of("10\u207B\u2076 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "microwattPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#microwattPerSquareMetre" ), "microwattPerSquareMetre", Set.of(new LangString( "microwatt per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5W/m\u00B2"), Optional.of("D85"), Optional.of("wattPerSquareMetre"), Optional.of("10\u207B\u2076 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY)) ) );
UNITS_BY_NAME.put( "mil", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mil" ), "mil", Set.of(new LangString( "mil", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mil"), Optional.of("M43"), Optional.of("radian"), Optional.of("9.817477 \u00D7 10\u207B\u2074 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "mileBasedOnUsSurveyFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mileBasedOnUsSurveyFoot" ), "mileBasedOnUsSurveyFoot", Set.of(new LangString( "mile (based on U.S. survey foot)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi (US survey)"), Optional.of("M52"), Optional.of("metre"), Optional.of("1.609347 \u00D7 10\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "milePerHourStatuteMile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milePerHourStatuteMile" ), "milePerHourStatuteMile", Set.of(new LangString( "mile per hour (statute mile)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mile/h"), Optional.of("HM"), Optional.of("metrePerSecond"), Optional.of("0.44704 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "milePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milePerMinute" ), "milePerMinute", Set.of(new LangString( "mile per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi/min"), Optional.of("M57"), Optional.of("metrePerSecond"), Optional.of("26.8224 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "milePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milePerSecond" ), "milePerSecond", Set.of(new LangString( "mile per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi/s"), Optional.of("M58"), Optional.of("metrePerSecond"), Optional.of("1.609344 \u00D7 10\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "mileStatuteMile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mileStatuteMile" ), "mileStatuteMile", Set.of(new LangString( "mile (statute mile)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mile"), Optional.of("SMI"), Optional.of("metre"), Optional.of("1609.344 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "mileStatuteMilePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mileStatuteMilePerSecondSquared" ), "mileStatuteMilePerSecondSquared", Set.of(new LangString( "mile (statute mile) per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi/s\u00B2"), Optional.of("M42"), Optional.of("metrePerSecondSquared"), Optional.of("1.609344 \u00D7 10\u00B3 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "mille", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mille" ), "mille", Set.of(new LangString( "mille", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E12"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "milliInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliInch" ), "milliInch", Set.of(new LangString( "milli-inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mil"), Optional.of("77"), Optional.of("metre"), Optional.of("25.4 \u00D7 10\u207B\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "milliampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliampere" ), "milliampere", Set.of(new LangString( "milliampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA"), Optional.of("4K"), Optional.of("ampere"), Optional.of("10\u207B\u00B3 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_LINKAGE, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.ELECTRIC_CURRENT)) ) );
UNITS_BY_NAME.put( "milliampereHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliampereHour" ), "milliampereHour", Set.of(new LangString( "milliampere hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA\u00B7h"), Optional.of("E09"), Optional.of("coulomb"), Optional.of("3.6 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "milliamperePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliamperePerBar" ), "milliamperePerBar", Set.of(new LangString( "milliampere per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA/bar"), Optional.of("F59"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliamperePerInch" ), "milliamperePerInch", Set.of(new LangString( "milliampere per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA/in"), Optional.of("F08"), Optional.empty(), Optional.of("3.93700787401575 \u00D7 10\u207B\u00B2 A \u00D7 m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerLitreMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliamperePerLitreMinute" ), "milliamperePerLitreMinute", Set.of(new LangString( "milliampere per litre minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA/(l\u00B7min)"), Optional.of("G59"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 m\u207B\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliamperePerMillimetre" ), "milliamperePerMillimetre", Set.of(new LangString( "milliampere per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA/mm"), Optional.of("F76"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerPoundForcePerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliamperePerPoundForcePerSquareInch" ), "milliamperePerPoundForcePerSquareInch", Set.of(new LangString( "milliampere per pound-force per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mA/(lbf/in\u00B2)"), Optional.of("F57"), Optional.empty(), Optional.of("1.45038 \u00D7 10\u207B\u2077 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliard" ), "milliard", Set.of(new LangString( "milliard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MLD"), Optional.empty(), Optional.of("10\u2079"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millibar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millibar" ), "millibar", Set.of(new LangString( "millibar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mbar"), Optional.of("MBR"), Optional.of("pascal"), Optional.of("10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "millibarCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millibarCubicMetrePerSecond" ), "millibarCubicMetrePerSecond", Set.of(new LangString( "millibar cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mbar\u00B7m\u00B3/s"), Optional.of("F96"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "millibarLitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millibarLitrePerSecond" ), "millibarLitrePerSecond", Set.of(new LangString( "millibar litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mbar\u00B7l/s"), Optional.of("F95"), Optional.empty(), Optional.of("10\u207B\u00B9 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "millibarPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millibarPerBar" ), "millibarPerBar", Set.of(new LangString( "millibar per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mbar/bar"), Optional.of("F04"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "millibarPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millibarPerKelvin" ), "millibarPerKelvin", Set.of(new LangString( "millibar per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mbar/K"), Optional.of("F84"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "millicandela", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millicandela" ), "millicandela", Set.of(new LangString( "millicandela", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mcd"), Optional.of("P34"), Optional.of("candela"), Optional.of("10\u207B\u00B3 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "millicoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millicoulomb" ), "millicoulomb", Set.of(new LangString( "millicoulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mC"), Optional.of("D86"), Optional.of("coulomb"), Optional.of("10\u207B\u00B3 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "millicoulombPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millicoulombPerCubicMetre" ), "millicoulombPerCubicMetre", Set.of(new LangString( "millicoulomb per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mC/m\u00B3"), Optional.of("D88"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u207B\u00B3 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "millicoulombPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millicoulombPerKilogram" ), "millicoulombPerKilogram", Set.of(new LangString( "millicoulomb per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mC/kg"), Optional.of("C8"), Optional.of("coulombPerKilogram"), Optional.of("10\u207B\u00B3 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "millicoulombPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millicoulombPerSquareMetre" ), "millicoulombPerSquareMetre", Set.of(new LangString( "millicoulomb per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mC/m\u00B2"), Optional.of("D89"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u207B\u00B3 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "millicurie", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millicurie" ), "millicurie", Set.of(new LangString( "millicurie", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mCi"), Optional.of("MCU"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u2077 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "milliequivalenceCausticPotashPerGramOfProduct", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliequivalenceCausticPotashPerGramOfProduct" ), "milliequivalenceCausticPotashPerGramOfProduct", Set.of(new LangString( "milliequivalence caustic potash per gram of product", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("KO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millifarad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millifarad" ), "millifarad", Set.of(new LangString( "millifarad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mF"), Optional.of("C10"), Optional.of("farad"), Optional.of("10\u207B\u00B3 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "milligal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligal" ), "milligal", Set.of(new LangString( "milligal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mGal"), Optional.of("C11"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u2075 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "milligram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligram" ), "milligram", Set.of(new LangString( "milligram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg"), Optional.of("MGM"), Optional.of("kilogram"), Optional.of("10\u207B\u2076 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "milligramPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerBar" ), "milligramPerBar", Set.of(new LangString( "milligram per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/bar"), Optional.of("F75"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerCubicMetre" ), "milligramPerCubicMetre", Set.of(new LangString( "milligram per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/m\u00B3"), Optional.of("GP"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u2076 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "milligramPerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerCubicMetreBar" ), "milligramPerCubicMetreBar", Set.of(new LangString( "milligram per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mg/m\u00B3)/bar"), Optional.of("L18"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("10\u207B\u00B9\u00B9 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerCubicMetreKelvin" ), "milligramPerCubicMetreKelvin", Set.of(new LangString( "milligram per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mg/m\u00B3)/K"), Optional.of("L17"), Optional.empty(), Optional.of("10\u207B\u2076 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerDay" ), "milligramPerDay", Set.of(new LangString( "milligram per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/d"), Optional.of("F32"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerDayBar" ), "milligramPerDayBar", Set.of(new LangString( "milligram per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(d\u00B7bar)"), Optional.of("F70"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2076 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerDayKelvin" ), "milligramPerDayKelvin", Set.of(new LangString( "milligram per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(d\u00B7K)"), Optional.of("F43"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerGram" ), "milligramPerGram", Set.of(new LangString( "milligram per gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/g"), Optional.of("H64"), Optional.empty(), Optional.of("10\u207B\u00B3 1"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerHour" ), "milligramPerHour", Set.of(new LangString( "milligram per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/h"), Optional.of("4M"), Optional.of("kilogramPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerHourBar" ), "milligramPerHourBar", Set.of(new LangString( "milligram per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(h\u00B7bar)"), Optional.of("F71"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2075 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerHourKelvin" ), "milligramPerHourKelvin", Set.of(new LangString( "milligram per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(h\u00B7K)"), Optional.of("F44"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerKelvin" ), "milligramPerKelvin", Set.of(new LangString( "milligram per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/K"), Optional.of("F16"), Optional.empty(), Optional.of("10\u207B\u2076 kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerKilogram" ), "milligramPerKilogram", Set.of(new LangString( "milligram per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/kg"), Optional.of("NA"), Optional.empty(), Optional.of("10\u207B\u2076 1"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "milligramPerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerLitre" ), "milligramPerLitre", Set.of(new LangString( "milligram per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/l"), Optional.of("M1"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerMetre" ), "milligramPerMetre", Set.of(new LangString( "milligram per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/m"), Optional.of("C12"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u2076 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerMinute" ), "milligramPerMinute", Set.of(new LangString( "milligram per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/min"), Optional.of("F33"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerMinuteBar" ), "milligramPerMinuteBar", Set.of(new LangString( "milligram per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(min\u00B7bar)"), Optional.of("F72"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B3 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerMinuteKelvin" ), "milligramPerMinuteKelvin", Set.of(new LangString( "milligram per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(min\u00B7K)"), Optional.of("F45"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerSecond" ), "milligramPerSecond", Set.of(new LangString( "milligram per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/s"), Optional.of("F34"), Optional.empty(), Optional.of("10\u207B\u2076 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerSecondBar" ), "milligramPerSecondBar", Set.of(new LangString( "milligram per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(s\u00B7bar)"), Optional.of("F73"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerSecondKelvin" ), "milligramPerSecondKelvin", Set.of(new LangString( "milligram per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/(s\u00B7K)"), Optional.of("F46"), Optional.empty(), Optional.of("10\u207B\u2076 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
}

private static void init10() {
UNITS_BY_NAME.put( "milligramPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerSquareCentimetre" ), "milligramPerSquareCentimetre", Set.of(new LangString( "milligram per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/cm\u00B2"), Optional.of("H63"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 kg"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "milligramPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligramPerSquareMetre" ), "milligramPerSquareMetre", Set.of(new LangString( "milligram per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mg/m\u00B2"), Optional.of("GO"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u207B\u2076 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "milligray", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligray" ), "milligray", Set.of(new LangString( "milligray", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mGy"), Optional.of("C13"), Optional.of("gray"), Optional.of("10\u207B\u00B3 Gy"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENERGY_IMPARTED, QuantityKinds.MASSIC_ENERGY_IMPARTED)) ) );
UNITS_BY_NAME.put( "milligrayPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligrayPerHour" ), "milligrayPerHour", Set.of(new LangString( "milligray per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mGy/h"), Optional.of("P62"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2077 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "milligrayPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligrayPerMinute" ), "milligrayPerMinute", Set.of(new LangString( "milligray per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mGy/min"), Optional.of("P58"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2075 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "milligrayPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milligrayPerSecond" ), "milligrayPerSecond", Set.of(new LangString( "milligray per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mGy/s"), Optional.of("P54"), Optional.of("grayPerSecond"), Optional.of("10\u207B\u00B3 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "millihenry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millihenry" ), "millihenry", Set.of(new LangString( "millihenry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mH"), Optional.of("C14"), Optional.of("henry"), Optional.of("10\u207B\u00B3 H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE)) ) );
UNITS_BY_NAME.put( "millihenryPerKiloohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millihenryPerKiloohm" ), "millihenryPerKiloohm", Set.of(new LangString( "millihenry per kiloohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mH/k\u2126"), Optional.of("H05"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "millihenryPerOhm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millihenryPerOhm" ), "millihenryPerOhm", Set.of(new LangString( "millihenry per ohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mH/\u2126"), Optional.of("H06"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "millijoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millijoule" ), "millijoule", Set.of(new LangString( "millijoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mJ"), Optional.of("C15"), Optional.of("joule"), Optional.of("10\u207B\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "millilitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitre" ), "millilitre", Set.of(new LangString( "millilitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml"), Optional.of("MLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerBar" ), "millilitrePerBar", Set.of(new LangString( "millilitre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/bar"), Optional.of("G97"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerCubicMetre" ), "millilitrePerCubicMetre", Set.of(new LangString( "millilitre per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/m\u00B3"), Optional.of("H65"), Optional.empty(), Optional.of("10\u207B\u2076 1"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerDay" ), "millilitrePerDay", Set.of(new LangString( "millilitre per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/d"), Optional.of("G54"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerDayBar" ), "millilitrePerDayBar", Set.of(new LangString( "millilitre per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(d\u00B7bar)"), Optional.of("G90"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2076 \u00D7 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerDayKelvin" ), "millilitrePerDayKelvin", Set.of(new LangString( "millilitre per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(d\u00B7K)"), Optional.of("G73"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerHour" ), "millilitrePerHour", Set.of(new LangString( "millilitre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/h"), Optional.of("G55"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerHourBar" ), "millilitrePerHourBar", Set.of(new LangString( "millilitre per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(h\u00B7bar)"), Optional.of("G91"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2075 \u00D7 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerHourKelvin" ), "millilitrePerHourKelvin", Set.of(new LangString( "millilitre per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(h\u00B7K)"), Optional.of("G74"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerKelvin" ), "millilitrePerKelvin", Set.of(new LangString( "millilitre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/K"), Optional.of("G30"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "millilitrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerKilogram" ), "millilitrePerKilogram", Set.of(new LangString( "millilitre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/kg"), Optional.of("KX"), Optional.of("cubicMetrePerKilogram"), Optional.of("10\u207B\u2076 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerLitre" ), "millilitrePerLitre", Set.of(new LangString( "millilitre per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/l"), Optional.of("L19"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "millilitrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerMinute" ), "millilitrePerMinute", Set.of(new LangString( "millilitre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/min"), Optional.of("41"), Optional.of("cubicMetrePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerMinuteBar" ), "millilitrePerMinuteBar", Set.of(new LangString( "millilitre per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(min\u00B7bar)"), Optional.of("G92"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B3 \u00D7 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerMinuteKelvin" ), "millilitrePerMinuteKelvin", Set.of(new LangString( "millilitre per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(min\u00B7K)"), Optional.of("G75"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerSecond" ), "millilitrePerSecond", Set.of(new LangString( "millilitre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/s"), Optional.of("40"), Optional.of("cubicMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerSecondBar" ), "millilitrePerSecondBar", Set.of(new LangString( "millilitre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(s\u00B7bar)"), Optional.of("G93"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerSecondKelvin" ), "millilitrePerSecondKelvin", Set.of(new LangString( "millilitre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(s\u00B7K)"), Optional.of("G76"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSquareCentimetreMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerSquareCentimetreMinute" ), "millilitrePerSquareCentimetreMinute", Set.of(new LangString( "millilitre per square centimetre minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(ml/min)/cm\u00B2"), Optional.of("M22"), Optional.of("cubicMetrePerSecondSquareMetre"), Optional.of("2.777778 \u00D7 10\u207B\u2076 (m\u00B3/s)/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "millilitrePerSquareCentimetreSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millilitrePerSquareCentimetreSecond" ), "millilitrePerSquareCentimetreSecond", Set.of(new LangString( "millilitre per square centimetre second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ml/(cm\u00B2\u00B7s)"), Optional.of("35"), Optional.of("metrePerSecond"), Optional.of("10\u207B\u00B2 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.POROSITY)) ) );
UNITS_BY_NAME.put( "millimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetre" ), "millimetre", Set.of(new LangString( "millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm"), Optional.of("MMT"), Optional.of("metre"), Optional.of("10\u207B\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "millimetrePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerBar" ), "millimetrePerBar", Set.of(new LangString( "millimetre per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/bar"), Optional.of("G06"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "millimetrePerDegreeCelsiusMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerDegreeCelsiusMetre" ), "millimetrePerDegreeCelsiusMetre", Set.of(new LangString( "millimetre per degree Celsius metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/(\u00B0C\u00B7m)"), Optional.of("E97"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u00B3 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "millimetrePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerHour" ), "millimetrePerHour", Set.of(new LangString( "millimetre per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/h"), Optional.of("H67"), Optional.empty(), Optional.of("0.277777778 \u00D7 10\u207B\u2077 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetrePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerKelvin" ), "millimetrePerKelvin", Set.of(new LangString( "millimetre per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/K"), Optional.of("F53"), Optional.empty(), Optional.of("10\u207B\u00B3 m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "millimetrePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerMinute" ), "millimetrePerMinute", Set.of(new LangString( "millimetre per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/min"), Optional.of("H81"), Optional.empty(), Optional.of("1.666666667 \u00D7 10\u207B\u2075 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerSecond" ), "millimetrePerSecond", Set.of(new LangString( "millimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/s"), Optional.of("C16"), Optional.of("metrePerSecond"), Optional.of("10\u207B\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetrePerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerSecondSquared" ), "millimetrePerSecondSquared", Set.of(new LangString( "millimetre per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/s\u00B2"), Optional.of("M41"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u00B3 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY)) ) );
UNITS_BY_NAME.put( "millimetrePerYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetrePerYear" ), "millimetrePerYear", Set.of(new LangString( "millimetre per year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm/y"), Optional.of("H66"), Optional.empty(), Optional.of("3.15576 \u00D7 10\u2074 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetreSquaredPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetreSquaredPerSecond" ), "millimetreSquaredPerSecond", Set.of(new LangString( "millimetre squared per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm\u00B2/s"), Optional.of("C17"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "millimetreToTheFourthPower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimetreToTheFourthPower" ), "millimetreToTheFourthPower", Set.of(new LangString( "millimetre to the fourth power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm\u2074"), Optional.of("G77"), Optional.of("metreToTheFourthPower"), Optional.of("10\u207B\u00B9\u00B2 m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_MOMENT_OF_AREA, QuantityKinds.SECOND_AXIAL_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "millimole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimole" ), "millimole", Set.of(new LangString( "millimole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mmol"), Optional.of("C18"), Optional.of("mole"), Optional.of("10\u207B\u00B3 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "millimolePerGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimolePerGram" ), "millimolePerGram", Set.of(new LangString( "millimole per gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mmol/g"), Optional.of("H68"), Optional.empty(), Optional.of("mol \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "millimolePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimolePerKilogram" ), "millimolePerKilogram", Set.of(new LangString( "millimole per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mmol/kg"), Optional.of("D87"), Optional.of("molePerKilogram"), Optional.of("10\u207B\u00B3 mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "millimolePerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millimolePerLitre" ), "millimolePerLitre", Set.of(new LangString( "millimole per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mmol/l"), Optional.of("M33"), Optional.of("molePerCubicMetre"), Optional.of("mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "millinewton", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millinewton" ), "millinewton", Set.of(new LangString( "millinewton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mN"), Optional.of("C20"), Optional.of("newton"), Optional.of("10\u207B\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "millinewtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millinewtonMetre" ), "millinewtonMetre", Set.of(new LangString( "millinewton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mN\u00B7m"), Optional.of("D83"), Optional.of("newtonMetre"), Optional.of("10\u207B\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "millinewtonPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millinewtonPerMetre" ), "millinewtonPerMetre", Set.of(new LangString( "millinewton per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mN/m"), Optional.of("C22"), Optional.of("newtonPerMetre"), Optional.of("10\u207B\u00B3 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "milliohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliohm" ), "milliohm", Set.of(new LangString( "milliohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u2126"), Optional.of("E45"), Optional.of("ohm"), Optional.of("10\u207B\u00B3 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.REACTANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.IMPEDANCE, QuantityKinds.COMPLEX_IMPEDANCES)) ) );
UNITS_BY_NAME.put( "milliohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliohmMetre" ), "milliohmMetre", Set.of(new LangString( "milliohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u2126\u00B7m"), Optional.of("C23"), Optional.of("ohmMetre"), Optional.of("10\u207B\u00B3 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "milliohmPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliohmPerMetre" ), "milliohmPerMetre", Set.of(new LangString( "milliohm per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u2126/m"), Optional.of("F54"), Optional.of("ohmPerMetre"), Optional.of("10\u207B\u00B3 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "million", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#million" ), "million", Set.of(new LangString( "million", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MIO"), Optional.empty(), Optional.of("10\u2076"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionBtuItPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millionBtuItPerHour" ), "millionBtuItPerHour", Set.of(new LangString( "million Btu(IT) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("BtuIT/h"), Optional.of("E16"), Optional.of("watt"), Optional.of("293071.1 W"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionBtuPer1000CubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millionBtuPer1000CubicFoot" ), "millionBtuPer1000CubicFoot", Set.of(new LangString( "million Btu per 1000 cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MBTU/kft\u00B3"), Optional.of("M9"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millionCubicMetre" ), "millionCubicMetre", Set.of(new LangString( "million cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Mm\u00B3"), Optional.of("HMQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionInternationalUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millionInternationalUnit" ), "millionInternationalUnit", Set.of(new LangString( "million international unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MIU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millipascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millipascal" ), "millipascal", Set.of(new LangString( "millipascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mPa"), Optional.of("74"), Optional.of("pascal"), Optional.of("10\u207B\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "millipascalPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millipascalPerMetre" ), "millipascalPerMetre", Set.of(new LangString( "millipascal per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mPa/m"), Optional.of("P80"), Optional.empty(), Optional.of("10\u207B\u00B3 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "millipascalSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millipascalSecond" ), "millipascalSecond", Set.of(new LangString( "millipascal second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mPa\u00B7s"), Optional.of("C24"), Optional.of("pascalSecond"), Optional.of("10\u207B\u00B3 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "millipascalSecondPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millipascalSecondPerBar" ), "millipascalSecondPerBar", Set.of(new LangString( "millipascal second per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mPa\u00B7s/bar"), Optional.of("L16"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2078 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "millipascalSecondPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millipascalSecondPerKelvin" ), "millipascalSecondPerKelvin", Set.of(new LangString( "millipascal second per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mPa\u00B7s/K"), Optional.of("L15"), Optional.empty(), Optional.of("10\u207B\u00B3 Pa \u00D7 s/K"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "milliradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliradian" ), "milliradian", Set.of(new LangString( "milliradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mrad"), Optional.of("C25"), Optional.of("radian"), Optional.of("10\u207B\u00B3 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "milliroentgen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliroentgen" ), "milliroentgen", Set.of(new LangString( "milliroentgen", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mR"), Optional.of("2Y"), Optional.of("coulombPerKilogram"), Optional.of("2.58 \u00D7 10\u207B\u2077 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "milliroentgenAequivalentMen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliroentgenAequivalentMen" ), "milliroentgenAequivalentMen", Set.of(new LangString( "milliroentgen aequivalent men", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mrem"), Optional.of("L31"), Optional.of("sievert"), Optional.of("10\u207B\u2075 Sv"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "millisecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisecond" ), "millisecond", Set.of(new LangString( "millisecond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ms"), Optional.of("C26"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "millisiemens", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisiemens" ), "millisiemens", Set.of(new LangString( "millisiemens", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mS"), Optional.of("C27"), Optional.of("siemens"), Optional.of("10\u207B\u00B3 S"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE)) ) );
UNITS_BY_NAME.put( "millisiemensPerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisiemensPerCentimetre" ), "millisiemensPerCentimetre", Set.of(new LangString( "millisiemens per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mS/cm"), Optional.of("H61"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u00B9 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "millisievert", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisievert" ), "millisievert", Set.of(new LangString( "millisievert", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mSv"), Optional.of("C28"), Optional.of("sievert"), Optional.of("10\u207B\u00B3 Sv"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "millisievertPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisievertPerHour" ), "millisievertPerHour", Set.of(new LangString( "millisievert per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mSv/h"), Optional.of("P71"), Optional.of("sievertPerSecond"), Optional.of("0.277777778 \u00D7 10\u207B\u2077 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "millisievertPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisievertPerMinute" ), "millisievertPerMinute", Set.of(new LangString( "millisievert per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mSv/min"), Optional.of("P75"), Optional.of("sievertPerSecond"), Optional.of("1.666666667 \u00D7 10\u207B\u2075 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "millisievertPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millisievertPerSecond" ), "millisievertPerSecond", Set.of(new LangString( "millisievert per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mSv/s"), Optional.of("P66"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u00B3 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "millitesla", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millitesla" ), "millitesla", Set.of(new LangString( "millitesla", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mT"), Optional.of("C29"), Optional.of("tesla"), Optional.of("10\u207B\u00B3 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "millivolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millivolt" ), "millivolt", Set.of(new LangString( "millivolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mV"), Optional.of("2Z"), Optional.of("voltDc"), Optional.of("10\u207B\u00B3 V"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.TENSION, QuantityKinds.VOLTAGE, QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.ELECTROMOTIVE_FORCE)) ) );
UNITS_BY_NAME.put( "millivoltAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millivoltAmpere" ), "millivoltAmpere", Set.of(new LangString( "millivolt - ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mV\u00B7A"), Optional.of("M35"), Optional.of("voltAmpere"), Optional.of("10\u207B\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "millivoltPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millivoltPerKelvin" ), "millivoltPerKelvin", Set.of(new LangString( "millivolt per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mV/K"), Optional.of("D49"), Optional.of("voltPerKelvin"), Optional.of("10\u207B\u00B3 V/K"), new HashSet<>(Arrays.asList(QuantityKinds.THOMPSON_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "millivoltPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millivoltPerMetre" ), "millivoltPerMetre", Set.of(new LangString( "millivolt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mV/m"), Optional.of("C30"), Optional.of("voltPerMetre"), Optional.of("10\u207B\u00B3 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "millivoltPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#millivoltPerMinute" ), "millivoltPerMinute", Set.of(new LangString( "millivolt per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mV/min"), Optional.of("H62"), Optional.empty(), Optional.of("1.666666667 \u00D7 10\u207B\u2075 m\u00B2 \u00D7 kg \u00D7 s\u207B\u2074 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "milliwatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliwatt" ), "milliwatt", Set.of(new LangString( "milliwatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mW"), Optional.of("C31"), Optional.of("watt"), Optional.of("10\u207B\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER, QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "milliwattPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliwattPerSquareMetre" ), "milliwattPerSquareMetre", Set.of(new LangString( "milliwatt per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mW/m\u00B2"), Optional.of("C32"), Optional.of("wattPerSquareMetre"), Optional.of("10\u207B\u00B3 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY)) ) );
UNITS_BY_NAME.put( "milliweber", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#milliweber" ), "milliweber", Set.of(new LangString( "milliweber", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mWb"), Optional.of("C33"), Optional.of("weber"), Optional.of("10\u207B\u00B3 Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX)) ) );
UNITS_BY_NAME.put( "minuteUnitOfAngle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#minuteUnitOfAngle" ), "minuteUnitOfAngle", Set.of(new LangString( "minute [unit of angle]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("'"), Optional.of("D61"), Optional.of("radian"), Optional.of("2.908882 \u00D7 10\u207B\u2074 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "minuteUnitOfTime", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#minuteUnitOfTime" ), "minuteUnitOfTime", Set.of(new LangString( "minute [unit of time]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("min"), Optional.of("MIN"), Optional.of("secondUnitOfTime"), Optional.of("60 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "mmscfPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mmscfPerDay" ), "mmscfPerDay", Set.of(new LangString( "MMSCF/day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("5E"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "moduleWidth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#moduleWidth" ), "moduleWidth", Set.of(new LangString( "module width", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("MW"), Optional.of("H77"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "molPerCubicMetrePascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molPerCubicMetrePascal" ), "molPerCubicMetrePascal", Set.of(new LangString( "mol per cubic metre pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/m\u00B3)/Pa"), Optional.of("P52"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "molPerKilogramPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molPerKilogramPascal" ), "molPerKilogramPascal", Set.of(new LangString( "mol per kilogram pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/kg)/Pa"), Optional.of("P51"), Optional.empty(), Optional.of("m \u00D7 kg\u207B\u00B2 \u00D7 s\u00B2 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "mole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mole" ), "mole", Set.of(new LangString( "mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol"), Optional.of("C34"), Optional.empty(), Optional.of("mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "molePerCubicDecimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerCubicDecimetre" ), "molePerCubicDecimetre", Set.of(new LangString( "mole per cubic decimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/dm\u00B3"), Optional.of("C35"), Optional.of("molePerCubicMetre"), Optional.of("10\u00B3 mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "molePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerCubicMetre" ), "molePerCubicMetre", Set.of(new LangString( "mole per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/m\u00B3"), Optional.of("C36"), Optional.empty(), Optional.of("mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerCubicMetreBar" ), "molePerCubicMetreBar", Set.of(new LangString( "mole per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/m\u00B3)/bar"), Optional.of("L29"), Optional.of("molPerCubicMetrePascal"), Optional.of("10\u207B\u2075 (mol/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerCubicMetreKelvin" ), "molePerCubicMetreKelvin", Set.of(new LangString( "mole per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/m\u00B3)/K"), Optional.of("L28"), Optional.empty(), Optional.of("(mol/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerCubicMetreToThePowerSumOfStoichiometricNumbers", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerCubicMetreToThePowerSumOfStoichiometricNumbers" ), "molePerCubicMetreToThePowerSumOfStoichiometricNumbers", Set.of(new LangString( "mole per cubic metre to the power sum of stoichiometric numbers", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/m\u00B3)\u2211\u03BDB"), Optional.of("P99"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "molePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerHour" ), "molePerHour", Set.of(new LangString( "mole per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/h"), Optional.of("L23"), Optional.of("molePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
UNITS_BY_NAME.put( "molePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerKilogram" ), "molePerKilogram", Set.of(new LangString( "mole per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/kg"), Optional.of("C19"), Optional.empty(), Optional.of("mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MOLALITY_OF_SOLUTE_B)) ) );
UNITS_BY_NAME.put( "molePerKilogramBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerKilogramBar" ), "molePerKilogramBar", Set.of(new LangString( "mole per kilogram bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/kg)/bar"), Optional.of("L25"), Optional.of("molPerKilogramPascal"), Optional.of("10\u207B\u2075 (mol/kg)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "molePerKilogramKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerKilogramKelvin" ), "molePerKilogramKelvin", Set.of(new LangString( "mole per kilogram kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/kg)/K"), Optional.of("L24"), Optional.empty(), Optional.of("(mol/kg)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "molePerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerLitre" ), "molePerLitre", Set.of(new LangString( "mole per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/l"), Optional.of("C38"), Optional.of("molePerCubicMetre"), Optional.of("10\u00B3 mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerLitreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerLitreBar" ), "molePerLitreBar", Set.of(new LangString( "mole per litre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/l)/bar"), Optional.of("L27"), Optional.of("molPerCubicMetrePascal"), Optional.of("10\u207B\u00B2 (mol/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerLitreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerLitreKelvin" ), "molePerLitreKelvin", Set.of(new LangString( "mole per litre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(mol/l)/K"), Optional.of("L26"), Optional.of("molePerCubicMetreKelvin"), Optional.of("10\u00B3 (mol/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "molePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerMinute" ), "molePerMinute", Set.of(new LangString( "mole per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/min"), Optional.of("L30"), Optional.of("molePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
}

private static void init11() {
UNITS_BY_NAME.put( "molePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#molePerSecond" ), "molePerSecond", Set.of(new LangString( "mole per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol/s"), Optional.of("E95"), Optional.empty(), Optional.of("s\u207B\u00B9 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "monetaryValue", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#monetaryValue" ), "monetaryValue", Set.of(new LangString( "monetary value", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("M4"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "month", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#month" ), "month", Set.of(new LangString( "month", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mo"), Optional.of("MON"), Optional.of("secondUnitOfTime"), Optional.of("2.629800 \u00D7 10\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "mutuallyDefined", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#mutuallyDefined" ), "mutuallyDefined", Set.of(new LangString( "mutually defined", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ZZ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "nanoampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanoampere" ), "nanoampere", Set.of(new LangString( "nanoampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nA"), Optional.of("C39"), Optional.of("ampere"), Optional.of("10\u207B\u2079 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "nanocoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanocoulomb" ), "nanocoulomb", Set.of(new LangString( "nanocoulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nC"), Optional.of("C40"), Optional.of("coulomb"), Optional.of("10\u207B\u2079 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "nanofarad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanofarad" ), "nanofarad", Set.of(new LangString( "nanofarad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nF"), Optional.of("C41"), Optional.of("farad"), Optional.of("10\u207B\u2079 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "nanofaradPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanofaradPerMetre" ), "nanofaradPerMetre", Set.of(new LangString( "nanofarad per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nF/m"), Optional.of("C42"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u2079 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY, QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.ELECTRIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "nanogramPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanogramPerKilogram" ), "nanogramPerKilogram", Set.of(new LangString( "nanogram per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ng/kg"), Optional.of("L32"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "nanograyPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanograyPerHour" ), "nanograyPerHour", Set.of(new LangString( "nanogray per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nGy/h"), Optional.of("P64"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B3 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "nanograyPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanograyPerMinute" ), "nanograyPerMinute", Set.of(new LangString( "nanogray per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nGy/min"), Optional.of("P60"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B9 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "nanograyPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanograyPerSecond" ), "nanograyPerSecond", Set.of(new LangString( "nanogray per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nGy/s"), Optional.of("P56"), Optional.of("grayPerSecond"), Optional.of("10\u207B\u2079 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "nanohenry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanohenry" ), "nanohenry", Set.of(new LangString( "nanohenry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nH"), Optional.of("C43"), Optional.of("henry"), Optional.of("10\u207B\u2079 H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "nanohenryPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanohenryPerMetre" ), "nanohenryPerMetre", Set.of(new LangString( "nanohenry per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nH/m"), Optional.of("C44"), Optional.of("henryPerMetre"), Optional.of("10\u207B\u2079 H/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_CONSTANT, QuantityKinds.PERMEABILITY, QuantityKinds.PERMEABILITY_OF_VACUUM)) ) );
UNITS_BY_NAME.put( "nanometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanometre" ), "nanometre", Set.of(new LangString( "nanometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nm"), Optional.of("C45"), Optional.of("metre"), Optional.of("10\u207B\u2079 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "nanoohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanoohm" ), "nanoohm", Set.of(new LangString( "nanoohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("n\u2126"), Optional.of("P22"), Optional.of("ohm"), Optional.of("10\u207B\u2079 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.REACTANCE, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT)) ) );
UNITS_BY_NAME.put( "nanoohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanoohmMetre" ), "nanoohmMetre", Set.of(new LangString( "nanoohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("n\u2126\u00B7m"), Optional.of("C46"), Optional.empty(), Optional.of("10\u207B\u2079 \u2126\u00B7\u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "nanosecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanosecond" ), "nanosecond", Set.of(new LangString( "nanosecond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ns"), Optional.of("C47"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2079 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "nanosiemensPerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanosiemensPerCentimetre" ), "nanosiemensPerCentimetre", Set.of(new LangString( "nanosiemens per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nS/cm"), Optional.of("G44"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2077 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "nanosiemensPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanosiemensPerMetre" ), "nanosiemensPerMetre", Set.of(new LangString( "nanosiemens per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nS/m"), Optional.of("G45"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2079 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "nanosievertPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanosievertPerHour" ), "nanosievertPerHour", Set.of(new LangString( "nanosievert per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nSv/h"), Optional.of("P73"), Optional.of("sievertPerSecond"), Optional.of("0.277777778 \u00D7 10\u207B\u00B9\u00B3 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "nanosievertPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanosievertPerMinute" ), "nanosievertPerMinute", Set.of(new LangString( "nanosievert per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nSv/min"), Optional.of("P77"), Optional.of("sievertPerSecond"), Optional.of("1.666666667 \u00D7 10\u207B\u00B9\u00B9 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "nanosievertPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanosievertPerSecond" ), "nanosievertPerSecond", Set.of(new LangString( "nanosievert per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nSv/s"), Optional.of("P68"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u2079 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "nanotesla", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanotesla" ), "nanotesla", Set.of(new LangString( "nanotesla", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nT"), Optional.of("C48"), Optional.of("tesla"), Optional.of("10\u207B\u2079 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "nanowatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nanowatt" ), "nanowatt", Set.of(new LangString( "nanowatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nW"), Optional.of("C49"), Optional.of("watt"), Optional.of("10\u207B\u2079 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "naturalUnitOfInformation", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#naturalUnitOfInformation" ), "naturalUnitOfInformation", Set.of(new LangString( "natural unit of information", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nat"), Optional.of("Q16"), Optional.empty(), Optional.of("nat"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "naturalUnitOfInformationPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#naturalUnitOfInformationPerSecond" ), "naturalUnitOfInformationPerSecond", Set.of(new LangString( "natural unit of information per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("nat/s"), Optional.of("Q19"), Optional.empty(), Optional.of("nat/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "nauticalMile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nauticalMile" ), "nauticalMile", Set.of(new LangString( "nautical mile", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("n mile"), Optional.of("NMI"), Optional.of("metre"), Optional.of("1852 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "neper", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#neper" ), "neper", Set.of(new LangString( "neper", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Np"), Optional.of("C50"), Optional.empty(), Optional.of("Np"), new HashSet<>(Arrays.asList(QuantityKinds.LOGARITHMIC_DECREMENT, QuantityKinds.LEVEL_OF_A_POWER_QUANTITY, QuantityKinds.LEVEL_OF_A_FIELD_QUANTITY)) ) );
UNITS_BY_NAME.put( "neperPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#neperPerSecond" ), "neperPerSecond", Set.of(new LangString( "neper per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Np/s"), Optional.of("C51"), Optional.empty(), Optional.of("Np/s"), new HashSet<>(Arrays.asList(QuantityKinds.DAMPING_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "netKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#netKilogram" ), "netKilogram", Set.of(new LangString( "net kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("58"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "netTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#netTon" ), "netTon", Set.of(new LangString( "net ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "newton", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newton" ), "newton", Set.of(new LangString( "newton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N"), Optional.of("NEW"), Optional.empty(), Optional.of("(kg \u00D7 m)/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "newtonCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonCentimetre" ), "newtonCentimetre", Set.of(new LangString( "newton centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7cm"), Optional.of("F88"), Optional.empty(), Optional.of("10\u207B\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetre" ), "newtonMetre", Set.of(new LangString( "newton metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m"), Optional.of("NU"), Optional.empty(), Optional.of("N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetrePerAmpere" ), "newtonMetrePerAmpere", Set.of(new LangString( "newton metre per ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m/A"), Optional.of("F90"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerDegree", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetrePerDegree" ), "newtonMetrePerDegree", Set.of(new LangString( "newton metre per degree", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Nm/\u00B0"), Optional.of("F89"), Optional.empty(), Optional.of("57.295788 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 rad\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetrePerKilogram" ), "newtonMetrePerKilogram", Set.of(new LangString( "newton metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m/kg"), Optional.of("G19"), Optional.empty(), Optional.of("m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetrePerMetre" ), "newtonMetrePerMetre", Set.of(new LangString( "newton metre per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m/m\u00B2"), Optional.of("Q27"), Optional.empty(), Optional.of("m \u00D7 kg \u00D7 s\u207B\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "newtonMetrePerRadian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetrePerRadian" ), "newtonMetrePerRadian", Set.of(new LangString( "newton metre per radian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m/rad"), Optional.of("M93"), Optional.empty(), Optional.of("m\u00B2 \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 rad\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetrePerSquareMetre" ), "newtonMetrePerSquareMetre", Set.of(new LangString( "newton metre per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m/m\u00B2"), Optional.of("M34"), Optional.empty(), Optional.of("N \u00D7 m/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORSIONAL_STIFFNESS, QuantityKinds.AREA_RELATED_TORSIONAL_MOMENT)) ) );
UNITS_BY_NAME.put( "newtonMetreSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetreSecond" ), "newtonMetreSecond", Set.of(new LangString( "newton metre second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m\u00B7s"), Optional.of("C53"), Optional.empty(), Optional.of("N \u00D7 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_IMPULSE)) ) );
UNITS_BY_NAME.put( "newtonMetreSquaredPerKilogramSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetreSquaredPerKilogramSquared" ), "newtonMetreSquaredPerKilogramSquared", Set.of(new LangString( "newton metre squared per kilogram squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m\u00B2/kg\u00B2"), Optional.of("C54"), Optional.empty(), Optional.of("N \u00D7 m\u00B2/kg\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.GRAVITATIONAL_CONSTANT)) ) );
UNITS_BY_NAME.put( "newtonMetreWattToThePowerMinus0point5", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonMetreWattToThePowerMinus0point5" ), "newtonMetreWattToThePowerMinus0point5", Set.of(new LangString( "newton metre watt to the power minus 0,5", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m\u00B7W\u207B\u2070\u2027\u2075"), Optional.of("H41"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 W\u207B\u2070\u2027\u2075"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonPerAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerAmpere" ), "newtonPerAmpere", Set.of(new LangString( "newton per ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/A"), Optional.of("H40"), Optional.empty(), Optional.of("kg \u00D7 m \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonPerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerCentimetre" ), "newtonPerCentimetre", Set.of(new LangString( "newton per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/cm"), Optional.of("M23"), Optional.of("newtonPerMetre"), Optional.of("10\u00B2 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "newtonPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerMetre" ), "newtonPerMetre", Set.of(new LangString( "newton per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/m"), Optional.of("4P"), Optional.empty(), Optional.of("N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "newtonPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerMillimetre" ), "newtonPerMillimetre", Set.of(new LangString( "newton per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/mm"), Optional.of("F47"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerSquareCentimetre" ), "newtonPerSquareCentimetre", Set.of(new LangString( "newton per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/cm\u00B2"), Optional.of("E01"), Optional.of("pascal"), Optional.of("10\u2074 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "newtonPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerSquareMetre" ), "newtonPerSquareMetre", Set.of(new LangString( "newton per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/m\u00B2"), Optional.of("C55"), Optional.of("pascal"), Optional.of("Pa"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "newtonPerSquareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonPerSquareMillimetre" ), "newtonPerSquareMillimetre", Set.of(new LangString( "newton per square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N/mm\u00B2"), Optional.of("C56"), Optional.of("pascal"), Optional.of("10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "newtonSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonSecond" ), "newtonSecond", Set.of(new LangString( "newton second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7s"), Optional.of("C57"), Optional.empty(), Optional.of("N \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.IMPULSE)) ) );
UNITS_BY_NAME.put( "newtonSecondPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonSecondPerMetre" ), "newtonSecondPerMetre", Set.of(new LangString( "newton second per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7s/m"), Optional.of("C58"), Optional.empty(), Optional.of("N \u00D7 s/m"), new HashSet<>(Arrays.asList(QuantityKinds.MECHANICAL_IMPEDANCE)) ) );
UNITS_BY_NAME.put( "newtonSecondPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonSecondPerSquareMetre" ), "newtonSecondPerSquareMetre", Set.of(new LangString( "newton second per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(N/m\u00B2)\u00B7s"), Optional.of("N36"), Optional.of("pascalSecond"), Optional.of("Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "newtonSquareMetrePerAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#newtonSquareMetrePerAmpere" ), "newtonSquareMetrePerAmpere", Set.of(new LangString( "newton square metre per ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("N\u00B7m\u00B2/A"), Optional.of("P49"), Optional.empty(), Optional.of("m\u00B3 \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_DIPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "nil", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#nil" ), "nil", Set.of(new LangString( "nil", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("()"), Optional.of("NIL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfArticles", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfArticles" ), "numberOfArticles", Set.of(new LangString( "number of articles", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NAR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfCells", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfCells" ), "numberOfCells", Set.of(new LangString( "number of cells", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NCL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfInternationalUnits", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfInternationalUnits" ), "numberOfInternationalUnits", Set.of(new LangString( "number of international units", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NIU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfJewels", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfJewels" ), "numberOfJewels", Set.of(new LangString( "number of jewels", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("JWL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfPacks", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfPacks" ), "numberOfPacks", Set.of(new LangString( "number of packs", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NMP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfParts", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfParts" ), "numberOfParts", Set.of(new LangString( "number of parts", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("NPT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfWords", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#numberOfWords" ), "numberOfWords", Set.of(new LangString( "number of words", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("D68"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "octave", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#octave" ), "octave", Set.of(new LangString( "octave", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("C59"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY_INTERVAL)) ) );
UNITS_BY_NAME.put( "octet", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#octet" ), "octet", Set.of(new LangString( "octet", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("o"), Optional.of("Q12"), Optional.of("bit"), Optional.of("8 bit"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "octetPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#octetPerSecond" ), "octetPerSecond", Set.of(new LangString( "octet per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("o/s"), Optional.of("Q13"), Optional.of("bitPerSecond"), Optional.of("8 bit/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohm" ), "ohm", Set.of(new LangString( "ohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126"), Optional.of("OHM"), Optional.empty(), Optional.of("\u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.REACTANCE, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT)) ) );
UNITS_BY_NAME.put( "ohmCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmCentimetre" ), "ohmCentimetre", Set.of(new LangString( "ohm centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126\u00B7cm"), Optional.of("C60"), Optional.of("ohmMetre"), Optional.of("10\u207B\u00B2 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmCircularMilPerFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmCircularMilPerFoot" ), "ohmCircularMilPerFoot", Set.of(new LangString( "ohm circular-mil per foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126\u00B7cmil/ft"), Optional.of("P23"), Optional.of("ohmMetre"), Optional.of("1.662426 \u00D7 10\u207B\u2079 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmKilometre" ), "ohmKilometre", Set.of(new LangString( "ohm kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126\u00B7km"), Optional.of("M24"), Optional.of("ohmMetre"), Optional.of("10\u00B3 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmMetre" ), "ohmMetre", Set.of(new LangString( "ohm metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126\u00B7m"), Optional.of("C61"), Optional.empty(), Optional.of("\u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY, QuantityKinds.RESIDUAL_RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmPerKilometre" ), "ohmPerKilometre", Set.of(new LangString( "ohm per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126/km"), Optional.of("F56"), Optional.of("ohmPerMetre"), Optional.of("10\u207B\u00B3 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "ohmPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmPerMetre" ), "ohmPerMetre", Set.of(new LangString( "ohm per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126/m"), Optional.of("H26"), Optional.empty(), Optional.of("\u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "ohmPerMileStatuteMile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ohmPerMileStatuteMile" ), "ohmPerMileStatuteMile", Set.of(new LangString( "ohm per mile (statute mile)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2126/mi"), Optional.of("F55"), Optional.of("ohmPerMetre"), Optional.of("6.21371 \u00D7 10\u207B\u2074  \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "one", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#one" ), "one", Set.of(new LangString( "one", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1"), Optional.of("C62"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_SUSCEPTIBILITY, QuantityKinds.REST_MASS_OF_PROTON, QuantityKinds.REFRACTIVE_INDEX, QuantityKinds.GRAND_PARTITION_FUNCTION, QuantityKinds.PACKING_FRACTION, QuantityKinds.POISSON_NUMBER, QuantityKinds.RELATIVE_PERMITTIVITY, QuantityKinds.SHEAR_STRAIN, QuantityKinds.SHORT_RANGE_ORDER_PARAMETER, QuantityKinds.SPECTRAL_TRANSMITTANCE, QuantityKinds.MICROCANONICAL_PARTITION_FUNCTION, QuantityKinds.ELECTRIC_SUSCEPTIBILITY, QuantityKinds.HYPERFINE_STRUCTURE_QUANTUM_NUMBER, QuantityKinds.ACTIVITY_COEFFICIENT_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.GRASHOF_NUMBER, QuantityKinds.NEUTRON_YIELD_PER_FISSION, QuantityKinds.CHARGE_NUMBER_OF_ION, QuantityKinds.PARTITION_FUNCTION_OF_A_MOLECULE, QuantityKinds.ATOMIC_NUMBER, QuantityKinds.RATIO_OF_THE_SPECIFIC_HEAT_CAPACITIES, QuantityKinds.STANTON_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.RATIO_OF_THE_MASSIC_HEAT_CAPACITY, QuantityKinds.EFFECTIVE_MULTIPLICATION_FACTOR, QuantityKinds.RELATIVE_MASS_DENSITY, QuantityKinds.CURRENT_FRACTION_OF_ION_B, QuantityKinds.FROUDE_NUMBER, QuantityKinds.MADELUNG_CONSTANT, QuantityKinds.MOLECULAR_PARTITION_FUNCTION, QuantityKinds.ORDER_OF_REFLEXION, QuantityKinds.MASS_NUMBER, QuantityKinds.PRANDTL_NUMBER, QuantityKinds.COUPLING_COEFFICIENT, QuantityKinds.REST_MASS_OF_ELECTRON, QuantityKinds.RESONANCE_ESCAPE_PROBABILITY, QuantityKinds.SPECTRAL_ABSORPTION_FACTOR, QuantityKinds.MAGNETIC_REYNOLDS_NUMBER, QuantityKinds.WEBER_NUMBER, QuantityKinds.POISSON_RATIO, QuantityKinds.DISSIPANCE, QuantityKinds.TOTAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER, QuantityKinds.EMISSIVITY_AT_A_SPECIFIED_WAVELENGTH, QuantityKinds.DIRECTIONAL_SPECTRAL_EMISSIVITY, QuantityKinds.RELATIVE_DENSITY, QuantityKinds.DISSIPATION_FACTOR, QuantityKinds.GRASHOF_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.TRANSMITTANCE, QuantityKinds.CANONICAL_PARTITION_FUNCTION, QuantityKinds.MAGNETIC_QUANTUM_NUMBER, QuantityKinds.EMISSIVITY, QuantityKinds.CIE_COLORIMETRIC_FUNCTIONS, QuantityKinds.REST_MASS_OF_NEUTRON, QuantityKinds.SPECTRAL_REFLECTIONFACTOR, QuantityKinds.LEWIS_NUMBER, QuantityKinds.NUCLEAR_SPIN_QUANTUM_NUMBER, QuantityKinds.NUMBER_OF_TURNS_IN_A_WINDING, QuantityKinds.G_FACTOR_OF_NUCLEUS, QuantityKinds.PRINCIPLE_QUANTUM_NUMBER, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_LIQUID_OR_A_SOLID_MIXTURE, QuantityKinds.STANTON_NUMBER, QuantityKinds.MULTIPLICATION_FACTOR, QuantityKinds.SCHMIDT_NUMBER, QuantityKinds.MACH_NUMBER, QuantityKinds.FRICTION_FACTOR, QuantityKinds.ALFVEN_NUMBER, QuantityKinds.LUMINOUS_EFFICIENCY_AT_A_SPECIFIED_WAVELENGTH, QuantityKinds.NUSSELT_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.LEAKAGE_COEFFICIENT, QuantityKinds.ORBITAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER, QuantityKinds.TRANSPORT_NUMBER_OF_ION_B, QuantityKinds.RELATIVE_PERMEABILITY, QuantityKinds.ACTIVITY_COEFFICIENT_OF_B_IN_A_LIQUID_AS_A_SOLID_MIXTURE, QuantityKinds.ABSOLUTE_ACTIVITY, QuantityKinds.PECLET_NUMBER, QuantityKinds.FOURIER_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.RELATIVE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.SPECTRAL_LUMINOUS_EFFICIENCY, QuantityKinds.SPECTRAL_EMISSIVITY, QuantityKinds.REFLECTION_FACTOR, QuantityKinds.COEFFICIENT_OF_FRICTION, QuantityKinds.THERMAL_DIFFUSION_FACTOR, QuantityKinds.DEBYE_WALLE_FACTOR, QuantityKinds.LINEAR_STRAIN, QuantityKinds.STANDARD_EQUILIBRIUM_CONSTANT, QuantityKinds.NUMBER_OF_PAIRS_OF_POLES, QuantityKinds.INFINITE_MEDIUM_MULTIPLICATION_FACTOR, QuantityKinds.GRUENEISEN_PARAMETER, QuantityKinds.RELATIVE_MASS_EXCESS, QuantityKinds.STROUHAL_NUMBER, QuantityKinds.TRANSMISSION_FACTOR, QuantityKinds.SPECTRAL_TRANSMISSION_FACTOR, QuantityKinds.PROTON_NUMBER, QuantityKinds.NUSSELT_NUMBER, QuantityKinds.OPTICAL_DENSITY, QuantityKinds.THERMAL_UTILIZATION_FACTOR, QuantityKinds.ACTIVITY_OF_SOLVENT_A, QuantityKinds.NON_LEAKAGE_PROBABILITY, QuantityKinds.MOLE_FRACTION_OF_B, QuantityKinds.COWLING_NUMBER, QuantityKinds.MOLE_RATIO_OF_SOLUTE_B, QuantityKinds.NUMBER_OF_MOLECULES_OR_OTHER_ELEMENTARY_ENTITIES, QuantityKinds.FINE_STRUCTURE_CONSTANT, QuantityKinds.REFLECTANCE, QuantityKinds.KNUDSEN_NUMBER, QuantityKinds.VOLUME_OR_BULK_STRAIN, QuantityKinds.LANDAU_GINZBURG_NUMBER, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.PECLET_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.MOBILITY_RATIO, QuantityKinds.NEUTRON_NUMBER, QuantityKinds.OSMOTIC_COEFFICIENT_OF_THE_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.EULER_NUMBER, QuantityKinds.REYNOLDS_NUMBER, QuantityKinds.SPECTRAL_ABSORPTANCE, QuantityKinds.RAYLEIGH_NUMBER, QuantityKinds.ISENTROPIC_EXPONENT, QuantityKinds.STOICHIOMETRIC_NUMBER_OF_B, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_GASEOUS_MIXTURE, QuantityKinds.BINDING_FRACTION, QuantityKinds.DEGREE_OF_DISSOCIATION, QuantityKinds.HARTMANN_NUMBER, QuantityKinds.THERMAL_DIFFUSION_RATIO, QuantityKinds.SPECTRAL_REFLECTANCE, QuantityKinds.REACTIVITY, QuantityKinds.RELATIVE_ELONGATION, QuantityKinds.FAST_FISSION_FACTOR, QuantityKinds.ABSORBANCE, QuantityKinds.RELATIVE_ATOMIC_MASS, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.FOURIER_NUMBER, QuantityKinds.LUMINOUS_EFFICIENCY, QuantityKinds.STATISTICAL_WEIGHT, QuantityKinds.SPIN_ANGULAR_MOMENTUM_QUANTUM_NUMBER, QuantityKinds.G_FACTOR_OF_ATOM_OR_ELECTRON, QuantityKinds.GRAND_CANONICAL_PARTITION_FUNCTION, QuantityKinds.NEUTRON_YIELD_PER_ABSORPTION, QuantityKinds.NUCLEON_NUMBER, QuantityKinds.LONG_RANGE_ORDER_PARAMETER, QuantityKinds.NUMBER_OF_PHASES, QuantityKinds.RELATIVE_MASS_DEFECT, QuantityKinds.SPECTRAL_RADIANCE_FACTOR, QuantityKinds.AVERAGE_LOGARITHMIC_ENERGY_DECREMENT, QuantityKinds.RELATIVE_MOLECULAR_MASS, QuantityKinds.ABSORPTION_FACTOR, QuantityKinds.INTERNAL_CONVERSION_FACTOR, QuantityKinds.COORDINATES_TRICHROMATIC, QuantityKinds.LETHARGY)) ) );
UNITS_BY_NAME.put( "onePerOne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#onePerOne" ), "onePerOne", Set.of(new LangString( "one per one", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/1"), Optional.of("Q26"), Optional.empty(), Optional.of("1/1"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ounceAvoirdupois", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupois" ), "ounceAvoirdupois", Set.of(new LangString( "ounce (avoirdupois)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz"), Optional.of("ONZ"), Optional.of("kilogram"), Optional.of("2.834952 \u00D7 10\u207B\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisForce", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisForce" ), "ounceAvoirdupoisForce", Set.of(new LangString( "ounce (avoirdupois)-force", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ozf"), Optional.of("L40"), Optional.of("newton"), Optional.of("0.2780139 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisForceInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisForceInch" ), "ounceAvoirdupoisForceInch", Set.of(new LangString( "ounce (avoirdupois)-force inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ozf\u00B7in"), Optional.of("L41"), Optional.of("newtonMetre"), Optional.of("7.061552 \u00D7 10\u207B\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerCubicInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerCubicInch" ), "ounceAvoirdupoisPerCubicInch", Set.of(new LangString( "ounce (avoirdupois) per cubic inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/in\u00B3"), Optional.of("L39"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.729994 \u00D7 10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerCubicYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerCubicYard" ), "ounceAvoirdupoisPerCubicYard", Set.of(new LangString( "ounce (avoirdupois) per cubic yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/yd\u00B3"), Optional.of("G32"), Optional.empty(), Optional.of("3.70798 \u00D7 10\u207B\u00B2 kg \u00D7 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerDay" ), "ounceAvoirdupoisPerDay", Set.of(new LangString( "ounce (avoirdupois) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/d"), Optional.of("L33"), Optional.of("kilogramPerSecond"), Optional.of("3.281194 \u00D7 10\u207B\u2077kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerGallonUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerGallonUk" ), "ounceAvoirdupoisPerGallonUk", Set.of(new LangString( "ounce (avoirdupois) per gallon (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/gal (UK)"), Optional.of("L37"), Optional.of("kilogramPerCubicMetre"), Optional.of("6.236023 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerGallonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerGallonUs" ), "ounceAvoirdupoisPerGallonUs", Set.of(new LangString( "ounce (avoirdupois) per gallon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/gal (US)"), Optional.of("L38"), Optional.of("kilogramPerCubicMetre"), Optional.of("7.489152 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerHour" ), "ounceAvoirdupoisPerHour", Set.of(new LangString( "ounce (avoirdupois) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/h"), Optional.of("L34"), Optional.of("kilogramPerSecond"), Optional.of("7.874867 \u00D7 10\u207B\u2076 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerMinute" ), "ounceAvoirdupoisPerMinute", Set.of(new LangString( "ounce (avoirdupois) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/min"), Optional.of("L35"), Optional.of("kilogramPerSecond"), Optional.of("4.72492 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerSecond" ), "ounceAvoirdupoisPerSecond", Set.of(new LangString( "ounce (avoirdupois) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/s"), Optional.of("L36"), Optional.of("kilogramPerSecond"), Optional.of("2.834952 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceAvoirdupoisPerSquareInch" ), "ounceAvoirdupoisPerSquareInch", Set.of(new LangString( "ounce (avoirdupois) per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/in\u00B2"), Optional.of("N22"), Optional.of("kilogramPerSquareMetre"), Optional.of("4.394185 \u00D7 10 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "ounceFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceFoot" ), "ounceFoot", Set.of(new LangString( "ounce foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz\u00B7ft"), Optional.of("4R"), Optional.of("kilogramMetre"), Optional.of("8.640934 \u00D7 10\u207B\u00B3 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "ounceInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceInch" ), "ounceInch", Set.of(new LangString( "ounce inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz\u00B7in"), Optional.of("4Q"), Optional.of("kilogramMetre"), Optional.of("7.200778 \u00D7 10\u207B\u2074 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "ouncePerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ouncePerSquareFoot" ), "ouncePerSquareFoot", Set.of(new LangString( "ounce per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/ft\u00B2"), Optional.of("37"), Optional.of("kilogramPerSquareMetre"), Optional.of("0.3051517 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "ouncePerSquareFootPer0point01inch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ouncePerSquareFootPer0point01inch" ), "ouncePerSquareFootPer0point01inch", Set.of(new LangString( "ounce per square foot per 0,01inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/(ft\u00B2/cin)"), Optional.of("38"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ouncePerSquareYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ouncePerSquareYard" ), "ouncePerSquareYard", Set.of(new LangString( "ounce per square yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("oz/yd\u00B2"), Optional.of("ON"), Optional.of("kilogramPerSquareMetre"), Optional.of("3.390575 \u00D7 10\u207B\u00B2 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUkFluidPerDay" ), "ounceUkFluidPerDay", Set.of(new LangString( "ounce (UK fluid) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (UK)/d"), Optional.of("J95"), Optional.of("cubicMetrePerSecond"), Optional.of("3.288549 \u00D7 10\u207B\u00B9\u2070 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUkFluidPerHour" ), "ounceUkFluidPerHour", Set.of(new LangString( "ounce (UK fluid) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (UK)/h"), Optional.of("J96"), Optional.of("cubicMetrePerSecond"), Optional.of("7.892517 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUkFluidPerMinute" ), "ounceUkFluidPerMinute", Set.of(new LangString( "ounce (UK fluid) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (UK)/min"), Optional.of("J97"), Optional.of("cubicMetrePerSecond"), Optional.of("4.73551 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUkFluidPerSecond" ), "ounceUkFluidPerSecond", Set.of(new LangString( "ounce (UK fluid) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (UK)/s"), Optional.of("J98"), Optional.of("cubicMetrePerSecond"), Optional.of("2.841306 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUsFluidPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUsFluidPerDay" ), "ounceUsFluidPerDay", Set.of(new LangString( "ounce (US fluid) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (US)/d"), Optional.of("J99"), Optional.of("cubicMetrePerSecond"), Optional.of("3.422862 \u00D7 10\u207B\u00B9\u2070 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUsFluidPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUsFluidPerHour" ), "ounceUsFluidPerHour", Set.of(new LangString( "ounce (US fluid) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (US)/h"), Optional.of("K10"), Optional.of("cubicMetrePerSecond"), Optional.of("8.214869 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUsFluidPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUsFluidPerMinute" ), "ounceUsFluidPerMinute", Set.of(new LangString( "ounce (US fluid) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (US)/min"), Optional.of("K11"), Optional.of("cubicMetrePerSecond"), Optional.of("4.928922 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
}

private static void init12() {
UNITS_BY_NAME.put( "ounceUsFluidPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ounceUsFluidPerSecond" ), "ounceUsFluidPerSecond", Set.of(new LangString( "ounce (US fluid) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("fl oz (US)/s"), Optional.of("K12"), Optional.of("cubicMetrePerSecond"), Optional.of("2.957353 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "outfit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#outfit" ), "outfit", Set.of(new LangString( "outfit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("11"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "overtimeHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#overtimeHour" ), "overtimeHour", Set.of(new LangString( "overtime hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("OT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ozoneDepletionEquivalent", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ozoneDepletionEquivalent" ), "ozoneDepletionEquivalent", Set.of(new LangString( "ozone depletion equivalent", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ODE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pad" ), "pad", Set.of(new LangString( "pad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("PD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "page", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#page" ), "page", Set.of(new LangString( "page", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ZP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pageFacsimile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pageFacsimile" ), "pageFacsimile", Set.of(new LangString( "page - facsimile", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("QA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pageHardcopy", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pageHardcopy" ), "pageHardcopy", Set.of(new LangString( "page - hardcopy", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("QB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pagePerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pagePerInch" ), "pagePerInch", Set.of(new LangString( "page per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ppi"), Optional.of("PQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pair", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pair" ), "pair", Set.of(new LangString( "pair", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("PR"), Optional.empty(), Optional.of("2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "panel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#panel" ), "panel", Set.of(new LangString( "panel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("OA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "parsec", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#parsec" ), "parsec", Set.of(new LangString( "parsec", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pc"), Optional.of("C63"), Optional.of("metre"), Optional.of("3.085678 \u00D7 10\u00B9\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "partPerBillionUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#partPerBillionUs" ), "partPerBillionUs", Set.of(new LangString( "part per billion (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ppb"), Optional.of("61"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u2079"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerHundredThousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#partPerHundredThousand" ), "partPerHundredThousand", Set.of(new LangString( "part per hundred thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ppht"), Optional.of("E40"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u2075"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerMillion", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#partPerMillion" ), "partPerMillion", Set.of(new LangString( "part per million", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ppm"), Optional.of("59"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u2076"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerQuadrillionUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#partPerQuadrillionUs" ), "partPerQuadrillionUs", Set.of(new LangString( "Part per quadrillion (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ppq"), Optional.empty(), Optional.empty(), Optional.of("1 \u00D7 10\u00AF\u00B9\u2075"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerThousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#partPerThousand" ), "partPerThousand", Set.of(new LangString( "part per thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2030"), Optional.of("NX"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerTrillionUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#partPerTrillionUs" ), "partPerTrillionUs", Set.of(new LangString( "part per trillion (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ppt"), Optional.empty(), Optional.empty(), Optional.of("1 \u00D7 10\u00AF\u00B9\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascal" ), "pascal", Set.of(new LangString( "pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa"), Optional.of("PAL"), Optional.empty(), Optional.of("Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.INSTANTANEOUS_SOUND_PRESSURE, QuantityKinds.STATIC_PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.FUGACITY_OF_B_IN_A_GASEOUS_MIXTURE, QuantityKinds.PARTIAL_PRESSURE_OF_B_IN_A_GASEOUS_MIXTURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.OSMOTIC_PRESSURE, QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "pascalCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalCubicMetrePerSecond" ), "pascalCubicMetrePerSecond", Set.of(new LangString( "pascal cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7m\u00B3/s"), Optional.of("G01"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "pascalLitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalLitrePerSecond" ), "pascalLitrePerSecond", Set.of(new LangString( "pascal litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7l/s"), Optional.of("F99"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "pascalPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalPerBar" ), "pascalPerBar", Set.of(new LangString( "pascal per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa/bar"), Optional.of("F07"), Optional.empty(), Optional.of("10\u207B\u2075"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "pascalPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalPerKelvin" ), "pascalPerKelvin", Set.of(new LangString( "pascal per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa/K"), Optional.of("C64"), Optional.empty(), Optional.of("Pa/K"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "pascalPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalPerMetre" ), "pascalPerMetre", Set.of(new LangString( "pascal per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa/m"), Optional.of("H42"), Optional.empty(), Optional.of("m\u207B\u00B2 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "pascalSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSecond" ), "pascalSecond", Set.of(new LangString( "pascal second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7s"), Optional.of("C65"), Optional.empty(), Optional.of("Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "pascalSecondPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSecondPerBar" ), "pascalSecondPerBar", Set.of(new LangString( "pascal second per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7s/bar"), Optional.of("H07"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2075 s"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "pascalSecondPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSecondPerCubicMetre" ), "pascalSecondPerCubicMetre", Set.of(new LangString( "pascal second per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7s/m\u00B3"), Optional.of("C66"), Optional.empty(), Optional.of("Pa \u00D7 s/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ACOUSTIC_IMPEDANCE)) ) );
UNITS_BY_NAME.put( "pascalSecondPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSecondPerKelvin" ), "pascalSecondPerKelvin", Set.of(new LangString( "pascal second per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa.s/K"), Optional.of("F77"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "pascalSecondPerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSecondPerLitre" ), "pascalSecondPerLitre", Set.of(new LangString( "pascal second per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7s/l"), Optional.of("M32"), Optional.of("pascalSecondPerCubicMetre"), Optional.of("10\u00B3 Pa \u00D7 s/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ACOUSTIC_IMPEDANCE)) ) );
UNITS_BY_NAME.put( "pascalSecondPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSecondPerMetre" ), "pascalSecondPerMetre", Set.of(new LangString( "pascal second per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B7 s/m"), Optional.of("C67"), Optional.empty(), Optional.of("Pa \u00D7 s/m"), new HashSet<>(Arrays.asList(QuantityKinds.CHARACTERISTIC_IMPEDANCE_OF_A_MEDIUM)) ) );
UNITS_BY_NAME.put( "pascalSquareMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSquareMetrePerKilogram" ), "pascalSquareMetrePerKilogram", Set.of(new LangString( "pascal square metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa/(kg/m\u00B2)"), Optional.of("P79"), Optional.of("metrePerSecondSquared"), Optional.of("m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.BURST_INDEX)) ) );
UNITS_BY_NAME.put( "pascalSquaredSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalSquaredSecond" ), "pascalSquaredSecond", Set.of(new LangString( "pascal squared second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u00B2\u00B7s"), Optional.of("P42"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_EXPOSURE)) ) );
UNITS_BY_NAME.put( "pascalToThePowerSumOfStoichiometricNumbers", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pascalToThePowerSumOfStoichiometricNumbers" ), "pascalToThePowerSumOfStoichiometricNumbers", Set.of(new LangString( "pascal to the power sum of stoichiometric numbers", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u03A3\u03BDB"), Optional.of("P98"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pebibitPerCubicMetre" ), "pebibitPerCubicMetre", Set.of(new LangString( "pebibit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pibit/m\u00B3"), Optional.of("E82"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pebibitPerMetre" ), "pebibitPerMetre", Set.of(new LangString( "pebibit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pibit/m"), Optional.of("E80"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pebibitPerSquareMetre" ), "pebibitPerSquareMetre", Set.of(new LangString( "pebibit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pibit/m\u00B2"), Optional.of("E81"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pebibyte" ), "pebibyte", Set.of(new LangString( "Pebibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("PiB"), Optional.of("E60"), Optional.of("byte"), Optional.of("2\u2075\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "peck", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peck" ), "peck", Set.of(new LangString( "peck", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (US)"), Optional.of("G23"), Optional.of("cubicMetre"), Optional.of("8.809768 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "peckUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUk" ), "peckUk", Set.of(new LangString( "peck (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (UK)"), Optional.of("L43"), Optional.of("cubicMetre"), Optional.of("9.092181 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "peckUkPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUkPerDay" ), "peckUkPerDay", Set.of(new LangString( "peck (UK) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (UK)/d"), Optional.of("L44"), Optional.of("cubicMetrePerSecond"), Optional.of("1.052336 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUkPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUkPerHour" ), "peckUkPerHour", Set.of(new LangString( "peck (UK) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (UK)/h"), Optional.of("L45"), Optional.of("cubicMetrePerSecond"), Optional.of("2.525606 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUkPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUkPerMinute" ), "peckUkPerMinute", Set.of(new LangString( "peck (UK) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (UK)/min"), Optional.of("L46"), Optional.of("cubicMetrePerSecond"), Optional.of("1.5153635 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUkPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUkPerSecond" ), "peckUkPerSecond", Set.of(new LangString( "peck (UK) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (UK)/s"), Optional.of("L47"), Optional.of("cubicMetrePerSecond"), Optional.of("9.092181 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUsDryPerDay" ), "peckUsDryPerDay", Set.of(new LangString( "peck (US dry) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (US dry)/d"), Optional.of("L48"), Optional.of("cubicMetrePerSecond"), Optional.of("1.019649 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUsDryPerHour" ), "peckUsDryPerHour", Set.of(new LangString( "peck (US dry) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (US dry)/h"), Optional.of("L49"), Optional.of("cubicMetrePerSecond"), Optional.of("2.447158 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUsDryPerMinute" ), "peckUsDryPerMinute", Set.of(new LangString( "peck (US dry) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (US dry)/min"), Optional.of("L50"), Optional.of("cubicMetrePerSecond"), Optional.of("1.468295 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#peckUsDryPerSecond" ), "peckUsDryPerSecond", Set.of(new LangString( "peck (US dry) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pk (US dry)/s"), Optional.of("L51"), Optional.of("cubicMetrePerSecond"), Optional.of("8.809768 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "penCalorie", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#penCalorie" ), "penCalorie", Set.of(new LangString( "pen calorie", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("N1"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "penGramProtein", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#penGramProtein" ), "penGramProtein", Set.of(new LangString( "pen gram (protein)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("D23"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pennyweight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pennyweight" ), "pennyweight", Set.of(new LangString( "pennyweight", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DWT"), Optional.of("gram"), Optional.of("1.555174 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "perMillePerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#perMillePerPsi" ), "perMillePerPsi", Set.of(new LangString( "per mille per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u2030/psi"), Optional.of("J12"), Optional.of("reciprocalPascalOrPascalToThePowerMinusOne"), Optional.of("1.450377 \u00D7 10\u207B\u2077 Pa\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percent", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percent" ), "percent", Set.of(new LangString( "percent", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%"), Optional.of("P1"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DIMENSIONLESS)) ) );
UNITS_BY_NAME.put( "percentPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerBar" ), "percentPerBar", Set.of(new LangString( "percent per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/bar"), Optional.of("H96"), Optional.of("reciprocalPascalOrPascalToThePowerMinusOne"), Optional.of("10\u207B\u2077 Pa\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerDecakelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerDecakelvin" ), "percentPerDecakelvin", Set.of(new LangString( "percent per decakelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/daK"), Optional.of("H73"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u00B3 K\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerDegree", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerDegree" ), "percentPerDegree", Set.of(new LangString( "percent per degree", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/\u00B0"), Optional.of("H90"), Optional.empty(), Optional.of("0.5729578 rad\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerDegreeCelsius" ), "percentPerDegreeCelsius", Set.of(new LangString( "percent per degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/\u00B0C"), Optional.of("M25"), Optional.empty(), Optional.of("10\u207B\u00B2 \u00B0C\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerHectobar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerHectobar" ), "percentPerHectobar", Set.of(new LangString( "percent per hectobar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/hbar"), Optional.of("H72"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerHundred", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerHundred" ), "percentPerHundred", Set.of(new LangString( "percent per hundred", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/100"), Optional.of("H93"), Optional.empty(), Optional.of("10\u207B\u2074"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerInch" ), "percentPerInch", Set.of(new LangString( "percent per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/in"), Optional.of("H98"), Optional.of("reciprocalMetre"), Optional.of("0.3937008 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerKelvin" ), "percentPerKelvin", Set.of(new LangString( "percent per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/K"), Optional.of("H25"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u00B2 K\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerMetre" ), "percentPerMetre", Set.of(new LangString( "percent per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/m"), Optional.of("H99"), Optional.of("reciprocalMetre"), Optional.of("10\u207B\u00B2 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerMillimetre" ), "percentPerMillimetre", Set.of(new LangString( "percent per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/mm"), Optional.of("J10"), Optional.of("reciprocalMetre"), Optional.of("10 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerMonth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerMonth" ), "percentPerMonth", Set.of(new LangString( "percent per month", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/mo"), Optional.of("H71"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerOhm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerOhm" ), "percentPerOhm", Set.of(new LangString( "percent per ohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/\u2126"), Optional.of("H89"), Optional.empty(), Optional.of("10\u207B\u00B2 \u2126\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerOneHundredThousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerOneHundredThousand" ), "percentPerOneHundredThousand", Set.of(new LangString( "percent per one hundred thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/100000"), Optional.of("H92"), Optional.empty(), Optional.of("10\u207B\u2077"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerTenThousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerTenThousand" ), "percentPerTenThousand", Set.of(new LangString( "percent per ten thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/10000"), Optional.of("H91"), Optional.empty(), Optional.of("10\u207B\u2076"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerThousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerThousand" ), "percentPerThousand", Set.of(new LangString( "percent per thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/1000"), Optional.of("H94"), Optional.empty(), Optional.of("10\u207B\u2075"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerVolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentPerVolt" ), "percentPerVolt", Set.of(new LangString( "percent per volt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("%/V"), Optional.of("H95"), Optional.empty(), Optional.of("10\u207B\u00B2 V\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentVolume", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentVolume" ), "percentVolume", Set.of(new LangString( "percent volume", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("VP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentWeight", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#percentWeight" ), "percentWeight", Set.of(new LangString( "percent weight", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("60"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "perm0Degreesc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#perm0Degreesc" ), "perm0Degreesc", Set.of(new LangString( "perm (0 \u00B0C)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("perm (0 \u00B0C)"), Optional.of("P91"), Optional.of("kilogramPerSquareMetrePascalSecond"), Optional.of("5.72135 \u00D7 10\u207B\u00B9\u00B9 kg/(m\u00B2 \u00D7 Pa \u00D7 s)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "perm23Degreesc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#perm23Degreesc" ), "perm23Degreesc", Set.of(new LangString( "perm (23 \u00B0C)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("perm (23 \u00B0C)"), Optional.of("P92"), Optional.of("kilogramPerSquareMetrePascalSecond"), Optional.of("5.74525 \u00D7 10\u207B\u00B9\u00B9 kg/(m\u00B2 \u00D7 Pa \u00D7 s)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "person", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#person" ), "person", Set.of(new LangString( "person", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("IE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "petabit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#petabit" ), "petabit", Set.of(new LangString( "petabit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pbit"), Optional.of("E78"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "petabitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#petabitPerSecond" ), "petabitPerSecond", Set.of(new LangString( "petabit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pbit/s"), Optional.of("E79"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "petabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#petabyte" ), "petabyte", Set.of(new LangString( "petabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pbyte"), Optional.of("E36"), Optional.of("byte"), Optional.of("10\u00B9\u2075 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "petajoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#petajoule" ), "petajoule", Set.of(new LangString( "petajoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("PJ"), Optional.of("C68"), Optional.of("joule"), Optional.of("10\u00B9\u2075 J"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "pferdestaerke", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pferdestaerke" ), "pferdestaerke", Set.of(new LangString( "Pferdestaerke", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("PS"), Optional.of("N12"), Optional.of("watt"), Optional.of("7.354988 \u00D7 10\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "pfund", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pfund" ), "pfund", Set.of(new LangString( "pfund", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pfd"), Optional.of("M86"), Optional.of("kilogram"), Optional.of("0.5 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "phon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#phon" ), "phon", Set.of(new LangString( "phon", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("C69"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.LOUDNESS_LEVEL)) ) );
UNITS_BY_NAME.put( "phot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#phot" ), "phot", Set.of(new LangString( "phot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ph"), Optional.of("P26"), Optional.empty(), Optional.of("10\u2074 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "pica", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pica" ), "pica", Set.of(new LangString( "pica", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("R1"), Optional.of("metre"), Optional.of("4.217518 \u00D7 10\u207B\u00B3 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "picoampere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picoampere" ), "picoampere", Set.of(new LangString( "picoampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pA"), Optional.of("C70"), Optional.of("ampere"), Optional.of("10\u207B\u00B9\u00B2 A"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "picocoulomb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picocoulomb" ), "picocoulomb", Set.of(new LangString( "picocoulomb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pC"), Optional.of("C71"), Optional.of("coulomb"), Optional.of("10\u207B\u00B9\u00B2 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "picofarad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picofarad" ), "picofarad", Set.of(new LangString( "picofarad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pF"), Optional.of("4T"), Optional.of("farad"), Optional.of("10\u207B\u00B9\u00B2 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "picofaradPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picofaradPerMetre" ), "picofaradPerMetre", Set.of(new LangString( "picofarad per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pF/m"), Optional.of("C72"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u00B9\u00B2 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.PERMITTIVITY, QuantityKinds.ELECTRIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "picohenry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picohenry" ), "picohenry", Set.of(new LangString( "picohenry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pH"), Optional.of("C73"), Optional.of("henry"), Optional.of("10\u207B\u00B9\u00B2 H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "picometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picometre" ), "picometre", Set.of(new LangString( "picometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pm"), Optional.of("C52"), Optional.of("metre"), Optional.of("10\u207B\u00B9\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.THICKNESS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "picopascalPerKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picopascalPerKilometre" ), "picopascalPerKilometre", Set.of(new LangString( "picopascal per kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pPa/km"), Optional.of("H69"), Optional.empty(), Optional.of("10\u207B\u00B9\u2075 m\u207B\u00B2 \u00D7 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "picosecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picosecond" ), "picosecond", Set.of(new LangString( "picosecond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ps"), Optional.of("H70"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B9\u00B2 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "picosiemens", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picosiemens" ), "picosiemens", Set.of(new LangString( "picosiemens", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pS"), Optional.of("N92"), Optional.of("siemens"), Optional.of("10\u207B\u00B9\u00B2 S"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "picosiemensPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picosiemensPerMetre" ), "picosiemensPerMetre", Set.of(new LangString( "picosiemens per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pS/m"), Optional.of("L42"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u00B9\u00B2 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "picovolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picovolt" ), "picovolt", Set.of(new LangString( "picovolt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pV"), Optional.of("N99"), Optional.of("voltDc"), Optional.of("10\u207B\u00B9\u00B2 V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.TENSION, QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.VOLTAGE, QuantityKinds.POTENTIAL_DIFFERENCE)) ) );
UNITS_BY_NAME.put( "picowatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picowatt" ), "picowatt", Set.of(new LangString( "picowatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pW"), Optional.of("C75"), Optional.of("watt"), Optional.of("10\u207B\u00B9\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "picowattPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#picowattPerSquareMetre" ), "picowattPerSquareMetre", Set.of(new LangString( "picowatt per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pW/m\u00B2"), Optional.of("C76"), Optional.of("wattPerSquareMetre"), Optional.of("10\u207B\u00B9\u00B2 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY)) ) );
UNITS_BY_NAME.put( "piece", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#piece" ), "piece", Set.of(new LangString( "piece", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("H87"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ping", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ping" ), "ping", Set.of(new LangString( "ping", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E19"), Optional.of("squareMetre"), Optional.of("3.305 m\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pintUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUk" ), "pintUk", Set.of(new LangString( "pint (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (UK)"), Optional.of("PTI"), Optional.of("cubicMetre"), Optional.of("5. 68261 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "pintUkPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUkPerDay" ), "pintUkPerDay", Set.of(new LangString( "pint (UK) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (UK)/d"), Optional.of("L53"), Optional.of("cubicMetrePerSecond"), Optional.of("6.577098 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUkPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUkPerHour" ), "pintUkPerHour", Set.of(new LangString( "pint (UK) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (UK)/h"), Optional.of("L54"), Optional.of("cubicMetrePerSecond"), Optional.of("1.578504 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
}

private static void init13() {
UNITS_BY_NAME.put( "pintUkPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUkPerMinute" ), "pintUkPerMinute", Set.of(new LangString( "pint (UK) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (UK)/min"), Optional.of("L55"), Optional.of("cubicMetrePerSecond"), Optional.of("9.471022 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUkPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUkPerSecond" ), "pintUkPerSecond", Set.of(new LangString( "pint (UK) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (UK)/s"), Optional.of("L56"), Optional.of("cubicMetrePerSecond"), Optional.of("5.682613 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUsLiquidPerDay" ), "pintUsLiquidPerDay", Set.of(new LangString( "pint (US liquid) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (US liq.)/d"), Optional.of("L57"), Optional.of("cubicMetrePerSecond"), Optional.of("5.476580 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUsLiquidPerHour" ), "pintUsLiquidPerHour", Set.of(new LangString( "pint (US liquid) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (US liq.)/h"), Optional.of("L58"), Optional.of("cubicMetrePerSecond"), Optional.of("1.314379 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUsLiquidPerMinute" ), "pintUsLiquidPerMinute", Set.of(new LangString( "pint (US liquid) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (US liq.)/min"), Optional.of("L59"), Optional.of("cubicMetrePerSecond"), Optional.of("7.886275 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pintUsLiquidPerSecond" ), "pintUsLiquidPerSecond", Set.of(new LangString( "pint (US liquid) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pt (US liq.)/s"), Optional.of("L60"), Optional.of("cubicMetrePerSecond"), Optional.of("4.731765 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pipelineJoint", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pipelineJoint" ), "pipelineJoint", Set.of(new LangString( "pipeline joint", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("JNT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pitch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pitch" ), "pitch", Set.of(new LangString( "pitch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("PI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pixel", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pixel" ), "pixel", Set.of(new LangString( "pixel", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E37"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poise", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poise" ), "poise", Set.of(new LangString( "poise", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("P"), Optional.of("89"), Optional.of("pascalSecond"), Optional.of("0.1 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poisePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poisePerBar" ), "poisePerBar", Set.of(new LangString( "poise per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("P/bar"), Optional.of("F06"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poisePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poisePerKelvin" ), "poisePerKelvin", Set.of(new LangString( "poise per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("P/K"), Optional.of("F86"), Optional.empty(), Optional.of("10\u207B\u00B9 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poisePerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poisePerPascal" ), "poisePerPascal", Set.of(new LangString( "poise per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("P/Pa"), Optional.of("N35"), Optional.of("secondUnitOfTime"), Optional.of("0.1 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "pond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pond" ), "pond", Set.of(new LangString( "pond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("p"), Optional.of("M78"), Optional.of("newton"), Optional.of("9.80665 \u00D7 10\u207B\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "pound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#pound" ), "pound", Set.of(new LangString( "pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb"), Optional.of("LBR"), Optional.of("kilogram"), Optional.of("0.45359237 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicFootDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerCubicFootDegreeFahrenheit" ), "poundAvoirdupoisPerCubicFootDegreeFahrenheit", Set.of(new LangString( "pound (avoirdupois) per cubic foot degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/ft\u00B3)/\u00B0F"), Optional.of("K69"), Optional.empty(), Optional.of("28.83323 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicFootPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerCubicFootPsi" ), "poundAvoirdupoisPerCubicFootPsi", Set.of(new LangString( "pound (avoirdupois) per cubic foot psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/ft\u00B3)/psi"), Optional.of("K70"), Optional.empty(), Optional.of("2.323282 \u00D7 10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicInchDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerCubicInchDegreeFahrenheit" ), "poundAvoirdupoisPerCubicInchDegreeFahrenheit", Set.of(new LangString( "pound (avoirdupois) per cubic inch degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/in\u00B3)/\u00B0F"), Optional.of("K75"), Optional.empty(), Optional.of("4.982384 \u00D7 10\u2074 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicInchPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerCubicInchPsi" ), "poundAvoirdupoisPerCubicInchPsi", Set.of(new LangString( "pound (avoirdupois) per cubic inch psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/in\u00B3)/psi"), Optional.of("K76"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("4.014632 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerDay" ), "poundAvoirdupoisPerDay", Set.of(new LangString( "pound (avoirdupois) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/d"), Optional.of("K66"), Optional.of("kilogramPerSecond"), Optional.of("5.249912 \u00D7 10\u207B\u2076 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerDegreeFahrenheit" ), "poundAvoirdupoisPerDegreeFahrenheit", Set.of(new LangString( "pound (avoirdupois) per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/\u00B0F"), Optional.of("K64"), Optional.of("kilogramPerKelvin"), Optional.of("0.8164663 kg/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerGallonUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerGallonUk" ), "poundAvoirdupoisPerGallonUk", Set.of(new LangString( "pound (avoirdupois) per gallon (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/gal (UK)"), Optional.of("K71"), Optional.of("kilogramPerCubicMetre"), Optional.of("99.77637 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerHourDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerHourDegreeFahrenheit" ), "poundAvoirdupoisPerHourDegreeFahrenheit", Set.of(new LangString( "pound (avoirdupois) per hour degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/h)/\u00B0F"), Optional.of("K73"), Optional.empty(), Optional.of("2.267962 \u00D7 10\u207B\u2074 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerHourPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerHourPsi" ), "poundAvoirdupoisPerHourPsi", Set.of(new LangString( "pound (avoirdupois) per hour psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/h)/psi"), Optional.of("K74"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.827445 \u00D7 10\u207B\u2078 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerMinute" ), "poundAvoirdupoisPerMinute", Set.of(new LangString( "pound (avoirdupois) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/min"), Optional.of("K78"), Optional.of("kilogramPerSecond"), Optional.of("7.559873 \u00D7 10\u207B\u00B3 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerMinuteDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerMinuteDegreeFahrenheit" ), "poundAvoirdupoisPerMinuteDegreeFahrenheit", Set.of(new LangString( "pound (avoirdupois) per minute degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/(min\u00B7\u00B0F)"), Optional.of("K79"), Optional.empty(), Optional.of("1.360777 \u00D7 10\u207B\u00B2 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerMinutePsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerMinutePsi" ), "poundAvoirdupoisPerMinutePsi", Set.of(new LangString( "pound (avoirdupois) per minute psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/min)/psi"), Optional.of("K80"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.096467 \u00D7 10\u207B\u2076 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerPsi" ), "poundAvoirdupoisPerPsi", Set.of(new LangString( "pound (avoirdupois) per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/psi"), Optional.of("K77"), Optional.of("kilogramPerPascal"), Optional.of("6.578802 \u00D7 10\u207B\u2075 kg/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerSecond" ), "poundAvoirdupoisPerSecond", Set.of(new LangString( "pound (avoirdupois) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/s"), Optional.of("K81"), Optional.of("kilogramPerSecond"), Optional.of("0.4535924 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerSecondDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerSecondDegreeFahrenheit" ), "poundAvoirdupoisPerSecondDegreeFahrenheit", Set.of(new LangString( "pound (avoirdupois) per second degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/s)/\u00B0F"), Optional.of("K82"), Optional.empty(), Optional.of("0.8164663 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerSecondPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisPerSecondPsi" ), "poundAvoirdupoisPerSecondPsi", Set.of(new LangString( "pound (avoirdupois) per second psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(lb/s)/psi"), Optional.of("K83"), Optional.of("kilogramPerSecondPascal"), Optional.of("6.578802 \u00D7 10\u207B\u2075 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundAvoirdupoisSquareFoot" ), "poundAvoirdupoisSquareFoot", Set.of(new LangString( "pound (avoirdupois) square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb\u00B7ft\u00B2"), Optional.of("K65"), Optional.of("kilogramMetreSquared"), Optional.of("4.214011 \u00D7 10\u207B\u00B2 kg \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA)) ) );
UNITS_BY_NAME.put( "poundFootPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundFootPerSecond" ), "poundFootPerSecond", Set.of(new LangString( "pound foot per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb\u00B7(ft/s)"), Optional.of("N10"), Optional.of("kilogramMetrePerSecond"), Optional.of("1.382550 \u00D7 10\u207B\u00B9 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "poundForce", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForce" ), "poundForce", Set.of(new LangString( "pound-force", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf"), Optional.of("C78"), Optional.of("newton"), Optional.of("4.448222 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "poundForceFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceFoot" ), "poundForceFoot", Set.of(new LangString( "pound-force foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7ft"), Optional.of("M92"), Optional.of("newtonMetre"), Optional.of("1.355818 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "poundForceFootPerAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceFootPerAmpere" ), "poundForceFootPerAmpere", Set.of(new LangString( "pound-force foot per ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7ft/A"), Optional.of("F22"), Optional.empty(), Optional.of("1.35582 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundForceFootPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceFootPerInch" ), "poundForceFootPerInch", Set.of(new LangString( "pound-force foot per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7ft/in"), Optional.of("P89"), Optional.empty(), Optional.of("53.37866 m \u00D7 kg / s\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poundForceFootPerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceFootPerPound" ), "poundForceFootPerPound", Set.of(new LangString( "pound-force foot per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7ft/lb"), Optional.of("G20"), Optional.empty(), Optional.of("2.98907 m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.WORK_PER_UNIT_WEIGHT, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundForceInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceInch" ), "poundForceInch", Set.of(new LangString( "pound-force inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7in"), Optional.of("F21"), Optional.empty(), Optional.of("1.12985 \u00D7 10\u207B\u00B9 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundForceInchPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceInchPerInch" ), "poundForceInchPerInch", Set.of(new LangString( "pound-force inch per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7in/in"), Optional.of("P90"), Optional.empty(), Optional.of("4.448222 m \u00D7 kg / s\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poundForcePerFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForcePerFoot" ), "poundForcePerFoot", Set.of(new LangString( "pound-force per foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf/ft"), Optional.of("F17"), Optional.empty(), Optional.of("1.45939 \u00D7 10\u00B9 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.FORCE_DIVIDED_BY_LENGTH)) ) );
UNITS_BY_NAME.put( "poundForcePerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForcePerInch" ), "poundForcePerInch", Set.of(new LangString( "pound-force per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf/in"), Optional.of("F48"), Optional.empty(), Optional.of("1.75127 \u00D7 10\u00B2 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.FORCE_DIVIDED_BY_LENGTH)) ) );
UNITS_BY_NAME.put( "poundForcePerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForcePerSquareFoot" ), "poundForcePerSquareFoot", Set.of(new LangString( "pound-force per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf/ft\u00B2"), Optional.of("K85"), Optional.of("pascal"), Optional.of("47.88026 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "poundForcePerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForcePerSquareInch" ), "poundForcePerSquareInch", Set.of(new LangString( "pound-force per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf/in\u00B2"), Optional.of("PS"), Optional.of("pascal"), Optional.of("6.894757 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "poundForcePerSquareInchDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForcePerSquareInchDegreeFahrenheit" ), "poundForcePerSquareInchDegreeFahrenheit", Set.of(new LangString( "pound-force per square inch degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi/\u00B0F"), Optional.of("K86"), Optional.of("pascalPerKelvin"), Optional.of("1.241056 \u00D7 10\u2074 Pa/K"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "poundForcePerYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForcePerYard" ), "poundForcePerYard", Set.of(new LangString( "pound-force per yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf/yd"), Optional.of("N33"), Optional.of("newtonPerMetre"), Optional.of("4.864635 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "poundForceSecondPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceSecondPerSquareFoot" ), "poundForceSecondPerSquareFoot", Set.of(new LangString( "pound-force second per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7s/ft\u00B2"), Optional.of("K91"), Optional.of("pascalSecond"), Optional.of("47.88026 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundForceSecondPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundForceSecondPerSquareInch" ), "poundForceSecondPerSquareInch", Set.of(new LangString( "pound-force second per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbf\u00B7s/in\u00B2"), Optional.of("K92"), Optional.of("pascalSecond"), Optional.of("6.894757 \u00D7 10\u00B3 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundInchPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundInchPerSecond" ), "poundInchPerSecond", Set.of(new LangString( "pound inch per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb\u00B7(in/s)"), Optional.of("N11"), Optional.of("kilogramMetrePerSecond"), Optional.of("1.152125 \u00D7 10\u207B\u00B2 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "poundInchSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundInchSquared" ), "poundInchSquared", Set.of(new LangString( "pound inch squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb\u00B7in\u00B2"), Optional.of("F20"), Optional.of("kilogramMetreSquared"), Optional.of("2.926397 \u00D7 10\u207B\u2074 kg \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA)) ) );
UNITS_BY_NAME.put( "poundMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundMole" ), "poundMole", Set.of(new LangString( "pound mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbmol"), Optional.of("P44"), Optional.of("mole"), Optional.of("453.5924 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "poundMolePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundMolePerMinute" ), "poundMolePerMinute", Set.of(new LangString( "pound mole per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbmol/h"), Optional.of("P46"), Optional.of("molePerSecond"), Optional.of("7.559873 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "poundMolePerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundMolePerPound" ), "poundMolePerPound", Set.of(new LangString( "pound mole per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbmol/lb"), Optional.of("P48"), Optional.of("molePerKilogram"), Optional.of("10\u00B3 mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "poundMolePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundMolePerSecond" ), "poundMolePerSecond", Set.of(new LangString( "pound mole per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lbmol/s"), Optional.of("P45"), Optional.of("molePerSecond"), Optional.of("4.535924 \u00D7 10\u00B2 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "poundPerCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerCubicFoot" ), "poundPerCubicFoot", Set.of(new LangString( "pound per cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/ft\u00B3"), Optional.of("87"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.601846 \u00D7 10\u00B9 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "poundPerCubicInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerCubicInch" ), "poundPerCubicInch", Set.of(new LangString( "pound per cubic inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/in\u00B3"), Optional.of("LA"), Optional.of("kilogramPerCubicMetre"), Optional.of("2.767990 \u00D7 10\u2074 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerCubicYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerCubicYard" ), "poundPerCubicYard", Set.of(new LangString( "pound per cubic yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/yd\u00B3"), Optional.of("K84"), Optional.of("kilogramPerCubicMetre"), Optional.of("0.5932764 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerFoot" ), "poundPerFoot", Set.of(new LangString( "pound per foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/ft"), Optional.of("P2"), Optional.of("kilogramPerMetre"), Optional.of("1.488164 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerFootDay" ), "poundPerFootDay", Set.of(new LangString( "pound per foot day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/(ft\u00B7d)"), Optional.of("N44"), Optional.of("pascalSecond"), Optional.of("1.722412 \u00D7 10\u207B\u2075 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerFootHour" ), "poundPerFootHour", Set.of(new LangString( "pound per foot hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/(ft\u00B7h)"), Optional.of("K67"), Optional.of("pascalSecond"), Optional.of("4.133789 \u00D7 10\u207B\u2074 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerFootMinute" ), "poundPerFootMinute", Set.of(new LangString( "pound per foot minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/(ft\u00B7min)"), Optional.of("N43"), Optional.of("pascalSecond"), Optional.of("2.480273 \u00D7 10\u207B\u00B2 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerFootSecond" ), "poundPerFootSecond", Set.of(new LangString( "pound per foot second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/(ft\u00B7s)"), Optional.of("K68"), Optional.of("pascalSecond"), Optional.of("1.488164 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerGallonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerGallonUs" ), "poundPerGallonUs", Set.of(new LangString( "pound per gallon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/gal (US)"), Optional.of("GE"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.198264 \u00D7 10\u00B2 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerHour" ), "poundPerHour", Set.of(new LangString( "pound per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/h"), Optional.of("4U"), Optional.of("kilogramPerSecond"), Optional.of("1.259979 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundPerInchOfLengthUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerInchOfLengthUnit" ), "poundPerInchOfLengthUnit", Set.of(new LangString( "pound per inch of length", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/in"), Optional.of("PO"), Optional.of("kilogramPerMetre"), Optional.of("1.785797 \u00D7 10\u00B9 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerPound" ), "poundPerPound", Set.of(new LangString( "pound per pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/lb"), Optional.of("M91"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "poundPerReam", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerReam" ), "poundPerReam", Set.of(new LangString( "pound per ream", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("RP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poundPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerSquareFoot" ), "poundPerSquareFoot", Set.of(new LangString( "pound per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/ft\u00B2"), Optional.of("FP"), Optional.of("kilogramPerSquareMetre"), Optional.of("4.882428 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "poundPerSquareInchAbsolute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerSquareInchAbsolute" ), "poundPerSquareInchAbsolute", Set.of(new LangString( "pound per square inch absolute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/in\u00B2"), Optional.of("80"), Optional.of("kilogramPerSquareMetre"), Optional.of("7.030696 \u00D7 10\u00B2 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "poundPerSquareYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerSquareYard" ), "poundPerSquareYard", Set.of(new LangString( "pound per square yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/yd\u00B2"), Optional.of("N25"), Optional.of("kilogramPerSquareMetre"), Optional.of("5.424919 \u00D7 10\u207B\u00B9 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "poundPerYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundPerYard" ), "poundPerYard", Set.of(new LangString( "pound per yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("lb/yd"), Optional.of("M84"), Optional.of("kilogramPerMetre"), Optional.of("4.960546 \u00D7 10\u207B\u00B9 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "poundal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundal" ), "poundal", Set.of(new LangString( "poundal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pdl"), Optional.of("M76"), Optional.of("newton"), Optional.of("1.382550 \u00D7 10\u207B\u00B9 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "poundalFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalFoot" ), "poundalFoot", Set.of(new LangString( "poundal foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pdl\u00B7ft"), Optional.of("M95"), Optional.of("newtonMetre"), Optional.of("4.214011 \u00D7 10\u207B\u00B2 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "poundalInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalInch" ), "poundalInch", Set.of(new LangString( "poundal inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pdl\u00B7in"), Optional.of("M96"), Optional.of("newtonMetre"), Optional.of("3.51167710\u207B\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundalPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalPerInch" ), "poundalPerInch", Set.of(new LangString( "poundal per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pdl/in"), Optional.of("N32"), Optional.of("newtonPerMetre"), Optional.of("5.443110 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "poundalPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalPerSquareFoot" ), "poundalPerSquareFoot", Set.of(new LangString( "poundal per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pdl/ft\u00B2"), Optional.of("N21"), Optional.of("pascal"), Optional.of("1.488164 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "poundalPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalPerSquareInch" ), "poundalPerSquareInch", Set.of(new LangString( "poundal per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("pdl/in\u00B2"), Optional.of("N26"), Optional.of("pascal"), Optional.of("2.142957 \u00D7 10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "poundalSecondPerSquareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalSecondPerSquareFoot" ), "poundalSecondPerSquareFoot", Set.of(new LangString( "poundal second per square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(pdl/ft\u00B2)\u00B7s"), Optional.of("N34"), Optional.of("pascalSecond"), Optional.of("1.488164 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundalSecondPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#poundalSecondPerSquareInch" ), "poundalSecondPerSquareInch", Set.of(new LangString( "poundal second per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(pdl/in\u00B2)\u00B7s"), Optional.of("N42"), Optional.of("pascalSecond"), Optional.of("2.142957 \u00D7 10\u00B2 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "printPoint", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#printPoint" ), "printPoint", Set.of(new LangString( "print point", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("N3"), Optional.empty(), Optional.of("0.0138 in (appro\u00D7)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "proofGallon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#proofGallon" ), "proofGallon", Set.of(new LangString( "proof gallon", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("PGL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "proofLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#proofLitre" ), "proofLitre", Set.of(new LangString( "proof litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("PFL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "psiCubicInchPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#psiCubicInchPerSecond" ), "psiCubicInchPerSecond", Set.of(new LangString( "psi cubic inch per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi\u00B7in\u00B3/s"), Optional.of("K87"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("0.112985 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#psiCubicMetrePerSecond" ), "psiCubicMetrePerSecond", Set.of(new LangString( "psi cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi\u00B7m\u00B3/s"), Optional.of("K89"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("6.894757 \u00D7 10\u00B3 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiCubicYardPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#psiCubicYardPerSecond" ), "psiCubicYardPerSecond", Set.of(new LangString( "psi cubic yard per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi\u00B7yd\u00B3/s"), Optional.of("K90"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("5.271420 \u00D7 10\u00B3 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiLitrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#psiLitrePerSecond" ), "psiLitrePerSecond", Set.of(new LangString( "psi litre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi\u00B7l/s"), Optional.of("K88"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("6.894757 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#psiPerInch" ), "psiPerInch", Set.of(new LangString( "psi per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi/in"), Optional.of("P86"), Optional.empty(), Optional.of("2.714471 \u00D7 10\u2075 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "psiPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#psiPerPsi" ), "psiPerPsi", Set.of(new LangString( "psi per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("psi/psi"), Optional.of("L52"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "quad1015Btuit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quad1015Btuit" ), "quad1015Btuit", Set.of(new LangString( "quad (1015 BtuIT)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("quad"), Optional.of("N70"), Optional.of("joule"), Optional.of("1.055056 \u00D7 10\u00B9\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENERGY, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENTHALPY)) ) );
UNITS_BY_NAME.put( "quartUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUk" ), "quartUk", Set.of(new LangString( "quart (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (UK)"), Optional.of("QTI"), Optional.of("cubicMetre"), Optional.of("1.1365225 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUkLiquidPerDay" ), "quartUkLiquidPerDay", Set.of(new LangString( "quart (UK liquid) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (UK liq.)/d"), Optional.of("K94"), Optional.of("cubicMetrePerSecond"), Optional.of("1.315420 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUkLiquidPerHour" ), "quartUkLiquidPerHour", Set.of(new LangString( "quart (UK liquid) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (UK liq.)/h"), Optional.of("K95"), Optional.of("cubicMetrePerSecond"), Optional.of("3.157008 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUkLiquidPerMinute" ), "quartUkLiquidPerMinute", Set.of(new LangString( "quart (UK liquid) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (UK liq.)/min"), Optional.of("K96"), Optional.of("cubicMetrePerSecond"), Optional.of("1.894205 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUkLiquidPerSecond" ), "quartUkLiquidPerSecond", Set.of(new LangString( "quart (UK liquid) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (UK liq.)/s"), Optional.of("K97"), Optional.of("cubicMetrePerSecond"), Optional.of("1.136523 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUsLiquidPerDay" ), "quartUsLiquidPerDay", Set.of(new LangString( "quart (US liquid) per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (US liq.)/d"), Optional.of("K98"), Optional.of("cubicMetrePerSecond"), Optional.of("1.095316 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUsLiquidPerHour" ), "quartUsLiquidPerHour", Set.of(new LangString( "quart (US liquid) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (US liq.)/h"), Optional.of("K99"), Optional.of("cubicMetrePerSecond"), Optional.of("2.628758 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUsLiquidPerMinute" ), "quartUsLiquidPerMinute", Set.of(new LangString( "quart (US liquid) per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (US liq.)/min"), Optional.of("L10"), Optional.of("cubicMetrePerSecond"), Optional.of("1.577255 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quartUsLiquidPerSecond" ), "quartUsLiquidPerSecond", Set.of(new LangString( "quart (US liquid) per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qt (US liq.)/s"), Optional.of("L11"), Optional.of("cubicMetrePerSecond"), Optional.of("9.463529 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quarterOfAYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quarterOfAYear" ), "quarterOfAYear", Set.of(new LangString( "quarter (of a year)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("QAN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "quarterUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quarterUk" ), "quarterUk", Set.of(new LangString( "quarter (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Qr (UK)"), Optional.of("QTR"), Optional.of("kilogram"), Optional.of("12.700 59 kg"), Collections.emptySet() ) );
}

private static void init14() {
UNITS_BY_NAME.put( "quire", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#quire" ), "quire", Set.of(new LangString( "quire", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("qr"), Optional.of("QR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "rackUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#rackUnit" ), "rackUnit", Set.of(new LangString( "rack unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("U or RU"), Optional.of("H80"), Optional.of("metre"), Optional.of("4.445 \u00D7 10\u207B\u00B2 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "rad", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#rad" ), "rad", Set.of(new LangString( "rad", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad"), Optional.of("C80"), Optional.of("gray"), Optional.of("10\u207B\u00B2 Gy"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE)) ) );
UNITS_BY_NAME.put( "radian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#radian" ), "radian", Set.of(new LangString( "radian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad"), Optional.of("C81"), Optional.empty(), Optional.of("rad"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_DISPLACEMENT, QuantityKinds.ANGLE_PLANE, QuantityKinds.LOSS_ANGLE, QuantityKinds.ANGLE_OF_OPTICAL_ROTATION, QuantityKinds.BRAGG_ANGLE, QuantityKinds.PHASE_DIFFERENCE)) ) );
UNITS_BY_NAME.put( "radianPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#radianPerMetre" ), "radianPerMetre", Set.of(new LangString( "radian per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad/m"), Optional.of("C84"), Optional.empty(), Optional.of("rad/m"), new HashSet<>(Arrays.asList(QuantityKinds.DEBYE_ANGULAR_REPETENCY, QuantityKinds.ANGULAR_REPETENCY, QuantityKinds.DEBYE_ANGULAR_WAVE_NUMBER, QuantityKinds.ANGULAR_WAVE_NUMBER, QuantityKinds.ANGULAR_WAVENUMBER)) ) );
UNITS_BY_NAME.put( "radianPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#radianPerSecond" ), "radianPerSecond", Set.of(new LangString( "radian per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad/s"), Optional.of("2A"), Optional.empty(), Optional.of("rad/s"), new HashSet<>(Arrays.asList(QuantityKinds.NUCLEAR_PRECESSION, QuantityKinds.ANGULAR_VELOCITY, QuantityKinds.CIRCULAR_FREQUENCY, QuantityKinds.ANGULAR_FREQUENCY, QuantityKinds.PULSATANCE, QuantityKinds.CYCLOTRON_ANGULAR_FREQUENCY, QuantityKinds.DEBYE_ANGULAR_FREQUENCY)) ) );
UNITS_BY_NAME.put( "radianPerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#radianPerSecondSquared" ), "radianPerSecondSquared", Set.of(new LangString( "radian per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad/s\u00B2"), Optional.of("2B"), Optional.empty(), Optional.of("rad/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_ACCELERATION)) ) );
UNITS_BY_NAME.put( "radianSquareMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#radianSquareMetrePerKilogram" ), "radianSquareMetrePerKilogram", Set.of(new LangString( "radian square metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad\u00B7m\u00B2/kg"), Optional.of("C83"), Optional.empty(), Optional.of("rad \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_OPTICAL_ROTATORY_POWER, QuantityKinds.ROTATORY_POWER, QuantityKinds.MASSIC_OPTICAL)) ) );
UNITS_BY_NAME.put( "radianSquareMetrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#radianSquareMetrePerMole" ), "radianSquareMetrePerMole", Set.of(new LangString( "radian square metre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rad\u00B7m\u00B2/mol"), Optional.of("C82"), Optional.empty(), Optional.of("rad \u00D7 m\u00B2/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_OPTICAL_ROTATORY_POWER)) ) );
UNITS_BY_NAME.put( "rate", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#rate" ), "rate", Set.of(new LangString( "rate", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("A9"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ration", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ration" ), "ration", Set.of(new LangString( "ration", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("13"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ream", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#ream" ), "ream", Set.of(new LangString( "ream", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("RM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalAngstrom", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalAngstrom" ), "reciprocalAngstrom", Set.of(new LangString( "reciprocal angstrom", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00C5\u207B\u00B9"), Optional.of("C85"), Optional.of("reciprocalMetre"), Optional.of("10\u00B9\u2070 m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FERMI_ANGULAR_WAVE_NUMBER, QuantityKinds.FERMI_ANGULAR_REPETENCY)) ) );
UNITS_BY_NAME.put( "reciprocalBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalBar" ), "reciprocalBar", Set.of(new LangString( "reciprocal bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/bar"), Optional.of("F58"), Optional.empty(), Optional.of("bar\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.COMPRESSIBILITY, QuantityKinds.BULK_COMPRESSIBILITY)) ) );
UNITS_BY_NAME.put( "reciprocalCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCentimetre" ), "reciprocalCentimetre", Set.of(new LangString( "reciprocal centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u207B\u00B9"), Optional.of("E90"), Optional.of("reciprocalMetre"), Optional.of("10\u00B2 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalCubicCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicCentimetre" ), "reciprocalCubicCentimetre", Set.of(new LangString( "reciprocal cubic centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u207B\u00B3"), Optional.of("H50"), Optional.of("reciprocalCubicMetre"), Optional.of("10\u2076 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicFoot" ), "reciprocalCubicFoot", Set.of(new LangString( "reciprocal cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/ft\u00B3"), Optional.of("K20"), Optional.of("reciprocalCubicMetre"), Optional.of("35.31466 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "reciprocalCubicInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicInch" ), "reciprocalCubicInch", Set.of(new LangString( "reciprocal cubic inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/in\u00B3"), Optional.of("K49"), Optional.of("reciprocalCubicMetre"), Optional.of("6.1023759 \u00D7 10\u2074 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "reciprocalCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicMetre" ), "reciprocalCubicMetre", Set.of(new LangString( "reciprocal cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u207B\u00B3"), Optional.of("C86"), Optional.empty(), Optional.of("m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_HOLE_NUMBER, QuantityKinds.ION_NUMBER_DENSITY, QuantityKinds.HOLE_NUMBER_DENSITY, QuantityKinds.VOLUMIC_ELECTRON_NUMBER, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.INTRINSIC_NUMBER_DENSITY, QuantityKinds.DONOR_NUMBER_DENSITY, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.VOLUMIC_INTRINSIS_NUMBER, QuantityKinds.ACCEPTOR_NUMBER_DENSITY, QuantityKinds.ION_DENSITY, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.NEUTRON_NUMBER_DENSITY, QuantityKinds.VOLUMIC_ACCEPTOR_NUMBER, QuantityKinds.ELECTRON_NUMBER_DENSITY, QuantityKinds.VOLUMIC_DONOR_NUMBER)) ) );
UNITS_BY_NAME.put( "reciprocalCubicMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicMetrePerSecond" ), "reciprocalCubicMetrePerSecond", Set.of(new LangString( "reciprocal cubic metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u207B\u00B3/s"), Optional.of("C87"), Optional.empty(), Optional.of("m\u207B\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.SLOWING_DOWN_DENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalCubicMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicMillimetre" ), "reciprocalCubicMillimetre", Set.of(new LangString( "reciprocal cubic millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/mm\u00B3"), Optional.of("L20"), Optional.of("reciprocalCubicMetre"), Optional.of("10\u2079 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalCubicYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalCubicYard" ), "reciprocalCubicYard", Set.of(new LangString( "reciprocal cubic yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/yd\u00B3"), Optional.of("M10"), Optional.of("reciprocalCubicMetre"), Optional.of("1.307951 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalDay" ), "reciprocalDay", Set.of(new LangString( "reciprocal day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("d\u207B\u00B9"), Optional.of("E91"), Optional.of("reciprocalSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2075 s\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalDegreeFahrenheit" ), "reciprocalDegreeFahrenheit", Set.of(new LangString( "reciprocal degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/\u00B0F"), Optional.of("J26"), Optional.empty(), Optional.of("1.81/K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "reciprocalElectronVoltPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalElectronVoltPerCubicMetre" ), "reciprocalElectronVoltPerCubicMetre", Set.of(new LangString( "reciprocal electron volt per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("eV\u207B\u00B9/m\u00B3"), Optional.of("C88"), Optional.of("reciprocalJoulePerCubicMetre"), Optional.of("6.24146 \u00D7 10\u00B9\u2078 J\u207B\u00B9/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_STATES)) ) );
UNITS_BY_NAME.put( "reciprocalHenry", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalHenry" ), "reciprocalHenry", Set.of(new LangString( "reciprocal henry", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("H\u207B\u00B9"), Optional.of("C89"), Optional.empty(), Optional.of("H\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RELUCTANCE)) ) );
UNITS_BY_NAME.put( "reciprocalHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalHour" ), "reciprocalHour", Set.of(new LangString( "reciprocal hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/h"), Optional.of("H10"), Optional.of("reciprocalSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalInch" ), "reciprocalInch", Set.of(new LangString( "reciprocal inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/in"), Optional.of("Q24"), Optional.of("reciprocalMetre"), Optional.of("39.37008 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalJoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalJoule" ), "reciprocalJoule", Set.of(new LangString( "reciprocal joule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/J"), Optional.of("N91"), Optional.empty(), Optional.of("1/J"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalJoulePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalJoulePerCubicMetre" ), "reciprocalJoulePerCubicMetre", Set.of(new LangString( "reciprocal joule per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("J\u207B\u00B9/m\u00B3"), Optional.of("C90"), Optional.empty(), Optional.of("J\u207B\u00B9/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_STATES)) ) );
UNITS_BY_NAME.put( "reciprocalKelvinOrKelvinToThePowerMinusOne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalKelvinOrKelvinToThePowerMinusOne" ), "reciprocalKelvinOrKelvinToThePowerMinusOne", Set.of(new LangString( "reciprocal kelvin or kelvin to the power minus one", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("K\u207B\u00B9"), Optional.of("C91"), Optional.empty(), Optional.of("K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RELATIVE_PRESSURE_COEFFICIENT, QuantityKinds.LINEAR_EXPANSION_COEFFICIENT, QuantityKinds.CUBIC_EXPANSION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalKilovoltAmpereReciprocalHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalKilovoltAmpereReciprocalHour" ), "reciprocalKilovoltAmpereReciprocalHour", Set.of(new LangString( "reciprocal kilovolt - ampere reciprocal hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/kVAh"), Optional.of("M21"), Optional.empty(), Optional.of("2.777778 \u00D7 10\u207B\u2077 (V \u00D7 A \u00D7 s)\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalLitre" ), "reciprocalLitre", Set.of(new LangString( "reciprocal litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/l"), Optional.of("K63"), Optional.of("reciprocalCubicMetre"), Optional.of("10\u00B3 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "reciprocalMegakelvinOrMegakelvinToThePowerMinusOne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalMegakelvinOrMegakelvinToThePowerMinusOne" ), "reciprocalMegakelvinOrMegakelvinToThePowerMinusOne", Set.of(new LangString( "reciprocal megakelvin or megakelvin to the power minus one", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/MK"), Optional.of("M20"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u2076 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.CUBIC_EXPANSION_COEFFICIENT, QuantityKinds.LINEAR_EXPANSION_COEFFICIENT, QuantityKinds.RELATIVE_PRESSURE_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalMetre" ), "reciprocalMetre", Set.of(new LangString( "reciprocal metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u207B\u00B9"), Optional.of("C92"), Optional.empty(), Optional.of("m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.WAVENUMBER, QuantityKinds.LINEAR_IONIZATION_BY_A_PARTICLE, QuantityKinds.WAVE_NUMBER, QuantityKinds.ANGULAR_RECIPROCAL_LATTICE_VECTOR, QuantityKinds.LENS_POWER, QuantityKinds.MACROSCOPIC_CROSS_SECTION, QuantityKinds.CURVATURE, QuantityKinds.ATTENUATION_COEFFICIENT, QuantityKinds.FUNDAMENTAL_RECIPROCAL_LATTICE_VECTOR, QuantityKinds.MACROSCOPIC_TOTAL_CROSS_SECTION, QuantityKinds.PHASE_COEFFICIENT, QuantityKinds.VOLUMIC_TOTAL_CROSS_SECTION, QuantityKinds.ANGULAR_REPETENCY, QuantityKinds.LINEAR_ATTENUATION_COEFFICIENT, QuantityKinds.LINEAR_ABSORPTION_COEFFICIENT, QuantityKinds.VERGENCE, QuantityKinds.RYDBERG_CONSTANT, QuantityKinds.PROPAGATION_COEFFICIENT, QuantityKinds.ANGULAR_WAVE_NUMBER, QuantityKinds.VOLUMIC_CROSS_SECTION, QuantityKinds.LINEAR_EXTINCTION_COEFFICIENT, QuantityKinds.REPETENCY, QuantityKinds.TOTAL_IONIZATION_BY_A_PARTICLE)) ) );
UNITS_BY_NAME.put( "reciprocalMetreSquaredReciprocalSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalMetreSquaredReciprocalSecond" ), "reciprocalMetreSquaredReciprocalSecond", Set.of(new LangString( "reciprocal metre squared reciprocal second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u207B\u00B2/s"), Optional.of("B81"), Optional.empty(), Optional.of("m\u207B\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.NEUTRON_FLUENCE_RATE, QuantityKinds.PARTICLE_FLUENCE_RATE, QuantityKinds.CURRENT_DENSITY_OF_PARTICLES, QuantityKinds.PARTICAL_FLUX_DENSITY, QuantityKinds.NEUTRONFLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalMinute" ), "reciprocalMinute", Set.of(new LangString( "reciprocal minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("min\u207B\u00B9"), Optional.of("C94"), Optional.of("secondUnitOfTime"), Optional.of("1.666667 \u00D7 10\u207B\u00B2 s"), new HashSet<>(Arrays.asList(QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalMole" ), "reciprocalMole", Set.of(new LangString( "reciprocal mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mol\u207B\u00B9"), Optional.of("C95"), Optional.empty(), Optional.of("mol\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.AVOGADRO_CONSTANT)) ) );
UNITS_BY_NAME.put( "reciprocalMonth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalMonth" ), "reciprocalMonth", Set.of(new LangString( "reciprocal month", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/mo"), Optional.of("H11"), Optional.of("reciprocalSecond"), Optional.of("3.80257 \u00D7 10\u207B\u2077 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalPascalOrPascalToThePowerMinusOne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalPascalOrPascalToThePowerMinusOne" ), "reciprocalPascalOrPascalToThePowerMinusOne", Set.of(new LangString( "reciprocal pascal or pascal to the power minus one", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Pa\u207B\u00B9"), Optional.of("C96"), Optional.empty(), Optional.of("Pa\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ISOTHERMAL_COMPRESSIBILITY, QuantityKinds.ISENTROPIC_COMPRESSIBILITY, QuantityKinds.COMPRESSIBILITY, QuantityKinds.BULK_COMPRESSIBILITY)) ) );
UNITS_BY_NAME.put( "reciprocalPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalPsi" ), "reciprocalPsi", Set.of(new LangString( "reciprocal psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/psi"), Optional.of("K93"), Optional.of("reciprocalPascalOrPascalToThePowerMinusOne"), Optional.of("1.450377 \u00D7 10\u207B\u2074 Pa\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "reciprocalRadian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalRadian" ), "reciprocalRadian", Set.of(new LangString( "reciprocal radian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/rad"), Optional.of("P97"), Optional.empty(), Optional.of("1/rad"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalSecond" ), "reciprocalSecond", Set.of(new LangString( "reciprocal second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s\u207B\u00B9"), Optional.of("C97"), Optional.empty(), Optional.of("s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.LARMOR_ANGULAR_FREQUENCY, QuantityKinds.ANGULAR_FREQUENCY, QuantityKinds.CIRCULAR_FREQUENCY, QuantityKinds.DAMPING_COEFFICIENT, QuantityKinds.ROTATIONAL_FREQUENCY, QuantityKinds.DECAY_CONSTANT, QuantityKinds.PULSATANCE, QuantityKinds.DISINTEGRATION_CONSTANT, QuantityKinds.DEBYE_ANGULAR_FREQUENCY, QuantityKinds.PHOTON_FLUX)) ) );
UNITS_BY_NAME.put( "reciprocalSecondPerMetreSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalSecondPerMetreSquared" ), "reciprocalSecondPerMetreSquared", Set.of(new LangString( "reciprocal second per metre squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s\u207B\u00B9/m\u00B2"), Optional.of("C99"), Optional.empty(), Optional.of("s\u207B\u00B9/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_EXITANCE, QuantityKinds.IRRADIANCE)) ) );
UNITS_BY_NAME.put( "reciprocalSecondPerSteradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalSecondPerSteradian" ), "reciprocalSecondPerSteradian", Set.of(new LangString( "reciprocal second per steradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s\u207B\u00B9/sr"), Optional.of("D1"), Optional.empty(), Optional.of("s\u207B\u00B9/sr"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_INTENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalSecondPerSteradianMetreSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalSecondPerSteradianMetreSquared" ), "reciprocalSecondPerSteradianMetreSquared", Set.of(new LangString( "reciprocal second per steradian metre squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s\u207B\u00B9/(sr\u00B7m\u00B2)"), Optional.of("D2"), Optional.empty(), Optional.of("s\u207B\u00B9/(sr \u00D7 m\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_RADIANCE, QuantityKinds.PHOTON_LUMINANCE)) ) );
UNITS_BY_NAME.put( "reciprocalSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalSquareInch" ), "reciprocalSquareInch", Set.of(new LangString( "reciprocal square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/in\u00B2"), Optional.of("P78"), Optional.of("reciprocalSquareMetre"), Optional.of("1.550003 \u00D7 10\u00B3 m\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PARTICLE_FLUENCE)) ) );
UNITS_BY_NAME.put( "reciprocalSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalSquareMetre" ), "reciprocalSquareMetre", Set.of(new LangString( "reciprocal square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u207B\u00B2"), Optional.of("C93"), Optional.empty(), Optional.of("m\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_EXPOSURE, QuantityKinds.PARTICLE_FLUENCE)) ) );
UNITS_BY_NAME.put( "reciprocalVolt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalVolt" ), "reciprocalVolt", Set.of(new LangString( "reciprocal volt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/V"), Optional.of("P96"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B3 \u00D7 A"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalVoltAmpereReciprocalSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalVoltAmpereReciprocalSecond" ), "reciprocalVoltAmpereReciprocalSecond", Set.of(new LangString( "reciprocal volt - ampere reciprocal second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/(V\u00B7A\u00B7s)"), Optional.of("M30"), Optional.empty(), Optional.of("(V \u00D7 A \u00D7 s)\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalWeek", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalWeek" ), "reciprocalWeek", Set.of(new LangString( "reciprocal week", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/wk"), Optional.of("H85"), Optional.of("reciprocalSecond"), Optional.of("1.647989452868 \u00D7 10\u207B\u2076 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#reciprocalYear" ), "reciprocalYear", Set.of(new LangString( "reciprocal year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/y"), Optional.of("H09"), Optional.of("reciprocalSecond"), Optional.of("3.16881 \u00D7 10\u207B\u2078 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "rem", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#rem" ), "rem", Set.of(new LangString( "rem", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rem"), Optional.of("D91"), Optional.of("sievert"), Optional.of("10\u207B\u00B2 Sv"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "remPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#remPerSecond" ), "remPerSecond", Set.of(new LangString( "rem per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rem/s"), Optional.of("P69"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u00B2 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "revenueTonMile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#revenueTonMile" ), "revenueTonMile", Set.of(new LangString( "revenue ton mile", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("RT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "revolution", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#revolution" ), "revolution", Set.of(new LangString( "revolution", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rev"), Optional.of("M44"), Optional.of("radian"), Optional.of("6.283185 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "revolutionPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#revolutionPerMinute" ), "revolutionPerMinute", Set.of(new LangString( "revolution per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("r/min"), Optional.of("M46"), Optional.of("radianPerSecond"), Optional.of("0.1047198 rad/s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_VELOCITY)) ) );
UNITS_BY_NAME.put( "revolutionsPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#revolutionsPerMinute" ), "revolutionsPerMinute", Set.of(new LangString( "revolutions per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("r/min"), Optional.of("RPM"), Optional.empty(), Optional.of("1.67 \u00D7 10\u207B\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "revolutionsPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#revolutionsPerSecond" ), "revolutionsPerSecond", Set.of(new LangString( "revolutions per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("r/s"), Optional.of("RPS"), Optional.empty(), Optional.of("1/s"), new HashSet<>(Arrays.asList(QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "rhe", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#rhe" ), "rhe", Set.of(new LangString( "rhe", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rhe"), Optional.of("P88"), Optional.empty(), Optional.of("10 m \u00D7 kg\u207B\u00B9 \u00D7 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "rodUnitOfDistance", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#rodUnitOfDistance" ), "rodUnitOfDistance", Set.of(new LangString( "rod [unit of distance]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("rd (US)"), Optional.of("F49"), Optional.of("metre"), Optional.of("5.029210 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "roentgen", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#roentgen" ), "roentgen", Set.of(new LangString( "roentgen", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("R"), Optional.of("2C"), Optional.of("coulombPerKilogram"), Optional.of("2.58 \u00D7 10\u207B\u2074 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "roentgenPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#roentgenPerSecond" ), "roentgenPerSecond", Set.of(new LangString( "roentgen per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("R/s"), Optional.of("D6"), Optional.of("coulombPerKilogramSecond"), Optional.of("2.58 \u00D7 10\u207B\u2074 C/(kg \u00D7 s)"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE_RATE)) ) );
UNITS_BY_NAME.put( "room", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#room" ), "room", Set.of(new LangString( "room", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("ROM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "round", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#round" ), "round", Set.of(new LangString( "round", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("D65"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "runFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#runFoot" ), "runFoot", Set.of(new LangString( "run foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E52"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "runningOrOperatingHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#runningOrOperatingHour" ), "runningOrOperatingHour", Set.of(new LangString( "running or operating hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("RH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "score", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#score" ), "score", Set.of(new LangString( "score", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SCO"), Optional.empty(), Optional.of("20"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "scruple", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#scruple" ), "scruple", Set.of(new LangString( "scruple", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SCR"), Optional.of("gram"), Optional.of("1.295982 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "secondPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#secondPerCubicMetre" ), "secondPerCubicMetre", Set.of(new LangString( "second per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s/m\u00B3"), Optional.of("D93"), Optional.empty(), Optional.of("s/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_DOSE)) ) );
UNITS_BY_NAME.put( "secondPerCubicMetreRadian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#secondPerCubicMetreRadian" ), "secondPerCubicMetreRadian", Set.of(new LangString( "second per cubic metre radian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s/(rad\u00B7m\u00B3)"), Optional.of("D94"), Optional.empty(), Optional.of("s/(rad \u00D7 m\u00B3)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CONCENTRATION_OF_VIBRATIONAL_MODES_IN_TERMS_OF_ANGULAR_FREQUENCY)) ) );
UNITS_BY_NAME.put( "secondPerKilogramm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#secondPerKilogramm" ), "secondPerKilogramm", Set.of(new LangString( "second per kilogramm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s/kg"), Optional.of("Q20"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "secondPerRadianCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#secondPerRadianCubicMetre" ), "secondPerRadianCubicMetre", Set.of(new LangString( "second per radian cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/(Hz\u00B7rad\u00B7m\u00B3)"), Optional.of("Q22"), Optional.empty(), Optional.of("m\u207B\u00B3 \u00D7 s \u00D7 rad\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "secondUnitOfAngle", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#secondUnitOfAngle" ), "secondUnitOfAngle", Set.of(new LangString( "second [unit of angle]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\""), Optional.of("D62"), Optional.of("radian"), Optional.of("4.848137 \u00D7 10\u207B\u2076 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "secondUnitOfTime", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#secondUnitOfTime" ), "secondUnitOfTime", Set.of(new LangString( "second [unit of time]", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("s"), Optional.of("SEC"), Optional.empty(), Optional.of("s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME, QuantityKinds.HALF_LIFE, QuantityKinds.REVERBERATION_TIME, QuantityKinds.RELAXATION_TIME, QuantityKinds.TIME_CONSTANT, QuantityKinds.PERIOD, QuantityKinds.REACTOR_TIME_CONSTANT, QuantityKinds.CARRIER_LIFE_TIME, QuantityKinds.PERIODIC_TIME, QuantityKinds.MEAN_LIFE)) ) );
UNITS_BY_NAME.put( "segment", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#segment" ), "segment", Set.of(new LangString( "segment", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "serviceUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#serviceUnit" ), "serviceUnit", Set.of(new LangString( "service unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E48"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "set", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#set" ), "set", Set.of(new LangString( "set", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SET"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shake", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#shake" ), "shake", Set.of(new LangString( "shake", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("shake"), Optional.of("M56"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2078 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "shannon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#shannon" ), "shannon", Set.of(new LangString( "shannon", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Sh"), Optional.of("Q14"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shannonPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#shannonPerSecond" ), "shannonPerSecond", Set.of(new LangString( "shannon per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Sh/s"), Optional.of("Q17"), Optional.empty(), Optional.of("Sh/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shares", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#shares" ), "shares", Set.of(new LangString( "shares", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shipment", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#shipment" ), "shipment", Set.of(new LangString( "shipment", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SX"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#shot" ), "shot", Set.of(new LangString( "shot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("14"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "siderealYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#siderealYear" ), "siderealYear", Set.of(new LangString( "sidereal year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("y (sidereal)"), Optional.of("L96"), Optional.of("secondUnitOfTime"), Optional.of("3.155815 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "siemens", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#siemens" ), "siemens", Set.of(new LangString( "siemens", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("S"), Optional.of("SIE"), Optional.empty(), Optional.of("A/V"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.ADMITTANCE)) ) );
UNITS_BY_NAME.put( "siemensPerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#siemensPerCentimetre" ), "siemensPerCentimetre", Set.of(new LangString( "siemens per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("S/cm"), Optional.of("H43"), Optional.of("siemensPerMetre"), Optional.of("10\u00B2 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "siemensPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#siemensPerMetre" ), "siemensPerMetre", Set.of(new LangString( "siemens per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("S/m"), Optional.of("D10"), Optional.empty(), Optional.of("S/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTROLYTIC_CONDUCTIVITY, QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "siemensSquareMetrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#siemensSquareMetrePerMole" ), "siemensSquareMetrePerMole", Set.of(new LangString( "siemens square metre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("S\u00B7m\u00B2/mol"), Optional.of("D12"), Optional.empty(), Optional.of("S \u00D7 m\u00B2/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "sievert", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#sievert" ), "sievert", Set.of(new LangString( "sievert", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Sv"), Optional.of("D13"), Optional.empty(), Optional.of("m\u00B2/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "sievertPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#sievertPerHour" ), "sievertPerHour", Set.of(new LangString( "sievert per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Sv/h"), Optional.of("P70"), Optional.of("sievertPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "sievertPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#sievertPerMinute" ), "sievertPerMinute", Set.of(new LangString( "sievert per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Sv/min"), Optional.of("P74"), Optional.of("sievertPerSecond"), Optional.of("0.016666 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "sievertPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#sievertPerSecond" ), "sievertPerSecond", Set.of(new LangString( "sievert per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Sv/s"), Optional.of("P65"), Optional.empty(), Optional.of("Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "sitas", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#sitas" ), "sitas", Set.of(new LangString( "sitas", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("56"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "skein", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#skein" ), "skein", Set.of(new LangString( "skein", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "slug", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slug" ), "slug", Set.of(new LangString( "slug", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug"), Optional.of("F13"), Optional.of("kilogram"), Optional.of("1.459390 \u00D7 10\u00B9 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "slugPerCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slugPerCubicFoot" ), "slugPerCubicFoot", Set.of(new LangString( "slug per cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug/ft\u00B3"), Optional.of("L65"), Optional.of("kilogramPerCubicMetre"), Optional.of("5.153788 \u00D7 10\u00B2 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "slugPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slugPerDay" ), "slugPerDay", Set.of(new LangString( "slug per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug/d"), Optional.of("L63"), Optional.of("kilogramPerSecond"), Optional.of("1.689109 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "slugPerFootSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slugPerFootSecond" ), "slugPerFootSecond", Set.of(new LangString( "slug per foot second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug/(ft\u00B7s)"), Optional.of("L64"), Optional.of("pascalSecond"), Optional.of("47.88026 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "slugPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slugPerHour" ), "slugPerHour", Set.of(new LangString( "slug per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug/h"), Optional.of("L66"), Optional.of("kilogramPerSecond"), Optional.of("4.053861 \u00D7 10\u207B\u00B3 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
}

private static void init15() {
UNITS_BY_NAME.put( "slugPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slugPerMinute" ), "slugPerMinute", Set.of(new LangString( "slug per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug/min"), Optional.of("L67"), Optional.of("kilogramPerSecond"), Optional.of("0.2432317 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "slugPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#slugPerSecond" ), "slugPerSecond", Set.of(new LangString( "slug per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("slug/s"), Optional.of("L68"), Optional.of("kilogramPerSecond"), Optional.of("14.59390 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "sone", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#sone" ), "sone", Set.of(new LangString( "sone", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("D15"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.LOUDNESS)) ) );
UNITS_BY_NAME.put( "square", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#square" ), "square", Set.of(new LangString( "square", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "squareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareCentimetre" ), "squareCentimetre", Set.of(new LangString( "square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B2"), Optional.of("CMK"), Optional.of("squareMetre"), Optional.of("10\u207B\u2074 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerErg", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareCentimetrePerErg" ), "squareCentimetrePerErg", Set.of(new LangString( "square centimetre per erg", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B2/erg"), Optional.of("D16"), Optional.of("squareMetrePerJoule"), Optional.of("10\u00B3 m\u00B2/J"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerGram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareCentimetrePerGram" ), "squareCentimetrePerGram", Set.of(new LangString( "square centimetre per gram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B2/g"), Optional.of("H15"), Optional.empty(), Optional.of("10\u207B\u00B9 kg\u207B\u00B9 \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareCentimetrePerSecond" ), "squareCentimetrePerSecond", Set.of(new LangString( "square centimetre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B2/s"), Optional.of("M81"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2074 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerSteradianErg", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareCentimetrePerSteradianErg" ), "squareCentimetrePerSteradianErg", Set.of(new LangString( "square centimetre per steradian erg", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("cm\u00B2/(sr\u00B7erg)"), Optional.of("D17"), Optional.of("squareMetrePerSteradianJoule"), Optional.of("10\u00B3 m\u00B2/(sr \u00D7 J)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareDecametre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareDecametre" ), "squareDecametre", Set.of(new LangString( "square decametre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dam\u00B2"), Optional.of("H16"), Optional.of("squareMetre"), Optional.of("10\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareDecimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareDecimetre" ), "squareDecimetre", Set.of(new LangString( "square decimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("dm\u00B2"), Optional.of("DMK"), Optional.of("squareMetre"), Optional.of("10\u207B\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareFoot" ), "squareFoot", Set.of(new LangString( "square foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B2"), Optional.of("FTK"), Optional.of("squareMetre"), Optional.of("9.290304 \u00D7 10\u207B\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareFootPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareFootPerHour" ), "squareFootPerHour", Set.of(new LangString( "square foot per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B2/h"), Optional.of("M79"), Optional.of("squareMetrePerSecond"), Optional.of("2.58064 \u00D7 10\u207B\u2075 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareFootPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareFootPerSecond" ), "squareFootPerSecond", Set.of(new LangString( "square foot per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ft\u00B2/s"), Optional.of("S3"), Optional.of("squareMetrePerSecond"), Optional.of("0.09290304 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY, QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareHectometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareHectometre" ), "squareHectometre", Set.of(new LangString( "square hectometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("hm\u00B2"), Optional.of("H18"), Optional.of("squareMetre"), Optional.of("10\u2074 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareInch" ), "squareInch", Set.of(new LangString( "square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B2"), Optional.of("INK"), Optional.of("squareMetre"), Optional.of("6.4516 \u00D7 10\u207B\u2074 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareInchPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareInchPerSecond" ), "squareInchPerSecond", Set.of(new LangString( "square inch per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("in\u00B2/s"), Optional.of("G08"), Optional.empty(), Optional.of("6.4516 \u00D7 10\u207B\u2074 m\u00B2 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareKilometre" ), "squareKilometre", Set.of(new LangString( "square kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km\u00B2"), Optional.of("KMK"), Optional.of("squareMetre"), Optional.of("10\u2076 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetre" ), "squareMetre", Set.of(new LangString( "square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2"), Optional.of("MTK"), Optional.empty(), Optional.of("m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENT_ABSORPTION_AREA_OF_A_SURFACE_OR_OBJECT, QuantityKinds.SLOWING_DOWN_AREA, QuantityKinds.CROSS_SECTION, QuantityKinds.DIFFUSION_AREA, QuantityKinds.MIGRATION_AREA, QuantityKinds.ATOMIC_ATTENUATION_COEFFICIENT, QuantityKinds.AREA, QuantityKinds.NUCLEAR_QUADRUPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "squareMetreHourDegreeCelsiusPerKilocalorieInternationalTable", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetreHourDegreeCelsiusPerKilocalorieInternationalTable" ), "squareMetreHourDegreeCelsiusPerKilocalorieInternationalTable", Set.of(new LangString( "square metre hour degree Celsius per kilocalorie (international table)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2\u00B7h\u00B7\u00B0C/kcal"), Optional.of("L14"), Optional.empty(), Optional.of("0.8598452 m\u00B2 \u00D7 s \u00D7 K/J"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "squareMetreKelvinPerWatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetreKelvinPerWatt" ), "squareMetreKelvinPerWatt", Set.of(new LangString( "square metre kelvin per watt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2\u00B7K/W"), Optional.of("D19"), Optional.empty(), Optional.of("m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "squareMetrePerJoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerJoule" ), "squareMetrePerJoule", Set.of(new LangString( "square metre per joule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/J"), Optional.of("D20"), Optional.empty(), Optional.of("m\u00B2/J"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareMetrePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerKilogram" ), "squareMetrePerKilogram", Set.of(new LangString( "square metre per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/kg"), Optional.of("D21"), Optional.empty(), Optional.of("m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_ENERGY_TRANSFER_COEFFICIENT, QuantityKinds.MASS_ATTENUATION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "squareMetrePerLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerLitre" ), "squareMetrePerLitre", Set.of(new LangString( "square metre per litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/l"), Optional.of("E31"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "squareMetrePerMole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerMole" ), "squareMetrePerMole", Set.of(new LangString( "square metre per mole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/mol"), Optional.of("D22"), Optional.empty(), Optional.of("m\u00B2/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_ATTENUATION_COEFFICIENT, QuantityKinds.MOLAR_ABSORPTION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "squareMetrePerNewton", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerNewton" ), "squareMetrePerNewton", Set.of(new LangString( "square metre per newton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/N"), Optional.of("H59"), Optional.empty(), Optional.of("m \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerSecond" ), "squareMetrePerSecond", Set.of(new LangString( "square metre per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/s"), Optional.of("S4"), Optional.empty(), Optional.of("m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.DIFFUSION_COEFFICIENT, QuantityKinds.KINEMATIC_VISCOSITY, QuantityKinds.DIFFUSION_COEFFICIENT_FOR_NEUTRON_NUMBER_DENSITY, QuantityKinds.THERMAL_DIFFUSIVITY, QuantityKinds.THERMAL_DIFFUSION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerSecondBar" ), "squareMetrePerSecondBar", Set.of(new LangString( "square metre per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/(s\u00B7bar)"), Optional.of("G41"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerSecondKelvin" ), "squareMetrePerSecondKelvin", Set.of(new LangString( "square metre per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/(s\u00B7K)"), Optional.of("G09"), Optional.empty(), Optional.of("m\u00B2 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecondPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerSecondPascal" ), "squareMetrePerSecondPascal", Set.of(new LangString( "square metre per second pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(m\u00B2/s)/Pa"), Optional.of("M82"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSteradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerSteradian" ), "squareMetrePerSteradian", Set.of(new LangString( "square metre per steradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/sr"), Optional.of("D24"), Optional.empty(), Optional.of("m\u00B2/sr"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareMetrePerSteradianJoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerSteradianJoule" ), "squareMetrePerSteradianJoule", Set.of(new LangString( "square metre per steradian joule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/(sr\u00B7J)"), Optional.of("D25"), Optional.empty(), Optional.of("m\u00B2/(sr \u00D7 J)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareMetrePerVoltSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMetrePerVoltSecond" ), "squareMetrePerVoltSecond", Set.of(new LangString( "square metre per volt second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("m\u00B2/(V\u00B7s)"), Optional.of("D26"), Optional.empty(), Optional.of("m\u00B2/(V \u00D7 s)"), new HashSet<>(Arrays.asList(QuantityKinds.MOBILITY)) ) );
UNITS_BY_NAME.put( "squareMicrometreSquareMicron", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMicrometreSquareMicron" ), "squareMicrometreSquareMicron", Set.of(new LangString( "square micrometre (square micron)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("\u00B5m\u00B2"), Optional.of("H30"), Optional.of("squareMetre"), Optional.of("10\u207B\u00B9\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMileBasedOnUsSurveyFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMileBasedOnUsSurveyFoot" ), "squareMileBasedOnUsSurveyFoot", Set.of(new LangString( "square mile (based on U.S. survey foot)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi\u00B2 (US survey)"), Optional.of("M48"), Optional.of("squareMetre"), Optional.of("2.589998 \u00D7 10\u2076 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMileStatuteMile", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMileStatuteMile" ), "squareMileStatuteMile", Set.of(new LangString( "square mile (statute mile)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mi\u00B2"), Optional.of("MIK"), Optional.of("squareKilometre"), Optional.of("2.589988 km\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareMillimetre" ), "squareMillimetre", Set.of(new LangString( "square millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mm\u00B2"), Optional.of("MMK"), Optional.of("squareMetre"), Optional.of("10\u207B\u2076 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareRoofing", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareRoofing" ), "squareRoofing", Set.of(new LangString( "square, roofing", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SQR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "squareYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#squareYard" ), "squareYard", Set.of(new LangString( "square yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd\u00B2"), Optional.of("YDK"), Optional.of("squareMetre"), Optional.of("8.361274 \u00D7 10\u207B\u00B9 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "standard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#standard" ), "standard", Set.of(new LangString( "standard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("std"), Optional.of("WSD"), Optional.of("cubicMetre"), Optional.of("4.672 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "standardAccelerationOfFreeFall", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#standardAccelerationOfFreeFall" ), "standardAccelerationOfFreeFall", Set.of(new LangString( "standard acceleration of free fall", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gn"), Optional.of("K40"), Optional.of("metrePerSecondSquared"), Optional.of("9.80665 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "standardAtmosphere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#standardAtmosphere" ), "standardAtmosphere", Set.of(new LangString( "standard atmosphere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("atm"), Optional.of("ATM"), Optional.of("pascal"), Optional.of("101325 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "standardAtmospherePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#standardAtmospherePerMetre" ), "standardAtmospherePerMetre", Set.of(new LangString( "standard atmosphere per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Atm/m"), Optional.of("P83"), Optional.empty(), Optional.of("1.01325 \u00D7 10\u2075 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "standardKilolitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#standardKilolitre" ), "standardKilolitre", Set.of(new LangString( "standard kilolitre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DMO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "standardLitre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#standardLitre" ), "standardLitre", Set.of(new LangString( "standard litre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("STL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "steradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#steradian" ), "steradian", Set.of(new LangString( "steradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("sr"), Optional.of("D27"), Optional.empty(), Optional.of("sr"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "stere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stere" ), "stere", Set.of(new LangString( "stere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("st"), Optional.of("G26"), Optional.of("cubicMetre"), Optional.of("m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "stick", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stick" ), "stick", Set.of(new LangString( "stick", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("STC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "stickCigarette", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stickCigarette" ), "stickCigarette", Set.of(new LangString( "stick, cigarette", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("STK"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "stickMilitary", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stickMilitary" ), "stickMilitary", Set.of(new LangString( "stick, military", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("15"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "stilb", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stilb" ), "stilb", Set.of(new LangString( "stilb", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("sb"), Optional.of("P31"), Optional.of("candelaPerSquareMetre"), Optional.of("10\u2074 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "stokes", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stokes" ), "stokes", Set.of(new LangString( "stokes", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("St"), Optional.of("91"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2074 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stokesPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stokesPerBar" ), "stokesPerBar", Set.of(new LangString( "stokes per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("St/bar"), Optional.of("G46"), Optional.empty(), Optional.of("10\u207B\u2079 kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stokesPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stokesPerKelvin" ), "stokesPerKelvin", Set.of(new LangString( "stokes per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("St/K"), Optional.of("G10"), Optional.empty(), Optional.of("10\u207B\u2074 m\u00B2 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stokesPerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stokesPerPascal" ), "stokesPerPascal", Set.of(new LangString( "stokes per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("St/Pa"), Optional.of("M80"), Optional.empty(), Optional.of("10\u207B\u2074 kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stoneUk", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#stoneUk" ), "stoneUk", Set.of(new LangString( "stone (UK)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("st"), Optional.of("STI"), Optional.of("kilogram"), Optional.of("6.350293 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "strand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#strand" ), "strand", Set.of(new LangString( "strand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E30"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "straw", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#straw" ), "straw", Set.of(new LangString( "straw", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("STW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "strip", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#strip" ), "strip", Set.of(new LangString( "strip", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "syringe", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#syringe" ), "syringe", Set.of(new LangString( "syringe", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("SYR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tablespoonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tablespoonUs" ), "tablespoonUs", Set.of(new LangString( "tablespoon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("tablespoon (US)"), Optional.of("G24"), Optional.of("cubicMetre"), Optional.of("1.478676 \u00D7 10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tablet", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tablet" ), "tablet", Set.of(new LangString( "tablet", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("U2"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "teaspoonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#teaspoonUs" ), "teaspoonUs", Set.of(new LangString( "teaspoon (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("teaspoon (US)"), Optional.of("G25"), Optional.of("cubicMetre"), Optional.of("4.928922 \u00D7 10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tebibitPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tebibitPerCubicMetre" ), "tebibitPerCubicMetre", Set.of(new LangString( "tebibit per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Tibit/m\u00B3"), Optional.of("E86"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tebibitPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tebibitPerMetre" ), "tebibitPerMetre", Set.of(new LangString( "tebibit per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Tibit/m"), Optional.of("E85"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tebibitPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tebibitPerSquareMetre" ), "tebibitPerSquareMetre", Set.of(new LangString( "tebibit per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Tibit/m\u00B2"), Optional.of("E87"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tebibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tebibyte" ), "tebibyte", Set.of(new LangString( "Tebibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("TiB"), Optional.of("E61"), Optional.of("byte"), Optional.of("2\u2074\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "technicalAtmospherePerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#technicalAtmospherePerMetre" ), "technicalAtmospherePerMetre", Set.of(new LangString( "technical atmosphere per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("at/m"), Optional.of("P84"), Optional.empty(), Optional.of("9.80665 \u00D7 10\u2074 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "telecommunicationLineInService", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#telecommunicationLineInService" ), "telecommunicationLineInService", Set.of(new LangString( "telecommunication line in service", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("T0"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "telecommunicationLineInServiceAverage", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#telecommunicationLineInServiceAverage" ), "telecommunicationLineInServiceAverage", Set.of(new LangString( "telecommunication line in service average", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("UB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "telecommunicationPort", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#telecommunicationPort" ), "telecommunicationPort", Set.of(new LangString( "telecommunication port", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("UC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tenDay" ), "tenDay", Set.of(new LangString( "ten day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("DAD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenPack", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tenPack" ), "tenPack", Set.of(new LangString( "ten pack", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenPair", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tenPair" ), "tenPair", Set.of(new LangString( "ten pair", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TPR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenSet", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tenSet" ), "tenSet", Set.of(new LangString( "ten set", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TST"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenThousandSticks", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tenThousandSticks" ), "tenThousandSticks", Set.of(new LangString( "ten thousand sticks", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TTS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "terabit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terabit" ), "terabit", Set.of(new LangString( "terabit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Tbit"), Optional.of("E83"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "terabitPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terabitPerSecond" ), "terabitPerSecond", Set.of(new LangString( "terabit per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Tbit/s"), Optional.of("E84"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "terabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terabyte" ), "terabyte", Set.of(new LangString( "Terabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("TB"), Optional.of("E35"), Optional.of("byte"), Optional.of("10\u00B9\u00B2 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "terahertz", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terahertz" ), "terahertz", Set.of(new LangString( "terahertz", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("THz"), Optional.of("D29"), Optional.of("hertz"), Optional.of("10\u00B9\u00B2 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "terajoule", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terajoule" ), "terajoule", Set.of(new LangString( "terajoule", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("TJ"), Optional.of("D30"), Optional.of("joule"), Optional.of("10\u00B9\u00B2 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "teraohm", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#teraohm" ), "teraohm", Set.of(new LangString( "teraohm", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("T\u2126"), Optional.of("H44"), Optional.of("ohm"), Optional.of("10\u00B9\u00B2 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.REACTANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.IMPEDANCE)) ) );
UNITS_BY_NAME.put( "terawatt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terawatt" ), "terawatt", Set.of(new LangString( "terawatt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("TW"), Optional.of("D31"), Optional.of("watt"), Optional.of("10\u00B9\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "terawattHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#terawattHour" ), "terawattHour", Set.of(new LangString( "terawatt hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("TW\u00B7h"), Optional.of("D32"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u00B9\u2075 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "tesla", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tesla" ), "tesla", Set.of(new LangString( "tesla", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("T"), Optional.of("D33"), Optional.empty(), Optional.of("T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.LOWER_CRITICAL_MAGNETIC_FLUX_DENSITY, QuantityKinds.UPPER_CRITICAL_MAGNETIC_FLUX_DENSITY, QuantityKinds.THERMODYNAMIC_CRITICAL_MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "test", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#test" ), "test", Set.of(new LangString( "test", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E53"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "teu", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#teu" ), "teu", Set.of(new LangString( "TEU", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E22"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tex", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tex" ), "tex", Set.of(new LangString( "tex", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("tex (g/km)"), Optional.of("D34"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u2076 kg/m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "theoreticalPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#theoreticalPound" ), "theoreticalPound", Set.of(new LangString( "theoretical pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("24"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "theoreticalTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#theoreticalTon" ), "theoreticalTon", Set.of(new LangString( "theoretical ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("27"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thermEc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thermEc" ), "thermEc", Set.of(new LangString( "therm (EC)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("thm (EC)"), Optional.of("N71"), Optional.of("joule"), Optional.of("1.05506 \u00D7 10\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "thermUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thermUs" ), "thermUs", Set.of(new LangString( "therm (U.S.)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("thm (US)"), Optional.of("N72"), Optional.of("joule"), Optional.of("1.054804 \u00D7 10\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "thirtyDayMonth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thirtyDayMonth" ), "thirtyDayMonth", Set.of(new LangString( "30-day month", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("mo (30 days)"), Optional.of("M36"), Optional.of("secondUnitOfTime"), Optional.of("2.592000 \u00D7 10\u2076 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousand", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousand" ), "thousand", Set.of(new LangString( "thousand", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MIL"), Optional.empty(), Optional.of("10\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandBoardFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandBoardFoot" ), "thousandBoardFoot", Set.of(new LangString( "thousand board foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MBF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandCubicFoot", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandCubicFoot" ), "thousandCubicFoot", Set.of(new LangString( "thousand cubic foot", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("kft\u00B3"), Optional.of("FC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandCubicMetre" ), "thousandCubicMetre", Set.of(new LangString( "thousand cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("R9"), Optional.of("cubicMetre"), Optional.of("10\u00B3m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandCubicMetrePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandCubicMetrePerDay" ), "thousandCubicMetrePerDay", Set.of(new LangString( "thousand cubic metre per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("km\u00B3/d"), Optional.of("TQD"), Optional.of("cubicMetrePerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u00B2 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandPiece", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandPiece" ), "thousandPiece", Set.of(new LangString( "thousand piece", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("T3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandSquareInch" ), "thousandSquareInch", Set.of(new LangString( "thousand square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}

private static void init16() {
UNITS_BY_NAME.put( "thousandStandardBrickEquivalent", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#thousandStandardBrickEquivalent" ), "thousandStandardBrickEquivalent", Set.of(new LangString( "thousand standard brick equivalent", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("MBE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tonAssay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonAssay" ), "tonAssay", Set.of(new LangString( "ton, assay", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("M85"), Optional.of("kilogram"), Optional.of("2.916667 \u00D7 10\u207B\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonForceUsShort", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonForceUsShort" ), "tonForceUsShort", Set.of(new LangString( "ton-force (US short)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton.sh-force"), Optional.of("L94"), Optional.of("newton"), Optional.of("8.896443 \u00D7 10\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "tonLongPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonLongPerDay" ), "tonLongPerDay", Set.of(new LangString( "ton long per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (UK)/d"), Optional.of("L85"), Optional.of("kilogramPerSecond"), Optional.of("1.175980 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonRegister", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonRegister" ), "tonRegister", Set.of(new LangString( "ton, register", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("RT"), Optional.of("M70"), Optional.of("cubicMetre"), Optional.of("2.831685 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tonShortPerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonShortPerDay" ), "tonShortPerDay", Set.of(new LangString( "ton short per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (US)/d"), Optional.of("L88"), Optional.of("kilogramPerSecond"), Optional.of("1.049982 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonShortPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonShortPerDegreeFahrenheit" ), "tonShortPerDegreeFahrenheit", Set.of(new LangString( "ton short per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (US)/\u00B0F"), Optional.of("L87"), Optional.of("kilogramPerKelvin"), Optional.of("1.632932 \u00D7 10\u00B3 kg/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonShortPerHourDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonShortPerHourDegreeFahrenheit" ), "tonShortPerHourDegreeFahrenheit", Set.of(new LangString( "ton short per hour degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (US)/(h\u00B7\u00B0F)"), Optional.of("L89"), Optional.empty(), Optional.of("0.4535922 kg/s \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonShortPerHourPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonShortPerHourPsi" ), "tonShortPerHourPsi", Set.of(new LangString( "ton short per hour psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(ton (US)/h)/psi"), Optional.of("L90"), Optional.of("kilogramPerSecondPascal"), Optional.of("3.654889 \u00D7 10\u207B\u2075 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonShortPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonShortPerPsi" ), "tonShortPerPsi", Set.of(new LangString( "ton short per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (US)/psi"), Optional.of("L91"), Optional.empty(), Optional.of("0.131576"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "tonUkLongPerCubicYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUkLongPerCubicYard" ), "tonUkLongPerCubicYard", Set.of(new LangString( "ton (UK long) per cubic yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton.l/yd\u00B3 (UK)"), Optional.of("L92"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.328939 \u00D7 10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "tonUkOrLongTonUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUkOrLongTonUs" ), "tonUkOrLongTonUs", Set.of(new LangString( "ton (UK) or long ton (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (UK)"), Optional.of("LTN"), Optional.of("kilogram"), Optional.of("1.016047 \u00D7 10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonUkShipping", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUkShipping" ), "tonUkShipping", Set.of(new LangString( "ton (UK shipping)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("British shipping ton"), Optional.of("L84"), Optional.of("cubicMetre"), Optional.of("1.1893 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tonUsOrShortTonUkorus", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUsOrShortTonUkorus" ), "tonUsOrShortTonUkorus", Set.of(new LangString( "ton (US) or short ton (UK/US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (US)"), Optional.of("STN"), Optional.of("kilogram"), Optional.of("0.9071847 \u00D7 10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonUsPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUsPerHour" ), "tonUsPerHour", Set.of(new LangString( "ton (US) per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton (US) /h"), Optional.of("4W"), Optional.of("kilogramPerSecond"), Optional.of("2.519958 \u00D7 10\u207B\u00B9 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonUsShipping", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUsShipping" ), "tonUsShipping", Set.of(new LangString( "ton (US shipping)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(US) shipping ton"), Optional.of("L86"), Optional.of("cubicMetre"), Optional.of("1.1326 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tonUsShortPerCubicYard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonUsShortPerCubicYard" ), "tonUsShortPerCubicYard", Set.of(new LangString( "ton (US short) per cubic yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ton.s/yd\u00B3 (US)"), Optional.of("L93"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.186553 \u00D7 10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "tonneKilometre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonneKilometre" ), "tonneKilometre", Set.of(new LangString( "tonne kilometre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t\u00B7km"), Optional.of("TKM"), Optional.of("kilogramMetre"), Optional.of("10\u2076 kg \u00D7 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tonneMetricTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonneMetricTon" ), "tonneMetricTon", Set.of(new LangString( "tonne (metric ton)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t"), Optional.of("TNE"), Optional.of("kilogram"), Optional.of("10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonnePerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerBar" ), "tonnePerBar", Set.of(new LangString( "tonne per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/bar"), Optional.of("L70"), Optional.of("kilogramPerPascal"), Optional.of("10\u207B\u00B2 kg/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "tonnePerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerCubicMetre" ), "tonnePerCubicMetre", Set.of(new LangString( "tonne per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/m\u00B3"), Optional.of("D41"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "tonnePerCubicMetreBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerCubicMetreBar" ), "tonnePerCubicMetreBar", Set.of(new LangString( "tonne per cubic metre bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/m\u00B3)/bar"), Optional.of("L77"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("10\u207B\u00B2 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "tonnePerCubicMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerCubicMetreKelvin" ), "tonnePerCubicMetreKelvin", Set.of(new LangString( "tonne per cubic metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/m\u00B3)/K"), Optional.of("L76"), Optional.empty(), Optional.of("10\u00B3 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "tonnePerDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerDay" ), "tonnePerDay", Set.of(new LangString( "tonne per day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/d"), Optional.of("L71"), Optional.of("kilogramPerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerDayBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerDayBar" ), "tonnePerDayBar", Set.of(new LangString( "tonne per day bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/d)/bar"), Optional.of("L73"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.15741 \u00D7 10\u207B\u2077 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerDayKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerDayKelvin" ), "tonnePerDayKelvin", Set.of(new LangString( "tonne per day kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/d)/K"), Optional.of("L72"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B2 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerHour" ), "tonnePerHour", Set.of(new LangString( "tonne per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/h"), Optional.of("E18"), Optional.of("kilogramPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerHourBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerHourBar" ), "tonnePerHourBar", Set.of(new LangString( "tonne per hour bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/h)/bar"), Optional.of("L75"), Optional.of("kilogramPerSecondPascal"), Optional.of("2.77778 \u00D7 10\u207B\u2076 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerHourKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerHourKelvin" ), "tonnePerHourKelvin", Set.of(new LangString( "tonne per hour kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/h)/K"), Optional.of("L74"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerKelvin" ), "tonnePerKelvin", Set.of(new LangString( "tonne per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/K"), Optional.of("L69"), Optional.of("kilogramPerKelvin"), Optional.of("10\u00B3 kg/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonnePerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerMinute" ), "tonnePerMinute", Set.of(new LangString( "tonne per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/min"), Optional.of("L78"), Optional.of("kilogramPerSecond"), Optional.of("16.6667 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerMinuteBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerMinuteBar" ), "tonnePerMinuteBar", Set.of(new LangString( "tonne per minute bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/min)/bar"), Optional.of("L80"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.66667 \u00D7 10\u207B\u2074 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerMinuteKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerMinuteKelvin" ), "tonnePerMinuteKelvin", Set.of(new LangString( "tonne per minute kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/min)/K"), Optional.of("L79"), Optional.empty(), Optional.of("16.6667 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerMonth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerMonth" ), "tonnePerMonth", Set.of(new LangString( "tonne per month", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/mo"), Optional.of("M88"), Optional.of("kilogramPerSecond"), Optional.of("3.80257053768 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerSecond" ), "tonnePerSecond", Set.of(new LangString( "tonne per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/s"), Optional.of("L81"), Optional.of("kilogramPerSecond"), Optional.of("10\u00B3 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerSecondBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerSecondBar" ), "tonnePerSecondBar", Set.of(new LangString( "tonne per second bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/s)/bar"), Optional.of("L83"), Optional.of("kilogramPerSecondPascal"), Optional.of("10\u207B\u00B2 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerSecondKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerSecondKelvin" ), "tonnePerSecondKelvin", Set.of(new LangString( "tonne per second kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("(t/s)/K"), Optional.of("L82"), Optional.empty(), Optional.of("10\u00B3 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tonnePerYear" ), "tonnePerYear", Set.of(new LangString( "tonne per year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("t/y"), Optional.of("M89"), Optional.of("kilogramPerSecond"), Optional.of("3.168808781 \u00D7 10\u207B\u2075 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "torrPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#torrPerMetre" ), "torrPerMetre", Set.of(new LangString( "torr per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Torr/m"), Optional.of("P85"), Optional.empty(), Optional.of("1.333224 \u00D7 10\u00B2 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "totalAcidNumber", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#totalAcidNumber" ), "totalAcidNumber", Set.of(new LangString( "total acid number", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("TAN"), Optional.of("TAN"), Optional.empty(), Optional.of("mg KOH/g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "treatment", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#treatment" ), "treatment", Set.of(new LangString( "treatment", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("U1"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "trillionEur", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#trillionEur" ), "trillionEur", Set.of(new LangString( "trillion (EUR)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("TRL"), Optional.empty(), Optional.of("10\u00B9\u2078"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "trip", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#trip" ), "trip", Set.of(new LangString( "trip", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E54"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tropicalYear", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tropicalYear" ), "tropicalYear", Set.of(new LangString( "tropical year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("y (tropical)"), Optional.of("D42"), Optional.of("secondUnitOfTime"), Optional.of("3.155692 5 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "troyOunceOrApothecaryOunce", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#troyOunceOrApothecaryOunce" ), "troyOunceOrApothecaryOunce", Set.of(new LangString( "troy ounce or apothecary ounce", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("tr oz"), Optional.of("APZ"), Optional.of("kilogram"), Optional.of("3.110348 \u00D7 10\u207B\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "troyPoundUs", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#troyPoundUs" ), "troyPoundUs", Set.of(new LangString( "troy pound (US)", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("LBT"), Optional.of("gram"), Optional.of("373.2417 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "twentyFootContainer", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#twentyFootContainer" ), "twentyFootContainer", Set.of(new LangString( "twenty foot container", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("20"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tyre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#tyre" ), "tyre", Set.of(new LangString( "tyre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E23"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "unifiedAtomicMassUnit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#unifiedAtomicMassUnit" ), "unifiedAtomicMassUnit", Set.of(new LangString( "unified atomic mass unit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("u"), Optional.of("D43"), Optional.of("kilogram"), Optional.of("1.660 538 782 \u00D7 10\u207B\u00B2\u2077 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_OF_ATOM_OF_A_NUCLIDE_X, QuantityKinds.UNIFIED_ATOMIC_MASS_CONSTANT, QuantityKinds.NUCLIDIC_MASS, QuantityKinds.MASS_OF_MOLECULE, QuantityKinds.MASS_DEFECT)) ) );
UNITS_BY_NAME.put( "unitPole", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#unitPole" ), "unitPole", Set.of(new LangString( "unit pole", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("unit pole"), Optional.of("P53"), Optional.of("weber"), Optional.of("1.256637 \u00D7 10\u207B\u2077 Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_QUANTUM)) ) );
UNITS_BY_NAME.put( "usGallonPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#usGallonPerMinute" ), "usGallonPerMinute", Set.of(new LangString( "US gallon per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("gal (US) /min"), Optional.of("G2"), Optional.of("cubicMetrePerSecond"), Optional.of("6.309020 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "use", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#use" ), "use", Set.of(new LangString( "use", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E55"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "var", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#var" ), "var", Set.of(new LangString( "var", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("var"), Optional.of("D44"), Optional.of("voltAmpere"), Optional.of("V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.REACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "volt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#volt" ), "volt", Set.of(new LangString( "volt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V"), Optional.of("VLT"), Optional.empty(), Optional.of("V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.PELTIER_COEFFICIENT_FOR_SUBSTANCES_A_AND_B, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.TENSION, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.THERMOELECTROMOTIVE_FORCE_BETWEEN_SUBSTANCES_A_AND_B, QuantityKinds.VOLTAGE)) ) );
UNITS_BY_NAME.put( "voltAc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltAc" ), "voltAc", Set.of(new LangString( "volt AC", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V"), Optional.of("2G"), Optional.empty(), Optional.of("V"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "voltAmpere", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltAmpere" ), "voltAmpere", Set.of(new LangString( "volt - ampere", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V\u00B7A"), Optional.of("D46"), Optional.of("watt"), Optional.of("W"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "voltAmperePerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltAmperePerKilogram" ), "voltAmperePerKilogram", Set.of(new LangString( "volt - ampere per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V\u00B7A / kg"), Optional.of("VA"), Optional.empty(), Optional.of("V \u00D7 A / kg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "voltDc", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltDc" ), "voltDc", Set.of(new LangString( "volt DC", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V"), Optional.of("2H"), Optional.empty(), Optional.of("V"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "voltPerBar", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerBar" ), "voltPerBar", Set.of(new LangString( "volt per bar", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/bar"), Optional.of("G60"), Optional.empty(), Optional.of("10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerCentimetre" ), "voltPerCentimetre", Set.of(new LangString( "volt per centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/cm"), Optional.of("D47"), Optional.empty(), Optional.of("V/m \u00D7 10\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerInch" ), "voltPerInch", Set.of(new LangString( "volt per inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/in"), Optional.of("H23"), Optional.empty(), Optional.of("3.937007874 \u00D7 10\u00B9 m \u00D7 kg \u00D7 s\u207B\u00B3 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerKelvin" ), "voltPerKelvin", Set.of(new LangString( "volt per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/K"), Optional.of("D48"), Optional.empty(), Optional.of("V/K"), new HashSet<>(Arrays.asList(QuantityKinds.SEEBECK_COEFFICIENT_FOR_SUBSTANCES_A_AND_B)) ) );
UNITS_BY_NAME.put( "voltPerLitreMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerLitreMinute" ), "voltPerLitreMinute", Set.of(new LangString( "volt per litre minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/(l\u00B7min)"), Optional.of("F87"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u00B9 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u2074 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerMetre" ), "voltPerMetre", Set.of(new LangString( "volt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/m"), Optional.of("D50"), Optional.empty(), Optional.of("V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerMicrosecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerMicrosecond" ), "voltPerMicrosecond", Set.of(new LangString( "volt per microsecond", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/\u00B5s"), Optional.of("H24"), Optional.of("voltPerSecond"), Optional.of("10\u2076 V/s"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerMillimetre" ), "voltPerMillimetre", Set.of(new LangString( "volt per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/mm"), Optional.of("D51"), Optional.of("voltPerMetre"), Optional.of("10\u00B3 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerPascal", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerPascal" ), "voltPerPascal", Set.of(new LangString( "volt per pascal", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/Pa"), Optional.of("N98"), Optional.empty(), Optional.of("m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltPerSecond" ), "voltPerSecond", Set.of(new LangString( "volt per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/s"), Optional.of("H46"), Optional.empty(), Optional.of("m\u00B2 \u00D7 kg \u00D7 s\u207B\u2074 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltSecondPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltSecondPerMetre" ), "voltSecondPerMetre", Set.of(new LangString( "volt second per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V\u00B7s/m"), Optional.of("H45"), Optional.empty(), Optional.of("m \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltSquareInchPerPoundForce", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltSquareInchPerPoundForce" ), "voltSquareInchPerPoundForce", Set.of(new LangString( "volt square inch per pound-force", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V/(lbf/in\u00B2)"), Optional.of("H22"), Optional.empty(), Optional.of("1.4503774398 \u00D7 10\u207B\u2074 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltSquaredPerKelvinSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#voltSquaredPerKelvinSquared" ), "voltSquaredPerKelvinSquared", Set.of(new LangString( "volt squared per kelvin squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("V\u00B2/K\u00B2"), Optional.of("D45"), Optional.empty(), Optional.of("V\u00B2/K\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "waterHorsePower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#waterHorsePower" ), "waterHorsePower", Set.of(new LangString( "water horse power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("F80"), Optional.of("watt"), Optional.of("7.46043 \u00D7 10\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "watt", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#watt" ), "watt", Set.of(new LangString( "watt", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W"), Optional.of("WTT"), Optional.empty(), Optional.of("W"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGYFLUX, QuantityKinds.RADIANT_POWER, QuantityKinds.POWER, QuantityKinds.ACTIVE_POWER, QuantityKinds.SOUND_POWER, QuantityKinds.HEAT_FLOW_RATE, QuantityKinds.POWER_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "wattHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattHour" ), "wattHour", Set.of(new LangString( "watt hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W\u00B7h"), Optional.of("WHR"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVE_ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "wattPerCubicMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerCubicMetre" ), "wattPerCubicMetre", Set.of(new LangString( "watt per cubic metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/m\u00B3"), Optional.of("H47"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 kg \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_EXITANCE, QuantityKinds.IRRADIANCE, QuantityKinds.FIRST_RADIATION_CONSTANT, QuantityKinds.RADIANT_ENERGY, QuantityKinds.FLUENCE_RATE)) ) );
UNITS_BY_NAME.put( "wattPerKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerKelvin" ), "wattPerKelvin", Set.of(new LangString( "watt per kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/K"), Optional.of("D52"), Optional.empty(), Optional.of("W/K"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTANCE)) ) );
UNITS_BY_NAME.put( "wattPerKilogram", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerKilogram" ), "wattPerKilogram", Set.of(new LangString( "watt per kilogram", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/kg"), Optional.of("WA"), Optional.empty(), Optional.of("1 W/kg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wattPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerMetre" ), "wattPerMetre", Set.of(new LangString( "watt per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/m"), Optional.of("H74"), Optional.empty(), Optional.of("W m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FIRST_RADIATION_CONSTANT, QuantityKinds.RADIANT_EXITANCE, QuantityKinds.IRRADIANCE, QuantityKinds.FLUENCE_RATE, QuantityKinds.RADIANT_ENERGY)) ) );
UNITS_BY_NAME.put( "wattPerMetreDegreeCelsius", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerMetreDegreeCelsius" ), "wattPerMetreDegreeCelsius", Set.of(new LangString( "watt per metre degree Celsius", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/(m\u00B7\u00B0C)"), Optional.of("N80"), Optional.of("wattPerMetreKelvin"), Optional.of("W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "wattPerMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerMetreKelvin" ), "wattPerMetreKelvin", Set.of(new LangString( "watt per metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/(m\u00B7K)"), Optional.of("D53"), Optional.empty(), Optional.of("W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "wattPerSquareCentimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSquareCentimetre" ), "wattPerSquareCentimetre", Set.of(new LangString( "watt per square centimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/cm\u00B2"), Optional.of("N48"), Optional.of("wattPerSquareMetre"), Optional.of("10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "wattPerSquareInch", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSquareInch" ), "wattPerSquareInch", Set.of(new LangString( "watt per square inch", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/in\u00B2"), Optional.of("N49"), Optional.of("wattPerSquareMetre"), Optional.of("1.550003 \u00D7 10\u00B3 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "wattPerSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSquareMetre" ), "wattPerSquareMetre", Set.of(new LangString( "watt per square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/m\u00B2"), Optional.of("D54"), Optional.empty(), Optional.of("W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.IRRADIANCE, QuantityKinds.FIRST_RADIATION_CONSTANT, QuantityKinds.FLUENCE_RATE, QuantityKinds.RADIANT_ENERGY, QuantityKinds.SOUND_INTENSITY, QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE, QuantityKinds.RADIANT_EXITANCE, QuantityKinds.POYNTING_VECTOR, QuantityKinds.ENERGY_FLUX_DENSITY, QuantityKinds.ENERGY_FLUENCE_RATE)) ) );
UNITS_BY_NAME.put( "wattPerSquareMetreKelvin", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSquareMetreKelvin" ), "wattPerSquareMetreKelvin", Set.of(new LangString( "watt per square metre kelvin", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/(m\u00B2\u00B7K)"), Optional.of("D55"), Optional.empty(), Optional.of("W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "wattPerSquareMetreKelvinToTheFourthPower", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSquareMetreKelvinToTheFourthPower" ), "wattPerSquareMetreKelvinToTheFourthPower", Set.of(new LangString( "watt per square metre kelvin to the fourth power", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/(m\u00B2\u00B7K\u2074)"), Optional.of("D56"), Optional.empty(), Optional.of("W/(m\u00B2 \u00D7 K\u2074)"), new HashSet<>(Arrays.asList(QuantityKinds.STEFAN_BOLTZMANN_CONSTANT)) ) );
UNITS_BY_NAME.put( "wattPerSteradian", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSteradian" ), "wattPerSteradian", Set.of(new LangString( "watt per steradian", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/sr"), Optional.of("D57"), Optional.empty(), Optional.of("W/sr"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_INTENSITY)) ) );
UNITS_BY_NAME.put( "wattPerSteradianSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattPerSteradianSquareMetre" ), "wattPerSteradianSquareMetre", Set.of(new LangString( "watt per steradian square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W/(sr\u00B7m\u00B2)"), Optional.of("D58"), Optional.empty(), Optional.of("W/(sr \u00D7 m\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANCE)) ) );
UNITS_BY_NAME.put( "wattSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattSecond" ), "wattSecond", Set.of(new LangString( "watt second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W\u00B7s"), Optional.of("J55"), Optional.empty(), Optional.of("W \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.HEAT, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "wattSquareMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wattSquareMetre" ), "wattSquareMetre", Set.of(new LangString( "watt square metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("W\u00B7m\u00B2"), Optional.of("Q21"), Optional.empty(), Optional.of("m\u2074 \u00D7 kg \u00D7 s\u207B\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "weber", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#weber" ), "weber", Set.of(new LangString( "weber", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Wb"), Optional.of("WEB"), Optional.empty(), Optional.of("Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_QUANTUM, QuantityKinds.MAGNETIC_FLUX)) ) );
UNITS_BY_NAME.put( "weberMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#weberMetre" ), "weberMetre", Set.of(new LangString( "weber metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Wb\u00B7m"), Optional.of("P50"), Optional.empty(), Optional.of("m\u00B3 \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_DIPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "weberPerMetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#weberPerMetre" ), "weberPerMetre", Set.of(new LangString( "weber per metre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Wb/m"), Optional.of("D59"), Optional.empty(), Optional.of("Wb/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_VECTOR_POTENTIAL)) ) );
UNITS_BY_NAME.put( "weberPerMillimetre", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#weberPerMillimetre" ), "weberPerMillimetre", Set.of(new LangString( "weber per millimetre", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("Wb/mm"), Optional.of("D60"), Optional.of("weberPerMetre"), Optional.of("10\u00B3 Wb/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_VECTOR_POTENTIAL)) ) );
UNITS_BY_NAME.put( "weberToThePowerMinusOne", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#weberToThePowerMinusOne" ), "weberToThePowerMinusOne", Set.of(new LangString( "weber to the power minus one", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("1/Wb"), Optional.of("Q23"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2 \u00D7 A"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "week", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#week" ), "week", Set.of(new LangString( "week", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("wk"), Optional.of("WEE"), Optional.of("secondUnitOfTime"), Optional.of("6.048 \u00D7 10\u2075 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "well", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#well" ), "well", Set.of(new LangString( "well", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E56"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wetKilo", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wetKilo" ), "wetKilo", Set.of(new LangString( "wet kilo", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("W2"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wetPound", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wetPound" ), "wetPound", Set.of(new LangString( "wet pound", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("WB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wetTon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wetTon" ), "wetTon", Set.of(new LangString( "wet ton", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("WE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wineGallon", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#wineGallon" ), "wineGallon", Set.of(new LangString( "wine gallon", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("WG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}

private static void init17() {
UNITS_BY_NAME.put( "workingDay", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#workingDay" ), "workingDay", Set.of(new LangString( "working day", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E49"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "workingMonth", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#workingMonth" ), "workingMonth", Set.of(new LangString( "working month", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("WM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "yard", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yard" ), "yard", Set.of(new LangString( "yard", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd"), Optional.of("YRD"), Optional.of("metre"), Optional.of("0.9144 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DIAMETER)) ) );
UNITS_BY_NAME.put( "yardPerDegreeFahrenheit", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yardPerDegreeFahrenheit" ), "yardPerDegreeFahrenheit", Set.of(new LangString( "yard per degree Fahrenheit", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd/\u00B0F"), Optional.of("L98"), Optional.of("metrePerKelvin"), Optional.of("1.64592 m/K"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER)) ) );
UNITS_BY_NAME.put( "yardPerHour", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yardPerHour" ), "yardPerHour", Set.of(new LangString( "yard per hour", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd/h"), Optional.of("M66"), Optional.of("metrePerSecond"), Optional.of("2.54 \u00D7 10\u207B\u2074 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "yardPerMinute", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yardPerMinute" ), "yardPerMinute", Set.of(new LangString( "yard per minute", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd/min"), Optional.of("M65"), Optional.of("metrePerSecond"), Optional.of("1.524 \u00D7 10\u207B\u00B2 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "yardPerPsi", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yardPerPsi" ), "yardPerPsi", Set.of(new LangString( "yard per psi", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd/psi"), Optional.of("L99"), Optional.of("metrePerPascal"), Optional.of("1.326225 \u00D7 10\u207B\u2074 m/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "yardPerSecond", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yardPerSecond" ), "yardPerSecond", Set.of(new LangString( "yard per second", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd/s"), Optional.of("M64"), Optional.of("metrePerSecond"), Optional.of("9.144 \u00D7 10\u207B\u00B9 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "yardPerSecondSquared", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yardPerSecondSquared" ), "yardPerSecondSquared", Set.of(new LangString( "yard per second squared", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("yd/s\u00B2"), Optional.of("M40"), Optional.of("metrePerSecondSquared"), Optional.of("9.144 \u00D7 10\u207B\u00B9 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "year", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#year" ), "year", Set.of(new LangString( "year", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("y"), Optional.of("ANN"), Optional.of("secondUnitOfTime"), Optional.of("3.15576 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "yobibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yobibyte" ), "yobibyte", Set.of(new LangString( "Yobibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("YiB"), Optional.empty(), Optional.of("byte"), Optional.of("2\u2078\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "yottabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#yottabyte" ), "yottabyte", Set.of(new LangString( "Yottabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("YB"), Optional.empty(), Optional.of("byte"), Optional.of("10\u00B2\u2074 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "zebibyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#zebibyte" ), "zebibyte", Set.of(new LangString( "Zebibyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ZiB"), Optional.empty(), Optional.of("byte"), Optional.of("2\u2077\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "zettabyte", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#zettabyte" ), "zettabyte", Set.of(new LangString( "Zettabyte", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.of("ZB"), Optional.empty(), Optional.of("byte"), Optional.of("10\u00B2\u00B9 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "zone", new DefaultUnit( new MetaModelBaseAttributes( AspectModelUrn.fromUrn( "urn:samm:org.eclipse.esmf.samm:unit:2.1.0#zone" ), "zone", Set.of(new LangString( "zone", Locale.forLanguageTag( "en" ) )), Set.of(), List.of() ), Optional.empty(), Optional.of("E57"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}


   /**
    * Returns the unit with a given name
    */
   public static synchronized Optional<Unit> fromName( final String name ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
init0();
init1();
init2();
init3();
init4();
init5();
init6();
init7();
init8();
init9();
init10();
init11();
init12();
init13();
init14();
init15();
init16();
init17();
      }
      return Optional.ofNullable( UNITS_BY_NAME.get( name ) );
   }

   /**
    * Finds the unit with a given code
    */
   public static Optional<Unit> fromCode( final String code ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
         fromName( "" );
      }
      return UNITS_BY_NAME.values()
            .stream()
            .filter( unit -> unit.getCode().map( code2 -> code2.equals( code ) ).orElse( false ) )
            .findAny();
   }

   /**
    * Finds the units with a given symbol. In most, but not all cases, this set will contain exactly one unit.
    */
   public static Set<Unit> fromSymbol( final String symbol ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
         fromName( "" );
      }
      return UNITS_BY_NAME.values()
            .stream()
            .flatMap( unit -> unit.getSymbol()
                  .filter( symbol2 -> symbol2.equals( symbol ) )
                  .stream()
                  .map( symbol2 -> unit ) )
            .collect( Collectors.toSet() );
   }

   /**
    * Returns all units of a given quantity kind
    */
   public static Set<Unit> unitsWithQuantityKind( final QuantityKind quantityKind ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
         fromName( "" );
      }
      return UNITS_BY_NAME.values()
            .stream()
            .filter( unit -> unit.getQuantityKinds().contains( quantityKind ) )
            .collect( Collectors.toSet() );
   }
}
