/*
 * Copyright (c) 2024 Robert Bosch Manufacturing Solutions GmbH
 *
 * See the AUTHORS file(s) distributed with this work for additional
 * information regarding authorship.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * SPDX-License-Identifier: MPL-2.0
 */

package org.eclipse.esmf.metamodel;

import java.util.Arrays;
import java.util.Optional;

import org.eclipse.esmf.aspectmodel.AspectModelFile;
import org.eclipse.esmf.aspectmodel.resolver.modelfile.MetaModelFile;
import org.eclipse.esmf.aspectmodel.urn.AspectModelUrn;
import org.eclipse.esmf.aspectmodel.visitor.AspectVisitor;
import org.eclipse.esmf.metamodel.vocabulary.SammNs;

/**
 * Enumeration of Quantity Kinds as defined in <a href="http://tfig.unece.org/contents/recommendation-20.htm">Recommendation 20</a>
 * by the <a href="http://www.unece.org/info/ece-homepage.html">UNECE</a> (United Nations Economic Commission for Europe).
 */
public enum QuantityKinds implements QuantityKind {
AMOUNT_OF_SUBSTANCE( "amountOfSubstance", "amount of substance" ),
LUMINOUS_EFFICIENCY( "luminousEfficiency", "luminous efficiency" ),
ANGLE_OF_OPTICAL_ROTATION( "angleOfOpticalRotation", "angle of optical rotation" ),
SPECIFIC_OPTICAL_ROTATORY_POWER( "specificOpticalRotatoryPower", "specific optical rotatory power" ),
AFFINITY_OF_A_CHEMICAL_REACTION( "affinityOfAChemicalReaction", "affinity (of a chemical reaction)" ),
FAHRENHEIT_TEMPERATURE( "fahrenheitTemperature", "fahrenheit temperature" ),
RESIDUAL_RESISTIVITY( "residualResistivity", "residual resistivity" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION( "standardAbsoluteActivityOfSoluteBEspeciallyInADiluteSolution", "standard absolute activity of solute B (especially in a dilute solution)" ),
VELOCITY_OF_SOUND_PHASE_VELOCITY( "velocityOfSoundPhaseVelocity", "velocity of sound (phase velocity)" ),
MASSIEU_FUNCTION( "massieuFunction", "massieu function" ),
THICKNESS( "thickness", "thickness" ),
MEAN_LIFE( "meanLife", "mean life" ),
DENSITY_OF_STATES( "densityOfStates", "density of states" ),
MAGNETIC_POLARIZATION( "magneticPolarization", "magnetic polarization" ),
REACTANCE( "reactance", "reactance" ),
ISENTROPIC_EXPONENT( "isentropicExponent", "isentropic exponent" ),
STROUHAL_NUMBER( "strouhalNumber", "Strouhal number" ),
LOUDNESS_LEVEL( "loudnessLevel", "loudness level" ),
RELATIVE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION( "relativeActivityOfSolventAEspeciallyInADiluteSolution", "relative activity of solvent A (especially in a dilute solution)" ),
INSTANTANEOUS_SOUND_PRESSURE( "instantaneousSoundPressure", "(instantaneous) sound pressure" ),
POISSON_RATIO( "poissonRatio", "poisson ratio" ),
PLANCK_CONSTANT( "planckConstant", "Planck constant" ),
PELTIER_COEFFICIENT_FOR_SUBSTANCES_A_AND_B( "peltierCoefficientForSubstancesAAndB", "Peltier coefficient for substances a and b" ),
LEVEL_OF_A_POWER_QUANTITY( "levelOfAPowerQuantity", "level of a power quantity" ),
HALF_THICKNESS( "halfThickness", "half-thickness" ),
MOLAR_OPTICAL_ROTATORY_POWER( "molarOpticalRotatoryPower", "molar optical rotatory power" ),
BRAGG_ANGLE( "braggAngle", "Bragg angle" ),
DEBYE_TEMPERATURE( "debyeTemperature", "Debye temperature" ),
ENERGY_FLUENCE_RATE( "energyFluenceRate", "energy fluence rate" ),
WEIGHT( "weight", "weight" ),
RESISTANCE_LOAD_PER_UNIT_LENGTH( "resistanceLoadPerUnitLength", "resistance load per unit length" ),
RADIANCE_EXPOSURE( "radianceExposure", "radiance exposure" ),
CARRIER_LIFE_TIME( "carrierLifeTime", "carrier life time" ),
WORK_FUNCTION( "workFunction", "work function" ),
MOBILITY( "mobility", "mobility" ),
MAGNETIC_REYNOLDS_NUMBER( "magneticReynoldsNumber", "magnetic Reynolds number" ),
LOUDNESS( "loudness", "loudness" ),
NUCLEAR_RADIUS( "nuclearRadius", "nuclear radius" ),
MASS_ENERGY_TRANSFER_COEFFICIENT( "massEnergyTransferCoefficient", "mass energy transfer coefficient" ),
MASS_RATIO( "massRatio", "mass ratio" ),
DIRECTIONAL_SPECTRAL_EMISSIVITY( "directionalSpectralEmissivity", "directional spectral emissivity" ),
WEBER_NUMBER( "weberNumber", "Weber number" ),
RELATIVE_MOLECULAR_MASS( "relativeMolecularMass", "relative molecular mass" ),
REVERBERATION_TIME( "reverberationTime", "reverberation time" ),
MOLALITY_OF_SOLUTE_B( "molalityOfSoluteB", "molality of solute B" ),
SURFACE_TENSION( "surfaceTension", "surface tension" ),
INSTANTANEOUS_VOLUME_FLOW_RATE( "instantaneousVolumeFlowRate", "(instantaneous) volume flow rate" ),
DIMENSIONLESS( "dimensionless", "dimensionless" ),
RAYLEIGH_NUMBER( "rayleighNumber", "Rayleigh number" ),
BINDING_FRACTION( "bindingFraction", "binding fraction" ),
DEBYE_ANGULAR_REPETENCY( "debyeAngularRepetency", "Debye angular repetency" ),
SLOWING_DOWN_DENSITY( "slowingDownDensity", "slowing down density" ),
BREADTH( "breadth", "breadth" ),
REST_MASS_OF_PROTON( "restMassOfProton", "(rest) mass of proton" ),
MODULUS_OF_COMPRESSION( "modulusOfCompression", "modulus of compression" ),
ELECTRON_NUMBER_DENSITY( "electronNumberDensity", "electron number density" ),
ACCELERATION_OF_FREE_FALL( "accelerationOfFreeFall", "acceleration of free fall" ),
GYROMAGNETIC_RATIO( "gyromagneticRatio", "gyromagnetic ratio" ),
SHORT_RANGE_ORDER_PARAMETER( "shortRangeOrderParameter", "short-range order parameter" ),
MAGNETIC_VECTOR_POTENTIAL( "magneticVectorPotential", "magnetic vector potential" ),
CHARACTERISTIC_IMPEDANCE_OF_A_MEDIUM( "characteristicImpedanceOfAMedium", "characteristic impedance of a medium" ),
IONIC_STRENGTH( "ionicStrength", "ionic strength" ),
RESISTANCE_TO_DIRECT_CURRENT( "resistanceToDirectCurrent", "resistance (to direct current)" ),
RADIANT_POWER( "radiantPower", "radiant power" ),
MAGNETIC_DIPOLE_MOMENT( "magneticDipoleMoment", "magnetic dipole moment" ),
RESISTIVITY( "resistivity", "resistivity" ),
POWER( "power", "power" ),
TOTAL_IONIZATION_BY_A_PARTICLE( "totalIonizationByAParticle", "total ionization by a particle" ),
VOLUMIC( "volumic", "volumic" ),
MASS_DENSITY( "massDensity", "mass density" ),
SPECTRAL_ABSORPTION_FACTOR( "spectralAbsorptionFactor", "spectral absorption factor" ),
INSTANTANEOUS_SOUND_PARTICLE_VELOCITY( "instantaneousSoundParticleVelocity", "(instantaneous) sound particle velocity" ),
ELECTRIC_POLARIZABILITY_OF_A_MOLECULE( "electricPolarizabilityOfAMolecule", "electric polarizability of a molecule" ),
LENGTH_OF_PATH( "lengthOfPath", "length of path" ),
BOLTZMANN_CONSTANT( "boltzmannConstant", "Boltzmann constant" ),
INTERNAL_CONVERSION_FACTOR( "internalConversionFactor", "internal conversion factor" ),
ELECTRIC_CONSTANT( "electricConstant", "electric constant" ),
GRUENEISEN_PARAMETER( "grueneisenParameter", "Grüneisen parameter" ),
PHASE_DIFFERENCE( "phaseDifference", "phase difference" ),
LINEAR_ENERGY_TRANSFER( "linearEnergyTransfer", "linear energy transfer" ),
RELATIVE_MASS_DENSITY( "relativeMassDensity", "relative mass density" ),
PERMEANCE( "permeance", "permeance" ),
LOGARITHMIC_DECREMENT( "logarithmicDecrement", "logarithmic decrement" ),
PERFORMANCE_CHARACTERISTIC( "performanceCharacteristic", "performance characteristic" ),
INFORMATION_ENTROPY( "informationEntropy", "Information entropy" ),
SPECTRAL_LUMINOUS_EFFICACY( "spectralLuminousEfficacy", "spectral luminous efficacy" ),
NUSSELT_NUMBER_FOR_MASS_TRANSFER( "nusseltNumberForMassTransfer", "Nusselt number for mass transfer" ),
DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUENCE_RATE( "diffusionCoefficientForNeutronFluenceRate", "diffusion coefficient for neutron fluence rate" ),
TOTAL_ATOMIC_STOPPING_POWER( "totalAtomicStoppingPower", "total atomic stopping power" ),
MOMENT_OF_MOMENTUM( "momentOfMomentum", "moment of momentum" ),
FINE_STRUCTURE_CONSTANT( "fineStructureConstant", "fine structure constant" ),
REACTIVE_POWER( "reactivePower", "reactive power" ),
ANGULAR_MOMENTUM( "angularMomentum", "angular momentum" ),
PECLET_NUMBER( "pecletNumber", "Peclet number" ),
ACTIVITY( "activity", "activity" ),
RADIANT_ENERGY( "radiantEnergy", "radiant energy" ),
THERMAL_INSULANCE( "thermalInsulance", "thermal insulance" ),
CUBIC_EXPANSION_COEFFICIENT( "cubicExpansionCoefficient", "cubic expansion coefficient" ),
NEEL_TEMPERATURE( "neelTemperature", "Néel temperature" ),
MIGRATION_AREA( "migrationArea", "migration area" ),
REACTIVITY( "reactivity", "reactivity" ),
REFRACTIVE_INDEX( "refractiveIndex", "refractive index" ),
VOLUMIC_DONOR_NUMBER( "volumicDonorNumber", "volumic donor number" ),
TEMPERATURE( "temperature", "temperature" ),
ENERGY_FLUENCE( "energyFluence", "energy fluence" ),
NEUTRON_FLUENCE_RATE( "neutronFluenceRate", "neutron fluence rate" ),
EFFECTIVE_MASS( "effectiveMass", "effective mass" ),
QUANTITY_OF_HEAT( "quantityOfHeat", "quantity of heat" ),
PECLET_NUMBER_FOR_MASS_TRANSFER( "pecletNumberForMassTransfer", "Peclet number for mass transfer" ),
IMPEDANCE( "impedance", "impedance" ),
ABSORBANCE( "absorbance", "absorbance" ),
CONDUCTANCE_FOR_ALTERNATING_CURRENT( "conductanceForAlternatingCurrent", "conductance (for alternating current)" ),
FUNDAMENTAL_RECIPROCAL_LATTICE_VECTOR( "fundamentalReciprocalLatticeVector", "fundamental reciprocal lattice vector" ),
SOLID_ANGLE( "solidAngle", "solid angle" ),
LUMINOUS_EFFICACY_AT_A_SPECIFIED_WAVELENGTH( "luminousEfficacyAtASpecifiedWavelength", "luminous efficacy at a specified wavelength" ),
MAGNETIC_FLUX_QUANTUM( "magneticFluxQuantum", "magnetic flux quantum" ),
NUMBER_OF_TURNS_IN_A_WINDING( "numberOfTurnsInAWinding", "number of turns in a winding" ),
WAVENUMBER( "wavenumber", "wavenumber" ),
DIAMETER( "diameter", "diameter" ),
CHARGE_DENSITY( "chargeDensity", "charge density" ),
SPECTRAL_CONCENTRATION_OF_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVELENGTH( "spectralConcentrationOfRadiantEnergyDensityInTermsOfWavelength", "spectral concentration of radiant energy density (in terms of wavelength)" ),
PHASE_DISPLACEMENT( "phaseDisplacement", "phase displacement" ),
ACTIVE_ENERGY( "activeEnergy", "active energy" ),
EULER_NUMBER( "eulerNumber", "Euler number" ),
NUCLEAR_SPIN_QUANTUM_NUMBER( "nuclearSpinQuantumNumber", "nuclear spin quantum number" ),
ABSORPTION_FACTOR( "absorptionFactor", "absorption factor" ),
ION_DENSITY( "ionDensity", "ion density" ),
HEIGHT( "height", "height" ),
NEUTRON_YIELD_PER_ABSORPTION( "neutronYieldPerAbsorption", "neutron yield per absorption" ),
LENGTH( "length", "length" ),
ACTIVITY_OF_SOLVENT_A( "activityOfSolventA", "activity of solvent A" ),
CHEMICAL_POTENTIAL( "chemicalPotential", "chemical potential" ),
RELATIVE_PERMEABILITY( "relativePermeability", "relative permeability" ),
RESISTANCE_TO_ALTERNATING_CURRENT( "resistanceToAlternatingCurrent", "resistance (to alternating current)" ),
MOLE_FRACTION_OF_B( "moleFractionOfB", "mole fraction of B" ),
RELATIVE_PRESSURE_COEFFICIENT( "relativePressureCoefficient", "relative pressure coefficient" ),
TIME_CONSTANT( "timeConstant", "time constant" ),
MAGNETIC_AREA_MOMENT( "magneticAreaMoment", "magnetic area moment" ),
ELECTROMOTIVE_FORCE( "electromotiveForce", "electromotive force" ),
MASS_FLOW_RATE( "massFlowRate", "mass flow rate" ),
PHOTON_EXITANCE( "photonExitance", "photon exitance" ),
HEAT_CAPACITY( "heatCapacity", "heat capacity" ),
CONCENTRATION_OF_B( "concentrationOfB", "concentration of B" ),
SECOND_MOMENT_OF_AREA( "secondMomentOfArea", "second moment of area" ),
TRANSPORT_NUMBER_OF_ION_B( "transportNumberOfIonB", "transport number of ion B" ),
STOICHIOMETRIC_NUMBER_OF_B( "stoichiometricNumberOfB", "stoichiometric number of B" ),
INTRINSIC_NUMBER_DENSITY( "intrinsicNumberDensity", "intrinsic number density" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_LIQUID_OR_A_SOLID_MIXTURE( "standardAbsoluteActivityOfBInALiquidOrASolidMixture", "standard absolute activity of B (in a liquid or a solid mixture)" ),
ELECTRIC_POTENTIAL( "electricPotential", "electric potential" ),
PARTICAL_FLUX_DENSITY( "particalFluxDensity", "partical flux density" ),
SOUND_PRESSURE_LEVEL( "soundPressureLevel", "sound pressure level" ),
RICHARDSON_CONSTANT( "richardsonConstant", "Richardson constant" ),
FAST_FISSION_FACTOR( "fastFissionFactor", "fast fission factor" ),
PROPAGATION_COEFFICIENT( "propagationCoefficient", "propagation coefficient" ),
RADIANT_ENERGYFLUX( "radiantEnergyflux", "radiant energyflux" ),
SPECTRAL_REFLECTIONFACTOR( "spectralReflectionfactor", "spectral reflectionfactor" ),
ANGULAR_CROSS_SECTION( "angularCrossSection", "angular cross-section" ),
PROTON_NUMBER( "protonNumber", "proton number" ),
NEUTRON_NUMBER_DENSITY( "neutronNumberDensity", "neutron number density" ),
RADIANT_EXITANCE( "radiantExitance", "radiant exitance" ),
UPPER_CRITICAL_MAGNETIC_FLUX_DENSITY( "upperCriticalMagneticFluxDensity", "upper critical magnetic flux density" ),
MASS_OF_MOLECULE( "massOfMolecule", "mass of molecule" ),
NUMBER_OF_PAIRS_OF_POLES( "numberOfPairsOfPoles", "number of pairs of poles" ),
AVERAGE_LOGARITHMIC_ENERGY_DECREMENT( "averageLogarithmicEnergyDecrement", "average logarithmic energy decrement" ),
MOLAR_HEAT_CAPACITY( "molarHeatCapacity", "molar heat capacity" ),
POWER_FOR_DIRECT_CURRENT( "powerForDirectCurrent", "power (for direct current)" ),
STATIC_PRESSURE( "staticPressure", "static pressure" ),
RELATIVE_ELONGATION( "relativeElongation", "relative elongation" ),
NUCLEAR_QUADRUPOLE_MOMENT( "nuclearQuadrupoleMoment", "nuclear quadrupole moment" ),
RADIUS_OF_CURVATURE( "radiusOfCurvature", "radius of curvature" ),
NUMBER_OF_PHASES( "numberOfPhases", "number of phases" ),
EXPOSURE_RATE( "exposureRate", "exposure rate" ),
LINEAR_IONIZATION_BY_A_PARTICLE( "linearIonizationByAParticle", "linear ionization by a particle" ),
LONG_RANGE_ORDER_PARAMETER( "longRangeOrderParameter", "long-range order parameter" ),
DISSIPANCE( "dissipance", "dissipance" ),
DISPLACEMENT_VECTOR_OF_ION_OR_ATOM( "displacementVectorOfIonOrAtom", "displacement vector of ion or atom" ),
RELUCTANCE( "reluctance", "reluctance" ),
MEAN_ENERGY_IMPARTED( "meanEnergyImparted", "mean energy imparted" ),
FERMI_ANGULAR_WAVE_NUMBER( "fermiAngularWaveNumber", "Fermi angular wave number" ),
LIGHT_EXPOSURE( "lightExposure", "light exposure" ),
LENS_POWER( "lensPower", "lens power" ),
POTENTIAL_ENERGY( "potentialEnergy", "potential energy" ),
ACCEPTOR_IONIZATION_ENERGY( "acceptorIonizationEnergy", "acceptor ionization energy" ),
MAGNETOMOTIVE_FORCE( "magnetomotiveForce", "magnetomotive force" ),
REFLECTION_FACTOR( "reflectionFactor", "reflection factor" ),
LINEAR_ABSORPTION_COEFFICIENT( "linearAbsorptionCoefficient", "linear absorption coefficient" ),
ANGULAR_IMPULSE( "angularImpulse", "angular impulse" ),
STANDARD_EQUILIBRIUM_CONSTANT( "standardEquilibriumConstant", "standard equilibrium constant" ),
APPARENT_POWER( "apparentPower", "apparent power" ),
ELECTROMAGNETIC_MOMENT( "electromagneticMoment", "electromagnetic moment" ),
LINEIC_RESISTANCE( "lineicResistance", "lineic resistance" ),
G_FACTOR_OF_NUCLEUS( "gFactorOfNucleus", "g-factor of nucleus" ),
MASSIC_THERMODYNAMIC_ENERGY( "massicThermodynamicEnergy", "massic thermodynamic energy" ),
MOLECULAR_CONCENTRATION_OF_B( "molecularConcentrationOfB", "molecular concentration of B" ),
MAGNETIC_CONSTANT( "magneticConstant", "magnetic constant" ),
THERMAL_UTILIZATION_FACTOR( "thermalUtilizationFactor", "thermal utilization factor" ),
PARTICLE_FLUENCE( "particleFluence", "particle fluence" ),
HALF_LIFE( "halfLife", "half life" ),
MASSIC_VOLUME( "massicVolume", "massic volume" ),
VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES( "volumicNumberOfMoleculesOrParticles", "volumic number of molecules (or particles)" ),
SPECTRAL_TRANSMITTANCE( "spectralTransmittance", "spectral transmittance" ),
ROTATIONAL_FREQUENCY( "rotationalFrequency", "rotational frequency" ),
TIME( "time", "time" ),
LOWER_CRITICAL_MAGNETIC_FLUX_DENSITY( "lowerCriticalMagneticFluxDensity", "lower critical magnetic flux density" ),
PERMITTIVITY( "permittivity", "permittivity" ),
SOUND_POWER_LEVEL( "soundPowerLevel", "sound power level" ),
FLUENCE_RATE( "fluenceRate", "fluence rate" ),
KINEMATIC_VISCOSITY( "kinematicViscosity", "kinematic viscosity" ),
LEVEL_OF_A_FIELD_QUANTITY( "levelOfAFieldQuantity", "level of a field quantity" ),
VOLUME_RATIO( "volumeRatio", "volume ratio" ),
MADELUNG_CONSTANT( "madelungConstant", "Madelung constant" ),
ENERGY_DENSITY( "energyDensity", "energy density" ),
COEFFICIENT( "coefficient", "coefficient" ),
LEAKAGE_RATE_OF_GAS( "leakageRateOfGas", "leakage rate of gas" ),
MOLECULAR_PARTITION_FUNCTION( "molecularPartitionFunction", "molecular partition function" ),
SPECIFIC_ACTIVITY_IN_A_SAMPLE( "specificActivityInASample", "specific activity in a sample" ),
LINEAR_EXPANSION_COEFFICIENT( "linearExpansionCoefficient", "linear expansion coefficient" ),
RADIANT_ENERGY_DENSITY( "radiantEnergyDensity", "radiant energy density" ),
CROSS_SECTION( "crossSection", "cross-section" ),
ENERGY( "energy", "energy" ),
LANDAU_GINZBURG_NUMBER( "landauGinzburgNumber", "Landau-Ginzburg number" ),
CURRENT_DENSITY( "currentDensity", "current density" ),
ISOTHERMAL_COMPRESSIBILITY( "isothermalCompressibility", "isothermal compressibility" ),
ORBITAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER( "orbitalAngularMomentumQuantumNumber", "orbital angular momentum quantum number" ),
LATTICE_PLANE_SPACING( "latticePlaneSpacing", "lattice plane spacing" ),
AREA_RELATED_TORSIONAL_MOMENT( "areaRelatedTorsionalMoment", "area-related torsional moment" ),
SEEBECK_COEFFICIENT_FOR_SUBSTANCES_A_AND_B( "seebeckCoefficientForSubstancesAAndB", "Seebeck coefficient for substances a and b" ),
WAVE_NUMBER( "waveNumber", "wave number" ),
VOLUMIC_INTRINSIS_NUMBER( "volumicIntrinsisNumber", "volumic intrinsis number" ),
MASS( "mass", "mass" ),
NORMAL_STRESS( "normalStress", "normal stress" ),
MODULUS_OF_ADMITTANCE( "modulusOfAdmittance", "modulus of admittance" ),
TENSION( "tension", "tension" ),
VOLUMIC_ACTIVITY( "volumicActivity", "volumic activity" ),
OBJECT_DISTANCE( "objectDistance", "object distance" ),
HALF_VALUE_THICKNESS( "halfValueThickness", "half-value thickness" ),
MOLAR_VOLUME( "molarVolume", "molar volume" ),
REYNOLDS_NUMBER( "reynoldsNumber", "Reynolds number" ),
SPECTRAL_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVE_LENGTH( "spectralRadiantEnergyDensityInTermsOfWaveLength", "spectral radiant energy density (in terms of wave length)" ),
THERMAL_DIFFUSION_FACTOR( "thermalDiffusionFactor", "thermal diffusion factor" ),
THOMPSON_COEFFICIENT( "thompsonCoefficient", "Thompson coefficient" ),
EMISSIVITY_AT_A_SPECIFIED_WAVELENGTH( "emissivityAtASpecifiedWavelength", "emissivity at a specified wavelength" ),
REST_MASS_OF_ELECTRON( "restMassOfElectron", "(rest) mass of electron" ),
PERMITTIVITY_OF_VACUUM( "permittivityOfVacuum", "permittivity of vacuum" ),
FUNDAMENTAL_LATTICE_VECTOR( "fundamentalLatticeVector", "fundamental lattice vector" ),
GRASHOF_NUMBER_FOR_MASS_TRANSFER( "grashofNumberForMassTransfer", "Grashof number for mass transfer" ),
OPTICAL_DENSITY( "opticalDensity", "optical density" ),
SPECIFIC_HEAT_CAPACITY_AT_SATURATION( "specificHeatCapacityAtSaturation", "specific heat capacity at saturation" ),
STANTON_NUMBER( "stantonNumber", "Stanton number" ),
REFLECTANCE( "reflectance", "reflectance" ),
PERIOD( "period", "period" ),
GRAVITATIONAL_CONSTANT( "gravitationalConstant", "gravitational constant" ),
PARTIAL_PRESSURE_OF_B_IN_A_GASEOUS_MIXTURE( "partialPressureOfBInAGaseousMixture", "partial pressure of B (in a gaseous mixture)" ),
EFFECTIVE_MULTIPLICATION_FACTOR( "effectiveMultiplicationFactor", "effective multiplication factor" ),
ELECTRON_AFFINITY( "electronAffinity", "electron affinity" ),
EMISSIVITY( "emissivity", "emissivity" ),
MOLAR_ABSORPTION_COEFFICIENT( "molarAbsorptionCoefficient", "molar absorption coefficient" ),
DENSITY_OF_HEAT_FLOW_RATE( "densityOfHeatFlowRate", "density of heat flow rate" ),
SOUND_EXPOSURE( "soundExposure", "sound exposure" ),
RADIANT_ENERGY_FLUENCE( "radiantEnergyFluence", "radiant energy fluence" ),
LOSS_ANGLE( "lossAngle", "loss angle" ),
ACOUSTIC_IMPEDANCE( "acousticImpedance", "acoustic impedance" ),
ELECTRIC_CURRENT( "electricCurrent", "electric current" ),
VOLUMIC_ELECTROMAGNETIC_ENERGY( "volumicElectromagneticEnergy", "volumic electromagnetic energy" ),
SHEAR_MODULUS( "shearModulus", "shear modulus" ),
SPECTRAL_CROSS_SECTION( "spectralCrossSection", "spectral cross-section" ),
DEBYE_ANGULAR_WAVE_NUMBER( "debyeAngularWaveNumber", "Debye angular wave number" ),
CYCLOTRON_ANGULAR_FREQUENCY( "cyclotronAngularFrequency", "cyclotron angular frequency" ),
ANGULAR_WAVENUMBER( "angularWavenumber", "angular wavenumber" ),
MAGNETIC_FLUX( "magneticFlux", "magnetic flux" ),
LONDON_PENETRATION_DEPTH( "londonPenetrationDepth", "London penetration depth" ),
THERMAL_DIFFUSION_COEFFICIENT( "thermalDiffusionCoefficient", "thermal diffusion coefficient" ),
INSTANTANEOUS_SOUND_PARTICLE_DISPLACEMENT( "instantaneousSoundParticleDisplacement", "(instantaneous) sound particle displacement" ),
ADMITTANCE( "admittance", "admittance" ),
MAGNETIC_MOMENT_OF_PARTICLE( "magneticMomentOfParticle", "magnetic moment of particle" ),
MIGRATION_LENGTH( "migrationLength", "migration length" ),
VOLUMIC_MASS( "volumicMass", "volumic mass" ),
HARDNESS_INDEX( "hardnessIndex", "hardness index" ),
QUANTITY_OF_ELECTRICITY( "quantityOfElectricity", "quantity of electricity" ),
TORQUE( "torque", "torque" ),
TRANSMISSION_FACTOR( "transmissionFactor", "transmission factor" ),
ELECTRIC_FLUX_DENSITY( "electricFluxDensity", "electric flux density" ),
LINEAR_STRAIN( "linearStrain", "linear strain" ),
COWLING_NUMBER( "cowlingNumber", "Cowling number" ),
ANGULAR_REPETENCY( "angularRepetency", "angular repetency" ),
ISENTROPIC_COMPRESSIBILITY( "isentropicCompressibility", "isentropic compressibility" ),
PARTICLE_POSITION_VECTOR( "particlePositionVector", "particle position vector" ),
MEAN_FREE_PATH( "meanFreePath", "mean free path" ),
LINEAR_ATTENUATION_COEFFICIENT( "linearAttenuationCoefficient", "linear attenuation coefficient" ),
SLOWING_DOWN_AREA( "slowingDownArea", "slowing down area" ),
COMPRESSIBILITY( "compressibility", "compressibility" ),
MAGNETIZATION( "magnetization", "magnetization" ),
THERMAL_DIFFUSION_RATIO( "thermalDiffusionRatio", "thermal diffusion ratio" ),
ELECTROMAGNETIC_ENERGY_DENSITY( "electromagneticEnergyDensity", "electromagnetic energy density" ),
DECAY_CONSTANT( "decayConstant", "decay constant" ),
ELECTRIC_SUSCEPTIBILITY( "electricSusceptibility", "electric susceptibility" ),
MASS_EXCESS( "massExcess", "mass excess" ),
LUMINOUS_EXITANCE( "luminousExitance", "luminous exitance" ),
CARTESIAN_COORDINATES( "cartesianCoordinates", "cartesian coordinates" ),
RADIANT_INTENSITY( "radiantIntensity", "radiant intensity" ),
NEUTRON_NUMBER( "neutronNumber", "neutron number" ),
ANGULAR_RECIPROCAL_LATTICE_VECTOR( "angularReciprocalLatticeVector", "angular reciprocal lattice vector" ),
CANONICAL_PARTITION_FUNCTION( "canonicalPartitionFunction", "canonical partition function" ),
TOTAL_MASS_STOPPING_POWER( "totalMassStoppingPower", "total mass stopping power" ),
NEUTRON_YIELD_PER_FISSION( "neutronYieldPerFission", "neutron yield per fission" ),
SOUND_REDUCTION_INDEX( "soundReductionIndex", "sound reduction index" ),
DIFFUSION_COEFFICIENT_FOR_NEUTRON_NUMBER_DENSITY( "diffusionCoefficientForNeutronNumberDensity", "diffusion coefficient for neutron number density" ),
REACTOR_TIME_CONSTANT( "reactorTimeConstant", "reactor time constant" ),
UNIFIED_ATOMIC_MASS_CONSTANT( "unifiedAtomicMassConstant", "unified atomic mass constant" ),
FUGACITY_OF_B_IN_A_GASEOUS_MIXTURE( "fugacityOfBInAGaseousMixture", "fugacity of B (in a gaseous mixture)" ),
LEWIS_NUMBER( "lewisNumber", "Lewis number" ),
FERMI_ENERGY( "fermiEnergy", "Fermi energy" ),
ENTHALPY( "enthalpy", "enthalpy" ),
SHEAR_STRESS( "shearStress", "shear stress" ),
PRESSURE_COEFFICIENT( "pressureCoefficient", "pressure coefficient" ),
MODULUS_OF_ELASTICITY( "modulusOfElasticity", "modulus of elasticity" ),
RELATIVE_ATOMIC_MASS( "relativeAtomicMass", "relative atomic mass" ),
IRRADIANCE( "irradiance", "irradiance" ),
HELMHOLTZ_FREE_ENERGY( "helmholtzFreeEnergy", "Helmholtz free energy" ),
GRAND_PARTITION_FUNCTION( "grandPartitionFunction", "grand partition function" ),
STANTON_NUMBER_FOR_MASS_TRANSFER( "stantonNumberForMassTransfer", "Stanton number for mass transfer" ),
ENTROPY( "entropy", "entropy" ),
FREQUENCY_INTERVAL( "frequencyInterval", "frequency interval" ),
THERMAL_DIFFUSIVITY( "thermalDiffusivity", "thermal diffusivity" ),
PHASE_VELOCITY( "phaseVelocity", "phase velocity" ),
LINEAR_ELECTRIC_CURRENT_DENSITY( "linearElectricCurrentDensity", "linear electric current density" ),
SPECIFIC_ENERGY_IMPARTED( "specificEnergyImparted", "specific energy imparted" ),
MASSIC_OPTICAL( "massicOptical", "massic optical" ),
PHOTON_RADIANCE( "photonRadiance", "photon radiance" ),
REPETENCY( "repetency", "repetency" ),
PRESSURE_RATIO( "pressureRatio", "pressure ratio" ),
BULK_COMPRESSIBILITY( "bulkCompressibility", "bulk compressibility" ),
RELATIVE_DENSITY( "relativeDensity", "relative density" ),
SPECTRAL_RADIANCE_FACTOR( "spectralRadianceFactor", "spectral radiance factor" ),
MEAN_LINEAR_RANGE( "meanLinearRange", "mean linear range" ),
HYPERFINE_STRUCTURE_QUANTUM_NUMBER( "hyperfineStructureQuantumNumber", "hyperfine structure quantum number" ),
MAXIMUM_SPECTRAL_LUMINOUS_EFFICACY( "maximumSpectralLuminousEfficacy", "maximum spectral luminous efficacy" ),
LINEAR_MASS( "linearMass", "linear mass" ),
MOLAR_THERMODYNAMIC_ENERGY( "molarThermodynamicEnergy", "molar thermodynamic energy" ),
DENSITY( "density", "density" ),
FIRST_RADIATION_CONSTANT( "firstRadiationConstant", "first radiation constant" ),
SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE( "specificHeatCapacityAtConstantPressure", "specific heat capacity at constant pressure" ),
BOHR_RADIUS( "bohrRadius", "Bohr radius" ),
FREQUENCY( "frequency", "frequency" ),
COHERENCE_LENGTH( "coherenceLength", "coherence length" ),
MASSIC_ENERGY_IMPARTED( "massicEnergyImparted", "massic energy imparted" ),
SPECTRAL_ABSORPTANCE( "spectralAbsorptance", "spectral absorptance" ),
MOLE_RATIO_OF_SOLUTE_B( "moleRatioOfSoluteB", "mole ratio of solute B" ),
PHASE_SPEED_OF_ELECTROMAGNETIC_WAVES( "phaseSpeedOfElectromagneticWaves", "phase speed of electromagnetic waves" ),
GRASHOF_NUMBER( "grashofNumber", "Grashof number" ),
IMPULSE( "impulse", "impulse" ),
ACTIVITY_COEFFICIENT_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION( "activityCoefficientOfSoluteBEspeciallyInADiluteSolution", "activity coefficient of solute B (especially in a dilute solution)" ),
ALFVEN_NUMBER( "alfvenNumber", "Alfven number" ),
PHOTON_EXPOSURE( "photonExposure", "photon exposure" ),
SOUND_ENERGY_DENSITY( "soundEnergyDensity", "sound energy density" ),
MASS_DEFECT( "massDefect", "mass defect" ),
HARTMANN_NUMBER( "hartmannNumber", "Hartmann number" ),
VOLUME_OR_BULK_STRAIN( "volumeOrBulkStrain", "volume or bulk strain" ),
LINEAR_EXTINCTION_COEFFICIENT( "linearExtinctionCoefficient", "linear extinction coefficient" ),
PHOTON_FLUX( "photonFlux", "photon flux" ),
RATIO_OF_THE_MASSIC_HEAT_CAPACITY( "ratioOfTheMassicHeatCapacity", "ratio of the massic heat capacity" ),
MASS_OF_ATOM_OF_A_NUCLIDE_X( "massOfAtomOfANuclideX", "mass of atom (of a nuclide x)" ),
RELATIVE_MASS_DEFECT( "relativeMassDefect", "relative mass defect" ),
PARTICLE_FLUENCE_RATE( "particleFluenceRate", "particle fluence rate" ),
HEAT_FLOW_RATE( "heatFlowRate", "heat flow rate" ),
NUCLEAR_MAGNETON_ORNUCLEUS( "nuclearMagnetonOrnucleus", "nuclear magneton ornucleus" ),
ATOMIC_ATTENUATION_COEFFICIENT( "atomicAttenuationCoefficient", "atomic attenuation coefficient" ),
FOCAL_DISTANCE( "focalDistance", "focal distance" ),
ANGLE_PLANE( "anglePlane", "angle (plane)" ),
THERMAL_CONDUCTIVITY( "thermalConductivity", "thermal conductivity" ),
PERIODIC_TIME( "periodicTime", "periodic time" ),
WORK( "work", "work" ),
PULSATANCE( "pulsatance", "pulsatance" ),
DEGREE_OF_DISSOCIATION( "degreeOfDissociation", "degree of dissociation" ),
COMPLEX_ADMITTANCE( "complexAdmittance", "complex admittance" ),
AREIC_MASS( "areicMass", "areic mass" ),
KNUDSEN_NUMBER( "knudsenNumber", "Knudsen number" ),
MOLAR_ENTROPY( "molarEntropy", "molar entropy" ),
SURFACE_DENSITY_OF_CHARGE( "surfaceDensityOfCharge", "surface density of charge" ),
LETHARGY( "lethargy", "lethargy" ),
CATALYTIC_ACTIVITY( "catalyticActivity", "catalytic activity" ),
VERGENCE( "vergence", "vergence" ),
MACROSCOPIC_TOTAL_CROSS_SECTION( "macroscopicTotalCrossSection", "macroscopic total cross-section" ),
VOLUMIC_HOLE_NUMBER( "volumicHoleNumber", "volumic hole number" ),
ANGULAR_ACCELERATION( "angularAcceleration", "angular acceleration" ),
MUTUAL_INDUCTANCE( "mutualInductance", "mutual inductance" ),
HARTREE_ENERGY( "hartreeEnergy", "Hartree energy" ),
TOTAL_LINEAR_STOPPING_POWER( "totalLinearStoppingPower", "total linear stopping power" ),
COORDINATES_TRICHROMATIC( "coordinatesTrichromatic", "coordinates trichromatic" ),
GYROMAGNETIC_COEFFICIENT( "gyromagneticCoefficient", "gyromagnetic coefficient" ),
MASS_CONCENTRATION_OF_B( "massConcentrationOfB", "mass concentration of B" ),
THERMOELECTROMOTIVE_FORCE_BETWEEN_SUBSTANCES_A_AND_B( "thermoelectromotiveForceBetweenSubstancesAAndB", "thermoelectromotive force between substances a and b" ),
SECOND_AXIAL_MOMENT_OF_AREA( "secondAxialMomentOfArea", "second axial moment of area" ),
KINETIC_ENERGY( "kineticEnergy", "kinetic energy" ),
HELMHOLTZ_FUNCTION( "helmholtzFunction", "Helmholtz function" ),
CURRENT_LINKAGE( "currentLinkage", "current linkage" ),
COMPTON_WAVELENGTH( "comptonWavelength", "Compton wavelength" ),
LUMINANCE( "luminance", "luminance" ),
SPECTRAL_TRANSMISSION_FACTOR( "spectralTransmissionFactor", "spectral transmission factor" ),
PARTITION_FUNCTION_OF_A_MOLECULE( "partitionFunctionOfAMolecule", "partition function of a molecule" ),
RECOMBINATION_COEFFICIENT( "recombinationCoefficient", "recombination coefficient" ),
MAGNETIC_FLUX_DENSITY( "magneticFluxDensity", "magnetic flux density" ),
SELF_INDUCTANCE( "selfInductance", "self inductance" ),
SCHMIDT_NUMBER( "schmidtNumber", "Schmidt number" ),
ANGULAR_WAVE_NUMBER( "angularWaveNumber", "angular wave number" ),
TRANSMITTANCE( "transmittance", "transmittance" ),
FARADAY_CONSTANT( "faradayConstant", "Faraday constant" ),
MOLAR_FLUX( "molarFlux", "molar flux" ),
BURGERS_VECTOR( "burgersVector", "Burgers vector" ),
MASSIC_ENTHALPY( "massicEnthalpy", "massic enthalpy" ),
SPECTRAL_CONCENTRATION_OF_VIBRATIONAL_MODES_IN_TERMS_OF_ANGULAR_FREQUENCY( "spectralConcentrationOfVibrationalModesInTermsOfAngularFrequency", "spectral concentration of vibrational modes (in terms of angular frequency)" ),
ELECTRIC_DIPOLE_MOMENT_OF_MOLECULE( "electricDipoleMomentOfMolecule", "electric dipole moment of molecule" ),
MASSIC_ENERGY( "massicEnergy", "massic energy" ),
VOLUME_PER_TEMPERATURE( "volumePerTemperature", "volume per temperature" ),
VELOCITY_SPEED_ON_PROPAGATION_OF_ELECTROMAGNETIC_WAVES_IN_VACUO( "velocitySpeedOnPropagationOfElectromagneticWavesInVacuo", "velocity (speed) on propagation of electromagnetic waves in vacuo" ),
LINEIC_CHARGE( "lineicCharge", "lineic charge" ),
DOSE_EQUIVALENT( "doseEquivalent", "dose equivalent" ),
HEAT( "heat", "heat" ),
DISTANCE( "distance", "distance" ),
WAVELENGTH( "wavelength", "wavelength" ),
NUCLIDIC_MASS( "nuclidicMass", "nuclidic mass" ),
MACH_NUMBER( "machNumber", "Mach number" ),
ABSORBED_DOSE( "absorbedDose", "absorbed dose" ),
CAPACITANCE( "capacitance", "capacitance" ),
PHASE_VELOCITY_OF_ELECTROMAGNETIC_WAVES( "phaseVelocityOfElectromagneticWaves", "phase velocity of electromagnetic waves" ),
IMAGE_DISTANCE( "imageDistance", "image distance" ),
CONDUCTIVITY( "conductivity", "conductivity" ),
ELECTRIC_DIPOLE_MOMENT( "electricDipoleMoment", "electric dipole moment" ),
RELATIVE_PERMITTIVITY( "relativePermittivity", "relative permittivity" ),
VOLTAGE( "voltage", "voltage" ),
DAMPING_COEFFICIENT( "dampingCoefficient", "damping coefficient" ),
ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT( "electricFluxFluxOfDisplacement", "electric flux (flux of displacement)" ),
MAGNETIC_FIELD_STRENGTH( "magneticFieldStrength", "magnetic field strength" ),
SPECIFIC_ENTHALPY( "specificEnthalpy", "specific enthalpy" ),
EQUIVALENT_ABSORPTION_AREA_OF_A_SURFACE_OR_OBJECT( "equivalentAbsorptionAreaOfASurfaceOrObject", "equivalent absorption area of a surface or object" ),
MOLAR_GAS_CONSTANT( "molarGasConstant", "molar gas constant" ),
FERMI_TEMPERATURE( "fermiTemperature", "Fermi temperature" ),
ACTIVITY_COEFFICIENT_OF_B_IN_A_LIQUID_AS_A_SOLID_MIXTURE( "activityCoefficientOfBInALiquidAsASolidMixture", "activity coefficient of B (in a liquid as a solid mixture)" ),
RESONANCE_ESCAPE_PROBABILITY( "resonanceEscapeProbability", "resonance escape probability" ),
THERMODYNAMIC( "thermodynamic", "thermodynamic" ),
SPECIFIC_VOLUME( "specificVolume", "specific volume" ),
NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES( "numberDensityOfMoleculesOrParticles", "number density of molecules  (or particles)" ),
ELECTRIC_FIELD_STRENGTH( "electricFieldStrength", "electric field strength" ),
SUPERCONDUCTOR_ENERGY_GAP( "superconductorEnergyGap", "superconductor energy gap" ),
MAGNETIC_SUSCEPTIBILITY( "magneticSusceptibility", "magnetic susceptibility" ),
SECTION_MODULUS( "sectionModulus", "section modulus" ),
VISCOSITY_DYNAMIC_VISCOSITY( "viscosityDynamicViscosity", "viscosity (dynamic viscosity)" ),
DONOR_NUMBER_DENSITY( "donorNumberDensity", "donor number density" ),
MACROSCOPIC_CROSS_SECTION( "macroscopicCrossSection", "macroscopic cross-section" ),
LINEIC_ELECTRIC_CURRENT( "lineicElectricCurrent", "lineic electric current" ),
PLANCK_FUNCTION( "planckFunction", "planck function" ),
BULK_MODULUS( "bulkModulus", "bulk modulus" ),
GAP_ENERGY( "gapEnergy", "gap energy" ),
DIFFUSION_LENGTH( "diffusionLength", "diffusion length" ),
DATA_RATE( "dataRate", "Data rate" ),
NUSSELT_NUMBER( "nusseltNumber", "Nusselt number" ),
PERMEABILITY( "permeability", "permeability" ),
SPECIFIC_ENERGY( "specificEnergy", "specific energy" ),
PHASE_COEFFICIENT( "phaseCoefficient", "phase coefficient" ),
SPIN_ANGULAR_MOMENTUM_QUANTUM_NUMBER( "spinAngularMomentumQuantumNumber", "spin angular momentum quantum number" ),
PRINCIPLE_QUANTUM_NUMBER( "principleQuantumNumber", "principle quantum number" ),
MAGNETIC_INDUCTION( "magneticInduction", "magnetic induction" ),
ACTIVITY_CONCENTRATION( "activityConcentration", "activity concentration" ),
NUCLEON_NUMBER( "nucleonNumber", "nucleon number" ),
LUMINIOUS_EFFICACY( "luminiousEfficacy", "luminious efficacy" ),
CURRENT_FRACTION_OF_ION_B( "currentFractionOfIonB", "current fraction of ion B" ),
SPECTRAL_ANGULAR_CROSS_SECTION( "spectralAngularCrossSection", "spectral angular cross-section" ),
EQUIVALENCE_DOSE_OUTPUT( "equivalenceDoseOutput", "equivalence dose output" ),
MODULUS_OF_RIGIDITY( "modulusOfRigidity", "modulus of rigidity" ),
VOLUMIC_CROSS_SECTION( "volumicCrossSection", "volumic cross-section" ),
ELECTRON_RADIUS( "electronRadius", "electron radius" ),
LUMINOUS_EFFICIENCY_AT_A_SPECIFIED_WAVELENGTH( "luminousEfficiencyAtASpecifiedWavelength", "luminous efficiency at a specified wavelength" ),
COEFFICIENT_OF_FRICTION( "coefficientOfFriction", "coefficient of friction" ),
COEFFICIENT_OF_THERMAL_INSULATION( "coefficientOfThermalInsulation", "coefficient of thermal insulation" ),
INFINITE_MEDIUM_MULTIPLICATION_FACTOR( "infiniteMediumMultiplicationFactor", "infinite medium multiplication factor" ),
COUPLING_COEFFICIENT( "couplingCoefficient", "coupling coefficient" ),
POYNTING_VECTOR( "poyntingVector", "Poynting vector" ),
FERMI_ANGULAR_REPETENCY( "fermiAngularRepetency", "Fermi angular repetency" ),
THERMAL_RESISTANCE( "thermalResistance", "thermal resistance" ),
DONOR_IONIZATION_ENERGY( "donorIonizationEnergy", "donor ionization energy" ),
POTENTIAL_DIFFERENCE( "potentialDifference", "potential difference" ),
ABSOLUTE_ACTIVITY( "absoluteActivity", "absolute activity" ),
BETA_DISINTEGRATION_ENERGY( "betaDisintegrationEnergy", "beta disintegration energy" ),
NEUTRON_SPEED( "neutronSpeed", "neutron speed" ),
MAGNETIC_QUANTUM_NUMBER( "magneticQuantumNumber", "magnetic quantum number" ),
EQUILIBRIUM_POSITION_VECTOR_OF_ION_OR_ATOM( "equilibriumPositionVectorOfIonOrAtom", "equilibrium position vector of ion or atom" ),
MAXIMUM_BETA_PARTICLE_ENERGY( "maximumBetaParticleEnergy", "maximum beta particle energy" ),
SURFACE_DENSITY( "surfaceDensity", "surface density" ),
MOBILITY_RATIO( "mobilityRatio", "mobility ratio" ),
ELEMENTARY_CHARGE( "elementaryCharge", "elementary charge" ),
MOLAR_ATTENUATION_COEFFICIENT( "molarAttenuationCoefficient", "molar attenuation coefficient" ),
MOMENT_OF_FORCE( "momentOfForce", "moment of force" ),
OSMOTIC_COEFFICIENT_OF_THE_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION( "osmoticCoefficientOfTheSolventAEspeciallyInADiluteSolution", "osmotic coefficient of the solvent A (especially in a dilute solution)" ),
MAGNETIC_POTENTIAL_DIFFERENCE( "magneticPotentialDifference", "magnetic potential difference" ),
INSTANTANEOUS_SOUND_PARTICLE_ACCELERATION( "instantaneousSoundParticleAcceleration", "(instantaneous) sound particle acceleration" ),
BOHR_MAGNETON( "bohrMagneton", "Bohr magneton" ),
VOLUMIC_TOTAL_CROSS_SECTION( "volumicTotalCrossSection", "volumic total cross-section" ),
STATISTICAL_WEIGHT( "statisticalWeight", "statistical weight" ),
RESONANCE_ENERGY( "resonanceEnergy", "resonance energy" ),
CHARGE_NUMBER_OF_ION( "chargeNumberOfIon", "charge number of ion" ),
ENERGY_FLUX_DENSITY( "energyFluxDensity", "energy flux density" ),
RADIUS( "radius", "radius" ),
PACKING_FRACTION( "packingFraction", "packing fraction" ),
ACCELERATION( "acceleration", "acceleration" ),
PRESSURE( "pressure", "pressure" ),
HOLE_NUMBER_DENSITY( "holeNumberDensity", "hole number density" ),
VOLUME_DENSITY_OF_CHARGE( "volumeDensityOfCharge", "volume density of charge" ),
MECHANICAL_IMPEDANCE( "mechanicalImpedance", "mechanical impedance" ),
VOLUMIC_DOSE( "volumicDose", "volumic dose" ),
NEUTRONFLUX_DENSITY( "neutronfluxDensity", "neutronflux density" ),
TOTAL_CROSS_SECTION( "totalCrossSection", "total cross-section" ),
THERMODYNAMIC_ENERGY( "thermodynamicEnergy", "thermodynamic energy" ),
SPECTRAL_REFLECTANCE( "spectralReflectance", "spectral reflectance" ),
VELOCITY( "velocity", "velocity" ),
MOMENTUM( "momentum", "momentum" ),
THERMODYNAMIC_CRITICAL_MAGNETIC_FLUX_DENSITY( "thermodynamicCriticalMagneticFluxDensity", "thermodynamic critical magnetic flux density" ),
MASS_NUMBER( "massNumber", "mass number" ),
ANGULAR_FREQUENCY( "angularFrequency", "angular frequency" ),
SECOND_POLAR_MOMENT_OF_AREA( "secondPolarMomentOfArea", "second polar moment of area" ),
COEFFICIENT_OF_HEAT_TRANSFER( "coefficientOfHeatTransfer", "coefficient of heat transfer" ),
FROUDE_NUMBER( "froudeNumber", "Froude number" ),
CURVATURE( "curvature", "curvature" ),
PERMEABILITY_OF_VACUUM( "permeabilityOfVacuum", "permeability of vacuum" ),
SLOWING_DOWN_LENGTH( "slowingDownLength", "slowing-down length" ),
LEAKAGE_COEFFICIENT( "leakageCoefficient", "leakage coefficient" ),
SOUND_ENERGY( "soundEnergy", "sound energy" ),
MOLAR_MASS( "molarMass", "molar mass" ),
AVOGADRO_CONSTANT( "avogadroConstant", "Avogadro constant" ),
RADIANCE( "radiance", "radiance" ),
SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME( "specificHeatCapacityAtConstantVolume", "specific heat capacity at constant volume" ),
RYDBERG_CONSTANT( "rydbergConstant", "Rydberg constant" ),
AREA( "area", "area" ),
BURST_INDEX( "burstIndex", "burst index" ),
DISSIPATION_FACTOR( "dissipationFactor", "dissipation factor" ),
SOUND_POWER( "soundPower", "sound power" ),
CONDUCTANCE_FOR_DIRECT_CURRENT( "conductanceForDirectCurrent", "conductance (for direct current)" ),
ACCEPTOR_NUMBER_DENSITY( "acceptorNumberDensity", "acceptor number density" ),
ATTENUATION_COEFFICIENT( "attenuationCoefficient", "attenuation coefficient" ),
ABSORBED_DOSE_RATE( "absorbedDoseRate", "absorbed dose rate" ),
MOMENT_OF_A_COUPLE( "momentOfACouple", "moment of a couple" ),
LATTICE_VECTOR( "latticeVector", "lattice vector" ),
OSMOTIC_PRESSURE( "osmoticPressure", "osmotic pressure" ),
MEAN_MASS_RANGE( "meanMassRange", "mean mass range" ),
GROUP_VELOCITY( "groupVelocity", "group velocity" ),
THERMAL_CONDUCTANCE( "thermalConductance", "thermal conductance" ),
DEBYE_ANGULAR_FREQUENCY( "debyeAngularFrequency", "Debye angular frequency" ),
ACCELERATION_DUE_TO_GRAVITY( "accelerationDueToGravity", "acceleration due to gravity" ),
LUMINOUS_INTENSITY( "luminousIntensity", "luminous intensity" ),
POISSON_NUMBER( "poissonNumber", "poisson number" ),
PRANDTL_NUMBER( "prandtlNumber", "Prandtl number" ),
MEAN_FREE_PATH_OF_PHONONS_OR_ELECTRONS( "meanFreePathOfPhononsOrElectrons", "mean free path of phonons or electrons" ),
NUCLEAR_PRECESSION( "nuclearPrecession", "nuclear precession" ),
VOLUMIC_ACCEPTOR_NUMBER( "volumicAcceptorNumber", "volumic acceptor number" ),
ACTIVE_POWER( "activePower", "active power" ),
SHEAR_STRAIN( "shearStrain", "shear strain" ),
ATOMIC_NUMBER( "atomicNumber", "atomic number" ),
TOTAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER( "totalAngularMomentumQuantumNumber", "total angular momentum quantum number" ),
VOLUME_FLOW_RATE( "volumeFlowRate", "volume flow rate" ),
EXPOSURE( "exposure", "exposure" ),
NUMBER_OF_MOLECULES_OR_OTHER_ELEMENTARY_ENTITIES( "numberOfMoleculesOrOtherElementaryEntities", "number of molecules or other elementary entities" ),
SURFACE_COEFFICIENT_OF_HEAT_TRANSFER( "surfaceCoefficientOfHeatTransfer", "surface coefficient of heat transfer" ),
VOLUMIC_CHARGE( "volumicCharge", "volumic charge" ),
WORK_PER_UNIT_WEIGHT( "workPerUnitWeight", "work per unit weight" ),
TORSIONAL_STIFFNESS( "torsionalStiffness", "torsional stiffness" ),
HALL_COEFFICIENT( "hallCoefficient", "Hall coefficient" ),
FORCE( "force", "force" ),
ANGULAR_VELOCITY( "angularVelocity", "angular velocity" ),
ALPHA_DISINTEGRATION_ENERGY( "alphaDisintegrationEnergy", "alpha disintegration energy" ),
DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUX_DENSITY( "diffusionCoefficientForNeutronFluxDensity", "diffusion coefficient for neutron flux density" ),
CURRENT_DENSITY_OF_PARTICLES( "currentDensityOfParticles", "current density of particles" ),
COMPLEX_IMPEDANCES( "complexImpedances", "complex impedances" ),
FOURIER_NUMBER_FOR_MASS_TRANSFER( "fourierNumberForMassTransfer", "Fourier number for mass transfer" ),
FORCE_DIVIDED_BY_LENGTH( "forceDividedByLength", "force divided by length" ),
PHOTON_LUMINANCE( "photonLuminance", "photon luminance" ),
QUANTITY_OF_LIGHT( "quantityOfLight", "quantity of light" ),
REST_MASS_OF_NEUTRON( "restMassOfNeutron", "(rest) mass of neutron" ),
LUMINOUS_FLUX( "luminousFlux", "luminous flux" ),
RELAXATION_TIME( "relaxationTime", "relaxation time" ),
CURIE_TEMPERATURE( "curieTemperature", "Curie temperature" ),
ENERGY_IMPARTED( "energyImparted", "energy imparted" ),
DEBYE_WALLE_FACTOR( "debyeWalleFactor", "Debye-Walle factor" ),
DISPLACEMENT_ELECTRIC_POLARIZATION( "displacementElectricPolarization", "displacement electric polarization" ),
STEFAN_BOLTZMANN_CONSTANT( "stefanBoltzmannConstant", "Stefan-Boltzmann constant" ),
DISINTEGRATION_CONSTANT( "disintegrationConstant", "disintegration constant" ),
FOURIER_NUMBER( "fourierNumber", "Fourier number" ),
DIFFUSION_AREA( "diffusionArea", "diffusion area" ),
MULTIPLICATION_FACTOR( "multiplicationFactor", "multiplication factor" ),
TEMPERATURE_VARIATION_OVER_TIME( "temperatureVariationOverTime", "temperature variation over time" ),
LINEAR_DENSITY( "linearDensity", "linear density" ),
ROTATORY_POWER( "rotatoryPower", "rotatory power" ),
ORDER_OF_REFLEXION( "orderOfReflexion", "order of reflexion" ),
G_FACTOR_OF_ATOM_OR_ELECTRON( "gFactorOfAtomOrElectron", "g-factor of atom or electron" ),
LEVEL_WIDTH( "levelWidth", "level width" ),
RATIO_OF_THE_SPECIFIC_HEAT_CAPACITIES( "ratioOfTheSpecificHeatCapacities", "ratio of the specific heat capacities" ),
SUPER_CONDUCTOR_TRANSITION_TEMPERATURE( "superConductorTransitionTemperature", "Super conductor transition temperature" ),
SECOND_RADIATION_CONSTANT( "secondRadiationConstant", "second radiation constant" ),
VOLUMIC_ELECTRON_NUMBER( "volumicElectronNumber", "volumic electron number" ),
SPECIFIC_THERMODYNAMIC_ENERGY( "specificThermodynamicEnergy", "specific thermodynamic energy" ),
MOLAR_CONDUCTIVITY( "molarConductivity", "molar conductivity" ),
ILLUMINANCE( "illuminance", "illuminance" ),
SOUND_INTENSITY( "soundIntensity", "sound intensity" ),
FRICTION_FACTOR( "frictionFactor", "friction factor" ),
RELATIVE_MASS_EXCESS( "relativeMassExcess", "relative mass excess" ),
VOLUME( "volume", "volume" ),
ELECTRIC_CHARGE( "electricCharge", "electric charge" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_GASEOUS_MIXTURE( "standardAbsoluteActivityOfBInAGaseousMixture", "standard absolute activity of B (in a gaseous mixture)" ),
EXCHANGE_INTERGRAL( "exchangeIntergral", "exchange intergral" ),
SPECTRAL_LUMINOUS_EFFICIENCY( "spectralLuminousEfficiency", "spectral luminous efficiency" ),
DIFFUSION_COEFFICIENT( "diffusionCoefficient", "diffusion coefficient" ),
MASS_ATTENUATION_COEFFICIENT( "massAttenuationCoefficient", "mass attenuation coefficient" ),
GRAND_CANONICAL_PARTITION_FUNCTION( "grandCanonicalPartitionFunction", "grand-canonical partition function" ),
MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA( "momentOfInertiaDynamicMomentOfInertia", "moment of inertia (dynamic moment of inertia)" ),
MODULUS_OF_IMPEDANCE( "modulusOfImpedance", "modulus of impedance" ),
CIE_COLORIMETRIC_FUNCTIONS( "cieColorimetricFunctions", "CIE colorimetric functions" ),
REACTION_ENERGY( "reactionEnergy", "reaction energy" ),
STANDARD_ABSOLUTE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION( "standardAbsoluteActivityOfSolventAEspeciallyInADiluteSolution", "standard absolute activity of solvent A (especially in a dilute solution)" ),
CIRCULAR_FREQUENCY( "circularFrequency", "circular frequency" ),
LARMOR_ANGULAR_FREQUENCY( "larmorAngularFrequency", "Larmor angular frequency" ),
ELECTROLYTIC_CONDUCTIVITY( "electrolyticConductivity", "electrolytic conductivity" ),
PHOTON_INTENSITY( "photonIntensity", "photon intensity" ),
MAGNETIC_MOMENT( "magneticMoment", "magnetic moment" ),
SPECTRAL_EMISSIVITY( "spectralEmissivity", "spectral emissivity" ),
POROSITY( "porosity", "porosity" ),
NON_LEAKAGE_PROBABILITY( "nonLeakageProbability", "non leakage probability" ),
ION_NUMBER_DENSITY( "ionNumberDensity", "ion number density" ),
MASSIC_HELMHOLTZ_FREE_ENERGY( "massicHelmholtzFreeEnergy", "massic Helmholtz free energy" ),
MICROCANONICAL_PARTITION_FUNCTION( "microcanonicalPartitionFunction", "microcanonical partition function" );

   private final String name;
   private final String label;

   QuantityKinds( final String name, final String label ) {
      this.name = name;
      this.label = label;
   }

   @Override
   public AspectModelUrn urn() {
      return AspectModelUrn.fromUrn( SammNs.UNIT.urn( name ) );
   }

   @Override
   public AspectModelFile getSourceFile() {
      return MetaModelFile.UNITS;
   }

   /**
    * Returns the quantity kind's unique name
    */
   @Override
   public String getName() {
      return name;
   }

   /**
    * Returns the quantity kind's human-readable name
    */
   @Override
   public String getLabel() {
      return label;
   }

   /**
    * Returns the quantity kind's human-readable name
    */
   @Override
   public String toString() {
      return getLabel();
   }

   @Override
   public <T, C> T accept( final AspectVisitor<T, C> visitor, final C context ) {
      return visitor.visitQuantityKind( this, context );
   }

   /**
    * Finds the quantity kind with a given name
    */
   public static Optional<QuantityKind> fromName( final String name ) {
      return Arrays.stream( QuantityKinds.values() )
            .filter( quantityKind -> quantityKind.getName().equals( name ) )
            .map( QuantityKind.class::cast )
            .findAny();
   }
}
