/*
 * Copyright (c) 2024 Robert Bosch Manufacturing Solutions GmbH
 *
 * See the AUTHORS file(s) distributed with this work for additional
 * information regarding authorship.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * SPDX-License-Identifier: MPL-2.0
 */

package org.eclipse.esmf.metamodel;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.HashSet;
import java.util.Collections;
import java.util.stream.Collectors;

import org.eclipse.esmf.aspectmodel.AspectModelFile;
import org.eclipse.esmf.aspectmodel.loader.MetaModelBaseAttributes;
import org.eclipse.esmf.aspectmodel.resolver.modelfile.MetaModelFile;
import org.eclipse.esmf.aspectmodel.urn.AspectModelUrn;
import org.eclipse.esmf.metamodel.datatype.LangString;
import org.eclipse.esmf.metamodel.impl.DefaultUnit;
import org.eclipse.esmf.metamodel.vocabulary.SammNs;

/**
 * Enumeration of Units as defined in <a href="https://tfig.unece.org/contents/recommendation-20.htm">Recommendation 20</a> by
 * the <a href="https://www.unece.org/info/ece-homepage.html">UNECE</a> (United Nations Economic Commission for Europe).
 */
public class Units {
   private static final Map<String, Unit> UNITS_BY_NAME = new HashMap<>();

   private Units() {
   }

private static void init0() {
UNITS_BY_NAME.put( "accessLine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "accessLine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "access line" ).build(), Optional.empty(), Optional.of("AL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "accountingUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "accountingUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "accounting unit" ).build(), Optional.empty(), Optional.of("E50"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "acre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "acre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "acre" ).build(), Optional.of("acre"), Optional.of("ACR"), Optional.of("squareMetre"), Optional.of("4046.873 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "acreFootBasedOnUsSurveyFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "acreFootBasedOnUsSurveyFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "acre-foot (based on U.S. survey foot)" ).build(), Optional.of("acre-ft (US survey)"), Optional.of("M67"), Optional.of("cubicMetre"), Optional.of("1.233489 \u00D7 10\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "activeUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "activeUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "active unit" ).build(), Optional.empty(), Optional.of("E25"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "activityUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "activityUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "activity" ).build(), Optional.empty(), Optional.of("ACT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "actualPer360", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "actualPer360" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "actual/360" ).build(), Optional.of("y (360 days)"), Optional.of("M37"), Optional.of("secondUnitOfTime"), Optional.of("3.1104000 \u00D7 10\u2077 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "additionalMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "additionalMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "additional minute" ).build(), Optional.empty(), Optional.of("AH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "airDryMetricTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "airDryMetricTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "air dry metric ton" ).build(), Optional.empty(), Optional.of("MD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "airDryTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "airDryTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "air dry ton" ).build(), Optional.empty(), Optional.of("E28"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "alcoholicStrengthByMass", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "alcoholicStrengthByMass" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "alcoholic strength by mass" ).build(), Optional.empty(), Optional.of("ASM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "alcoholicStrengthByVolume", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "alcoholicStrengthByVolume" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "alcoholic strength by volume" ).build(), Optional.empty(), Optional.of("ASU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere" ).build(), Optional.of("A"), Optional.of("AMP"), Optional.empty(), Optional.of("A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "ampereHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampereHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere hour" ).build(), Optional.of("A\u00B7h"), Optional.of("AMH"), Optional.of("coulomb"), Optional.of("3.6 \u00D7 10\u00B3 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "ampereMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampereMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere minute" ).build(), Optional.of("A\u00B7min"), Optional.of("N95"), Optional.of("coulomb"), Optional.of("60 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "amperePerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per centimetre" ).build(), Optional.of("A/cm"), Optional.of("A2"), Optional.of("amperePerMetre"), Optional.of("10\u00B2 A/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per kilogram" ).build(), Optional.of("A/kg"), Optional.of("H31"), Optional.empty(), Optional.of("A \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per metre" ).build(), Optional.of("A/m"), Optional.of("AE"), Optional.empty(), Optional.of("A/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIZATION, QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per millimetre" ).build(), Optional.of("A/mm"), Optional.of("A3"), Optional.of("amperePerMetre"), Optional.of("10\u00B3 A/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY, QuantityKinds.LINEIC_ELECTRIC_CURRENT)) ) );
UNITS_BY_NAME.put( "amperePerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per pascal" ).build(), Optional.of("A/Pa"), Optional.of("N93"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per square centimetre" ).build(), Optional.of("A/cm\u00B2"), Optional.of("A4"), Optional.of("amperePerSquareMetre"), Optional.of("10\u2074 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per square metre" ).build(), Optional.of("A/m\u00B2"), Optional.of("A41"), Optional.empty(), Optional.of("A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "amperePerSquareMetreKelvinSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerSquareMetreKelvinSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per square metre kelvin squared" ).build(), Optional.of("A/(m\u00B2\u00B7K\u00B2)"), Optional.of("A6"), Optional.empty(), Optional.of("A/(m\u00B2 \u00D7 K\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.RICHARDSON_CONSTANT)) ) );
UNITS_BY_NAME.put( "amperePerSquareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "amperePerSquareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere per square millimetre" ).build(), Optional.of("A/mm\u00B2"), Optional.of("A7"), Optional.of("amperePerSquareMetre"), Optional.of("10\u2076 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "ampereSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampereSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere second" ).build(), Optional.of("A\u00B7s"), Optional.of("A8"), Optional.of("coulomb"), Optional.of("C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY)) ) );
UNITS_BY_NAME.put( "ampereSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampereSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere square metre" ).build(), Optional.of("A\u00B7m\u00B2"), Optional.of("A5"), Optional.empty(), Optional.of("A \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.NUCLEAR_MAGNETON_ORNUCLEUS, QuantityKinds.BOHR_MAGNETON, QuantityKinds.ELECTROMAGNETIC_MOMENT, QuantityKinds.MAGNETIC_MOMENT, QuantityKinds.MAGNETIC_MOMENT_OF_PARTICLE, QuantityKinds.MAGNETIC_AREA_MOMENT)) ) );
UNITS_BY_NAME.put( "ampereSquareMetrePerJouleSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampereSquareMetrePerJouleSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere square metre per joule second" ).build(), Optional.of("A\u00B7m\u00B2/(J\u00B7s)"), Optional.of("A10"), Optional.empty(), Optional.of("(A \u00D7 s)/kg"), new HashSet<>(Arrays.asList(QuantityKinds.GYROMAGNETIC_RATIO, QuantityKinds.GYROMAGNETIC_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "ampereSquaredSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ampereSquaredSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ampere squared second" ).build(), Optional.of("A\u00B2\u00B7s"), Optional.of("H32"), Optional.empty(), Optional.of("A\u00B2 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "angstrom", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "angstrom" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "angstrom" ).build(), Optional.of("\u00C5"), Optional.of("A11"), Optional.of("metre"), Optional.of("10\u207B\u00B9\u2070 m"), new HashSet<>(Arrays.asList(QuantityKinds.BURGERS_VECTOR, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.WAVELENGTH, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.BOHR_RADIUS, QuantityKinds.LATTICE_PLANE_SPACING, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "antiHemophilicFactorAhfUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "antiHemophilicFactorAhfUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "anti-hemophilic factor (AHF) unit" ).build(), Optional.empty(), Optional.of("AQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "assembly", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "assembly" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "assembly" ).build(), Optional.empty(), Optional.of("AY"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "assortment", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "assortment" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "assortment" ).build(), Optional.empty(), Optional.of("AS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "astronomicalUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "astronomicalUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "astronomical unit" ).build(), Optional.of("ua"), Optional.of("A12"), Optional.of("metre"), Optional.of("1.49597870 \u00D7 10\u00B9\u00B9 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "attofarad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "attofarad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "attofarad" ).build(), Optional.of("aF"), Optional.of("H48"), Optional.empty(), Optional.of("10\u207B\u00B9\u2078 m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u2074 \u00D7 A\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "attojoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "attojoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "attojoule" ).build(), Optional.of("aJ"), Optional.of("A13"), Optional.of("joule"), Optional.of("10\u207B\u00B9\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.DONOR_IONIZATION_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ACCEPTOR_IONIZATION_ENERGY, QuantityKinds.EXCHANGE_INTERGRAL, QuantityKinds.ELECTRON_AFFINITY, QuantityKinds.SUPERCONDUCTOR_ENERGY_GAP)) ) );
UNITS_BY_NAME.put( "averageMinutePerCall", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "averageMinutePerCall" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "average minute per call" ).build(), Optional.empty(), Optional.of("AI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ball", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ball" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ball" ).build(), Optional.empty(), Optional.of("AA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "barCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bar cubic metre per second" ).build(), Optional.of("bar\u00B7m\u00B3/s"), Optional.of("F92"), Optional.empty(), Optional.of("10\u2075 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "barLitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barLitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bar litre per second" ).build(), Optional.of("bar\u00B7l/s"), Optional.of("F91"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "barPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bar per bar" ).build(), Optional.of("bar/bar"), Optional.of("J56"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "barPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bar per kelvin" ).build(), Optional.of("bar/K"), Optional.of("F81"), Optional.empty(), Optional.of("10\u2075 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "barUnitOfPressure", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barUnitOfPressure" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bar [unit of pressure]" ).build(), Optional.of("bar"), Optional.of("BAR"), Optional.of("pascal"), Optional.of("10\u2075 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.INSTANTANEOUS_SOUND_PRESSURE, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.STATIC_PRESSURE)) ) );
UNITS_BY_NAME.put( "barn", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barn" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barn" ).build(), Optional.of("b"), Optional.of("A14"), Optional.of("squareMetre"), Optional.of("10\u207B\u00B2\u2078 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barnPerElectronvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barnPerElectronvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barn per electronvolt" ).build(), Optional.of("b/eV"), Optional.of("A15"), Optional.of("squareMetrePerJoule"), Optional.of("6.241 51 \u00D7 10\u207B\u00B9\u2070 m\u00B2/J"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barnPerSteradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barnPerSteradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barn per steradian" ).build(), Optional.of("b/sr"), Optional.of("A17"), Optional.of("squareMetrePerSteradian"), Optional.of("1 \u00D7 10\u207B\u00B2\u2078 m\u00B2/sr"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barnPerSteradianElectronvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barnPerSteradianElectronvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barn per steradian electronvolt" ).build(), Optional.of("b/(sr\u00B7eV)"), Optional.of("A16"), Optional.empty(), Optional.of("6.241 51 \u00D7 10\u207B\u00B9\u2070 m\u00B2/(sr \u00D7J)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "barrelImperial", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelImperial" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel, imperial" ).build(), Optional.empty(), Optional.of("B4"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "barrelUkPetroleum", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUkPetroleum" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (UK petroleum)" ).build(), Optional.of("bbl (UK liq.)"), Optional.of("J57"), Optional.of("cubicMetre"), Optional.of("0.15911315 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUkPetroleumPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (UK petroleum) per day" ).build(), Optional.of("bbl (UK liq.)/d"), Optional.of("J59"), Optional.of("cubicMetrePerSecond"), Optional.of("1.8415874 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUkPetroleumPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (UK petroleum) per hour" ).build(), Optional.of("bbl (UK liq.)/h"), Optional.of("J60"), Optional.of("cubicMetrePerSecond"), Optional.of("4.419810 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUkPetroleumPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (UK petroleum) per minute" ).build(), Optional.of("bbl (UK liq.)/min"), Optional.of("J58"), Optional.of("cubicMetrePerSecond"), Optional.of("2.651886 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUkPetroleumPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUkPetroleumPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (UK petroleum) per second" ).build(), Optional.of("bbl (UK liq.)/s"), Optional.of("J61"), Optional.of("cubicMetrePerSecond"), Optional.of("0.15911315 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (US)" ).build(), Optional.of("barrel (US)"), Optional.of("BLL"), Optional.of("cubicMetre"), Optional.of("158.9873 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "barrelUsPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUsPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (US) per day" ).build(), Optional.of("barrel\u00A0(US)/d"), Optional.of("B1"), Optional.of("cubicMetrePerSecond"), Optional.of("1.84013 \u00D7 10\u207B\u2076 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "barrelUsPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUsPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (US) per minute" ).build(), Optional.of("barrel (US)/min"), Optional.of("5A"), Optional.of("cubicMetrePerSecond"), Optional.of("2.64979 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUsPetroleumPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUsPetroleumPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (US petroleum) per hour" ).build(), Optional.of("bbl (US)/h"), Optional.of("J62"), Optional.of("cubicMetrePerSecond"), Optional.of("4.416314 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "barrelUsPetroleumPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "barrelUsPetroleumPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "barrel (US petroleum) per second" ).build(), Optional.of("bbl (US)/s"), Optional.of("J63"), Optional.of("cubicMetrePerSecond"), Optional.of("0.1589873 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "baseBox", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "baseBox" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "base box" ).build(), Optional.empty(), Optional.of("BB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "batch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "batch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "batch" ).build(), Optional.empty(), Optional.of("5B"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "battingPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "battingPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "batting pound" ).build(), Optional.empty(), Optional.of("B3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "baud", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "baud" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "baud" ).build(), Optional.of("Bd"), Optional.of("J38"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "beaufort", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "beaufort" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Beaufort" ).build(), Optional.of("Bft"), Optional.of("M19"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "becquerel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "becquerel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "becquerel" ).build(), Optional.of("Bq"), Optional.of("BQL"), Optional.of("curie"), Optional.of("27.027 \u00D7 10\u207B\u00B9\u00B2 Ci"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "becquerelPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "becquerelPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "becquerel per cubic metre" ).build(), Optional.of("Bq/m\u00B3"), Optional.of("A19"), Optional.empty(), Optional.of("Bq/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_ACTIVITY, QuantityKinds.ACTIVITY_CONCENTRATION)) ) );
UNITS_BY_NAME.put( "becquerelPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "becquerelPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "becquerel per kilogram" ).build(), Optional.of("Bq/kg"), Optional.of("A18"), Optional.of("curiePerKilogram"), Optional.of("27.027 \u00D7 10\u207B\u00B9\u00B2 Ci/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "bel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bel" ).build(), Optional.of("B"), Optional.of("M72"), Optional.empty(), Optional.of("B"), new HashSet<>(Arrays.asList(QuantityKinds.LEVEL_OF_A_POWER_QUANTITY, QuantityKinds.LEVEL_OF_A_FIELD_QUANTITY)) ) );
UNITS_BY_NAME.put( "belPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "belPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bel per metre" ).build(), Optional.of("B/m"), Optional.of("P43"), Optional.empty(), Optional.of("B/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_POWER_LEVEL)) ) );
UNITS_BY_NAME.put( "bigPoint", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bigPoint" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "big point" ).build(), Optional.of("bp"), Optional.of("H82"), Optional.of("metre"), Optional.of("0.3527778 \u00D7 10\u207B\u00B3 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "billionEur", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "billionEur" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "billion (EUR)" ).build(), Optional.empty(), Optional.of("BIL"), Optional.empty(), Optional.of("10\u00B9\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "biot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "biot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "biot" ).build(), Optional.of("Bi"), Optional.of("N96"), Optional.of("ampere"), Optional.of("10\u00B9 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "bit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bit" ).build(), Optional.of("b"), Optional.of("A99"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "bitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bit per cubic metre" ).build(), Optional.of("bit/m\u00B3"), Optional.of("F01"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bit per metre" ).build(), Optional.of("bit/m"), Optional.of("E88"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bit per second" ).build(), Optional.of("bit/s"), Optional.of("B10"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bit per square metre" ).build(), Optional.of("bit/m\u00B2"), Optional.of("E89"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "blank", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "blank" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "blank" ).build(), Optional.empty(), Optional.of("H21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "boardFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "boardFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "board foot" ).build(), Optional.of("fbm"), Optional.of("BFT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "book", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "book" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "book" ).build(), Optional.empty(), Optional.of("D63"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "brakeHorsePower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "brakeHorsePower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "brake horse power" ).build(), Optional.of("BHP"), Optional.of("BHP"), Optional.of("watt"), Optional.of("7.457 \u00D7 10\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "britishThermalUnit39Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnit39Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (39 °F)" ).build(), Optional.of("Btu (39 \u00B0F)"), Optional.of("N66"), Optional.of("joule"), Optional.of("1.05967 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnit59Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnit59Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (59 °F)" ).build(), Optional.of("Btu (59 \u00B0F)"), Optional.of("N67"), Optional.of("joule"), Optional.of("1.05480 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENTHALPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnit60Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnit60Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (60 °F)" ).build(), Optional.of("Btu (60 \u00B0F)"), Optional.of("N68"), Optional.of("joule"), Optional.of("1.05468 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.ENTHALPY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTable", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTable" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table)" ).build(), Optional.of("BtuIT"), Optional.of("BTU"), Optional.of("joule"), Optional.of("1.055056 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTableFootPerHourSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTableFootPerHourSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) foot per hour square foot degree Fahrenheit" ).build(), Optional.of("BtuIT\u00B7ft/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J40"), Optional.of("wattPerMetreKelvin"), Optional.of("1.730735 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTableInchPerHourSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTableInchPerHourSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) inch per hour square foot degree Fahrenheit" ).build(), Optional.of("BtuIT\u00B7in/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J41"), Optional.of("wattPerMetreKelvin"), Optional.of("0.1442279 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTableInchPerSecondSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTableInchPerSecondSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) inch per second square foot degree Fahrenheit" ).build(), Optional.of("BtuIT\u00B7in/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J42"), Optional.of("wattPerMetreKelvin"), Optional.of("5.192204 \u00D7 10\u00B2 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per cubic foot" ).build(), Optional.of("BtuIT/ft\u00B3"), Optional.of("N58"), Optional.of("joulePerCubicMetre"), Optional.of("3.725895 \u00D710\u2074 J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per degree Fahrenheit" ).build(), Optional.of("BtuIT/\u00B0F"), Optional.of("N60"), Optional.of("joulePerKelvin"), Optional.of("1.899101 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per degree Rankine" ).build(), Optional.of("BtuIT/\u00B0R"), Optional.of("N62"), Optional.of("joulePerKelvin"), Optional.of("1.899101 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per hour" ).build(), Optional.of("BtuIT/h"), Optional.of("2I"), Optional.of("watt"), Optional.of("2.930711\u00D7 10\u207B\u00B9 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerHourSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerHourSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per hour square foot degree Fahrenheit" ).build(), Optional.of("BtuIT/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N74"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("5.678263 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerHourSquareFootDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerHourSquareFootDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per hour square foot degree Rankine" ).build(), Optional.of("BtuIT/(h\u00B7ft\u00B2\u00B7\u00B0R)"), Optional.of("A23"), Optional.empty(), Optional.of("5.67826 W/ (m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per minute" ).build(), Optional.of("BtuIT/min"), Optional.of("J44"), Optional.of("watt"), Optional.of("17.584266 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per pound" ).build(), Optional.of("BtuIT/lb"), Optional.of("AZ"), Optional.of("joulePerKilogram"), Optional.of("2326 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASSIC_HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerPoundDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerPoundDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per pound degree Fahrenheit" ).build(), Optional.of("BtuIT/(lb\u00B7\u00B0F)"), Optional.of("J43"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.1868 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerPoundDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerPoundDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per pound degree Rankine" ).build(), Optional.of("Btu/IT(lb\u00B7\u00B0R)"), Optional.of("A21"), Optional.of("joulePerKilogramKelvin"), Optional.of("4186.8 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_SATURATION)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per second" ).build(), Optional.of("BtuIT/s"), Optional.of("J45"), Optional.of("watt"), Optional.of("1.055056 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecondFootDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSecondFootDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per second foot degree Rankine" ).build(), Optional.of("BtuIT/(s\u00B7ft\u00B7\u00B0R)"), Optional.of("A22"), Optional.of("wattPerMetreKelvin"), Optional.of("6230.64 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecondSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSecondSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per second square foot degree Fahrenheit" ).build(), Optional.of("BtuIT/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N76"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("2.044175 \u00D7 10\u2074 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSecondSquareFootDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSecondSquareFootDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per second square foot degree Rankine" ).build(), Optional.of("BtuIT/(s\u00B7ft\u00B2\u00B7\u00B0R)"), Optional.of("A20"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("20441.7 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
}

private static void init1() {
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per square foot" ).build(), Optional.of("BtuIT/ft\u00B2"), Optional.of("P37"), Optional.of("joulePerSquareMetre"), Optional.of("1.135653 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareFootHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSquareFootHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per square foot hour" ).build(), Optional.of("BtuIT/(ft\u00B2\u00B7h)"), Optional.of("N50"), Optional.of("wattPerSquareMetre"), Optional.of("3.154591 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareFootSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSquareFootSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per square foot second" ).build(), Optional.of("BtuIT/(ft\u00B2\u00B7s)"), Optional.of("N53"), Optional.of("wattPerSquareMetre"), Optional.of("1.135653 \u00D7 10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitInternationalTablePerSquareInchSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitInternationalTablePerSquareInchSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (international table) per square inch second" ).build(), Optional.of("BtuIT/(in\u00B2\u00B7s)"), Optional.of("N55"), Optional.of("wattPerSquareMetre"), Optional.of("1.634246 \u00D7 10\u2076 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitMean", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitMean" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (mean)" ).build(), Optional.of("Btu"), Optional.of("J39"), Optional.of("joule"), Optional.of("1.05587 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalFootPerHourSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalFootPerHourSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) foot per hour square foot degree Fahrenheit" ).build(), Optional.of("Btuth\u00B7ft/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J46"), Optional.of("wattPerMetreKelvin"), Optional.of("1.729577 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalInchPerHourSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalInchPerHourSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) inch per hour square foot degree Fahrenheit" ).build(), Optional.of("Btuth\u00B7in/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J48"), Optional.of("wattPerMetreKelvin"), Optional.of("0.1441314 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalInchPerSecondSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalInchPerSecondSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) inch per second square foot degree Fahrenheit" ).build(), Optional.of("Btuth\u00B7in/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("J49"), Optional.of("wattPerMetreKelvin"), Optional.of("5.188732 \u00D7 10\u00B2 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per cubic foot" ).build(), Optional.of("Btuth/ft\u00B3"), Optional.of("N59"), Optional.of("joulePerCubicMetre"), Optional.of("3.723403 \u00D710\u2074 J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per degree Fahrenheit" ).build(), Optional.of("Btuth/\u00B0F"), Optional.of("N61"), Optional.of("joulePerKelvin"), Optional.of("1.897830 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per degree Rankine" ).build(), Optional.of("Btuth/\u00B0R"), Optional.of("N63"), Optional.of("joulePerKelvin"), Optional.of("1.897830 \u00D7 10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per hour" ).build(), Optional.of("Btuth/h"), Optional.of("J47"), Optional.of("watt"), Optional.of("0.2928751 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerHourSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerHourSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per hour square foot degree Fahrenheit" ).build(), Optional.of("Btuth/(h\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N75"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("5.674466 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per minute" ).build(), Optional.of("Btuth/min"), Optional.of("J51"), Optional.of("watt"), Optional.of("17.57250 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per pound" ).build(), Optional.of("Btuth/lb"), Optional.of("N73"), Optional.of("joulePerKilogram"), Optional.of("2.324444 \u00D7 10\u00B3 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASSIC_HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerPoundDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerPoundDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per pound degree Fahrenheit" ).build(), Optional.of("Btuth/(lb\u00B7\u00B0F)"), Optional.of("J50"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.184 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerPoundDegreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerPoundDegreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per pound degree Rankine" ).build(), Optional.of("(Btuth/\u00B0R)/lb"), Optional.of("N64"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.184 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per second" ).build(), Optional.of("Btuth/s"), Optional.of("J52"), Optional.of("watt"), Optional.of("1.054350 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSecondSquareFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerSecondSquareFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per second square foot degree Fahrenheit" ).build(), Optional.of("Btuth/(s\u00B7ft\u00B2\u00B7\u00B0F)"), Optional.of("N77"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("2.042808 \u00D7 10\u2074 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per square foot" ).build(), Optional.of("Btuth/ft\u00B2"), Optional.of("P38"), Optional.of("joulePerSquareMetre"), Optional.of("1.134893 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFootHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerSquareFootHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per square foot hour" ).build(), Optional.of("Btuth/(ft\u00B2\u00B7h)"), Optional.of("N51"), Optional.of("wattPerSquareMetre"), Optional.of("3.152481 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFootMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerSquareFootMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per square foot minute" ).build(), Optional.of("Btuth/(ft\u00B2\u00B7min)"), Optional.of("N52"), Optional.of("wattPerSquareMetre"), Optional.of("1.891489 \u00D7 10\u00B2 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "britishThermalUnitThermochemicalPerSquareFootSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "britishThermalUnitThermochemicalPerSquareFootSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "British thermal unit (thermochemical) per square foot second" ).build(), Optional.of("Btuth/(ft\u00B2\u00B7s)"), Optional.of("N54"), Optional.of("wattPerSquareMetre"), Optional.of("1.134893 \u00D7 10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bulkPack", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bulkPack" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bulk pack" ).build(), Optional.of("pk"), Optional.of("AB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "bushelUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (UK)" ).build(), Optional.of("bushel (UK)"), Optional.of("BUI"), Optional.of("cubicMetre"), Optional.of("3.636872 \u00D7 10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "bushelUkPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUkPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (UK) per day" ).build(), Optional.of("bu (UK)/d"), Optional.of("J64"), Optional.of("cubicMetrePerSecond"), Optional.of("4.209343 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUkPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUkPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (UK) per hour" ).build(), Optional.of("bu (UK)/h"), Optional.of("J65"), Optional.of("cubicMetrePerSecond"), Optional.of("1.010242 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUkPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUkPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (UK) per minute" ).build(), Optional.of("bu (UK)/min"), Optional.of("J66"), Optional.of("cubicMetrePerSecond"), Optional.of("6.061453 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUkPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUkPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (UK) per second" ).build(), Optional.of("bu (UK)/s"), Optional.of("J67"), Optional.of("cubicMetrePerSecond"), Optional.of("3.636872 \u00D7 10\u207B\u00B2 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (US)" ).build(), Optional.of("bu (US)"), Optional.of("BUA"), Optional.of("cubicMetre"), Optional.of("3.523907 \u00D7 10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUsDryPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (US dry) per day" ).build(), Optional.of("bu (US dry)/d"), Optional.of("J68"), Optional.of("cubicMetrePerSecond"), Optional.of("4.078596 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUsDryPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (US dry) per hour" ).build(), Optional.of("bu (US dry)/h"), Optional.of("J69"), Optional.of("cubicMetrePerSecond"), Optional.of("9.788631 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUsDryPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (US dry) per minute" ).build(), Optional.of("bu (US dry)/min"), Optional.of("J70"), Optional.of("cubicMetrePerSecond"), Optional.of("5.873178 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "bushelUsDryPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bushelUsDryPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "bushel (US dry) per second" ).build(), Optional.of("bu (US dry)/s"), Optional.of("J71"), Optional.of("cubicMetrePerSecond"), Optional.of("3.523907 \u00D7 10\u207B\u00B2 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "byte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "byte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "byte" ).build(), Optional.of("B"), Optional.of("AD"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "bytePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "bytePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "byte per second" ).build(), Optional.of("byte/s"), Optional.of("P93"), Optional.empty(), Optional.of("byte/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cake", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cake" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cake" ).build(), Optional.empty(), Optional.of("KA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "call", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "call" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "call" ).build(), Optional.empty(), Optional.of("C0"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "calorie20Degreesc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorie20Degreesc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (20 °C)" ).build(), Optional.of("cal\u2082\u2080"), Optional.of("N69"), Optional.of("joule"), Optional.of("4.18190"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "calorieInternationalTablePerGramDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieInternationalTablePerGramDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (international table) per gram degree Celsius" ).build(), Optional.of("calIT/(g\u00B7\u00B0C)"), Optional.of("J76"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.1868 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "calorieMean", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieMean" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (mean)" ).build(), Optional.of("cal"), Optional.of("J75"), Optional.of("joule"), Optional.of("4.19002 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerCentimetreSecondDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerCentimetreSecondDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per centimetre second degree Celsius" ).build(), Optional.of("calth/(cm\u00B7s\u00B7\u00B0C)"), Optional.of("J78"), Optional.of("wattPerMetreKelvin"), Optional.of("4.184 \u00D7 10\u00B2 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerGramDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerGramDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per gram degree Celsius" ).build(), Optional.of("calth/(g\u00B7\u00B0C)"), Optional.of("J79"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.184 \u00D7 10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per minute" ).build(), Optional.of("calth/min"), Optional.of("J81"), Optional.of("watt"), Optional.of("6.973333 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per second" ).build(), Optional.of("calth/s"), Optional.of("J82"), Optional.of("watt"), Optional.of("4.184 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per square centimetre" ).build(), Optional.of("calth/cm\u00B2"), Optional.of("P39"), Optional.of("joulePerSquareMetre"), Optional.of("4.184 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSquareCentimetreMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerSquareCentimetreMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per square centimetre minute" ).build(), Optional.of("calth/(cm\u00B2\u00B7min)"), Optional.of("N56"), Optional.of("wattPerSquareMetre"), Optional.of("6.973333 \u00D7 10\u00B2 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "calorieThermochemicalPerSquareCentimetreSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "calorieThermochemicalPerSquareCentimetreSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "calorie (thermochemical) per square centimetre second" ).build(), Optional.of("calth/(cm\u00B2\u00B7s)"), Optional.of("N57"), Optional.of("wattPerSquareMetre"), Optional.of("4.184 \u00D7 10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "candela", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "candela" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "candela" ).build(), Optional.of("cd"), Optional.of("CDL"), Optional.empty(), Optional.of("cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "candelaPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "candelaPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "candela per square foot" ).build(), Optional.of("cd/ft\u00B2"), Optional.of("P32"), Optional.of("candelaPerSquareMetre"), Optional.of("1.076391 \u00D7 10 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "candelaPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "candelaPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "candela per square inch" ).build(), Optional.of("cd/in\u00B2"), Optional.of("P28"), Optional.of("candelaPerSquareMetre"), Optional.of("1.550003 \u00D7 10\u00B3 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "candelaPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "candelaPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "candela per square metre" ).build(), Optional.of("cd/m\u00B2"), Optional.of("A24"), Optional.empty(), Optional.of("cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "card", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "card" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "card" ).build(), Optional.empty(), Optional.of("CG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "carryingCapacityInMetricTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "carryingCapacityInMetricTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "carrying capacity in metric ton" ).build(), Optional.empty(), Optional.of("CCT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "centalUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centalUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cental (UK)" ).build(), Optional.empty(), Optional.of("CNT"), Optional.of("kilogram"), Optional.of("45.359237 kg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "centigram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centigram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centigram" ).build(), Optional.of("cg"), Optional.of("CGM"), Optional.of("kilogram"), Optional.of("10\u207B\u2075 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "centilitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centilitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centilitre" ).build(), Optional.of("cl"), Optional.of("CLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "centimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre" ).build(), Optional.of("cm"), Optional.of("CMT"), Optional.of("metre"), Optional.of("10\u207B\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "centimetreOfMercury0Degreesc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetreOfMercury0Degreesc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre of mercury (0 °C)" ).build(), Optional.of("cmHg (0 \u00B0C)"), Optional.of("N13"), Optional.of("pascal"), Optional.of("1.33322 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "centimetreOfWater4Degreesc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetreOfWater4Degreesc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre of water (4 °C)" ).build(), Optional.of("cmH\u2082O (4 \u00B0C)"), Optional.of("N14"), Optional.of("pascal"), Optional.of("9.80638 \u00D7 10 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "centimetrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per bar" ).build(), Optional.of("cm/bar"), Optional.of("G04"), Optional.empty(), Optional.of("10\u207B\u2077 kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "centimetrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per hour" ).build(), Optional.of("cm/h"), Optional.of("H49"), Optional.empty(), Optional.of("0.277777778 \u00D7 10\u207B\u2076 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per kelvin" ).build(), Optional.of("cm/K"), Optional.of("F51"), Optional.empty(), Optional.of("10\u207B\u00B2 m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.RADIUS, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "centimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per second" ).build(), Optional.of("cm/s"), Optional.of("2M"), Optional.of("metrePerSecond"), Optional.of("10\u207B\u00B2 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per second bar" ).build(), Optional.of("(cm/s)/bar"), Optional.of("J85"), Optional.of("metrePerSecondPascal"), Optional.of("10\u207B\u2077 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per second kelvin" ).build(), Optional.of("(cm/s)/K"), Optional.of("J84"), Optional.of("metrePerSecondKelvin"), Optional.of("10\u207B\u00B2 (m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "centimetrePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centimetrePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centimetre per second squared" ).build(), Optional.of("cm/s\u00B2"), Optional.of("M39"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u00B2 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "centinewtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centinewtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centinewton metre" ).build(), Optional.of("cN\u00B7m"), Optional.of("J72"), Optional.of("newtonMetre"), Optional.of("10\u207B\u00B2 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "centipoise", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centipoise" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centipoise" ).build(), Optional.of("cP"), Optional.of("C7"), Optional.of("pascalSecond"), Optional.of("10\u207B\u00B3 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "centipoisePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centipoisePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centipoise per bar" ).build(), Optional.of("cP/bar"), Optional.of("J74"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2078 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "centipoisePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centipoisePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centipoise per kelvin" ).build(), Optional.of("cP/K"), Optional.of("J73"), Optional.empty(), Optional.of("10\u207B\u00B3 Pa \u00D7 s/K"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "centistokes", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "centistokes" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "centistokes" ).build(), Optional.of("cSt"), Optional.of("4C"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "chainBasedOnUsSurveyFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "chainBasedOnUsSurveyFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "chain (based on U.S. survey foot)" ).build(), Optional.of("ch (US survey)"), Optional.of("M49"), Optional.of("metre"), Optional.of("2.011684 \u00D7 10 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.DISTANCE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "circularMil", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "circularMil" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "circular mil" ).build(), Optional.of("cmil"), Optional.of("M47"), Optional.of("squareMetre"), Optional.of("5.067075 \u00D7 10\u207B\u00B9\u2070 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "clo", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "clo" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "clo" ).build(), Optional.of("clo"), Optional.of("J83"), Optional.of("squareMetreKelvinPerWatt"), Optional.of("0.155 m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "coilGroup", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coilGroup" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coil group" ).build(), Optional.empty(), Optional.of("C9"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "commonYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "commonYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "common year" ).build(), Optional.of("y (365 days)"), Optional.of("L95"), Optional.of("secondUnitOfTime"), Optional.of("3.1536 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "contentGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "contentGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "content gram" ).build(), Optional.empty(), Optional.of("CTG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "contentTonMetric", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "contentTonMetric" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "content ton (metric)" ).build(), Optional.empty(), Optional.of("CTN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "conventionalMetreOfWater", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "conventionalMetreOfWater" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "conventional metre of water" ).build(), Optional.of("mH\u2082O"), Optional.of("N23"), Optional.of("pascal"), Optional.of("9.80665 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "cord", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cord" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cord" ).build(), Optional.empty(), Optional.of("WCD"), Optional.of("cubicMetre"), Optional.of("3.63 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cord128Ft3", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cord128Ft3" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cord (128 ft3)" ).build(), Optional.of("cord"), Optional.of("M68"), Optional.of("cubicMetre"), Optional.of("3.624556 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "coulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb" ).build(), Optional.of("C"), Optional.of("COU"), Optional.of("ampereSecond"), Optional.of("A \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELEMENTARY_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "coulombMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb metre" ).build(), Optional.of("C\u00B7m"), Optional.of("A26"), Optional.empty(), Optional.of("A \u00D7 s \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_DIPOLE_MOMENT_OF_MOLECULE, QuantityKinds.ELECTRIC_DIPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "coulombMetreSquaredPerVolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombMetreSquaredPerVolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb metre squared per volt" ).build(), Optional.of("C\u00B7m\u00B2/V"), Optional.of("A27"), Optional.empty(), Optional.of("A\u00B2 \u00D7 s\u2074/kg"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_POLARIZABILITY_OF_A_MOLECULE)) ) );
UNITS_BY_NAME.put( "coulombPerCubicCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerCubicCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per cubic centimetre" ).build(), Optional.of("C/cm\u00B3"), Optional.of("A28"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2076 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE)) ) );
UNITS_BY_NAME.put( "coulombPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per cubic metre" ).build(), Optional.of("C/m\u00B3"), Optional.of("A29"), Optional.empty(), Optional.of("C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "coulombPerCubicMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerCubicMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per cubic millimetre" ).build(), Optional.of("C/mm\u00B3"), Optional.of("A30"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2079 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUME_DENSITY_OF_CHARGE)) ) );
UNITS_BY_NAME.put( "coulombPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per kilogram" ).build(), Optional.of("C/kg"), Optional.of("CKG"), Optional.empty(), Optional.of("A \u00D7 s/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "coulombPerKilogramSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerKilogramSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per kilogram second" ).build(), Optional.of("C/(kg\u00B7s)"), Optional.of("A31"), Optional.of("amperePerKilogram"), Optional.of("A/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE_RATE)) ) );
UNITS_BY_NAME.put( "coulombPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per metre" ).build(), Optional.of("C/m"), Optional.of("P10"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 s \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_CHARGE)) ) );
UNITS_BY_NAME.put( "coulombPerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per mole" ).build(), Optional.of("C/mol"), Optional.of("A32"), Optional.empty(), Optional.of("A \u00D7 s/mol"), new HashSet<>(Arrays.asList(QuantityKinds.FARADAY_CONSTANT)) ) );
UNITS_BY_NAME.put( "coulombPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per square centimetre" ).build(), Optional.of("C/cm\u00B2"), Optional.of("A33"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u2074 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "coulombPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per square metre" ).build(), Optional.of("C/m\u00B2"), Optional.of("A34"), Optional.empty(), Optional.of("C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "coulombPerSquareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombPerSquareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb per square millimetre" ).build(), Optional.of("C/mm\u00B2"), Optional.of("A35"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u2076 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "coulombSquareMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "coulombSquareMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "coulomb square metre per kilogram" ).build(), Optional.of("C\u00B7m\u00B2/kg"), Optional.of("J53"), Optional.empty(), Optional.of("C \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "credit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "credit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "credit" ).build(), Optional.empty(), Optional.of("B17"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cubicCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre" ).build(), Optional.of("cm\u00B3"), Optional.of("CMQ"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per bar" ).build(), Optional.of("cm\u00B3/bar"), Optional.of("G94"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per cubic metre" ).build(), Optional.of("cm\u00B3/m\u00B3"), Optional.of("J87"), Optional.empty(), Optional.of("10\u207B\u2076"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
}

private static void init2() {
UNITS_BY_NAME.put( "cubicCentimetrePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per day" ).build(), Optional.of("cm\u00B3/d"), Optional.of("G47"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per day bar" ).build(), Optional.of("cm\u00B3/(d\u00B7bar)"), Optional.of("G78"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2076 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per day kelvin" ).build(), Optional.of("cm\u00B3/(d\u00B7K)"), Optional.of("G61"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per hour" ).build(), Optional.of("cm\u00B3/h"), Optional.of("G48"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per hour bar" ).build(), Optional.of("cm\u00B3/(h\u00B7bar)"), Optional.of("G79"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2075 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per hour kelvin" ).build(), Optional.of("cm\u00B3/(h\u00B7K)"), Optional.of("G62"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per kelvin" ).build(), Optional.of("cm\u00B3/K"), Optional.of("G27"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per minute" ).build(), Optional.of("cm\u00B3/min"), Optional.of("G49"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per minute bar" ).build(), Optional.of("cm\u00B3/(min\u00B7bar)"), Optional.of("G80"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B3 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per minute kelvin" ).build(), Optional.of("cm\u00B3/(min\u00B7K)"), Optional.of("G63"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per mole" ).build(), Optional.of("cm\u00B3/mol"), Optional.of("A36"), Optional.of("cubicMetrePerMole"), Optional.of("10\u207B\u2076 m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per second" ).build(), Optional.of("cm\u00B3/s"), Optional.of("2J"), Optional.of("cubicMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per second bar" ).build(), Optional.of("cm\u00B3/(s\u00B7bar)"), Optional.of("G81"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicCentimetrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicCentimetrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic centimetre per second kelvin" ).build(), Optional.of("cm\u00B3/(s\u00B7K)"), Optional.of("G64"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecametre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecametre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decametre" ).build(), Optional.of("dam\u00B3"), Optional.of("DMA"), Optional.of("cubicMetre"), Optional.of("10\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre" ).build(), Optional.of("dm\u00B3"), Optional.of("DMQ"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per cubic metre" ).build(), Optional.of("dm\u00B3/m\u00B3"), Optional.of("J91"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per day" ).build(), Optional.of("dm\u00B3/d"), Optional.of("J90"), Optional.of("cubicMetrePerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per hour" ).build(), Optional.of("dm\u00B3/h"), Optional.of("E92"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per kilogram" ).build(), Optional.of("dm\u00B3/kg"), Optional.of("N28"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per minute" ).build(), Optional.of("dm\u00B3/min"), Optional.of("J92"), Optional.of("cubicMetrePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per mole" ).build(), Optional.of("dm\u00B3/mol"), Optional.of("A37"), Optional.of("cubicMetrePerMole"), Optional.of("10\u207B\u00B3 m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicDecimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicDecimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic decimetre per second" ).build(), Optional.of("dm\u00B3/s"), Optional.of("J93"), Optional.of("cubicMetrePerSecond"), Optional.of("10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot" ).build(), Optional.of("ft\u00B3"), Optional.of("FTQ"), Optional.of("cubicMetre"), Optional.of("2.831685 \u00D7 10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per day" ).build(), Optional.of("ft\u00B3/d"), Optional.of("K22"), Optional.of("cubicMetrePerSecond"), Optional.of("3.277413 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFootPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per degree Fahrenheit" ).build(), Optional.of("ft\u00B3/\u00B0F"), Optional.of("K21"), Optional.of("cubicMetrePerKelvin"), Optional.of("5.097033 \u00D7 10\u207B\u00B2 m\u00B3/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per hour" ).build(), Optional.of("ft\u00B3/h"), Optional.of("2K"), Optional.of("cubicMetrePerSecond"), Optional.of("7.86579 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFootPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per minute" ).build(), Optional.of("ft\u00B3/min"), Optional.of("2L"), Optional.of("cubicMetrePerSecond"), Optional.of("4.719474 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicFootPerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per pound" ).build(), Optional.of("ft\u00B3/lb"), Optional.of("N29"), Optional.of("cubicMetrePerKilogram"), Optional.of("6.242796 \u00D7 10\u207B\u00B2 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per psi" ).build(), Optional.of("ft\u00B3/psi"), Optional.of("K23"), Optional.of("cubicMetrePerPascal"), Optional.of("4.107012 \u00D7 10\u207B\u2076 m\u00B3/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicFootPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicFootPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic foot per second" ).build(), Optional.of("ft\u00B3/s"), Optional.of("E17"), Optional.of("cubicMetrePerSecond"), Optional.of("2.831685 \u00D7 10\u207B\u00B2 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "cubicHectometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicHectometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic hectometre" ).build(), Optional.of("hm\u00B3"), Optional.of("H19"), Optional.of("cubicMetre"), Optional.of("10\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic inch" ).build(), Optional.of("in\u00B3"), Optional.of("INQ"), Optional.of("cubicMetre"), Optional.of("16.387064 \u00D7 10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME, QuantityKinds.SECTION_MODULUS)) ) );
UNITS_BY_NAME.put( "cubicInchPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicInchPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic inch per hour" ).build(), Optional.of("in\u00B3/h"), Optional.of("G56"), Optional.empty(), Optional.of("4.55196 \u00D7 10\u207B\u2079 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicInchPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicInchPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic inch per minute" ).build(), Optional.of("in\u00B3/min"), Optional.of("G57"), Optional.empty(), Optional.of("2.73118 \u00D7 10\u207B\u2077 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicInchPerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicInchPerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic inch per pound" ).build(), Optional.of("in\u00B3/lb"), Optional.of("N30"), Optional.of("cubicMetrePerKilogram"), Optional.of("3.612728 \u00D7 10\u207B\u2075 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicInchPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicInchPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic inch per second" ).build(), Optional.of("in\u00B3/s"), Optional.of("G58"), Optional.empty(), Optional.of("1.63871 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic kilometre" ).build(), Optional.of("km\u00B3"), Optional.of("H20"), Optional.of("cubicMetre"), Optional.of("10\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre" ).build(), Optional.of("m\u00B3"), Optional.of("MTQ"), Optional.empty(), Optional.of("m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME, QuantityKinds.SECTION_MODULUS)) ) );
UNITS_BY_NAME.put( "cubicMetrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per bar" ).build(), Optional.of("m\u00B3/bar"), Optional.of("G96"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerCoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerCoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per coulomb" ).build(), Optional.of("m\u00B3/C"), Optional.of("A38"), Optional.empty(), Optional.of("m\u00B3/A \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.HALL_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "cubicMetrePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per cubic metre" ).build(), Optional.of("m\u00B3/m\u00B3"), Optional.of("H60"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "cubicMetrePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per day" ).build(), Optional.of("m\u00B3/d"), Optional.of("G52"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per day bar" ).build(), Optional.of("m\u00B3/(d\u00B7bar)"), Optional.of("G86"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2070 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per day kelvin" ).build(), Optional.of("m\u00B3/(d\u00B7K)"), Optional.of("G69"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per hour" ).build(), Optional.of("m\u00B3/h"), Optional.of("MQH"), Optional.of("cubicMetrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per hour bar" ).build(), Optional.of("m\u00B3/(h\u00B7bar)"), Optional.of("G87"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2079 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per hour kelvin" ).build(), Optional.of("m\u00B3/(h\u00B7K)"), Optional.of("G70"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per kelvin" ).build(), Optional.of("m\u00B3/K"), Optional.of("G29"), Optional.empty(), Optional.of("m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per kilogram" ).build(), Optional.of("m\u00B3/kg"), Optional.of("A39"), Optional.empty(), Optional.of("m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per minute" ).build(), Optional.of("m\u00B3/min"), Optional.of("G53"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per minute bar" ).build(), Optional.of("m\u00B3/(min\u00B7bar)"), Optional.of("G88"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2077 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per minute kelvin" ).build(), Optional.of("m\u00B3/(min\u00B7K)"), Optional.of("G71"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per mole" ).build(), Optional.of("m\u00B3/mol"), Optional.of("A40"), Optional.empty(), Optional.of("m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per pascal" ).build(), Optional.of("m\u00B3/Pa"), Optional.of("M71"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per second" ).build(), Optional.of("m\u00B3/s"), Optional.of("MQS"), Optional.empty(), Optional.of("m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE, QuantityKinds.RECOMBINATION_COEFFICIENT, QuantityKinds.INSTANTANEOUS_VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per second bar" ).build(), Optional.of("m\u00B3/(s\u00B7bar)"), Optional.of("G89"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per second kelvin" ).build(), Optional.of("m\u00B3/(s\u00B7K)"), Optional.of("G72"), Optional.empty(), Optional.of("m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerSecondPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per second pascal" ).build(), Optional.of("(m\u00B3/s)/Pa"), Optional.of("N45"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicMetrePerSecondSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMetrePerSecondSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic metre per second square metre" ).build(), Optional.of("(m\u00B3/s)/m\u00B2"), Optional.of("P87"), Optional.of("metrePerSecond"), Optional.of("m/s"), new HashSet<>(Arrays.asList(QuantityKinds.POROSITY)) ) );
UNITS_BY_NAME.put( "cubicMicrometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMicrometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic micrometre" ).build(), Optional.of("\u00B5m\u00B3"), Optional.empty(), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B9\u2078 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMileUkStatute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMileUkStatute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic mile (UK statute)" ).build(), Optional.of("mi\u00B3"), Optional.of("M69"), Optional.of("cubicMetre"), Optional.of("4.168182 \u00D7 10\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic millimetre" ).build(), Optional.of("mm\u00B3"), Optional.of("MMQ"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicMillimetrePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicMillimetrePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic millimetre per cubic metre" ).build(), Optional.of("mm\u00B3/m\u00B3"), Optional.of("L21"), Optional.empty(), Optional.of("10\u2079"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "cubicYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard" ).build(), Optional.of("yd\u00B3"), Optional.of("YDQ"), Optional.of("cubicMetre"), Optional.of("0.764555 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicYardPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYardPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard per day" ).build(), Optional.of("yd\u00B3/d"), Optional.of("M12"), Optional.of("cubicMetrePerSecond"), Optional.of("8.849015 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicYardPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYardPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard per degree Fahrenheit" ).build(), Optional.of("yd\u00B3/\u00B0F"), Optional.of("M11"), Optional.of("cubicMetrePerKelvin"), Optional.of("1.376199 m\u00B3/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicYardPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYardPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard per hour" ).build(), Optional.of("yd\u00B3/h"), Optional.of("M13"), Optional.of("cubicMetrePerSecond"), Optional.of("2.123764 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicYardPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYardPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard per minute" ).build(), Optional.of("yd\u00B3/min"), Optional.of("M15"), Optional.of("cubicMetrePerSecond"), Optional.of("1.274258 \u00D7 10\u207B\u00B2 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cubicYardPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYardPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard per psi" ).build(), Optional.of("yd\u00B3/psi"), Optional.of("M14"), Optional.of("cubicMetrePerPascal"), Optional.of("1.108893 \u00D7 10\u207B\u2074 m\u00B3/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "cubicYardPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cubicYardPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cubic yard per second" ).build(), Optional.of("yd\u00B3/s"), Optional.of("M16"), Optional.of("cubicMetrePerSecond"), Optional.of("0.7645549 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "cupUnitOfVolume", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cupUnitOfVolume" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cup [unit of volume]" ).build(), Optional.of("cup (US)"), Optional.of("G21"), Optional.of("cubicMetre"), Optional.of("2.365882 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "curie", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "curie" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "curie" ).build(), Optional.of("Ci"), Optional.of("CUR"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u00B9\u2070 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "curiePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "curiePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "curie per kilogram" ).build(), Optional.of("Ci/kg"), Optional.of("A42"), Optional.of("becquerelPerKilogram"), Optional.of("3.7 \u00D7 10\u00B9\u2070 Bq/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "cycle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "cycle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "cycle" ).build(), Optional.empty(), Optional.of("B7"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "day", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "day" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "day" ).build(), Optional.of("d"), Optional.of("DAY"), Optional.of("secondUnitOfTime"), Optional.of("86400 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "deadweightTonnage", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "deadweightTonnage" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "deadweight tonnage" ).build(), Optional.of("dwt"), Optional.of("A43"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "decade", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decade" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decade" ).build(), Optional.empty(), Optional.of("DEC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "decadeLogarithmic", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decadeLogarithmic" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decade (logarithmic)" ).build(), Optional.of("dec"), Optional.of("P41"), Optional.empty(), Optional.of("dec"), new HashSet<>(Arrays.asList(QuantityKinds.LOGARITHMIC_DECREMENT)) ) );
UNITS_BY_NAME.put( "decagram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decagram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decagram" ).build(), Optional.of("dag"), Optional.of("DJ"), Optional.of("kilogram"), Optional.of("10\u207B\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "decalitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decalitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decalitre" ).build(), Optional.of("dal"), Optional.of("A44"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B2 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "decametre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decametre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decametre" ).build(), Optional.of("dam"), Optional.of("A45"), Optional.of("metre"), Optional.of("10 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "decapascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decapascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decapascal" ).build(), Optional.of("daPa"), Optional.of("H75"), Optional.of("pascal"), Optional.of("10\u00B9 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "decare", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decare" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decare" ).build(), Optional.of("daa"), Optional.of("DAA"), Optional.of("squareMetre"), Optional.of("10\u00B3 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "decibel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decibel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decibel" ).build(), Optional.of("dB"), Optional.of("2N"), Optional.of("neper"), Optional.of("0.1151293 Np"), new HashSet<>(Arrays.asList(QuantityKinds.LEVEL_OF_A_POWER_QUANTITY, QuantityKinds.LEVEL_OF_A_FIELD_QUANTITY, QuantityKinds.SOUND_REDUCTION_INDEX, QuantityKinds.SOUND_POWER_LEVEL, QuantityKinds.SOUND_PRESSURE_LEVEL)) ) );
UNITS_BY_NAME.put( "decibelPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decibelPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decibel per kilometre" ).build(), Optional.of("dB/km"), Optional.of("H51"), Optional.of("belPerMetre"), Optional.of("10\u207B\u2074 B/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_POWER_LEVEL)) ) );
UNITS_BY_NAME.put( "decibelPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decibelPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decibel per metre" ).build(), Optional.of("dB/m"), Optional.of("H52"), Optional.of("belPerMetre"), Optional.of("10\u207B\u00B9 B/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_PRESSURE_LEVEL, QuantityKinds.SOUND_POWER_LEVEL)) ) );
UNITS_BY_NAME.put( "decigram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decigram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decigram" ).build(), Optional.of("dg"), Optional.of("DG"), Optional.of("kilogram"), Optional.of("10\u207B\u2074 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "decilitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decilitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decilitre" ).build(), Optional.of("dl"), Optional.of("DLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "decilitrePerGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decilitrePerGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decilitre per gram" ).build(), Optional.of("dl/g"), Optional.of("22"), Optional.of("cubicMetrePerKilogram"), Optional.of("10\u207B\u00B9 \u00D7 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "decimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decimetre" ).build(), Optional.of("dm"), Optional.of("DMT"), Optional.of("metre"), Optional.of("10\u207B\u00B9 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "decinewtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decinewtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decinewton metre" ).build(), Optional.of("dN\u00B7m"), Optional.of("DN"), Optional.of("newtonMetre"), Optional.of("10\u207B\u00B9 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "decitex", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decitex" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decitex" ).build(), Optional.of("dtex (g/10km)"), Optional.of("A47"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "decitonne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "decitonne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "decitonne" ).build(), Optional.of("dt or dtn"), Optional.of("DTN"), Optional.of("kilogram"), Optional.of("10\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "degreeApi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeApi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree API" ).build(), Optional.of("\u00B0API"), Optional.of("J13"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBalling", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeBalling" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Balling" ).build(), Optional.of("\u00B0Balling"), Optional.of("J17"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBaumeOriginScale", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeBaumeOriginScale" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Baume (origin scale)" ).build(), Optional.of("\u00B0B\u00E9"), Optional.of("J14"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBaumeUsHeavy", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeBaumeUsHeavy" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Baume (US heavy)" ).build(), Optional.of("\u00B0B\u00E9 (US heavy)"), Optional.of("J15"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBaumeUsLight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeBaumeUsLight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Baume (US light)" ).build(), Optional.of("\u00B0B\u00E9 (US light)"), Optional.of("J16"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeBrix", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeBrix" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Brix" ).build(), Optional.of("\u00B0Bx"), Optional.of("J18"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}

private static void init3() {
UNITS_BY_NAME.put( "degreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Celsius" ).build(), Optional.of("\u00B0C"), Optional.of("CEL"), Optional.of("kelvin"), Optional.of("1 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeCelsiusPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Celsius per bar" ).build(), Optional.of("\u00B0C/bar"), Optional.of("F60"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeCelsiusPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Celsius per hour" ).build(), Optional.of("\u00B0C/h"), Optional.of("H12"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 s\u207B\u00B9 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeCelsiusPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Celsius per kelvin" ).build(), Optional.of("\u00B0C/K"), Optional.of("E98"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeCelsiusPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Celsius per minute" ).build(), Optional.of("\u00B0C/min"), Optional.of("H13"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 s\u207B\u00B9 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeCelsiusPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeCelsiusPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Celsius per second" ).build(), Optional.of("\u00B0C/s"), Optional.of("H14"), Optional.empty(), Optional.of("s\u207B\u00B9 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree day" ).build(), Optional.of("deg da"), Optional.of("E10"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit" ).build(), Optional.of("\u00B0F"), Optional.of("FAH"), Optional.empty(), Optional.of("5/9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.FAHRENHEIT_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourPerBritishThermalUnitInternationalTable", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitHourPerBritishThermalUnitInternationalTable" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit hour per British thermal unit (international table)" ).build(), Optional.of("\u00B0F/(BtuIT/h)"), Optional.of("N84"), Optional.of("kelvinPerWatt"), Optional.of("1.895634 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourPerBritishThermalUnitThermochemical", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitHourPerBritishThermalUnitThermochemical" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit hour per British thermal unit (thermochemical)" ).build(), Optional.of("\u00B0F/(Btuth/h)"), Optional.of("N85"), Optional.of("kelvinPerWatt"), Optional.of("1.896903 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTable", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTable" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit hour square foot per British thermal unit (international table)" ).build(), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/BtuIT"), Optional.of("J22"), Optional.of("squareMetreKelvinPerWatt"), Optional.of("0.1761102 m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTableInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitHourSquareFootPerBritishThermalUnitInternationalTableInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit hour square foot per British thermal unit (international table) inch" ).build(), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/(BtuIT\u00B7in)"), Optional.of("N88"), Optional.of("kelvinMetrePerWatt"), Optional.of("6.933472 K \u00D7 m/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemical", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemical" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit hour square foot per British thermal unit (thermochemical)" ).build(), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/Btuth"), Optional.of("J19"), Optional.of("squareMetreKelvinPerWatt"), Optional.of("0.176228 m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemicalInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitHourSquareFootPerBritishThermalUnitThermochemicalInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit hour square foot per British thermal unit (thermochemical) inch" ).build(), Optional.of("\u00B0F\u00B7h\u00B7ft\u00B2/(Btuth\u00B7in)"), Optional.of("N89"), Optional.of("kelvinMetrePerWatt"), Optional.of("6.938112 K \u00D7 m/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit per bar" ).build(), Optional.of("\u00B0F/bar"), Optional.of("J21"), Optional.of("kelvinPerPascal"), Optional.of("0.5555556 \u00D7 10\u207B\u2075 K/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit per hour" ).build(), Optional.of("\u00B0F/h"), Optional.of("J23"), Optional.of("kelvinPerSecond"), Optional.of("1.543210 \u00D7 10\u207B\u2074 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit per kelvin" ).build(), Optional.of("\u00B0F/K"), Optional.of("J20"), Optional.empty(), Optional.of("0.5555556"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit per minute" ).build(), Optional.of("\u00B0F/min"), Optional.of("J24"), Optional.of("kelvinPerSecond"), Optional.of("9.259259 \u00D7 10\u207B\u00B3 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit per second" ).build(), Optional.of("\u00B0F/s"), Optional.of("J25"), Optional.of("kelvinPerSecond"), Optional.of("0.5555556 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitSecondPerBritishThermalUnitInternationalTable", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitSecondPerBritishThermalUnitInternationalTable" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit second per British thermal unit (international table)" ).build(), Optional.of("\u00B0F/(BtuIT/s)"), Optional.of("N86"), Optional.of("kelvinPerWatt"), Optional.of("5.265651 \u00D7 10\u207B\u2074 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeFahrenheitSecondPerBritishThermalUnitThermochemical", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeFahrenheitSecondPerBritishThermalUnitThermochemical" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Fahrenheit second per British thermal unit (thermochemical)" ).build(), Optional.of("\u00B0F/(Btuth/s)"), Optional.of("N87"), Optional.of("kelvinPerWatt"), Optional.of("5.269175 \u00D7 10\u207B\u2074 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "degreeOechsle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeOechsle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Oechsle" ).build(), Optional.of("\u00B0Oechsle"), Optional.of("J27"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree per metre" ).build(), Optional.of("\u00B0/m"), Optional.of("H27"), Optional.of("radianPerMetre"), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad/m"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "degreePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree per second" ).build(), Optional.of("\u00B0/s"), Optional.of("E96"), Optional.empty(), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "degreePlato", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreePlato" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Plato" ).build(), Optional.of("\u00B0P"), Optional.of("PLA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeRankine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeRankine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Rankine" ).build(), Optional.of("\u00B0R"), Optional.of("A48"), Optional.empty(), Optional.of("5/9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "degreeRankinePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeRankinePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Rankine per hour" ).build(), Optional.of("\u00B0R/h"), Optional.of("J28"), Optional.of("kelvinPerSecond"), Optional.of("1.543210 \u00D7 10\u207B\u2074 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeRankinePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeRankinePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Rankine per minute" ).build(), Optional.of("\u00B0R/min"), Optional.of("J29"), Optional.of("kelvinPerSecond"), Optional.of("9.259259 \u00D7 10\u207B\u00B3 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeRankinePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeRankinePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Rankine per second" ).build(), Optional.of("\u00B0R/s"), Optional.of("J30"), Optional.of("kelvinPerSecond"), Optional.of("0.5555556 K/s"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE_VARIATION_OVER_TIME)) ) );
UNITS_BY_NAME.put( "degreeTwaddell", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeTwaddell" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree Twaddell" ).build(), Optional.of("\u00B0Tw"), Optional.of("J31"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "degreeUnitOfAngle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeUnitOfAngle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree [unit of angle]" ).build(), Optional.of("\u00B0"), Optional.of("DD"), Optional.of("radian"), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad"), new HashSet<>(Arrays.asList(QuantityKinds.BRAGG_ANGLE, QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "degreeUnitOfAnglePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "degreeUnitOfAnglePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "degree [unit of angle] per second squared" ).build(), Optional.of("\u00B0/s\u00B2"), Optional.of("M45"), Optional.empty(), Optional.of("1.745329 \u00D7 10\u207B\u00B2 rad / s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_ACCELERATION)) ) );
UNITS_BY_NAME.put( "denier", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "denier" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "denier" ).build(), Optional.of("den (g/9 km)"), Optional.of("M83"), Optional.of("kilogramPerMetre"), Optional.of("1.111111 \u00D7 10\u207B\u2077 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "digit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "digit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "digit" ).build(), Optional.empty(), Optional.of("B19"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dioptre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dioptre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dioptre" ).build(), Optional.of("dpt"), Optional.of("Q25"), Optional.of("reciprocalMetre"), Optional.of("m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "displacementTonnage", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "displacementTonnage" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "displacement tonnage" ).build(), Optional.empty(), Optional.of("DPT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dose", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dose" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dose" ).build(), Optional.empty(), Optional.of("E27"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dotsPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dotsPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dots per inch" ).build(), Optional.of("dpi"), Optional.of("E39"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dozen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dozen" ).build(), Optional.of("DOZ"), Optional.of("DZN"), Optional.empty(), Optional.of("12"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenPack", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dozenPack" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dozen pack" ).build(), Optional.empty(), Optional.of("DZP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenPair", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dozenPair" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dozen pair" ).build(), Optional.empty(), Optional.of("DPR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenPiece", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dozenPiece" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dozen piece" ).build(), Optional.empty(), Optional.of("DPC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dozenRoll", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dozenRoll" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dozen roll" ).build(), Optional.empty(), Optional.of("DRL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dramUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dramUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dram (UK)" ).build(), Optional.empty(), Optional.of("DRI"), Optional.of("gram"), Optional.of("1.771745 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dramUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dramUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dram (US)" ).build(), Optional.empty(), Optional.of("DRA"), Optional.of("gram"), Optional.of("3.887935 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dryBarrelUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dryBarrelUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dry barrel (US)" ).build(), Optional.of("bbl (US)"), Optional.of("BLD"), Optional.of("cubicMetre"), Optional.of("1.15627 \u00D7 10\u207B\u00B9 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryGallonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dryGallonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dry gallon (US)" ).build(), Optional.of("dry gal (US)"), Optional.of("GLD"), Optional.of("cubicMetre"), Optional.of("4.404884 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryPintUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dryPintUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dry pint (US)" ).build(), Optional.of("dry pt (US)"), Optional.of("PTD"), Optional.of("cubicMetre"), Optional.of("5.506105 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dryPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dry pound" ).build(), Optional.empty(), Optional.of("DB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dryQuartUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dryQuartUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dry quart (US)" ).build(), Optional.of("dry qt (US)"), Optional.of("QTD"), Optional.of("cubicMetre"), Optional.of("1.101221 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "dryTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dryTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dry ton" ).build(), Optional.empty(), Optional.of("DT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "dyneMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "dyneMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "dyne metre" ).build(), Optional.of("dyn\u00B7m"), Optional.of("M97"), Optional.of("newtonMetre"), Optional.of("10\u207B\u2075 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "each", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "each" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "each" ).build(), Optional.empty(), Optional.of("EA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "eightPartCloudCover", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "eightPartCloudCover" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "8-part cloud cover" ).build(), Optional.empty(), Optional.of("A59"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "electronicMailBox", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "electronicMailBox" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "electronic mail box" ).build(), Optional.empty(), Optional.of("EB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "electronvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "electronvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "electronvolt" ).build(), Optional.of("eV"), Optional.of("A53"), Optional.of("joule"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J"), new HashSet<>(Arrays.asList(QuantityKinds.MAXIMUM_BETA_PARTICLE_ENERGY, QuantityKinds.BETA_DISINTEGRATION_ENERGY, QuantityKinds.FERMI_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.RESONANCE_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "electronvoltPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "electronvoltPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "electronvolt per metre" ).build(), Optional.of("eV/m"), Optional.of("A54"), Optional.of("joulePerMetre"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J/m"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_LINEAR_STOPPING_POWER, QuantityKinds.LINEAR_ENERGY_TRANSFER)) ) );
UNITS_BY_NAME.put( "electronvoltSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "electronvoltSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "electronvolt square metre" ).build(), Optional.of("eV\u00B7m\u00B2"), Optional.of("A55"), Optional.of("jouleSquareMetre"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_ATOMIC_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "electronvoltSquareMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "electronvoltSquareMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "electronvolt square metre per kilogram" ).build(), Optional.of("eV\u00B7m\u00B2/kg"), Optional.of("A56"), Optional.of("jouleSquareMetrePerKilogram"), Optional.of("1.602176487 \u00D7 10\u207B\u00B9\u2079 J \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_MASS_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "equivalentGallon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "equivalentGallon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "equivalent gallon" ).build(), Optional.empty(), Optional.of("EQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "erlang", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "erlang" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "erlang" ).build(), Optional.of("E"), Optional.of("Q11"), Optional.empty(), Optional.of("1 E"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exabitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exabitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "exabit per second" ).build(), Optional.of("Ebit/s"), Optional.of("E58"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Exabyte" ).build(), Optional.of("EB"), Optional.empty(), Optional.of("byte"), Optional.of("10\u00B9\u2078 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "exajoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exajoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "exajoule" ).build(), Optional.of("EJ"), Optional.of("A68"), Optional.of("joule"), Optional.of("10\u00B9\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "exbibitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exbibitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "exbibit per cubic metre" ).build(), Optional.of("Eibit/m\u00B3"), Optional.of("E67"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exbibitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exbibitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "exbibit per metre" ).build(), Optional.of("Eibit/m"), Optional.of("E65"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exbibitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exbibitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "exbibit per square metre" ).build(), Optional.of("Eibit/m\u00B2"), Optional.of("E66"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "exbibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "exbibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "exbibyte" ).build(), Optional.of("Eibyte"), Optional.of("E59"), Optional.of("byte"), Optional.of("2\u2076\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "failuresInTime", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "failuresInTime" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "failures in time" ).build(), Optional.of("FIT"), Optional.of("FIT"), Optional.of("reciprocalSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B3 s\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "farad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "farad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "farad" ).build(), Optional.of("F"), Optional.of("FAR"), Optional.empty(), Optional.of("F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "faradPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "faradPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "farad per kilometre" ).build(), Optional.of("F/km"), Optional.of("H33"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u00B3 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CONSTANT, QuantityKinds.PERMITTIVITY, QuantityKinds.PERMITTIVITY_OF_VACUUM)) ) );
UNITS_BY_NAME.put( "faradPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "faradPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "farad per metre" ).build(), Optional.of("F/m"), Optional.of("A69"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u207B\u00B3 \u00D7 s\u2074 \u00D7 A\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.PERMITTIVITY, QuantityKinds.ELECTRIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "fathom", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fathom" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "fathom" ).build(), Optional.of("fth"), Optional.of("AK"), Optional.of("metre"), Optional.of("1.8288 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "femtojoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "femtojoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "femtojoule" ).build(), Optional.of("fJ"), Optional.of("A70"), Optional.of("joule"), Optional.of("10\u207B\u00B9\u2075 J"), new HashSet<>(Arrays.asList(QuantityKinds.GAP_ENERGY, QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "femtometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "femtometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "femtometre" ).build(), Optional.of("fm"), Optional.of("A71"), Optional.of("metre"), Optional.of("10\u207B\u00B9\u2075 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "fibreMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fibreMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "fibre metre" ).build(), Optional.empty(), Optional.of("FBM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "fivePack", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fivePack" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "five pack" ).build(), Optional.empty(), Optional.of("P5"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "fixedRate", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fixedRate" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "fixed rate" ).build(), Optional.empty(), Optional.of("1I"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "flakeTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "flakeTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "flake ton" ).build(), Optional.empty(), Optional.of("FL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "fluidOunceUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fluidOunceUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "fluid ounce (UK)" ).build(), Optional.of("fl oz (UK)"), Optional.of("OZI"), Optional.of("cubicMetre"), Optional.of("2.841306 \u00D7 10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "fluidOunceUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fluidOunceUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "fluid ounce (US)" ).build(), Optional.of("fl oz (US)"), Optional.of("OZA"), Optional.of("cubicMetre"), Optional.of("2.957353 \u00D7 10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "foot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "foot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot" ).build(), Optional.of("ft"), Optional.of("FOT"), Optional.of("metre"), Optional.of("0.3048 m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "footOfWater392Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footOfWater392Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot of water (39.2 °F)" ).build(), Optional.of("ftH\u2082O (39,2 \u00B0F)"), Optional.of("N15"), Optional.of("pascal"), Optional.of("2.98898 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "footPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per degree Fahrenheit" ).build(), Optional.of("ft/\u00B0F"), Optional.of("K13"), Optional.of("metrePerKelvin"), Optional.of("0.54864 m/K"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "footPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per hour" ).build(), Optional.of("ft/h"), Optional.of("K14"), Optional.of("metrePerSecond"), Optional.of("8.466667 \u00D7 10\u207B\u2075m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per minute" ).build(), Optional.of("ft/min"), Optional.of("FR"), Optional.of("metrePerSecond"), Optional.of("5.08 \u00D7 10\u207B\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per psi" ).build(), Optional.of("ft/psi"), Optional.of("K17"), Optional.of("metrePerPascal"), Optional.of("4.420750 \u00D7 10\u207B\u2075 m/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "footPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per second" ).build(), Optional.of("ft/s"), Optional.of("FS"), Optional.of("metrePerSecond"), Optional.of("0.3048 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerSecondDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerSecondDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per second degree Fahrenheit" ).build(), Optional.of("(ft/s)/\u00B0F"), Optional.of("K18"), Optional.of("metrePerSecondKelvin"), Optional.of("0.54864 (m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerSecondPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerSecondPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per second psi" ).build(), Optional.of("(ft/s)/psi"), Optional.of("K19"), Optional.of("metrePerSecondPascal"), Optional.of("4.420750 \u00D7 10\u207B\u2075 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "footPerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per second squared" ).build(), Optional.of("ft/s\u00B2"), Optional.of("A73"), Optional.of("metrePerSecondSquared"), Optional.of("0.3048 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY)) ) );
UNITS_BY_NAME.put( "footPerThousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPerThousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot per thousand" ).build(), Optional.empty(), Optional.of("E33"), Optional.of("metre"), Optional.of("3.048 \u00D7 10\u207B\u2074 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "footPoundForce", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPoundForce" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot pound-force" ).build(), Optional.of("ft\u00B7lbf"), Optional.of("85"), Optional.of("joule"), Optional.of("1.355818 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY)) ) );
UNITS_BY_NAME.put( "footPoundForcePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPoundForcePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot pound-force per hour" ).build(), Optional.of("ft\u00B7lbf/h"), Optional.of("K15"), Optional.of("watt"), Optional.of("3.766161 \u00D7 10\u207B\u2074 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "footPoundForcePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPoundForcePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot pound-force per minute" ).build(), Optional.of("ft\u00B7lbf/min"), Optional.of("K16"), Optional.of("watt"), Optional.of("2.259697 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "footPoundForcePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPoundForcePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot pound-force per second" ).build(), Optional.of("ft\u00B7lbf/s"), Optional.of("A74"), Optional.of("watt"), Optional.of("1.355818 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "footPoundal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footPoundal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot poundal" ).build(), Optional.of("ft\u00B7pdl"), Optional.of("N46"), Optional.of("joule"), Optional.of("4.214011 \u00D7 10\u207B\u00B2 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "footToTheFourthPower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footToTheFourthPower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot to the fourth power" ).build(), Optional.of("ft\u2074"), Optional.of("N27"), Optional.of("metreToTheFourthPower"), Optional.of("8.630975 \u00D7 10\u207B\u00B3 m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_POLAR_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "footUsSurvey", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footUsSurvey" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "foot (U.S. survey)" ).build(), Optional.of("ft (US survey)"), Optional.of("M51"), Optional.of("metre"), Optional.of("3.048006 \u00D7 10\u207B\u00B9 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES)) ) );
UNITS_BY_NAME.put( "footcandle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footcandle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "footcandle" ).build(), Optional.of("ftc"), Optional.of("P27"), Optional.empty(), Optional.of("1.076391 \u00D7 10\u00B9 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
}

private static void init4() {
UNITS_BY_NAME.put( "footlambert", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "footlambert" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "footlambert" ).build(), Optional.of("ftL"), Optional.of("P29"), Optional.of("candelaPerSquareMetre"), Optional.of("3.426259 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "fortyFootContainer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "fortyFootContainer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "forty foot container" ).build(), Optional.empty(), Optional.of("21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "franklin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "franklin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "franklin" ).build(), Optional.of("Fr"), Optional.of("N94"), Optional.of("coulomb"), Optional.of("3.335641 \u00D7 10\u207B\u00B9\u2070 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "freightTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "freightTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "freight ton" ).build(), Optional.empty(), Optional.of("A75"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "frenchGauge", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "frenchGauge" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "French gauge" ).build(), Optional.of("Fg"), Optional.of("H79"), Optional.of("metre"), Optional.of("0.333333333 \u00D7 10\u207B\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.RADIUS, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "furlong", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "furlong" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "furlong" ).build(), Optional.of("fur"), Optional.of("M50"), Optional.of("metre"), Optional.of("2.01168 \u00D7 10\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.DISTANCE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "gal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gal" ).build(), Optional.of("Gal"), Optional.of("A76"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u00B2 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY)) ) );
UNITS_BY_NAME.put( "gallonUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (UK)" ).build(), Optional.of("gal (UK)"), Optional.of("GLI"), Optional.of("cubicMetre"), Optional.of("4.546092 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "gallonUkPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUkPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (UK) per day" ).build(), Optional.of("gal (UK)/d"), Optional.of("K26"), Optional.of("cubicMetrePerSecond"), Optional.of("5.261678 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUkPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUkPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (UK) per hour" ).build(), Optional.of("gal (UK)/h"), Optional.of("K27"), Optional.of("cubicMetrePerSecond"), Optional.of("1.262803 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUkPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUkPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (UK) per second" ).build(), Optional.of("gal (UK)/s"), Optional.of("K28"), Optional.of("cubicMetrePerSecond"), Optional.of("4.54609 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (US)" ).build(), Optional.of("gal (US)"), Optional.of("GLL"), Optional.of("cubicMetre"), Optional.of("3.785412 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "gallonUsLiquidPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUsLiquidPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (US liquid) per second" ).build(), Optional.of("gal (US liq.)/s"), Optional.of("K30"), Optional.of("cubicMetrePerSecond"), Optional.of("3.785412 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gallonUsPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUsPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (US) per day" ).build(), Optional.of("gal (US)/d"), Optional.of("GB"), Optional.of("cubicMetrePerSecond"), Optional.of("4.381264 \u00D7 10\u207B\u2078 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gallonUsPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gallonUsPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gallon (US) per hour" ).build(), Optional.of("gal/h"), Optional.of("G50"), Optional.empty(), Optional.of("1.0515 \u00D7 10\u207B\u2076 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gamma", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gamma" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gamma" ).build(), Optional.of("\u03B3"), Optional.of("P12"), Optional.of("tesla"), Optional.of("10\u207B\u2079 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "gibibit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gibibit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gibibit" ).build(), Optional.of("Gibit"), Optional.of("B30"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gibibitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gibibit per cubic metre" ).build(), Optional.of("Gibit/m\u00B3"), Optional.of("E71"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gibibitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gibibit per metre" ).build(), Optional.of("Gibit/m"), Optional.of("E69"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gibibitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gibibit per square metre" ).build(), Optional.of("Gibit/m\u00B2"), Optional.of("E70"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gibibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gibibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Gibibyte" ).build(), Optional.of("Gibyte"), Optional.of("E62"), Optional.of("byte"), Optional.of("2\u00B3\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "gigabecquerel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigabecquerel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigabecquerel" ).build(), Optional.of("GBq"), Optional.of("GBQ"), Optional.of("becquerel"), Optional.of("10\u2079 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "gigabit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigabit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigabit" ).build(), Optional.of("Gbit"), Optional.of("B68"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gigabitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigabitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigabit per second" ).build(), Optional.of("Gbit/s"), Optional.of("B80"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gigabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Gigabyte" ).build(), Optional.of("GB"), Optional.of("E34"), Optional.of("byte"), Optional.of("10\u2079 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "gigabytePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigabytePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigabyte per second" ).build(), Optional.of("Gbyte/s"), Optional.of("E68"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gigacoulombPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigacoulombPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigacoulomb per cubic metre" ).build(), Optional.of("GC/m\u00B3"), Optional.of("A84"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2079 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "gigaelectronvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigaelectronvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigaelectronvolt" ).build(), Optional.of("GeV"), Optional.of("A85"), Optional.of("electronvolt"), Optional.of("10\u2079 eV"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "gigahertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigahertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigahertz" ).build(), Optional.of("GHz"), Optional.of("A86"), Optional.of("hertz"), Optional.of("10\u2079 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "gigahertzMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigahertzMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigahertz metre" ).build(), Optional.of("GHz\u00B7m"), Optional.of("M18"), Optional.of("hertzMetre"), Optional.of("10\u2079 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "gigajoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigajoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigajoule" ).build(), Optional.of("GJ"), Optional.of("GV"), Optional.of("joule"), Optional.of("10\u2079 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "gigaohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigaohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigaohm" ).build(), Optional.of("G\u2126"), Optional.of("A87"), Optional.of("ohm"), Optional.of("10\u2079 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.REACTANCE, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_IMPEDANCES)) ) );
UNITS_BY_NAME.put( "gigaohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigaohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigaohm metre" ).build(), Optional.of("G\u2126\u00B7m"), Optional.of("A88"), Optional.of("ohmMetre"), Optional.of("10\u2079 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "gigaohmPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigaohmPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigaohm per metre" ).build(), Optional.of("G\u2126/m"), Optional.of("M26"), Optional.of("ohmPerMetre"), Optional.of("10\u2079 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_LOAD_PER_UNIT_LENGTH)) ) );
UNITS_BY_NAME.put( "gigapascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigapascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigapascal" ).build(), Optional.of("GPa"), Optional.of("A89"), Optional.of("pascal"), Optional.of("10\u2079 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "gigawatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigawatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigawatt" ).build(), Optional.of("GW"), Optional.of("A90"), Optional.of("watt"), Optional.of("10\u2079 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.POWER, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "gigawattHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gigawattHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gigawatt hour" ).build(), Optional.of("GW\u00B7h"), Optional.of("GWH"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u00B9\u00B2 J"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY)) ) );
UNITS_BY_NAME.put( "gilbert", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gilbert" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gilbert" ).build(), Optional.of("Gi"), Optional.of("N97"), Optional.of("ampere"), Optional.of("7.957747 \u00D7 10\u207B\u00B9 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.CURRENT_LINKAGE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE)) ) );
UNITS_BY_NAME.put( "gillUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (UK)" ).build(), Optional.of("gi (UK)"), Optional.of("GII"), Optional.of("cubicMetre"), Optional.of("1.420653 \u00D7 10\u207B\u2074 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gillUkPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUkPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (UK) per day" ).build(), Optional.of("gi (UK)/d"), Optional.of("K32"), Optional.of("cubicMetrePerSecond"), Optional.of("1.644274 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUkPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUkPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (UK) per hour" ).build(), Optional.of("gi (UK)/h"), Optional.of("K33"), Optional.of("cubicMetrePerSecond"), Optional.of("3.946258 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUkPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUkPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (UK) per minute" ).build(), Optional.of("gi (UK)/min"), Optional.of("K34"), Optional.of("cubicMetrePerSecond"), Optional.of("0.02367755 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUkPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUkPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (UK) per second" ).build(), Optional.of("gi (UK)/s"), Optional.of("K35"), Optional.of("cubicMetrePerSecond"), Optional.of("1.420653 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (US)" ).build(), Optional.of("gi (US)"), Optional.of("GIA"), Optional.of("cubicMetre"), Optional.of("1.182941 \u00D7 10\u207B\u2074 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gillUsPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUsPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (US) per day" ).build(), Optional.of("gi (US)/d"), Optional.of("K36"), Optional.of("cubicMetrePerSecond"), Optional.of("1.369145 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUsPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUsPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (US) per hour" ).build(), Optional.of("gi (US)/h"), Optional.of("K37"), Optional.of("cubicMetrePerSecond"), Optional.of("3.285947 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUsPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUsPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (US) per minute" ).build(), Optional.of("gi (US)/min"), Optional.of("K38"), Optional.of("cubicMetrePerSecond"), Optional.of("1.971568 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gillUsPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gillUsPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gill (US) per second" ).build(), Optional.of("gi (US)/s"), Optional.of("K39"), Optional.of("cubicMetrePerSecond"), Optional.of("1.182941 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gon" ).build(), Optional.of("gon"), Optional.of("A91"), Optional.of("radian"), Optional.of("1.570796 \u00D7 10\u207B\u00B2 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "grain", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "grain" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "grain" ).build(), Optional.of("gr"), Optional.of("GRN"), Optional.of("kilogram"), Optional.of("64.79891 \u00D7 10\u207B\u2076 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "grainPerGallonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "grainPerGallonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "grain per gallon (US)" ).build(), Optional.of("gr/gal (US)"), Optional.of("K41"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.711806 \u00D7 10\u207B\u00B2 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram" ).build(), Optional.of("g"), Optional.of("GRM"), Optional.of("kilogram"), Optional.of("10\u207B\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "gramCentimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramCentimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram centimetre per second" ).build(), Optional.of("g\u00B7(cm/s)"), Optional.of("M99"), Optional.of("kilogramMetrePerSecond"), Optional.of("10\u207B\u2075 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "gramDryWeight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramDryWeight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram, dry weight" ).build(), Optional.empty(), Optional.of("GDW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramForcePerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramForcePerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram-force per square centimetre" ).build(), Optional.of("gf/cm\u00B2"), Optional.of("K31"), Optional.of("pascal"), Optional.of("98.0665 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "gramIncludingContainer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramIncludingContainer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram, including container" ).build(), Optional.empty(), Optional.of("GIC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramIncludingInnerPackaging", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramIncludingInnerPackaging" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram, including inner packaging" ).build(), Optional.empty(), Optional.of("GIP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram millimetre" ).build(), Optional.of("g\u00B7mm"), Optional.of("H84"), Optional.of("kilogramMetre"), Optional.of("10\u207B\u2076 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "gramOfFissileIsotope", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramOfFissileIsotope" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram of fissile isotope" ).build(), Optional.of("gi F/S"), Optional.of("GFI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gramPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per bar" ).build(), Optional.of("g/bar"), Optional.of("F74"), Optional.empty(), Optional.of("10\u207B\u2078 m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCentimetreSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCentimetreSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per centimetre second" ).build(), Optional.of("g/(cm\u00B7s)"), Optional.of("N41"), Optional.of("pascalSecond"), Optional.of("0.1 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic centimetre" ).build(), Optional.of("g/cm\u00B3"), Optional.of("23"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicCentimetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicCentimetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic centimetre bar" ).build(), Optional.of("g/(cm\u00B3\u00B7bar)"), Optional.of("G11"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicCentimetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicCentimetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic centimetre kelvin" ).build(), Optional.of("g/(cm\u00B3\u00B7K)"), Optional.of("G33"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicDecimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicDecimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic decimetre" ).build(), Optional.of("g/dm\u00B3"), Optional.of("F23"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicDecimetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicDecimetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic decimetre bar" ).build(), Optional.of("g/(dm\u00B3\u00B7bar)"), Optional.of("G12"), Optional.empty(), Optional.of("10\u207B\u2075 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicDecimetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicDecimetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic decimetre kelvin" ).build(), Optional.of("g/(dm\u00B3\u00B7K)"), Optional.of("G34"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic metre" ).build(), Optional.of("g/m\u00B3"), Optional.of("A93"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic metre bar" ).build(), Optional.of("g/(m\u00B3\u00B7bar)"), Optional.of("G14"), Optional.empty(), Optional.of("10\u207B\u2078 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per cubic metre kelvin" ).build(), Optional.of("g/(m\u00B3\u00B7K)"), Optional.of("G36"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per day" ).build(), Optional.of("g/d"), Optional.of("F26"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per day bar" ).build(), Optional.of("g/(d\u00B7bar)"), Optional.of("F62"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B3 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per day kelvin" ).build(), Optional.of("g/(d\u00B7K)"), Optional.of("F35"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerHertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per hertz" ).build(), Optional.of("g/Hz"), Optional.of("F25"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per hour" ).build(), Optional.of("g/h"), Optional.of("F27"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per hour bar" ).build(), Optional.of("g/(h\u00B7bar)"), Optional.of("F63"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B2 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per hour kelvin" ).build(), Optional.of("g/(h\u00B7K)"), Optional.of("F36"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per kelvin" ).build(), Optional.of("g/K"), Optional.of("F14"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per litre" ).build(), Optional.of("g/l"), Optional.of("GL"), Optional.of("kilogramPerCubicMetre"), Optional.of("kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerLitreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerLitreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per litre bar" ).build(), Optional.of("g/(l\u00B7bar)"), Optional.of("G13"), Optional.empty(), Optional.of("10\u207B\u2075 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerLitreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerLitreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per litre kelvin" ).build(), Optional.of("g/(l\u00B7K)"), Optional.of("G35"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMetreGramPer100Centimetres", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMetreGramPer100Centimetres" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per metre (gram per 100 centimetres)" ).build(), Optional.of("g/m"), Optional.of("GF"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u00B3 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillilitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMillilitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per millilitre" ).build(), Optional.of("g/ml"), Optional.of("GJ"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillilitreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMillilitreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per millilitre bar" ).build(), Optional.of("g/(ml\u00B7bar)"), Optional.of("G15"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillilitreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMillilitreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per millilitre kelvin" ).build(), Optional.of("g/(ml\u00B7K)"), Optional.of("G37"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per millimetre" ).build(), Optional.of("g/mm"), Optional.of("H76"), Optional.empty(), Optional.of("10\u00B9 kg \u00D7 m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "gramPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per minute" ).build(), Optional.of("g/min"), Optional.of("F28"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per minute bar" ).build(), Optional.of("g/(min\u00B7bar)"), Optional.of("F64"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u2070 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per minute kelvin" ).build(), Optional.of("g/(min\u00B7K)"), Optional.of("F37"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per mole" ).build(), Optional.of("g/mol"), Optional.of("A94"), Optional.of("kilogramPerMole"), Optional.of("10\u207B\u00B3 kg/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_MASS)) ) );
UNITS_BY_NAME.put( "gramPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per second" ).build(), Optional.of("g/s"), Optional.of("F29"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per second bar" ).build(), Optional.of("g/(s\u00B7bar)"), Optional.of("F65"), Optional.empty(), Optional.of("10\u207B\u2078 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per second kelvin" ).build(), Optional.of("g/(s\u00B7K)"), Optional.of("F38"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "gramPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per square centimetre" ).build(), Optional.of("g/cm\u00B2"), Optional.of("25"), Optional.of("kilogramPerSquareMetre"), Optional.of("10 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "gramPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per square metre" ).build(), Optional.of("g/m\u00B2"), Optional.of("GM"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u207B\u00B3 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "gramPerSquareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gramPerSquareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gram per square millimetre" ).build(), Optional.of("g/mm\u00B2"), Optional.of("N24"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u00B3 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "gray", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gray" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gray" ).build(), Optional.of("Gy"), Optional.of("A95"), Optional.empty(), Optional.of("m\u00B2/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENERGY_IMPARTED, QuantityKinds.MASSIC_ENERGY_IMPARTED)) ) );
UNITS_BY_NAME.put( "grayPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "grayPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gray per hour" ).build(), Optional.of("Gy/h"), Optional.of("P61"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "grayPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "grayPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gray per minute" ).build(), Optional.of("Gy/min"), Optional.of("P57"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "grayPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "grayPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gray per second" ).build(), Optional.of("Gy/s"), Optional.of("A96"), Optional.empty(), Optional.of("m\u00B2/s\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
}

private static void init5() {
UNITS_BY_NAME.put( "greatGross", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "greatGross" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "great gross" ).build(), Optional.empty(), Optional.of("GGR"), Optional.empty(), Optional.of("1728"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "gross", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "gross" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gross" ).build(), Optional.of("gr"), Optional.of("GRO"), Optional.empty(), Optional.of("144"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "grossKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "grossKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "gross kilogram" ).build(), Optional.empty(), Optional.of("E4"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "group", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "group" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "group" ).build(), Optional.empty(), Optional.of("10"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "guntersChain", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "guntersChain" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Gunter's chain" ).build(), Optional.of("ch (UK)"), Optional.of("X1"), Optional.of("metre"), Optional.of("20.1168 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH)) ) );
UNITS_BY_NAME.put( "halfYear6Months", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "halfYear6Months" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "half year (6 months)" ).build(), Optional.empty(), Optional.of("SAN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hangingContainer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hangingContainer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hanging container" ).build(), Optional.empty(), Optional.of("Z11"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hank", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hank" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hank" ).build(), Optional.empty(), Optional.of("HA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hartley", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hartley" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hartley" ).build(), Optional.of("Hart"), Optional.of("Q15"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hartleyPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hartleyPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hartley per second" ).build(), Optional.of("Hart/s"), Optional.of("Q18"), Optional.empty(), Optional.of("Hart/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "head", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "head" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "head" ).build(), Optional.empty(), Optional.of("HEA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hectobar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectobar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectobar" ).build(), Optional.of("hbar"), Optional.of("HBA"), Optional.of("pascal"), Optional.of("10\u2077 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "hectogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectogram" ).build(), Optional.of("hg"), Optional.of("HGM"), Optional.of("kilogram"), Optional.of("10\u207B\u00B9 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "hectolitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectolitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectolitre" ).build(), Optional.of("hl"), Optional.of("HLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B9 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "hectolitreOfPureAlcohol", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectolitreOfPureAlcohol" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectolitre of pure alcohol" ).build(), Optional.empty(), Optional.of("HPA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hectometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectometre" ).build(), Optional.of("hm"), Optional.of("HMT"), Optional.of("metre"), Optional.of("10\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.THICKNESS, QuantityKinds.BREADTH)) ) );
UNITS_BY_NAME.put( "hectopascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectopascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectopascal" ).build(), Optional.of("hPa"), Optional.of("A97"), Optional.of("pascal"), Optional.of("10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "hectopascalCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectopascalCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectopascal cubic metre per second" ).build(), Optional.of("hPa\u00B7m\u00B3/s"), Optional.of("F94"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "hectopascalLitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectopascalLitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectopascal litre per second" ).build(), Optional.of("hPa\u00B7l/s"), Optional.of("F93"), Optional.empty(), Optional.of("10\u207B\u00B9 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "hectopascalPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectopascalPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectopascal per bar" ).build(), Optional.of("hPa/bar"), Optional.of("E99"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "hectopascalPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectopascalPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectopascal per kelvin" ).build(), Optional.of("hPa/K"), Optional.of("F82"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "hectopascalPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hectopascalPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hectopascal per metre" ).build(), Optional.of("hPa/m"), Optional.of("P82"), Optional.empty(), Optional.of("10\u00B2 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "hefnerKerze", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hefnerKerze" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Hefner-Kerze" ).build(), Optional.of("HK"), Optional.of("P35"), Optional.of("candela"), Optional.of("0.903 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "henry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "henry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "henry" ).build(), Optional.of("H"), Optional.of("81"), Optional.empty(), Optional.of("H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "henryPerKiloohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "henryPerKiloohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "henry per kiloohm" ).build(), Optional.of("H/k\u2126"), Optional.of("H03"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.SELF_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "henryPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "henryPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "henry per metre" ).build(), Optional.of("H/m"), Optional.of("A98"), Optional.empty(), Optional.of("H/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEABILITY_OF_VACUUM, QuantityKinds.PERMEABILITY, QuantityKinds.MAGNETIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "henryPerOhm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "henryPerOhm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "henry per ohm" ).build(), Optional.of("H/\u2126"), Optional.of("H04"), Optional.of("secondUnitOfTime"), Optional.of("s"), new HashSet<>(Arrays.asList(QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "hertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hertz" ).build(), Optional.of("Hz"), Optional.of("HTZ"), Optional.empty(), Optional.of("Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "hertzMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hertzMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hertz metre" ).build(), Optional.of("Hz\u00B7m"), Optional.of("H34"), Optional.empty(), Optional.of("Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "horsepowerBoiler", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "horsepowerBoiler" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "horsepower (boiler)" ).build(), Optional.of("boiler hp"), Optional.of("K42"), Optional.of("watt"), Optional.of("9.80950 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "horsepowerElectric", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "horsepowerElectric" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "horsepower (electric)" ).build(), Optional.of("electric hp"), Optional.of("K43"), Optional.of("watt"), Optional.of("746 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "hour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hour" ).build(), Optional.of("h"), Optional.of("HUR"), Optional.of("secondUnitOfTime"), Optional.of("3600 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "hundred", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundred" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred" ).build(), Optional.empty(), Optional.of("CEN"), Optional.empty(), Optional.of("100"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredBoardFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredBoardFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred board foot" ).build(), Optional.empty(), Optional.of("BP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredBoxes", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredBoxes" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred boxes" ).build(), Optional.empty(), Optional.of("HBX"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredCount", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredCount" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred count" ).build(), Optional.empty(), Optional.of("HC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred cubic foot" ).build(), Optional.empty(), Optional.of("HH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred cubic metre" ).build(), Optional.empty(), Optional.of("FF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredInternationalUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredInternationalUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred international unit" ).build(), Optional.empty(), Optional.of("HIU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredKilogramDryWeight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredKilogramDryWeight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred kilogram, dry weight" ).build(), Optional.empty(), Optional.of("HDW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredKilogramNetMass", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredKilogramNetMass" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred kilogram, net mass" ).build(), Optional.empty(), Optional.of("HKM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredLeave", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredLeave" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred leave" ).build(), Optional.empty(), Optional.of("CLF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred metre" ).build(), Optional.empty(), Optional.of("JPS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredPack", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredPack" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred pack" ).build(), Optional.empty(), Optional.of("CNP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "hundredPoundCwtOrHundredWeightUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredPoundCwtOrHundredWeightUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred pound (cwt) / hundred weight (US)" ).build(), Optional.of("cwt (US)"), Optional.of("CWA"), Optional.of("kilogram"), Optional.of("45.3592 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "hundredWeightUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hundredWeightUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hundred weight (UK)" ).build(), Optional.of("cwt (UK)"), Optional.of("CWI"), Optional.of("kilogram"), Optional.of("50.80235 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "hydraulicHorsePower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "hydraulicHorsePower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "hydraulic horse power" ).build(), Optional.empty(), Optional.of("5J"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "imperialGallonPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "imperialGallonPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Imperial gallon per minute" ).build(), Optional.of("gal (UK) /min"), Optional.of("G3"), Optional.of("cubicMetrePerSecond"), Optional.of("7.57682 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "inch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch" ).build(), Optional.of("in"), Optional.of("INH"), Optional.of("metre"), Optional.of("25.4 \u00D7 10\u207B\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "inchOfMercury", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchOfMercury" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch of mercury" ).build(), Optional.of("inHg"), Optional.of("F79"), Optional.empty(), Optional.of("3.38639 \u00D7 10\u00B3 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "inchOfMercury32Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchOfMercury32Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch of mercury (32 °F)" ).build(), Optional.of("inHG (32 \u00B0F)"), Optional.of("N16"), Optional.of("pascal"), Optional.of("3.38638 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "inchOfMercury60Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchOfMercury60Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch of mercury (60 °F)" ).build(), Optional.of("inHg (60 \u00B0F)"), Optional.of("N17"), Optional.of("pascal"), Optional.of("3.37685 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "inchOfWater", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchOfWater" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch of water" ).build(), Optional.of("inH\u2082O"), Optional.of("F78"), Optional.empty(), Optional.of("2.49089 \u00D7 10\u00B2 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "inchOfWater392Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchOfWater392Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch of water (39.2 °F)" ).build(), Optional.of("inH\u2082O (39,2 \u00B0F)"), Optional.of("N18"), Optional.of("pascal"), Optional.of("2.49082 \u00D7 10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "inchOfWater60Degreesf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchOfWater60Degreesf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch of water (60 °F)" ).build(), Optional.of("inH\u2082O (60 \u00B0F)"), Optional.of("N19"), Optional.of("pascal"), Optional.of("2.4884 \u00D7 10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "inchPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per degree Fahrenheit" ).build(), Optional.of("in/\u00B0F"), Optional.of("K45"), Optional.of("metrePerKelvin"), Optional.of("4.572 \u00D7 10\u207B\u00B2 m/K"), new HashSet<>(Arrays.asList(QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "inchPerLinearFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerLinearFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per linear foot" ).build(), Optional.empty(), Optional.of("B82"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "inchPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per minute" ).build(), Optional.of("in/min"), Optional.of("M63"), Optional.of("metrePerSecond"), Optional.of("4.233333 \u00D7 10\u207B\u2074 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per psi" ).build(), Optional.of("in/psi"), Optional.of("K46"), Optional.of("metrePerPascal"), Optional.of("3.683959 \u00D7 10\u207B\u2076 m/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "inchPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per second" ).build(), Optional.of("in/s"), Optional.of("IU"), Optional.of("metrePerSecond"), Optional.of("0.0254 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerSecondDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerSecondDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per second degree Fahrenheit" ).build(), Optional.of("(in/s)/\u00B0F"), Optional.of("K47"), Optional.of("metrePerSecondKelvin"), Optional.of("4.572 \u00D7 10\u207B\u00B2 (m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerSecondPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerSecondPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per second psi" ).build(), Optional.of("(in/s)/psi"), Optional.of("K48"), Optional.of("metrePerSecondPascal"), Optional.of("3.683959 \u00D7 10\u207B\u2076 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per second squared" ).build(), Optional.of("in/s\u00B2"), Optional.of("IV"), Optional.of("metrePerSecondSquared"), Optional.of("0.0254 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL)) ) );
UNITS_BY_NAME.put( "inchPerTwoPiRadiant", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerTwoPiRadiant" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per two pi radiant" ).build(), Optional.of("in/revolution"), Optional.of("H57"), Optional.empty(), Optional.of("2.54 \u00D7 10\u207B\u00B2 m/(2 \u00D7 \u03C0 \u00D7 rad)"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "inchPerYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPerYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch per year" ).build(), Optional.of("in/y"), Optional.of("M61"), Optional.of("metrePerSecond"), Optional.of("8.048774 \u00D7 10\u207B\u00B9\u2070 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "inchPoundPoundInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPoundPoundInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch pound (pound inch)" ).build(), Optional.of("in\u00B7lb"), Optional.of("IA"), Optional.of("kilogramMetre"), Optional.of("1.15212 \u00D7 10\u207B\u00B2 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "inchPoundal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchPoundal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch poundal" ).build(), Optional.of("in\u00B7pdl"), Optional.of("N47"), Optional.of("joule"), Optional.of("3.511677 \u00D7 10\u207B\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "inchToTheFourthPower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "inchToTheFourthPower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "inch to the fourth power" ).build(), Optional.of("in\u2074"), Optional.of("D69"), Optional.of("metreToTheFourthPower"), Optional.of("41.62314 \u00D7 10\u207B\u2078 m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_POLAR_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "internationalCandle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "internationalCandle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "international candle" ).build(), Optional.of("IK"), Optional.of("P36"), Optional.of("candela"), Optional.of("1.019 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "internationalSugarDegree", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "internationalSugarDegree" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "international sugar degree" ).build(), Optional.empty(), Optional.of("ISD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "job", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "job" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "job" ).build(), Optional.empty(), Optional.of("E51"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "joule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule" ).build(), Optional.of("J"), Optional.of("JOU"), Optional.empty(), Optional.of("J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.MEAN_ENERGY_IMPARTED, QuantityKinds.ENERGY_IMPARTED, QuantityKinds.ALPHA_DISINTEGRATION_ENERGY, QuantityKinds.LEVEL_WIDTH, QuantityKinds.HARTREE_ENERGY, QuantityKinds.HEAT, QuantityKinds.WORK_FUNCTION, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.RADIANT_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.WORK, QuantityKinds.ENTHALPY, QuantityKinds.REACTION_ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.ACTIVE_ENERGY)) ) );
UNITS_BY_NAME.put( "joulePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per cubic metre" ).build(), Optional.of("J/m\u00B3"), Optional.of("B8"), Optional.empty(), Optional.of("J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC, QuantityKinds.RADIANT_ENERGY_DENSITY, QuantityKinds.SOUND_ENERGY, QuantityKinds.SOUND_ENERGY_DENSITY, QuantityKinds.VOLUMIC_ELECTROMAGNETIC_ENERGY, QuantityKinds.ELECTROMAGNETIC_ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "joulePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per day" ).build(), Optional.of("J/d"), Optional.of("P17"), Optional.of("watt"), Optional.of("1.15741 \u00D7 10\u207B\u2075 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per gram" ).build(), Optional.of("J/g"), Optional.of("D95"), Optional.empty(), Optional.of("J/(10\u207B\u00B3 \u00D7 kg)"), new HashSet<>(Arrays.asList(QuantityKinds.MASSIC_THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "joulePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per hour" ).build(), Optional.of("J/h"), Optional.of("P16"), Optional.of("watt"), Optional.of("2.77778 \u00D7 10\u207B\u2074 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per kelvin" ).build(), Optional.of("J/K"), Optional.of("JE"), Optional.empty(), Optional.of("J/K"), new HashSet<>(Arrays.asList(QuantityKinds.PLANCK_FUNCTION, QuantityKinds.BOLTZMANN_CONSTANT, QuantityKinds.ENTROPY, QuantityKinds.MASSIEU_FUNCTION, QuantityKinds.HEAT_CAPACITY)) ) );
UNITS_BY_NAME.put( "joulePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per kilogram" ).build(), Optional.of("J/kg"), Optional.of("J2"), Optional.empty(), Optional.of("J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENERGY, QuantityKinds.MASSIC_ENERGY)) ) );
UNITS_BY_NAME.put( "joulePerKilogramKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerKilogramKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per kilogram kelvin" ).build(), Optional.of("J/(kg\u00B7K)"), Optional.of("B11"), Optional.empty(), Optional.of("J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_SATURATION, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE)) ) );
UNITS_BY_NAME.put( "joulePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per metre" ).build(), Optional.of("J/m"), Optional.of("B12"), Optional.empty(), Optional.of("J/m"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_LINEAR_STOPPING_POWER, QuantityKinds.LINEAR_ENERGY_TRANSFER)) ) );
UNITS_BY_NAME.put( "joulePerMetreToTheFourthPower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerMetreToTheFourthPower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per metre to the fourth power" ).build(), Optional.of("J/m\u2074"), Optional.of("B14"), Optional.empty(), Optional.of("J/m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVE_LENGTH, QuantityKinds.SPECTRAL_CONCENTRATION_OF_RADIANT_ENERGY_DENSITY_IN_TERMS_OF_WAVELENGTH)) ) );
UNITS_BY_NAME.put( "joulePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per minute" ).build(), Optional.of("J/min"), Optional.of("P15"), Optional.of("watt"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per mole" ).build(), Optional.of("J/mol"), Optional.of("B15"), Optional.empty(), Optional.of("J/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_THERMODYNAMIC_ENERGY, QuantityKinds.CHEMICAL_POTENTIAL, QuantityKinds.AFFINITY_OF_A_CHEMICAL_REACTION)) ) );
UNITS_BY_NAME.put( "joulePerMoleKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerMoleKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per mole kelvin" ).build(), Optional.of("J/(mol\u00B7K)"), Optional.of("B16"), Optional.empty(), Optional.of("J/(mol \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_HEAT_CAPACITY, QuantityKinds.MOLAR_GAS_CONSTANT, QuantityKinds.MOLAR_ENTROPY)) ) );
UNITS_BY_NAME.put( "joulePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per second" ).build(), Optional.of("J/s"), Optional.of("P14"), Optional.of("watt"), Optional.of("W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "joulePerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per square centimetre" ).build(), Optional.of("J/cm\u00B2"), Optional.of("E43"), Optional.of("joulePerSquareMetre"), Optional.of("10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "joulePerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per square metre" ).build(), Optional.of("J/m\u00B2"), Optional.of("B13"), Optional.empty(), Optional.of("J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE, QuantityKinds.ENERGY_FLUENCE)) ) );
UNITS_BY_NAME.put( "joulePerTesla", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "joulePerTesla" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule per tesla" ).build(), Optional.of("J/T"), Optional.of("Q10"), Optional.empty(), Optional.of("m\u00B2 \u00D7 A"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "jouleSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "jouleSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule second" ).build(), Optional.of("J\u00B7s"), Optional.of("B18"), Optional.empty(), Optional.of("J \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.PLANCK_CONSTANT)) ) );
UNITS_BY_NAME.put( "jouleSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "jouleSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule square metre" ).build(), Optional.of("J\u00B7m\u00B2"), Optional.of("D73"), Optional.empty(), Optional.of("J \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_ATOMIC_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "jouleSquareMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "jouleSquareMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "joule square metre per kilogram" ).build(), Optional.of("J\u00B7m\u00B2/kg"), Optional.of("B20"), Optional.empty(), Optional.of("J \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.TOTAL_MASS_STOPPING_POWER)) ) );
UNITS_BY_NAME.put( "katal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "katal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "katal" ).build(), Optional.of("kat"), Optional.of("KAT"), Optional.empty(), Optional.of("s\u207B\u00B9 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "kelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin" ).build(), Optional.of("K"), Optional.of("KEL"), Optional.empty(), Optional.of("K"), new HashSet<>(Arrays.asList(QuantityKinds.CURIE_TEMPERATURE, QuantityKinds.SUPER_CONDUCTOR_TRANSITION_TEMPERATURE, QuantityKinds.THERMODYNAMIC, QuantityKinds.NEEL_TEMPERATURE, QuantityKinds.DEBYE_TEMPERATURE, QuantityKinds.FERMI_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinMetrePerWatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinMetrePerWatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin metre per watt" ).build(), Optional.of("K\u00B7m/W"), Optional.of("H35"), Optional.empty(), Optional.of("K \u00D7 m\u207B\u00B9 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "kelvinPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per bar" ).build(), Optional.of("K/bar"), Optional.of("F61"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per hour" ).build(), Optional.of("K/h"), Optional.of("F10"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 s\u207B\u00B9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per kelvin" ).build(), Optional.of("K/K"), Optional.of("F02"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per minute" ).build(), Optional.of("K/min"), Optional.of("F11"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 s\u207B\u00B9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per pascal" ).build(), Optional.of("K/Pa"), Optional.of("N79"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "kelvinPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per second" ).build(), Optional.of("K/s"), Optional.of("F12"), Optional.empty(), Optional.of("s\u207B\u00B9 \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
}

private static void init6() {
UNITS_BY_NAME.put( "kelvinPerWatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kelvinPerWatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kelvin per watt" ).build(), Optional.of("K/W"), Optional.of("B21"), Optional.empty(), Optional.of("K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_RESISTANCE)) ) );
UNITS_BY_NAME.put( "kibibit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kibibit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kibibit" ).build(), Optional.of("Kibit"), Optional.of("C21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kibibitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kibibit per cubic metre" ).build(), Optional.of("Kibit/m\u00B3"), Optional.of("E74"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kibibitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kibibit per metre" ).build(), Optional.of("Kibit/m"), Optional.of("E72"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kibibitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kibibit per square metre" ).build(), Optional.of("Kibit/m\u00B2"), Optional.of("E73"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kibibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kibibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kibibyte" ).build(), Optional.of("KiB"), Optional.of("E64"), Optional.of("byte"), Optional.of("2\u00B9\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "kiloampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloampere" ).build(), Optional.of("kA"), Optional.of("B22"), Optional.of("ampere"), Optional.of("10\u00B3 A"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE, QuantityKinds.MAGNETOMOTIVE_FORCE)) ) );
UNITS_BY_NAME.put( "kiloampereHourThousandAmpereHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloampereHourThousandAmpereHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloampere hour (thousand ampere hour)" ).build(), Optional.of("kA\u00B7h"), Optional.of("TAH"), Optional.of("coulomb"), Optional.of("3.6 \u00D7 10\u2076 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "kiloamperePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloamperePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloampere per metre" ).build(), Optional.of("kA/m"), Optional.of("B24"), Optional.of("amperePerMetre"), Optional.of("10\u00B3 A/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "kiloamperePerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloamperePerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloampere per square metre" ).build(), Optional.of("kA/m\u00B2"), Optional.of("B23"), Optional.of("amperePerSquareMetre"), Optional.of("10\u00B3 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "kilobar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobar" ).build(), Optional.of("kbar"), Optional.of("KBA"), Optional.of("pascal"), Optional.of("10\u2078 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "kilobaud", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobaud" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobaud" ).build(), Optional.of("kBd"), Optional.of("K50"), Optional.of("baud"), Optional.of("10\u00B3 Bd"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilobecquerel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobecquerel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobecquerel" ).build(), Optional.of("kBq"), Optional.of("2Q"), Optional.of("becquerel"), Optional.of("10\u00B3 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "kilobecquerelPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobecquerelPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobecquerel per kilogram" ).build(), Optional.of("kBq/kg"), Optional.of("B25"), Optional.of("becquerelPerKilogram"), Optional.of("10\u00B3 Bq/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "kilobit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobit" ).build(), Optional.of("kbit"), Optional.of("C37"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilobitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobit per second" ).build(), Optional.of("kbit/s"), Optional.of("C74"), Optional.of("bitPerSecond"), Optional.of("10\u00B3 bit/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilobyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Kilobyte" ).build(), Optional.of("kB"), Optional.of("2P"), Optional.of("byte"), Optional.of("10\u00B3 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "kilobytePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilobytePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilobyte per second" ).build(), Optional.of("kbyte/s"), Optional.of("P94"), Optional.of("bytePerSecond"), Optional.of("10\u00B3 byte/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilocalorieInternationalTable", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieInternationalTable" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (international table)" ).build(), Optional.of("kcalIT"), Optional.of("E14"), Optional.of("joule"), Optional.of("4.1868 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.ENTHALPY, QuantityKinds.ENERGY, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "kilocalorieInternationalTablePerGramKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieInternationalTablePerGramKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (international table) per gram kelvin" ).build(), Optional.of("(kcalIT/K)/g"), Optional.of("N65"), Optional.of("joulePerKilogramKelvin"), Optional.of("4.1868 \u00D7 10\u2076 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "kilocalorieInternationalTablePerHourMetreDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieInternationalTablePerHourMetreDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (international table) per hour metre degree Celsius" ).build(), Optional.of("kcal/(m\u00B7h\u00B7\u00B0C)"), Optional.of("K52"), Optional.empty(), Optional.of("1.163 J/(m \u00D7 s \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilocalorieMean", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieMean" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (mean)" ).build(), Optional.of("kcal"), Optional.of("K51"), Optional.of("joule"), Optional.of("4.19002 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT, QuantityKinds.ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.ENTHALPY)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemical", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieThermochemical" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (thermochemical)" ).build(), Optional.of("kcalth"), Optional.of("K53"), Optional.of("joule"), Optional.of("4.184 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HEAT)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemicalPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieThermochemicalPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (thermochemical) per hour" ).build(), Optional.of("kcalth/h"), Optional.of("E15"), Optional.of("watt"), Optional.of("1.16222 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemicalPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieThermochemicalPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (thermochemical) per minute" ).build(), Optional.of("kcalth/min"), Optional.of("K54"), Optional.of("watt"), Optional.of("69.73333 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilocalorieThermochemicalPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocalorieThermochemicalPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocalorie (thermochemical) per second" ).build(), Optional.of("kcalth/s"), Optional.of("K55"), Optional.of("watt"), Optional.of("4.184 \u00D7 10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilocandela", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocandela" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocandela" ).build(), Optional.of("kcd"), Optional.of("P33"), Optional.of("candela"), Optional.of("10\u00B3 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "kilocharacter", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocharacter" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocharacter" ).build(), Optional.empty(), Optional.of("KB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilocoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocoulomb" ).build(), Optional.of("kC"), Optional.of("B26"), Optional.of("coulomb"), Optional.of("10\u00B3 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "kilocoulombPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocoulombPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocoulomb per cubic metre" ).build(), Optional.of("kC/m\u00B3"), Optional.of("B27"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u00B3 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUMIC_CHARGE)) ) );
UNITS_BY_NAME.put( "kilocoulombPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocoulombPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocoulomb per square metre" ).build(), Optional.of("kC/m\u00B2"), Optional.of("B28"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u00B3 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "kilocurie", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilocurie" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilocurie" ).build(), Optional.of("kCi"), Optional.of("2R"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u00B9\u00B3 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "kiloelectronvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloelectronvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloelectronvolt" ).build(), Optional.of("keV"), Optional.of("B29"), Optional.of("electronvolt"), Optional.of("10\u00B3 eV"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "kilofarad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilofarad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilofarad" ).build(), Optional.of("kF"), Optional.of("N90"), Optional.of("farad"), Optional.of("10\u00B3 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "kilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram" ).build(), Optional.of("kg"), Optional.of("KGM"), Optional.empty(), Optional.of("kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS, QuantityKinds.NUCLIDIC_MASS, QuantityKinds.EFFECTIVE_MASS, QuantityKinds.MASS_OF_MOLECULE, QuantityKinds.MASS_EXCESS, QuantityKinds.MASS_OF_ATOM_OF_A_NUCLIDE_X)) ) );
UNITS_BY_NAME.put( "kilogramCentimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramCentimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram centimetre per second" ).build(), Optional.of("kg\u00B7(cm/s)"), Optional.of("M98"), Optional.of("kilogramMetrePerSecond"), Optional.of("10\u207B\u00B2 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "kilogramDrainedNetWeight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramDrainedNetWeight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram drained net weight" ).build(), Optional.of("kg/net eda"), Optional.of("KDW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramDryWeight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramDryWeight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram, dry weight" ).build(), Optional.empty(), Optional.of("MND"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramForceMetrePerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramForceMetrePerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram-force metre per square centimetre" ).build(), Optional.of("kgf\u00B7m/cm\u00B2"), Optional.of("E44"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramForcePerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramForcePerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram-force per square centimetre" ).build(), Optional.of("kgf/cm\u00B2"), Optional.of("E42"), Optional.of("pascal"), Optional.of("9.80665 \u00D7 10\u2074 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "kilogramForcePerSquareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramForcePerSquareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram-force per square millimetre" ).build(), Optional.of("kgf/mm\u00B2"), Optional.of("E41"), Optional.of("pascal"), Optional.of("9.80665 \u00D7 10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "kilogramIncludingContainer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramIncludingContainer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram, including container" ).build(), Optional.empty(), Optional.of("KIC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramIncludingInnerPackaging", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramIncludingInnerPackaging" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram, including inner packaging" ).build(), Optional.empty(), Optional.of("KIP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram metre" ).build(), Optional.of("kg\u00B7m"), Optional.of("M94"), Optional.empty(), Optional.of("kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "kilogramMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram metre per second" ).build(), Optional.of("kg\u00B7m/s"), Optional.of("B31"), Optional.empty(), Optional.of("kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "kilogramMetrePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramMetrePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram metre per second squared" ).build(), Optional.of("kg\u00B7m/s\u00B2"), Optional.of("M77"), Optional.empty(), Optional.of("(kg \u00D7 m)/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "kilogramMetreSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramMetreSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram metre squared" ).build(), Optional.of("kg\u00B7m\u00B2"), Optional.of("B32"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA)) ) );
UNITS_BY_NAME.put( "kilogramMetreSquaredPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramMetreSquaredPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram metre squared per second" ).build(), Optional.of("kg\u00B7m\u00B2/s"), Optional.of("B33"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_MOMENTUM, QuantityKinds.ANGULAR_MOMENTUM)) ) );
UNITS_BY_NAME.put( "kilogramNamedSubstance", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramNamedSubstance" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram named substance" ).build(), Optional.empty(), Optional.of("KNS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfCholineChloride", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfCholineChloride" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of choline chloride" ).build(), Optional.of("kg C\u2085 H\u2081\u2084ClNO"), Optional.of("KCC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfHydrogenPeroxide", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfHydrogenPeroxide" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of hydrogen peroxide" ).build(), Optional.of("kg H\u2082O\u2082"), Optional.of("KHY"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfImportedMeatLessOffal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfImportedMeatLessOffal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of imported meat, less offal" ).build(), Optional.empty(), Optional.of("TMS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfMethylamine", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfMethylamine" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of methylamine" ).build(), Optional.of("kg met.am."), Optional.of("KMA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfNitrogen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfNitrogen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of nitrogen" ).build(), Optional.of("kg N"), Optional.of("KNI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfPhosphorusPentoxidePhosphoricAnhydride", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfPhosphorusPentoxidePhosphoricAnhydride" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of phosphorus pentoxide (phosphoric anhydride)" ).build(), Optional.empty(), Optional.of("KPP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfPotassiumHydroxideCausticPotash", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfPotassiumHydroxideCausticPotash" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of potassium hydroxide (caustic potash)" ).build(), Optional.of("kg KOH"), Optional.of("KPH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfPotassiumOxide", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfPotassiumOxide" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of potassium oxide" ).build(), Optional.of("kg K\u2082O"), Optional.of("KPO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfSodiumHydroxideCausticSoda", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfSodiumHydroxideCausticSoda" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of sodium hydroxide (caustic soda)" ).build(), Optional.of("kg NaOH"), Optional.of("KSH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfSubstance90PercentDry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfSubstance90PercentDry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of substance 90 % dry" ).build(), Optional.of("kg 90 % sdt"), Optional.of("KSD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfTungstenTrioxide", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfTungstenTrioxide" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of tungsten trioxide" ).build(), Optional.of("kg WO\u2083"), Optional.of("KWO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramOfUranium", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramOfUranium" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram of uranium" ).build(), Optional.of("kg U"), Optional.of("KUR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per bar" ).build(), Optional.of("kg/bar"), Optional.of("H53"), Optional.empty(), Optional.of("10\u207B\u2075 m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic centimetre" ).build(), Optional.of("kg/cm\u00B3"), Optional.of("G31"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicCentimetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicCentimetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic centimetre bar" ).build(), Optional.of("kg/(cm\u00B3\u00B7bar)"), Optional.of("G16"), Optional.empty(), Optional.of("10\u00B9 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicCentimetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicCentimetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic centimetre kelvin" ).build(), Optional.of("kg/(cm\u00B3\u00B7K)"), Optional.of("G38"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicDecimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicDecimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic decimetre" ).build(), Optional.of("kg/dm\u00B3"), Optional.of("B34"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicDecimetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicDecimetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic decimetre bar" ).build(), Optional.of("(kg/dm\u00B3)/bar"), Optional.of("H55"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicDecimetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicDecimetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic decimetre kelvin" ).build(), Optional.of("(kg/dm\u00B3)/K"), Optional.of("H54"), Optional.empty(), Optional.of("10\u00B3 m\u207B\u00B3 \u00D7 kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic metre" ).build(), Optional.of("kg/m\u00B3"), Optional.of("KMQ"), Optional.empty(), Optional.of("kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic metre bar" ).build(), Optional.of("kg/(m\u00B3\u00B7bar)"), Optional.of("G18"), Optional.empty(), Optional.of("10\u207B\u2075 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic metre kelvin" ).build(), Optional.of("kg/(m\u00B3\u00B7K)"), Optional.of("G40"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerCubicMetrePascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerCubicMetrePascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per cubic metre pascal" ).build(), Optional.of("(kg/m\u00B3)/Pa"), Optional.of("M73"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per day" ).build(), Optional.of("kg/d"), Optional.of("F30"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per day bar" ).build(), Optional.of("kg/(d\u00B7bar)"), Optional.of("F66"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2070 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per day kelvin" ).build(), Optional.of("kg/(d\u00B7K)"), Optional.of("F39"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2075 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per hour" ).build(), Optional.of("kg/h"), Optional.of("E93"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per hour bar" ).build(), Optional.of("kg/(h\u00B7bar)"), Optional.of("F67"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2079 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per hour kelvin" ).build(), Optional.of("kg/(h\u00B7K)"), Optional.of("F40"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2074 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per kelvin" ).build(), Optional.of("kg/K"), Optional.of("F15"), Optional.empty(), Optional.of("kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per kilogram" ).build(), Optional.of("kg/kg"), Optional.of("M29"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "kilogramPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per kilometre" ).build(), Optional.of("kg/km"), Optional.of("M31"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u00B3 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerKilomol", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerKilomol" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per kilomol" ).build(), Optional.of("kg/kmol"), Optional.of("F24"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 mol\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per litre" ).build(), Optional.of("kg/l or kg/L"), Optional.of("B35"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "kilogramPerLitreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerLitreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per litre bar" ).build(), Optional.of("kg/(l\u00B7bar)"), Optional.of("G17"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerLitreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerLitreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per litre kelvin" ).build(), Optional.of("kg/(l\u00B7K)"), Optional.of("G39"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per metre" ).build(), Optional.of("kg/m"), Optional.of("KL"), Optional.empty(), Optional.of("kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMetreDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per metre day" ).build(), Optional.of("kg/(m\u00B7d)"), Optional.of("N39"), Optional.of("pascalSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2075 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMetreHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per metre hour" ).build(), Optional.of("kg/(m\u00B7h)"), Optional.of("N40"), Optional.of("pascalSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMetreMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per metre minute" ).build(), Optional.of("kg/(m\u00B7min)"), Optional.of("N38"), Optional.of("pascalSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMetreSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMetreSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per metre second" ).build(), Optional.of("kg/(m\u00B7s)"), Optional.of("N37"), Optional.of("pascalSecond"), Optional.of("Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per millimetre" ).build(), Optional.of("kg/mm"), Optional.of("KW"), Optional.of("kilogramPerMetre"), Optional.of("10\u00B3 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerMillimetreWidth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMillimetreWidth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per millimetre width" ).build(), Optional.empty(), Optional.of("KI"), Optional.of("kilogramPerMetre"), Optional.of("10\u00B3 kg/m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per minute" ).build(), Optional.of("kg/min"), Optional.of("F31"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per minute bar" ).build(), Optional.of("kg/(min\u00B7bar)"), Optional.of("F68"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2077 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per minute kelvin" ).build(), Optional.of("kg/(min\u00B7K)"), Optional.of("F41"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per mole" ).build(), Optional.of("kg/mol"), Optional.of("D74"), Optional.empty(), Optional.of("kg/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_MASS)) ) );
UNITS_BY_NAME.put( "kilogramPerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per pascal" ).build(), Optional.of("kg/Pa"), Optional.of("M74"), Optional.empty(), Optional.of("m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per second" ).build(), Optional.of("kg/s"), Optional.of("KGS"), Optional.empty(), Optional.of("kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per second bar" ).build(), Optional.of("kg/(s\u00B7bar)"), Optional.of("F69"), Optional.empty(), Optional.of("10\u207B\u2075 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per second kelvin" ).build(), Optional.of("kg/(s\u00B7K)"), Optional.of("F42"), Optional.empty(), Optional.of("kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
}

private static void init7() {
UNITS_BY_NAME.put( "kilogramPerSecondPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSecondPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per second pascal" ).build(), Optional.of("(kg/s)/Pa"), Optional.of("M87"), Optional.empty(), Optional.of("m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per square centimetre" ).build(), Optional.of("kg/cm\u00B2"), Optional.of("D5"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u2074 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "kilogramPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per square metre" ).build(), Optional.of("kg/m\u00B2"), Optional.of("28"), Optional.empty(), Optional.of("kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.MEAN_MASS_RANGE, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "kilogramPerSquareMetrePascalSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSquareMetrePascalSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per square metre pascal second" ).build(), Optional.of("kg/(m\u00B2\u00B7Pa\u00B7s)"), Optional.of("Q28"), Optional.empty(), Optional.of("kg/(m\u00B2 \u00D7 Pa \u00D7 s)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilogramPerSquareMetreSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramPerSquareMetreSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram per square metre second" ).build(), Optional.of("kg/(m\u00B2\u00B7s)"), Optional.of("H56"), Optional.empty(), Optional.of("kg m\u207B\u00B2 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilogramSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram square centimetre" ).build(), Optional.of("kg\u00B7cm\u00B2"), Optional.of("F18"), Optional.empty(), Optional.of("10\u207B\u2074 kg m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilogramSquareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilogramSquareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilogram square millimetre" ).build(), Optional.of("kg\u00B7mm\u00B2"), Optional.of("F19"), Optional.empty(), Optional.of("10\u207B\u2076 kg m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilohenry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilohenry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilohenry" ).build(), Optional.of("kH"), Optional.of("P24"), Optional.of("henry"), Optional.of("10\u00B3 H"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "kilohertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilohertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilohertz" ).build(), Optional.of("kHz"), Optional.of("KHZ"), Optional.of("hertz"), Optional.of("10\u00B3 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "kilohertzMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilohertzMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilohertz metre" ).build(), Optional.of("kHz\u00B7m"), Optional.of("M17"), Optional.of("hertzMetre"), Optional.of("10\u00B3 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "kilojoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule" ).build(), Optional.of("kJ"), Optional.of("KJO"), Optional.of("joule"), Optional.of("10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "kilojoulePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per day" ).build(), Optional.of("kJ/d"), Optional.of("P21"), Optional.of("watt"), Optional.of("1.15741 \u00D7 10\u207B\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilojoulePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per hour" ).build(), Optional.of("kJ/h"), Optional.of("P20"), Optional.of("watt"), Optional.of("2.77778 \u00D7 10\u207B\u00B9 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilojoulePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per kelvin" ).build(), Optional.of("kJ/K"), Optional.of("B41"), Optional.of("joulePerKelvin"), Optional.of("10\u00B3 J/K"), new HashSet<>(Arrays.asList(QuantityKinds.HEAT_CAPACITY, QuantityKinds.ENTROPY)) ) );
UNITS_BY_NAME.put( "kilojoulePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per kilogram" ).build(), Optional.of("kJ/kg"), Optional.of("B42"), Optional.of("joulePerKilogram"), Optional.of("10\u00B3 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENTHALPY, QuantityKinds.MASSIC_ENTHALPY)) ) );
UNITS_BY_NAME.put( "kilojoulePerKilogramKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerKilogramKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per kilogram kelvin" ).build(), Optional.of("kJ/(kg\u00B7K)"), Optional.of("B43"), Optional.of("joulePerKilogramKelvin"), Optional.of("10\u00B3 J/(kg \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_VOLUME, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_SATURATION, QuantityKinds.SPECIFIC_HEAT_CAPACITY_AT_CONSTANT_PRESSURE)) ) );
UNITS_BY_NAME.put( "kilojoulePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per minute" ).build(), Optional.of("kJ/min"), Optional.of("P19"), Optional.of("watt"), Optional.of("1.66667 \u00D7 10 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilojoulePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per mole" ).build(), Optional.of("kJ/mol"), Optional.of("B44"), Optional.of("joulePerMole"), Optional.of("10\u00B3 J/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "kilojoulePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilojoulePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilojoule per second" ).build(), Optional.of("kJ/s"), Optional.of("P18"), Optional.of("watt"), Optional.of("10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilolitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilolitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilolitre" ).build(), Optional.of("kl"), Optional.of("K6"), Optional.of("cubicMetre"), Optional.of("m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "kilolitrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilolitrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilolitre per hour" ).build(), Optional.of("kl/h"), Optional.of("4X"), Optional.of("cubicMetrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilolux", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilolux" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilolux" ).build(), Optional.of("klx"), Optional.of("KLX"), Optional.empty(), Optional.of("10\u00B3 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "kilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilometre" ).build(), Optional.of("km"), Optional.of("KMT"), Optional.of("metre"), Optional.of("10\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "kilometrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilometrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilometre per hour" ).build(), Optional.of("km/h"), Optional.of("KMH"), Optional.of("metrePerSecond"), Optional.of("0.277778 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "kilometrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilometrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilometre per second" ).build(), Optional.of("km/s"), Optional.of("M62"), Optional.of("metrePerSecond"), Optional.of("10\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "kilometrePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilometrePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilometre per second squared" ).build(), Optional.of("km/s\u00B2"), Optional.of("M38"), Optional.of("metrePerSecondSquared"), Optional.of("10\u00B3 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_OF_FREE_FALL)) ) );
UNITS_BY_NAME.put( "kilomole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole" ).build(), Optional.of("kmol"), Optional.of("B45"), Optional.of("mole"), Optional.of("10\u00B3 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "kilomolePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per cubic metre" ).build(), Optional.of("kmol/m\u00B3"), Optional.of("B46"), Optional.of("molePerCubicMetre"), Optional.of("10\u00B3 mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "kilomolePerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per cubic metre bar" ).build(), Optional.of("(kmol/m\u00B3)/bar"), Optional.of("K60"), Optional.of("molPerCubicMetrePascal"), Optional.of("10\u207B\u00B2 (mol/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "kilomolePerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per cubic metre kelvin" ).build(), Optional.of("(kmol/m\u00B3)/K"), Optional.of("K59"), Optional.of("molePerCubicMetreKelvin"), Optional.of("10\u00B3 (mol/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "kilomolePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per hour" ).build(), Optional.of("kmol/h"), Optional.of("K58"), Optional.of("molePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
UNITS_BY_NAME.put( "kilomolePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per kilogram" ).build(), Optional.of("kmol/kg"), Optional.of("P47"), Optional.of("molePerKilogram"), Optional.of("10\u00B3 mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "kilomolePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per minute" ).build(), Optional.of("kmol/min"), Optional.of("K61"), Optional.of("molePerSecond"), Optional.of("16.6667 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
UNITS_BY_NAME.put( "kilomolePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilomolePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilomole per second" ).build(), Optional.of("kmol/s"), Optional.of("E94"), Optional.empty(), Optional.of("10\u00B3 s\u207B\u00B9 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "kilonewton", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilonewton" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilonewton" ).build(), Optional.of("kN"), Optional.of("B47"), Optional.of("newton"), Optional.of("10\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "kilonewtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilonewtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilonewton metre" ).build(), Optional.of("kN\u00B7m"), Optional.of("B48"), Optional.of("newtonMetre"), Optional.of("10\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "kilonewtonPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilonewtonPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilonewton per metre" ).build(), Optional.of("kN/m"), Optional.of("N31"), Optional.of("newtonPerMetre"), Optional.of("10\u00B3 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "kiloohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloohm" ).build(), Optional.of("k\u2126"), Optional.of("B49"), Optional.of("ohm"), Optional.of("10\u00B3 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.REACTANCE, QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "kiloohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloohm metre" ).build(), Optional.of("k\u2126\u00B7m"), Optional.of("B50"), Optional.of("ohmMetre"), Optional.of("10\u00B3 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "kilopascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopascal" ).build(), Optional.of("kPa"), Optional.of("KPA"), Optional.of("pascal"), Optional.of("10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "kilopascalPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopascalPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopascal per bar" ).build(), Optional.of("kPa/bar"), Optional.of("F03"), Optional.empty(), Optional.of("10\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "kilopascalPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopascalPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopascal per kelvin" ).build(), Optional.of("kPa/K"), Optional.of("F83"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "kilopascalPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopascalPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopascal per metre" ).build(), Optional.of("kPa/m"), Optional.of("P81"), Optional.empty(), Optional.of("10\u00B3 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "kilopascalPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopascalPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopascal per millimetre" ).build(), Optional.of("kPa/mm"), Optional.of("34"), Optional.empty(), Optional.of("10\u2076 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "kilopascalSquareMetrePerGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopascalSquareMetrePerGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopascal square metre per gram" ).build(), Optional.of("kPa\u00B7m\u00B2/g"), Optional.of("33"), Optional.of("metrePerSecondSquared"), Optional.of("10\u2076 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.BURST_INDEX)) ) );
UNITS_BY_NAME.put( "kilopoundForce", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopoundForce" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopound-force" ).build(), Optional.of("kip"), Optional.of("M75"), Optional.of("newton"), Optional.of("4.448222 \u00D7 10\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "kilopoundPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilopoundPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilopound per hour" ).build(), Optional.of("klb/h"), Optional.of("M90"), Optional.of("kilogramPerSecond"), Optional.of("0.125997889 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kiloroentgen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloroentgen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloroentgen" ).build(), Optional.of("kR"), Optional.of("KR"), Optional.of("coulombPerKilogram"), Optional.of("2.58 \u00D7 10\u207B\u00B9 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "kilosecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilosecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilosecond" ).build(), Optional.of("ks"), Optional.of("B52"), Optional.of("secondUnitOfTime"), Optional.of("10\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "kilosegment", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilosegment" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilosegment" ).build(), Optional.empty(), Optional.of("KJ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilosiemens", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilosiemens" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilosiemens" ).build(), Optional.of("kS"), Optional.of("B53"), Optional.of("siemens"), Optional.of("10\u00B3 S"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "kilosiemensPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilosiemensPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilosiemens per metre" ).build(), Optional.of("kS/m"), Optional.of("B54"), Optional.of("siemensPerMetre"), Optional.of("10\u00B3 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilotesla", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilotesla" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilotesla" ).build(), Optional.of("kT"), Optional.of("P13"), Optional.of("tesla"), Optional.of("10\u00B3 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "kilotonne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilotonne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilotonne" ).build(), Optional.of("kt"), Optional.of("KTN"), Optional.of("kilogram"), Optional.of("10\u2076 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "kilovar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovar" ).build(), Optional.of("kvar"), Optional.of("KVR"), Optional.of("voltAmpere"), Optional.of("10\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.REACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "kilovolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovolt" ).build(), Optional.of("kV"), Optional.of("KVT"), Optional.of("voltDc"), Optional.of("10\u00B3 V"), new HashSet<>(Arrays.asList(QuantityKinds.TENSION, QuantityKinds.VOLTAGE, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.ELECTRIC_POTENTIAL)) ) );
UNITS_BY_NAME.put( "kilovoltAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovoltAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovolt - ampere" ).build(), Optional.of("kV\u00B7A"), Optional.of("KVA"), Optional.of("voltAmpere"), Optional.of("10\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "kilovoltAmpereHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovoltAmpereHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovolt ampere hour" ).build(), Optional.of("kVAh"), Optional.of("C79"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilovoltAmpereReactiveDemand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovoltAmpereReactiveDemand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovolt ampere reactive demand" ).build(), Optional.empty(), Optional.of("K2"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilovoltAmpereReactiveHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovoltAmpereReactiveHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovolt ampere reactive hour" ).build(), Optional.of("kvar\u00B7h"), Optional.of("K3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilovoltPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilovoltPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilovolt per metre" ).build(), Optional.of("kV/m"), Optional.of("B55"), Optional.of("voltPerMetre"), Optional.of("10\u00B3 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "kilowatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt" ).build(), Optional.of("kW"), Optional.of("KWT"), Optional.of("watt"), Optional.of("10\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.POWER, QuantityKinds.HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "kilowattDemand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattDemand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt demand" ).build(), Optional.empty(), Optional.of("K1"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt hour" ).build(), Optional.of("kW\u00B7h"), Optional.of("KWH"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u2076 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "kilowattHourPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattHourPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt hour per cubic metre" ).build(), Optional.of("kW\u00B7h/m\u00B3"), Optional.of("E46"), Optional.of("joulePerCubicMetre"), Optional.of("3.6 \u00D7 10\u2076 J/m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattHourPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattHourPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt hour per hour" ).build(), Optional.of("kW\u00B7h/h"), Optional.of("D03"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattHourPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattHourPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt hour per kelvin" ).build(), Optional.of("kW\u00B7h/K"), Optional.of("E47"), Optional.of("joulePerKelvin"), Optional.of("3.6 \u00D7 10\u2076 J/K"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "kilowattPerMetreDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattPerMetreDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt per metre degree Celsius" ).build(), Optional.of("kW/(m\u00B7\u00B0C)"), Optional.of("N82"), Optional.of("wattPerMetreKelvin"), Optional.of("10\u00B3 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilowattPerMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattPerMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt per metre kelvin" ).build(), Optional.of("kW/(m\u00B7K)"), Optional.of("N81"), Optional.of("wattPerMetreKelvin"), Optional.of("10\u00B3 W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "kilowattPerSquareMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kilowattPerSquareMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kilowatt per square metre kelvin" ).build(), Optional.of("kW/(m\u00B2\u00B7K)"), Optional.of("N78"), Optional.of("wattPerSquareMetreKelvin"), Optional.of("10\u00B3 W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "kiloweber", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloweber" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloweber" ).build(), Optional.of("kWb"), Optional.of("P11"), Optional.of("weber"), Optional.of("10\u00B3 Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX)) ) );
UNITS_BY_NAME.put( "kiloweberPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kiloweberPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kiloweber per metre" ).build(), Optional.of("kWb/m"), Optional.of("B56"), Optional.of("weberPerMetre"), Optional.of("10\u00B3 Wb/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_VECTOR_POTENTIAL)) ) );
UNITS_BY_NAME.put( "kipPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kipPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kip per square inch" ).build(), Optional.of("ksi"), Optional.of("N20"), Optional.of("pascal"), Optional.of("6.894757 \u00D7 10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "kit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "kit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "kit" ).build(), Optional.empty(), Optional.of("KT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "knot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "knot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "knot" ).build(), Optional.of("kn"), Optional.of("KNT"), Optional.of("metrePerSecond"), Optional.of("0.514444 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "labourHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "labourHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "labour hour" ).build(), Optional.empty(), Optional.of("LH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lacticDryMaterialPercentage", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lacticDryMaterialPercentage" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lactic dry material percentage" ).build(), Optional.empty(), Optional.of("KLK"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lactoseExcessPercentage", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lactoseExcessPercentage" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lactose excess percentage" ).build(), Optional.empty(), Optional.of("LAC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lambert", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lambert" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lambert" ).build(), Optional.of("Lb"), Optional.of("P30"), Optional.of("candelaPerSquareMetre"), Optional.of("3.183099 \u00D7 10\u00B3 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "langley", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "langley" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "langley" ).build(), Optional.of("Ly"), Optional.of("P40"), Optional.of("joulePerSquareMetre"), Optional.of("4.184 \u00D7 10\u2074 J/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_FLUENCE, QuantityKinds.RADIANCE_EXPOSURE)) ) );
UNITS_BY_NAME.put( "layer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "layer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "layer" ).build(), Optional.empty(), Optional.of("LR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "leaf", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "leaf" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "leaf" ).build(), Optional.empty(), Optional.of("LEF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lengthUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lengthUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "length" ).build(), Optional.empty(), Optional.of("LN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lightYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lightYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "light year" ).build(), Optional.of("ly"), Optional.of("B57"), Optional.of("metre"), Optional.of("9.46073 \u00D7 10\u00B9\u2075 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "linearFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "linearFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "linear foot" ).build(), Optional.empty(), Optional.of("LF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "linearMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "linearMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "linear metre" ).build(), Optional.empty(), Optional.of("LM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "linearYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "linearYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "linear yard" ).build(), Optional.empty(), Optional.of("LY"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "link", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "link" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "link" ).build(), Optional.empty(), Optional.of("LK"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "liquidPintUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "liquidPintUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "liquid pint (US)" ).build(), Optional.of("liq pt (US)"), Optional.of("PTL"), Optional.of("cubicMetre"), Optional.of("4. 731765 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "liquidPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "liquidPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "liquid pound" ).build(), Optional.empty(), Optional.of("LP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "liquidQuartUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "liquidQuartUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "liquid quart (US)" ).build(), Optional.of("liq qt (US)"), Optional.of("QTL"), Optional.of("cubicMetre"), Optional.of("9.463529 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "litre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre" ).build(), Optional.of("l"), Optional.of("LTR"), Optional.of("cubicMetre"), Optional.of("10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "litreOfPureAlcohol", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litreOfPureAlcohol" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre of pure alcohol" ).build(), Optional.empty(), Optional.of("LPA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "litrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per bar" ).build(), Optional.of("l/bar"), Optional.of("G95"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "litrePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per day" ).build(), Optional.of("l/d"), Optional.of("LD"), Optional.of("cubicMetrePerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per day bar" ).build(), Optional.of("l/(d\u00B7bar)"), Optional.of("G82"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B3 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per day kelvin" ).build(), Optional.of("l/(d\u00B7K)"), Optional.of("G65"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per hour" ).build(), Optional.of("l/h"), Optional.of("E32"), Optional.of("cubicMetrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2077 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "litrePerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per hour bar" ).build(), Optional.of("l/(h\u00B7bar)"), Optional.of("G83"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B2 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per hour kelvin" ).build(), Optional.of("l/(h\u00B7K)"), Optional.of("G66"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u2077 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
}

private static void init8() {
UNITS_BY_NAME.put( "litrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per kelvin" ).build(), Optional.of("l/K"), Optional.of("G28"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "litrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per kilogram" ).build(), Optional.of("l/kg"), Optional.of("H83"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "litrePerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per litre" ).build(), Optional.of("l/l"), Optional.of("K62"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "litrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per minute" ).build(), Optional.of("l/min"), Optional.of("L2"), Optional.of("cubicMetrePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per minute bar" ).build(), Optional.of("l/(min\u00B7bar)"), Optional.of("G84"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u2070 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per minute kelvin" ).build(), Optional.of("l/(min\u00B7K)"), Optional.of("G67"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per mole" ).build(), Optional.of("l/mol"), Optional.of("B58"), Optional.of("cubicMetrePerMole"), Optional.of("10\u207B\u00B3 m\u00B3/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_VOLUME)) ) );
UNITS_BY_NAME.put( "litrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per second" ).build(), Optional.of("l/s"), Optional.of("G51"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per second bar" ).build(), Optional.of("l/(s\u00B7bar)"), Optional.of("G85"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "litrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "litrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "litre per second kelvin" ).build(), Optional.of("l/(s\u00B7K)"), Optional.of("G68"), Optional.empty(), Optional.of("10\u207B\u00B3 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "load", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "load" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "load" ).build(), Optional.empty(), Optional.of("NL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lotUnitOfProcurement", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lotUnitOfProcurement" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lot  [unit of procurement]" ).build(), Optional.empty(), Optional.of("LO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lotUnitOfWeight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lotUnitOfWeight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lot  [unit of weight]" ).build(), Optional.empty(), Optional.of("D04"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lumen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lumen" ).build(), Optional.of("lm"), Optional.of("LUM"), Optional.empty(), Optional.of("cd \u00D7 sr"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_FLUX)) ) );
UNITS_BY_NAME.put( "lumenHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumenHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lumen hour" ).build(), Optional.of("lm\u00B7h"), Optional.of("B59"), Optional.empty(), Optional.of("3.6 \u00D7 10\u00B3 s \u00D7 cd \u00D7 sr"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_LIGHT)) ) );
UNITS_BY_NAME.put( "lumenPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumenPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lumen per square foot" ).build(), Optional.of("lm/ft\u00B2"), Optional.of("P25"), Optional.empty(), Optional.of("1.076391 \u00D7 10\u00B9 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "lumenPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumenPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lumen per square metre" ).build(), Optional.of("lm/m\u00B2"), Optional.of("B60"), Optional.empty(), Optional.of("cd \u00D7 sr/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_EXITANCE)) ) );
UNITS_BY_NAME.put( "lumenPerWatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumenPerWatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lumen per watt" ).build(), Optional.of("lm/W"), Optional.of("B61"), Optional.empty(), Optional.of("cd \u00D7 sr/W"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_LUMINOUS_EFFICACY, QuantityKinds.MAXIMUM_SPECTRAL_LUMINOUS_EFFICACY, QuantityKinds.LUMINOUS_EFFICACY_AT_A_SPECIFIED_WAVELENGTH, QuantityKinds.LUMINIOUS_EFFICACY)) ) );
UNITS_BY_NAME.put( "lumenSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumenSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lumen second" ).build(), Optional.of("lm\u00B7s"), Optional.of("B62"), Optional.empty(), Optional.of("s \u00D7 cd \u00D7 sr"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_LIGHT)) ) );
UNITS_BY_NAME.put( "lumpSum", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lumpSum" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lump sum" ).build(), Optional.empty(), Optional.of("LS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "lux", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "lux" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lux" ).build(), Optional.of("lx"), Optional.of("LUX"), Optional.empty(), Optional.of("cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "luxHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "luxHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lux hour" ).build(), Optional.of("lx\u00B7h"), Optional.of("B63"), Optional.empty(), Optional.of("3.6 \u00D7 10\u00B3 s \u00D7 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LIGHT_EXPOSURE)) ) );
UNITS_BY_NAME.put( "luxSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "luxSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "lux second" ).build(), Optional.of("lx\u00B7s"), Optional.of("B64"), Optional.empty(), Optional.of("s \u00D7 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LIGHT_EXPOSURE)) ) );
UNITS_BY_NAME.put( "manmonth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "manmonth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "manmonth" ).build(), Optional.empty(), Optional.of("3C"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "meal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "meal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "meal" ).build(), Optional.empty(), Optional.of("Q3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mebibit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mebibit" ).build(), Optional.of("Mibit"), Optional.of("D11"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mebibitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mebibit per cubic metre" ).build(), Optional.of("Mibit/m\u00B3"), Optional.of("E77"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mebibitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mebibit per metre" ).build(), Optional.of("Mibit/m"), Optional.of("E75"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mebibitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mebibit per square metre" ).build(), Optional.of("Mibit/m\u00B2"), Optional.of("E76"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mebibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mebibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mebibyte" ).build(), Optional.of("Mibyte"), Optional.of("E63"), Optional.of("byte"), Optional.of("2\u00B2\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "megaampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaampere" ).build(), Optional.of("MA"), Optional.of("H38"), Optional.of("ampere"), Optional.of("10\u2076 A"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "megaamperePerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaamperePerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaampere per square metre" ).build(), Optional.of("MA/m\u00B2"), Optional.of("B66"), Optional.of("amperePerSquareMetre"), Optional.of("10\u2076 A/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "megabaud", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabaud" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megabaud" ).build(), Optional.of("MBd"), Optional.of("J54"), Optional.of("baud"), Optional.of("10\u2076 Bd"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megabecquerel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabecquerel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megabecquerel" ).build(), Optional.of("MBq"), Optional.of("4N"), Optional.of("becquerel"), Optional.of("10\u2076 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "megabecquerelPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabecquerelPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megabecquerel per kilogram" ).build(), Optional.of("MBq/kg"), Optional.of("B67"), Optional.of("becquerelPerKilogram"), Optional.of("10\u2076 Bq/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ACTIVITY_IN_A_SAMPLE)) ) );
UNITS_BY_NAME.put( "megabit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megabit" ).build(), Optional.of("Mbit"), Optional.of("D36"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megabitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megabit per second" ).build(), Optional.of("Mbit/s"), Optional.of("E20"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Megabyte" ).build(), Optional.of("MB"), Optional.of("4L"), Optional.of("byte"), Optional.of("10\u2076 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "megabytePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megabytePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megabyte per second" ).build(), Optional.of("Mbyte/s"), Optional.of("P95"), Optional.of("bytePerSecond"), Optional.of("10\u2076 byte/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megacoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megacoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megacoulomb" ).build(), Optional.of("MC"), Optional.of("D77"), Optional.of("coulomb"), Optional.of("10\u2076 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "megacoulombPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megacoulombPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megacoulomb per cubic metre" ).build(), Optional.of("MC/m\u00B3"), Optional.of("B69"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u2076 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE)) ) );
UNITS_BY_NAME.put( "megacoulombPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megacoulombPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megacoulomb per square metre" ).build(), Optional.of("MC/m\u00B2"), Optional.of("B70"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u2076 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.SURFACE_DENSITY_OF_CHARGE)) ) );
UNITS_BY_NAME.put( "megaelectronvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaelectronvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaelectronvolt" ).build(), Optional.of("MeV"), Optional.of("B71"), Optional.of("electronvolt"), Optional.of("10\u2076 eV"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "megagram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megagram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megagram" ).build(), Optional.of("Mg"), Optional.of("2U"), Optional.of("kilogram"), Optional.of("10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "megagramPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megagramPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megagram per cubic metre" ).build(), Optional.of("Mg/m\u00B3"), Optional.of("B72"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "megahertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megahertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megahertz" ).build(), Optional.of("MHz"), Optional.of("MHZ"), Optional.of("hertz"), Optional.of("10\u2076 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "megahertzKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megahertzKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megahertz kilometre" ).build(), Optional.of("MHz\u00B7km"), Optional.of("H39"), Optional.of("hertzMetre"), Optional.of("10\u2079 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "megahertzMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megahertzMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megahertz metre" ).build(), Optional.of("MHz\u00B7m"), Optional.of("M27"), Optional.of("hertzMetre"), Optional.of("10\u2076 Hz \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "megajoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megajoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megajoule" ).build(), Optional.of("MJ"), Optional.of("3B"), Optional.of("joule"), Optional.of("10\u2076 J"), new HashSet<>(Arrays.asList(QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "megajoulePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megajoulePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megajoule per cubic metre" ).build(), Optional.of("MJ/m\u00B3"), Optional.of("JM"), Optional.of("joulePerCubicMetre"), Optional.of("10\u2076 J/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY_DENSITY)) ) );
UNITS_BY_NAME.put( "megajoulePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megajoulePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megajoule per kilogram" ).build(), Optional.of("MJ/kg"), Optional.of("JK"), Optional.of("joulePerKilogram"), Optional.of("10\u2076 J/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_THERMODYNAMIC_ENERGY)) ) );
UNITS_BY_NAME.put( "megajoulePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megajoulePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megajoule per second" ).build(), Optional.of("MJ/s"), Optional.of("D78"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megalitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megalitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megalitre" ).build(), Optional.of("Ml"), Optional.of("MAL"), Optional.of("cubicMetre"), Optional.of("10\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "megametre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megametre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megametre" ).build(), Optional.of("Mm"), Optional.of("MAM"), Optional.of("metre"), Optional.of("10\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "meganewton", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "meganewton" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "meganewton" ).build(), Optional.of("MN"), Optional.of("B73"), Optional.of("newton"), Optional.of("10\u2076 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "meganewtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "meganewtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "meganewton metre" ).build(), Optional.of("MN\u00B7m"), Optional.of("B74"), Optional.of("newtonMetre"), Optional.of("10\u2076 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "megaohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaohm" ).build(), Optional.of("M\u2126"), Optional.of("B75"), Optional.of("ohm"), Optional.of("10\u2076 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.IMPEDANCE, QuantityKinds.REACTANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.COMPLEX_IMPEDANCES)) ) );
UNITS_BY_NAME.put( "megaohmKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaohmKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaohm kilometre" ).build(), Optional.of("M\u2126\u00B7km"), Optional.of("H88"), Optional.of("ohmMetre"), Optional.of("10\u2079 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "megaohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaohm metre" ).build(), Optional.of("M\u2126\u00B7m"), Optional.of("B76"), Optional.of("ohmMetre"), Optional.of("10\u2076 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "megaohmPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaohmPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaohm per kilometre" ).build(), Optional.of("M\u2126/km"), Optional.of("H36"), Optional.of("ohmPerMetre"), Optional.of("10\u00B3 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "megaohmPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megaohmPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megaohm per metre" ).build(), Optional.of("M\u2126/m"), Optional.of("H37"), Optional.of("ohmPerMetre"), Optional.of("10\u2076 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "megapascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megapascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megapascal" ).build(), Optional.of("MPa"), Optional.of("MPA"), Optional.of("pascal"), Optional.of("10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "megapascalCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megapascalCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megapascal cubic metre per second" ).build(), Optional.of("MPa\u00B7m\u00B3/s"), Optional.of("F98"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "megapascalLitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megapascalLitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megapascal litre per second" ).build(), Optional.of("MPa\u00B7l/s"), Optional.of("F97"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "megapascalPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megapascalPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megapascal per bar" ).build(), Optional.of("MPa/bar"), Optional.of("F05"), Optional.empty(), Optional.of("10\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "megapascalPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megapascalPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megapascal per kelvin" ).build(), Optional.of("MPa/K"), Optional.of("F85"), Optional.empty(), Optional.of("10\u2076 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "megapixel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megapixel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megapixel" ).build(), Optional.empty(), Optional.of("E38"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megasiemensPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megasiemensPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megasiemens per metre" ).build(), Optional.of("MS/m"), Optional.of("B77"), Optional.of("siemensPerMetre"), Optional.of("10\u2076 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "megavar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megavar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megavar" ).build(), Optional.of("Mvar"), Optional.of("MAR"), Optional.of("voltAmpere"), Optional.of("10\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.REACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "megavolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megavolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megavolt" ).build(), Optional.of("MV"), Optional.of("B78"), Optional.of("voltDc"), Optional.of("10\u2076 V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.VOLTAGE, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.TENSION, QuantityKinds.ELECTRIC_POTENTIAL)) ) );
UNITS_BY_NAME.put( "megavoltAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megavoltAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megavolt - ampere" ).build(), Optional.of("MV\u00B7A"), Optional.of("MVA"), Optional.of("voltAmpere"), Optional.of("10\u2076 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "megavoltAmpereReactiveHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megavoltAmpereReactiveHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megavolt ampere reactive hour" ).build(), Optional.of("Mvar\u00B7h"), Optional.of("MAH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megavoltPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megavoltPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megavolt per metre" ).build(), Optional.of("MV/m"), Optional.of("B79"), Optional.of("voltPerMetre"), Optional.of("10\u2076 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "megawatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megawatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megawatt" ).build(), Optional.of("MW"), Optional.of("MAW"), Optional.of("watt"), Optional.of("10\u2076 W"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "megawattHour1000Kwh", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megawattHour1000Kwh" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megawatt hour (1000 kW.h)" ).build(), Optional.of("MW\u00B7h"), Optional.of("MWH"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u2079 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "megawattHourPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megawattHourPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megawatt hour per hour" ).build(), Optional.of("MW\u00B7h/h"), Optional.of("E07"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "megawattPerHertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "megawattPerHertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "megawatt per hertz" ).build(), Optional.of("MW/Hz"), Optional.of("E08"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "mesh", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mesh" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mesh" ).build(), Optional.empty(), Optional.of("57"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "message", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "message" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "message" ).build(), Optional.empty(), Optional.of("NF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre" ).build(), Optional.of("m"), Optional.of("MTR"), Optional.empty(), Optional.of("m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.EQUILIBRIUM_POSITION_VECTOR_OF_ION_OR_ATOM, QuantityKinds.HEIGHT, QuantityKinds.PARTICLE_POSITION_VECTOR, QuantityKinds.ELECTRON_RADIUS, QuantityKinds.BOHR_RADIUS, QuantityKinds.FUNDAMENTAL_LATTICE_VECTOR, QuantityKinds.LONDON_PENETRATION_DEPTH, QuantityKinds.LENGTH, QuantityKinds.COHERENCE_LENGTH, QuantityKinds.BREADTH, QuantityKinds.OBJECT_DISTANCE, QuantityKinds.IMAGE_DISTANCE, QuantityKinds.INSTANTANEOUS_SOUND_PARTICLE_DISPLACEMENT, QuantityKinds.MIGRATION_LENGTH, QuantityKinds.DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUX_DENSITY, QuantityKinds.MEAN_LINEAR_RANGE, QuantityKinds.NUCLEAR_RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.MEAN_FREE_PATH, QuantityKinds.MEAN_FREE_PATH_OF_PHONONS_OR_ELECTRONS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.SLOWING_DOWN_LENGTH, QuantityKinds.DIFFUSION_LENGTH, QuantityKinds.DISPLACEMENT_VECTOR_OF_ION_OR_ATOM, QuantityKinds.WAVELENGTH, QuantityKinds.DISTANCE, QuantityKinds.COMPTON_WAVELENGTH, QuantityKinds.RADIUS, QuantityKinds.FOCAL_DISTANCE, QuantityKinds.LATTICE_VECTOR, QuantityKinds.DIFFUSION_COEFFICIENT_FOR_NEUTRON_FLUENCE_RATE, QuantityKinds.HALF_VALUE_THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.HALF_THICKNESS)) ) );
UNITS_BY_NAME.put( "metreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre kelvin" ).build(), Optional.of("m\u00B7K"), Optional.of("D18"), Optional.empty(), Optional.of("m \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_RADIATION_CONSTANT)) ) );
UNITS_BY_NAME.put( "metrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per bar" ).build(), Optional.of("m/bar"), Optional.of("G05"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "metrePerDegreeCelsiusMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerDegreeCelsiusMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per degree Celsius metre" ).build(), Optional.of("m/(\u00B0C\u00B7m)"), Optional.of("N83"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "metrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per hour" ).build(), Optional.of("m/h"), Optional.of("M60"), Optional.of("metrePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per kelvin" ).build(), Optional.of("m/K"), Optional.of("F52"), Optional.empty(), Optional.of("m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "metrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per minute" ).build(), Optional.of("m/min"), Optional.of("2X"), Optional.of("metrePerSecond"), Optional.of("0.016666 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per pascal" ).build(), Optional.of("m/Pa"), Optional.of("M53"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "metrePerRadiant", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerRadiant" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per radiant" ).build(), Optional.of("m/rad"), Optional.of("M55"), Optional.empty(), Optional.of("m/rad"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "metrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per second" ).build(), Optional.of("m/s"), Optional.of("MTS"), Optional.empty(), Optional.of("m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.PHASE_SPEED_OF_ELECTROMAGNETIC_WAVES, QuantityKinds.VELOCITY_OF_SOUND_PHASE_VELOCITY, QuantityKinds.PHASE_VELOCITY_OF_ELECTROMAGNETIC_WAVES, QuantityKinds.NEUTRON_SPEED, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY_SPEED_ON_PROPAGATION_OF_ELECTROMAGNETIC_WAVES_IN_VACUO, QuantityKinds.INSTANTANEOUS_SOUND_PARTICLE_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per second bar" ).build(), Optional.of("(m/s)/bar"), Optional.of("L13"), Optional.of("metrePerSecondPascal"), Optional.of("10\u207B\u2075 (m/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per second kelvin" ).build(), Optional.of("(m/s)/K"), Optional.of("L12"), Optional.empty(), Optional.of("(m/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerSecondPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per second pascal" ).build(), Optional.of("(m/s)/Pa"), Optional.of("M59"), Optional.empty(), Optional.of("m\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "metrePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per second squared" ).build(), Optional.of("m/s\u00B2"), Optional.of("MSK"), Optional.empty(), Optional.of("m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.INSTANTANEOUS_SOUND_PARTICLE_ACCELERATION, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "metrePerVoltSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metrePerVoltSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre per volt second" ).build(), Optional.of("m/(V\u00B7s)"), Optional.of("H58"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.MOBILITY)) ) );
UNITS_BY_NAME.put( "metreToTheFourthPower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metreToTheFourthPower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metre to the fourth power" ).build(), Optional.of("m\u2074"), Optional.of("B83"), Optional.empty(), Optional.of("m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_MOMENT_OF_AREA, QuantityKinds.SECOND_AXIAL_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "metricCarat", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metricCarat" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metric carat" ).build(), Optional.empty(), Optional.of("CTM"), Optional.of("milligram"), Optional.of("200 mg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metricTonIncludingContainer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metricTonIncludingContainer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metric ton, including container" ).build(), Optional.empty(), Optional.of("TIC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metricTonIncludingInnerPackaging", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metricTonIncludingInnerPackaging" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metric ton, including inner packaging" ).build(), Optional.empty(), Optional.of("TIP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "metricTonLubricatingOil", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "metricTonLubricatingOil" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "metric ton, lubricating oil" ).build(), Optional.empty(), Optional.of("LUB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "microInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micro-inch" ).build(), Optional.of("\u00B5in"), Optional.of("M7"), Optional.of("metre"), Optional.of("25.4 \u00D7 10\u207B\u2079 m"), new HashSet<>(Arrays.asList(QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.THICKNESS, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE)) ) );
}

private static void init9() {
UNITS_BY_NAME.put( "microampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microampere" ).build(), Optional.of("\u00B5A"), Optional.of("B84"), Optional.of("ampere"), Optional.of("10\u207B\u2076 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_LINKAGE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.MAGNETOMOTIVE_FORCE)) ) );
UNITS_BY_NAME.put( "microbar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microbar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microbar" ).build(), Optional.of("\u00B5bar"), Optional.of("B85"), Optional.of("pascal"), Optional.of("10\u207B\u00B9 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "microbecquerel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microbecquerel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microbecquerel" ).build(), Optional.of("\u00B5Bq"), Optional.of("H08"), Optional.of("becquerel"), Optional.of("10\u207B\u2076 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "microcoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microcoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microcoulomb" ).build(), Optional.of("\u00B5C"), Optional.of("B86"), Optional.of("coulomb"), Optional.of("10\u207B\u2076 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE)) ) );
UNITS_BY_NAME.put( "microcoulombPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microcoulombPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microcoulomb per cubic metre" ).build(), Optional.of("\u00B5C/m\u00B3"), Optional.of("B87"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u207B\u2076 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.VOLUMIC_CHARGE, QuantityKinds.CHARGE_DENSITY)) ) );
UNITS_BY_NAME.put( "microcoulombPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microcoulombPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microcoulomb per square metre" ).build(), Optional.of("\u00B5C/m\u00B2"), Optional.of("B88"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u207B\u2076 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_DENSITY, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.SURFACE_DENSITY_OF_CHARGE)) ) );
UNITS_BY_NAME.put( "microcurie", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microcurie" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microcurie" ).build(), Optional.of("\u00B5Ci"), Optional.of("M5"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u2074 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "microfarad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microfarad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microfarad" ).build(), Optional.of("\u00B5F"), Optional.of("4O"), Optional.of("farad"), Optional.of("10\u207B\u2076 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "microfaradPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microfaradPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microfarad per kilometre" ).build(), Optional.of("\u00B5F/km"), Optional.of("H28"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u2079 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CONSTANT, QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.PERMITTIVITY)) ) );
UNITS_BY_NAME.put( "microfaradPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microfaradPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microfarad per metre" ).build(), Optional.of("\u00B5F/m"), Optional.of("B89"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u2076 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY, QuantityKinds.ELECTRIC_CONSTANT, QuantityKinds.PERMITTIVITY_OF_VACUUM)) ) );
UNITS_BY_NAME.put( "microgram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microgram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgram" ).build(), Optional.of("\u00B5g"), Optional.of("MC"), Optional.of("kilogram"), Optional.of("10\u207B\u2079 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "microgramPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microgramPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgram per cubic metre" ).build(), Optional.of("\u00B5g/m\u00B3"), Optional.of("GQ"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u2079 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "microgramPerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microgramPerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgram per cubic metre bar" ).build(), Optional.of("(\u00B5g/m\u00B3)/bar"), Optional.of("J35"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("10\u207B\u00B9\u2074 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "microgramPerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microgramPerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgram per cubic metre kelvin" ).build(), Optional.of("(\u00B5g/m\u00B3)/K"), Optional.of("J34"), Optional.empty(), Optional.of("10\u207B\u2079 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "microgramPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microgramPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgram per kilogram" ).build(), Optional.of("\u00B5g/kg"), Optional.of("J33"), Optional.empty(), Optional.of("10\u207B\u2079"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "microgramPerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microgramPerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgram per litre" ).build(), Optional.of("\u00B5g/l"), Optional.of("H29"), Optional.empty(), Optional.of("10\u207B\u2076 m\u207B\u00B3 \u00D7 kg"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "micrograyPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micrograyPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgray per hour" ).build(), Optional.of("\u00B5Gy/h"), Optional.of("P63"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "micrograyPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micrograyPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgray per minute" ).build(), Optional.of("\u00B5Gy/min"), Optional.of("P59"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2078 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "micrograyPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micrograyPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microgray per second" ).build(), Optional.of("\u00B5Gy/s"), Optional.of("P55"), Optional.of("grayPerSecond"), Optional.of("10\u207B\u2076 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "microhenry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microhenry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microhenry" ).build(), Optional.of("\u00B5H"), Optional.of("B90"), Optional.of("henry"), Optional.of("10\u207B\u2076 H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE)) ) );
UNITS_BY_NAME.put( "microhenryPerKiloohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microhenryPerKiloohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microhenry per kiloohm" ).build(), Optional.of("\u00B5H/k\u2126"), Optional.of("G98"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2079 s"), new HashSet<>(Arrays.asList(QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "microhenryPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microhenryPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microhenry per metre" ).build(), Optional.of("\u00B5H/m"), Optional.of("B91"), Optional.of("henryPerMetre"), Optional.of("10\u207B\u2076 H/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEABILITY_OF_VACUUM, QuantityKinds.PERMEABILITY, QuantityKinds.MAGNETIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "microhenryPerOhm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microhenryPerOhm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microhenry per ohm" ).build(), Optional.of("\u00B5H/\u2126"), Optional.of("G99"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.SELF_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "microlitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microlitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microlitre" ).build(), Optional.of("\u00B5l"), Optional.of("4G"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2079 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "microlitrePerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microlitrePerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microlitre per litre" ).build(), Optional.of("\u00B5l/l"), Optional.of("J36"), Optional.empty(), Optional.of("10\u207B\u2076"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "micrometreMicron", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micrometreMicron" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micrometre (micron)" ).build(), Optional.of("\u00B5m"), Optional.of("4H"), Optional.of("metre"), Optional.of("10\u207B\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.HEIGHT, QuantityKinds.LENGTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.THICKNESS, QuantityKinds.RADIUS, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "micrometrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micrometrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micrometre per kelvin" ).build(), Optional.of("\u00B5m/K"), Optional.of("F50"), Optional.empty(), Optional.of("10\u207B\u2076 m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.RADIUS, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "micromole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micromole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micromole" ).build(), Optional.of("\u00B5mol"), Optional.of("FH"), Optional.of("mole"), Optional.of("10\u207B\u2076 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "micronewton", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micronewton" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micronewton" ).build(), Optional.of("\u00B5N"), Optional.of("B92"), Optional.of("newton"), Optional.of("10\u207B\u2076 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "micronewtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micronewtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micronewton metre" ).build(), Optional.of("\u00B5N\u00B7m"), Optional.of("B93"), Optional.of("newtonMetre"), Optional.of("10\u207B\u2076 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "microohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microohm" ).build(), Optional.of("\u00B5\u2126"), Optional.of("B94"), Optional.of("ohm"), Optional.of("10\u207B\u2076 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.REACTANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "microohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microohm metre" ).build(), Optional.of("\u00B5\u2126\u00B7m"), Optional.of("B95"), Optional.of("ohmMetre"), Optional.of("10\u207B\u2076 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "micropascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micropascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micropascal" ).build(), Optional.of("\u00B5Pa"), Optional.of("B96"), Optional.of("pascal"), Optional.of("10\u207B\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "micropoise", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "micropoise" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "micropoise" ).build(), Optional.of("\u00B5P"), Optional.of("J32"), Optional.of("pascalSecond"), Optional.of("10\u207B\u2076 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "microradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microradian" ).build(), Optional.of("\u00B5rad"), Optional.of("B97"), Optional.of("radian"), Optional.of("10\u207B\u2076 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "microsecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsecond" ).build(), Optional.of("\u00B5s"), Optional.of("B98"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "microsiemens", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsiemens" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsiemens" ).build(), Optional.of("\u00B5S"), Optional.of("B99"), Optional.of("siemens"), Optional.of("10\u207B\u2076 S"), new HashSet<>(Arrays.asList(QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE)) ) );
UNITS_BY_NAME.put( "microsiemensPerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsiemensPerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsiemens per centimetre" ).build(), Optional.of("\u00B5S/cm"), Optional.of("G42"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2074 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.ADMITTANCE)) ) );
UNITS_BY_NAME.put( "microsiemensPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsiemensPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsiemens per metre" ).build(), Optional.of("\u00B5S/m"), Optional.of("G43"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2076 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.ADMITTANCE)) ) );
UNITS_BY_NAME.put( "microsievertPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsievertPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsievert per hour" ).build(), Optional.of("\u00B5Sv/h"), Optional.of("P72"), Optional.of("sievertPerSecond"), Optional.of("0.277777778 \u00D7 10\u207B\u00B9\u2070 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "microsievertPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsievertPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsievert per minute" ).build(), Optional.of("\u00B5Sv/min"), Optional.of("P76"), Optional.of("sievertPerSecond"), Optional.of("1.666666667 \u00D7 10\u207B\u2078 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "microsievertPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microsievertPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microsievert per second" ).build(), Optional.of("\u00B5Sv/s"), Optional.of("P67"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u2076 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "microtesla", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microtesla" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microtesla" ).build(), Optional.of("\u00B5T"), Optional.of("D81"), Optional.of("tesla"), Optional.of("10\u207B\u2076 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_INDUCTION, QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "microvolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microvolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microvolt" ).build(), Optional.of("\u00B5V"), Optional.of("D82"), Optional.of("voltDc"), Optional.of("10\u207B\u2076 V"), new HashSet<>(Arrays.asList(QuantityKinds.VOLTAGE, QuantityKinds.TENSION, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.ELECTRIC_POTENTIAL)) ) );
UNITS_BY_NAME.put( "microvoltPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microvoltPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microvolt per metre" ).build(), Optional.of("\u00B5V/m"), Optional.of("C3"), Optional.of("voltPerMetre"), Optional.of("10\u207B\u2076 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "microwatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microwatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microwatt" ).build(), Optional.of("\u00B5W"), Optional.of("D80"), Optional.of("watt"), Optional.of("10\u207B\u2076 W"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "microwattPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "microwattPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "microwatt per square metre" ).build(), Optional.of("\u00B5W/m\u00B2"), Optional.of("D85"), Optional.of("wattPerSquareMetre"), Optional.of("10\u207B\u2076 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY)) ) );
UNITS_BY_NAME.put( "mil", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mil" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mil" ).build(), Optional.of("mil"), Optional.of("M43"), Optional.of("radian"), Optional.of("9.817477 \u00D7 10\u207B\u2074 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "mileBasedOnUsSurveyFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mileBasedOnUsSurveyFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mile (based on U.S. survey foot)" ).build(), Optional.of("mi (US survey)"), Optional.of("M52"), Optional.of("metre"), Optional.of("1.609347 \u00D7 10\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "milePerHourStatuteMile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milePerHourStatuteMile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mile per hour (statute mile)" ).build(), Optional.of("mile/h"), Optional.of("HM"), Optional.of("metrePerSecond"), Optional.of("0.44704 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "milePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mile per minute" ).build(), Optional.of("mi/min"), Optional.of("M57"), Optional.of("metrePerSecond"), Optional.of("26.8224 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "milePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mile per second" ).build(), Optional.of("mi/s"), Optional.of("M58"), Optional.of("metrePerSecond"), Optional.of("1.609344 \u00D7 10\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "mileStatuteMile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mileStatuteMile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mile (statute mile)" ).build(), Optional.of("mile"), Optional.of("SMI"), Optional.of("metre"), Optional.of("1609.344 m"), new HashSet<>(Arrays.asList(QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "mileStatuteMilePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mileStatuteMilePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mile (statute mile) per second squared" ).build(), Optional.of("mi/s\u00B2"), Optional.of("M42"), Optional.of("metrePerSecondSquared"), Optional.of("1.609344 \u00D7 10\u00B3 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION_OF_FREE_FALL)) ) );
UNITS_BY_NAME.put( "mille", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mille" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mille" ).build(), Optional.empty(), Optional.of("E12"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "milliInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milli-inch" ).build(), Optional.of("mil"), Optional.of("77"), Optional.of("metre"), Optional.of("25.4 \u00D7 10\u207B\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE, QuantityKinds.LENGTH, QuantityKinds.BREADTH, QuantityKinds.HEIGHT, QuantityKinds.DIAMETER, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "milliampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere" ).build(), Optional.of("mA"), Optional.of("4K"), Optional.of("ampere"), Optional.of("10\u207B\u00B3 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.ELECTRIC_CURRENT)) ) );
UNITS_BY_NAME.put( "milliampereHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliampereHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere hour" ).build(), Optional.of("mA\u00B7h"), Optional.of("E09"), Optional.of("coulomb"), Optional.of("3.6 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY)) ) );
UNITS_BY_NAME.put( "milliamperePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliamperePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere per bar" ).build(), Optional.of("mA/bar"), Optional.of("F59"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliamperePerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere per inch" ).build(), Optional.of("mA/in"), Optional.of("F08"), Optional.empty(), Optional.of("3.93700787401575 \u00D7 10\u207B\u00B2 A \u00D7 m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerLitreMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliamperePerLitreMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere per litre minute" ).build(), Optional.of("mA/(l\u00B7min)"), Optional.of("G59"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B2 m\u207B\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliamperePerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere per millimetre" ).build(), Optional.of("mA/mm"), Optional.of("F76"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FIELD_STRENGTH, QuantityKinds.LINEIC_ELECTRIC_CURRENT, QuantityKinds.LINEAR_ELECTRIC_CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliamperePerPoundForcePerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliamperePerPoundForcePerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliampere per pound-force per square inch" ).build(), Optional.of("mA/(lbf/in\u00B2)"), Optional.of("F57"), Optional.empty(), Optional.of("1.45038 \u00D7 10\u207B\u2077 kg\u207B\u00B9 \u00D7 m \u00D7 s\u00B2 \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.CURRENT_DENSITY)) ) );
UNITS_BY_NAME.put( "milliard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliard" ).build(), Optional.empty(), Optional.of("MLD"), Optional.empty(), Optional.of("10\u2079"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millibar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millibar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millibar" ).build(), Optional.of("mbar"), Optional.of("MBR"), Optional.of("pascal"), Optional.of("10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "millibarCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millibarCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millibar cubic metre per second" ).build(), Optional.of("mbar\u00B7m\u00B3/s"), Optional.of("F96"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION)) ) );
UNITS_BY_NAME.put( "millibarLitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millibarLitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millibar litre per second" ).build(), Optional.of("mbar\u00B7l/s"), Optional.of("F95"), Optional.empty(), Optional.of("10\u207B\u00B9 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "millibarPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millibarPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millibar per bar" ).build(), Optional.of("mbar/bar"), Optional.of("F04"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "millibarPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millibarPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millibar per kelvin" ).build(), Optional.of("mbar/K"), Optional.of("F84"), Optional.empty(), Optional.of("10\u00B2 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B2 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "millicandela", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millicandela" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millicandela" ).build(), Optional.of("mcd"), Optional.of("P34"), Optional.of("candela"), Optional.of("10\u207B\u00B3 cd"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINOUS_INTENSITY)) ) );
UNITS_BY_NAME.put( "millicoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millicoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millicoulomb" ).build(), Optional.of("mC"), Optional.of("D86"), Optional.of("coulomb"), Optional.of("10\u207B\u00B3 C"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "millicoulombPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millicoulombPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millicoulomb per cubic metre" ).build(), Optional.of("mC/m\u00B3"), Optional.of("D88"), Optional.of("coulombPerCubicMetre"), Optional.of("10\u207B\u00B3 C/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_DENSITY_OF_CHARGE, QuantityKinds.CHARGE_DENSITY, QuantityKinds.VOLUMIC_CHARGE)) ) );
UNITS_BY_NAME.put( "millicoulombPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millicoulombPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millicoulomb per kilogram" ).build(), Optional.of("mC/kg"), Optional.of("C8"), Optional.of("coulombPerKilogram"), Optional.of("10\u207B\u00B3 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "millicoulombPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millicoulombPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millicoulomb per square metre" ).build(), Optional.of("mC/m\u00B2"), Optional.of("D89"), Optional.of("coulombPerSquareMetre"), Optional.of("10\u207B\u00B3 C/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY_OF_CHARGE, QuantityKinds.DISPLACEMENT_ELECTRIC_POLARIZATION, QuantityKinds.ELECTRIC_FLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "millicurie", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millicurie" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millicurie" ).build(), Optional.of("mCi"), Optional.of("MCU"), Optional.of("becquerel"), Optional.of("3.7 \u00D7 10\u2077 Bq"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVITY)) ) );
UNITS_BY_NAME.put( "milliequivalenceCausticPotashPerGramOfProduct", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliequivalenceCausticPotashPerGramOfProduct" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliequivalence caustic potash per gram of product" ).build(), Optional.empty(), Optional.of("KO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millifarad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millifarad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millifarad" ).build(), Optional.of("mF"), Optional.of("C10"), Optional.of("farad"), Optional.of("10\u207B\u00B3 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "milligal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligal" ).build(), Optional.of("mGal"), Optional.of("C11"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u2075 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY)) ) );
UNITS_BY_NAME.put( "milligram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram" ).build(), Optional.of("mg"), Optional.of("MGM"), Optional.of("kilogram"), Optional.of("10\u207B\u2076 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "milligramPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per bar" ).build(), Optional.of("mg/bar"), Optional.of("F75"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 m \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per cubic metre" ).build(), Optional.of("mg/m\u00B3"), Optional.of("GP"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u2076 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per cubic metre bar" ).build(), Optional.of("(mg/m\u00B3)/bar"), Optional.of("L18"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("10\u207B\u00B9\u00B9 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per cubic metre kelvin" ).build(), Optional.of("(mg/m\u00B3)/K"), Optional.of("L17"), Optional.empty(), Optional.of("10\u207B\u2076 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per day" ).build(), Optional.of("mg/d"), Optional.of("F32"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per day bar" ).build(), Optional.of("mg/(d\u00B7bar)"), Optional.of("F70"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2076 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per day kelvin" ).build(), Optional.of("mg/(d\u00B7K)"), Optional.of("F43"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per gram" ).build(), Optional.of("mg/g"), Optional.of("H64"), Optional.empty(), Optional.of("10\u207B\u00B3 1"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per hour" ).build(), Optional.of("mg/h"), Optional.of("4M"), Optional.of("kilogramPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per hour bar" ).build(), Optional.of("mg/(h\u00B7bar)"), Optional.of("F71"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2075 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per hour kelvin" ).build(), Optional.of("mg/(h\u00B7K)"), Optional.of("F44"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per kelvin" ).build(), Optional.of("mg/K"), Optional.of("F16"), Optional.empty(), Optional.of("10\u207B\u2076 kg \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per kilogram" ).build(), Optional.of("mg/kg"), Optional.of("NA"), Optional.empty(), Optional.of("10\u207B\u2076 1"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "milligramPerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per litre" ).build(), Optional.of("mg/l"), Optional.of("M1"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u207B\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "milligramPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per metre" ).build(), Optional.of("mg/m"), Optional.of("C12"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u2076 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "milligramPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per minute" ).build(), Optional.of("mg/min"), Optional.of("F33"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per minute bar" ).build(), Optional.of("mg/(min\u00B7bar)"), Optional.of("F72"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B3 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per minute kelvin" ).build(), Optional.of("mg/(min\u00B7K)"), Optional.of("F45"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per second" ).build(), Optional.of("mg/s"), Optional.of("F34"), Optional.empty(), Optional.of("10\u207B\u2076 kg \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per second bar" ).build(), Optional.of("mg/(s\u00B7bar)"), Optional.of("F73"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "milligramPerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per second kelvin" ).build(), Optional.of("mg/(s\u00B7K)"), Optional.of("F46"), Optional.empty(), Optional.of("10\u207B\u2076 kg \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
}

private static void init10() {
UNITS_BY_NAME.put( "milligramPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per square centimetre" ).build(), Optional.of("mg/cm\u00B2"), Optional.of("H63"), Optional.empty(), Optional.of("10\u207B\u00B2 m\u207B\u00B2 \u00D7 kg"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "milligramPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligramPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligram per square metre" ).build(), Optional.of("mg/m\u00B2"), Optional.of("GO"), Optional.of("kilogramPerSquareMetre"), Optional.of("10\u207B\u2076 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "milligray", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligray" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligray" ).build(), Optional.of("mGy"), Optional.of("C13"), Optional.of("gray"), Optional.of("10\u207B\u00B3 Gy"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_ENERGY_IMPARTED, QuantityKinds.MASSIC_ENERGY_IMPARTED)) ) );
UNITS_BY_NAME.put( "milligrayPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligrayPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligray per hour" ).build(), Optional.of("mGy/h"), Optional.of("P62"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2077 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "milligrayPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligrayPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligray per minute" ).build(), Optional.of("mGy/min"), Optional.of("P58"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2075 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "milligrayPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milligrayPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milligray per second" ).build(), Optional.of("mGy/s"), Optional.of("P54"), Optional.of("grayPerSecond"), Optional.of("10\u207B\u00B3 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "millihenry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millihenry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millihenry" ).build(), Optional.of("mH"), Optional.of("C14"), Optional.of("henry"), Optional.of("10\u207B\u00B3 H"), new HashSet<>(Arrays.asList(QuantityKinds.SELF_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "millihenryPerKiloohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millihenryPerKiloohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millihenry per kiloohm" ).build(), Optional.of("mH/k\u2126"), Optional.of("H05"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "millihenryPerOhm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millihenryPerOhm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millihenry per ohm" ).build(), Optional.of("mH/\u2126"), Optional.of("H06"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "millijoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millijoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millijoule" ).build(), Optional.of("mJ"), Optional.of("C15"), Optional.of("joule"), Optional.of("10\u207B\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY, QuantityKinds.WORK, QuantityKinds.KINETIC_ENERGY)) ) );
UNITS_BY_NAME.put( "millilitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre" ).build(), Optional.of("ml"), Optional.of("MLT"), Optional.of("cubicMetre"), Optional.of("10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per bar" ).build(), Optional.of("ml/bar"), Optional.of("G97"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per cubic metre" ).build(), Optional.of("ml/m\u00B3"), Optional.of("H65"), Optional.empty(), Optional.of("10\u207B\u2076 1"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per day" ).build(), Optional.of("ml/d"), Optional.of("G54"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per day bar" ).build(), Optional.of("ml/(d\u00B7bar)"), Optional.of("G90"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u2076 \u00D7 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per day kelvin" ).build(), Optional.of("ml/(d\u00B7K)"), Optional.of("G73"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B9\u00B9 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per hour" ).build(), Optional.of("ml/h"), Optional.of("G55"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per hour bar" ).build(), Optional.of("ml/(h\u00B7bar)"), Optional.of("G91"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2075 \u00D7 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per hour kelvin" ).build(), Optional.of("ml/(h\u00B7K)"), Optional.of("G74"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u2070 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per kelvin" ).build(), Optional.of("ml/K"), Optional.of("G30"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_PER_TEMPERATURE)) ) );
UNITS_BY_NAME.put( "millilitrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per kilogram" ).build(), Optional.of("ml/kg"), Optional.of("KX"), Optional.of("cubicMetrePerKilogram"), Optional.of("10\u207B\u2076 m\u00B3/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "millilitrePerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per litre" ).build(), Optional.of("ml/l"), Optional.of("L19"), Optional.empty(), Optional.of("10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_RATIO)) ) );
UNITS_BY_NAME.put( "millilitrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per minute" ).build(), Optional.of("ml/min"), Optional.of("41"), Optional.of("cubicMetrePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per minute bar" ).build(), Optional.of("ml/(min\u00B7bar)"), Optional.of("G92"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B3 \u00D7 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per minute kelvin" ).build(), Optional.of("ml/(min\u00B7K)"), Optional.of("G75"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u207B\u2078 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per second" ).build(), Optional.of("ml/s"), Optional.of("40"), Optional.of("cubicMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per second bar" ).build(), Optional.of("ml/(s\u00B7bar)"), Optional.of("G93"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B9 kg\u207B\u00B9 \u00D7 m\u2074 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per second kelvin" ).build(), Optional.of("ml/(s\u00B7K)"), Optional.of("G76"), Optional.empty(), Optional.of("10\u207B\u2076 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "millilitrePerSquareCentimetreMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerSquareCentimetreMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per square centimetre minute" ).build(), Optional.of("(ml/min)/cm\u00B2"), Optional.of("M22"), Optional.of("cubicMetrePerSecondSquareMetre"), Optional.of("2.777778 \u00D7 10\u207B\u2076 (m\u00B3/s)/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "millilitrePerSquareCentimetreSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millilitrePerSquareCentimetreSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millilitre per square centimetre second" ).build(), Optional.of("ml/(cm\u00B2\u00B7s)"), Optional.of("35"), Optional.of("metrePerSecond"), Optional.of("10\u207B\u00B2 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.POROSITY)) ) );
UNITS_BY_NAME.put( "millimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre" ).build(), Optional.of("mm"), Optional.of("MMT"), Optional.of("metre"), Optional.of("10\u207B\u00B3 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DIAMETER, QuantityKinds.THICKNESS, QuantityKinds.LENGTH, QuantityKinds.RADIUS, QuantityKinds.HEIGHT, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.BREADTH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.DISTANCE)) ) );
UNITS_BY_NAME.put( "millimetrePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per bar" ).build(), Optional.of("mm/bar"), Optional.of("G06"), Optional.empty(), Optional.of("10\u207B\u2078 kg\u207B\u00B9 \u00D7 m\u00B2 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.HEIGHT, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.DIAMETER, QuantityKinds.LENGTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.BREADTH, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "millimetrePerDegreeCelsiusMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerDegreeCelsiusMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per degree Celsius metre" ).build(), Optional.of("mm/(\u00B0C\u00B7m)"), Optional.of("E97"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u00B3 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "millimetrePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per hour" ).build(), Optional.of("mm/h"), Optional.of("H67"), Optional.empty(), Optional.of("0.277777778 \u00D7 10\u207B\u2077 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetrePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per kelvin" ).build(), Optional.of("mm/K"), Optional.of("F53"), Optional.empty(), Optional.of("10\u207B\u00B3 m \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "millimetrePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per minute" ).build(), Optional.of("mm/min"), Optional.of("H81"), Optional.empty(), Optional.of("1.666666667 \u00D7 10\u207B\u2075 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per second" ).build(), Optional.of("mm/s"), Optional.of("C16"), Optional.of("metrePerSecond"), Optional.of("10\u207B\u00B3 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetrePerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per second squared" ).build(), Optional.of("mm/s\u00B2"), Optional.of("M41"), Optional.of("metrePerSecondSquared"), Optional.of("10\u207B\u00B3 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY, QuantityKinds.ACCELERATION)) ) );
UNITS_BY_NAME.put( "millimetrePerYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetrePerYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre per year" ).build(), Optional.of("mm/y"), Optional.of("H66"), Optional.empty(), Optional.of("3.15576 \u00D7 10\u2074 m \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "millimetreSquaredPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetreSquaredPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre squared per second" ).build(), Optional.of("mm\u00B2/s"), Optional.of("C17"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2076 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "millimetreToTheFourthPower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimetreToTheFourthPower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimetre to the fourth power" ).build(), Optional.of("mm\u2074"), Optional.of("G77"), Optional.of("metreToTheFourthPower"), Optional.of("10\u207B\u00B9\u00B2 m\u2074"), new HashSet<>(Arrays.asList(QuantityKinds.SECOND_MOMENT_OF_AREA, QuantityKinds.SECOND_AXIAL_MOMENT_OF_AREA)) ) );
UNITS_BY_NAME.put( "millimole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimole" ).build(), Optional.of("mmol"), Optional.of("C18"), Optional.of("mole"), Optional.of("10\u207B\u00B3 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "millimolePerGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimolePerGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimole per gram" ).build(), Optional.of("mmol/g"), Optional.of("H68"), Optional.empty(), Optional.of("mol \u00D7 kg\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "millimolePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimolePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimole per kilogram" ).build(), Optional.of("mmol/kg"), Optional.of("D87"), Optional.of("molePerKilogram"), Optional.of("10\u207B\u00B3 mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "millimolePerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millimolePerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millimole per litre" ).build(), Optional.of("mmol/l"), Optional.of("M33"), Optional.of("molePerCubicMetre"), Optional.of("mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "millinewton", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millinewton" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millinewton" ).build(), Optional.of("mN"), Optional.of("C20"), Optional.of("newton"), Optional.of("10\u207B\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "millinewtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millinewtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millinewton metre" ).build(), Optional.of("mN\u00B7m"), Optional.of("D83"), Optional.of("newtonMetre"), Optional.of("10\u207B\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "millinewtonPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millinewtonPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millinewton per metre" ).build(), Optional.of("mN/m"), Optional.of("C22"), Optional.of("newtonPerMetre"), Optional.of("10\u207B\u00B3 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "milliohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliohm" ).build(), Optional.of("m\u2126"), Optional.of("E45"), Optional.of("ohm"), Optional.of("10\u207B\u00B3 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.REACTANCE, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.IMPEDANCE)) ) );
UNITS_BY_NAME.put( "milliohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliohm metre" ).build(), Optional.of("m\u2126\u00B7m"), Optional.of("C23"), Optional.of("ohmMetre"), Optional.of("10\u207B\u00B3 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "milliohmPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliohmPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliohm per metre" ).build(), Optional.of("m\u2126/m"), Optional.of("F54"), Optional.of("ohmPerMetre"), Optional.of("10\u207B\u00B3 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "million", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "million" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "million" ).build(), Optional.empty(), Optional.of("MIO"), Optional.empty(), Optional.of("10\u2076"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionBtuItPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millionBtuItPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "million Btu(IT) per hour" ).build(), Optional.of("BtuIT/h"), Optional.of("E16"), Optional.of("watt"), Optional.of("293071.1 W"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionBtuPer1000CubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millionBtuPer1000CubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "million Btu per 1000 cubic foot" ).build(), Optional.of("MBTU/kft\u00B3"), Optional.of("M9"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millionCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "million cubic metre" ).build(), Optional.of("Mm\u00B3"), Optional.of("HMQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millionInternationalUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millionInternationalUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "million international unit" ).build(), Optional.empty(), Optional.of("MIU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "millipascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millipascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millipascal" ).build(), Optional.of("mPa"), Optional.of("74"), Optional.of("pascal"), Optional.of("10\u207B\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "millipascalPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millipascalPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millipascal per metre" ).build(), Optional.of("mPa/m"), Optional.of("P80"), Optional.empty(), Optional.of("10\u207B\u00B3 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "millipascalSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millipascalSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millipascal second" ).build(), Optional.of("mPa\u00B7s"), Optional.of("C24"), Optional.of("pascalSecond"), Optional.of("10\u207B\u00B3 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "millipascalSecondPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millipascalSecondPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millipascal second per bar" ).build(), Optional.of("mPa\u00B7s/bar"), Optional.of("L16"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2078 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "millipascalSecondPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millipascalSecondPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millipascal second per kelvin" ).build(), Optional.of("mPa\u00B7s/K"), Optional.of("L15"), Optional.empty(), Optional.of("10\u207B\u00B3 Pa \u00D7 s/K"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "milliradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliradian" ).build(), Optional.of("mrad"), Optional.of("C25"), Optional.of("radian"), Optional.of("10\u207B\u00B3 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "milliroentgen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliroentgen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliroentgen" ).build(), Optional.of("mR"), Optional.of("2Y"), Optional.of("coulombPerKilogram"), Optional.of("2.58 \u00D7 10\u207B\u2077 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "milliroentgenAequivalentMen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliroentgenAequivalentMen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliroentgen aequivalent men" ).build(), Optional.of("mrem"), Optional.of("L31"), Optional.of("sievert"), Optional.of("10\u207B\u2075 Sv"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "millisecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisecond" ).build(), Optional.of("ms"), Optional.of("C26"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B3 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "millisiemens", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisiemens" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisiemens" ).build(), Optional.of("mS"), Optional.of("C27"), Optional.of("siemens"), Optional.of("10\u207B\u00B3 S"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.ADMITTANCE)) ) );
UNITS_BY_NAME.put( "millisiemensPerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisiemensPerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisiemens per centimetre" ).build(), Optional.of("mS/cm"), Optional.of("H61"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u00B9 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "millisievert", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisievert" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisievert" ).build(), Optional.of("mSv"), Optional.of("C28"), Optional.of("sievert"), Optional.of("10\u207B\u00B3 Sv"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "millisievertPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisievertPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisievert per hour" ).build(), Optional.of("mSv/h"), Optional.of("P71"), Optional.of("sievertPerSecond"), Optional.of("0.277777778 \u00D7 10\u207B\u2077 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "millisievertPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisievertPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisievert per minute" ).build(), Optional.of("mSv/min"), Optional.of("P75"), Optional.of("sievertPerSecond"), Optional.of("1.666666667 \u00D7 10\u207B\u2075 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "millisievertPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millisievertPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millisievert per second" ).build(), Optional.of("mSv/s"), Optional.of("P66"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u00B3 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "millitesla", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millitesla" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millitesla" ).build(), Optional.of("mT"), Optional.of("C29"), Optional.of("tesla"), Optional.of("10\u207B\u00B3 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "millivolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millivolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millivolt" ).build(), Optional.of("mV"), Optional.of("2Z"), Optional.of("voltDc"), Optional.of("10\u207B\u00B3 V"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.TENSION, QuantityKinds.VOLTAGE, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.ELECTRIC_POTENTIAL)) ) );
UNITS_BY_NAME.put( "millivoltAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millivoltAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millivolt - ampere" ).build(), Optional.of("mV\u00B7A"), Optional.of("M35"), Optional.of("voltAmpere"), Optional.of("10\u207B\u00B3 V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "millivoltPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millivoltPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millivolt per kelvin" ).build(), Optional.of("mV/K"), Optional.of("D49"), Optional.of("voltPerKelvin"), Optional.of("10\u207B\u00B3 V/K"), new HashSet<>(Arrays.asList(QuantityKinds.THOMPSON_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "millivoltPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millivoltPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millivolt per metre" ).build(), Optional.of("mV/m"), Optional.of("C30"), Optional.of("voltPerMetre"), Optional.of("10\u207B\u00B3 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "millivoltPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "millivoltPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "millivolt per minute" ).build(), Optional.of("mV/min"), Optional.of("H62"), Optional.empty(), Optional.of("1.666666667 \u00D7 10\u207B\u2075 m\u00B2 \u00D7 kg \u00D7 s\u207B\u2074 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "milliwatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliwatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliwatt" ).build(), Optional.of("mW"), Optional.of("C31"), Optional.of("watt"), Optional.of("10\u207B\u00B3 W"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVE_POWER, QuantityKinds.POWER, QuantityKinds.POWER_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "milliwattPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliwattPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliwatt per square metre" ).build(), Optional.of("mW/m\u00B2"), Optional.of("C32"), Optional.of("wattPerSquareMetre"), Optional.of("10\u207B\u00B3 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY)) ) );
UNITS_BY_NAME.put( "milliweber", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "milliweber" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "milliweber" ).build(), Optional.of("mWb"), Optional.of("C33"), Optional.of("weber"), Optional.of("10\u207B\u00B3 Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX)) ) );
UNITS_BY_NAME.put( "minuteUnitOfAngle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "minuteUnitOfAngle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "minute [unit of angle]" ).build(), Optional.of("'"), Optional.of("D61"), Optional.of("radian"), Optional.of("2.908882 \u00D7 10\u207B\u2074 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "minuteUnitOfTime", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "minuteUnitOfTime" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "minute [unit of time]" ).build(), Optional.of("min"), Optional.of("MIN"), Optional.of("secondUnitOfTime"), Optional.of("60 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "mmscfPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mmscfPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "MMSCF/day" ).build(), Optional.empty(), Optional.of("5E"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "moduleWidth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "moduleWidth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "module width" ).build(), Optional.of("MW"), Optional.of("H77"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "molPerCubicMetrePascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molPerCubicMetrePascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mol per cubic metre pascal" ).build(), Optional.of("(mol/m\u00B3)/Pa"), Optional.of("P52"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "molPerKilogramPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molPerKilogramPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mol per kilogram pascal" ).build(), Optional.of("(mol/kg)/Pa"), Optional.of("P51"), Optional.empty(), Optional.of("m \u00D7 kg\u207B\u00B2 \u00D7 s\u00B2 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "mole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole" ).build(), Optional.of("mol"), Optional.of("C34"), Optional.empty(), Optional.of("mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "molePerCubicDecimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerCubicDecimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per cubic decimetre" ).build(), Optional.of("mol/dm\u00B3"), Optional.of("C35"), Optional.of("molePerCubicMetre"), Optional.of("10\u00B3 mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "molePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per cubic metre" ).build(), Optional.of("mol/m\u00B3"), Optional.of("C36"), Optional.empty(), Optional.of("mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "molePerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per cubic metre bar" ).build(), Optional.of("(mol/m\u00B3)/bar"), Optional.of("L29"), Optional.of("molPerCubicMetrePascal"), Optional.of("10\u207B\u2075 (mol/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "molePerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per cubic metre kelvin" ).build(), Optional.of("(mol/m\u00B3)/K"), Optional.of("L28"), Optional.empty(), Optional.of("(mol/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "molePerCubicMetreToThePowerSumOfStoichiometricNumbers", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerCubicMetreToThePowerSumOfStoichiometricNumbers" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per cubic metre to the power sum of stoichiometric numbers" ).build(), Optional.of("(mol/m\u00B3)\u2211\u03BDB"), Optional.of("P99"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "molePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per hour" ).build(), Optional.of("mol/h"), Optional.of("L23"), Optional.of("molePerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
UNITS_BY_NAME.put( "molePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per kilogram" ).build(), Optional.of("mol/kg"), Optional.of("C19"), Optional.empty(), Optional.of("mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MOLALITY_OF_SOLUTE_B)) ) );
UNITS_BY_NAME.put( "molePerKilogramBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerKilogramBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per kilogram bar" ).build(), Optional.of("(mol/kg)/bar"), Optional.of("L25"), Optional.of("molPerKilogramPascal"), Optional.of("10\u207B\u2075 (mol/kg)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerKilogramKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerKilogramKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per kilogram kelvin" ).build(), Optional.of("(mol/kg)/K"), Optional.of("L24"), Optional.empty(), Optional.of("(mol/kg)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "molePerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per litre" ).build(), Optional.of("mol/l"), Optional.of("C38"), Optional.of("molePerCubicMetre"), Optional.of("10\u00B3 mol/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.MASS_DENSITY, QuantityKinds.CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "molePerLitreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerLitreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per litre bar" ).build(), Optional.of("(mol/l)/bar"), Optional.of("L27"), Optional.of("molPerCubicMetrePascal"), Optional.of("10\u207B\u00B2 (mol/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "molePerLitreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerLitreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per litre kelvin" ).build(), Optional.of("(mol/l)/K"), Optional.of("L26"), Optional.of("molePerCubicMetreKelvin"), Optional.of("10\u00B3 (mol/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.DENSITY, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "molePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per minute" ).build(), Optional.of("mol/min"), Optional.of("L30"), Optional.of("molePerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B2 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_FLUX)) ) );
}

private static void init11() {
UNITS_BY_NAME.put( "molePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "molePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mole per second" ).build(), Optional.of("mol/s"), Optional.of("E95"), Optional.empty(), Optional.of("s\u207B\u00B9 \u00D7 mol"), new HashSet<>(Arrays.asList(QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "monetaryValue", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "monetaryValue" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "monetary value" ).build(), Optional.empty(), Optional.of("M4"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "month", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "month" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "month" ).build(), Optional.of("mo"), Optional.of("MON"), Optional.of("secondUnitOfTime"), Optional.of("2.629800 \u00D7 10\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "mutuallyDefined", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "mutuallyDefined" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "mutually defined" ).build(), Optional.empty(), Optional.of("ZZ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "nanoampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanoampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanoampere" ).build(), Optional.of("nA"), Optional.of("C39"), Optional.of("ampere"), Optional.of("10\u207B\u2079 A"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "nanocoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanocoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanocoulomb" ).build(), Optional.of("nC"), Optional.of("C40"), Optional.of("coulomb"), Optional.of("10\u207B\u2079 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "nanofarad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanofarad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanofarad" ).build(), Optional.of("nF"), Optional.of("C41"), Optional.of("farad"), Optional.of("10\u207B\u2079 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "nanofaradPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanofaradPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanofarad per metre" ).build(), Optional.of("nF/m"), Optional.of("C42"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u2079 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.ELECTRIC_CONSTANT, QuantityKinds.PERMITTIVITY)) ) );
UNITS_BY_NAME.put( "nanogramPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanogramPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanogram per kilogram" ).build(), Optional.of("ng/kg"), Optional.of("L32"), Optional.empty(), Optional.of("10\u207B\u00B9\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "nanograyPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanograyPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanogray per hour" ).build(), Optional.of("nGy/h"), Optional.of("P64"), Optional.of("grayPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9\u00B3 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "nanograyPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanograyPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanogray per minute" ).build(), Optional.of("nGy/min"), Optional.of("P60"), Optional.of("grayPerSecond"), Optional.of("1.66667 \u00D7 10\u207B\u00B9\u00B9 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "nanograyPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanograyPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanogray per second" ).build(), Optional.of("nGy/s"), Optional.of("P56"), Optional.of("grayPerSecond"), Optional.of("10\u207B\u2079 Gy/s"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE_RATE)) ) );
UNITS_BY_NAME.put( "nanohenry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanohenry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanohenry" ).build(), Optional.of("nH"), Optional.of("C43"), Optional.of("henry"), Optional.of("10\u207B\u2079 H"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE, QuantityKinds.MUTUAL_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "nanohenryPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanohenryPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanohenry per metre" ).build(), Optional.of("nH/m"), Optional.of("C44"), Optional.of("henryPerMetre"), Optional.of("10\u207B\u2079 H/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMEABILITY, QuantityKinds.MAGNETIC_CONSTANT, QuantityKinds.PERMEABILITY_OF_VACUUM)) ) );
UNITS_BY_NAME.put( "nanometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanometre" ).build(), Optional.of("nm"), Optional.of("C45"), Optional.of("metre"), Optional.of("10\u207B\u2079 m"), new HashSet<>(Arrays.asList(QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.HEIGHT)) ) );
UNITS_BY_NAME.put( "nanoohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanoohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanoohm" ).build(), Optional.of("n\u2126"), Optional.of("P22"), Optional.of("ohm"), Optional.of("10\u207B\u2079 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.REACTANCE, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "nanoohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanoohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanoohm metre" ).build(), Optional.of("n\u2126\u00B7m"), Optional.of("C46"), Optional.empty(), Optional.of("10\u207B\u2079 \u2126\u00B7\u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "nanosecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanosecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanosecond" ).build(), Optional.of("ns"), Optional.of("C47"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2079 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "nanosiemensPerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanosiemensPerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanosiemens per centimetre" ).build(), Optional.of("nS/cm"), Optional.of("G44"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2077 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "nanosiemensPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanosiemensPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanosiemens per metre" ).build(), Optional.of("nS/m"), Optional.of("G45"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u2079 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "nanosievertPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanosievertPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanosievert per hour" ).build(), Optional.of("nSv/h"), Optional.of("P73"), Optional.of("sievertPerSecond"), Optional.of("0.277777778 \u00D7 10\u207B\u00B9\u00B3 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "nanosievertPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanosievertPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanosievert per minute" ).build(), Optional.of("nSv/min"), Optional.of("P77"), Optional.of("sievertPerSecond"), Optional.of("1.666666667 \u00D7 10\u207B\u00B9\u00B9 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "nanosievertPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanosievertPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanosievert per second" ).build(), Optional.of("nSv/s"), Optional.of("P68"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u2079 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "nanotesla", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanotesla" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanotesla" ).build(), Optional.of("nT"), Optional.of("C48"), Optional.of("tesla"), Optional.of("10\u207B\u2079 T"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_INDUCTION, QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION)) ) );
UNITS_BY_NAME.put( "nanowatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nanowatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nanowatt" ).build(), Optional.of("nW"), Optional.of("C49"), Optional.of("watt"), Optional.of("10\u207B\u2079 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "naturalUnitOfInformation", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "naturalUnitOfInformation" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "natural unit of information" ).build(), Optional.of("nat"), Optional.of("Q16"), Optional.empty(), Optional.of("nat"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "naturalUnitOfInformationPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "naturalUnitOfInformationPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "natural unit of information per second" ).build(), Optional.of("nat/s"), Optional.of("Q19"), Optional.empty(), Optional.of("nat/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "nauticalMile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nauticalMile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nautical mile" ).build(), Optional.of("n mile"), Optional.of("NMI"), Optional.of("metre"), Optional.of("1852 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH_OF_PATH, QuantityKinds.BREADTH, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.RADIUS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.HEIGHT, QuantityKinds.LENGTH)) ) );
UNITS_BY_NAME.put( "neper", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "neper" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "neper" ).build(), Optional.of("Np"), Optional.of("C50"), Optional.empty(), Optional.of("Np"), new HashSet<>(Arrays.asList(QuantityKinds.LOGARITHMIC_DECREMENT, QuantityKinds.LEVEL_OF_A_POWER_QUANTITY, QuantityKinds.LEVEL_OF_A_FIELD_QUANTITY)) ) );
UNITS_BY_NAME.put( "neperPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "neperPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "neper per second" ).build(), Optional.of("Np/s"), Optional.of("C51"), Optional.empty(), Optional.of("Np/s"), new HashSet<>(Arrays.asList(QuantityKinds.DAMPING_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "netKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "netKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "net kilogram" ).build(), Optional.empty(), Optional.of("58"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "netTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "netTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "net ton" ).build(), Optional.empty(), Optional.of("NT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "newton", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newton" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton" ).build(), Optional.of("N"), Optional.of("NEW"), Optional.empty(), Optional.of("(kg \u00D7 m)/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "newtonCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton centimetre" ).build(), Optional.of("N\u00B7cm"), Optional.of("F88"), Optional.empty(), Optional.of("10\u207B\u00B2 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre" ).build(), Optional.of("N\u00B7m"), Optional.of("NU"), Optional.empty(), Optional.of("N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetrePerAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre per ampere" ).build(), Optional.of("N\u00B7m/A"), Optional.of("F90"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerDegree", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetrePerDegree" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre per degree" ).build(), Optional.of("Nm/\u00B0"), Optional.of("F89"), Optional.empty(), Optional.of("57.295788 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 rad\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre per kilogram" ).build(), Optional.of("N\u00B7m/kg"), Optional.of("G19"), Optional.empty(), Optional.of("m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetrePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre per metre" ).build(), Optional.of("N\u00B7m/m\u00B2"), Optional.of("Q27"), Optional.empty(), Optional.of("m \u00D7 kg \u00D7 s\u207B\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "newtonMetrePerRadian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetrePerRadian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre per radian" ).build(), Optional.of("N\u00B7m/rad"), Optional.of("M93"), Optional.empty(), Optional.of("m\u00B2 \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 rad\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonMetrePerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetrePerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre per square metre" ).build(), Optional.of("N\u00B7m/m\u00B2"), Optional.of("M34"), Optional.empty(), Optional.of("N \u00D7 m/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORSIONAL_STIFFNESS, QuantityKinds.AREA_RELATED_TORSIONAL_MOMENT)) ) );
UNITS_BY_NAME.put( "newtonMetreSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetreSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre second" ).build(), Optional.of("N\u00B7m\u00B7s"), Optional.of("C53"), Optional.empty(), Optional.of("N \u00D7 m \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_IMPULSE)) ) );
UNITS_BY_NAME.put( "newtonMetreSquaredPerKilogramSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetreSquaredPerKilogramSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre squared per kilogram squared" ).build(), Optional.of("N\u00B7m\u00B2/kg\u00B2"), Optional.of("C54"), Optional.empty(), Optional.of("N \u00D7 m\u00B2/kg\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.GRAVITATIONAL_CONSTANT)) ) );
UNITS_BY_NAME.put( "newtonMetreWattToThePowerMinus0point5", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonMetreWattToThePowerMinus0point5" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton metre watt to the power minus 0,5" ).build(), Optional.of("N\u00B7m\u00B7W\u207B\u2070\u2027\u2075"), Optional.of("H41"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 W\u207B\u2070\u2027\u2075"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonPerAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per ampere" ).build(), Optional.of("N/A"), Optional.of("H40"), Optional.empty(), Optional.of("kg \u00D7 m \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonPerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per centimetre" ).build(), Optional.of("N/cm"), Optional.of("M23"), Optional.of("newtonPerMetre"), Optional.of("10\u00B2 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "newtonPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per metre" ).build(), Optional.of("N/m"), Optional.of("4P"), Optional.empty(), Optional.of("N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "newtonPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per millimetre" ).build(), Optional.of("N/mm"), Optional.of("F47"), Optional.empty(), Optional.of("10\u00B3 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "newtonPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per square centimetre" ).build(), Optional.of("N/cm\u00B2"), Optional.of("E01"), Optional.of("pascal"), Optional.of("10\u2074 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "newtonPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per square metre" ).build(), Optional.of("N/m\u00B2"), Optional.of("C55"), Optional.of("pascal"), Optional.of("Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "newtonPerSquareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonPerSquareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton per square millimetre" ).build(), Optional.of("N/mm\u00B2"), Optional.of("C56"), Optional.of("pascal"), Optional.of("10\u2076 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "newtonSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton second" ).build(), Optional.of("N\u00B7s"), Optional.of("C57"), Optional.empty(), Optional.of("N \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.IMPULSE)) ) );
UNITS_BY_NAME.put( "newtonSecondPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonSecondPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton second per metre" ).build(), Optional.of("N\u00B7s/m"), Optional.of("C58"), Optional.empty(), Optional.of("N \u00D7 s/m"), new HashSet<>(Arrays.asList(QuantityKinds.MECHANICAL_IMPEDANCE)) ) );
UNITS_BY_NAME.put( "newtonSecondPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonSecondPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton second per square metre" ).build(), Optional.of("(N/m\u00B2)\u00B7s"), Optional.of("N36"), Optional.of("pascalSecond"), Optional.of("Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "newtonSquareMetrePerAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "newtonSquareMetrePerAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "newton square metre per ampere" ).build(), Optional.of("N\u00B7m\u00B2/A"), Optional.of("P49"), Optional.empty(), Optional.of("m\u00B3 \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_DIPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "nil", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "nil" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "nil" ).build(), Optional.of("()"), Optional.of("NIL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfArticles", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfArticles" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of articles" ).build(), Optional.empty(), Optional.of("NAR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfCells", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfCells" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of cells" ).build(), Optional.empty(), Optional.of("NCL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfInternationalUnits", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfInternationalUnits" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of international units" ).build(), Optional.empty(), Optional.of("NIU"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfJewels", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfJewels" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of jewels" ).build(), Optional.empty(), Optional.of("JWL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfPacks", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfPacks" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of packs" ).build(), Optional.empty(), Optional.of("NMP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfParts", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfParts" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of parts" ).build(), Optional.empty(), Optional.of("NPT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "numberOfWords", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "numberOfWords" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "number of words" ).build(), Optional.empty(), Optional.of("D68"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "octave", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "octave" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "octave" ).build(), Optional.empty(), Optional.of("C59"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY_INTERVAL)) ) );
UNITS_BY_NAME.put( "octet", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "octet" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "octet" ).build(), Optional.of("o"), Optional.of("Q12"), Optional.of("bit"), Optional.of("8 bit"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "octetPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "octetPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "octet per second" ).build(), Optional.of("o/s"), Optional.of("Q13"), Optional.of("bitPerSecond"), Optional.of("8 bit/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm" ).build(), Optional.of("\u2126"), Optional.of("OHM"), Optional.empty(), Optional.of("\u2126"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_IMPEDANCES, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.IMPEDANCE, QuantityKinds.REACTANCE)) ) );
UNITS_BY_NAME.put( "ohmCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm centimetre" ).build(), Optional.of("\u2126\u00B7cm"), Optional.of("C60"), Optional.of("ohmMetre"), Optional.of("10\u207B\u00B2 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmCircularMilPerFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmCircularMilPerFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm circular-mil per foot" ).build(), Optional.of("\u2126\u00B7cmil/ft"), Optional.of("P23"), Optional.of("ohmMetre"), Optional.of("1.662426 \u00D7 10\u207B\u2079 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm kilometre" ).build(), Optional.of("\u2126\u00B7km"), Optional.of("M24"), Optional.of("ohmMetre"), Optional.of("10\u00B3 \u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm metre" ).build(), Optional.of("\u2126\u00B7m"), Optional.of("C61"), Optional.empty(), Optional.of("\u2126 \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTIVITY, QuantityKinds.RESIDUAL_RESISTIVITY)) ) );
UNITS_BY_NAME.put( "ohmPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm per kilometre" ).build(), Optional.of("\u2126/km"), Optional.of("F56"), Optional.of("ohmPerMetre"), Optional.of("10\u207B\u00B3 \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "ohmPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm per metre" ).build(), Optional.of("\u2126/m"), Optional.of("H26"), Optional.empty(), Optional.of("\u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "ohmPerMileStatuteMile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ohmPerMileStatuteMile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ohm per mile (statute mile)" ).build(), Optional.of("\u2126/mi"), Optional.of("F55"), Optional.of("ohmPerMetre"), Optional.of("6.21371 \u00D7 10\u207B\u2074  \u2126/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEIC_RESISTANCE)) ) );
UNITS_BY_NAME.put( "one", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "one" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "one" ).build(), Optional.of("1"), Optional.of("C62"), Optional.empty(), Optional.of("1"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_REYNOLDS_NUMBER, QuantityKinds.DISSIPANCE, QuantityKinds.FOURIER_NUMBER, QuantityKinds.SHORT_RANGE_ORDER_PARAMETER, QuantityKinds.RELATIVE_DENSITY, QuantityKinds.NEUTRON_YIELD_PER_FISSION, QuantityKinds.PACKING_FRACTION, QuantityKinds.GRAND_PARTITION_FUNCTION, QuantityKinds.MACH_NUMBER, QuantityKinds.REFLECTANCE, QuantityKinds.PRINCIPLE_QUANTUM_NUMBER, QuantityKinds.NUMBER_OF_TURNS_IN_A_WINDING, QuantityKinds.COUPLING_COEFFICIENT, QuantityKinds.HARTMANN_NUMBER, QuantityKinds.LETHARGY, QuantityKinds.NUCLEAR_SPIN_QUANTUM_NUMBER, QuantityKinds.SPECTRAL_ABSORPTANCE, QuantityKinds.SPECTRAL_ABSORPTION_FACTOR, QuantityKinds.FROUDE_NUMBER, QuantityKinds.SCHMIDT_NUMBER, QuantityKinds.RATIO_OF_THE_MASSIC_HEAT_CAPACITY, QuantityKinds.RELATIVE_ELONGATION, QuantityKinds.INFINITE_MEDIUM_MULTIPLICATION_FACTOR, QuantityKinds.PECLET_NUMBER, QuantityKinds.REFRACTIVE_INDEX, QuantityKinds.COORDINATES_TRICHROMATIC, QuantityKinds.NEUTRON_YIELD_PER_ABSORPTION, QuantityKinds.SPIN_ANGULAR_MOMENTUM_QUANTUM_NUMBER, QuantityKinds.FRICTION_FACTOR, QuantityKinds.PROTON_NUMBER, QuantityKinds.RATIO_OF_THE_SPECIFIC_HEAT_CAPACITIES, QuantityKinds.CURRENT_FRACTION_OF_ION_B, QuantityKinds.MAGNETIC_SUSCEPTIBILITY, QuantityKinds.EULER_NUMBER, QuantityKinds.PARTITION_FUNCTION_OF_A_MOLECULE, QuantityKinds.DEBYE_WALLE_FACTOR, QuantityKinds.EMISSIVITY, QuantityKinds.MULTIPLICATION_FACTOR, QuantityKinds.ORBITAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER, QuantityKinds.KNUDSEN_NUMBER, QuantityKinds.ISENTROPIC_EXPONENT, QuantityKinds.SPECTRAL_EMISSIVITY, QuantityKinds.ACTIVITY_OF_SOLVENT_A, QuantityKinds.THERMAL_UTILIZATION_FACTOR, QuantityKinds.ELECTRIC_SUSCEPTIBILITY, QuantityKinds.REYNOLDS_NUMBER, QuantityKinds.RELATIVE_PERMEABILITY, QuantityKinds.REST_MASS_OF_ELECTRON, QuantityKinds.RESONANCE_ESCAPE_PROBABILITY, QuantityKinds.REACTIVITY, QuantityKinds.RELATIVE_MASS_EXCESS, QuantityKinds.TOTAL_ANGULAR_MOMENTUM_QUANTUM_NUMBER, QuantityKinds.CANONICAL_PARTITION_FUNCTION, QuantityKinds.MADELUNG_CONSTANT, QuantityKinds.AVERAGE_LOGARITHMIC_ENERGY_DECREMENT, QuantityKinds.DIRECTIONAL_SPECTRAL_EMISSIVITY, QuantityKinds.POISSON_RATIO, QuantityKinds.DISSIPATION_FACTOR, QuantityKinds.FOURIER_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.ATOMIC_NUMBER, QuantityKinds.POISSON_NUMBER, QuantityKinds.RELATIVE_MASS_DENSITY, QuantityKinds.RELATIVE_ATOMIC_MASS, QuantityKinds.NUMBER_OF_PAIRS_OF_POLES, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.FINE_STRUCTURE_CONSTANT, QuantityKinds.STOICHIOMETRIC_NUMBER_OF_B, QuantityKinds.CHARGE_NUMBER_OF_ION, QuantityKinds.MAGNETIC_QUANTUM_NUMBER, QuantityKinds.NUMBER_OF_MOLECULES_OR_OTHER_ELEMENTARY_ENTITIES, QuantityKinds.TRANSMITTANCE, QuantityKinds.MICROCANONICAL_PARTITION_FUNCTION, QuantityKinds.MOBILITY_RATIO, QuantityKinds.COWLING_NUMBER, QuantityKinds.GRUENEISEN_PARAMETER, QuantityKinds.PRANDTL_NUMBER, QuantityKinds.ABSORPTION_FACTOR, QuantityKinds.MOLECULAR_PARTITION_FUNCTION, QuantityKinds.MOLE_FRACTION_OF_B, QuantityKinds.ORDER_OF_REFLEXION, QuantityKinds.LEWIS_NUMBER, QuantityKinds.RAYLEIGH_NUMBER, QuantityKinds.SPECTRAL_TRANSMISSION_FACTOR, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_GASEOUS_MIXTURE, QuantityKinds.SHEAR_STRAIN, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.WEBER_NUMBER, QuantityKinds.TRANSPORT_NUMBER_OF_ION_B, QuantityKinds.PECLET_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.G_FACTOR_OF_NUCLEUS, QuantityKinds.THERMAL_DIFFUSION_RATIO, QuantityKinds.OPTICAL_DENSITY, QuantityKinds.NON_LEAKAGE_PROBABILITY, QuantityKinds.ACTIVITY_COEFFICIENT_OF_SOLUTE_B_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.HYPERFINE_STRUCTURE_QUANTUM_NUMBER, QuantityKinds.EFFECTIVE_MULTIPLICATION_FACTOR, QuantityKinds.MOLE_RATIO_OF_SOLUTE_B, QuantityKinds.NUSSELT_NUMBER, QuantityKinds.STANTON_NUMBER, QuantityKinds.BINDING_FRACTION, QuantityKinds.GRASHOF_NUMBER, QuantityKinds.ABSOLUTE_ACTIVITY, QuantityKinds.INTERNAL_CONVERSION_FACTOR, QuantityKinds.RELATIVE_ACTIVITY_OF_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.REST_MASS_OF_NEUTRON, QuantityKinds.RELATIVE_PERMITTIVITY, QuantityKinds.CIE_COLORIMETRIC_FUNCTIONS, QuantityKinds.STATISTICAL_WEIGHT, QuantityKinds.REST_MASS_OF_PROTON, QuantityKinds.NEUTRON_NUMBER, QuantityKinds.SPECTRAL_REFLECTIONFACTOR, QuantityKinds.COEFFICIENT_OF_FRICTION, QuantityKinds.STANTON_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.SPECTRAL_REFLECTANCE, QuantityKinds.ALFVEN_NUMBER, QuantityKinds.MASS_NUMBER, QuantityKinds.LEAKAGE_COEFFICIENT, QuantityKinds.STROUHAL_NUMBER, QuantityKinds.FAST_FISSION_FACTOR, QuantityKinds.LUMINOUS_EFFICIENCY_AT_A_SPECIFIED_WAVELENGTH, QuantityKinds.LUMINOUS_EFFICIENCY, QuantityKinds.LINEAR_STRAIN, QuantityKinds.ABSORBANCE, QuantityKinds.REFLECTION_FACTOR, QuantityKinds.NUCLEON_NUMBER, QuantityKinds.SPECTRAL_TRANSMITTANCE, QuantityKinds.DEGREE_OF_DISSOCIATION, QuantityKinds.STANDARD_ABSOLUTE_ACTIVITY_OF_B_IN_A_LIQUID_OR_A_SOLID_MIXTURE, QuantityKinds.OSMOTIC_COEFFICIENT_OF_THE_SOLVENT_A_ESPECIALLY_IN_A_DILUTE_SOLUTION, QuantityKinds.STANDARD_EQUILIBRIUM_CONSTANT, QuantityKinds.VOLUME_OR_BULK_STRAIN, QuantityKinds.EMISSIVITY_AT_A_SPECIFIED_WAVELENGTH, QuantityKinds.NUMBER_OF_PHASES, QuantityKinds.SPECTRAL_RADIANCE_FACTOR, QuantityKinds.NUSSELT_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.RELATIVE_MASS_DEFECT, QuantityKinds.SPECTRAL_LUMINOUS_EFFICIENCY, QuantityKinds.GRASHOF_NUMBER_FOR_MASS_TRANSFER, QuantityKinds.ACTIVITY_COEFFICIENT_OF_B_IN_A_LIQUID_AS_A_SOLID_MIXTURE, QuantityKinds.GRAND_CANONICAL_PARTITION_FUNCTION, QuantityKinds.THERMAL_DIFFUSION_FACTOR, QuantityKinds.LANDAU_GINZBURG_NUMBER, QuantityKinds.G_FACTOR_OF_ATOM_OR_ELECTRON, QuantityKinds.RELATIVE_MOLECULAR_MASS, QuantityKinds.LONG_RANGE_ORDER_PARAMETER, QuantityKinds.TRANSMISSION_FACTOR)) ) );
UNITS_BY_NAME.put( "onePerOne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "onePerOne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "one per one" ).build(), Optional.of("1/1"), Optional.of("Q26"), Optional.empty(), Optional.of("1/1"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ounceAvoirdupois", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupois" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois)" ).build(), Optional.of("oz"), Optional.of("ONZ"), Optional.of("kilogram"), Optional.of("2.834952 \u00D7 10\u207B\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisForce", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisForce" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois)-force" ).build(), Optional.of("ozf"), Optional.of("L40"), Optional.of("newton"), Optional.of("0.2780139 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisForceInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisForceInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois)-force inch" ).build(), Optional.of("ozf\u00B7in"), Optional.of("L41"), Optional.of("newtonMetre"), Optional.of("7.061552 \u00D7 10\u207B\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerCubicInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerCubicInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per cubic inch" ).build(), Optional.of("oz/in\u00B3"), Optional.of("L39"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.729994 \u00D7 10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerCubicYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerCubicYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per cubic yard" ).build(), Optional.of("oz/yd\u00B3"), Optional.of("G32"), Optional.empty(), Optional.of("3.70798 \u00D7 10\u207B\u00B2 kg \u00D7 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per day" ).build(), Optional.of("oz/d"), Optional.of("L33"), Optional.of("kilogramPerSecond"), Optional.of("3.281194 \u00D7 10\u207B\u2077kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerGallonUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerGallonUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per gallon (UK)" ).build(), Optional.of("oz/gal (UK)"), Optional.of("L37"), Optional.of("kilogramPerCubicMetre"), Optional.of("6.236023 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerGallonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerGallonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per gallon (US)" ).build(), Optional.of("oz/gal (US)"), Optional.of("L38"), Optional.of("kilogramPerCubicMetre"), Optional.of("7.489152 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per hour" ).build(), Optional.of("oz/h"), Optional.of("L34"), Optional.of("kilogramPerSecond"), Optional.of("7.874867 \u00D7 10\u207B\u2076 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per minute" ).build(), Optional.of("oz/min"), Optional.of("L35"), Optional.of("kilogramPerSecond"), Optional.of("4.72492 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per second" ).build(), Optional.of("oz/s"), Optional.of("L36"), Optional.of("kilogramPerSecond"), Optional.of("2.834952 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceAvoirdupoisPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceAvoirdupoisPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (avoirdupois) per square inch" ).build(), Optional.of("oz/in\u00B2"), Optional.of("N22"), Optional.of("kilogramPerSquareMetre"), Optional.of("4.394185 \u00D7 10 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "ounceFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce foot" ).build(), Optional.of("oz\u00B7ft"), Optional.of("4R"), Optional.of("kilogramMetre"), Optional.of("8.640934 \u00D7 10\u207B\u00B3 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "ounceInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce inch" ).build(), Optional.of("oz\u00B7in"), Optional.of("4Q"), Optional.of("kilogramMetre"), Optional.of("7.200778 \u00D7 10\u207B\u2074 kg \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "ouncePerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ouncePerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce per square foot" ).build(), Optional.of("oz/ft\u00B2"), Optional.of("37"), Optional.of("kilogramPerSquareMetre"), Optional.of("0.3051517 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "ouncePerSquareFootPer0point01inch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ouncePerSquareFootPer0point01inch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce per square foot per 0,01inch" ).build(), Optional.of("oz/(ft\u00B2/cin)"), Optional.of("38"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ouncePerSquareYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ouncePerSquareYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce per square yard" ).build(), Optional.of("oz/yd\u00B2"), Optional.of("ON"), Optional.of("kilogramPerSquareMetre"), Optional.of("3.390575 \u00D7 10\u207B\u00B2 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_DENSITY, QuantityKinds.AREIC_MASS)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUkFluidPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (UK fluid) per day" ).build(), Optional.of("fl oz (UK)/d"), Optional.of("J95"), Optional.of("cubicMetrePerSecond"), Optional.of("3.288549 \u00D7 10\u207B\u00B9\u2070 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUkFluidPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (UK fluid) per hour" ).build(), Optional.of("fl oz (UK)/h"), Optional.of("J96"), Optional.of("cubicMetrePerSecond"), Optional.of("7.892517 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUkFluidPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (UK fluid) per minute" ).build(), Optional.of("fl oz (UK)/min"), Optional.of("J97"), Optional.of("cubicMetrePerSecond"), Optional.of("4.73551 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUkFluidPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUkFluidPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (UK fluid) per second" ).build(), Optional.of("fl oz (UK)/s"), Optional.of("J98"), Optional.of("cubicMetrePerSecond"), Optional.of("2.841306 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUsFluidPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUsFluidPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (US fluid) per day" ).build(), Optional.of("fl oz (US)/d"), Optional.of("J99"), Optional.of("cubicMetrePerSecond"), Optional.of("3.422862 \u00D7 10\u207B\u00B9\u2070 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUsFluidPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUsFluidPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (US fluid) per hour" ).build(), Optional.of("fl oz (US)/h"), Optional.of("K10"), Optional.of("cubicMetrePerSecond"), Optional.of("8.214869 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "ounceUsFluidPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUsFluidPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (US fluid) per minute" ).build(), Optional.of("fl oz (US)/min"), Optional.of("K11"), Optional.of("cubicMetrePerSecond"), Optional.of("4.928922 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
}

private static void init12() {
UNITS_BY_NAME.put( "ounceUsFluidPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ounceUsFluidPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ounce (US fluid) per second" ).build(), Optional.of("fl oz (US)/s"), Optional.of("K12"), Optional.of("cubicMetrePerSecond"), Optional.of("2.957353 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "outfit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "outfit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "outfit" ).build(), Optional.empty(), Optional.of("11"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "overtimeHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "overtimeHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "overtime hour" ).build(), Optional.empty(), Optional.of("OT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ozoneDepletionEquivalent", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ozoneDepletionEquivalent" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ozone depletion equivalent" ).build(), Optional.empty(), Optional.of("ODE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pad" ).build(), Optional.empty(), Optional.of("PD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "page", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "page" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "page" ).build(), Optional.empty(), Optional.of("ZP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pageFacsimile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pageFacsimile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "page - facsimile" ).build(), Optional.empty(), Optional.of("QA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pageHardcopy", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pageHardcopy" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "page - hardcopy" ).build(), Optional.empty(), Optional.of("QB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pagePerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pagePerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "page per inch" ).build(), Optional.of("ppi"), Optional.of("PQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pair", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pair" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pair" ).build(), Optional.empty(), Optional.of("PR"), Optional.empty(), Optional.of("2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "panel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "panel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "panel" ).build(), Optional.empty(), Optional.of("OA"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "parsec", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "parsec" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "parsec" ).build(), Optional.of("pc"), Optional.of("C63"), Optional.of("metre"), Optional.of("3.085678 \u00D7 10\u00B9\u2076 m"), new HashSet<>(Arrays.asList(QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.THICKNESS, QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.HEIGHT, QuantityKinds.RADIUS)) ) );
UNITS_BY_NAME.put( "partPerBillionUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "partPerBillionUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "part per billion (US)" ).build(), Optional.of("ppb"), Optional.of("61"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u2079"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerHundredThousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "partPerHundredThousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "part per hundred thousand" ).build(), Optional.of("ppht"), Optional.of("E40"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u2075"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerMillion", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "partPerMillion" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "part per million" ).build(), Optional.of("ppm"), Optional.of("59"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u2076"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerQuadrillionUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "partPerQuadrillionUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Part per quadrillion (US)" ).build(), Optional.of("ppq"), Optional.empty(), Optional.empty(), Optional.of("1 \u00D7 10\u00AF\u00B9\u2075"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerThousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "partPerThousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "part per thousand" ).build(), Optional.of("\u2030"), Optional.of("NX"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "partPerTrillionUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "partPerTrillionUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "part per trillion (US)" ).build(), Optional.of("ppt"), Optional.empty(), Optional.empty(), Optional.of("1 \u00D7 10\u00AF\u00B9\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal" ).build(), Optional.of("Pa"), Optional.of("PAL"), Optional.empty(), Optional.of("Pa"), new HashSet<>(Arrays.asList(QuantityKinds.OSMOTIC_PRESSURE, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.FUGACITY_OF_B_IN_A_GASEOUS_MIXTURE, QuantityKinds.STATIC_PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.PARTIAL_PRESSURE_OF_B_IN_A_GASEOUS_MIXTURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.INSTANTANEOUS_SOUND_PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "pascalCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal cubic metre per second" ).build(), Optional.of("Pa\u00B7m\u00B3/s"), Optional.of("G01"), Optional.empty(), Optional.of("kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "pascalLitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalLitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal litre per second" ).build(), Optional.of("Pa\u00B7l/s"), Optional.of("F99"), Optional.empty(), Optional.of("10\u207B\u00B3 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY)) ) );
UNITS_BY_NAME.put( "pascalPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal per bar" ).build(), Optional.of("Pa/bar"), Optional.of("F07"), Optional.empty(), Optional.of("10\u207B\u2075"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "pascalPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal per kelvin" ).build(), Optional.of("Pa/K"), Optional.of("C64"), Optional.empty(), Optional.of("Pa/K"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "pascalPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal per metre" ).build(), Optional.of("Pa/m"), Optional.of("H42"), Optional.empty(), Optional.of("m\u207B\u00B2 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "pascalSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal second" ).build(), Optional.of("Pa\u00B7s"), Optional.of("C65"), Optional.empty(), Optional.of("Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "pascalSecondPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSecondPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal second per bar" ).build(), Optional.of("Pa\u00B7s/bar"), Optional.of("H07"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2075 s"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE)) ) );
UNITS_BY_NAME.put( "pascalSecondPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSecondPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal second per cubic metre" ).build(), Optional.of("Pa\u00B7s/m\u00B3"), Optional.of("C66"), Optional.empty(), Optional.of("Pa \u00D7 s/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ACOUSTIC_IMPEDANCE)) ) );
UNITS_BY_NAME.put( "pascalSecondPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSecondPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal second per kelvin" ).build(), Optional.of("Pa.s/K"), Optional.of("F77"), Optional.empty(), Optional.of("kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "pascalSecondPerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSecondPerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal second per litre" ).build(), Optional.of("Pa\u00B7s/l"), Optional.of("M32"), Optional.of("pascalSecondPerCubicMetre"), Optional.of("10\u00B3 Pa \u00D7 s/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.ACOUSTIC_IMPEDANCE)) ) );
UNITS_BY_NAME.put( "pascalSecondPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSecondPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal second per metre" ).build(), Optional.of("Pa\u00B7 s/m"), Optional.of("C67"), Optional.empty(), Optional.of("Pa \u00D7 s/m"), new HashSet<>(Arrays.asList(QuantityKinds.CHARACTERISTIC_IMPEDANCE_OF_A_MEDIUM)) ) );
UNITS_BY_NAME.put( "pascalSquareMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSquareMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal square metre per kilogram" ).build(), Optional.of("Pa/(kg/m\u00B2)"), Optional.of("P79"), Optional.of("metrePerSecondSquared"), Optional.of("m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.BURST_INDEX)) ) );
UNITS_BY_NAME.put( "pascalSquaredSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalSquaredSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal squared second" ).build(), Optional.of("Pa\u00B2\u00B7s"), Optional.of("P42"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u00B2 \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_EXPOSURE)) ) );
UNITS_BY_NAME.put( "pascalToThePowerSumOfStoichiometricNumbers", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pascalToThePowerSumOfStoichiometricNumbers" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pascal to the power sum of stoichiometric numbers" ).build(), Optional.of("Pa\u03A3\u03BDB"), Optional.of("P98"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pebibitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pebibit per cubic metre" ).build(), Optional.of("Pibit/m\u00B3"), Optional.of("E82"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pebibitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pebibit per metre" ).build(), Optional.of("Pibit/m"), Optional.of("E80"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pebibitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pebibit per square metre" ).build(), Optional.of("Pibit/m\u00B2"), Optional.of("E81"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pebibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pebibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Pebibyte" ).build(), Optional.of("PiB"), Optional.of("E60"), Optional.of("byte"), Optional.of("2\u2075\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "peck", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peck" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck" ).build(), Optional.of("pk (US)"), Optional.of("G23"), Optional.of("cubicMetre"), Optional.of("8.809768 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "peckUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (UK)" ).build(), Optional.of("pk (UK)"), Optional.of("L43"), Optional.of("cubicMetre"), Optional.of("9.092181 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "peckUkPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUkPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (UK) per day" ).build(), Optional.of("pk (UK)/d"), Optional.of("L44"), Optional.of("cubicMetrePerSecond"), Optional.of("1.052336 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUkPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUkPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (UK) per hour" ).build(), Optional.of("pk (UK)/h"), Optional.of("L45"), Optional.of("cubicMetrePerSecond"), Optional.of("2.525606 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUkPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUkPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (UK) per minute" ).build(), Optional.of("pk (UK)/min"), Optional.of("L46"), Optional.of("cubicMetrePerSecond"), Optional.of("1.5153635 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUkPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUkPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (UK) per second" ).build(), Optional.of("pk (UK)/s"), Optional.of("L47"), Optional.of("cubicMetrePerSecond"), Optional.of("9.092181 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUsDryPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (US dry) per day" ).build(), Optional.of("pk (US dry)/d"), Optional.of("L48"), Optional.of("cubicMetrePerSecond"), Optional.of("1.019649 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUsDryPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (US dry) per hour" ).build(), Optional.of("pk (US dry)/h"), Optional.of("L49"), Optional.of("cubicMetrePerSecond"), Optional.of("2.447158 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUsDryPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (US dry) per minute" ).build(), Optional.of("pk (US dry)/min"), Optional.of("L50"), Optional.of("cubicMetrePerSecond"), Optional.of("1.468295 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "peckUsDryPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "peckUsDryPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "peck (US dry) per second" ).build(), Optional.of("pk (US dry)/s"), Optional.of("L51"), Optional.of("cubicMetrePerSecond"), Optional.of("8.809768 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "penCalorie", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "penCalorie" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pen calorie" ).build(), Optional.empty(), Optional.of("N1"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "penGramProtein", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "penGramProtein" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pen gram (protein)" ).build(), Optional.empty(), Optional.of("D23"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pennyweight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pennyweight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pennyweight" ).build(), Optional.empty(), Optional.of("DWT"), Optional.of("gram"), Optional.of("1.555174 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "perMillePerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "perMillePerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "per mille per psi" ).build(), Optional.of("\u2030/psi"), Optional.of("J12"), Optional.of("reciprocalPascalOrPascalToThePowerMinusOne"), Optional.of("1.450377 \u00D7 10\u207B\u2077 Pa\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percent", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percent" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent" ).build(), Optional.of("%"), Optional.of("P1"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DIMENSIONLESS)) ) );
UNITS_BY_NAME.put( "percentPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per bar" ).build(), Optional.of("%/bar"), Optional.of("H96"), Optional.of("reciprocalPascalOrPascalToThePowerMinusOne"), Optional.of("10\u207B\u2077 Pa\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerDecakelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerDecakelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per decakelvin" ).build(), Optional.of("%/daK"), Optional.of("H73"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u00B3 K\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerDegree", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerDegree" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per degree" ).build(), Optional.of("%/\u00B0"), Optional.of("H90"), Optional.empty(), Optional.of("0.5729578 rad\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per degree Celsius" ).build(), Optional.of("%/\u00B0C"), Optional.of("M25"), Optional.empty(), Optional.of("10\u207B\u00B2 \u00B0C\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerHectobar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerHectobar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per hectobar" ).build(), Optional.of("%/hbar"), Optional.of("H72"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerHundred", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerHundred" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per hundred" ).build(), Optional.of("%/100"), Optional.of("H93"), Optional.empty(), Optional.of("10\u207B\u2074"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per inch" ).build(), Optional.of("%/in"), Optional.of("H98"), Optional.of("reciprocalMetre"), Optional.of("0.3937008 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per kelvin" ).build(), Optional.of("%/K"), Optional.of("H25"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u00B2 K\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per metre" ).build(), Optional.of("%/m"), Optional.of("H99"), Optional.of("reciprocalMetre"), Optional.of("10\u207B\u00B2 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per millimetre" ).build(), Optional.of("%/mm"), Optional.of("J10"), Optional.of("reciprocalMetre"), Optional.of("10 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerMonth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerMonth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per month" ).build(), Optional.of("%/mo"), Optional.of("H71"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerOhm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerOhm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per ohm" ).build(), Optional.of("%/\u2126"), Optional.of("H89"), Optional.empty(), Optional.of("10\u207B\u00B2 \u2126\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerOneHundredThousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerOneHundredThousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per one hundred thousand" ).build(), Optional.of("%/100000"), Optional.of("H92"), Optional.empty(), Optional.of("10\u207B\u2077"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerTenThousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerTenThousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per ten thousand" ).build(), Optional.of("%/10000"), Optional.of("H91"), Optional.empty(), Optional.of("10\u207B\u2076"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerThousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerThousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per thousand" ).build(), Optional.of("%/1000"), Optional.of("H94"), Optional.empty(), Optional.of("10\u207B\u2075"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentPerVolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentPerVolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent per volt" ).build(), Optional.of("%/V"), Optional.of("H95"), Optional.empty(), Optional.of("10\u207B\u00B2 V\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentVolume", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentVolume" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent volume" ).build(), Optional.empty(), Optional.of("VP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "percentWeight", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "percentWeight" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "percent weight" ).build(), Optional.empty(), Optional.of("60"), Optional.empty(), Optional.of("1 \u00D7 10\u207B\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "perm0Degreesc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "perm0Degreesc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "perm (0 °C)" ).build(), Optional.of("perm (0 \u00B0C)"), Optional.of("P91"), Optional.of("kilogramPerSquareMetrePascalSecond"), Optional.of("5.72135 \u00D7 10\u207B\u00B9\u00B9 kg/(m\u00B2 \u00D7 Pa \u00D7 s)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "perm23Degreesc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "perm23Degreesc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "perm (23 °C)" ).build(), Optional.of("perm (23 \u00B0C)"), Optional.of("P92"), Optional.of("kilogramPerSquareMetrePascalSecond"), Optional.of("5.74525 \u00D7 10\u207B\u00B9\u00B9 kg/(m\u00B2 \u00D7 Pa \u00D7 s)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "person", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "person" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "person" ).build(), Optional.empty(), Optional.of("IE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "petabit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "petabit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "petabit" ).build(), Optional.of("Pbit"), Optional.of("E78"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "petabitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "petabitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "petabit per second" ).build(), Optional.of("Pbit/s"), Optional.of("E79"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "petabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "petabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "petabyte" ).build(), Optional.of("Pbyte"), Optional.of("E36"), Optional.of("byte"), Optional.of("10\u00B9\u2075 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "petajoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "petajoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "petajoule" ).build(), Optional.of("PJ"), Optional.of("C68"), Optional.of("joule"), Optional.of("10\u00B9\u2075 J"), new HashSet<>(Arrays.asList(QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY)) ) );
UNITS_BY_NAME.put( "pferdestaerke", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pferdestaerke" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Pferdestaerke" ).build(), Optional.of("PS"), Optional.of("N12"), Optional.of("watt"), Optional.of("7.354988 \u00D7 10\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "pfund", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pfund" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pfund" ).build(), Optional.of("pfd"), Optional.of("M86"), Optional.of("kilogram"), Optional.of("0.5 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "phon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "phon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "phon" ).build(), Optional.empty(), Optional.of("C69"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.LOUDNESS_LEVEL)) ) );
UNITS_BY_NAME.put( "phot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "phot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "phot" ).build(), Optional.of("ph"), Optional.of("P26"), Optional.empty(), Optional.of("10\u2074 cd \u00D7 sr / m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ILLUMINANCE)) ) );
UNITS_BY_NAME.put( "pica", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pica" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pica" ).build(), Optional.empty(), Optional.of("R1"), Optional.of("metre"), Optional.of("4.217518 \u00D7 10\u207B\u00B3 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "picoampere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picoampere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picoampere" ).build(), Optional.of("pA"), Optional.of("C70"), Optional.of("ampere"), Optional.of("10\u207B\u00B9\u00B2 A"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CURRENT, QuantityKinds.MAGNETOMOTIVE_FORCE, QuantityKinds.MAGNETIC_POTENTIAL_DIFFERENCE, QuantityKinds.CURRENT_LINKAGE)) ) );
UNITS_BY_NAME.put( "picocoulomb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picocoulomb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picocoulomb" ).build(), Optional.of("pC"), Optional.of("C71"), Optional.of("coulomb"), Optional.of("10\u207B\u00B9\u00B2 C"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_CHARGE, QuantityKinds.QUANTITY_OF_ELECTRICITY, QuantityKinds.ELECTRIC_FLUX_FLUX_OF_DISPLACEMENT)) ) );
UNITS_BY_NAME.put( "picofarad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picofarad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picofarad" ).build(), Optional.of("pF"), Optional.of("4T"), Optional.of("farad"), Optional.of("10\u207B\u00B9\u00B2 F"), new HashSet<>(Arrays.asList(QuantityKinds.CAPACITANCE)) ) );
UNITS_BY_NAME.put( "picofaradPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picofaradPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picofarad per metre" ).build(), Optional.of("pF/m"), Optional.of("C72"), Optional.of("faradPerMetre"), Optional.of("10\u207B\u00B9\u00B2 F/m"), new HashSet<>(Arrays.asList(QuantityKinds.PERMITTIVITY, QuantityKinds.PERMITTIVITY_OF_VACUUM, QuantityKinds.ELECTRIC_CONSTANT)) ) );
UNITS_BY_NAME.put( "picohenry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picohenry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picohenry" ).build(), Optional.of("pH"), Optional.of("C73"), Optional.of("henry"), Optional.of("10\u207B\u00B9\u00B2 H"), new HashSet<>(Arrays.asList(QuantityKinds.MUTUAL_INDUCTANCE, QuantityKinds.PERMEANCE, QuantityKinds.SELF_INDUCTANCE)) ) );
UNITS_BY_NAME.put( "picometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picometre" ).build(), Optional.of("pm"), Optional.of("C52"), Optional.of("metre"), Optional.of("10\u207B\u00B9\u00B2 m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.DIAMETER, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.LENGTH, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "picopascalPerKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picopascalPerKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picopascal per kilometre" ).build(), Optional.of("pPa/km"), Optional.of("H69"), Optional.empty(), Optional.of("10\u207B\u00B9\u2075 m\u207B\u00B2 \u00D7 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "picosecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picosecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picosecond" ).build(), Optional.of("ps"), Optional.of("H70"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u00B9\u00B2 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "picosiemens", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picosiemens" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picosiemens" ).build(), Optional.of("pS"), Optional.of("N92"), Optional.of("siemens"), Optional.of("10\u207B\u00B9\u00B2 S"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT)) ) );
UNITS_BY_NAME.put( "picosiemensPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picosiemensPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picosiemens per metre" ).build(), Optional.of("pS/m"), Optional.of("L42"), Optional.of("siemensPerMetre"), Optional.of("10\u207B\u00B9\u00B2 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "picovolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picovolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picovolt" ).build(), Optional.of("pV"), Optional.of("N99"), Optional.of("voltDc"), Optional.of("10\u207B\u00B9\u00B2 V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.VOLTAGE, QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.TENSION)) ) );
UNITS_BY_NAME.put( "picowatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picowatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picowatt" ).build(), Optional.of("pW"), Optional.of("C75"), Optional.of("watt"), Optional.of("10\u207B\u00B9\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "picowattPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "picowattPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "picowatt per square metre" ).build(), Optional.of("pW/m\u00B2"), Optional.of("C76"), Optional.of("wattPerSquareMetre"), Optional.of("10\u207B\u00B9\u00B2 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY)) ) );
UNITS_BY_NAME.put( "piece", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "piece" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "piece" ).build(), Optional.empty(), Optional.of("H87"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ping", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ping" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ping" ).build(), Optional.empty(), Optional.of("E19"), Optional.of("squareMetre"), Optional.of("3.305 m\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pintUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (UK)" ).build(), Optional.of("pt (UK)"), Optional.of("PTI"), Optional.of("cubicMetre"), Optional.of("5. 68261 \u00D7 10\u207B\u2074 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "pintUkPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUkPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (UK) per day" ).build(), Optional.of("pt (UK)/d"), Optional.of("L53"), Optional.of("cubicMetrePerSecond"), Optional.of("6.577098 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUkPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUkPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (UK) per hour" ).build(), Optional.of("pt (UK)/h"), Optional.of("L54"), Optional.of("cubicMetrePerSecond"), Optional.of("1.578504 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
}

private static void init13() {
UNITS_BY_NAME.put( "pintUkPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUkPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (UK) per minute" ).build(), Optional.of("pt (UK)/min"), Optional.of("L55"), Optional.of("cubicMetrePerSecond"), Optional.of("9.471022 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUkPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUkPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (UK) per second" ).build(), Optional.of("pt (UK)/s"), Optional.of("L56"), Optional.of("cubicMetrePerSecond"), Optional.of("5.682613 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUsLiquidPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (US liquid) per day" ).build(), Optional.of("pt (US liq.)/d"), Optional.of("L57"), Optional.of("cubicMetrePerSecond"), Optional.of("5.476580 \u00D7 10\u207B\u2079 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUsLiquidPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (US liquid) per hour" ).build(), Optional.of("pt (US liq.)/h"), Optional.of("L58"), Optional.of("cubicMetrePerSecond"), Optional.of("1.314379 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUsLiquidPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (US liquid) per minute" ).build(), Optional.of("pt (US liq.)/min"), Optional.of("L59"), Optional.of("cubicMetrePerSecond"), Optional.of("7.886275 \u00D7 10\u207B\u2076 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pintUsLiquidPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pintUsLiquidPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pint (US liquid) per second" ).build(), Optional.of("pt (US liq.)/s"), Optional.of("L60"), Optional.of("cubicMetrePerSecond"), Optional.of("4.731765 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "pipelineJoint", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pipelineJoint" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pipeline joint" ).build(), Optional.empty(), Optional.of("JNT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pitch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pitch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pitch" ).build(), Optional.empty(), Optional.of("PI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "pixel", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pixel" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pixel" ).build(), Optional.empty(), Optional.of("E37"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poise", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poise" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poise" ).build(), Optional.of("P"), Optional.of("89"), Optional.of("pascalSecond"), Optional.of("0.1 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poisePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poisePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poise per bar" ).build(), Optional.of("P/bar"), Optional.of("F06"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2076 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poisePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poisePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poise per kelvin" ).build(), Optional.of("P/K"), Optional.of("F86"), Optional.empty(), Optional.of("10\u207B\u00B9 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poisePerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poisePerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poise per pascal" ).build(), Optional.of("P/Pa"), Optional.of("N35"), Optional.of("secondUnitOfTime"), Optional.of("0.1 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "pond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pond" ).build(), Optional.of("p"), Optional.of("M78"), Optional.of("newton"), Optional.of("9.80665 \u00D7 10\u207B\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "pound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "pound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound" ).build(), Optional.of("lb"), Optional.of("LBR"), Optional.of("kilogram"), Optional.of("0.45359237 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicFootDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerCubicFootDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per cubic foot degree Fahrenheit" ).build(), Optional.of("(lb/ft\u00B3)/\u00B0F"), Optional.of("K69"), Optional.empty(), Optional.of("28.83323 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicFootPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerCubicFootPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per cubic foot psi" ).build(), Optional.of("(lb/ft\u00B3)/psi"), Optional.of("K70"), Optional.empty(), Optional.of("2.323282 \u00D7 10\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicInchDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerCubicInchDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per cubic inch degree Fahrenheit" ).build(), Optional.of("(lb/in\u00B3)/\u00B0F"), Optional.of("K75"), Optional.empty(), Optional.of("4.982384 \u00D7 10\u2074 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerCubicInchPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerCubicInchPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per cubic inch psi" ).build(), Optional.of("(lb/in\u00B3)/psi"), Optional.of("K76"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("4.014632 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per day" ).build(), Optional.of("lb/d"), Optional.of("K66"), Optional.of("kilogramPerSecond"), Optional.of("5.249912 \u00D7 10\u207B\u2076 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per degree Fahrenheit" ).build(), Optional.of("lb/\u00B0F"), Optional.of("K64"), Optional.of("kilogramPerKelvin"), Optional.of("0.8164663 kg/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerGallonUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerGallonUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per gallon (UK)" ).build(), Optional.of("lb/gal (UK)"), Optional.of("K71"), Optional.of("kilogramPerCubicMetre"), Optional.of("99.77637 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerHourDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerHourDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per hour degree Fahrenheit" ).build(), Optional.of("(lb/h)/\u00B0F"), Optional.of("K73"), Optional.empty(), Optional.of("2.267962 \u00D7 10\u207B\u2074 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerHourPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerHourPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per hour psi" ).build(), Optional.of("(lb/h)/psi"), Optional.of("K74"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.827445 \u00D7 10\u207B\u2078 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per minute" ).build(), Optional.of("lb/min"), Optional.of("K78"), Optional.of("kilogramPerSecond"), Optional.of("7.559873 \u00D7 10\u207B\u00B3 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerMinuteDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerMinuteDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per minute degree Fahrenheit" ).build(), Optional.of("lb/(min\u00B7\u00B0F)"), Optional.of("K79"), Optional.empty(), Optional.of("1.360777 \u00D7 10\u207B\u00B2 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerMinutePsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerMinutePsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per minute psi" ).build(), Optional.of("(lb/min)/psi"), Optional.of("K80"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.096467 \u00D7 10\u207B\u2076 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per psi" ).build(), Optional.of("lb/psi"), Optional.of("K77"), Optional.of("kilogramPerPascal"), Optional.of("6.578802 \u00D7 10\u207B\u2075 kg/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per second" ).build(), Optional.of("lb/s"), Optional.of("K81"), Optional.of("kilogramPerSecond"), Optional.of("0.4535924 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerSecondDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerSecondDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per second degree Fahrenheit" ).build(), Optional.of("(lb/s)/\u00B0F"), Optional.of("K82"), Optional.empty(), Optional.of("0.8164663 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisPerSecondPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisPerSecondPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) per second psi" ).build(), Optional.of("(lb/s)/psi"), Optional.of("K83"), Optional.of("kilogramPerSecondPascal"), Optional.of("6.578802 \u00D7 10\u207B\u2075 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundAvoirdupoisSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundAvoirdupoisSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound (avoirdupois) square foot" ).build(), Optional.of("lb\u00B7ft\u00B2"), Optional.of("K65"), Optional.of("kilogramMetreSquared"), Optional.of("4.214011 \u00D7 10\u207B\u00B2 kg \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA)) ) );
UNITS_BY_NAME.put( "poundFootPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundFootPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound foot per second" ).build(), Optional.of("lb\u00B7(ft/s)"), Optional.of("N10"), Optional.of("kilogramMetrePerSecond"), Optional.of("1.382550 \u00D7 10\u207B\u00B9 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "poundForce", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForce" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force" ).build(), Optional.of("lbf"), Optional.of("C78"), Optional.of("newton"), Optional.of("4.448222 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "poundForceFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force foot" ).build(), Optional.of("lbf\u00B7ft"), Optional.of("M92"), Optional.of("newtonMetre"), Optional.of("1.355818 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE)) ) );
UNITS_BY_NAME.put( "poundForceFootPerAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceFootPerAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force foot per ampere" ).build(), Optional.of("lbf\u00B7ft/A"), Optional.of("F22"), Optional.empty(), Optional.of("1.35582 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundForceFootPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceFootPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force foot per inch" ).build(), Optional.of("lbf\u00B7ft/in"), Optional.of("P89"), Optional.empty(), Optional.of("53.37866 m \u00D7 kg / s\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poundForceFootPerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceFootPerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force foot per pound" ).build(), Optional.of("lbf\u00B7ft/lb"), Optional.of("G20"), Optional.empty(), Optional.of("2.98907 m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.WORK_PER_UNIT_WEIGHT, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundForceInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force inch" ).build(), Optional.of("lbf\u00B7in"), Optional.of("F21"), Optional.empty(), Optional.of("1.12985 \u00D7 10\u207B\u00B9 kg \u00D7 m\u00B2 \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundForceInchPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceInchPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force inch per inch" ).build(), Optional.of("lbf\u00B7in/in"), Optional.of("P90"), Optional.empty(), Optional.of("4.448222 m \u00D7 kg / s\u00B2"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poundForcePerFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForcePerFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force per foot" ).build(), Optional.of("lbf/ft"), Optional.of("F17"), Optional.empty(), Optional.of("1.45939 \u00D7 10\u00B9 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.FORCE_DIVIDED_BY_LENGTH)) ) );
UNITS_BY_NAME.put( "poundForcePerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForcePerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force per inch" ).build(), Optional.of("lbf/in"), Optional.of("F48"), Optional.empty(), Optional.of("1.75127 \u00D7 10\u00B2 kg \u00D7 s\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.FORCE_DIVIDED_BY_LENGTH)) ) );
UNITS_BY_NAME.put( "poundForcePerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForcePerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force per square foot" ).build(), Optional.of("lbf/ft\u00B2"), Optional.of("K85"), Optional.of("pascal"), Optional.of("47.88026 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "poundForcePerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForcePerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force per square inch" ).build(), Optional.of("lbf/in\u00B2"), Optional.of("PS"), Optional.of("pascal"), Optional.of("6.894757 \u00D7 10\u00B3 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "poundForcePerSquareInchDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForcePerSquareInchDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force per square inch degree Fahrenheit" ).build(), Optional.of("psi/\u00B0F"), Optional.of("K86"), Optional.of("pascalPerKelvin"), Optional.of("1.241056 \u00D7 10\u2074 Pa/K"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_STRESS)) ) );
UNITS_BY_NAME.put( "poundForcePerYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForcePerYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force per yard" ).build(), Optional.of("lbf/yd"), Optional.of("N33"), Optional.of("newtonPerMetre"), Optional.of("4.864635 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "poundForceSecondPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceSecondPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force second per square foot" ).build(), Optional.of("lbf\u00B7s/ft\u00B2"), Optional.of("K91"), Optional.of("pascalSecond"), Optional.of("47.88026 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundForceSecondPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundForceSecondPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound-force second per square inch" ).build(), Optional.of("lbf\u00B7s/in\u00B2"), Optional.of("K92"), Optional.of("pascalSecond"), Optional.of("6.894757 \u00D7 10\u00B3 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundInchPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundInchPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound inch per second" ).build(), Optional.of("lb\u00B7(in/s)"), Optional.of("N11"), Optional.of("kilogramMetrePerSecond"), Optional.of("1.152125 \u00D7 10\u207B\u00B2 kg \u00D7 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENTUM)) ) );
UNITS_BY_NAME.put( "poundInchSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundInchSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound inch squared" ).build(), Optional.of("lb\u00B7in\u00B2"), Optional.of("F20"), Optional.of("kilogramMetreSquared"), Optional.of("2.926397 \u00D7 10\u207B\u2074 kg \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_INERTIA_DYNAMIC_MOMENT_OF_INERTIA)) ) );
UNITS_BY_NAME.put( "poundMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound mole" ).build(), Optional.of("lbmol"), Optional.of("P44"), Optional.of("mole"), Optional.of("453.5924 mol"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE)) ) );
UNITS_BY_NAME.put( "poundMolePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundMolePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound mole per minute" ).build(), Optional.of("lbmol/h"), Optional.of("P46"), Optional.of("molePerSecond"), Optional.of("7.559873 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "poundMolePerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundMolePerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound mole per pound" ).build(), Optional.of("lbmol/lb"), Optional.of("P48"), Optional.of("molePerKilogram"), Optional.of("10\u00B3 mol/kg"), new HashSet<>(Arrays.asList(QuantityKinds.IONIC_STRENGTH)) ) );
UNITS_BY_NAME.put( "poundMolePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundMolePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound mole per second" ).build(), Optional.of("lbmol/s"), Optional.of("P45"), Optional.of("molePerSecond"), Optional.of("4.535924 \u00D7 10\u00B2 mol/s"), new HashSet<>(Arrays.asList(QuantityKinds.CATALYTIC_ACTIVITY)) ) );
UNITS_BY_NAME.put( "poundPerCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per cubic foot" ).build(), Optional.of("lb/ft\u00B3"), Optional.of("87"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.601846 \u00D7 10\u00B9 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerCubicInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerCubicInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per cubic inch" ).build(), Optional.of("lb/in\u00B3"), Optional.of("LA"), Optional.of("kilogramPerCubicMetre"), Optional.of("2.767990 \u00D7 10\u2074 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerCubicYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerCubicYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per cubic yard" ).build(), Optional.of("lb/yd\u00B3"), Optional.of("K84"), Optional.of("kilogramPerCubicMetre"), Optional.of("0.5932764 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per foot" ).build(), Optional.of("lb/ft"), Optional.of("P2"), Optional.of("kilogramPerMetre"), Optional.of("1.488164 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerFootDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per foot day" ).build(), Optional.of("lb/(ft\u00B7d)"), Optional.of("N44"), Optional.of("pascalSecond"), Optional.of("1.722412 \u00D7 10\u207B\u2075 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerFootHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per foot hour" ).build(), Optional.of("lb/(ft\u00B7h)"), Optional.of("K67"), Optional.of("pascalSecond"), Optional.of("4.133789 \u00D7 10\u207B\u2074 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerFootMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per foot minute" ).build(), Optional.of("lb/(ft\u00B7min)"), Optional.of("N43"), Optional.of("pascalSecond"), Optional.of("2.480273 \u00D7 10\u207B\u00B2 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerFootSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerFootSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per foot second" ).build(), Optional.of("lb/(ft\u00B7s)"), Optional.of("K68"), Optional.of("pascalSecond"), Optional.of("1.488164 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundPerGallonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerGallonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per gallon (US)" ).build(), Optional.of("lb/gal (US)"), Optional.of("GE"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.198264 \u00D7 10\u00B2 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per hour" ).build(), Optional.of("lb/h"), Optional.of("4U"), Optional.of("kilogramPerSecond"), Optional.of("1.259979 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "poundPerInchOfLengthUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerInchOfLengthUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per inch of length" ).build(), Optional.of("lb/in"), Optional.of("PO"), Optional.of("kilogramPerMetre"), Optional.of("1.785797 \u00D7 10\u00B9 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "poundPerPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per pound" ).build(), Optional.of("lb/lb"), Optional.of("M91"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_RATIO)) ) );
UNITS_BY_NAME.put( "poundPerReam", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerReam" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per ream" ).build(), Optional.empty(), Optional.of("RP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "poundPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per square foot" ).build(), Optional.of("lb/ft\u00B2"), Optional.of("FP"), Optional.of("kilogramPerSquareMetre"), Optional.of("4.882428 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "poundPerSquareInchAbsolute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerSquareInchAbsolute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per square inch absolute" ).build(), Optional.of("lb/in\u00B2"), Optional.of("80"), Optional.of("kilogramPerSquareMetre"), Optional.of("7.030696 \u00D7 10\u00B2 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.PRESSURE, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.NORMAL_STRESS)) ) );
UNITS_BY_NAME.put( "poundPerSquareYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerSquareYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per square yard" ).build(), Optional.of("lb/yd\u00B2"), Optional.of("N25"), Optional.of("kilogramPerSquareMetre"), Optional.of("5.424919 \u00D7 10\u207B\u00B9 kg/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SHEAR_STRESS, QuantityKinds.NORMAL_STRESS, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.MODULUS_OF_ELASTICITY)) ) );
UNITS_BY_NAME.put( "poundPerYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundPerYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "pound per yard" ).build(), Optional.of("lb/yd"), Optional.of("M84"), Optional.of("kilogramPerMetre"), Optional.of("4.960546 \u00D7 10\u207B\u00B9 kg/m"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_MASS, QuantityKinds.LINEAR_DENSITY)) ) );
UNITS_BY_NAME.put( "poundal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal" ).build(), Optional.of("pdl"), Optional.of("M76"), Optional.of("newton"), Optional.of("1.382550 \u00D7 10\u207B\u00B9 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "poundalFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal foot" ).build(), Optional.of("pdl\u00B7ft"), Optional.of("M95"), Optional.of("newtonMetre"), Optional.of("4.214011 \u00D7 10\u207B\u00B2 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_A_COUPLE, QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE)) ) );
UNITS_BY_NAME.put( "poundalInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal inch" ).build(), Optional.of("pdl\u00B7in"), Optional.of("M96"), Optional.of("newtonMetre"), Optional.of("3.51167710\u207B\u00B3 N \u00D7 m"), new HashSet<>(Arrays.asList(QuantityKinds.MOMENT_OF_FORCE, QuantityKinds.TORQUE, QuantityKinds.MOMENT_OF_A_COUPLE)) ) );
UNITS_BY_NAME.put( "poundalPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal per inch" ).build(), Optional.of("pdl/in"), Optional.of("N32"), Optional.of("newtonPerMetre"), Optional.of("5.443110 N/m"), new HashSet<>(Arrays.asList(QuantityKinds.SURFACE_TENSION)) ) );
UNITS_BY_NAME.put( "poundalPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal per square foot" ).build(), Optional.of("pdl/ft\u00B2"), Optional.of("N21"), Optional.of("pascal"), Optional.of("1.488164 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.PRESSURE, QuantityKinds.BULK_MODULUS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "poundalPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal per square inch" ).build(), Optional.of("pdl/in\u00B2"), Optional.of("N26"), Optional.of("pascal"), Optional.of("2.142957 \u00D7 10\u00B2 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.BULK_MODULUS, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.SHEAR_MODULUS)) ) );
UNITS_BY_NAME.put( "poundalSecondPerSquareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalSecondPerSquareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal second per square foot" ).build(), Optional.of("(pdl/ft\u00B2)\u00B7s"), Optional.of("N34"), Optional.of("pascalSecond"), Optional.of("1.488164 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "poundalSecondPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "poundalSecondPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "poundal second per square inch" ).build(), Optional.of("(pdl/in\u00B2)\u00B7s"), Optional.of("N42"), Optional.of("pascalSecond"), Optional.of("2.142957 \u00D7 10\u00B2 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "printPoint", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "printPoint" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "print point" ).build(), Optional.empty(), Optional.of("N3"), Optional.empty(), Optional.of("0.0138 in (appro\u00D7)"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "proofGallon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "proofGallon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "proof gallon" ).build(), Optional.empty(), Optional.of("PGL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "proofLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "proofLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "proof litre" ).build(), Optional.empty(), Optional.of("PFL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "psiCubicInchPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "psiCubicInchPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "psi cubic inch per second" ).build(), Optional.of("psi\u00B7in\u00B3/s"), Optional.of("K87"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("0.112985 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "psiCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "psi cubic metre per second" ).build(), Optional.of("psi\u00B7m\u00B3/s"), Optional.of("K89"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("6.894757 \u00D7 10\u00B3 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiCubicYardPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "psiCubicYardPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "psi cubic yard per second" ).build(), Optional.of("psi\u00B7yd\u00B3/s"), Optional.of("K90"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("5.271420 \u00D7 10\u00B3 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiLitrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "psiLitrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "psi litre per second" ).build(), Optional.of("psi\u00B7l/s"), Optional.of("K88"), Optional.of("pascalCubicMetrePerSecond"), Optional.of("6.894757 Pa \u00D7 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.LEAKAGE_RATE_OF_GAS)) ) );
UNITS_BY_NAME.put( "psiPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "psiPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "psi per inch" ).build(), Optional.of("psi/in"), Optional.of("P86"), Optional.empty(), Optional.of("2.714471 \u00D7 10\u2075 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "psiPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "psiPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "psi per psi" ).build(), Optional.of("psi/psi"), Optional.of("L52"), Optional.empty(), Optional.of("1.0"), new HashSet<>(Arrays.asList(QuantityKinds.PRESSURE_RATIO)) ) );
UNITS_BY_NAME.put( "quad1015Btuit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quad1015Btuit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quad (1015 BtuIT)" ).build(), Optional.of("quad"), Optional.of("N70"), Optional.of("joule"), Optional.of("1.055056 \u00D7 10\u00B9\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.ENERGY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENTHALPY, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "quartUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (UK)" ).build(), Optional.of("qt (UK)"), Optional.of("QTI"), Optional.of("cubicMetre"), Optional.of("1.1365225 \u00D7 10\u207B\u00B3 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUkLiquidPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (UK liquid) per day" ).build(), Optional.of("qt (UK liq.)/d"), Optional.of("K94"), Optional.of("cubicMetrePerSecond"), Optional.of("1.315420 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUkLiquidPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (UK liquid) per hour" ).build(), Optional.of("qt (UK liq.)/h"), Optional.of("K95"), Optional.of("cubicMetrePerSecond"), Optional.of("3.157008 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUkLiquidPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (UK liquid) per minute" ).build(), Optional.of("qt (UK liq.)/min"), Optional.of("K96"), Optional.of("cubicMetrePerSecond"), Optional.of("1.894205 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUkLiquidPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUkLiquidPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (UK liquid) per second" ).build(), Optional.of("qt (UK liq.)/s"), Optional.of("K97"), Optional.of("cubicMetrePerSecond"), Optional.of("1.136523 \u00D7 10\u207B\u00B3 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUsLiquidPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (US liquid) per day" ).build(), Optional.of("qt (US liq.)/d"), Optional.of("K98"), Optional.of("cubicMetrePerSecond"), Optional.of("1.095316 \u00D7 10\u207B\u2078 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUsLiquidPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (US liquid) per hour" ).build(), Optional.of("qt (US liq.)/h"), Optional.of("K99"), Optional.of("cubicMetrePerSecond"), Optional.of("2.628758 \u00D7 10\u207B\u2077 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUsLiquidPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (US liquid) per minute" ).build(), Optional.of("qt (US liq.)/min"), Optional.of("L10"), Optional.of("cubicMetrePerSecond"), Optional.of("1.577255 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quartUsLiquidPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quartUsLiquidPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quart (US liquid) per second" ).build(), Optional.of("qt (US liq.)/s"), Optional.of("L11"), Optional.of("cubicMetrePerSecond"), Optional.of("9.463529 \u00D7 10\u207B\u2074 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "quarterOfAYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quarterOfAYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quarter (of a year)" ).build(), Optional.empty(), Optional.of("QAN"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "quarterUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quarterUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quarter (UK)" ).build(), Optional.of("Qr (UK)"), Optional.of("QTR"), Optional.of("kilogram"), Optional.of("12.700 59 kg"), Collections.emptySet() ) );
}

private static void init14() {
UNITS_BY_NAME.put( "quire", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "quire" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "quire" ).build(), Optional.of("qr"), Optional.of("QR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "rackUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "rackUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rack unit" ).build(), Optional.of("U or RU"), Optional.of("H80"), Optional.of("metre"), Optional.of("4.445 \u00D7 10\u207B\u00B2 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "rad", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "rad" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rad" ).build(), Optional.of("rad"), Optional.of("C80"), Optional.of("gray"), Optional.of("10\u207B\u00B2 Gy"), new HashSet<>(Arrays.asList(QuantityKinds.ABSORBED_DOSE)) ) );
UNITS_BY_NAME.put( "radian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "radian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "radian" ).build(), Optional.of("rad"), Optional.of("C81"), Optional.empty(), Optional.of("rad"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_DISPLACEMENT, QuantityKinds.BRAGG_ANGLE, QuantityKinds.PHASE_DIFFERENCE, QuantityKinds.ANGLE_OF_OPTICAL_ROTATION, QuantityKinds.LOSS_ANGLE, QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "radianPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "radianPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "radian per metre" ).build(), Optional.of("rad/m"), Optional.of("C84"), Optional.empty(), Optional.of("rad/m"), new HashSet<>(Arrays.asList(QuantityKinds.DEBYE_ANGULAR_WAVE_NUMBER, QuantityKinds.ANGULAR_WAVENUMBER, QuantityKinds.ANGULAR_REPETENCY, QuantityKinds.DEBYE_ANGULAR_REPETENCY, QuantityKinds.ANGULAR_WAVE_NUMBER)) ) );
UNITS_BY_NAME.put( "radianPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "radianPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "radian per second" ).build(), Optional.of("rad/s"), Optional.of("2A"), Optional.empty(), Optional.of("rad/s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_VELOCITY, QuantityKinds.PULSATANCE, QuantityKinds.CIRCULAR_FREQUENCY, QuantityKinds.DEBYE_ANGULAR_FREQUENCY, QuantityKinds.CYCLOTRON_ANGULAR_FREQUENCY, QuantityKinds.ANGULAR_FREQUENCY, QuantityKinds.NUCLEAR_PRECESSION)) ) );
UNITS_BY_NAME.put( "radianPerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "radianPerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "radian per second squared" ).build(), Optional.of("rad/s\u00B2"), Optional.of("2B"), Optional.empty(), Optional.of("rad/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_ACCELERATION)) ) );
UNITS_BY_NAME.put( "radianSquareMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "radianSquareMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "radian square metre per kilogram" ).build(), Optional.of("rad\u00B7m\u00B2/kg"), Optional.of("C83"), Optional.empty(), Optional.of("rad \u00D7 m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_OPTICAL_ROTATORY_POWER, QuantityKinds.ROTATORY_POWER, QuantityKinds.MASSIC_OPTICAL)) ) );
UNITS_BY_NAME.put( "radianSquareMetrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "radianSquareMetrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "radian square metre per mole" ).build(), Optional.of("rad\u00B7m\u00B2/mol"), Optional.of("C82"), Optional.empty(), Optional.of("rad \u00D7 m\u00B2/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_OPTICAL_ROTATORY_POWER)) ) );
UNITS_BY_NAME.put( "rate", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "rate" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rate" ).build(), Optional.empty(), Optional.of("A9"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ration", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ration" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ration" ).build(), Optional.empty(), Optional.of("13"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "ream", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "ream" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ream" ).build(), Optional.empty(), Optional.of("RM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalAngstrom", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalAngstrom" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal angstrom" ).build(), Optional.of("\u00C5\u207B\u00B9"), Optional.of("C85"), Optional.of("reciprocalMetre"), Optional.of("10\u00B9\u2070 m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FERMI_ANGULAR_WAVE_NUMBER, QuantityKinds.FERMI_ANGULAR_REPETENCY)) ) );
UNITS_BY_NAME.put( "reciprocalBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal bar" ).build(), Optional.of("1/bar"), Optional.of("F58"), Optional.empty(), Optional.of("bar\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.COMPRESSIBILITY, QuantityKinds.BULK_COMPRESSIBILITY)) ) );
UNITS_BY_NAME.put( "reciprocalCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal centimetre" ).build(), Optional.of("cm\u207B\u00B9"), Optional.of("E90"), Optional.of("reciprocalMetre"), Optional.of("10\u00B2 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalCubicCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic centimetre" ).build(), Optional.of("cm\u207B\u00B3"), Optional.of("H50"), Optional.of("reciprocalCubicMetre"), Optional.of("10\u2076 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic foot" ).build(), Optional.of("1/ft\u00B3"), Optional.of("K20"), Optional.of("reciprocalCubicMetre"), Optional.of("35.31466 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalCubicInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic inch" ).build(), Optional.of("1/in\u00B3"), Optional.of("K49"), Optional.of("reciprocalCubicMetre"), Optional.of("6.1023759 \u00D7 10\u2074 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic metre" ).build(), Optional.of("m\u207B\u00B3"), Optional.of("C86"), Optional.empty(), Optional.of("m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_ACCEPTOR_NUMBER, QuantityKinds.VOLUMIC_ELECTRON_NUMBER, QuantityKinds.ION_NUMBER_DENSITY, QuantityKinds.DONOR_NUMBER_DENSITY, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.VOLUMIC_DONOR_NUMBER, QuantityKinds.NEUTRON_NUMBER_DENSITY, QuantityKinds.HOLE_NUMBER_DENSITY, QuantityKinds.ACCEPTOR_NUMBER_DENSITY, QuantityKinds.ION_DENSITY, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.ELECTRON_NUMBER_DENSITY, QuantityKinds.VOLUMIC_HOLE_NUMBER, QuantityKinds.INTRINSIC_NUMBER_DENSITY, QuantityKinds.VOLUMIC_INTRINSIS_NUMBER)) ) );
UNITS_BY_NAME.put( "reciprocalCubicMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic metre per second" ).build(), Optional.of("m\u207B\u00B3/s"), Optional.of("C87"), Optional.empty(), Optional.of("m\u207B\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.SLOWING_DOWN_DENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalCubicMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic millimetre" ).build(), Optional.of("1/mm\u00B3"), Optional.of("L20"), Optional.of("reciprocalCubicMetre"), Optional.of("10\u2079 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalCubicYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalCubicYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal cubic yard" ).build(), Optional.of("1/yd\u00B3"), Optional.of("M10"), Optional.of("reciprocalCubicMetre"), Optional.of("1.307951 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES, QuantityKinds.MOLECULAR_CONCENTRATION_OF_B)) ) );
UNITS_BY_NAME.put( "reciprocalDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal day" ).build(), Optional.of("d\u207B\u00B9"), Optional.of("E91"), Optional.of("reciprocalSecond"), Optional.of("1.15741 \u00D7 10\u207B\u2075 s\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal degree Fahrenheit" ).build(), Optional.of("1/\u00B0F"), Optional.of("J26"), Optional.empty(), Optional.of("1.81/K"), new HashSet<>(Arrays.asList(QuantityKinds.TEMPERATURE)) ) );
UNITS_BY_NAME.put( "reciprocalElectronVoltPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalElectronVoltPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal electron volt per cubic metre" ).build(), Optional.of("eV\u207B\u00B9/m\u00B3"), Optional.of("C88"), Optional.of("reciprocalJoulePerCubicMetre"), Optional.of("6.24146 \u00D7 10\u00B9\u2078 J\u207B\u00B9/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_STATES)) ) );
UNITS_BY_NAME.put( "reciprocalHenry", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalHenry" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal henry" ).build(), Optional.of("H\u207B\u00B9"), Optional.of("C89"), Optional.empty(), Optional.of("H\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RELUCTANCE)) ) );
UNITS_BY_NAME.put( "reciprocalHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal hour" ).build(), Optional.of("1/h"), Optional.of("H10"), Optional.of("reciprocalSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal inch" ).build(), Optional.of("1/in"), Optional.of("Q24"), Optional.of("reciprocalMetre"), Optional.of("39.37008 m\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalJoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalJoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal joule" ).build(), Optional.of("1/J"), Optional.of("N91"), Optional.empty(), Optional.of("1/J"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalJoulePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalJoulePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal joule per cubic metre" ).build(), Optional.of("J\u207B\u00B9/m\u00B3"), Optional.of("C90"), Optional.empty(), Optional.of("J\u207B\u00B9/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_STATES)) ) );
UNITS_BY_NAME.put( "reciprocalKelvinOrKelvinToThePowerMinusOne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalKelvinOrKelvinToThePowerMinusOne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal kelvin or kelvin to the power minus one" ).build(), Optional.of("K\u207B\u00B9"), Optional.of("C91"), Optional.empty(), Optional.of("K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RELATIVE_PRESSURE_COEFFICIENT, QuantityKinds.LINEAR_EXPANSION_COEFFICIENT, QuantityKinds.CUBIC_EXPANSION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalKilovoltAmpereReciprocalHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalKilovoltAmpereReciprocalHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal kilovolt - ampere reciprocal hour" ).build(), Optional.of("1/kVAh"), Optional.of("M21"), Optional.empty(), Optional.of("2.777778 \u00D7 10\u207B\u2077 (V \u00D7 A \u00D7 s)\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal litre" ).build(), Optional.of("1/l"), Optional.of("K63"), Optional.of("reciprocalCubicMetre"), Optional.of("10\u00B3 m\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.MOLECULAR_CONCENTRATION_OF_B, QuantityKinds.VOLUMIC_NUMBER_OF_MOLECULES_OR_PARTICLES, QuantityKinds.NUMBER_DENSITY_OF_MOLECULES_OR_PARTICLES)) ) );
UNITS_BY_NAME.put( "reciprocalMegakelvinOrMegakelvinToThePowerMinusOne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalMegakelvinOrMegakelvinToThePowerMinusOne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal megakelvin or megakelvin to the power minus one" ).build(), Optional.of("1/MK"), Optional.of("M20"), Optional.of("reciprocalKelvinOrKelvinToThePowerMinusOne"), Optional.of("10\u207B\u2076 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RELATIVE_PRESSURE_COEFFICIENT, QuantityKinds.CUBIC_EXPANSION_COEFFICIENT, QuantityKinds.LINEAR_EXPANSION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal metre" ).build(), Optional.of("m\u207B\u00B9"), Optional.of("C92"), Optional.empty(), Optional.of("m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.LINEAR_EXTINCTION_COEFFICIENT, QuantityKinds.REPETENCY, QuantityKinds.PROPAGATION_COEFFICIENT, QuantityKinds.VERGENCE, QuantityKinds.RYDBERG_CONSTANT, QuantityKinds.CURVATURE, QuantityKinds.PHASE_COEFFICIENT, QuantityKinds.MACROSCOPIC_CROSS_SECTION, QuantityKinds.ANGULAR_RECIPROCAL_LATTICE_VECTOR, QuantityKinds.LINEAR_IONIZATION_BY_A_PARTICLE, QuantityKinds.VOLUMIC_CROSS_SECTION, QuantityKinds.ANGULAR_WAVE_NUMBER, QuantityKinds.WAVE_NUMBER, QuantityKinds.WAVENUMBER, QuantityKinds.ANGULAR_REPETENCY, QuantityKinds.LINEAR_ATTENUATION_COEFFICIENT, QuantityKinds.LINEAR_ABSORPTION_COEFFICIENT, QuantityKinds.MACROSCOPIC_TOTAL_CROSS_SECTION, QuantityKinds.FUNDAMENTAL_RECIPROCAL_LATTICE_VECTOR, QuantityKinds.LENS_POWER, QuantityKinds.TOTAL_IONIZATION_BY_A_PARTICLE, QuantityKinds.VOLUMIC_TOTAL_CROSS_SECTION, QuantityKinds.ATTENUATION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalMetreSquaredReciprocalSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalMetreSquaredReciprocalSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal metre squared reciprocal second" ).build(), Optional.of("m\u207B\u00B2/s"), Optional.of("B81"), Optional.empty(), Optional.of("m\u207B\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.PARTICAL_FLUX_DENSITY, QuantityKinds.PARTICLE_FLUENCE_RATE, QuantityKinds.CURRENT_DENSITY_OF_PARTICLES, QuantityKinds.NEUTRON_FLUENCE_RATE, QuantityKinds.NEUTRONFLUX_DENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal minute" ).build(), Optional.of("min\u207B\u00B9"), Optional.of("C94"), Optional.of("secondUnitOfTime"), Optional.of("1.666667 \u00D7 10\u207B\u00B2 s"), new HashSet<>(Arrays.asList(QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal mole" ).build(), Optional.of("mol\u207B\u00B9"), Optional.of("C95"), Optional.empty(), Optional.of("mol\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.AVOGADRO_CONSTANT)) ) );
UNITS_BY_NAME.put( "reciprocalMonth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalMonth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal month" ).build(), Optional.of("1/mo"), Optional.of("H11"), Optional.of("reciprocalSecond"), Optional.of("3.80257 \u00D7 10\u207B\u2077 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalPascalOrPascalToThePowerMinusOne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalPascalOrPascalToThePowerMinusOne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal pascal or pascal to the power minus one" ).build(), Optional.of("Pa\u207B\u00B9"), Optional.of("C96"), Optional.empty(), Optional.of("Pa\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ISOTHERMAL_COMPRESSIBILITY, QuantityKinds.ISENTROPIC_COMPRESSIBILITY, QuantityKinds.COMPRESSIBILITY, QuantityKinds.BULK_COMPRESSIBILITY)) ) );
UNITS_BY_NAME.put( "reciprocalPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal psi" ).build(), Optional.of("1/psi"), Optional.of("K93"), Optional.of("reciprocalPascalOrPascalToThePowerMinusOne"), Optional.of("1.450377 \u00D7 10\u207B\u2074 Pa\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.AMOUNT_OF_SUBSTANCE, QuantityKinds.CONCENTRATION_OF_B, QuantityKinds.MASS_CONCENTRATION_OF_B, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalRadian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalRadian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal radian" ).build(), Optional.of("1/rad"), Optional.of("P97"), Optional.empty(), Optional.of("1/rad"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal second" ).build(), Optional.of("s\u207B\u00B9"), Optional.of("C97"), Optional.empty(), Optional.of("s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.DISINTEGRATION_CONSTANT, QuantityKinds.CIRCULAR_FREQUENCY, QuantityKinds.PULSATANCE, QuantityKinds.DECAY_CONSTANT, QuantityKinds.PHOTON_FLUX, QuantityKinds.ANGULAR_FREQUENCY, QuantityKinds.DAMPING_COEFFICIENT, QuantityKinds.LARMOR_ANGULAR_FREQUENCY, QuantityKinds.DEBYE_ANGULAR_FREQUENCY, QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalSecondPerMetreSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalSecondPerMetreSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal second per metre squared" ).build(), Optional.of("s\u207B\u00B9/m\u00B2"), Optional.of("C99"), Optional.empty(), Optional.of("s\u207B\u00B9/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_EXITANCE, QuantityKinds.IRRADIANCE)) ) );
UNITS_BY_NAME.put( "reciprocalSecondPerSteradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalSecondPerSteradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal second per steradian" ).build(), Optional.of("s\u207B\u00B9/sr"), Optional.of("D1"), Optional.empty(), Optional.of("s\u207B\u00B9/sr"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_INTENSITY)) ) );
UNITS_BY_NAME.put( "reciprocalSecondPerSteradianMetreSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalSecondPerSteradianMetreSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal second per steradian metre squared" ).build(), Optional.of("s\u207B\u00B9/(sr\u00B7m\u00B2)"), Optional.of("D2"), Optional.empty(), Optional.of("s\u207B\u00B9/(sr \u00D7 m\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_RADIANCE, QuantityKinds.PHOTON_LUMINANCE)) ) );
UNITS_BY_NAME.put( "reciprocalSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal square inch" ).build(), Optional.of("1/in\u00B2"), Optional.of("P78"), Optional.of("reciprocalSquareMetre"), Optional.of("1.550003 \u00D7 10\u00B3 m\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PARTICLE_FLUENCE)) ) );
UNITS_BY_NAME.put( "reciprocalSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal square metre" ).build(), Optional.of("m\u207B\u00B2"), Optional.of("C93"), Optional.empty(), Optional.of("m\u207B\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.PHOTON_EXPOSURE, QuantityKinds.PARTICLE_FLUENCE)) ) );
UNITS_BY_NAME.put( "reciprocalVolt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalVolt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal volt" ).build(), Optional.of("1/V"), Optional.of("P96"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B3 \u00D7 A"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "reciprocalVoltAmpereReciprocalSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalVoltAmpereReciprocalSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal volt - ampere reciprocal second" ).build(), Optional.of("1/(V\u00B7A\u00B7s)"), Optional.of("M30"), Optional.empty(), Optional.of("(V \u00D7 A \u00D7 s)\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.PERFORMANCE_CHARACTERISTIC, QuantityKinds.COEFFICIENT)) ) );
UNITS_BY_NAME.put( "reciprocalWeek", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalWeek" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal week" ).build(), Optional.of("1/wk"), Optional.of("H85"), Optional.of("reciprocalSecond"), Optional.of("1.647989452868 \u00D7 10\u207B\u2076 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "reciprocalYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "reciprocalYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "reciprocal year" ).build(), Optional.of("1/y"), Optional.of("H09"), Optional.of("reciprocalSecond"), Optional.of("3.16881 \u00D7 10\u207B\u2078 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "rem", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "rem" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rem" ).build(), Optional.of("rem"), Optional.of("D91"), Optional.of("sievert"), Optional.of("10\u207B\u00B2 Sv"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "remPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "remPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rem per second" ).build(), Optional.of("rem/s"), Optional.of("P69"), Optional.of("sievertPerSecond"), Optional.of("10\u207B\u00B2 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "revenueTonMile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "revenueTonMile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "revenue ton mile" ).build(), Optional.empty(), Optional.of("RT"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "revolution", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "revolution" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "revolution" ).build(), Optional.of("rev"), Optional.of("M44"), Optional.of("radian"), Optional.of("6.283185 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "revolutionPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "revolutionPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "revolution per minute" ).build(), Optional.of("r/min"), Optional.of("M46"), Optional.of("radianPerSecond"), Optional.of("0.1047198 rad/s"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_VELOCITY)) ) );
UNITS_BY_NAME.put( "revolutionsPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "revolutionsPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "revolutions per minute" ).build(), Optional.of("r/min"), Optional.of("RPM"), Optional.empty(), Optional.of("1.67 \u00D7 10\u207B\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "revolutionsPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "revolutionsPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "revolutions per second" ).build(), Optional.of("r/s"), Optional.of("RPS"), Optional.empty(), Optional.of("1/s"), new HashSet<>(Arrays.asList(QuantityKinds.ROTATIONAL_FREQUENCY)) ) );
UNITS_BY_NAME.put( "rhe", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "rhe" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rhe" ).build(), Optional.of("rhe"), Optional.of("P88"), Optional.empty(), Optional.of("10 m \u00D7 kg\u207B\u00B9 \u00D7 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "rodUnitOfDistance", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "rodUnitOfDistance" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "rod [unit of distance]" ).build(), Optional.of("rd (US)"), Optional.of("F49"), Optional.of("metre"), Optional.of("5.029210 m"), new HashSet<>(Arrays.asList(QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.BREADTH, QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.HEIGHT, QuantityKinds.LENGTH, QuantityKinds.DISTANCE, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.RADIUS, QuantityKinds.THICKNESS)) ) );
UNITS_BY_NAME.put( "roentgen", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "roentgen" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "roentgen" ).build(), Optional.of("R"), Optional.of("2C"), Optional.of("coulombPerKilogram"), Optional.of("2.58 \u00D7 10\u207B\u2074 C/kg"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE)) ) );
UNITS_BY_NAME.put( "roentgenPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "roentgenPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "roentgen per second" ).build(), Optional.of("R/s"), Optional.of("D6"), Optional.of("coulombPerKilogramSecond"), Optional.of("2.58 \u00D7 10\u207B\u2074 C/(kg \u00D7 s)"), new HashSet<>(Arrays.asList(QuantityKinds.EXPOSURE_RATE)) ) );
UNITS_BY_NAME.put( "room", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "room" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "room" ).build(), Optional.empty(), Optional.of("ROM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "round", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "round" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "round" ).build(), Optional.empty(), Optional.of("D65"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "runFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "runFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "run foot" ).build(), Optional.empty(), Optional.of("E52"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "runningOrOperatingHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "runningOrOperatingHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "running or operating hour" ).build(), Optional.empty(), Optional.of("RH"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "score", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "score" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "score" ).build(), Optional.empty(), Optional.of("SCO"), Optional.empty(), Optional.of("20"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "scruple", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "scruple" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "scruple" ).build(), Optional.empty(), Optional.of("SCR"), Optional.of("gram"), Optional.of("1.295982 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "secondPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "secondPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "second per cubic metre" ).build(), Optional.of("s/m\u00B3"), Optional.of("D93"), Optional.empty(), Optional.of("s/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_DOSE)) ) );
UNITS_BY_NAME.put( "secondPerCubicMetreRadian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "secondPerCubicMetreRadian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "second per cubic metre radian" ).build(), Optional.of("s/(rad\u00B7m\u00B3)"), Optional.of("D94"), Optional.empty(), Optional.of("s/(rad \u00D7 m\u00B3)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CONCENTRATION_OF_VIBRATIONAL_MODES_IN_TERMS_OF_ANGULAR_FREQUENCY)) ) );
UNITS_BY_NAME.put( "secondPerKilogramm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "secondPerKilogramm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "second per kilogramm" ).build(), Optional.of("s/kg"), Optional.of("Q20"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "secondPerRadianCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "secondPerRadianCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "second per radian cubic metre" ).build(), Optional.of("1/(Hz\u00B7rad\u00B7m\u00B3)"), Optional.of("Q22"), Optional.empty(), Optional.of("m\u207B\u00B3 \u00D7 s \u00D7 rad\u207B\u00B9"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "secondUnitOfAngle", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "secondUnitOfAngle" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "second [unit of angle]" ).build(), Optional.of("\""), Optional.of("D62"), Optional.of("radian"), Optional.of("4.848137 \u00D7 10\u207B\u2076 rad"), new HashSet<>(Arrays.asList(QuantityKinds.ANGLE_PLANE)) ) );
UNITS_BY_NAME.put( "secondUnitOfTime", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "secondUnitOfTime" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "second [unit of time]" ).build(), Optional.of("s"), Optional.of("SEC"), Optional.empty(), Optional.of("s"), new HashSet<>(Arrays.asList(QuantityKinds.REACTOR_TIME_CONSTANT, QuantityKinds.TIME_CONSTANT, QuantityKinds.RELAXATION_TIME, QuantityKinds.MEAN_LIFE, QuantityKinds.PERIODIC_TIME, QuantityKinds.PERIOD, QuantityKinds.CARRIER_LIFE_TIME, QuantityKinds.REVERBERATION_TIME, QuantityKinds.HALF_LIFE, QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "segment", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "segment" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "segment" ).build(), Optional.empty(), Optional.of("SG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "serviceUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "serviceUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "service unit" ).build(), Optional.empty(), Optional.of("E48"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "set", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "set" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "set" ).build(), Optional.empty(), Optional.of("SET"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shake", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "shake" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "shake" ).build(), Optional.of("shake"), Optional.of("M56"), Optional.of("secondUnitOfTime"), Optional.of("10\u207B\u2078 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "shannon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "shannon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "shannon" ).build(), Optional.of("Sh"), Optional.of("Q14"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shannonPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "shannonPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "shannon per second" ).build(), Optional.of("Sh/s"), Optional.of("Q17"), Optional.empty(), Optional.of("Sh/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shares", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "shares" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "shares" ).build(), Optional.empty(), Optional.of("E21"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shipment", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "shipment" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "shipment" ).build(), Optional.empty(), Optional.of("SX"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "shot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "shot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "shot" ).build(), Optional.empty(), Optional.of("14"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "siderealYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "siderealYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sidereal year" ).build(), Optional.of("y (sidereal)"), Optional.of("L96"), Optional.of("secondUnitOfTime"), Optional.of("3.155815 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "siemens", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "siemens" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "siemens" ).build(), Optional.of("S"), Optional.of("SIE"), Optional.empty(), Optional.of("A/V"), new HashSet<>(Arrays.asList(QuantityKinds.COMPLEX_ADMITTANCE, QuantityKinds.ADMITTANCE, QuantityKinds.MODULUS_OF_ADMITTANCE, QuantityKinds.CONDUCTANCE_FOR_ALTERNATING_CURRENT, QuantityKinds.CONDUCTANCE_FOR_DIRECT_CURRENT)) ) );
UNITS_BY_NAME.put( "siemensPerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "siemensPerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "siemens per centimetre" ).build(), Optional.of("S/cm"), Optional.of("H43"), Optional.of("siemensPerMetre"), Optional.of("10\u00B2 S/m"), new HashSet<>(Arrays.asList(QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "siemensPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "siemensPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "siemens per metre" ).build(), Optional.of("S/m"), Optional.of("D10"), Optional.empty(), Optional.of("S/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTROLYTIC_CONDUCTIVITY, QuantityKinds.CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "siemensSquareMetrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "siemensSquareMetrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "siemens square metre per mole" ).build(), Optional.of("S\u00B7m\u00B2/mol"), Optional.of("D12"), Optional.empty(), Optional.of("S \u00D7 m\u00B2/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "sievert", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "sievert" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sievert" ).build(), Optional.of("Sv"), Optional.of("D13"), Optional.empty(), Optional.of("m\u00B2/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DOSE_EQUIVALENT)) ) );
UNITS_BY_NAME.put( "sievertPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "sievertPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sievert per hour" ).build(), Optional.of("Sv/h"), Optional.of("P70"), Optional.of("sievertPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u2074 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "sievertPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "sievertPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sievert per minute" ).build(), Optional.of("Sv/min"), Optional.of("P74"), Optional.of("sievertPerSecond"), Optional.of("0.016666 Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "sievertPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "sievertPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sievert per second" ).build(), Optional.of("Sv/s"), Optional.of("P65"), Optional.empty(), Optional.of("Sv/s"), new HashSet<>(Arrays.asList(QuantityKinds.EQUIVALENCE_DOSE_OUTPUT)) ) );
UNITS_BY_NAME.put( "sitas", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "sitas" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sitas" ).build(), Optional.empty(), Optional.of("56"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "skein", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "skein" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "skein" ).build(), Optional.empty(), Optional.of("SW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "slug", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slug" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug" ).build(), Optional.of("slug"), Optional.of("F13"), Optional.of("kilogram"), Optional.of("1.459390 \u00D7 10\u00B9 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "slugPerCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slugPerCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug per cubic foot" ).build(), Optional.of("slug/ft\u00B3"), Optional.of("L65"), Optional.of("kilogramPerCubicMetre"), Optional.of("5.153788 \u00D7 10\u00B2 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "slugPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slugPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug per day" ).build(), Optional.of("slug/d"), Optional.of("L63"), Optional.of("kilogramPerSecond"), Optional.of("1.689109 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "slugPerFootSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slugPerFootSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug per foot second" ).build(), Optional.of("slug/(ft\u00B7s)"), Optional.of("L64"), Optional.of("pascalSecond"), Optional.of("47.88026 Pa \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.VISCOSITY_DYNAMIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "slugPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slugPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug per hour" ).build(), Optional.of("slug/h"), Optional.of("L66"), Optional.of("kilogramPerSecond"), Optional.of("4.053861 \u00D7 10\u207B\u00B3 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
}

private static void init15() {
UNITS_BY_NAME.put( "slugPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slugPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug per minute" ).build(), Optional.of("slug/min"), Optional.of("L67"), Optional.of("kilogramPerSecond"), Optional.of("0.2432317 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "slugPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "slugPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "slug per second" ).build(), Optional.of("slug/s"), Optional.of("L68"), Optional.of("kilogramPerSecond"), Optional.of("14.59390 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "sone", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "sone" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "sone" ).build(), Optional.empty(), Optional.of("D15"), Optional.empty(), Optional.empty(), new HashSet<>(Arrays.asList(QuantityKinds.LOUDNESS)) ) );
UNITS_BY_NAME.put( "square", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "square" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square" ).build(), Optional.empty(), Optional.of("SQ"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "squareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square centimetre" ).build(), Optional.of("cm\u00B2"), Optional.of("CMK"), Optional.of("squareMetre"), Optional.of("10\u207B\u2074 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerErg", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareCentimetrePerErg" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square centimetre per erg" ).build(), Optional.of("cm\u00B2/erg"), Optional.of("D16"), Optional.of("squareMetrePerJoule"), Optional.of("10\u00B3 m\u00B2/J"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerGram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareCentimetrePerGram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square centimetre per gram" ).build(), Optional.of("cm\u00B2/g"), Optional.of("H15"), Optional.empty(), Optional.of("10\u207B\u00B9 kg\u207B\u00B9 \u00D7 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SPECIFIC_VOLUME, QuantityKinds.MASSIC_VOLUME)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareCentimetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square centimetre per second" ).build(), Optional.of("cm\u00B2/s"), Optional.of("M81"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2074 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareCentimetrePerSteradianErg", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareCentimetrePerSteradianErg" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square centimetre per steradian erg" ).build(), Optional.of("cm\u00B2/(sr\u00B7erg)"), Optional.of("D17"), Optional.of("squareMetrePerSteradianJoule"), Optional.of("10\u00B3 m\u00B2/(sr \u00D7 J)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareDecametre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareDecametre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square decametre" ).build(), Optional.of("dam\u00B2"), Optional.of("H16"), Optional.of("squareMetre"), Optional.of("10\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareDecimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareDecimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square decimetre" ).build(), Optional.of("dm\u00B2"), Optional.of("DMK"), Optional.of("squareMetre"), Optional.of("10\u207B\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square foot" ).build(), Optional.of("ft\u00B2"), Optional.of("FTK"), Optional.of("squareMetre"), Optional.of("9.290304 \u00D7 10\u207B\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareFootPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareFootPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square foot per hour" ).build(), Optional.of("ft\u00B2/h"), Optional.of("M79"), Optional.of("squareMetrePerSecond"), Optional.of("2.58064 \u00D7 10\u207B\u2075 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareFootPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareFootPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square foot per second" ).build(), Optional.of("ft\u00B2/s"), Optional.of("S3"), Optional.of("squareMetrePerSecond"), Optional.of("0.09290304 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSIVITY, QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareHectometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareHectometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square hectometre" ).build(), Optional.of("hm\u00B2"), Optional.of("H18"), Optional.of("squareMetre"), Optional.of("10\u2074 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square inch" ).build(), Optional.of("in\u00B2"), Optional.of("INK"), Optional.of("squareMetre"), Optional.of("6.4516 \u00D7 10\u207B\u2074 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareInchPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareInchPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square inch per second" ).build(), Optional.of("in\u00B2/s"), Optional.of("G08"), Optional.empty(), Optional.of("6.4516 \u00D7 10\u207B\u2074 m\u00B2 \u00D7 s\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square kilometre" ).build(), Optional.of("km\u00B2"), Optional.of("KMK"), Optional.of("squareMetre"), Optional.of("10\u2076 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre" ).build(), Optional.of("m\u00B2"), Optional.of("MTK"), Optional.empty(), Optional.of("m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.MIGRATION_AREA, QuantityKinds.CROSS_SECTION, QuantityKinds.EQUIVALENT_ABSORPTION_AREA_OF_A_SURFACE_OR_OBJECT, QuantityKinds.DIFFUSION_AREA, QuantityKinds.NUCLEAR_QUADRUPOLE_MOMENT, QuantityKinds.ATOMIC_ATTENUATION_COEFFICIENT, QuantityKinds.AREA, QuantityKinds.SLOWING_DOWN_AREA)) ) );
UNITS_BY_NAME.put( "squareMetreHourDegreeCelsiusPerKilocalorieInternationalTable", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetreHourDegreeCelsiusPerKilocalorieInternationalTable" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre hour degree Celsius per kilocalorie (international table)" ).build(), Optional.of("m\u00B2\u00B7h\u00B7\u00B0C/kcal"), Optional.of("L14"), Optional.empty(), Optional.of("0.8598452 m\u00B2 \u00D7 s \u00D7 K/J"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "squareMetreKelvinPerWatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetreKelvinPerWatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre kelvin per watt" ).build(), Optional.of("m\u00B2\u00B7K/W"), Optional.of("D19"), Optional.empty(), Optional.of("m\u00B2 \u00D7 K/W"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_INSULANCE, QuantityKinds.COEFFICIENT_OF_THERMAL_INSULATION)) ) );
UNITS_BY_NAME.put( "squareMetrePerJoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerJoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per joule" ).build(), Optional.of("m\u00B2/J"), Optional.of("D20"), Optional.empty(), Optional.of("m\u00B2/J"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareMetrePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per kilogram" ).build(), Optional.of("m\u00B2/kg"), Optional.of("D21"), Optional.empty(), Optional.of("m\u00B2/kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_ENERGY_TRANSFER_COEFFICIENT, QuantityKinds.MASS_ATTENUATION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "squareMetrePerLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per litre" ).build(), Optional.of("m\u00B2/l"), Optional.of("E31"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "squareMetrePerMole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerMole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per mole" ).build(), Optional.of("m\u00B2/mol"), Optional.of("D22"), Optional.empty(), Optional.of("m\u00B2/mol"), new HashSet<>(Arrays.asList(QuantityKinds.MOLAR_ATTENUATION_COEFFICIENT, QuantityKinds.MOLAR_ABSORPTION_COEFFICIENT)) ) );
UNITS_BY_NAME.put( "squareMetrePerNewton", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerNewton" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per newton" ).build(), Optional.of("m\u00B2/N"), Optional.of("H59"), Optional.empty(), Optional.of("m \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per second" ).build(), Optional.of("m\u00B2/s"), Optional.of("S4"), Optional.empty(), Optional.of("m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_DIFFUSION_COEFFICIENT, QuantityKinds.KINEMATIC_VISCOSITY, QuantityKinds.DIFFUSION_COEFFICIENT_FOR_NEUTRON_NUMBER_DENSITY, QuantityKinds.DIFFUSION_COEFFICIENT, QuantityKinds.THERMAL_DIFFUSIVITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per second bar" ).build(), Optional.of("m\u00B2/(s\u00B7bar)"), Optional.of("G41"), Optional.empty(), Optional.of("10\u207B\u2075 kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per second kelvin" ).build(), Optional.of("m\u00B2/(s\u00B7K)"), Optional.of("G09"), Optional.empty(), Optional.of("m\u00B2 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSecondPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerSecondPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per second pascal" ).build(), Optional.of("(m\u00B2/s)/Pa"), Optional.of("M82"), Optional.empty(), Optional.of("kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "squareMetrePerSteradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerSteradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per steradian" ).build(), Optional.of("m\u00B2/sr"), Optional.of("D24"), Optional.empty(), Optional.of("m\u00B2/sr"), new HashSet<>(Arrays.asList(QuantityKinds.ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareMetrePerSteradianJoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerSteradianJoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per steradian joule" ).build(), Optional.of("m\u00B2/(sr\u00B7J)"), Optional.of("D25"), Optional.empty(), Optional.of("m\u00B2/(sr \u00D7 J)"), new HashSet<>(Arrays.asList(QuantityKinds.SPECTRAL_ANGULAR_CROSS_SECTION)) ) );
UNITS_BY_NAME.put( "squareMetrePerVoltSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMetrePerVoltSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square metre per volt second" ).build(), Optional.of("m\u00B2/(V\u00B7s)"), Optional.of("D26"), Optional.empty(), Optional.of("m\u00B2/(V \u00D7 s)"), new HashSet<>(Arrays.asList(QuantityKinds.MOBILITY)) ) );
UNITS_BY_NAME.put( "squareMicrometreSquareMicron", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMicrometreSquareMicron" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square micrometre (square micron)" ).build(), Optional.of("\u00B5m\u00B2"), Optional.of("H30"), Optional.of("squareMetre"), Optional.of("10\u207B\u00B9\u00B2 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMileBasedOnUsSurveyFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMileBasedOnUsSurveyFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square mile (based on U.S. survey foot)" ).build(), Optional.of("mi\u00B2 (US survey)"), Optional.of("M48"), Optional.of("squareMetre"), Optional.of("2.589998 \u00D7 10\u2076 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMileStatuteMile", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMileStatuteMile" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square mile (statute mile)" ).build(), Optional.of("mi\u00B2"), Optional.of("MIK"), Optional.of("squareKilometre"), Optional.of("2.589988 km\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square millimetre" ).build(), Optional.of("mm\u00B2"), Optional.of("MMK"), Optional.of("squareMetre"), Optional.of("10\u207B\u2076 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "squareRoofing", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareRoofing" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square, roofing" ).build(), Optional.empty(), Optional.of("SQR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "squareYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "squareYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "square yard" ).build(), Optional.of("yd\u00B2"), Optional.of("YDK"), Optional.of("squareMetre"), Optional.of("8.361274 \u00D7 10\u207B\u00B9 m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.AREA)) ) );
UNITS_BY_NAME.put( "standard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "standard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "standard" ).build(), Optional.of("std"), Optional.of("WSD"), Optional.of("cubicMetre"), Optional.of("4.672 m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "standardAccelerationOfFreeFall", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "standardAccelerationOfFreeFall" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "standard acceleration of free fall" ).build(), Optional.of("gn"), Optional.of("K40"), Optional.of("metrePerSecondSquared"), Optional.of("9.80665 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY)) ) );
UNITS_BY_NAME.put( "standardAtmosphere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "standardAtmosphere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "standard atmosphere" ).build(), Optional.of("atm"), Optional.of("ATM"), Optional.of("pascal"), Optional.of("101325 Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MODULUS_OF_ELASTICITY, QuantityKinds.SHEAR_STRESS, QuantityKinds.MODULUS_OF_RIGIDITY, QuantityKinds.PRESSURE, QuantityKinds.NORMAL_STRESS, QuantityKinds.MODULUS_OF_COMPRESSION, QuantityKinds.SHEAR_MODULUS, QuantityKinds.BULK_MODULUS)) ) );
UNITS_BY_NAME.put( "standardAtmospherePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "standardAtmospherePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "standard atmosphere per metre" ).build(), Optional.of("Atm/m"), Optional.of("P83"), Optional.empty(), Optional.of("1.01325 \u00D7 10\u2075 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "standardKilolitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "standardKilolitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "standard kilolitre" ).build(), Optional.empty(), Optional.of("DMO"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "standardLitre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "standardLitre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "standard litre" ).build(), Optional.empty(), Optional.of("STL"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "steradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "steradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "steradian" ).build(), Optional.of("sr"), Optional.of("D27"), Optional.empty(), Optional.of("sr"), new HashSet<>(Arrays.asList(QuantityKinds.SOLID_ANGLE)) ) );
UNITS_BY_NAME.put( "stere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stere" ).build(), Optional.of("st"), Optional.of("G26"), Optional.of("cubicMetre"), Optional.of("m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "stick", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stick" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stick" ).build(), Optional.empty(), Optional.of("STC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "stickCigarette", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stickCigarette" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stick, cigarette" ).build(), Optional.empty(), Optional.of("STK"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "stickMilitary", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stickMilitary" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stick, military" ).build(), Optional.empty(), Optional.of("15"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "stilb", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stilb" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stilb" ).build(), Optional.of("sb"), Optional.of("P31"), Optional.of("candelaPerSquareMetre"), Optional.of("10\u2074 cd/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.LUMINANCE)) ) );
UNITS_BY_NAME.put( "stokes", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stokes" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stokes" ).build(), Optional.of("St"), Optional.of("91"), Optional.of("squareMetrePerSecond"), Optional.of("10\u207B\u2074 m\u00B2/s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stokesPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stokesPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stokes per bar" ).build(), Optional.of("St/bar"), Optional.of("G46"), Optional.empty(), Optional.of("10\u207B\u2079 kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stokesPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stokesPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stokes per kelvin" ).build(), Optional.of("St/K"), Optional.of("G10"), Optional.empty(), Optional.of("10\u207B\u2074 m\u00B2 \u00D7 s\u207B\u00B9 \u00D7 K\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stokesPerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stokesPerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stokes per pascal" ).build(), Optional.of("St/Pa"), Optional.of("M80"), Optional.empty(), Optional.of("10\u207B\u2074 kg\u207B\u00B9 \u00D7 m\u00B3 \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.KINEMATIC_VISCOSITY)) ) );
UNITS_BY_NAME.put( "stoneUk", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "stoneUk" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "stone (UK)" ).build(), Optional.of("st"), Optional.of("STI"), Optional.of("kilogram"), Optional.of("6.350293 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "strand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "strand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "strand" ).build(), Optional.empty(), Optional.of("E30"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "straw", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "straw" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "straw" ).build(), Optional.empty(), Optional.of("STW"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "strip", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "strip" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "strip" ).build(), Optional.empty(), Optional.of("SR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "syringe", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "syringe" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "syringe" ).build(), Optional.empty(), Optional.of("SYR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tablespoonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tablespoonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tablespoon (US)" ).build(), Optional.of("tablespoon (US)"), Optional.of("G24"), Optional.of("cubicMetre"), Optional.of("1.478676 \u00D7 10\u207B\u2075 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tablet", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tablet" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tablet" ).build(), Optional.empty(), Optional.of("U2"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "teaspoonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "teaspoonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "teaspoon (US)" ).build(), Optional.of("teaspoon (US)"), Optional.of("G25"), Optional.of("cubicMetre"), Optional.of("4.928922 \u00D7 10\u207B\u2076 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tebibitPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tebibitPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tebibit per cubic metre" ).build(), Optional.of("Tibit/m\u00B3"), Optional.of("E86"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tebibitPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tebibitPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tebibit per metre" ).build(), Optional.of("Tibit/m"), Optional.of("E85"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tebibitPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tebibitPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tebibit per square metre" ).build(), Optional.of("Tibit/m\u00B2"), Optional.of("E87"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tebibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tebibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Tebibyte" ).build(), Optional.of("TiB"), Optional.of("E61"), Optional.of("byte"), Optional.of("2\u2074\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "technicalAtmospherePerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "technicalAtmospherePerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "technical atmosphere per metre" ).build(), Optional.of("at/m"), Optional.of("P84"), Optional.empty(), Optional.of("9.80665 \u00D7 10\u2074 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "telecommunicationLineInService", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "telecommunicationLineInService" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "telecommunication line in service" ).build(), Optional.empty(), Optional.of("T0"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "telecommunicationLineInServiceAverage", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "telecommunicationLineInServiceAverage" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "telecommunication line in service average" ).build(), Optional.empty(), Optional.of("UB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "telecommunicationPort", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "telecommunicationPort" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "telecommunication port" ).build(), Optional.empty(), Optional.of("UC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tenDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ten day" ).build(), Optional.empty(), Optional.of("DAD"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenPack", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tenPack" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ten pack" ).build(), Optional.empty(), Optional.of("TP"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenPair", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tenPair" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ten pair" ).build(), Optional.empty(), Optional.of("TPR"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenSet", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tenSet" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ten set" ).build(), Optional.empty(), Optional.of("TST"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tenThousandSticks", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tenThousandSticks" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ten thousand sticks" ).build(), Optional.empty(), Optional.of("TTS"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "terabit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terabit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "terabit" ).build(), Optional.of("Tbit"), Optional.of("E83"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "terabitPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terabitPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "terabit per second" ).build(), Optional.of("Tbit/s"), Optional.of("E84"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "terabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Terabyte" ).build(), Optional.of("TB"), Optional.of("E35"), Optional.of("byte"), Optional.of("10\u00B9\u00B2 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "terahertz", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terahertz" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "terahertz" ).build(), Optional.of("THz"), Optional.of("D29"), Optional.of("hertz"), Optional.of("10\u00B9\u00B2 Hz"), new HashSet<>(Arrays.asList(QuantityKinds.FREQUENCY)) ) );
UNITS_BY_NAME.put( "terajoule", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terajoule" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "terajoule" ).build(), Optional.of("TJ"), Optional.of("D30"), Optional.of("joule"), Optional.of("10\u00B9\u00B2 J"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY)) ) );
UNITS_BY_NAME.put( "teraohm", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "teraohm" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "teraohm" ).build(), Optional.of("T\u2126"), Optional.of("H44"), Optional.of("ohm"), Optional.of("10\u00B9\u00B2 \u2126"), new HashSet<>(Arrays.asList(QuantityKinds.RESISTANCE_TO_ALTERNATING_CURRENT, QuantityKinds.MODULUS_OF_IMPEDANCE, QuantityKinds.REACTANCE, QuantityKinds.RESISTANCE_TO_DIRECT_CURRENT, QuantityKinds.IMPEDANCE, QuantityKinds.COMPLEX_IMPEDANCES)) ) );
UNITS_BY_NAME.put( "terawatt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terawatt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "terawatt" ).build(), Optional.of("TW"), Optional.of("D31"), Optional.of("watt"), Optional.of("10\u00B9\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "terawattHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "terawattHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "terawatt hour" ).build(), Optional.of("TW\u00B7h"), Optional.of("D32"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u00B9\u2075 J"), new HashSet<>(Arrays.asList(QuantityKinds.KINETIC_ENERGY, QuantityKinds.ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "tesla", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tesla" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tesla" ).build(), Optional.of("T"), Optional.of("D33"), Optional.empty(), Optional.of("T"), new HashSet<>(Arrays.asList(QuantityKinds.THERMODYNAMIC_CRITICAL_MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_POLARIZATION, QuantityKinds.UPPER_CRITICAL_MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_FLUX_DENSITY, QuantityKinds.LOWER_CRITICAL_MAGNETIC_FLUX_DENSITY, QuantityKinds.MAGNETIC_INDUCTION)) ) );
UNITS_BY_NAME.put( "test", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "test" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "test" ).build(), Optional.empty(), Optional.of("E53"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "teu", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "teu" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "TEU" ).build(), Optional.empty(), Optional.of("E22"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tex", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tex" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tex" ).build(), Optional.of("tex (g/km)"), Optional.of("D34"), Optional.of("kilogramPerMetre"), Optional.of("10\u207B\u2076 kg/m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "theoreticalPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "theoreticalPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "theoretical pound" ).build(), Optional.empty(), Optional.of("24"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "theoreticalTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "theoreticalTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "theoretical ton" ).build(), Optional.empty(), Optional.of("27"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thermEc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thermEc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "therm (EC)" ).build(), Optional.of("thm (EC)"), Optional.of("N71"), Optional.of("joule"), Optional.of("1.05506 \u00D7 10\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY)) ) );
UNITS_BY_NAME.put( "thermUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thermUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "therm (U.S.)" ).build(), Optional.of("thm (US)"), Optional.of("N72"), Optional.of("joule"), Optional.of("1.054804 \u00D7 10\u2078 J"), new HashSet<>(Arrays.asList(QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.HEAT, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FUNCTION)) ) );
UNITS_BY_NAME.put( "thirtyDayMonth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thirtyDayMonth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "30-day month" ).build(), Optional.of("mo (30 days)"), Optional.of("M36"), Optional.of("secondUnitOfTime"), Optional.of("2.592000 \u00D7 10\u2076 s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousand", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousand" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand" ).build(), Optional.empty(), Optional.of("MIL"), Optional.empty(), Optional.of("10\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandBoardFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandBoardFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand board foot" ).build(), Optional.empty(), Optional.of("MBF"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandCubicFoot", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandCubicFoot" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand cubic foot" ).build(), Optional.of("kft\u00B3"), Optional.of("FC"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand cubic metre" ).build(), Optional.empty(), Optional.of("R9"), Optional.of("cubicMetre"), Optional.of("10\u00B3m\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandCubicMetrePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandCubicMetrePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand cubic metre per day" ).build(), Optional.of("km\u00B3/d"), Optional.of("TQD"), Optional.of("cubicMetrePerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u00B2 m\u00B3/s"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandPiece", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandPiece" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand piece" ).build(), Optional.empty(), Optional.of("T3"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "thousandSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand square inch" ).build(), Optional.empty(), Optional.of("TI"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}

private static void init16() {
UNITS_BY_NAME.put( "thousandStandardBrickEquivalent", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "thousandStandardBrickEquivalent" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "thousand standard brick equivalent" ).build(), Optional.empty(), Optional.of("MBE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tonAssay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonAssay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton, assay" ).build(), Optional.empty(), Optional.of("M85"), Optional.of("kilogram"), Optional.of("2.916667 \u00D7 10\u207B\u00B2 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonForceUsShort", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonForceUsShort" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton-force (US short)" ).build(), Optional.of("ton.sh-force"), Optional.of("L94"), Optional.of("newton"), Optional.of("8.896443 \u00D7 10\u00B3 N"), new HashSet<>(Arrays.asList(QuantityKinds.WEIGHT, QuantityKinds.FORCE)) ) );
UNITS_BY_NAME.put( "tonLongPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonLongPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton long per day" ).build(), Optional.of("ton (UK)/d"), Optional.of("L85"), Optional.of("kilogramPerSecond"), Optional.of("1.175980 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonRegister", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonRegister" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton, register" ).build(), Optional.of("RT"), Optional.of("M70"), Optional.of("cubicMetre"), Optional.of("2.831685 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tonShortPerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonShortPerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton short per day" ).build(), Optional.of("ton (US)/d"), Optional.of("L88"), Optional.of("kilogramPerSecond"), Optional.of("1.049982 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonShortPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonShortPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton short per degree Fahrenheit" ).build(), Optional.of("ton (US)/\u00B0F"), Optional.of("L87"), Optional.of("kilogramPerKelvin"), Optional.of("1.632932 \u00D7 10\u00B3 kg/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonShortPerHourDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonShortPerHourDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton short per hour degree Fahrenheit" ).build(), Optional.of("ton (US)/(h\u00B7\u00B0F)"), Optional.of("L89"), Optional.empty(), Optional.of("0.4535922 kg/s \u00D7 K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonShortPerHourPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonShortPerHourPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton short per hour psi" ).build(), Optional.of("(ton (US)/h)/psi"), Optional.of("L90"), Optional.of("kilogramPerSecondPascal"), Optional.of("3.654889 \u00D7 10\u207B\u2075 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonShortPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonShortPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton short per psi" ).build(), Optional.of("ton (US)/psi"), Optional.of("L91"), Optional.empty(), Optional.of("0.131576"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "tonUkLongPerCubicYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUkLongPerCubicYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (UK long) per cubic yard" ).build(), Optional.of("ton.l/yd\u00B3 (UK)"), Optional.of("L92"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.328939 \u00D7 10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "tonUkOrLongTonUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUkOrLongTonUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (UK) or long ton (US)" ).build(), Optional.of("ton (UK)"), Optional.of("LTN"), Optional.of("kilogram"), Optional.of("1.016047 \u00D7 10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonUkShipping", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUkShipping" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (UK shipping)" ).build(), Optional.of("British shipping ton"), Optional.of("L84"), Optional.of("cubicMetre"), Optional.of("1.1893 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tonUsOrShortTonUkorus", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUsOrShortTonUkorus" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (US) or short ton (UK/US)" ).build(), Optional.of("ton (US)"), Optional.of("STN"), Optional.of("kilogram"), Optional.of("0.9071847 \u00D7 10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonUsPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUsPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (US) per hour" ).build(), Optional.of("ton (US) /h"), Optional.of("4W"), Optional.of("kilogramPerSecond"), Optional.of("2.519958 \u00D7 10\u207B\u00B9 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonUsShipping", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUsShipping" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (US shipping)" ).build(), Optional.of("(US) shipping ton"), Optional.of("L86"), Optional.of("cubicMetre"), Optional.of("1.1326 m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME)) ) );
UNITS_BY_NAME.put( "tonUsShortPerCubicYard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonUsShortPerCubicYard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "ton (US short) per cubic yard" ).build(), Optional.of("ton.s/yd\u00B3 (US)"), Optional.of("L93"), Optional.of("kilogramPerCubicMetre"), Optional.of("1.186553 \u00D7 10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "tonneKilometre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonneKilometre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne kilometre" ).build(), Optional.of("t\u00B7km"), Optional.of("TKM"), Optional.of("kilogramMetre"), Optional.of("10\u2076 kg \u00D7 m"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tonneMetricTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonneMetricTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne (metric ton)" ).build(), Optional.of("t"), Optional.of("TNE"), Optional.of("kilogram"), Optional.of("10\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonnePerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per bar" ).build(), Optional.of("t/bar"), Optional.of("L70"), Optional.of("kilogramPerPascal"), Optional.of("10\u207B\u00B2 kg/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "tonnePerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per cubic metre" ).build(), Optional.of("t/m\u00B3"), Optional.of("D41"), Optional.of("kilogramPerCubicMetre"), Optional.of("10\u00B3 kg/m\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY)) ) );
UNITS_BY_NAME.put( "tonnePerCubicMetreBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerCubicMetreBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per cubic metre bar" ).build(), Optional.of("(t/m\u00B3)/bar"), Optional.of("L77"), Optional.of("kilogramPerCubicMetrePascal"), Optional.of("10\u207B\u00B2 (kg/m\u00B3)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY, QuantityKinds.MASS_DENSITY, QuantityKinds.VOLUMIC_MASS)) ) );
UNITS_BY_NAME.put( "tonnePerCubicMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerCubicMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per cubic metre kelvin" ).build(), Optional.of("(t/m\u00B3)/K"), Optional.of("L76"), Optional.empty(), Optional.of("10\u00B3 (kg/m\u00B3)/K"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUMIC_MASS, QuantityKinds.MASS_DENSITY, QuantityKinds.DENSITY)) ) );
UNITS_BY_NAME.put( "tonnePerDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per day" ).build(), Optional.of("t/d"), Optional.of("L71"), Optional.of("kilogramPerSecond"), Optional.of("1.15741 \u00D7 10\u207B\u00B2 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerDayBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerDayBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per day bar" ).build(), Optional.of("(t/d)/bar"), Optional.of("L73"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.15741 \u00D7 10\u207B\u2077 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerDayKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerDayKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per day kelvin" ).build(), Optional.of("(t/d)/K"), Optional.of("L72"), Optional.empty(), Optional.of("1.15741 \u00D7 10\u207B\u00B2 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per hour" ).build(), Optional.of("t/h"), Optional.of("E18"), Optional.of("kilogramPerSecond"), Optional.of("2.77778 \u00D7 10\u207B\u00B9 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerHourBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerHourBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per hour bar" ).build(), Optional.of("(t/h)/bar"), Optional.of("L75"), Optional.of("kilogramPerSecondPascal"), Optional.of("2.77778 \u00D7 10\u207B\u2076 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerHourKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerHourKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per hour kelvin" ).build(), Optional.of("(t/h)/K"), Optional.of("L74"), Optional.empty(), Optional.of("2.77778 \u00D7 10\u207B\u00B9 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per kelvin" ).build(), Optional.of("t/K"), Optional.of("L69"), Optional.of("kilogramPerKelvin"), Optional.of("10\u00B3 kg/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "tonnePerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per minute" ).build(), Optional.of("t/min"), Optional.of("L78"), Optional.of("kilogramPerSecond"), Optional.of("16.6667 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerMinuteBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerMinuteBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per minute bar" ).build(), Optional.of("(t/min)/bar"), Optional.of("L80"), Optional.of("kilogramPerSecondPascal"), Optional.of("1.66667 \u00D7 10\u207B\u2074 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerMinuteKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerMinuteKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per minute kelvin" ).build(), Optional.of("(t/min)/K"), Optional.of("L79"), Optional.empty(), Optional.of("16.6667 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerMonth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerMonth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per month" ).build(), Optional.of("t/mo"), Optional.of("M88"), Optional.of("kilogramPerSecond"), Optional.of("3.80257053768 \u00D7 10\u207B\u2074 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per second" ).build(), Optional.of("t/s"), Optional.of("L81"), Optional.of("kilogramPerSecond"), Optional.of("10\u00B3 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerSecondBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerSecondBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per second bar" ).build(), Optional.of("(t/s)/bar"), Optional.of("L83"), Optional.of("kilogramPerSecondPascal"), Optional.of("10\u207B\u00B2 (kg/s)/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerSecondKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerSecondKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per second kelvin" ).build(), Optional.of("(t/s)/K"), Optional.of("L82"), Optional.empty(), Optional.of("10\u00B3 (kg/s)/K"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "tonnePerYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tonnePerYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tonne per year" ).build(), Optional.of("t/y"), Optional.of("M89"), Optional.of("kilogramPerSecond"), Optional.of("3.168808781 \u00D7 10\u207B\u2075 kg/s"), new HashSet<>(Arrays.asList(QuantityKinds.MASS_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "torrPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "torrPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "torr per metre" ).build(), Optional.of("Torr/m"), Optional.of("P85"), Optional.empty(), Optional.of("1.333224 \u00D7 10\u00B2 kg/(m\u00B2 \u00D7 s\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.HARDNESS_INDEX)) ) );
UNITS_BY_NAME.put( "totalAcidNumber", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "totalAcidNumber" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "total acid number" ).build(), Optional.of("TAN"), Optional.of("TAN"), Optional.empty(), Optional.of("mg KOH/g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "treatment", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "treatment" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "treatment" ).build(), Optional.empty(), Optional.of("U1"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "trillionEur", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "trillionEur" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "trillion (EUR)" ).build(), Optional.empty(), Optional.of("TRL"), Optional.empty(), Optional.of("10\u00B9\u2078"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "trip", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "trip" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "trip" ).build(), Optional.empty(), Optional.of("E54"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tropicalYear", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tropicalYear" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tropical year" ).build(), Optional.of("y (tropical)"), Optional.of("D42"), Optional.of("secondUnitOfTime"), Optional.of("3.155692 5 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "troyOunceOrApothecaryOunce", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "troyOunceOrApothecaryOunce" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "troy ounce or apothecary ounce" ).build(), Optional.of("tr oz"), Optional.of("APZ"), Optional.of("kilogram"), Optional.of("3.110348 \u00D7 10\u207B\u00B3 kg"), new HashSet<>(Arrays.asList(QuantityKinds.MASS)) ) );
UNITS_BY_NAME.put( "troyPoundUs", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "troyPoundUs" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "troy pound (US)" ).build(), Optional.empty(), Optional.of("LBT"), Optional.of("gram"), Optional.of("373.2417 g"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "twentyFootContainer", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "twentyFootContainer" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "twenty foot container" ).build(), Optional.empty(), Optional.of("20"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "tyre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "tyre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "tyre" ).build(), Optional.empty(), Optional.of("E23"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "unifiedAtomicMassUnit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "unifiedAtomicMassUnit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "unified atomic mass unit" ).build(), Optional.of("u"), Optional.of("D43"), Optional.of("kilogram"), Optional.of("1.660 538 782 \u00D7 10\u207B\u00B2\u2077 kg"), new HashSet<>(Arrays.asList(QuantityKinds.UNIFIED_ATOMIC_MASS_CONSTANT, QuantityKinds.MASS_OF_MOLECULE, QuantityKinds.MASS_DEFECT, QuantityKinds.NUCLIDIC_MASS, QuantityKinds.MASS_OF_ATOM_OF_A_NUCLIDE_X)) ) );
UNITS_BY_NAME.put( "unitPole", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "unitPole" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "unit pole" ).build(), Optional.of("unit pole"), Optional.of("P53"), Optional.of("weber"), Optional.of("1.256637 \u00D7 10\u207B\u2077 Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_QUANTUM)) ) );
UNITS_BY_NAME.put( "usGallonPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "usGallonPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "US gallon per minute" ).build(), Optional.of("gal (US) /min"), Optional.of("G2"), Optional.of("cubicMetrePerSecond"), Optional.of("6.309020 \u00D7 10\u207B\u2075 m\u00B3/s"), new HashSet<>(Arrays.asList(QuantityKinds.VOLUME_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "use", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "use" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "use" ).build(), Optional.empty(), Optional.of("E55"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "var", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "var" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "var" ).build(), Optional.of("var"), Optional.of("D44"), Optional.of("voltAmpere"), Optional.of("V \u00D7 A"), new HashSet<>(Arrays.asList(QuantityKinds.REACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "volt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "volt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt" ).build(), Optional.of("V"), Optional.of("VLT"), Optional.empty(), Optional.of("V"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_POTENTIAL, QuantityKinds.ELECTROMOTIVE_FORCE, QuantityKinds.POTENTIAL_DIFFERENCE, QuantityKinds.VOLTAGE, QuantityKinds.THERMOELECTROMOTIVE_FORCE_BETWEEN_SUBSTANCES_A_AND_B, QuantityKinds.TENSION, QuantityKinds.PELTIER_COEFFICIENT_FOR_SUBSTANCES_A_AND_B)) ) );
UNITS_BY_NAME.put( "voltAc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltAc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt AC" ).build(), Optional.of("V"), Optional.of("2G"), Optional.empty(), Optional.of("V"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "voltAmpere", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltAmpere" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt - ampere" ).build(), Optional.of("V\u00B7A"), Optional.of("D46"), Optional.of("watt"), Optional.of("W"), new HashSet<>(Arrays.asList(QuantityKinds.APPARENT_POWER)) ) );
UNITS_BY_NAME.put( "voltAmperePerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltAmperePerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt - ampere per kilogram" ).build(), Optional.of("V\u00B7A / kg"), Optional.of("VA"), Optional.empty(), Optional.of("V \u00D7 A / kg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "voltDc", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltDc" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt DC" ).build(), Optional.of("V"), Optional.of("2H"), Optional.empty(), Optional.of("V"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "voltPerBar", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerBar" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per bar" ).build(), Optional.of("V/bar"), Optional.of("G60"), Optional.empty(), Optional.of("10\u207B\u2075 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per centimetre" ).build(), Optional.of("V/cm"), Optional.of("D47"), Optional.empty(), Optional.of("V/m \u00D7 10\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per inch" ).build(), Optional.of("V/in"), Optional.of("H23"), Optional.empty(), Optional.of("3.937007874 \u00D7 10\u00B9 m \u00D7 kg \u00D7 s\u207B\u00B3 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per kelvin" ).build(), Optional.of("V/K"), Optional.of("D48"), Optional.empty(), Optional.of("V/K"), new HashSet<>(Arrays.asList(QuantityKinds.SEEBECK_COEFFICIENT_FOR_SUBSTANCES_A_AND_B)) ) );
UNITS_BY_NAME.put( "voltPerLitreMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerLitreMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per litre minute" ).build(), Optional.of("V/(l\u00B7min)"), Optional.of("F87"), Optional.empty(), Optional.of("1.66667 \u00D7 10\u00B9 kg \u00D7 m\u207B\u00B9 \u00D7 s\u207B\u2074 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per metre" ).build(), Optional.of("V/m"), Optional.of("D50"), Optional.empty(), Optional.of("V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerMicrosecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerMicrosecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per microsecond" ).build(), Optional.of("V/\u00B5s"), Optional.of("H24"), Optional.of("voltPerSecond"), Optional.of("10\u2076 V/s"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per millimetre" ).build(), Optional.of("V/mm"), Optional.of("D51"), Optional.of("voltPerMetre"), Optional.of("10\u00B3 V/m"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerPascal", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerPascal" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per pascal" ).build(), Optional.of("V/Pa"), Optional.of("N98"), Optional.empty(), Optional.of("m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt per second" ).build(), Optional.of("V/s"), Optional.of("H46"), Optional.empty(), Optional.of("m\u00B2 \u00D7 kg \u00D7 s\u207B\u2074 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltSecondPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltSecondPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt second per metre" ).build(), Optional.of("V\u00B7s/m"), Optional.of("H45"), Optional.empty(), Optional.of("m \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltSquareInchPerPoundForce", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltSquareInchPerPoundForce" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt square inch per pound-force" ).build(), Optional.of("V/(lbf/in\u00B2)"), Optional.of("H22"), Optional.empty(), Optional.of("1.4503774398 \u00D7 10\u207B\u2074 m\u00B3 \u00D7 s\u207B\u00B9 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "voltSquaredPerKelvinSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "voltSquaredPerKelvinSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "volt squared per kelvin squared" ).build(), Optional.of("V\u00B2/K\u00B2"), Optional.of("D45"), Optional.empty(), Optional.of("V\u00B2/K\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ELECTRIC_FIELD_STRENGTH)) ) );
UNITS_BY_NAME.put( "waterHorsePower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "waterHorsePower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "water horse power" ).build(), Optional.empty(), Optional.of("F80"), Optional.of("watt"), Optional.of("7.46043 \u00D7 10\u00B2 W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER)) ) );
UNITS_BY_NAME.put( "watt", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "watt" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt" ).build(), Optional.of("W"), Optional.of("WTT"), Optional.empty(), Optional.of("W"), new HashSet<>(Arrays.asList(QuantityKinds.POWER_FOR_DIRECT_CURRENT, QuantityKinds.HEAT_FLOW_RATE, QuantityKinds.SOUND_POWER, QuantityKinds.POWER, QuantityKinds.RADIANT_POWER, QuantityKinds.RADIANT_ENERGYFLUX, QuantityKinds.ACTIVE_POWER)) ) );
UNITS_BY_NAME.put( "wattHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt hour" ).build(), Optional.of("W\u00B7h"), Optional.of("WHR"), Optional.of("joule"), Optional.of("3.6 \u00D7 10\u00B3 J"), new HashSet<>(Arrays.asList(QuantityKinds.ACTIVE_ENERGY, QuantityKinds.POTENTIAL_ENERGY, QuantityKinds.ENERGY, QuantityKinds.KINETIC_ENERGY, QuantityKinds.WORK)) ) );
UNITS_BY_NAME.put( "wattPerCubicMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerCubicMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per cubic metre" ).build(), Optional.of("W/m\u00B3"), Optional.of("H47"), Optional.empty(), Optional.of("m\u207B\u00B9 \u00D7 kg \u00D7 s\u207B\u00B3"), new HashSet<>(Arrays.asList(QuantityKinds.FIRST_RADIATION_CONSTANT, QuantityKinds.FLUENCE_RATE, QuantityKinds.IRRADIANCE, QuantityKinds.RADIANT_ENERGY, QuantityKinds.RADIANT_EXITANCE)) ) );
UNITS_BY_NAME.put( "wattPerKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per kelvin" ).build(), Optional.of("W/K"), Optional.of("D52"), Optional.empty(), Optional.of("W/K"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTANCE)) ) );
UNITS_BY_NAME.put( "wattPerKilogram", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerKilogram" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per kilogram" ).build(), Optional.of("W/kg"), Optional.of("WA"), Optional.empty(), Optional.of("1 W/kg"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wattPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per metre" ).build(), Optional.of("W/m"), Optional.of("H74"), Optional.empty(), Optional.of("W m\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_ENERGY, QuantityKinds.RADIANT_EXITANCE, QuantityKinds.FLUENCE_RATE, QuantityKinds.FIRST_RADIATION_CONSTANT, QuantityKinds.IRRADIANCE)) ) );
UNITS_BY_NAME.put( "wattPerMetreDegreeCelsius", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerMetreDegreeCelsius" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per metre degree Celsius" ).build(), Optional.of("W/(m\u00B7\u00B0C)"), Optional.of("N80"), Optional.of("wattPerMetreKelvin"), Optional.of("W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "wattPerMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per metre kelvin" ).build(), Optional.of("W/(m\u00B7K)"), Optional.of("D53"), Optional.empty(), Optional.of("W/(m \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.THERMAL_CONDUCTIVITY)) ) );
UNITS_BY_NAME.put( "wattPerSquareCentimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSquareCentimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per square centimetre" ).build(), Optional.of("W/cm\u00B2"), Optional.of("N48"), Optional.of("wattPerSquareMetre"), Optional.of("10\u2074 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "wattPerSquareInch", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSquareInch" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per square inch" ).build(), Optional.of("W/in\u00B2"), Optional.of("N49"), Optional.of("wattPerSquareMetre"), Optional.of("1.550003 \u00D7 10\u00B3 W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE)) ) );
UNITS_BY_NAME.put( "wattPerSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per square metre" ).build(), Optional.of("W/m\u00B2"), Optional.of("D54"), Optional.empty(), Optional.of("W/m\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.SOUND_INTENSITY, QuantityKinds.DENSITY_OF_HEAT_FLOW_RATE, QuantityKinds.FIRST_RADIATION_CONSTANT, QuantityKinds.FLUENCE_RATE, QuantityKinds.POYNTING_VECTOR, QuantityKinds.IRRADIANCE, QuantityKinds.ENERGY_FLUX_DENSITY, QuantityKinds.ENERGY_FLUENCE_RATE, QuantityKinds.RADIANT_EXITANCE, QuantityKinds.RADIANT_ENERGY)) ) );
UNITS_BY_NAME.put( "wattPerSquareMetreKelvin", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSquareMetreKelvin" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per square metre kelvin" ).build(), Optional.of("W/(m\u00B2\u00B7K)"), Optional.of("D55"), Optional.empty(), Optional.of("W/(m\u00B2 \u00D7 K)"), new HashSet<>(Arrays.asList(QuantityKinds.COEFFICIENT_OF_HEAT_TRANSFER)) ) );
UNITS_BY_NAME.put( "wattPerSquareMetreKelvinToTheFourthPower", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSquareMetreKelvinToTheFourthPower" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per square metre kelvin to the fourth power" ).build(), Optional.of("W/(m\u00B2\u00B7K\u2074)"), Optional.of("D56"), Optional.empty(), Optional.of("W/(m\u00B2 \u00D7 K\u2074)"), new HashSet<>(Arrays.asList(QuantityKinds.STEFAN_BOLTZMANN_CONSTANT)) ) );
UNITS_BY_NAME.put( "wattPerSteradian", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSteradian" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per steradian" ).build(), Optional.of("W/sr"), Optional.of("D57"), Optional.empty(), Optional.of("W/sr"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANT_INTENSITY)) ) );
UNITS_BY_NAME.put( "wattPerSteradianSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattPerSteradianSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt per steradian square metre" ).build(), Optional.of("W/(sr\u00B7m\u00B2)"), Optional.of("D58"), Optional.empty(), Optional.of("W/(sr \u00D7 m\u00B2)"), new HashSet<>(Arrays.asList(QuantityKinds.RADIANCE)) ) );
UNITS_BY_NAME.put( "wattSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt second" ).build(), Optional.of("W\u00B7s"), Optional.of("J55"), Optional.empty(), Optional.of("W \u00D7 s"), new HashSet<>(Arrays.asList(QuantityKinds.HELMHOLTZ_FUNCTION, QuantityKinds.ENERGY, QuantityKinds.ENTHALPY, QuantityKinds.QUANTITY_OF_HEAT, QuantityKinds.THERMODYNAMIC_ENERGY, QuantityKinds.HELMHOLTZ_FREE_ENERGY, QuantityKinds.HEAT)) ) );
UNITS_BY_NAME.put( "wattSquareMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wattSquareMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "watt square metre" ).build(), Optional.of("W\u00B7m\u00B2"), Optional.of("Q21"), Optional.empty(), Optional.of("m\u2074 \u00D7 kg \u00D7 s\u207B\u00B3"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "weber", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "weber" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "weber" ).build(), Optional.of("Wb"), Optional.of("WEB"), Optional.empty(), Optional.of("Wb"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_FLUX_QUANTUM, QuantityKinds.MAGNETIC_FLUX)) ) );
UNITS_BY_NAME.put( "weberMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "weberMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "weber metre" ).build(), Optional.of("Wb\u00B7m"), Optional.of("P50"), Optional.empty(), Optional.of("m\u00B3 \u00D7 kg \u00D7 s\u207B\u00B2 \u00D7 A\u207B\u00B9"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_DIPOLE_MOMENT)) ) );
UNITS_BY_NAME.put( "weberPerMetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "weberPerMetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "weber per metre" ).build(), Optional.of("Wb/m"), Optional.of("D59"), Optional.empty(), Optional.of("Wb/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_VECTOR_POTENTIAL)) ) );
UNITS_BY_NAME.put( "weberPerMillimetre", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "weberPerMillimetre" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "weber per millimetre" ).build(), Optional.of("Wb/mm"), Optional.of("D60"), Optional.of("weberPerMetre"), Optional.of("10\u00B3 Wb/m"), new HashSet<>(Arrays.asList(QuantityKinds.MAGNETIC_VECTOR_POTENTIAL)) ) );
UNITS_BY_NAME.put( "weberToThePowerMinusOne", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "weberToThePowerMinusOne" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "weber to the power minus one" ).build(), Optional.of("1/Wb"), Optional.of("Q23"), Optional.empty(), Optional.of("m\u207B\u00B2 \u00D7 kg\u207B\u00B9 \u00D7 s\u00B2 \u00D7 A"), Collections.emptySet() ) );
UNITS_BY_NAME.put( "week", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "week" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "week" ).build(), Optional.of("wk"), Optional.of("WEE"), Optional.of("secondUnitOfTime"), Optional.of("6.048 \u00D7 10\u2075 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "well", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "well" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "well" ).build(), Optional.empty(), Optional.of("E56"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wetKilo", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wetKilo" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "wet kilo" ).build(), Optional.empty(), Optional.of("W2"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wetPound", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wetPound" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "wet pound" ).build(), Optional.empty(), Optional.of("WB"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wetTon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wetTon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "wet ton" ).build(), Optional.empty(), Optional.of("WE"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "wineGallon", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "wineGallon" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "wine gallon" ).build(), Optional.empty(), Optional.of("WG"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}

private static void init17() {
UNITS_BY_NAME.put( "workingDay", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "workingDay" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "working day" ).build(), Optional.empty(), Optional.of("E49"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "workingMonth", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "workingMonth" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "working month" ).build(), Optional.empty(), Optional.of("WM"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
UNITS_BY_NAME.put( "yard", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yard" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard" ).build(), Optional.of("yd"), Optional.of("YRD"), Optional.of("metre"), Optional.of("0.9144 m"), new HashSet<>(Arrays.asList(QuantityKinds.DISTANCE, QuantityKinds.RADIUS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.THICKNESS, QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "yardPerDegreeFahrenheit", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yardPerDegreeFahrenheit" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard per degree Fahrenheit" ).build(), Optional.of("yd/\u00B0F"), Optional.of("L98"), Optional.of("metrePerKelvin"), Optional.of("1.64592 m/K"), new HashSet<>(Arrays.asList(QuantityKinds.RADIUS, QuantityKinds.DISTANCE, QuantityKinds.THICKNESS, QuantityKinds.RADIUS_OF_CURVATURE, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.BREADTH, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DIAMETER, QuantityKinds.LENGTH_OF_PATH)) ) );
UNITS_BY_NAME.put( "yardPerHour", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yardPerHour" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard per hour" ).build(), Optional.of("yd/h"), Optional.of("M66"), Optional.of("metrePerSecond"), Optional.of("2.54 \u00D7 10\u207B\u2074 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.GROUP_VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY)) ) );
UNITS_BY_NAME.put( "yardPerMinute", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yardPerMinute" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard per minute" ).build(), Optional.of("yd/min"), Optional.of("M65"), Optional.of("metrePerSecond"), Optional.of("1.524 \u00D7 10\u207B\u00B2 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.PHASE_VELOCITY, QuantityKinds.VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "yardPerPsi", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yardPerPsi" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard per psi" ).build(), Optional.of("yd/psi"), Optional.of("L99"), Optional.of("metrePerPascal"), Optional.of("1.326225 \u00D7 10\u207B\u2074 m/Pa"), new HashSet<>(Arrays.asList(QuantityKinds.DIAMETER, QuantityKinds.BREADTH, QuantityKinds.LENGTH, QuantityKinds.HEIGHT, QuantityKinds.RADIUS, QuantityKinds.THICKNESS, QuantityKinds.CARTESIAN_COORDINATES, QuantityKinds.DISTANCE, QuantityKinds.LENGTH_OF_PATH, QuantityKinds.RADIUS_OF_CURVATURE)) ) );
UNITS_BY_NAME.put( "yardPerSecond", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yardPerSecond" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard per second" ).build(), Optional.of("yd/s"), Optional.of("M64"), Optional.of("metrePerSecond"), Optional.of("9.144 \u00D7 10\u207B\u00B9 m/s"), new HashSet<>(Arrays.asList(QuantityKinds.VELOCITY, QuantityKinds.PHASE_VELOCITY, QuantityKinds.GROUP_VELOCITY)) ) );
UNITS_BY_NAME.put( "yardPerSecondSquared", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yardPerSecondSquared" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "yard per second squared" ).build(), Optional.of("yd/s\u00B2"), Optional.of("M40"), Optional.of("metrePerSecondSquared"), Optional.of("9.144 \u00D7 10\u207B\u00B9 m/s\u00B2"), new HashSet<>(Arrays.asList(QuantityKinds.ACCELERATION, QuantityKinds.ACCELERATION_OF_FREE_FALL, QuantityKinds.ACCELERATION_DUE_TO_GRAVITY)) ) );
UNITS_BY_NAME.put( "year", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "year" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "year" ).build(), Optional.of("y"), Optional.of("ANN"), Optional.of("secondUnitOfTime"), Optional.of("3.15576 \u00D7 10\u2077 s"), new HashSet<>(Arrays.asList(QuantityKinds.TIME)) ) );
UNITS_BY_NAME.put( "yobibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yobibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Yobibyte" ).build(), Optional.of("YiB"), Optional.empty(), Optional.of("byte"), Optional.of("2\u2078\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "yottabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "yottabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Yottabyte" ).build(), Optional.of("YB"), Optional.empty(), Optional.of("byte"), Optional.of("10\u00B2\u2074 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "zebibyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "zebibyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Zebibyte" ).build(), Optional.of("ZiB"), Optional.empty(), Optional.of("byte"), Optional.of("2\u2077\u2070 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "zettabyte", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "zettabyte" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "Zettabyte" ).build(), Optional.of("ZB"), Optional.empty(), Optional.of("byte"), Optional.of("10\u00B2\u00B9 B"), new HashSet<>(Arrays.asList(QuantityKinds.INFORMATION_ENTROPY)) ) );
UNITS_BY_NAME.put( "zone", new DefaultUnit( MetaModelBaseAttributes.builder().withUrn( SammNs.UNIT.urn( "zone" ) ).withPreferredName( Locale.forLanguageTag( "en" ), "zone" ).build(), Optional.empty(), Optional.of("E57"), Optional.empty(), Optional.empty(), Collections.emptySet() ) );
}


   /**
    * Returns the file that defines this unit
    */
   public AspectModelFile getSourceFile() {
      return MetaModelFile.UNITS;
   }

   /**
    * Returns the unit with a given name
    */
   public static synchronized Optional<Unit> fromName( final String name ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
init0();
init1();
init2();
init3();
init4();
init5();
init6();
init7();
init8();
init9();
init10();
init11();
init12();
init13();
init14();
init15();
init16();
init17();
      }
      return Optional.ofNullable( UNITS_BY_NAME.get( name ) );
   }

   /**
    * Finds the unit with a given code
    */
   public static Optional<Unit> fromCode( final String code ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
         fromName( "" );
      }
      return UNITS_BY_NAME.values()
            .stream()
            .filter( unit -> unit.getCode().map( code2 -> code2.equals( code ) ).orElse( false ) )
            .findAny();
   }

   /**
    * Finds the units with a given symbol. In most, but not all cases, this set will contain exactly one unit.
    */
   public static Set<Unit> fromSymbol( final String symbol ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
         fromName( "" );
      }
      return UNITS_BY_NAME.values()
            .stream()
            .flatMap( unit -> unit.getSymbol()
                  .filter( symbol2 -> symbol2.equals( symbol ) )
                  .stream()
                  .map( symbol2 -> unit ) )
            .collect( Collectors.toSet() );
   }

   /**
    * Returns all units of a given quantity kind
    */
   public static Set<Unit> unitsWithQuantityKind( final QuantityKind quantityKind ) {
      if ( UNITS_BY_NAME.isEmpty() ) {
         fromName( "" );
      }
      return UNITS_BY_NAME.values()
            .stream()
            .filter( unit -> unit.getQuantityKinds().contains( quantityKind ) )
            .collect( Collectors.toSet() );
   }
}
