/**
 * Copyright (C) 2011-2012 Typesafe Inc. <http://typesafe.com>
 */
package org.ekrich.config

import java.{lang => jl}

/**
 * The syntax of a character stream (<a href="http://json.org">JSON</a>, <a
 * href="https://github.com/lightbend/config/blob/master/HOCON.md">HOCON</a> aka
 * ".conf", or <a href=
 * "http://download.oracle.com/javase/7/docs/api/java/util/Properties.html#load%28java.io.Reader%29"
 * >Java properties</a>).
 */
final class ConfigSyntax private (name: String, ordinal: Int)
    extends jl.Enum[ConfigSyntax](name, ordinal)

object ConfigSyntax {

  /**
   * Pedantically strict <a href="http://json.org">JSON</a> format; no comments,
   * no unexpected commas, no duplicate keys in the same object. Associated with
   * the `.json` file extension and `application/json` Content-Type.
   */
  final val JSON = new ConfigSyntax("JSON", 0)

  /**
   * The JSON-superset <a
   * href="https://github.com/lightbend/config/blob/master/HOCON.md">HOCON</a>
   * format. Associated with the `.conf` file extension and `application/hocon`
   * Content-Type.
   */
  final val CONF = new ConfigSyntax("CONF", 1)

  /**
   * Standard <a href=
   * "http://download.oracle.com/javase/7/docs/api/java/util/Properties.html#load%28java.io.Reader%29"
   * >Java properties</a> format. Associated with the `.properties` file
   * extension and `text/x-java-properties` Content-Type.
   */
  final val PROPERTIES = new ConfigSyntax("PROPERTIES", 2)

  private[this] final val _values: Array[ConfigSyntax] =
    Array(JSON, CONF, PROPERTIES)

  def values: Array[ConfigSyntax] = _values.clone()

  def valueOf(name: String): ConfigSyntax =
    _values.find(_.name == name).getOrElse {
      throw new IllegalArgumentException("No enum const ConfigSyntax." + name)
    }
}
