/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.gradle.testclusters;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.elasticsearch.GradleServicesAdapter;
import org.elasticsearch.gradle.Distribution;
import org.elasticsearch.gradle.Version;
import org.elasticsearch.gradle.testclusters.TestClustersException;
import org.gradle.api.Action;
import org.gradle.api.file.CopySpec;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.internal.os.OperatingSystem;

public class ElasticsearchNode {
    private final Logger logger = Logging.getLogger(ElasticsearchNode.class);
    private final String name;
    private final GradleServicesAdapter services;
    private final AtomicBoolean configurationFrozen = new AtomicBoolean(false);
    private final Path artifactsExtractDir;
    private final Path workingDir;
    private static final int ES_DESTROY_TIMEOUT = 20;
    private static final TimeUnit ES_DESTROY_TIMEOUT_UNIT = TimeUnit.SECONDS;
    private static final int NODE_UP_TIMEOUT = 30;
    private static final TimeUnit NODE_UP_TIMEOUT_UNIT = TimeUnit.SECONDS;
    private final LinkedHashMap<String, Predicate<ElasticsearchNode>> waitConditions;
    private final Path confPathRepo;
    private final Path configFile;
    private final Path confPathData;
    private final Path confPathLogs;
    private final Path transportPortFile;
    private final Path httpPortsFile;
    private final Path esStdoutFile;
    private final Path esStderrFile;
    private Distribution distribution;
    private String version;
    private File javaHome;
    private volatile Process esProcess;
    private final String path;

    ElasticsearchNode(String path, String name, GradleServicesAdapter services, File artifactsExtractDir, File workingDirBase) {
        this.path = path;
        this.name = name;
        this.services = services;
        this.artifactsExtractDir = artifactsExtractDir.toPath();
        this.workingDir = workingDirBase.toPath().resolve(ElasticsearchNode.safeName(name)).toAbsolutePath();
        this.confPathRepo = this.workingDir.resolve("repo");
        this.configFile = this.workingDir.resolve("config/elasticsearch.yml");
        this.confPathData = this.workingDir.resolve("data");
        this.confPathLogs = this.workingDir.resolve("logs");
        this.transportPortFile = this.confPathLogs.resolve("transport.ports");
        this.httpPortsFile = this.confPathLogs.resolve("http.ports");
        this.esStdoutFile = this.confPathLogs.resolve("es.stdout.log");
        this.esStderrFile = this.confPathLogs.resolve("es.stderr.log");
        this.waitConditions = new LinkedHashMap();
        this.waitConditions.put("http ports file", node -> Files.exists(node.httpPortsFile, new LinkOption[0]));
        this.waitConditions.put("transport ports file", node -> Files.exists(node.transportPortFile, new LinkOption[0]));
        this.waitForUri("cluster health yellow", "/_cluster/health?wait_for_nodes=>=1&wait_for_status=yellow");
    }

    public String getName() {
        return this.name;
    }

    public String getVersion() {
        return this.version;
    }

    public void setVersion(String version) {
        Objects.requireNonNull(version, "null version passed when configuring test cluster `" + this + "`");
        this.checkFrozen();
        this.version = version;
    }

    public Distribution getDistribution() {
        return this.distribution;
    }

    public void setDistribution(Distribution distribution) {
        Objects.requireNonNull(distribution, "null distribution passed when configuring test cluster `" + this + "`");
        this.checkFrozen();
        this.distribution = distribution;
    }

    public void freeze() {
        Objects.requireNonNull(this.distribution, "null distribution passed when configuring test cluster `" + this + "`");
        Objects.requireNonNull(this.version, "null version passed when configuring test cluster `" + this + "`");
        Objects.requireNonNull(this.javaHome, "null javaHome passed when configuring test cluster `" + this + "`");
        this.logger.info("Locking configuration of `{}`", (Object)this);
        this.configurationFrozen.set(true);
    }

    public void setJavaHome(File javaHome) {
        Objects.requireNonNull(javaHome, "null javaHome passed when configuring test cluster `" + this + "`");
        this.checkFrozen();
        if (!javaHome.exists()) {
            throw new TestClustersException("java home for `" + this + "` does not exists: `" + javaHome + "`");
        }
        this.javaHome = javaHome;
    }

    public File getJavaHome() {
        return this.javaHome;
    }

    private void waitForUri(String description, String uri) {
        this.waitConditions.put(description, node -> {
            try {
                URL url = new URL("http://" + this.getHttpPortInternal().get(0) + uri);
                HttpURLConnection con = (HttpURLConnection)url.openConnection();
                con.setRequestMethod("GET");
                con.setConnectTimeout(500);
                con.setReadTimeout(500);
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(con.getInputStream()));){
                    String response = reader.lines().collect(Collectors.joining("\n"));
                    this.logger.info("{} -> {} ->\n{}", new Object[]{this, uri, response});
                }
                return true;
            }
            catch (IOException e) {
                throw new IllegalStateException("Connection attempt to " + this + " failed", e);
            }
        });
    }

    synchronized void start() {
        this.logger.info("Starting `{}`", (Object)this);
        Path distroArtifact = this.artifactsExtractDir.resolve(this.distribution.getFileExtension()).resolve(this.distribution.getFileName() + "-" + this.getVersion());
        if (!Files.exists(distroArtifact, new LinkOption[0])) {
            throw new TestClustersException("Can not start " + this + ", missing: " + distroArtifact);
        }
        if (!Files.isDirectory(distroArtifact, new LinkOption[0])) {
            throw new TestClustersException("Can not start " + this + ", is not a directory: " + distroArtifact);
        }
        this.services.sync((Action<? super CopySpec>)((Action)spec -> {
            spec.from(new Object[]{distroArtifact.resolve("config").toFile()});
            spec.into((Object)this.configFile.getParent());
        }));
        this.configure();
        this.startElasticsearchProcess(distroArtifact);
    }

    private void startElasticsearchProcess(Path distroArtifact) {
        this.logger.info("Running `bin/elasticsearch` in `{}` for {}", (Object)this.workingDir, (Object)this);
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]);
        if (OperatingSystem.current().isWindows()) {
            processBuilder.command("cmd", "/c", distroArtifact.resolve("\\bin\\elasticsearch.bat").toAbsolutePath().toString());
        } else {
            processBuilder.command(distroArtifact.resolve("bin/elasticsearch").toAbsolutePath().toString());
        }
        try {
            processBuilder.directory(this.workingDir.toFile());
            Map<String, String> environment = processBuilder.environment();
            environment.clear();
            environment.put("JAVA_HOME", this.getJavaHome().getAbsolutePath());
            environment.put("ES_PATH_CONF", this.configFile.getParent().toAbsolutePath().toString());
            environment.put("ES_JAVA_OPTIONS", "-Xms512m -Xmx512m");
            processBuilder.redirectError(ProcessBuilder.Redirect.appendTo(this.esStderrFile.toFile()));
            processBuilder.redirectOutput(ProcessBuilder.Redirect.appendTo(this.esStdoutFile.toFile()));
            this.esProcess = processBuilder.start();
        }
        catch (IOException e) {
            throw new TestClustersException("Failed to start ES process for " + this, e);
        }
    }

    public String getHttpSocketURI() {
        this.waitForAllConditions();
        return this.getHttpPortInternal().get(0);
    }

    public String getTransportPortURI() {
        this.waitForAllConditions();
        return this.getTransportPortInternal().get(0);
    }

    synchronized void stop(boolean tailLogs) {
        if (this.esProcess == null && tailLogs) {
            return;
        }
        this.logger.info("Stopping `{}`, tailLogs: {}", (Object)this, (Object)tailLogs);
        Objects.requireNonNull(this.esProcess, "Can't stop `" + this + "` as it was not started or already stopped.");
        this.stopHandle(this.esProcess.toHandle(), true);
        if (tailLogs) {
            this.logFileContents("Standard output of node", this.esStdoutFile);
            this.logFileContents("Standard error of node", this.esStderrFile);
        }
        this.esProcess = null;
    }

    private void stopHandle(ProcessHandle processHandle, boolean forcibly) {
        if (!processHandle.isAlive()) {
            this.logger.info("Process was not running when we tried to terminate it.");
            return;
        }
        processHandle.children().forEach(each -> this.stopHandle((ProcessHandle)each, forcibly));
        this.logProcessInfo("Terminating elasticsearch process" + (forcibly ? " forcibly " : "gracefully") + ":", processHandle.info());
        if (forcibly) {
            processHandle.destroyForcibly();
        } else {
            processHandle.destroy();
            this.waitForProcessToExit(processHandle);
            if (!processHandle.isAlive()) {
                return;
            }
            this.logger.info("process did not terminate after {} {}, stopping it forcefully", (Object)20, (Object)ES_DESTROY_TIMEOUT_UNIT);
            processHandle.destroyForcibly();
        }
        this.waitForProcessToExit(processHandle);
        if (processHandle.isAlive()) {
            throw new TestClustersException("Was not able to terminate elasticsearch process");
        }
    }

    private void logProcessInfo(String prefix, ProcessHandle.Info info) {
        this.logger.info(prefix + " commandLine:`{}` command:`{}` args:`{}`", new Object[]{info.commandLine().orElse("-"), info.command().orElse("-"), Arrays.stream(info.arguments().orElse(new String[0])).map(each -> "'" + each + "'").collect(Collectors.joining(" "))});
    }

    private void logFileContents(String description, Path from) {
        this.logger.error("{} `{}`", (Object)description, (Object)this);
        try (Stream<String> lines = Files.lines(from, StandardCharsets.UTF_8);){
            lines.map(line -> "  " + line).forEach(arg_0 -> ((Logger)this.logger).error(arg_0));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void waitForProcessToExit(ProcessHandle processHandle) {
        try {
            processHandle.onExit().get(20L, ES_DESTROY_TIMEOUT_UNIT);
        }
        catch (InterruptedException e) {
            this.logger.info("Interrupted while waiting for ES process", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            this.logger.info("Failure while waiting for process to exist", (Throwable)e);
        }
        catch (TimeoutException e) {
            this.logger.info("Timed out waiting for process to exit", (Throwable)e);
        }
    }

    private void configure() {
        try {
            Files.createDirectories(this.configFile.getParent(), new FileAttribute[0]);
            Files.createDirectories(this.confPathRepo, new FileAttribute[0]);
            Files.createDirectories(this.confPathData, new FileAttribute[0]);
            Files.createDirectories(this.confPathLogs, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        LinkedHashMap<String, Object> config = new LinkedHashMap<String, Object>();
        String nodeName = ElasticsearchNode.safeName(this.name);
        config.put("cluster.name", nodeName);
        config.put("node.name", nodeName);
        config.put("path.repo", this.confPathRepo.toAbsolutePath().toString());
        config.put("path.data", this.confPathData.toAbsolutePath().toString());
        config.put("path.logs", this.confPathLogs.toAbsolutePath().toString());
        config.put("path.shared_data", this.workingDir.resolve("sharedData").toString());
        config.put("node.attr.testattr", "test");
        config.put("node.portsfile", "true");
        config.put("http.port", "0");
        config.put("transport.tcp.port", "0");
        config.put("cluster.routing.allocation.disk.watermark.low", "1b");
        config.put("cluster.routing.allocation.disk.watermark.high", "1b");
        config.put("script.max_compilations_rate", "2048/1m");
        if (Version.fromString(this.version).getMajor() >= 6) {
            config.put("cluster.routing.allocation.disk.watermark.flood_stage", "1b");
        }
        if (Version.fromString(this.version).getMajor() >= 7) {
            config.put("cluster.initial_master_nodes", "[" + nodeName + "]");
        }
        try {
            Files.write(this.configFile, config.entrySet().stream().map(entry -> (String)entry.getKey() + ": " + (String)entry.getValue()).collect(Collectors.joining("\n")).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Could not write config file: " + this.configFile, e);
        }
        this.logger.info("Written config file:{} for {}", (Object)this.configFile, (Object)this);
    }

    private void checkFrozen() {
        if (this.configurationFrozen.get()) {
            throw new IllegalStateException("Configuration can not be altered, already locked");
        }
    }

    private static String safeName(String name) {
        return name.replaceAll("^[^a-zA-Z0-9]+", "").replaceAll("[^a-zA-Z0-9]+", "-");
    }

    private List<String> getTransportPortInternal() {
        try {
            return this.readPortsFile(this.transportPortFile);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read transport ports file: " + this.transportPortFile + " for " + this, e);
        }
    }

    private List<String> getHttpPortInternal() {
        try {
            return this.readPortsFile(this.httpPortsFile);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read http ports file: " + this.httpPortsFile + " for " + this, e);
        }
    }

    private List<String> readPortsFile(Path file) throws IOException {
        try (Stream<String> lines = Files.lines(file, StandardCharsets.UTF_8);){
            List<String> list = lines.map(String::trim).collect(Collectors.toList());
            return list;
        }
    }

    private void waitForAllConditions() {
        Objects.requireNonNull(this.esProcess, "Can't wait for `" + this + "` as it was stopped.");
        long startedAt = System.currentTimeMillis();
        this.logger.info("Starting to wait for cluster to come up");
        this.waitConditions.forEach((description, predicate) -> {
            long thisConditionStartedAt = System.currentTimeMillis();
            boolean conditionMet = false;
            Exception lastException = null;
            while (System.currentTimeMillis() - startedAt < TimeUnit.MILLISECONDS.convert(30L, NODE_UP_TIMEOUT_UNIT)) {
                if (!this.esProcess.isAlive()) {
                    throw new TestClustersException("process was found dead while waiting for " + description + ", " + this);
                }
                try {
                    if (predicate.test(this)) {
                        conditionMet = true;
                        break;
                    }
                }
                catch (TestClustersException e) {
                    throw new TestClustersException(e);
                }
                catch (Exception e) {
                    if (lastException == null) {
                        lastException = e;
                    }
                    e.addSuppressed(lastException);
                    lastException = e;
                }
                try {
                    Thread.sleep(500L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
            if (!conditionMet) {
                String message = "`" + this + "` failed to wait for " + description + " after 30 " + NODE_UP_TIMEOUT_UNIT;
                if (lastException == null) {
                    throw new TestClustersException(message);
                }
                throw new TestClustersException(message, lastException);
            }
            this.logger.info("{}: {} took {} seconds", new Object[]{this, description, TimeUnit.SECONDS.convert(System.currentTimeMillis() - thisConditionStartedAt, TimeUnit.MILLISECONDS)});
        });
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ElasticsearchNode that = (ElasticsearchNode)o;
        return Objects.equals(this.name, that.name) && Objects.equals(this.path, that.path);
    }

    public int hashCode() {
        return Objects.hash(this.name, this.path);
    }

    public String toString() {
        return "node{" + this.path + ":" + this.name + "}";
    }
}

