/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.common.blobstore.gcs;

import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.bytes.BytesReference;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;


public class GoogleCloudStorageBlobContainer extends AbstractBlobContainer {

    private final GoogleCloudStorageBlobStore blobStore;
    private final String path;

    GoogleCloudStorageBlobContainer(BlobPath path, GoogleCloudStorageBlobStore blobStore) {
        super(path);
        this.blobStore = blobStore;
        this.path = path.buildAsString();
    }

    @Override
    public boolean blobExists(String blobName) {
        try {
            return blobStore.blobExists(buildKey(blobName));
        } catch (Exception e) {
            throw new BlobStoreException("Failed to check if blob [" + blobName + "] exists", e);
        }
    }

    @Override
    public Map<String, BlobMetaData> listBlobs() throws IOException {
        return blobStore.listBlobs(path);
    }

    @Override
    public Map<String, BlobMetaData> listBlobsByPrefix(String prefix) throws IOException {
        return blobStore.listBlobsByPrefix(path, prefix);
    }

    @Override
    public InputStream readBlob(String blobName) throws IOException {
        return blobStore.readBlob(buildKey(blobName));
    }

    @Override
    public void writeBlob(String blobName, InputStream inputStream, long blobSize) throws IOException {
        blobStore.writeBlob(buildKey(blobName), inputStream, blobSize);
    }

    @Override
    public void writeBlob(String blobName, BytesReference bytes) throws IOException {
        writeBlob(blobName, bytes.streamInput(), bytes.length());
    }

    @Override
    public void deleteBlob(String blobName) throws IOException {
        blobStore.deleteBlob(buildKey(blobName));
    }

    @Override
    public void deleteBlobsByPrefix(String prefix) throws IOException {
        blobStore.deleteBlobsByPrefix(buildKey(prefix));
    }

    @Override
    public void deleteBlobs(Collection<String> blobNames) throws IOException {
        blobStore.deleteBlobs(buildKeys(blobNames));
    }

    @Override
    public void move(String sourceBlobName, String targetBlobName) throws IOException {
        blobStore.moveBlob(buildKey(sourceBlobName), buildKey(targetBlobName));
    }

    protected String buildKey(String blobName) {
        assert blobName != null;
        return path + blobName;
    }

    protected Set<String> buildKeys(Collection<String> blobNames) {
        Set<String> keys = new HashSet<>();
        if (blobNames != null) {
            keys.addAll(blobNames.stream().map(this::buildKey).collect(Collectors.toList()));
        }
        return keys;
    }
}
