/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.test.loggerusage;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.elasticsearch.test.loggerusage.SuppressForbidden;
import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.AbstractInsnNode;
import org.objectweb.asm.tree.IntInsnNode;
import org.objectweb.asm.tree.LdcInsnNode;
import org.objectweb.asm.tree.LineNumberNode;
import org.objectweb.asm.tree.MethodInsnNode;
import org.objectweb.asm.tree.MethodNode;
import org.objectweb.asm.tree.TypeInsnNode;
import org.objectweb.asm.tree.analysis.Analyzer;
import org.objectweb.asm.tree.analysis.AnalyzerException;
import org.objectweb.asm.tree.analysis.BasicInterpreter;
import org.objectweb.asm.tree.analysis.BasicValue;
import org.objectweb.asm.tree.analysis.Frame;
import org.objectweb.asm.tree.analysis.Interpreter;

public class ESLoggerUsageChecker {
    public static final String LOGGER_CLASS = "org.elasticsearch.common.logging.ESLogger";
    public static final String THROWABLE_CLASS = "java.lang.Throwable";
    public static final List<String> LOGGER_METHODS = Arrays.asList("trace", "debug", "info", "warn", "error");
    public static final String IGNORE_CHECKS_ANNOTATION = "org.elasticsearch.common.SuppressLoggerChecks";

    @SuppressForbidden(reason="command line tool")
    public static void main(String ... args) throws Exception {
        System.out.println("checking for wrong usages of ESLogger...");
        boolean[] wrongUsageFound = new boolean[1];
        ESLoggerUsageChecker.checkLoggerUsage(wrongLoggerUsage -> {
            System.err.println(wrongLoggerUsage.getErrorLines());
            wrongUsageFound[0] = true;
        }, args);
        if (wrongUsageFound[0]) {
            throw new Exception("Wrong logger usages found");
        }
        System.out.println("No wrong usages found");
    }

    private static void checkLoggerUsage(final Consumer<WrongLoggerUsage> wrongUsageCallback, String ... classDirectories) throws IOException {
        for (String classDirectory : classDirectories) {
            Path root = Paths.get(classDirectory, new String[0]);
            if (!Files.isDirectory(root, new LinkOption[0])) {
                throw new IllegalArgumentException(root + " should be an existing directory");
            }
            Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (Files.isRegularFile(file, new LinkOption[0]) && file.getFileName().toString().endsWith(".class")) {
                        try (InputStream in = Files.newInputStream(file, new OpenOption[0]);){
                            ESLoggerUsageChecker.check(wrongUsageCallback, in);
                        }
                    }
                    return super.visitFile(file, attrs);
                }
            });
        }
    }

    public static void check(Consumer<WrongLoggerUsage> wrongUsageCallback, InputStream inputStream) throws IOException {
        ESLoggerUsageChecker.check(wrongUsageCallback, inputStream, s -> true);
    }

    static void check(Consumer<WrongLoggerUsage> wrongUsageCallback, InputStream inputStream, Predicate<String> methodsToCheck) throws IOException {
        ClassReader cr = new ClassReader(inputStream);
        cr.accept((ClassVisitor)new ClassChecker(wrongUsageCallback, methodsToCheck), 0);
    }

    private static int calculateNumberOfPlaceHolders(String message) {
        int count = 0;
        for (int i = 1; i < message.length(); ++i) {
            if (message.charAt(i - 1) != '{' || message.charAt(i) != '}') continue;
            ++count;
            ++i;
        }
        return count;
    }

    private static BasicValue getStackValue(Frame<BasicValue> f, int index) {
        int top = f.getStackSize() - 1;
        return index <= top ? (BasicValue)f.getStack(top - index) : null;
    }

    private static final class ArraySizeInterpreter
    extends BasicInterpreter {
        private ArraySizeInterpreter() {
        }

        public BasicValue newOperation(AbstractInsnNode insnNode) throws AnalyzerException {
            switch (insnNode.getOpcode()) {
                case 3: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, 0);
                }
                case 4: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, 1);
                }
                case 5: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, 2);
                }
                case 6: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, 3);
                }
                case 7: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, 4);
                }
                case 8: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, 5);
                }
                case 16: 
                case 17: {
                    return new IntegerConstantBasicValue(Type.INT_TYPE, ((IntInsnNode)insnNode).operand);
                }
                case 18: {
                    Object constant = ((LdcInsnNode)insnNode).cst;
                    if (constant instanceof Integer) {
                        return new IntegerConstantBasicValue(Type.INT_TYPE, (Integer)constant);
                    }
                    return super.newOperation(insnNode);
                }
            }
            return super.newOperation(insnNode);
        }

        public BasicValue merge(BasicValue value1, BasicValue value2) {
            if (value1 instanceof IntegerConstantBasicValue && value2 instanceof IntegerConstantBasicValue) {
                IntegerConstantBasicValue c1 = (IntegerConstantBasicValue)value1;
                IntegerConstantBasicValue c2 = (IntegerConstantBasicValue)value2;
                return new IntegerConstantBasicValue(Type.INT_TYPE, Math.min(c1.minValue, c2.minValue), Math.max(c1.maxValue, c2.maxValue));
            }
            if (value1 instanceof ArraySizeBasicValue && value2 instanceof ArraySizeBasicValue) {
                ArraySizeBasicValue c1 = (ArraySizeBasicValue)value1;
                ArraySizeBasicValue c2 = (ArraySizeBasicValue)value2;
                return new ArraySizeBasicValue(Type.INT_TYPE, Math.min(c1.minValue, c2.minValue), Math.max(c1.maxValue, c2.maxValue));
            }
            return super.merge(value1, value2);
        }

        public BasicValue unaryOperation(AbstractInsnNode insnNode, BasicValue value) throws AnalyzerException {
            if (insnNode.getOpcode() == 189 && value instanceof IntegerConstantBasicValue) {
                IntegerConstantBasicValue constantBasicValue = (IntegerConstantBasicValue)value;
                String desc = ((TypeInsnNode)insnNode).desc;
                return new ArraySizeBasicValue(Type.getType((String)("[" + Type.getObjectType((String)desc))), constantBasicValue.minValue, constantBasicValue.maxValue);
            }
            return super.unaryOperation(insnNode, value);
        }

        public BasicValue ternaryOperation(AbstractInsnNode insnNode, BasicValue value1, BasicValue value2, BasicValue value3) throws AnalyzerException {
            if (insnNode.getOpcode() == 83 && value1 instanceof ArraySizeBasicValue) {
                return value1;
            }
            return super.ternaryOperation(insnNode, value1, value2, value3);
        }
    }

    private static final class PlaceHolderStringInterpreter
    extends BasicInterpreter {
        private PlaceHolderStringInterpreter() {
        }

        public BasicValue newOperation(AbstractInsnNode insnNode) throws AnalyzerException {
            Object constant;
            if (insnNode.getOpcode() == 18 && (constant = ((LdcInsnNode)insnNode).cst) instanceof String) {
                return new PlaceHolderStringBasicValue(ESLoggerUsageChecker.calculateNumberOfPlaceHolders((String)constant));
            }
            return super.newOperation(insnNode);
        }

        public BasicValue merge(BasicValue value1, BasicValue value2) {
            if (value1 instanceof PlaceHolderStringBasicValue && value2 instanceof PlaceHolderStringBasicValue && !value1.equals((Object)value2)) {
                PlaceHolderStringBasicValue c1 = (PlaceHolderStringBasicValue)value1;
                PlaceHolderStringBasicValue c2 = (PlaceHolderStringBasicValue)value2;
                return new PlaceHolderStringBasicValue(Math.min(c1.minValue, c2.minValue), Math.max(c1.maxValue, c2.maxValue));
            }
            return super.merge(value1, value2);
        }
    }

    private static final class IntegerConstantBasicValue
    extends IntMinMaxTrackingBasicValue {
        public IntegerConstantBasicValue(Type type, int constant) {
            super(type, constant);
        }

        public IntegerConstantBasicValue(Type type, int minConstant, int maxConstant) {
            super(type, minConstant, maxConstant);
        }
    }

    private static final class ArraySizeBasicValue
    extends IntMinMaxTrackingBasicValue {
        public ArraySizeBasicValue(Type type, int minArraySize, int maxArraySize) {
            super(type, minArraySize, maxArraySize);
        }
    }

    private static final class PlaceHolderStringBasicValue
    extends IntMinMaxTrackingBasicValue {
        public static final Type STRING_OBJECT_TYPE = Type.getObjectType((String)"java/lang/String");

        public PlaceHolderStringBasicValue(int placeHolders) {
            super(STRING_OBJECT_TYPE, placeHolders);
        }

        public PlaceHolderStringBasicValue(int minPlaceHolders, int maxPlaceHolders) {
            super(STRING_OBJECT_TYPE, minPlaceHolders, maxPlaceHolders);
        }
    }

    private static class IntMinMaxTrackingBasicValue
    extends BasicValue {
        protected final int minValue;
        protected final int maxValue;

        public IntMinMaxTrackingBasicValue(Type type, int value) {
            super(type);
            this.minValue = value;
            this.maxValue = value;
        }

        public IntMinMaxTrackingBasicValue(Type type, int minValue, int maxValue) {
            super(type);
            this.minValue = minValue;
            this.maxValue = maxValue;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || ((Object)((Object)this)).getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            IntMinMaxTrackingBasicValue that = (IntMinMaxTrackingBasicValue)((Object)o);
            if (this.minValue != that.minValue) {
                return false;
            }
            return this.maxValue == that.maxValue;
        }

        public int hashCode() {
            int result = super.hashCode();
            result = 31 * result + this.minValue;
            result = 31 * result + this.maxValue;
            return result;
        }

        public String toString() {
            return "IntMinMaxTrackingBasicValue{minValue=" + this.minValue + ", maxValue=" + this.maxValue + '}';
        }
    }

    private static class MethodChecker
    extends MethodVisitor {
        private final String className;
        private final Consumer<WrongLoggerUsage> wrongUsageCallback;
        private boolean ignoreChecks;

        public MethodChecker(String className, int access, String name, String desc, Consumer<WrongLoggerUsage> wrongUsageCallback) {
            super(327680, (MethodVisitor)new MethodNode(access, name, desc, null, null));
            this.className = className;
            this.wrongUsageCallback = wrongUsageCallback;
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            if (ESLoggerUsageChecker.IGNORE_CHECKS_ANNOTATION.equals(Type.getType((String)desc).getClassName())) {
                this.ignoreChecks = true;
            }
            return super.visitAnnotation(desc, visible);
        }

        public void visitEnd() {
            if (!this.ignoreChecks) {
                this.findBadLoggerUsages((MethodNode)this.mv);
            }
            super.visitEnd();
        }

        public void findBadLoggerUsages(MethodNode methodNode) {
            Analyzer stringPlaceHolderAnalyzer = new Analyzer((Interpreter)new PlaceHolderStringInterpreter());
            Analyzer arraySizeAnalyzer = new Analyzer((Interpreter)new ArraySizeInterpreter());
            try {
                stringPlaceHolderAnalyzer.analyze(this.className, methodNode);
                arraySizeAnalyzer.analyze(this.className, methodNode);
            }
            catch (AnalyzerException e) {
                throw new RuntimeException("Internal error: failed in analysis step", e);
            }
            Frame[] stringFrames = stringPlaceHolderAnalyzer.getFrames();
            Frame[] arraySizeFrames = arraySizeAnalyzer.getFrames();
            AbstractInsnNode[] insns = methodNode.instructions.toArray();
            int lineNumber = -1;
            for (int i = 0; i < insns.length; ++i) {
                AbstractInsnNode insn = insns[i];
                if (insn instanceof LineNumberNode) {
                    LineNumberNode lineNumberNode = (LineNumberNode)insn;
                    lineNumber = lineNumberNode.line;
                }
                if (insn.getOpcode() != 182) continue;
                MethodInsnNode methodInsn = (MethodInsnNode)insn;
                if (!Type.getObjectType((String)methodInsn.owner).getClassName().equals(ESLoggerUsageChecker.LOGGER_CLASS) || !LOGGER_METHODS.contains(methodInsn.name)) continue;
                BasicValue varArgsSizeObject = ESLoggerUsageChecker.getStackValue((Frame<BasicValue>)arraySizeFrames[i], 0);
                if (!(varArgsSizeObject instanceof ArraySizeBasicValue)) {
                    this.wrongUsageCallback.accept(new WrongLoggerUsage(this.className, methodNode.name, methodInsn.name, lineNumber, "Could not determine size of varargs array"));
                    continue;
                }
                ArraySizeBasicValue varArgsSize = (ArraySizeBasicValue)varArgsSizeObject;
                Type[] argumentTypes = Type.getArgumentTypes((String)methodInsn.desc);
                BasicValue logMessageLengthObject = ESLoggerUsageChecker.getStackValue((Frame<BasicValue>)stringFrames[i], argumentTypes.length - 1);
                if (!(logMessageLengthObject instanceof PlaceHolderStringBasicValue)) {
                    if (varArgsSize.minValue <= 0) continue;
                    this.wrongUsageCallback.accept(new WrongLoggerUsage(this.className, methodNode.name, methodInsn.name, lineNumber, "First argument must be a string constant so that we can statically ensure proper place holder usage"));
                    continue;
                }
                PlaceHolderStringBasicValue logMessageLength = (PlaceHolderStringBasicValue)logMessageLengthObject;
                if (logMessageLength.minValue != logMessageLength.maxValue) {
                    this.wrongUsageCallback.accept(new WrongLoggerUsage(this.className, methodNode.name, methodInsn.name, lineNumber, "Multiple log messages with conflicting number of place holders"));
                    continue;
                }
                if (varArgsSize.minValue != varArgsSize.maxValue) {
                    this.wrongUsageCallback.accept(new WrongLoggerUsage(this.className, methodNode.name, methodInsn.name, lineNumber, "Multiple parameter arrays with conflicting sizes"));
                    continue;
                }
                assert (logMessageLength.minValue == logMessageLength.maxValue && varArgsSize.minValue == varArgsSize.maxValue);
                if (logMessageLength.minValue == varArgsSize.minValue) continue;
                this.wrongUsageCallback.accept(new WrongLoggerUsage(this.className, methodNode.name, methodInsn.name, lineNumber, "Expected " + logMessageLength.minValue + " arguments but got " + varArgsSize.minValue));
            }
        }
    }

    private static class ClassChecker
    extends ClassVisitor {
        private String className;
        private boolean ignoreChecks;
        private final Consumer<WrongLoggerUsage> wrongUsageCallback;
        private final Predicate<String> methodsToCheck;

        public ClassChecker(Consumer<WrongLoggerUsage> wrongUsageCallback, Predicate<String> methodsToCheck) {
            super(327680);
            this.wrongUsageCallback = wrongUsageCallback;
            this.methodsToCheck = methodsToCheck;
        }

        public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
            this.className = name;
        }

        public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
            if (ESLoggerUsageChecker.IGNORE_CHECKS_ANNOTATION.equals(Type.getType((String)desc).getClassName())) {
                this.ignoreChecks = true;
            }
            return super.visitAnnotation(desc, visible);
        }

        public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions) {
            if (!this.ignoreChecks && this.methodsToCheck.test(name)) {
                return new MethodChecker(this.className, access, name, desc, this.wrongUsageCallback);
            }
            return super.visitMethod(access, name, desc, signature, exceptions);
        }
    }

    public static class WrongLoggerUsage {
        private final String className;
        private final String methodName;
        private final String logMethodName;
        private final int line;
        private final String errorMessage;

        public WrongLoggerUsage(String className, String methodName, String logMethodName, int line, String errorMessage) {
            this.className = className;
            this.methodName = methodName;
            this.logMethodName = logMethodName;
            this.line = line;
            this.errorMessage = errorMessage;
        }

        public String toString() {
            return "WrongLoggerUsage{className='" + this.className + '\'' + ", methodName='" + this.methodName + '\'' + ", logMethodName='" + this.logMethodName + '\'' + ", line=" + this.line + ", errorMessage='" + this.errorMessage + '\'' + '}';
        }

        public String getErrorLines() {
            String fullClassName = Type.getObjectType((String)this.className).getClassName();
            String simpleClassName = fullClassName.substring(fullClassName.lastIndexOf(".") + 1, fullClassName.length());
            int innerClassIndex = simpleClassName.indexOf("$");
            if (innerClassIndex > 0) {
                simpleClassName = simpleClassName.substring(0, innerClassIndex);
            }
            simpleClassName = simpleClassName + ".java";
            StringBuilder sb = new StringBuilder();
            sb.append("Bad usage of ");
            sb.append(ESLoggerUsageChecker.LOGGER_CLASS).append("#").append(this.logMethodName);
            sb.append(": ");
            sb.append(this.errorMessage);
            sb.append("\n\tat ");
            sb.append(fullClassName);
            sb.append(".");
            sb.append(this.methodName);
            sb.append("(");
            sb.append(simpleClassName);
            sb.append(":");
            sb.append(this.line);
            sb.append(")");
            return sb.toString();
        }
    }
}

