/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsoftware.elasticactors.indexing.elasticsearch.indexer;

import com.fasterxml.jackson.databind.util.ByteBufferBackedInputStream;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.io.CharStreams;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsResponse;
import org.elasticsearch.action.admin.indices.get.GetIndexRequestBuilder;
import org.elasticsearch.action.admin.indices.get.GetIndexResponse;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.client.Client;
import org.elasticsoftware.elasticactors.indexing.elasticsearch.IndexConfig;
import org.elasticsoftware.elasticactors.state.ActorLifecycleStep;
import org.elasticsoftware.elasticactors.state.ActorStateUpdate;
import org.elasticsoftware.elasticactors.state.ActorStateUpdateListener;
import org.elasticsoftware.elasticactors.util.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

public final class Indexer
implements ActorStateUpdateListener {
    private static final Logger logger = LoggerFactory.getLogger(Indexer.class);
    private Client client;
    private final Map<String, List<String>> oldVersionOfIndices = Maps.newHashMap();
    private final Set<String> activatedActors = Sets.newHashSet();
    private final Set<String> setupIndices = Sets.newHashSet();

    @Autowired
    public Indexer(Client client) {
        this.client = client;
    }

    public void onUpdate(List<? extends ActorStateUpdate> updates) {
        updates.stream().filter(update -> update.getActorClass().getAnnotation(IndexConfig.class) != null).forEach(update -> {
            IndexConfig indexConfig = update.getActorClass().getAnnotation(IndexConfig.class);
            Class messageClass = update.getMessageClass();
            ActorLifecycleStep lifecycleStep = update.getLifecycleStep();
            if (messageClass != null && ArrayUtils.contains((Object[])indexConfig.includedMessages(), (Object)messageClass)) {
                this.indexActorState(indexConfig, (ActorStateUpdate)update);
            } else if (lifecycleStep != null) {
                if (lifecycleStep == ActorLifecycleStep.DESTROY) {
                    this.deleteActorState(indexConfig, (ActorStateUpdate)update);
                } else if (Arrays.binarySearch(indexConfig.indexOn(), lifecycleStep) >= 0) {
                    this.indexActorState(indexConfig, (ActorStateUpdate)update);
                }
                if (lifecycleStep == ActorLifecycleStep.ACTIVATE && indexConfig.versioningStrategy() == IndexConfig.VersioningStrategy.REINDEX_ON_ACTIVATE && !this.activatedActors.contains(update.getActorRef().getActorId())) {
                    this.deleteOldVersionsOfActor(indexConfig, (ActorStateUpdate)update);
                    this.activatedActors.add(update.getActorRef().getActorId());
                }
            }
        });
    }

    private void deleteActorState(IndexConfig indexConfig, ActorStateUpdate update) {
        this.doDeleteActorState(this.constructIndexName(indexConfig, update), indexConfig.typeName(), update.getActorRef().getActorId());
    }

    private void deleteOldVersionsOfActor(IndexConfig indexConfig, ActorStateUpdate update) {
        List<String> previousIndices = this.findPreviousIndices(indexConfig, update);
        for (String indexName : previousIndices) {
            this.doDeleteActorState(indexName, indexConfig.typeName(), update.getActorRef().getActorId());
        }
    }

    private void doDeleteActorState(String indexName, String typeName, final String actorId) {
        this.client.prepareDelete(indexName, typeName, actorId).execute((ActionListener)new ActionListener<DeleteResponse>(){

            public void onResponse(DeleteResponse deleteResponse) {
                logger.debug("Successfully deleted actor {{}} from elasticsearch", (Object)actorId);
            }

            public void onFailure(Exception e) {
                logger.error("Failed to delete actor {{}} from elasticsearch", (Object)actorId, (Object)e);
            }
        });
    }

    private void indexActorState(IndexConfig indexConfig, final ActorStateUpdate update) {
        String indexName = this.constructIndexName(indexConfig, update);
        if (!this.setupIndices.contains(indexName)) {
            try {
                this.setUpIndexAliases(indexConfig, update);
                this.setupIndices.add(indexName);
            }
            catch (Exception e) {
                logger.error("Error while trying to setup aliases for index {{}}. Actor <{}> will not be indexed in elasticsearch", new Object[]{indexName, update.getActorRef(), e});
                return;
            }
        }
        try (ByteBufferBackedInputStream is = new ByteBufferBackedInputStream(update.getSerializedState());){
            String actorState = CharStreams.toString((Readable)new InputStreamReader((InputStream)is, StandardCharsets.UTF_8));
            this.client.prepareIndex(indexName, indexConfig.typeName(), update.getActorRef().getActorId()).setOpType(DocWriteRequest.OpType.INDEX).setSource(new Object[]{update.getActorRef().getActorId(), actorState}).execute((ActionListener)new ActionListener<IndexResponse>(){

                public void onResponse(IndexResponse indexResponse) {
                    logger.debug("Successfully indexed actor {{}} in elasticsearch", (Object)update.getActorRef());
                }

                public void onFailure(Exception e) {
                    logger.error("Failed to index actor {{}} in elasticsearch", (Object)update.getActorRef(), (Object)e);
                }
            });
        }
        catch (Exception e) {
            logger.error("Encountered error while trying to index actor {{}} in elasticsearch", (Object)update.getActorRef(), (Object)e);
        }
    }

    private String constructIndexName(IndexConfig indexConfig, ActorStateUpdate update) {
        return indexConfig.indexName() + "_" + this.constructCurrentIndexVersion(indexConfig, update);
    }

    private String constructCurrentIndexVersion(IndexConfig indexConfig, ActorStateUpdate update) {
        if (indexConfig.versioningStrategy().equals((Object)IndexConfig.VersioningStrategy.NONE)) {
            return "v1";
        }
        String[] versions = update.getVersion().split("\\.");
        if (versions.length > 1) {
            return "v" + versions[0] + "-" + versions[1];
        }
        return "v" + update.getVersion();
    }

    private void setUpIndexAliases(IndexConfig indexConfig, ActorStateUpdate update) throws Exception {
        String baseIndexName = indexConfig.indexName();
        String fullIndexName = this.constructIndexName(indexConfig, update);
        IndicesExistsResponse indicesExistsResponse = (IndicesExistsResponse)this.client.admin().indices().prepareExists(new String[]{fullIndexName}).execute().get();
        if (!indicesExistsResponse.isExists()) {
            this.client.admin().indices().prepareCreate(fullIndexName).execute().get();
            this.client.admin().indices().prepareAliases().addAlias(fullIndexName, baseIndexName).execute().get();
        }
    }

    private List<String> findPreviousIndices(IndexConfig indexConfig, ActorStateUpdate update) {
        if (this.oldVersionOfIndices.containsKey(indexConfig.indexName())) {
            return this.oldVersionOfIndices.get(indexConfig.indexName());
        }
        ArrayList previousIndices = Lists.newArrayList();
        try {
            String baseIndexName = indexConfig.indexName();
            String fullIndexName = this.constructIndexName(indexConfig, update);
            GetIndexResponse indexResponse = (GetIndexResponse)((GetIndexRequestBuilder)this.client.admin().indices().prepareGetIndex().addIndices(new String[]{baseIndexName + "*"})).execute().get();
            for (String index : indexResponse.indices()) {
                if (index.equals(baseIndexName) || !index.equals(fullIndexName)) continue;
                previousIndices.add(index);
            }
        }
        catch (Exception e) {
            logger.error("Encountered error while trying to find previous version of indices for base index name {" + indexConfig.indexName() + "} Old versions of actors indexed here won't be removed", (Throwable)e);
        }
        this.oldVersionOfIndices.put(indexConfig.indexName(), previousIndices);
        return previousIndices;
    }

    @VisibleForTesting
    Set<String> getActivatedActors() {
        return this.activatedActors;
    }
}

