/**
 * eobjects.org MetaModel
 * Copyright (C) 2010 eobjects.org
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.eobjects.metamodel.csv;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;

import org.eobjects.metamodel.AbstractUpdateCallback;
import org.eobjects.metamodel.UpdateCallback;
import org.eobjects.metamodel.create.TableCreationBuilder;
import org.eobjects.metamodel.insert.RowInsertionBuilder;
import org.eobjects.metamodel.schema.Schema;
import org.eobjects.metamodel.schema.Table;
import org.eobjects.metamodel.util.EqualsBuilder;
import org.eobjects.metamodel.util.FileHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import au.com.bytecode.opencsv.CSVWriter;

final class CsvUpdateCallback extends AbstractUpdateCallback implements
		UpdateCallback {

	private static final Logger logger = LoggerFactory
			.getLogger(CsvUpdateCallback.class);

	private final File _file;
	private final CsvConfiguration _configuration;
	private CSVWriter csvWriter;

	public CsvUpdateCallback(CsvDataContext dataContext) {
		super(dataContext);
		_file = dataContext.getFile();
		_configuration = dataContext.getConfiguration();
	}

	@Override
	public TableCreationBuilder createTable(Schema schema, String name)
			throws IllegalArgumentException, IllegalStateException {
		return new CsvCreateTableBuilder(this, schema, name);
	}

	@Override
	public RowInsertionBuilder insertInto(Table table)
			throws IllegalArgumentException, IllegalStateException {
		return new CsvInsertBuilder(this, table);
	}

	protected synchronized void writeRow(String[] stringValues, boolean append) {
		getCsvWriter(append);
		csvWriter.writeNext(stringValues);
	}

	private void getCsvWriter(boolean append) {
		if (csvWriter == null || !append) {
			boolean needsLineBreak = needsLineBreak(_file, _configuration);

			final Writer writer = FileHelper.getWriter(_file,
					_configuration.getEncoding(), append);
			if (needsLineBreak) {
				try {
					writer.write('\n');
				} catch (IOException e) {
					logger.debug("Failed to insert newline", e);
				}
			}
			csvWriter = new CSVWriter(writer,
					_configuration.getSeparatorChar(),
					_configuration.getQuoteChar(),
					_configuration.getEscapeChar());
		}
	}

	protected static boolean needsLineBreak(File file,
			CsvConfiguration configuration) {
		if (!file.exists() || file.length() == 0) {
			return false;
		}

		try {
			// find the bytes a newline would match under the encoding
			final byte[] bytesInLineBreak;
			{
				ByteBuffer encodedLineBreak = Charset.forName(
						configuration.getEncoding()).encode("\n");
				bytesInLineBreak = new byte[encodedLineBreak.capacity()];
				encodedLineBreak.get(bytesInLineBreak);
			}

			// find the last bytes of the file
			final byte[] bytesFromFile = new byte[bytesInLineBreak.length];
			{
				final RandomAccessFile randomAccessFile = new RandomAccessFile(
						file, "r");
				try {
					FileChannel channel = randomAccessFile.getChannel();
					try {
						long length = randomAccessFile.length();

						channel = channel.position(length
								- bytesInLineBreak.length);
						channel.read(ByteBuffer.wrap(bytesFromFile));
					} finally {
						channel.close();
					}
				} finally {
					randomAccessFile.close();
				}
			}

			// if the two byte arrays match, then the newline is not needed.
			if (EqualsBuilder.equals(bytesInLineBreak, bytesFromFile)) {
				return false;
			}
			return true;
		} catch (Exception e) {
			logger.error(
					"Error occurred while checking if file needs linebreak, omitting check",
					e);
		}
		return false;
	}

	/**
	 * Closes all open handles
	 */
	protected void close() {
		if (csvWriter != null) {
			try {
				csvWriter.flush();
			} catch (IOException e) {
				logger.warn("Failed to flush CSV writer", e);
			}
			try {
				csvWriter.close();
			} catch (IOException e) {
				logger.error("Failed to close CSV writer", e);
			} finally {
				csvWriter = null;
			}
		}
	}
}
