/**
 * eobjects.org MetaModel
 * Copyright (C) 2010 eobjects.org
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.eobjects.metamodel.csv;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.util.List;

import org.eobjects.metamodel.MetaModelException;
import org.eobjects.metamodel.QueryPostprocessDataContext;
import org.eobjects.metamodel.UpdateScript;
import org.eobjects.metamodel.UpdateableDataContext;
import org.eobjects.metamodel.data.DataSet;
import org.eobjects.metamodel.query.FilterItem;
import org.eobjects.metamodel.schema.Column;
import org.eobjects.metamodel.schema.Table;
import org.eobjects.metamodel.util.FileHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import au.com.bytecode.opencsv.CSVReader;

/**
 * DataContext implementation for reading CSV files.
 */
public final class CsvDataContext extends QueryPostprocessDataContext implements UpdateableDataContext {

    private static final Logger logger = LoggerFactory.getLogger(CsvDataContext.class);

    private final Object WRITE_LOCK = new Object();

    private final File _file;
    private final CsvConfiguration _configuration;
    private final boolean _writable;

    /**
     * Constructs a CSV DataContext based on a file
     * 
     * @param file
     * @param configuration
     */
    public CsvDataContext(File file, CsvConfiguration configuration) {
        if (file == null) {
            throw new IllegalArgumentException("File cannot be null");
        }
        if (configuration == null) {
            throw new IllegalArgumentException("CsvConfiguration cannot be null");
        }
        _file = file;
        _configuration = configuration;
        _writable = true;
    }

    /**
     * Constructs a CSV DataContext based on a {@link URL}
     * 
     * @param url
     * @param configuration
     */
    public CsvDataContext(URL url, CsvConfiguration configuration) {
        _file = createFileFromInputStream(url, configuration.getEncoding());
        _configuration = configuration;
        _writable = false;
    }

    /**
     * Constructs a CSV DataContext based on a file
     * 
     * @param file
     */
    public CsvDataContext(File file) {
        this(file, new CsvConfiguration());
    }

    /**
     * Constructs a CSV DataContext based on an {@link InputStream}
     * 
     * @param inputStream
     * @param configuration
     */
    public CsvDataContext(InputStream inputStream, CsvConfiguration configuration) {
        _file = createFileFromInputStream(inputStream, configuration.getEncoding());
        _configuration = configuration;
        _writable = false;
    }

    /**
     * @deprecated use {@link #CsvDataContext(File, CsvConfiguration)} instead.
     */
    @Deprecated
    public CsvDataContext(File file, char separatorChar) {
        this(file, separatorChar, CsvConfiguration.DEFAULT_QUOTE_CHAR);
    }

    /**
     * @deprecated use {@link #CsvDataContext(File, CsvConfiguration)} instead.
     */
    @Deprecated
    public CsvDataContext(File file, char separatorChar, char quoteChar) {
        this(file, new CsvConfiguration(CsvConfiguration.DEFAULT_COLUMN_NAME_LINE, FileHelper.DEFAULT_ENCODING,
                separatorChar, quoteChar, CsvConfiguration.DEFAULT_ESCAPE_CHAR));
    }

    /**
     * @deprecated use {@link #CsvDataContext(File, CsvConfiguration)} instead.
     */
    @Deprecated
    public CsvDataContext(File file, char separatorChar, char quoteChar, String encoding) {
        this(file, new CsvConfiguration(CsvConfiguration.DEFAULT_COLUMN_NAME_LINE, encoding, separatorChar, quoteChar,
                CsvConfiguration.DEFAULT_ESCAPE_CHAR));
    }

    /**
     * @deprecated use {@link #CsvDataContext(URL, CsvConfiguration)} instead.
     */
    @Deprecated
    public CsvDataContext(URL url, char separatorChar, char quoteChar) {
        this(url, separatorChar, quoteChar, FileHelper.DEFAULT_ENCODING);
    }

    /**
     * @deprecated use {@link #CsvDataContext(URL, CsvConfiguration)} instead.
     */
    @Deprecated
    public CsvDataContext(URL url, char separatorChar, char quoteChar, String encoding) {
        this(url, new CsvConfiguration(CsvConfiguration.DEFAULT_COLUMN_NAME_LINE, encoding, separatorChar, quoteChar,
                CsvConfiguration.DEFAULT_ESCAPE_CHAR));
    }

    /**
     * @deprecated use {@link #CsvDataContext(InputStream, CsvConfiguration)}
     *             instead.
     */
    @Deprecated
    public CsvDataContext(InputStream inputStream, char separatorChar, char quoteChar) {
        this(inputStream, new CsvConfiguration(CsvConfiguration.DEFAULT_COLUMN_NAME_LINE, FileHelper.DEFAULT_ENCODING,
                separatorChar, quoteChar, CsvConfiguration.DEFAULT_ESCAPE_CHAR));
    }

    /**
     * @deprecated use {@link #CsvDataContext(InputStream, CsvConfiguration)}
     *             instead.
     */
    @Deprecated
    public CsvDataContext(InputStream inputStream, char separatorChar, char quoteChar, String encoding) {
        this(inputStream, new CsvConfiguration(CsvConfiguration.DEFAULT_COLUMN_NAME_LINE, encoding, separatorChar,
                quoteChar, CsvConfiguration.DEFAULT_ESCAPE_CHAR));
    }

    /**
     * Gets the CSV configuration used
     * 
     * @return a CSV configuration
     */
    public CsvConfiguration getConfiguration() {
        return _configuration;
    }

    /**
     * Gets the CSV file being read
     * 
     * @return a file
     */
    public File getFile() {
        return _file;
    }

    private static File createFileFromInputStream(URL url, String encoding) {
        try {
            return createFileFromInputStream(url.openStream(), encoding);
        } catch (IOException e) {
            logger.error("Could not open URL input stream", e);
            throw new IllegalStateException(e);
        }
    }

    private static File createFileFromInputStream(InputStream inputStream, String encoding) {
        final File file;
        final File tempDir = FileHelper.getTempDir();

        File fileCandidate = null;
        boolean usableName = false;
        int index = 0;

        while (!usableName) {
            index++;
            fileCandidate = new File(tempDir, "metamodel" + index + ".csv");
            usableName = !fileCandidate.exists();
        }
        file = fileCandidate;

        final BufferedWriter writer = FileHelper.getBufferedWriter(file, encoding);
        final BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));

        try {
            file.createNewFile();
            file.deleteOnExit();

            boolean firstLine = true;

            for (String line = reader.readLine(); line != null; line = reader.readLine()) {
                if (firstLine) {
                    firstLine = false;
                } else {
                    writer.write('\n');
                }
                writer.write(line);
            }
        } catch (IOException e) {
            throw new IllegalStateException(e);
        } finally {
            FileHelper.safeClose(writer, reader);
        }

        return file;
    }

    @Override
    protected Number executeCountQuery(Table table, List<FilterItem> whereItems, boolean functionApproximationAllowed) {
        if (!functionApproximationAllowed) {
            return null;
        }

        // calculate approximate count by sampling the file.
        InputStream inputStream = null;
        try {
            final long length = _file.length();
            // read up to 5 megs of the file and approximate number of lines
            // based on that.

            final int sampleSize = (int) Math.min(length, 1024 * 1024 * 5);
            final int chunkSize = Math.min(sampleSize, 1024 * 1024);

            int readSize = 0;
            int newlines = 0;
            int carriageReturns = 0;
            byte[] byteBuffer = new byte[chunkSize];
            char[] charBuffer = new char[chunkSize];
            inputStream = new FileInputStream(_file);

            while (readSize < sampleSize) {
                final int read = inputStream.read(byteBuffer);
                if (read == -1) {
                    break;
                } else {
                    readSize += read;
                }

                Reader reader = new InputStreamReader(new ByteArrayInputStream(byteBuffer),
                        _configuration.getEncoding());
                reader.read(charBuffer);
                for (char c : charBuffer) {
                    if ('\n' == c) {
                        newlines++;
                    } else if ('\r' == c) {
                        carriageReturns++;
                    }
                }
            }

            int lines = Math.max(newlines, carriageReturns);

            logger.info("Found {} lines breaks in {} bytes", lines, sampleSize);

            long approxCount = (long) (lines * length / sampleSize);
            return approxCount;
        } catch (IOException e) {
            logger.error("Unexpected error during COUNT(*) approximation", e);
            throw new IllegalStateException(e);
        } finally {
            FileHelper.safeClose(inputStream);
        }
    }

    @Override
    public DataSet materializeMainSchemaTable(Table table, Column[] columns, int maxRows) {

        final int lineNumber = _configuration.getColumnNameLineNumber() + 1;
        final CSVReader reader = createCsvReader(lineNumber);
        final int columnCount = table.getColumnCount();
        if (maxRows < 0) {
            return new CsvDataSet(reader, columns, null, columnCount, _configuration.isFailOnInconsistentRowLength());
        } else {
            return new CsvDataSet(reader, columns, maxRows, columnCount, _configuration.isFailOnInconsistentRowLength());
        }
    }

    protected CSVReader createCsvReader(int skipLines) {
        final Reader fileReader = FileHelper.getReader(_file, _configuration.getEncoding());
        final CSVReader csvReader = new CSVReader(fileReader, _configuration.getSeparatorChar(),
                _configuration.getQuoteChar(), _configuration.getEscapeChar(), skipLines);
        return csvReader;
    }

    @Override
    protected CsvSchema getMainSchema() throws MetaModelException {
        CsvSchema schema = new CsvSchema(getMainSchemaName(), this);
        if (_file.exists()) {
            schema.setTable(new CsvTable(schema));
        }
        return schema;
    }

    @Override
    protected String getMainSchemaName() {
        return _file.getName();
    }

    protected boolean isWritable() {
        return _writable;
    }

    private void checkWritable() {
        if (!isWritable()) {
            throw new IllegalStateException(
                    "This CSV DataContext is not writable, as it based on a read-only resource.");
        }
    }

    @Override
    public void executeUpdate(UpdateScript update) {
        checkWritable();
        CsvUpdateCallback callback = new CsvUpdateCallback(this);
        synchronized (WRITE_LOCK) {
            try {
                update.run(callback);
            } finally {
                callback.close();
            }
        }
    }
}