/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2024 Objectionary.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.eolang.jeo;

import com.jcabi.log.Logger;
import java.io.File;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

/**
 * Converts EO to bytecode.
 * In other words, it assembles low-lever representation EO to bytecode that JVM can execute.
 * The mojo that converts EO to bytecode only.
 * It does not apply any improvements.
 *
 * @since 0.1.0
 */
@Mojo(name = "assemble", defaultPhase = LifecyclePhase.PROCESS_CLASSES)
public final class AssembleMojo extends AbstractMojo {

    /**
     * Maven project.
     *
     * @since 0.2
     */
    @Parameter(defaultValue = "${project}", required = true, readonly = true)
    private MavenProject project;

    /**
     * Source directory.
     *
     * @since 0.2.0
     * @checkstyle MemberNameCheck (6 lines)
     */
    @Parameter(
        property = "jeo.assemble.sourcesDir",
        defaultValue = "${project.build.directory}/generated-sources/jeo-xmir"
    )
    private File sourcesDir;

    /**
     * Target directory.
     *
     * @since 0.2.0
     * @checkstyle MemberNameCheck (6 lines)
     */
    @Parameter(
        property = "jeo.assemble.outputDir",
        defaultValue = "${project.build.outputDirectory}"
    )
    private File outputDir;

    /**
     * Skip bytecode verification.
     * We check all generated bytecode by default.
     * If you want to skip this verification, set this parameter to true.
     *
     * @since 0.2.0
     * @checkstyle MemberNameCheck (6 lines)
     */
    @Parameter(
        property = "jeo.assemble.skip.verification",
        defaultValue = "false"
    )
    private boolean skipVerification;

    /**
     * Whether the plugin is disabled.
     * If it's disabled, then it won't do anything.
     *
     * @since 0.2.0
     * @checkstyle MemberNameCheck (6 lines)
     */
    @Parameter(
        property = "jeo.disassemble.disabled",
        defaultValue = "false"
    )
    private boolean disabled;

    @Override
    public void execute() throws MojoExecutionException {
        try {
            new PluginStartup(this.project).init();
            if (this.disabled) {
                Logger.info(this, "Assemble mojo is disabled. Skipping.");
            } else {
                new Assembler(
                    this.sourcesDir.toPath(),
                    this.outputDir.toPath(),
                    !this.skipVerification
                ).assemble();
            }
        } catch (final DependencyResolutionRequiredException exception) {
            throw new MojoExecutionException(exception);
        }
    }
}
