/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.latch.Latch;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.log.LogBuffer;
import com.sleepycat.je.log.LogStatDefinition;
import com.sleepycat.je.utilint.AtomicLongStat;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.IntStat;
import com.sleepycat.je.utilint.LongStat;
import com.sleepycat.je.utilint.StatGroup;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.LinkedList;

class LogBufferPool {
    private static final String DEBUG_NAME = LogBufferPool.class.getName();
    private EnvironmentImpl envImpl = null;
    private int logBufferSize;
    private LinkedList<LogBuffer> bufferPool;
    private int dirtyStart = -1;
    private int dirtyEnd = -1;
    private LogBuffer currentWriteBuffer;
    private int currentWriteBufferIndex;
    private final FileManager fileManager;
    private final StatGroup stats;
    private final AtomicLongStat nNotResident;
    private final AtomicLongStat nCacheMiss;
    private final IntStat logBuffers;
    private final LongStat nBufferBytes;
    private final LongStat nNoFreeBuffer;
    private final boolean runInMemory;
    private final Latch bufferPoolLatch;
    private volatile long minBufferLsn = 0L;

    LogBufferPool(FileManager fileManager, EnvironmentImpl envImpl) throws DatabaseException {
        this.fileManager = fileManager;
        this.envImpl = envImpl;
        this.bufferPoolLatch = new Latch(DEBUG_NAME + "_FullLatch");
        DbConfigManager configManager = envImpl.getConfigManager();
        this.runInMemory = envImpl.isMemOnly();
        this.reset(configManager);
        this.currentWriteBuffer = this.bufferPool.getFirst();
        this.currentWriteBufferIndex = 0;
        this.stats = new StatGroup("LogBufferPool", "LogBufferPool statistics");
        this.nNotResident = new AtomicLongStat(this.stats, LogStatDefinition.LBFP_NOT_RESIDENT);
        this.nCacheMiss = new AtomicLongStat(this.stats, LogStatDefinition.LBFP_MISS);
        this.logBuffers = new IntStat(this.stats, LogStatDefinition.LBFP_LOG_BUFFERS);
        this.nBufferBytes = new LongStat(this.stats, LogStatDefinition.LBFP_BUFFER_BYTES);
        this.nNoFreeBuffer = new LongStat(this.stats, LogStatDefinition.LBFP_NO_FREE_BUFFER);
    }

    final int getLogBufferSize() {
        return this.logBufferSize;
    }

    void reset(DbConfigManager configManager) throws DatabaseException {
        if (this.runInMemory && this.bufferPool != null) {
            return;
        }
        if (this.currentWriteBuffer != null) {
            this.bumpAndWriteSynced(0, true);
        }
        int numBuffers = configManager.getInt(EnvironmentParams.NUM_LOG_BUFFERS);
        long logBufferBudget = this.envImpl.getMemoryBudget().getLogBufferBudget();
        long logFileSize = configManager.getLong(EnvironmentParams.LOG_FILE_MAX);
        int newBufferSize = (int)logBufferBudget / numBuffers;
        newBufferSize = Math.min(newBufferSize, (int)logFileSize);
        LinkedList<LogBuffer> newPool = new LinkedList<LogBuffer>();
        if (this.runInMemory) {
            numBuffers = 1;
        }
        for (int i = 0; i < numBuffers; ++i) {
            newPool.add(new LogBuffer(newBufferSize, this.envImpl));
        }
        this.bufferPoolLatch.acquire();
        this.bufferPool = newPool;
        this.logBufferSize = newBufferSize;
        this.currentWriteBuffer = this.bufferPool.getFirst();
        this.currentWriteBufferIndex = 0;
        this.bufferPoolLatch.release();
    }

    LogBuffer getWriteBuffer(int sizeNeeded, boolean flippedFile) throws IOException, DatabaseException {
        if (flippedFile) {
            this.bumpAndWriteSynced(sizeNeeded, true);
            if (!this.runInMemory) {
                this.fileManager.syncLogEndAndFinishFile();
            }
        } else if (!(this.currentWriteBuffer.hasRoom(sizeNeeded) || this.bumpCurrent(sizeNeeded) && this.currentWriteBuffer.hasRoom(sizeNeeded))) {
            this.bumpAndWriteSynced(sizeNeeded, false);
        }
        return this.currentWriteBuffer;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    void bumpAndWriteSynced(int sizeNeeded, boolean flushRequired) {
        if (!this.bumpCurrent(sizeNeeded)) {
            this.writeDirty(flushRequired);
            if (!this.bumpCurrent(sizeNeeded)) throw EnvironmentFailureException.unexpectedState(this.envImpl, "No free log buffers.");
            this.writeDirty(flushRequired);
            return;
        } else {
            this.writeDirty(flushRequired);
        }
    }

    private int getNextSlot(int slotNumber) {
        return slotNumber < this.bufferPool.size() - 1 ? ++slotNumber : 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void writeDirty(boolean flushRequired) {
        this.bufferPoolLatch.acquire();
        try {
            if (this.dirtyStart < 0) {
                Object var7_2 = null;
                this.bufferPoolLatch.releaseIfOwner();
                return;
            }
            boolean process = true;
            do {
                Object var5_7;
                LogBuffer lb = this.bufferPool.get(this.dirtyStart);
                lb.waitForZeroAndLatch();
                try {
                    this.writeBufferToFile(lb, flushRequired);
                    var5_7 = null;
                    lb.release();
                }
                catch (Throwable throwable) {
                    var5_7 = null;
                    lb.release();
                    throw throwable;
                }
                if (this.dirtyStart == this.dirtyEnd) {
                    process = false;
                    continue;
                }
                this.dirtyStart = this.getNextSlot(this.dirtyStart);
            } while (process);
            this.dirtyStart = -1;
            this.dirtyEnd = -1;
        }
        catch (Throwable throwable) {
            Object var7_4 = null;
            this.bufferPoolLatch.releaseIfOwner();
            throw throwable;
        }
        Object var7_3 = null;
        this.bufferPoolLatch.releaseIfOwner();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeBufferToFile(LogBuffer latchedBuffer, boolean flushRequired) {
        if (this.runInMemory) {
            return;
        }
        try {
            ByteBuffer currentByteBuffer = latchedBuffer.getDataBuffer();
            int savePosition = currentByteBuffer.position();
            int saveLimit = currentByteBuffer.limit();
            currentByteBuffer.flip();
            try {
                this.fileManager.writeLogBuffer(latchedBuffer, flushRequired);
            }
            catch (Throwable t) {
                currentByteBuffer.position(savePosition);
                currentByteBuffer.limit(saveLimit);
                if (FileManager.continueAfterWriteException() && t instanceof RuntimeException) {
                    throw (RuntimeException)t;
                }
                if (t instanceof EnvironmentFailureException) {
                    if (!this.envImpl.isValid()) {
                        throw (EnvironmentFailureException)t;
                    }
                    throw EnvironmentFailureException.unexpectedException(this.envImpl, (Exception)((EnvironmentFailureException)t));
                }
                if (t instanceof Error) {
                    this.envImpl.invalidate((Error)t);
                    throw (Error)t;
                }
                if (t instanceof Exception) {
                    throw EnvironmentFailureException.unexpectedException(this.envImpl, (Exception)t);
                }
                throw EnvironmentFailureException.unexpectedException(this.envImpl, t.getMessage(), null);
            }
            Object var8_7 = null;
            if (latchedBuffer != null) {
                latchedBuffer.release();
            }
        }
        catch (Throwable throwable) {
            Object var8_8 = null;
            if (latchedBuffer != null) {
                latchedBuffer.release();
            }
            throw throwable;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean bumpCurrent(int sizeNeeded) {
        LogBuffer latchedBuffer;
        block8: {
            block7: {
                block6: {
                    this.bufferPoolLatch.acquire();
                    this.currentWriteBuffer.latchForWrite();
                    latchedBuffer = this.currentWriteBuffer;
                    try {
                        if (this.currentWriteBuffer.getFirstLsn() != -1L) break block6;
                        boolean bl = true;
                        Object var7_7 = null;
                        latchedBuffer.release();
                        this.bufferPoolLatch.releaseIfOwner();
                        return bl;
                    }
                    catch (Throwable throwable) {
                        Object var7_11 = null;
                        latchedBuffer.release();
                        this.bufferPoolLatch.releaseIfOwner();
                        throw throwable;
                    }
                }
                if (!this.runInMemory) break block7;
                int bufferSize = this.logBufferSize > sizeNeeded ? this.logBufferSize : sizeNeeded;
                this.currentWriteBuffer = new LogBuffer(bufferSize, this.envImpl);
                this.bufferPool.add(this.currentWriteBuffer);
                this.currentWriteBufferIndex = this.bufferPool.size() - 1;
                boolean bl = true;
                Object var7_8 = null;
                latchedBuffer.release();
                this.bufferPoolLatch.releaseIfOwner();
                return bl;
            }
            if (this.dirtyStart < 0) {
                this.dirtyStart = this.currentWriteBufferIndex;
                break block8;
            }
            if (this.getNextSlot(this.currentWriteBufferIndex) != this.dirtyStart) break block8;
            this.nNoFreeBuffer.increment();
            boolean bufferSize = false;
            Object var7_9 = null;
            latchedBuffer.release();
            this.bufferPoolLatch.releaseIfOwner();
            return bufferSize;
        }
        this.dirtyEnd = this.currentWriteBufferIndex;
        this.currentWriteBufferIndex = this.getNextSlot(this.currentWriteBufferIndex);
        LogBuffer nextToUse = this.bufferPool.get(this.currentWriteBufferIndex);
        LogBuffer newInitialBuffer = this.bufferPool.get(this.getNextSlot(this.currentWriteBufferIndex));
        nextToUse.reinit();
        this.currentWriteBuffer = nextToUse;
        this.updateMinBufferLsn(newInitialBuffer);
        boolean bl = true;
        Object var7_10 = null;
        latchedBuffer.release();
        this.bufferPoolLatch.releaseIfOwner();
        return bl;
    }

    private void updateMinBufferLsn(LogBuffer newInitialBuffer) {
        long newMinLsn = newInitialBuffer.getFirstLsn();
        if (newMinLsn != -1L) {
            this.minBufferLsn = newMinLsn;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    LogBuffer getReadBufferByLsn(long lsn) throws DatabaseException {
        LogBuffer logBuffer;
        this.nNotResident.increment();
        if (DbLsn.compareTo(lsn, this.minBufferLsn) < 0) {
            this.nCacheMiss.increment();
            return null;
        }
        this.bufferPoolLatch.acquire();
        try {
            for (LogBuffer l : this.bufferPool) {
                if (!l.containsLsn(lsn)) continue;
                LogBuffer logBuffer2 = l;
                Object var7_5 = null;
                this.bufferPoolLatch.releaseIfOwner();
                return logBuffer2;
            }
            this.nCacheMiss.increment();
            logBuffer = null;
        }
        catch (Throwable throwable) {
            Object var7_7 = null;
            this.bufferPoolLatch.releaseIfOwner();
            throw throwable;
        }
        Object var7_6 = null;
        this.bufferPoolLatch.releaseIfOwner();
        return logBuffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    StatGroup loadStats(StatsConfig config) throws DatabaseException {
        this.bufferPoolLatch.acquire();
        long bufferBytes = 0L;
        int nLogBuffers = 0;
        try {
            for (LogBuffer l : this.bufferPool) {
                ++nLogBuffers;
                bufferBytes += (long)l.getCapacity();
            }
            Object var8_6 = null;
            this.bufferPoolLatch.release();
        }
        catch (Throwable throwable) {
            Object var8_7 = null;
            this.bufferPoolLatch.release();
            throw throwable;
        }
        this.logBuffers.set(nLogBuffers);
        this.nBufferBytes.set(bufferBytes);
        return this.stats.cloneGroup(config.getClear());
    }

    public long getNCacheMiss() {
        return this.nCacheMiss.get();
    }

    public StatGroup getBufferPoolLatchStats() {
        return this.bufferPoolLatch.getLatchStats();
    }
}

