/*
 * Copyright (C) 2006-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */

package org.filesys.smb.server.notify;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.filesys.server.filesys.NetworkFile;
import org.filesys.server.filesys.NotifyChange;
import org.filesys.smb.server.SMBSrvSession;


/**
 * Notify Change Request List Class
 *
 * @author gkspencer
 */
public class NotifyRequestList {

    //	List of notify change requests
    private List<NotifyRequest> m_requests;

    /**
     * Default constructor
     */
    public NotifyRequestList() {
        m_requests = new ArrayList<NotifyRequest>();
    }

    /**
     * Return the specified request
     *
     * @param idx int
     * @return NotifyRequest
     */
    public final synchronized NotifyRequest getRequest(int idx) {

        //	Range check the index
        if (idx >= m_requests.size())
            return null;

        //	Return the notify request
        return m_requests.get(idx);
    }

    /**
     * Return the global filter mask, generated by combining all of the pending notify request filters
     *
     * @return Set of NotifyChange
     */
    public final synchronized Set<NotifyChange> getGlobalFilter() {

        //	Loop through all the requests
        Set<NotifyChange> filter = EnumSet.noneOf( NotifyChange.class);

        if (m_requests.size() > 0) {

            //	Build the global filter mask from all pending requests
            for (int i = 0; i < m_requests.size(); i++) {
                NotifyRequest req = m_requests.get(i);
                filter.addAll( req.getFilter());
            }
        }

        //	Return the filter mask
        return filter;
    }

    /**
     * Add a request to the list
     *
     * @param req NotifyRequest
     */
    public final synchronized void addRequest(NotifyRequest req) {
        m_requests.add(req);
    }

    /**
     * Find the notify request for the matching ids
     *
     * @param mid int
     * @param tid int
     * @param uid int
     * @param pid int
     * @return NotifyRequest
     */
    public final synchronized NotifyRequest findRequest(int mid, int tid, int uid, int pid) {

        //	Search for the required request
        for (int i = 0; i < m_requests.size(); i++) {

            //	Get the current request
            NotifyRequest curReq = m_requests.get(i);

            if (curReq.getId() == mid &&
                    curReq.getTreeId() == tid &&
                    curReq.getUserId() == uid &&
                    curReq.getProcessId() == pid) {

                //	Return the request
                return curReq;
            }
        }

        //	Request not found in the list
        return null;
    }

    /**
     * Find the notify request for the matching id
     *
     * @param mid long
     * @return NotifyRequest
     */
    public final synchronized NotifyRequest findRequest(long mid) {

        //	Search for the required request
        for (int i = 0; i < m_requests.size(); i++) {

            //	Get the current request
            NotifyRequest curReq = m_requests.get(i);

            if (curReq.getId() == mid) {

                //	Return the request
                return curReq;
            }
        }

        //	Request not found in the list
        return null;
    }

    /**
     * Find the notify request for the specified directory and filter
     *
     * @param dir       NetworkFile
     * @param filter    Set of NotifyChange
     * @param watchTree boolean
     * @return NotifyRequest
     */
    public final synchronized NotifyRequest findRequest(NetworkFile dir, Set<NotifyChange> filter, boolean watchTree) {

        //	Search for the required request
        for (int i = 0; i < m_requests.size(); i++) {

            //	Get the current request
            NotifyRequest curReq = m_requests.get(i);
            if (curReq.getDirectory() == dir &&
                    curReq.getFilter() == filter &&
                    curReq.hasWatchTree() == watchTree) {

                //	Return the request
                return curReq;
            }
        }

        //	Request not found in the list
        return null;
    }

    /**
     * Remove a request from the list
     *
     * @param req NotifyRequest
     * @return NotifyRequest
     */
    public final synchronized NotifyRequest removeRequest(NotifyRequest req) {

        //	Search for the required request, and remove it from the list
        for (int i = 0; i < m_requests.size(); i++) {

            //	Get the current request
            NotifyRequest curReq = m_requests.get(i);
            if (curReq == req) {

                //	Remove the request from the list
                m_requests.remove(i);
                return curReq;
            }
        }

        //	Request not found in the list
        return null;
    }

    /**
     * Remove a request from the list
     *
     * @param idx int
     * @return NotifyRequest
     */
    public final synchronized NotifyRequest removeRequestAt(int idx) {

        //	Check if the request index is valid
        if (idx < 0 || idx >= m_requests.size())
            return null;

        //	Remove the specified request
        return m_requests.remove(idx);
    }

    /**
     * Remove all requests for the specified session
     *
     * @param sess SMBSrvSession
     */
    public final synchronized void removeAllRequestsForSession(SMBSrvSession sess) {

        //	Search for the required requests, and remove from the list
        int idx = 0;

        while (idx < m_requests.size()) {

            //	Get the current request
            NotifyRequest curReq = m_requests.get(idx);
            if (curReq.getSession() == sess) {

                //	Remove the request from the list
                m_requests.remove(idx);
            }
            else
                idx++;
        }
    }

    /**
     * Remove all requests for the specified session and tree connection
     *
     * @param sess SMBSrvSession
     * @param tid  int
     */
    public final synchronized void removeAllRequestsForSession(SMBSrvSession sess, int tid) {

        //	Search for the required requests, and remove from the list
        int idx = 0;

        while (idx < m_requests.size()) {

            //	Get the current request
            NotifyRequest curReq = m_requests.get(idx);
            if (curReq.getSession() == sess && curReq.getTreeId() == tid) {

                //	Remove the request from the list
                m_requests.remove(idx);
            }
            else
                idx++;
        }
    }

    /**
     * Remove all requests from the list
     */
    public final synchronized void clearRequestList() {
        m_requests.clear();
    }

    /**
     * Return the request list size
     *
     * @return int
     */
    public final synchronized int numberOfRequests() {
        return m_requests.size();
    }
}
