/*
 * Decompiled with CFR 0.152.
 */
package org.finos.legend.sdlc.entities;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;
import org.eclipse.collections.api.RichIterable;
import org.eclipse.collections.api.factory.Lists;
import org.eclipse.collections.api.list.MutableList;
import org.eclipse.collections.impl.utility.Iterate;
import org.finos.legend.sdlc.domain.model.entity.Entity;
import org.finos.legend.sdlc.serialization.EntitySerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntityReserializer {
    private static final Logger LOGGER = LoggerFactory.getLogger(EntityReserializer.class);
    private final EntitySerializer sourceSerializer;
    private final EntitySerializer targetSerializer;
    private final String targetFileExtension;

    private EntityReserializer(EntitySerializer sourceSerializer, EntitySerializer targetSerializer, String targetFileExtension) {
        this.sourceSerializer = sourceSerializer;
        this.targetSerializer = targetSerializer;
        this.targetFileExtension = targetFileExtension == null ? this.targetSerializer.getDefaultFileExtension() : targetFileExtension;
    }

    public List<String> reserializeDirectoryTree(Path sourceDirectory, Path targetDirectory) throws IOException {
        return this.reserializeDirectoryTree(sourceDirectory, null, targetDirectory);
    }

    public List<String> reserializeDirectoryTree(Path sourceDirectory, Predicate<? super Path> filter, Path targetDirectory) throws IOException {
        if (Files.notExists(sourceDirectory, new LinkOption[0])) {
            LOGGER.debug("Source directory {} does not exist: no entities reserialized to {}", (Object)sourceDirectory, (Object)targetDirectory);
            return Collections.emptyList();
        }
        Path targetEntitiesDirectory = targetDirectory.resolve("entities");
        MutableList entityPaths = Lists.mutable.empty();
        ArrayDeque<Path> directories = new ArrayDeque<Path>();
        directories.add(sourceDirectory);
        while (!directories.isEmpty()) {
            Path directory = (Path)directories.removeFirst();
            DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory);
            try {
                for (Path entry : dirStream) {
                    if (Files.isDirectory(entry, new LinkOption[0])) {
                        directories.addLast(entry);
                        continue;
                    }
                    if (filter != null && !filter.test(entry)) continue;
                    entityPaths.add(this.reserializeFile(entry, targetEntitiesDirectory));
                }
            }
            finally {
                if (dirStream == null) continue;
                dirStream.close();
            }
        }
        return entityPaths;
    }

    public Predicate<Path> getDefaultExtensionFilter() {
        return EntityReserializer.getExtensionFilter(this.sourceSerializer.getDefaultFileExtension());
    }

    private String reserializeFile(Path sourceFile, Path targetDirectory) throws IOException {
        Entity entity;
        LOGGER.debug("Reading {}", (Object)sourceFile);
        try (BufferedInputStream inputStream = new BufferedInputStream(Files.newInputStream(sourceFile, new OpenOption[0]));){
            entity = this.sourceSerializer.deserialize((InputStream)inputStream);
        }
        catch (Exception e) {
            StringBuilder builder = new StringBuilder("Error deserializing entity from ").append(sourceFile);
            String eMessage = e.getMessage();
            if (eMessage != null) {
                builder.append(": ").append(eMessage);
            }
            String message = builder.toString();
            LOGGER.debug(message, (Throwable)e);
            if (e instanceof IOException) {
                throw new IOException(message, e);
            }
            throw new RuntimeException(message, e);
        }
        LOGGER.debug("Finished reading {} from {}", (Object)entity.getPath(), (Object)sourceFile);
        Path targetFile = this.targetSerializer.filePathForEntity(entity, targetDirectory, this.targetFileExtension);
        LOGGER.debug("Writing {} to {}", (Object)entity.getPath(), (Object)targetFile);
        Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
        try (BufferedOutputStream outputStream = new BufferedOutputStream(Files.newOutputStream(targetFile, StandardOpenOption.CREATE_NEW));){
            this.targetSerializer.serialize(entity, (OutputStream)outputStream);
        }
        catch (Exception e) {
            StringBuilder builder = new StringBuilder("Error serializing entity '").append(entity.getPath()).append("' to ").append(targetFile);
            if (e instanceof FileAlreadyExistsException) {
                builder.append(": target file already exists");
            } else {
                String eMessage = e.getMessage();
                if (eMessage != null) {
                    builder.append(": ").append(eMessage);
                }
            }
            String message = builder.toString();
            LOGGER.debug(message, (Throwable)e);
            if (e instanceof IOException) {
                throw new IOException(message, e);
            }
            throw new RuntimeException(message, e);
        }
        LOGGER.debug("Finished writing {} to {}", (Object)entity.getPath(), (Object)targetFile);
        return entity.getPath();
    }

    public static EntityReserializer newReserializer(EntitySerializer sourceSerializer, EntitySerializer targetSerializer, String targetFileExtension) {
        return new EntityReserializer(sourceSerializer, targetSerializer, targetFileExtension);
    }

    public static EntityReserializer newReserializer(EntitySerializer sourceSerializer, EntitySerializer targetSerializer) {
        return EntityReserializer.newReserializer(sourceSerializer, targetSerializer, null);
    }

    public static Predicate<Path> getExtensionFilter(String extension) {
        String canonicalExtension = EntityReserializer.canonicalizeFileExtension(extension);
        return canonicalExtension == null ? EntityReserializer::hasNoExtension : p -> EntityReserializer.hasExtension(p, canonicalExtension);
    }

    public static Predicate<Path> getExtensionsFilter(String ... extensions) {
        return EntityReserializer.getExtensionsFilter(Arrays.asList(extensions));
    }

    public static Predicate<Path> getExtensionsFilter(Collection<? extends String> extensions) {
        switch (extensions.size()) {
            case 0: {
                return p -> false;
            }
            case 1: {
                return EntityReserializer.getExtensionFilter((String)Iterate.getFirst(extensions));
            }
        }
        MutableList canonicalExtensions = (MutableList)Iterate.collect(extensions, EntityReserializer::canonicalizeFileExtension, (Collection)Lists.mutable.ofInitialCapacity(extensions.size()));
        return p -> EntityReserializer.hasAnyExtension(p, (RichIterable<String>)canonicalExtensions);
    }

    private static boolean hasNoExtension(Path path) {
        return EntityReserializer.hasNoExtension(path.getFileName().toString());
    }

    private static boolean hasNoExtension(String fileName) {
        return fileName.lastIndexOf(46) == -1;
    }

    private static boolean hasExtension(Path path, String canonicalExtension) {
        return EntityReserializer.hasExtension(path.getFileName().toString(), canonicalExtension);
    }

    private static boolean hasExtension(String fileName, String canonicalExtension) {
        return fileName.endsWith(canonicalExtension);
    }

    private static boolean hasAnyExtension(Path path, RichIterable<String> canonicalExtensions) {
        String fileName = path.getFileName().toString();
        return canonicalExtensions.anySatisfy((org.eclipse.collections.api.block.predicate.Predicate & Serializable)ex -> ex == null ? EntityReserializer.hasNoExtension(fileName) : EntityReserializer.hasExtension(fileName, ex));
    }

    private static String canonicalizeFileExtension(String extension) {
        if (extension == null || extension.isEmpty()) {
            return null;
        }
        if (extension.charAt(0) != '.') {
            return "." + extension;
        }
        return extension;
    }
}

