/*
 * Decompiled with CFR 0.152.
 */
package org.finos.legend.sdlc.serialization;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Spliterators;
import java.util.function.Function;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.finos.legend.sdlc.domain.model.entity.Entity;
import org.finos.legend.sdlc.serialization.EntitySerializer;
import org.finos.legend.sdlc.serialization.EntitySerializers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntityLoader
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(EntityLoader.class);
    private static final EntitySerializer ENTITY_SERIALIZER = EntitySerializers.getDefaultJsonSerializer();
    private static final String ENTITIES_DIRECTORY = "entities";
    private static final String ENTITY_FILE_EXTENSION = "." + ENTITY_SERIALIZER.getDefaultFileExtension();
    private final List<EntityFileSearch> searchList;

    private EntityLoader(List<EntityFileSearch> searchList) {
        this.searchList = searchList;
    }

    public Entity getEntity(String entityPath) {
        String entityFilePath = EntityLoader.entityPathToFilePath(entityPath);
        return this.searchList.stream().map(s -> s.getPath(entityFilePath)).filter(EntityLoader::isPossiblyEntityFile).map(EntityLoader::readEntity).filter(Objects::nonNull).findFirst().orElse(null);
    }

    public Stream<Entity> getAllEntities() {
        try {
            return this.getEntitiesInDirectory(ENTITIES_DIRECTORY);
        }
        catch (Exception e) {
            StringBuilder builder = new StringBuilder("Error getting all entities");
            String eMessage = e.getMessage();
            if (eMessage != null) {
                builder.append(": ").append(eMessage);
            }
            throw new RuntimeException(builder.toString(), e);
        }
    }

    public Stream<Entity> getEntitiesInPackage(String packagePath) {
        try {
            return this.getEntitiesInDirectory(EntityLoader.packagePathToDirectoryPath(packagePath));
        }
        catch (Exception e) {
            StringBuilder builder = new StringBuilder("Error getting all entities from package '").append(packagePath).append('\'');
            String eMessage = e.getMessage();
            if (eMessage != null) {
                builder.append(": ").append(eMessage);
            }
            throw new RuntimeException(builder.toString(), e);
        }
    }

    @Override
    public synchronized void close() throws Exception {
        Exception exception = null;
        for (AutoCloseable autoCloseable : this.searchList) {
            try {
                autoCloseable.close();
            }
            catch (Exception e) {
                if (exception == null) {
                    exception = e;
                    continue;
                }
                try {
                    exception.addSuppressed(e);
                }
                catch (Exception ee) {
                    LOGGER.debug("Error adding suppressed exception", (Throwable)ee);
                }
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    private Stream<Entity> getEntitiesInDirectory(String directoryPath) {
        return this.searchList.stream().flatMap(s -> s.getPathsInDirectory(directoryPath)).filter(EntityLoader::isPossiblyEntityFile).map(EntityLoader::readEntity).filter(Objects::nonNull);
    }

    public static EntityLoader newEntityLoader(ClassLoader classLoader) {
        return new EntityLoader(Collections.singletonList(new ClassLoaderEntityFileSearch(classLoader)));
    }

    public static EntityLoader newEntityLoader(Path path) {
        EntityFileSearch search = EntityLoader.newPathEntityFileSearch(path);
        return new EntityLoader(search == null ? Collections.emptyList() : Collections.singletonList(search));
    }

    public static EntityLoader newEntityLoader(File path) {
        return EntityLoader.newEntityLoader(path.toPath());
    }

    public static EntityLoader newEntityLoader(URI uri) {
        EntityFileSearch search = EntityLoader.newURIEntityFileSearch(uri);
        return new EntityLoader(search == null ? Collections.emptyList() : Collections.singletonList(search));
    }

    public static EntityLoader newEntityLoader(Path ... paths) {
        return EntityLoader.newEntityLoader(paths, EntityLoader::newPathEntityFileSearch);
    }

    public static EntityLoader newEntityLoader(File ... files) {
        return EntityLoader.newEntityLoader(files, EntityLoader::newFileEntityFileSearch);
    }

    public static EntityLoader newEntityLoader(URI ... uris) {
        return EntityLoader.newEntityLoader(uris, EntityLoader::newURIEntityFileSearch);
    }

    public static EntityLoader newEntityLoader(ClassLoader classLoader, Path ... paths) {
        return EntityLoader.newEntityLoader(classLoader, paths, EntityLoader::newPathEntityFileSearch);
    }

    public static EntityLoader newEntityLoader(ClassLoader classLoader, File ... files) {
        return EntityLoader.newEntityLoader(classLoader, files, EntityLoader::newFileEntityFileSearch);
    }

    private static <T> EntityLoader newEntityLoader(T[] sources, Function<T, EntityFileSearch> searchFunction) {
        if (sources == null || sources.length == 0) {
            return new EntityLoader(Collections.emptyList());
        }
        if (sources.length == 1) {
            EntityFileSearch search = searchFunction.apply(sources[0]);
            return new EntityLoader(search == null ? Collections.emptyList() : Collections.singletonList(search));
        }
        ArrayList<EntityFileSearch> searchList = new ArrayList<EntityFileSearch>(sources.length);
        Arrays.stream(sources).map(searchFunction).filter(Objects::nonNull).forEach(searchList::add);
        return new EntityLoader(searchList);
    }

    private static <T> EntityLoader newEntityLoader(ClassLoader classLoader, T[] sources, Function<T, EntityFileSearch> searchFunction) {
        if (sources == null || sources.length == 0) {
            return EntityLoader.newEntityLoader(classLoader);
        }
        ArrayList<EntityFileSearch> searchList = new ArrayList<EntityFileSearch>(sources.length + 1);
        searchList.add(new ClassLoaderEntityFileSearch(classLoader));
        Arrays.stream(sources).map(searchFunction).filter(Objects::nonNull).forEach(searchList::add);
        return new EntityLoader(searchList);
    }

    private static Entity readEntity(Path path) {
        Entity entity;
        block8: {
            InputStream stream = Files.newInputStream(path, new OpenOption[0]);
            try {
                entity = ENTITY_SERIALIZER.deserialize(stream);
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    LOGGER.error("Error reading entity from file: {}", (Object)path, (Object)e);
                    return null;
                }
            }
            stream.close();
        }
        return entity;
    }

    private static String entityPathToFilePath(String entityPath) {
        StringBuilder builder = new StringBuilder(ENTITIES_DIRECTORY.length() + entityPath.length() + ENTITY_FILE_EXTENSION.length());
        return EntityLoader.writePackageablePathAsFilePath(builder.append(ENTITIES_DIRECTORY), entityPath).append(ENTITY_FILE_EXTENSION).toString();
    }

    private static String packagePathToDirectoryPath(String packagePath) {
        if ("::".equals(packagePath)) {
            return ENTITIES_DIRECTORY;
        }
        StringBuilder builder = new StringBuilder(ENTITIES_DIRECTORY.length() + packagePath.length());
        return EntityLoader.writePackageablePathAsFilePath(builder.append(ENTITIES_DIRECTORY), packagePath).toString();
    }

    private static StringBuilder writePackageablePathAsFilePath(StringBuilder builder, String packageablePath) {
        int current = 0;
        int nextDelim = packageablePath.indexOf(58);
        while (nextDelim != -1) {
            builder.append('/').append(packageablePath, current, nextDelim);
            current = nextDelim + 2;
            nextDelim = packageablePath.indexOf(58, current);
        }
        return builder.append('/').append(packageablePath, current, packageablePath.length());
    }

    private static boolean isPossiblyEntityFile(Path path) {
        return path != null && EntityLoader.isPossiblyEntityFileName(path.toString()) && Files.isRegularFile(path, new LinkOption[0]);
    }

    private static boolean isPossiblyEntityFileName(String name) {
        return name != null && name.regionMatches(true, name.length() - ENTITY_FILE_EXTENSION.length(), ENTITY_FILE_EXTENSION, 0, ENTITY_FILE_EXTENSION.length());
    }

    private static Stream<Path> getDirectoryStream(Path dirPath) {
        try {
            return Files.walk(dirPath, FileVisitOption.FOLLOW_LINKS);
        }
        catch (IOException ignore) {
            return Stream.empty();
        }
    }

    private static Path getPathFromURL(URL url) {
        try {
            return EntityLoader.getPathFromURI(url.toURI());
        }
        catch (URISyntaxException e) {
            LOGGER.error("Error converting URL to URI: {}", (Object)url, (Object)e);
            return null;
        }
    }

    private static Path getPathFromURI(URI uri) {
        try {
            return EntityLoader.getOrCreateFileSystem(uri).provider().getPath(uri);
        }
        catch (Exception e) {
            LOGGER.error("Error getting path from URI: {}", (Object)uri, (Object)e);
            return null;
        }
    }

    private static FileSystem getOrCreateFileSystem(URI uri) throws IOException {
        if ("file".equalsIgnoreCase(uri.getScheme())) {
            return FileSystems.getDefault();
        }
        try {
            return FileSystems.getFileSystem(uri);
        }
        catch (FileSystemNotFoundException ignore) {
            try {
                return FileSystems.newFileSystem(uri, Collections.emptyMap());
            }
            catch (FileSystemAlreadyExistsException ignoreAlso) {
                return FileSystems.getFileSystem(uri);
            }
        }
    }

    private static EntityFileSearch newFileEntityFileSearch(File file) {
        return EntityLoader.newPathEntityFileSearch(file.toPath());
    }

    private static EntityFileSearch newPathEntityFileSearch(Path path) {
        try {
            BasicFileAttributes attributes;
            try {
                attributes = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
            }
            catch (NoSuchFileException e) {
                return null;
            }
            if (attributes.isDirectory()) {
                return new DirectoryEntityFileSearch(path);
            }
            FileSystem fs = FileSystems.newFileSystem(path, EntityLoader.class.getClassLoader());
            return new DirectoryEntityFileSearchWithCloseable(fs.getPath(fs.getSeparator(), new String[0]), fs);
        }
        catch (Exception e) {
            StringBuilder builder = new StringBuilder("Error handling ").append(path);
            String eMessage = e.getMessage();
            if (eMessage != null) {
                builder.append(": ").append(eMessage);
            }
            throw new RuntimeException(builder.toString(), e);
        }
    }

    private static EntityFileSearch newURIEntityFileSearch(URI uri) {
        try {
            boolean fileSystemShouldBeClosed;
            FileSystem fileSystem;
            if ("file".equalsIgnoreCase(uri.getScheme())) {
                return EntityLoader.newPathEntityFileSearch(Paths.get(uri));
            }
            try {
                fileSystem = FileSystems.getFileSystem(uri);
                fileSystemShouldBeClosed = false;
            }
            catch (FileSystemNotFoundException ignore) {
                try {
                    fileSystem = FileSystems.newFileSystem(uri, Collections.emptyMap());
                    fileSystemShouldBeClosed = true;
                }
                catch (FileSystemAlreadyExistsException ignoreAlso) {
                    fileSystem = FileSystems.getFileSystem(uri);
                    fileSystemShouldBeClosed = false;
                }
            }
            try {
                Path path = fileSystem.provider().getPath(uri);
                return fileSystemShouldBeClosed ? new DirectoryEntityFileSearchWithCloseable(path, fileSystem) : new DirectoryEntityFileSearch(path);
            }
            catch (Exception e) {
                if (fileSystemShouldBeClosed) {
                    try {
                        fileSystem.close();
                    }
                    catch (Exception suppress) {
                        e.addSuppressed(suppress);
                    }
                }
                throw e;
            }
        }
        catch (Exception e) {
            StringBuilder builder = new StringBuilder("Error handling ").append(uri);
            String eMessage = e.getMessage();
            if (eMessage != null) {
                builder.append(": ").append(eMessage);
            }
            throw new RuntimeException(builder.toString(), e);
        }
    }

    private static class DirectoryEntityFileSearchWithCloseable
    extends DirectoryEntityFileSearch {
        private final AutoCloseable closeable;

        private DirectoryEntityFileSearchWithCloseable(Path directory, AutoCloseable closeable) {
            super(directory);
            this.closeable = closeable;
        }

        @Override
        public void close() throws Exception {
            this.closeable.close();
        }
    }

    private static class DirectoryEntityFileSearch
    implements EntityFileSearch {
        private final Path directory;

        private DirectoryEntityFileSearch(Path directory) {
            this.directory = directory;
        }

        @Override
        public Path getPath(String filePath) {
            return this.directory.resolve(filePath);
        }

        @Override
        public Stream<Path> getPathsInDirectory(String dirPath) {
            Path resolvedPath = this.directory.resolve(dirPath);
            return Files.isDirectory(resolvedPath, new LinkOption[0]) ? EntityLoader.getDirectoryStream(resolvedPath) : Stream.empty();
        }

        @Override
        public void close() throws Exception {
        }
    }

    private static class ClassLoaderEntityFileSearch
    implements EntityFileSearch {
        private final ClassLoader classLoader;

        private ClassLoaderEntityFileSearch(ClassLoader classLoader) {
            this.classLoader = Objects.requireNonNull(classLoader, "ClassLoader may not be null");
        }

        @Override
        public Path getPath(String filePath) {
            URL url = this.classLoader.getResource(filePath);
            return url == null ? null : EntityLoader.getPathFromURL(url);
        }

        @Override
        public Stream<Path> getPathsInDirectory(String dirPath) {
            Enumeration<URL> urls;
            try {
                urls = this.classLoader.getResources(dirPath);
            }
            catch (IOException ignore) {
                return Stream.empty();
            }
            return StreamSupport.stream(Spliterators.spliteratorUnknownSize(new Iterator<URL>(){

                @Override
                public boolean hasNext() {
                    return urls.hasMoreElements();
                }

                @Override
                public URL next() {
                    return (URL)urls.nextElement();
                }
            }, 0), false).map(x$0 -> EntityLoader.getPathFromURL(x$0)).filter(Objects::nonNull).filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).flatMap(x$0 -> EntityLoader.getDirectoryStream(x$0));
        }

        @Override
        public void close() {
        }
    }

    private static interface EntityFileSearch
    extends AutoCloseable {
        public Path getPath(String var1);

        public Stream<Path> getPathsInDirectory(String var1);
    }
}

