/*
 * Decompiled with CFR 0.152.
 */
package org.finos.tracdap.common.config;

import com.google.protobuf.Message;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.finos.tracdap.common.config.ConfigFormat;
import org.finos.tracdap.common.config.ConfigParser;
import org.finos.tracdap.common.config.IConfigLoader;
import org.finos.tracdap.common.exception.EStartup;
import org.finos.tracdap.common.plugin.IPluginManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigManager {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final IPluginManager plugins;
    private final URI configRootFile;
    private final URI configRootDir;

    public ConfigManager(String configUrl, Path workingDir, IPluginManager plugins) {
        this.plugins = plugins;
        this.configRootFile = this.resolveConfigRootFile(configUrl, workingDir);
        this.configRootDir = this.configRootFile.resolve(".").normalize();
        if ("file".equals(this.configRootDir.getScheme())) {
            this.log.info("Using config root: {}", (Object)Paths.get(this.configRootDir));
        } else {
            this.log.info("Using config root: {}", (Object)this.configRootDir);
        }
    }

    public URI configRoot() {
        return this.configRootDir;
    }

    public String loadConfigFile(String configUrl) {
        if (configUrl == null || configUrl.isBlank()) {
            throw new EStartup("Config URL is missing or blank");
        }
        URI requestedUrl = this.parseUrl(configUrl);
        return this.loadConfigFromUrl(requestedUrl);
    }

    public <TConfig, X extends Message> TConfig loadConfigObject(String configUrl, Class<TConfig> configClass) {
        if (configUrl == null || configUrl.isBlank()) {
            throw new EStartup("Config URL is missing or blank");
        }
        URI requestedUrl = this.parseUrl(configUrl);
        String configData = this.loadConfigFromUrl(requestedUrl);
        ConfigFormat configFormat = ConfigFormat.fromExtension(requestedUrl);
        if (Message.class.isAssignableFrom(configClass)) {
            ByteBuf configBytes = Unpooled.wrappedBuffer((byte[])configData.getBytes(StandardCharsets.UTF_8));
            Class<TConfig> configMsgClass = configClass;
            return ConfigParser.parseConfig(configBytes, configFormat, configMsgClass);
        }
        return ConfigParser.parseStructuredConfig(configData, configFormat, configClass);
    }

    public String loadRootConfigFile() {
        return this.loadConfigFile(this.configRootFile.toString());
    }

    public <TConfig> TConfig loadRootConfigObject(Class<TConfig> configClass) {
        return this.loadConfigObject(this.configRootFile.toString(), configClass);
    }

    private String loadConfigFromUrl(URI requestedUrl) {
        URI resolvedUrl = this.resolveUrl(requestedUrl);
        URI relativeUrl = this.configRootDir.relativize(resolvedUrl);
        this.log.info("Loading config file: {}", (Object)relativeUrl);
        String protocol = resolvedUrl.getScheme();
        IConfigLoader loader = this.configLoaderForProtocol(protocol);
        return loader.loadTextFile(resolvedUrl);
    }

    private URI resolveConfigRootFile(String suppliedConfigUrl, Path workingDir) {
        try {
            if (suppliedConfigUrl == null || suppliedConfigUrl.isBlank()) {
                throw new EStartup("URL for root config file is missing or blank");
            }
            URI configUrl = URI.create(suppliedConfigUrl);
            String configProtocol = configUrl.getScheme();
            if (configProtocol == null || configProtocol.isBlank() || configProtocol.equals("file")) {
                if (configUrl.isAbsolute()) {
                    return configUrl;
                }
                Path configPath = workingDir.resolve(configUrl.getPath());
                return configPath.toUri();
            }
            return configUrl;
        }
        catch (IllegalArgumentException e) {
            String message = String.format("URL for root config file is invalid: [%s]", suppliedConfigUrl);
            throw new EStartup(message);
        }
    }

    private URI parseUrl(String url) {
        if (url.startsWith("/") || url.startsWith("\\")) {
            try {
                return Paths.get(url, new String[0]).toUri().normalize();
            }
            catch (InvalidPathException invalidPathException) {
                // empty catch block
            }
        }
        try {
            return URI.create(url);
        }
        catch (IllegalArgumentException illegalArgumentException) {
            try {
                return Paths.get(url, new String[0]).toUri().normalize();
            }
            catch (InvalidPathException invalidPathException) {
                throw new EStartup("Requested config URL is not a valid URL or path: " + url);
            }
        }
    }

    private URI resolveUrl(URI url) {
        boolean isAbsolute;
        String ERROR_MSG_TEMPLATE = "Invalid URL for config file: %2$s [%1$s]";
        String protocol = url.getScheme();
        String path = url.getPath() != null ? url.getPath() : url.getSchemeSpecificPart();
        boolean bl = isAbsolute = path.startsWith("/") || path.startsWith("\\");
        if (isAbsolute) {
            if (protocol == null || protocol.isBlank()) {
                String message = String.format(ERROR_MSG_TEMPLATE, url, "Absolute URLs must specify an explicit protocol");
                throw new EStartup(message);
            }
        } else if (protocol != null && !protocol.isBlank()) {
            String message = String.format(ERROR_MSG_TEMPLATE, url, "Relative URLs cannot specify an explicit protocol");
            throw new EStartup(message);
        }
        if ("file".equals(protocol) && url.getHost() != null) {
            String message = String.format(ERROR_MSG_TEMPLATE, url, "Network file paths are not supported");
            throw new EStartup(message);
        }
        if (isAbsolute) {
            return url.normalize();
        }
        return this.configRootDir.resolve(url).normalize();
    }

    private IConfigLoader configLoaderForProtocol(String protocol) {
        if (protocol == null || protocol.isBlank()) {
            protocol = "file";
        }
        if (!this.plugins.isServiceAvailable(IConfigLoader.class, protocol)) {
            String message = String.format("No config loader available for protocol [%s]", protocol);
            this.log.error(message);
            throw new EStartup(message);
        }
        return this.plugins.createService(IConfigLoader.class, protocol);
    }
}

