/*
 * Decompiled with CFR 0.152.
 */
package org.finos.tracdap.common.config.local;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Properties;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.finos.tracdap.common.config.ConfigManager;
import org.finos.tracdap.common.config.ISecretLoader;
import org.finos.tracdap.common.exception.EConfigLoad;
import org.finos.tracdap.common.exception.EStartup;
import org.finos.tracdap.common.exception.ETracInternal;
import org.finos.tracdap.common.startup.StartupLog;
import org.slf4j.event.Level;

public class JksSecretLoader
implements ISecretLoader {
    public static final String DEFAULT_KEYSTORE_TYPE = "PKCS12";
    private final Properties properties;
    private KeyStore keystore;
    private boolean ready;
    String secretKey;

    public JksSecretLoader(Properties properties) {
        this.properties = properties;
        this.keystore = null;
        this.ready = false;
    }

    @Override
    public void init(ConfigManager configManager) {
        if (this.ready) {
            StartupLog.log(this, Level.ERROR, "JKS secret loader initialized twice");
            throw new EStartup("JKS secret loader initialized twice");
        }
        String keystoreType = this.properties.getProperty("secret.type", DEFAULT_KEYSTORE_TYPE);
        String keystoreUrl = this.properties.getProperty("secret.url");
        String keystoreKey = this.properties.getProperty("secret.key");
        try {
            StartupLog.log(this, Level.INFO, "Initializing JKS secret loader...");
            if (keystoreUrl == null || keystoreUrl.isBlank()) {
                String message = String.format("JKS secrets need %s in the main config file", "secret.url");
                StartupLog.log(this, Level.ERROR, message);
                throw new EStartup(message);
            }
            if (keystoreKey == null || keystoreKey.isBlank()) {
                String template = "JKS secrets need a secret key, use --secret-key or set %s in the environment";
                String message = String.format(template, "SECRET_KEY");
                StartupLog.log(this, Level.ERROR, message);
                throw new EStartup(message);
            }
            this.keystore = KeyStore.getInstance(keystoreType);
            byte[] keystoreBytes = configManager.loadBinaryConfig(keystoreUrl);
            try (ByteArrayInputStream stream = new ByteArrayInputStream(keystoreBytes);){
                this.keystore.load(stream, keystoreKey.toCharArray());
                this.ready = true;
                this.secretKey = keystoreKey;
            }
        }
        catch (KeyStoreException e) {
            String message = String.format("Keystore type is not supported: [%s]", keystoreType);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        catch (IOException e) {
            Throwable error = e.getCause() != null ? e.getCause() : e;
            String errorDetail = error.getMessage() + " (this normally means the secret key is wrong)";
            String message = String.format("Failed to open keystore [%s]: %s", keystoreUrl, errorDetail);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        catch (NoSuchAlgorithmException | CertificateException e) {
            String message = String.format("Failed to open keystore [%s]: %s", keystoreUrl, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
    }

    @Override
    public String loadPassword(String secretName) {
        try {
            if (!this.ready) {
                StartupLog.log(this, Level.ERROR, "JKS Secret loader has not been initialized");
                throw new ETracInternal("JKS Secret loader has not been initialized");
            }
            KeyStore.Entry entry = this.keystore.getEntry(secretName, new KeyStore.PasswordProtection(this.secretKey.toCharArray()));
            if (entry == null) {
                String message = String.format("Secret is not present in the store: [%s]", secretName);
                StartupLog.log(this, Level.ERROR, message);
                throw new EConfigLoad(message);
            }
            if (!(entry instanceof KeyStore.SecretKeyEntry)) {
                String message = String.format("Secret is not a secret key: [%s] is %s", secretName, entry.getClass().getSimpleName());
                StartupLog.log(this, Level.ERROR, message);
                throw new EConfigLoad(message);
            }
            KeyStore.SecretKeyEntry secret = (KeyStore.SecretKeyEntry)entry;
            String algorithm = secret.getSecretKey().getAlgorithm();
            SecretKeyFactory factory = SecretKeyFactory.getInstance(algorithm);
            Class<PBEKeySpec> keySpecType = PBEKeySpec.class;
            PBEKeySpec keySpec = (PBEKeySpec)factory.getKeySpec(secret.getSecretKey(), keySpecType);
            char[] password = keySpec.getPassword();
            return new String(password);
        }
        catch (GeneralSecurityException e) {
            String message = String.format("Secret could not be retrieved from the key store: [%s] %s", secretName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message, e);
        }
    }
}

