/*
 * Decompiled with CFR 0.152.
 */
package org.finos.tracdap.common.config;

import com.google.protobuf.Message;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Map;
import java.util.Properties;
import org.finos.tracdap.common.config.ConfigFormat;
import org.finos.tracdap.common.config.ConfigParser;
import org.finos.tracdap.common.config.IConfigLoader;
import org.finos.tracdap.common.config.ISecretLoader;
import org.finos.tracdap.common.config.local.JksSecretLoader;
import org.finos.tracdap.common.exception.EConfigLoad;
import org.finos.tracdap.common.exception.EStartup;
import org.finos.tracdap.common.plugin.IPluginManager;
import org.finos.tracdap.common.startup.StartupLog;
import org.finos.tracdap.config._ConfigFile;
import org.slf4j.event.Level;

public class ConfigManager {
    private final IPluginManager plugins;
    private final URI rootConfigFile;
    private final URI rootConfigDir;
    private final String secretKey;
    private ISecretLoader secrets = null;
    private byte[] rootConfigCache = null;

    public ConfigManager(String configUrl, Path workingDir, IPluginManager plugins) {
        this(configUrl, workingDir, plugins, null);
    }

    public ConfigManager(String configUrl, Path workingDir, IPluginManager plugins, String secretKey) {
        this.plugins = plugins;
        this.secretKey = secretKey;
        this.rootConfigFile = this.resolveRootUrl(this.parseUrl(configUrl), workingDir);
        this.rootConfigDir = this.rootConfigFile.resolve(".").normalize();
        String rootDirDisplay = "file".equals(this.rootConfigDir.getScheme()) ? Paths.get(this.rootConfigDir).toString() : this.rootConfigDir.toString();
        StartupLog.log(this, Level.INFO, String.format("Using config root: %s", rootDirDisplay));
    }

    public void prepareSecrets() {
        _ConfigFile rootConfig = this.loadRootConfigObject(_ConfigFile.class, true);
        Map configMap = rootConfig.getConfigMap();
        String secretType = configMap.getOrDefault("secret.type", "");
        String secretUrl = configMap.getOrDefault("secret.url", "");
        if (secretType == null || secretType.isBlank()) {
            StartupLog.log(this, Level.INFO, "Using secrets: [none]");
            return;
        }
        StartupLog.log(this, Level.INFO, String.format("Using secrets: [%s] %s", secretType, secretUrl));
        this.secrets = this.secretLoaderForProtocol(secretType, configMap);
    }

    public ISecretLoader getUserDb() {
        _ConfigFile config = this.loadRootConfigObject(_ConfigFile.class, true);
        if (!config.containsConfig("users.type")) {
            String template = "TRAC user database is not enabled (set config key [%s] to turn it on)";
            String message = String.format(template, "users.type");
            throw new EStartup(message);
        }
        if (!config.containsConfig("users.url")) {
            String message = "Missing required config key [users.url]";
            throw new EStartup(message);
        }
        String userDbType = config.getConfigOrThrow("users.type");
        String userDbUrl = config.getConfigOrThrow("users.url");
        String userDbSecret = config.getConfigOrDefault("users.key", "");
        Path userDbPath = Paths.get(this.resolveConfigFile(URI.create(userDbUrl)));
        String userDbKey = userDbSecret.isEmpty() ? this.secretKey : this.loadPassword(userDbSecret);
        Properties userDbProps = new Properties();
        userDbProps.put("secret.type", userDbType);
        userDbProps.put("secret.url", userDbPath.toString());
        userDbProps.put("secret.key", userDbKey);
        JksSecretLoader userDb = new JksSecretLoader(userDbProps);
        userDb.init(this);
        return userDb;
    }

    public URI configRoot() {
        return this.rootConfigDir;
    }

    public URI resolveConfigFile(URI relativePath) {
        return this.resolveUrl(relativePath);
    }

    public byte[] loadBinaryConfig(String configUrl) {
        URI parsed = this.parseUrl(configUrl);
        URI resolved = this.resolveUrl(parsed);
        return this.loadUrl(resolved);
    }

    public String loadTextConfig(String configUrl) {
        URI parsed = this.parseUrl(configUrl);
        URI resolved = this.resolveUrl(parsed);
        byte[] bytes = this.loadUrl(resolved);
        return new String(bytes, StandardCharsets.UTF_8);
    }

    public <TConfig extends Message> TConfig loadConfigObject(String configUrl, Class<TConfig> configClass, boolean leniency) {
        URI parsed = this.parseUrl(configUrl);
        URI resolved = this.resolveUrl(parsed);
        byte[] bytes = this.loadUrl(resolved);
        ConfigFormat format = ConfigFormat.fromExtension(resolved);
        return ConfigParser.parseConfig(bytes, format, configClass, leniency);
    }

    public <TConfig extends Message> TConfig loadConfigObject(String configUrl, Class<TConfig> configClass) {
        return this.loadConfigObject(configUrl, configClass, false);
    }

    public <TConfig extends Message> TConfig loadRootConfigObject(Class<TConfig> configClass, boolean leniency) {
        byte[] bytes = this.rootConfigCache;
        if (bytes == null) {
            URI parsed = this.parseUrl(this.rootConfigFile.toString());
            URI resolved = this.resolveUrl(parsed);
            this.rootConfigCache = bytes = this.loadUrl(resolved);
        }
        ConfigFormat format = ConfigFormat.fromExtension(this.rootConfigFile);
        return ConfigParser.parseConfig(bytes, format, configClass, leniency);
    }

    public <TConfig extends Message> TConfig loadRootConfigObject(Class<TConfig> configClass) {
        return this.loadRootConfigObject(configClass, false);
    }

    public boolean hasSecret(String secretName) {
        if (this.secrets == null) {
            return false;
        }
        return this.secrets.hasSecret(secretName);
    }

    public String loadPassword(String secretName) {
        if (this.secrets == null) {
            String message = String.format("Secrets are not enabled, to use secrets set secret.type in [%s]", this.rootConfigFile);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        return this.secrets.loadPassword(secretName);
    }

    public PublicKey loadPublicKey(String secretName) {
        if (this.secrets == null) {
            String message = String.format("Secrets are not enabled, to use secrets set secret.type in [%s]", this.rootConfigFile);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        return this.secrets.loadPublicKey(secretName);
    }

    public PrivateKey loadPrivateKey(String secretName) {
        if (this.secrets == null) {
            String message = String.format("Secrets are not enabled, to use secrets set secret.type in [%s]", this.rootConfigFile);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        return this.secrets.loadPrivateKey(secretName);
    }

    private URI parseUrl(String urlString) {
        if (urlString == null || urlString.isBlank()) {
            throw new EConfigLoad("Config URL is missing or blank");
        }
        Path path = null;
        URI url = null;
        try {
            path = Paths.get(urlString, new String[0]).normalize();
        }
        catch (InvalidPathException invalidPathException) {
            // empty catch block
        }
        try {
            url = URI.create(urlString);
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        if ((urlString.startsWith("/") || urlString.startsWith("\\") || urlString.startsWith(":\\", 1)) && path != null) {
            return path.toUri();
        }
        if (url != null) {
            return url;
        }
        if (path != null) {
            return path.toUri();
        }
        throw new EConfigLoad("Requested config URL is not a valid URL or path: " + urlString);
    }

    private URI resolveUrl(URI url) {
        boolean isAbsolute;
        String ERROR_MSG_TEMPLATE = "Invalid URL for config file: %2$s [%1$s]";
        String protocol = url.getScheme();
        String path = url.getPath() != null ? url.getPath() : url.getSchemeSpecificPart();
        boolean bl = isAbsolute = path.startsWith("/") || path.startsWith("\\");
        if (isAbsolute) {
            if (protocol == null || protocol.isBlank()) {
                String message = String.format(ERROR_MSG_TEMPLATE, url, "Absolute URLs must specify an explicit protocol");
                throw new EConfigLoad(message);
            }
        } else if (protocol != null && !protocol.isBlank()) {
            String message = String.format(ERROR_MSG_TEMPLATE, url, "Relative URLs cannot specify an explicit protocol");
            throw new EConfigLoad(message);
        }
        if ("file".equals(protocol) && url.getHost() != null) {
            String message = String.format(ERROR_MSG_TEMPLATE, url, "Network file paths are not supported");
            throw new EConfigLoad(message);
        }
        if (isAbsolute) {
            return url.normalize();
        }
        return this.rootConfigDir.resolve(url).normalize();
    }

    private URI resolveRootUrl(URI url, Path workingDir) {
        String protocol = url.getScheme();
        if (protocol == null || protocol.isBlank() || protocol.equals("file")) {
            if (url.isAbsolute()) {
                return url;
            }
            Path configPath = workingDir.resolve(url.getPath());
            return configPath.toUri();
        }
        if (!url.isAbsolute()) {
            String message = String.format("Relative URL is not allowed for root config file with protocol [%s]: [%s]", protocol, url);
            throw new EConfigLoad(message);
        }
        return url;
    }

    private byte[] loadUrl(URI absoluteUrl) {
        URI relativeUrl = this.rootConfigDir.relativize(absoluteUrl);
        String message = String.format("Loading config file: [%s]", relativeUrl);
        StartupLog.log(this, Level.INFO, message);
        String protocol = absoluteUrl.getScheme();
        IConfigLoader loader = this.configLoaderForProtocol(protocol);
        return loader.loadBinaryFile(absoluteUrl);
    }

    private IConfigLoader configLoaderForProtocol(String protocol) {
        if (protocol == null || protocol.isBlank()) {
            protocol = "file";
        }
        if (!this.plugins.isServiceAvailable(IConfigLoader.class, protocol)) {
            String message = String.format("No config loader available for protocol [%s]", protocol);
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message);
        }
        return this.plugins.createConfigService(IConfigLoader.class, protocol, new Properties());
    }

    private ISecretLoader secretLoaderForProtocol(String protocol, Map<String, String> configMap) {
        if (!this.plugins.isServiceAvailable(ISecretLoader.class, protocol)) {
            String message = String.format("No secret loader available for protocol [%s]", protocol);
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message);
        }
        Properties secretProps = this.buildSecretProps(configMap);
        ISecretLoader secretLoader = this.plugins.createConfigService(ISecretLoader.class, protocol, secretProps);
        secretLoader.init(this);
        return secretLoader;
    }

    private Properties buildSecretProps(Map<String, String> configMap) {
        Properties secretProps = new Properties();
        for (Map.Entry<String, String> secretEntry : configMap.entrySet()) {
            if (!secretEntry.getKey().startsWith("secret.") || secretEntry.getValue() == null) continue;
            secretProps.put(secretEntry.getKey(), secretEntry.getValue());
        }
        if (this.secretKey != null && !this.secretKey.isBlank()) {
            secretProps.put("secret.key", this.secretKey);
        }
        return secretProps;
    }
}

