/*
 * Decompiled with CFR 0.152.
 */
package org.finos.tracdap.common.config;

import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.finos.tracdap.common.exception.EConfigLoad;
import org.finos.tracdap.common.exception.ETracInternal;
import org.finos.tracdap.common.exception.EUnexpected;
import org.finos.tracdap.common.startup.StartupLog;
import org.slf4j.event.Level;

public class CryptoHelpers {
    private static final List<String> KEY_FACTORY_ALGORITHMS = List.of("EC", "RSA", "DSA", "DiffieHellman");

    public static void writeTextEntry(KeyStore keystore, String secretKey, String alias, String text) throws EConfigLoad {
        try {
            KeyStore.PasswordProtection protection = new KeyStore.PasswordProtection(secretKey.toCharArray());
            SecretKeyFactory factory = SecretKeyFactory.getInstance("PBE");
            PBEKeySpec spec = new PBEKeySpec(text.toCharArray());
            SecretKey secret = factory.generateSecret(spec);
            KeyStore.SecretKeyEntry entry = new KeyStore.SecretKeyEntry(secret);
            keystore.setEntry(alias, entry, protection);
        }
        catch (IllegalArgumentException | GeneralSecurityException e) {
            String message = String.format("Failed to write secret [%s]: %s", alias, e.getMessage());
            throw new EConfigLoad(message, e);
        }
    }

    public static void writeTextEntry(KeyStore keystore, String secretKey, String alias, String text, Map<String, String> attributes) throws EConfigLoad {
        CryptoHelpers.writeTextEntry(keystore, secretKey, alias, text);
        for (Map.Entry<String, String> attr : attributes.entrySet()) {
            String attrAlias = alias + "$" + attr.getKey();
            CryptoHelpers.writeTextEntry(keystore, secretKey, attrAlias, attr.getValue());
        }
    }

    public static void deleteEntry(KeyStore keystore, String alias) throws EConfigLoad {
        try {
            if (keystore.containsAlias(alias)) {
                keystore.deleteEntry(alias);
            }
        }
        catch (GeneralSecurityException e) {
            String message = String.format("Failed to read secret [%s]: %s", alias, e.getMessage());
            throw new EConfigLoad(message, e);
        }
    }

    public static boolean containsEntry(KeyStore keystore, String alias) throws EConfigLoad {
        try {
            return keystore.containsAlias(alias);
        }
        catch (GeneralSecurityException e) {
            String message = String.format("Failed to read secret [%s]: %s", alias, e.getMessage());
            throw new EConfigLoad(message, e);
        }
    }

    public static String readTextEntry(KeyStore keystore, String secretKey, String alias) throws EConfigLoad {
        try {
            KeyStore.Entry entry = keystore.getEntry(alias, new KeyStore.PasswordProtection(secretKey.toCharArray()));
            if (entry == null) {
                String message = String.format("Secret is not present in the store: [%s]", alias);
                StartupLog.log(CryptoHelpers.class, Level.ERROR, message);
                throw new EConfigLoad(message);
            }
            if (!(entry instanceof KeyStore.SecretKeyEntry)) {
                String message = String.format("Secret is not a secret key: [%s] is %s", alias, entry.getClass().getSimpleName());
                StartupLog.log(CryptoHelpers.class, Level.ERROR, message);
                throw new EConfigLoad(message);
            }
            KeyStore.SecretKeyEntry secret = (KeyStore.SecretKeyEntry)entry;
            String algorithm = secret.getSecretKey().getAlgorithm();
            SecretKeyFactory factory = SecretKeyFactory.getInstance(algorithm);
            Class<PBEKeySpec> keySpecType = PBEKeySpec.class;
            PBEKeySpec keySpec = (PBEKeySpec)factory.getKeySpec(secret.getSecretKey(), keySpecType);
            char[] password = keySpec.getPassword();
            return new String(password);
        }
        catch (IllegalArgumentException | GeneralSecurityException e) {
            String message = String.format("Failed to read secret [%s]: %s", alias, e.getMessage());
            throw new EConfigLoad(message, e);
        }
    }

    public static boolean containsAttribute(KeyStore keystore, String alias, String attrName) throws EConfigLoad {
        try {
            String attrAlias = alias + "$" + attrName;
            return keystore.containsAlias(attrAlias);
        }
        catch (GeneralSecurityException e) {
            String message = String.format("Failed to read secret [%s]: %s", alias, e.getMessage());
            throw new EConfigLoad(message, e);
        }
    }

    public static String readAttribute(KeyStore keystore, String secretKey, String alias, String attrName) throws EConfigLoad {
        String attrAlias = alias + "$" + attrName;
        return CryptoHelpers.readTextEntry(keystore, secretKey, attrAlias);
    }

    public static String encodeSSHA512(String password, byte[] salt) {
        try {
            MessageDigest hasher = MessageDigest.getInstance("SHA-512");
            hasher.update(password.getBytes(StandardCharsets.UTF_8));
            hasher.update(salt);
            int scheme = 6;
            byte[] hash = hasher.digest();
            Base64.Encoder b64 = Base64.getEncoder().withoutPadding();
            String encodedSalt = b64.encodeToString(salt);
            String encodedHash = b64.encodeToString(hash);
            return String.format("$%d$%s$%s", scheme, encodedSalt, encodedHash);
        }
        catch (NoSuchAlgorithmException e) {
            throw new EUnexpected(e);
        }
    }

    public static boolean validateSSHA512(String phc, String password) {
        try {
            String[] phcSections = phc.split("\\$");
            if (phcSections.length != 4 || !"6".equals(phcSections[1])) {
                throw new ETracInternal("Invalid password hash");
            }
            String encodedSalt = phcSections[2];
            String encodedHash = phcSections[3];
            Base64.Decoder b64 = Base64.getDecoder();
            byte[] originalSalt = b64.decode(encodedSalt);
            byte[] originalHash = b64.decode(encodedHash);
            MessageDigest hasher = MessageDigest.getInstance("SHA-512");
            hasher.update(password.getBytes(StandardCharsets.UTF_8));
            hasher.update(originalSalt);
            byte[] hash = hasher.digest();
            return Arrays.equals(hash, originalHash);
        }
        catch (NoSuchAlgorithmException e) {
            throw new EUnexpected(e);
        }
    }

    public static String encodePublicKey(PublicKey key, boolean mime) {
        try {
            KeyFactory factory = KeyFactory.getInstance(key.getAlgorithm());
            X509EncodedKeySpec spec = factory.getKeySpec(key, X509EncodedKeySpec.class);
            byte[] encoded = spec.getEncoded();
            if (mime) {
                String base64 = Base64.getMimeEncoder(80, "\n".getBytes()).encodeToString(encoded);
                return "-----BEGIN PUBLIC KEY-----\n" + base64 + "\n-----END PUBLIC KEY-----\n";
            }
            return Base64.getEncoder().encodeToString(encoded);
        }
        catch (IllegalArgumentException | GeneralSecurityException e) {
            String message = String.format("Failed to encode public key: " + e.getMessage(), new Object[0]);
            throw new EConfigLoad(message, e);
        }
    }

    public static String encodePrivateKey(PrivateKey key, boolean mime) {
        try {
            KeyFactory factory = KeyFactory.getInstance(key.getAlgorithm());
            PKCS8EncodedKeySpec spec = factory.getKeySpec(key, PKCS8EncodedKeySpec.class);
            byte[] encoded = spec.getEncoded();
            if (mime) {
                String base64 = Base64.getMimeEncoder(80, "\n".getBytes()).encodeToString(encoded);
                return "-----BEGIN PRIVATE KEY-----\n" + base64 + "\n-----END PRIVATE KEY-----\n";
            }
            return Base64.getEncoder().encodeToString(encoded);
        }
        catch (IllegalArgumentException | GeneralSecurityException e) {
            String message = String.format("Failed to encode private key: " + e.getMessage(), new Object[0]);
            throw new EConfigLoad(message, e);
        }
    }

    public static PublicKey decodePublicKey(String key, boolean mime) {
        try {
            byte[] encoded;
            if (mime) {
                String base64 = key.replace("-----BEGIN PUBLIC KEY-----", "").replaceAll("\\r", "").replaceAll("\\n", "").replace("-----END PUBLIC KEY-----", "");
                encoded = Base64.getMimeDecoder().decode(base64);
            } else {
                encoded = Base64.getDecoder().decode(key);
            }
            for (String algorithm : KEY_FACTORY_ALGORITHMS) {
                try {
                    X509EncodedKeySpec spec = new X509EncodedKeySpec(encoded);
                    KeyFactory factory = KeyFactory.getInstance(algorithm);
                    return factory.generatePublic(spec);
                }
                catch (Exception e) {
                    if (e instanceof InvalidKeySpecException) continue;
                    throw e;
                }
            }
            String algos = String.join((CharSequence)", ", KEY_FACTORY_ALGORITHMS);
            String message = String.format("Failed to decode public key: No suitable algorithm (available algorithms are %s)", algos);
            throw new EConfigLoad(message);
        }
        catch (IllegalArgumentException | GeneralSecurityException e) {
            String message = String.format("Failed to decode public key: " + e.getMessage(), new Object[0]);
            throw new EConfigLoad(message, e);
        }
    }

    public static PrivateKey decodePrivateKey(String key, boolean mime) {
        try {
            byte[] encoded;
            if (mime) {
                String base64 = key.replace("-----BEGIN PRIVATE KEY-----", "").replaceAll("\\r", "").replaceAll("\\n", "").replace("-----END PRIVATE KEY-----", "");
                encoded = Base64.getMimeDecoder().decode(base64);
            } else {
                encoded = Base64.getDecoder().decode(key);
            }
            for (String algorithm : KEY_FACTORY_ALGORITHMS) {
                try {
                    PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(encoded);
                    KeyFactory factory = KeyFactory.getInstance(algorithm);
                    return factory.generatePrivate(spec);
                }
                catch (Exception e) {
                    if (e instanceof InvalidKeySpecException) continue;
                    throw e;
                }
            }
            String algos = String.join((CharSequence)", ", KEY_FACTORY_ALGORITHMS);
            String message = String.format("Failed to decode public key: No suitable algorithm (available algorithms are %s)", algos);
            throw new EConfigLoad(message);
        }
        catch (IllegalArgumentException | GeneralSecurityException e) {
            String message = String.format("Failed to decode private key: " + e.getMessage(), new Object[0]);
            throw new EConfigLoad(message, e);
        }
    }
}

