/*
 * Decompiled with CFR 0.152.
 */
package org.finos.tracdap.common.config.local;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateException;
import java.util.Properties;
import org.finos.tracdap.common.config.ConfigManager;
import org.finos.tracdap.common.config.CryptoHelpers;
import org.finos.tracdap.common.config.ISecretLoader;
import org.finos.tracdap.common.exception.EConfigLoad;
import org.finos.tracdap.common.exception.EStartup;
import org.finos.tracdap.common.startup.StartupLog;
import org.slf4j.event.Level;

public class JksSecretLoader
implements ISecretLoader {
    public static final String DEFAULT_KEYSTORE_TYPE = "PKCS12";
    private final Properties properties;
    private KeyStore keystore;
    private boolean ready;
    String secretKey;

    public JksSecretLoader(Properties properties) {
        this.properties = properties;
        this.keystore = null;
        this.ready = false;
    }

    @Override
    public void init(ConfigManager configManager) {
        if (this.ready) {
            StartupLog.log(this, Level.ERROR, "JKS secret loader initialized twice");
            throw new EStartup("JKS secret loader initialized twice");
        }
        String keystoreType = this.properties.getProperty("secret.type", DEFAULT_KEYSTORE_TYPE);
        String keystoreUrl = this.properties.getProperty("secret.url");
        String keystoreKey = this.properties.getProperty("secret.key");
        try {
            StartupLog.log(this, Level.INFO, "Initializing JKS secret loader...");
            if (keystoreUrl == null || keystoreUrl.isBlank()) {
                String message = String.format("JKS secrets need %s in the main config file", "secret.url");
                StartupLog.log(this, Level.ERROR, message);
                throw new EStartup(message);
            }
            if (keystoreKey == null || keystoreKey.isBlank()) {
                String template = "JKS secrets need a secret key, use --secret-key or set %s in the environment";
                String message = String.format(template, "TRAC_SECRET_KEY");
                StartupLog.log(this, Level.ERROR, message);
                throw new EStartup(message);
            }
            this.keystore = KeyStore.getInstance(keystoreType);
            byte[] keystoreBytes = configManager.loadBinaryConfig(keystoreUrl);
            try (ByteArrayInputStream stream = new ByteArrayInputStream(keystoreBytes);){
                this.keystore.load(stream, keystoreKey.toCharArray());
                this.ready = true;
                this.secretKey = keystoreKey;
            }
        }
        catch (KeyStoreException e) {
            String message = String.format("Keystore type is not supported: [%s]", keystoreType);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        catch (IOException e) {
            Throwable error = e.getCause() != null ? e.getCause() : e;
            String errorDetail = error.getMessage() + " (this normally means the secret key is wrong)";
            String message = String.format("Failed to open keystore [%s]: %s", keystoreUrl, errorDetail);
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
        catch (NoSuchAlgorithmException | CertificateException e) {
            String message = String.format("Failed to open keystore [%s]: %s", keystoreUrl, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EStartup(message);
        }
    }

    @Override
    public boolean hasSecret(String secretName) {
        try {
            return this.keystore.containsAlias(secretName);
        }
        catch (GeneralSecurityException e) {
            String message = String.format("Secret could not be found in the key store: [%s] %s", secretName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message);
        }
    }

    @Override
    public String loadPassword(String secretName) {
        try {
            return CryptoHelpers.readTextEntry(this.keystore, this.secretKey, secretName);
        }
        catch (EConfigLoad e) {
            String message = String.format("Password could not be retrieved from the key store: [%s] %s", secretName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message, e);
        }
    }

    @Override
    public boolean hasAttr(String secretName, String attrName) {
        try {
            return CryptoHelpers.containsAttribute(this.keystore, secretName, attrName);
        }
        catch (EConfigLoad e) {
            String message = String.format("Password could not be retrieved from the key store: [%s, %s] %s", secretName, attrName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message, e);
        }
    }

    @Override
    public String loadAttr(String secretName, String attrName) {
        try {
            return CryptoHelpers.readAttribute(this.keystore, this.secretKey, secretName, attrName);
        }
        catch (EConfigLoad e) {
            String message = String.format("Attribute could not be retrieved from the key store: [%s, %s] %s", secretName, attrName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message, e);
        }
    }

    @Override
    public PublicKey loadPublicKey(String secretName) {
        try {
            String base64 = CryptoHelpers.readTextEntry(this.keystore, this.secretKey, secretName);
            return CryptoHelpers.decodePublicKey(base64, false);
        }
        catch (EConfigLoad e) {
            String message = String.format("Public key could not be retrieved from the key store: [%s] %s", secretName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message, e);
        }
    }

    @Override
    public PrivateKey loadPrivateKey(String secretName) {
        try {
            String base64 = CryptoHelpers.readTextEntry(this.keystore, this.secretKey, secretName);
            return CryptoHelpers.decodePrivateKey(base64, false);
        }
        catch (EConfigLoad e) {
            String message = String.format("Private key could not be retrieved from the key store: [%s] %s", secretName, e.getMessage());
            StartupLog.log(this, Level.ERROR, message);
            throw new EConfigLoad(message, e);
        }
    }
}

