/*
 *
 *  Copyright 2022 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.flowstep.http;

import org.bson.Document;
import org.flowstep.core.DataProviderException;
import org.flowstep.core.ExtractConnectionException;
import org.flowstep.core.FlowDataProvider;
import org.flowstep.core.connection.EnvironmentConnection;
import org.flowstep.core.context.FlowPackageContext;
import org.flowstep.core.context.FlowStepContext;
import org.flowstep.core.model.step.FlowGroup;
import org.flowstep.http.model.HttpStep;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.List;
import java.util.Map;
import java.util.Objects;

public class HttpStepBuilder implements FlowDataProvider {

    private EnvironmentConnection connection;
    private FlowPackageContext stepPackageContext;
    private HttpStep step;

    private FlowGroup stepGroup;

    public HttpStepBuilder setConnection(EnvironmentConnection connection) {
        this.connection = connection;
        return this;
    }

    public HttpStepBuilder setStepPackageContext(FlowPackageContext stepPackageContext) {
        this.stepPackageContext = stepPackageContext;
        return this;
    }

    public HttpStepBuilder setStep(HttpStep step) {
        this.step = step;
        return this;
    }

    public HttpStepBuilder setStepGroup(FlowGroup stepGroup) {
        this.stepGroup = stepGroup;
        return this;
    }

    @Override
    public void build() throws DataProviderException {
        RestTemplate restTemplate = connection.getConnectionTemplate();
        FlowStepContext stepContext = new FlowStepContext(stepPackageContext, stepGroup, step);
        ResponseEntity<List<Map<String, Object>>> result;

        try {
            result = restTemplate.exchange(connection.getEnvironmentItemSettings().getUri(),
                    HttpMethod.GET,
                    null,
                    new ParameterizedTypeReference<List<Map<String, Object>>>() {
                    });
        } catch (Exception e) {
            throw new ExtractConnectionException("Error while loading remote data", e);
        }

        Objects.requireNonNull(result.getBody())
                .forEach(dataRecord -> saveRecord(step, new Document(dataRecord), stepContext, stepPackageContext));

    }
}
