/*
 *
 *  Copyright 2022 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.flowstep.http;

import org.flowstep.core.FlowStepException;
import org.flowstep.core.connection.EnvironmentConnection;
import org.flowstep.core.context.FlowPackageContext;
import org.flowstep.core.factory.FlowEnvironmentFactory;
import org.flowstep.core.model.step.FlowGroup;
import org.flowstep.core.model.step.FlowStep;
import org.flowstep.core.processor.FlowProcessor;
import org.flowstep.core.processor.FlowProcessorType;
import org.flowstep.http.model.HttpStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class HttpStepProcessor implements FlowProcessor {
    private static final Logger logger = LoggerFactory.getLogger(HttpStepProcessor.class);

    private FlowEnvironmentFactory environmentFactory;

    @Autowired
    public HttpStepProcessor setEnvironmentFactory(FlowEnvironmentFactory environmentFactory) {
        this.environmentFactory = environmentFactory;
        return this;
    }

    @Override
    public FlowProcessorType getType() {
        return FlowProcessorType.PROCESSOR;
    }

    @Override
    public boolean canProcess(FlowStep step) {
        return step instanceof HttpStep;
    }

    @Override
    public boolean process(FlowGroup stepGroup, FlowStep step, FlowPackageContext stepPackageContext) {
        logger.info("    ->Process HttpStep");
        HttpStep httpStep = (HttpStep) step;
        EnvironmentConnection connection = environmentFactory.getEnvironmentConnection(httpStep.getEnvironment(),
                stepPackageContext.getStepPackage());

        if (connection != null) {
            try {
                new HttpStepBuilder().setStep(httpStep)
                        .setConnection(connection)
                        .setStepPackageContext(stepPackageContext)
                        .setStepGroup(stepGroup)
                        .build();
            } catch (Exception e) {
                throw new FlowStepException("HttpStep exception", e);
            }
            return true;
        } else
            logger.error("Error: couldn't find connection for environment: {}", httpStep.getEnvironment());

        return false;
    }
}
