/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.fryske_akademy.jsf.lazy;

/*-
 * #%L
 * primfacesgui
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.vectorprint.StringConverter;
import org.fryske_akademy.Util;
import org.fryske_akademy.jpa.EntityInterface;
import org.fryske_akademy.jpa.Param;
import org.fryske_akademy.jpa.Param.Builder;
import org.fryske_akademy.jsf.AbstractLazyController;
import org.fryske_akademy.jsf.Filtering;
import org.fryske_akademy.jsf.util.JsfUtil;
import org.fryske_akademy.services.CrudReadService;
import org.fryske_akademy.services.CrudReadService.SORTORDER;
import org.primefaces.model.LazyDataModel;
import org.primefaces.model.SortMeta;
import org.primefaces.model.SortOrder;

import javax.inject.Inject;
import javax.security.enterprise.SecurityContext;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Class to support lazy loading with filtering and sorting.
 * Create a subclass, annotate it as a CDI bean and then use {@link JsfUtil#findInContext(java.lang.Class) }
 * in the constructor of the accompanying {@link AbstractLazyController} (prevents cyclic reference).
 *
 * @author eduard
 */
public abstract class AbstractLazyModel<E extends EntityInterface> extends LazyDataModel<E> implements Filtering<E> {

    private final Class<E> clazz;
    private boolean useOr;
    private boolean syntaxInvalue = true;
    private boolean caseInsensitive = false;

    private AbstractLazyController abstractLazyController;

    public AbstractLazyController getLazyController() {
        return abstractLazyController;
    }

    /**
     * Called in @Postconstruct of {@link AbstractLazyController}
     *
     * @param lazyController
     */
    public final void setLazyController(AbstractLazyController lazyController) {
        if (abstractLazyController == null) {
            abstractLazyController = lazyController;
        } else {
            throw new IllegalStateException(String.format("replacing %s with %s not allowed", abstractLazyController,lazyController));
        }
    }

    public AbstractLazyModel(Class<E> clazz) {
        this.clazz = clazz;
    }

    /**
     * NOTE when your bean uses a passivation capable scope (i.e. SessionScoped) the field should be transient and
     * you should have a private readObject in which you initialize CrudReadService, see {@link Util#getBean(java.lang.Class, java.lang.annotation.Annotation...) }.
     * @return 
     */
    public abstract CrudReadService getCrudReadService();
    

    /**
     * map holding filter values, creates an entry with a null value if a key is
     * not yet in the map
     */
    private final Map<String, Object> filters = new HashMap<String, Object>(3) {
        @Override
        public Object get(Object key) {
            if (!containsKey(key)) {
                put(String.valueOf(key), null);
            }
            return super.get(key);
        }

    };

    /**
     * Calls {@link CrudReadService#find(java.io.Serializable, java.lang.Class)
     * } with Integer.valueOf(rowKey)
     *
     * @param rowKey
     * @return
     */
    @Override
    public E getRowData(String rowKey) {
        return "".equals(rowKey) ? null : getCrudReadService().find(Integer.valueOf(rowKey), clazz);
    }

    /**
     * creates a {@link Builder }
     * using {@link AbstractLazyController#isSyntaxInvalue() },
     * {@link Builder#DEFAULT_MAPPING} and {@link #isCaseInsensitive()}.
     *
     * @return
     */
    protected Param.Builder initParamBuilder() {
        return new Param.Builder(isSyntaxInvalue(), Param.Builder.DEFAULT_MAPPING, isCaseInsensitive());
    }

    /**
     * Call {@link #initParamBuilder() } and call {@link #addToParamBuilder(Builder, String, Object)} for each entry. Filters holding empty or null value are skipped.
     *
     * @param filters
     * @return
     */
    protected List<Param> convertFilters(Map<String, Object> filters) {
        if (filters == null) {
            return null;
        }
        Param.Builder builder = initParamBuilder();
        for (Map.Entry<String, Object> p : filters.entrySet()) {
            if ("".equals(p.getValue())||p.getValue()==null) {
                // skip
            } else {
                addToParamBuilder(builder, p.getKey(), p.getValue());
            }
        }
        return builder.build();
    }

    /**
     * Implement this method to call one of the add methods in {@link Builder}, if you want to benefit from syntax intelligence
     * for non String values, choose one of the add methods using a converter and use String.valueOf(value) as paramValue.
     * @see StringConverter#forClass(Class)
     * @see #isUseOr()
     * @param builder
     * @param key
     * @param value
     */
    protected abstract void addToParamBuilder(Param.Builder builder, String key, Object value);

    /**
     * Calls {@link #load(int, int, java.util.List, java.util.Map) }
     *
     * @param first
     * @param pageSize
     * @param sortField
     * @param sortOrder
     * @param filters
     * @return
     */
    @Override
    public final List<E> load(int first, int pageSize, String sortField, SortOrder sortOrder, Map<String, Object> filters) {
        List<SortMeta> s = new ArrayList<>(1);
        s.add(new SortMeta(null, sortField, sortOrder, null));
        return load(first, pageSize, s, filters);
    }

    /**
     * Calls {@link #convertSortMeta(java.util.List) }, {@link #convertFilters(java.util.Map) },
     * {@link CrudReadService#findDynamic(java.lang.Integer, java.lang.Integer, java.util.Map, java.util.List, java.lang.Class) },
     * {@link #setWrappedData(java.lang.Object) } and {@link #setRowCount(int) } with
     * {@link CrudReadService#countDynamic(java.util.List, java.lang.Class) }.
     *
     * @param first
     * @param pageSize
     * @param multiSortMeta
     * @param filters
     * @return
     */
    @Override
    public final List<E> load(int first, int pageSize, List<SortMeta> multiSortMeta, Map<String, Object> filters) {
        SORTORDER.Builder sortBuilder = convertSortMeta(multiSortMeta);
        List<Param> convertFilters = convertFilters(filters);
        List<E> load = getCrudReadService().findDynamic(first, pageSize, sortBuilder.build(), convertFilters, clazz);
        setWrappedData(load);
        setRowCount(getCrudReadService().countDynamic(convertFilters, clazz));
        return load;
    }

    /**
     * convert primefaces SortMeta list into builder for use in CrudReadService.
     * @see #convert(org.primefaces.model.SortOrder) 
     * @param multiSortMeta
     * @return 
     */
    protected SORTORDER.Builder convertSortMeta(List<SortMeta> multiSortMeta) {
        SORTORDER.Builder sortBuilder = new SORTORDER.Builder();
        if (multiSortMeta != null) {
            multiSortMeta.forEach((h) -> {
                sortBuilder.add(h.getSortField(), convert(h.getSortOrder()));
            });
        }
        return sortBuilder;
    }

    public static SORTORDER convert(SortOrder order) {
        if (order == null) {
            return null;
        }
        switch (order) {
            case ASCENDING:
                return SORTORDER.ASC;
            case UNSORTED:
                return SORTORDER.NONE;
            default:
                return SORTORDER.DESC;
        }
    }

    /**
     * Use this in primefaces EL expression for filteredValue
     *
     * @return
     */
    @Override
    public final List<E> getFiltered() {
        return getWrappedData();
    }

    @Override
    public final void setFiltered(List<E> filtered) {
    }

    /**
     * used this in EL expression for filterValue:
     * #{controller.filters['filtername']}
     *
     * @return
     */
    @Override
    public Map<String, Object> getFilters() {
        return filters;
    }

    @Override
    public Filtering<E> add(String key, Object value) {
        filters.put(key, value);
        return this;
    }

    @Override
    public Filtering<E> clear() {
        filters.clear();
        return this;
    }


    @Override
    public boolean hasFilter(String key) {
        return filters.containsKey(key)&&filters.get(key)!=null&&!String.valueOf(filters.get(key)).isEmpty();
    }

    public void setSyntaxInvalue(boolean syntaxInvalue) {
        this.syntaxInvalue = syntaxInvalue;
    }

    /**
     * when true use or when several parameters are given
     *
     * @see Param#getAndOr()
     * @return
     */
    public boolean isUseOr() {
        return useOr;
    }

    public void setUseOr(boolean useOr) {
        this.useOr = useOr;
    }

    /**
     * when true support syntax in parameter value
     *
     * @see Builder
     * @return
     */
    public boolean isSyntaxInvalue() {
        return syntaxInvalue;
    }

    public boolean isCaseInsensitive() {
        return caseInsensitive;
    }

    public void setCaseInsensitive(boolean caseInsensitive) {
        this.caseInsensitive = caseInsensitive;
    }

    /**
     * Wraps a String value in * so a contains search is performed, calls {@link Builder#add(String, Object, boolean)}  with
     * {@link #isUseOr() }.
     * @param builder
     * @param key
     * @param value 
     */
    protected void wrapStringInWildCards(Param.Builder builder, String key, Object value) {
        Object o = value;
        if (value instanceof String) {
            String s = (String)value;
            if (!Param.Builder.valueIsOperator(s,builder.isSyntaxInValue())) {
                o = "*" + s + "*";
                if (s.indexOf(Param.Builder.NEGATION) == 0) {
                    o = "!" + o;
                }
            }
        }
        builder.add(key, o, isUseOr());

    }

    /**
     * This method makes your lazy datatables work also if you don't define a rowkey attribute and function
     * @param object
     * @return
     */
    @Override
    public Object getRowKey(E object) {
        return object.getId()==null?-1:object.getId();
    }

    @Inject
    private transient SecurityContext securityContext;

    private void readObject(java.io.ObjectInputStream stream) throws ClassNotFoundException, IOException {
        stream.defaultReadObject();
        securityContext = Util.getBean(SecurityContext.class);
    }

    protected SecurityContext getSecurityContext() {
        return securityContext;
    }
}