package org.fryske_akademy.jsf.util;

/*-
 * #%L
 * primfacesgui
 * %%
 * Copyright (C) 2018 Fryske Akademy
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import org.fryske_akademy.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.faces.application.FacesMessage;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolationException;
import java.io.IOException;
import java.util.*;

public class JsfUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(JsfUtil.class.getName());

    /**
     * Calls {@link FacesContext#validationFailed() }, {@link Util#deepestCause(java.lang.Throwable)
     * } and {@link #addErrorMessage(java.lang.String) } with the prefix and the
     * localized message from the deepest exception. When the deepestCause is a ConstraintViolationException
     * {@link Util#formatConstraintException(ConstraintViolationException)} is called. Rethrows the exception
     *
     * @param ex
     * @param prefix
     */
    public static void handleException(Exception ex, String prefix) throws Exception {
        FacesContext.getCurrentInstance().validationFailed();
        Throwable t = Util.deepestCause(ex);
        String msg = t instanceof ConstraintViolationException ?
                Util.formatConstraintException((ConstraintViolationException)t).toString() : t.getLocalizedMessage();
        addErrorMessage(prefix + msg);
        throw ex;
    }

    public static boolean isValidationFailed() {
        return FacesContext.getCurrentInstance().isValidationFailed();
    }

    public static void addErrorMessage(Exception ex, String defaultMsg) {
        String msg = ex.getLocalizedMessage();
        if (msg != null && msg.length() > 0) {
            addErrorMessage(msg);
        } else {
            addErrorMessage(defaultMsg);
        }
    }

    public static void addErrorMessages(List<String> messages) {
        for (String message : messages) {
            addErrorMessage(message);
        }
    }

    /**
     * uses null as clientId
     *
     * @param msg
     */
    public static void addErrorMessage(String msg) {
        String cid = null;
        addErrorMessage(cid, msg);
    }

    public static void addErrorMessage(String clientId, String msg) {
        FacesMessage facesMsg = new FacesMessage(FacesMessage.SEVERITY_ERROR, msg, msg);
        FacesContext.getCurrentInstance().addMessage(clientId, facesMsg);
    }

    /**
     * Uses successInfo as client id
     *
     * @param msg
     */
    public static void addSuccessMessage(String msg) {
        addSuccessMessage("successInfo", msg);
    }

    public static void addSuccessMessage(String clientId, String msg) {
        FacesMessage facesMsg = new FacesMessage(FacesMessage.SEVERITY_INFO, msg, msg);
        FacesContext.getCurrentInstance().addMessage(clientId, facesMsg);
    }

    /**
     * Before using this expensive method try injection and look into lazy Injection! Calls {@link #findInContext(java.lang.String, java.lang.Class) } with
     * clazz.simpleName, first letter lowerCase.
     *
     * @param <T>
     * @param clazz
     * @return
     */
    public static <T> T findInContext(Class<T> clazz) {
        return findInContext(Character.toLowerCase(clazz.getSimpleName().charAt(0)) + clazz.getSimpleName().substring(1), clazz);
    }

    /**
     * Before using this expensive method try injection and look into lazy Injection! Lookup
     * a bean using the application ELResolver.
     * @param name
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> T findInContext(String name, Class<T> clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        T obj = (T) facesContext.getApplication().getELResolver().
                getValue(facesContext.getELContext(), null, name);
        return obj;
    }

    public static <T> T findOrCreateInContext(String name, Class<T> clazz) {
        FacesContext facesContext = FacesContext.getCurrentInstance();
        return facesContext.getApplication().evaluateExpressionGet(facesContext, name, clazz);
    }

    private static final Map<Class,Converter> CLASS_CONVERTER_MAP = new HashMap<>();

    /**
     * uses static cache
     * @param clazz
     * @return
     */
    public static Converter getConverter(Class clazz) {
        if (CLASS_CONVERTER_MAP.containsKey(clazz)) {
            return CLASS_CONVERTER_MAP.get(clazz);
        }
        FacesContext facesContext = FacesContext.getCurrentInstance();
        Converter converter = facesContext.getApplication().createConverter(clazz);
        if (converter==null) {
            LOGGER.warn(String.format("no converter found for %s",clazz));
        } else {
            CLASS_CONVERTER_MAP.put(clazz,converter);
        }
        return CLASS_CONVERTER_MAP.get(clazz);
    }

    /**
     * context path plus servlet path
     *
     * @return
     */
    public static String fullServletPath() {
        ExternalContext externalContext = FacesContext.getCurrentInstance().getExternalContext();
        return externalContext.getRequestContextPath() + externalContext.getRequestServletPath();
    }

    public static String contextPath() {
        ExternalContext externalContext = FacesContext.getCurrentInstance().getExternalContext();
        return externalContext.getRequestContextPath();
    }

    public static void redirectToServletPath(String servletPath) throws IOException {
        ExternalContext externalContext = FacesContext.getCurrentInstance().getExternalContext();
        externalContext.redirect(externalContext.getRequestContextPath() + servletPath);
    }

    public static void logout() throws ServletException {
        FacesContext.getCurrentInstance().getExternalContext().invalidateSession();
        ((HttpServletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest()).logout();
    }

    private static final ResourceBundle rootBundle = ResourceBundle.getBundle("EMPTY", Locale.ROOT);

    /**
     * return a resource for the current locale (from faces context or default
     * locale), or an empty bundle. Using this method prevents
     * MissingResourceException and logs the bundle (property file) that isn't
     * found.
     *
     * @param bundle
     * @return
     */
    public static ResourceBundle getLocaleBundle(String bundle) {
        FacesContext currentInstance = FacesContext.getCurrentInstance();
        Locale loc = currentInstance != null ? currentInstance.getViewRoot().getLocale() : Locale.getDefault();
        try {
            return ResourceBundle.getBundle(bundle, loc);
        } catch (MissingResourceException e) {
            LOGGER.warn("resource (properties file) not found: " + bundle + " for locale: " + loc + ", returning EMPTY");
            return rootBundle;
        }

    }

    /**
     * Return the message from the bundle or the key when it's not found. Keys not found are logged.
     * @param bundle
     * @param key
     * @return
     */
    public static String getFromBundle(ResourceBundle bundle, String key) {
        if (bundle.containsKey(key)) {
            return bundle.getString(key);
        } else {
            LOGGER.warn(String.format("%s not found in language bundle %s", key, bundle));
            return key;
        }
    }
}
