/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.scripting.jruby.admin;

import org.glassfish.api.ActionReport;
import org.glassfish.api.I18n;
import org.glassfish.api.Param;
import org.glassfish.api.admin.AdminCommand;
import org.glassfish.api.admin.AdminCommandContext;
import org.glassfish.api.admin.config.ConfigParser;
import org.glassfish.scripting.jruby.config.JrubyContainer;
import org.glassfish.scripting.jruby.config.JrubyRuntimePool;
import org.jvnet.hk2.annotations.Inject;
import org.jvnet.hk2.annotations.Scoped;
import org.jvnet.hk2.annotations.Service;
import org.jvnet.hk2.component.Habitat;
import org.jvnet.hk2.component.PerLookup;
import org.jvnet.hk2.config.ConfigSupport;
import org.jvnet.hk2.config.SingleConfigCode;
import org.jvnet.hk2.config.TransactionFailure;

import java.beans.PropertyVetoException;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.logging.Level;

/**
 * @author Vivek Pandey
 */
@Service(name = "jruby-container-config")
@I18n("jruby-container-config.command")

@Scoped(PerLookup.class)
public class JRubyContainerCommand implements AdminCommand {

    @Inject(optional = true)
    JrubyContainer container;

    @Inject
    ConfigParser parser;

    @Inject
    Habitat habitat;

    @Param(name = "jruby-home", optional = true)
    String jrubyHome;

    @Param(name = "jruby-runtime", optional = true, shortName = "runtime")
    String jrubyRuntime;

    @Param(name = "jruby-runtime-min", optional = true, shortName = "runtime-min")
    String jrubyRuntimeMin;

    @Param(name = "jruby-runtime-max", optional = true, shortName = "runtime-max")
    String jrubyRuntimeMax;

    @Param(name = "show", optional = true, acceptableValues = "true, false")
    String show;

    @Override
    public void execute(AdminCommandContext context) {
        ActionReport report = context.getActionReport();
        if (container == null) {
            URL xml = getClass().getClassLoader().getResource("/org/glassfish/scripting/jruby/config/jruby-container-config.xml");
            try {

                container = parser.parseContainerConfig(habitat, xml, JrubyContainer.class);
            }
            catch (IOException e) {
                report.failure(context.logger, "Failed to create <jruby-container> config element", e);
            }
        }

        if (show != null && Boolean.valueOf(show)) {
            report.setActionDescription(Messages.CURRENT_CONFIG);

            report.getTopMessagePart()
                    .addChild().setMessage("jruby-home=" + container.getJrubyHome());


            report.getTopMessagePart()
                    .addChild().setMessage("jruby-runtime=" + container.getJrubyRuntimePool().getJrubyRuntime());

            report.getTopMessagePart()
                    .addChild().setMessage("jruby-runtime-min=" + container.getJrubyRuntimePool().getJrubyRuntimeMin());

            report.getTopMessagePart()
                    .addChild().setMessage("jruby-runtime-max=" + container.getJrubyRuntimePool().getJrubyRuntimeMax());

            report.setActionExitCode(ActionReport.ExitCode.SUCCESS);

            return;
        }

        updateJRubyHome(context);

        updateJRubyRuntimePool(context);

    }

    private void updateJRubyHome(AdminCommandContext context) {
        ActionReport report = context.getActionReport();
        if (jrubyHome == null) {
            return;
        }

        if (!new File(jrubyHome).exists()) {
            report.failure(context.logger, Messages.format(Messages.JRUBY_HOME_NOT_EXIST, jrubyHome));
            return;
        }
        try {
            ConfigSupport.apply(new SingleConfigCode<JrubyContainer>() {
                @Override
                public Object run(JrubyContainer param) throws PropertyVetoException, TransactionFailure {
                    param.setJrubyHome(jrubyHome);
                    return null;
                }
            }, container);
            report.getTopMessagePart().setMessage(Messages.format(Messages.JRUBY_HOME_SUCCESS, jrubyHome));
        } catch (TransactionFailure e) {
            report.failure(context.getLogger(), Messages.format(Messages.JRUBY_HOME_FAILURE), e);
        }

    }

    private void updateJRubyRuntimePool(AdminCommandContext context) {
        if (jrubyRuntime == null && jrubyRuntimeMin == null && jrubyRuntimeMax == null) {
            return;
        }
        ActionReport report = context.getActionReport();
        final int[] settings = new int[3];
        int value = toInt(JrubyRuntimePool.JRUBY_RUNTIME, jrubyRuntime, context);
        settings[JRUBY_RUNTIME] = value;
        value = toInt(JrubyRuntimePool.JRUBY_RUNTIME_MIN, jrubyRuntimeMin, context);
        settings[JRUBY_RUNTIME_MIN] = value;
        value = toInt(JrubyRuntimePool.JRUBY_RUNTIME_MAX, jrubyRuntimeMax, context);
        settings[JRUBY_RUNTIME_MAX] = value;

        JrubyRuntimePool pool = container.getJrubyRuntimePool();

        try {
            ConfigSupport.apply(new SingleConfigCode<JrubyRuntimePool>() {
                @Override
                public Object run(JrubyRuntimePool param) throws PropertyVetoException, TransactionFailure {
                    if (settings[JRUBY_RUNTIME] > 0)
                        param.setJrubyRuntime(settings[JRUBY_RUNTIME]);
                    if (settings[JRUBY_RUNTIME_MIN] > 0)
                        param.setJrubyRuntimeMin(settings[JRUBY_RUNTIME_MIN]);
                    if (settings[JRUBY_RUNTIME_MAX] > 0)
                        param.setJrubyRuntimeMax(settings[JRUBY_RUNTIME_MAX]);
                    return null;
                }
            }, pool);
            report.getTopMessagePart().setMessage(Messages.format(Messages.JRUBY_RUNTIME_POOL_SUCCESS, pool.getJrubyRuntime(), pool.getJrubyRuntimeMin(), pool.getJrubyRuntimeMax()));
        } catch (TransactionFailure e) {
            context.logger.log(Level.SEVERE, Messages.JRUBY_RUNTIME_POOL_FAILURE, e);
        }
    }

    private int toInt(String property, String value, AdminCommandContext context) {
        if (value == null) {
            return INVALID_INT;
        }
        try {
            return Integer.valueOf(jrubyRuntime, 10);
        } catch (NumberFormatException e) {
            context.getActionReport().failure(context.getLogger(), Messages.format(Messages.INVALID_NUMBER, property, value), e);
            return INVALID_INT;
        }
    }

    //represents invalid runtime setting value
    private static final int INVALID_INT = -1;

    private static final int JRUBY_RUNTIME = 0;
    private static final int JRUBY_RUNTIME_MIN = 1;
    private static final int JRUBY_RUNTIME_MAX = 2;

}
