/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.scripting.jruby.admin;

import com.sun.enterprise.config.serverbeans.MonitoringService;
import org.glassfish.api.ActionReport;
import org.glassfish.api.I18n;
import org.glassfish.api.Param;
import org.glassfish.api.admin.AdminCommand;
import org.glassfish.api.admin.AdminCommandContext;
import org.glassfish.api.admin.config.ConfigParser;
import org.glassfish.api.monitoring.ContainerMonitoring;
import org.glassfish.scripting.jruby.config.JrubyContainer;
import org.glassfish.scripting.jruby.config.JrubyRuntimePool;
import org.jvnet.hk2.annotations.Inject;
import org.jvnet.hk2.annotations.Scoped;
import org.jvnet.hk2.annotations.Service;
import org.jvnet.hk2.component.Habitat;
import org.jvnet.hk2.component.PerLookup;
import org.jvnet.hk2.config.ConfigSupport;
import org.jvnet.hk2.config.SingleConfigCode;
import org.jvnet.hk2.config.TransactionFailure;

import java.beans.PropertyVetoException;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.logging.Level;

/**
 * @author Vivek Pandey
 */
@Service(name = "configure-jruby-container")
@I18n("configure-jruby-container.command")

@Scoped(PerLookup.class)
public class JRubyContainerCommand implements AdminCommand {

    @Inject(optional = true)
    JrubyContainer container;

    @Inject
    ConfigParser parser;

    @Inject
    Habitat habitat;

    @Inject
    MonitoringService monitoringService;

    @Param(name = "jruby-home", optional = true)
    String jrubyHome;

    @Param(name = "jruby-runtime", optional = true)
    String jrubyRuntime;

    @Param(name = "jruby-runtime-min", optional = true)
    String jrubyRuntimeMin;

    @Param(name = "jruby-runtime-max", optional = true)
    String jrubyRuntimeMax;

    @Param(name = "monitoring", optional = true, defaultValue = "false")
    Boolean monitoring;


    @Param(name = "show", optional = true, defaultValue = "true")
    Boolean show;


    @Override
    public void execute(AdminCommandContext context) {
        ActionReport report = context.getActionReport();
        if (container == null) {
            URL xml = getClass().getClassLoader().getResource("/org/glassfish/scripting/jruby/config/jruby-container-config.xml");
            try {
                container = parser.parseContainerConfig(habitat, xml, JrubyContainer.class);
                String home = System.getProperty("com.sun.aas.installRoot") + File.separator + "jruby";
                if (!(new File(home).exists())) {
                    home = "";
                }
                try {
                    final String home1 = home;
                    ConfigSupport.apply(new SingleConfigCode<JrubyContainer>() {
                        @Override
                        public Object run(JrubyContainer param) throws PropertyVetoException, TransactionFailure {
                            param.setJrubyHome(home1);
                            return null;
                        }
                    }, container);
                } catch (TransactionFailure e) {
                    report.failure(context.getLogger(), Messages.format(Messages.JRUBY_CONFIG_FAILED), e);
                }
            } catch (IOException e) {
                report.failure(context.logger, Messages.format(Messages.JRUBY_CONTAINER_CONFIG_FAILED), e);
            }
        }

        updateJRubyHome(context);

        updateJRubyRuntimePool(context);
        updateMonitoring(context);


        if (show != null && Boolean.valueOf(show)) {
            report.getTopMessagePart().addChild().setMessage(Messages.format(Messages.CURRENT_CONFIG));
            report.getTopMessagePart()
                    .addChild().setMessage(JrubyContainer.JRUBY_HOME + "=" + container.getJrubyHome());


            report.getTopMessagePart()
                    .addChild().setMessage(JrubyRuntimePool.INITIAL_POOL_SIZE + "=" + container.getJrubyRuntimePool().getInitialPoolSize());

            report.getTopMessagePart()
                    .addChild().setMessage(JrubyRuntimePool.MIN_POOL_SIZE + "=" + container.getJrubyRuntimePool().getMinPoolSize());

            report.getTopMessagePart()
                    .addChild().setMessage(JrubyRuntimePool.MAX_POOL_SIZE + "=" + container.getJrubyRuntimePool().getMaxPoolSize());

            report.getTopMessagePart()
                    .addChild().setMessage("monitoring" + "=" + monitoring);

            report.setActionExitCode(ActionReport.ExitCode.SUCCESS);

        }
    }

    private void updateJRubyHome(AdminCommandContext context) {
        ActionReport report = context.getActionReport();
        if (jrubyHome == null) {
            return;
        }

        if (!new File(jrubyHome).exists()) {
            report.failure(context.logger, Messages.format(Messages.JRUBY_HOME_NOT_EXIST, jrubyHome));
            return;
        }
        try {
            ConfigSupport.apply(new SingleConfigCode<JrubyContainer>() {
                @Override
                public Object run(JrubyContainer param) throws PropertyVetoException, TransactionFailure {
                    param.setJrubyHome(jrubyHome);
                    return null;
                }
            }, container);
            report.getTopMessagePart().setMessage(Messages.format(Messages.JRUBY_HOME_SUCCESS, jrubyHome));
        } catch (TransactionFailure e) {
            report.failure(context.getLogger(), Messages.format(Messages.JRUBY_HOME_FAILURE), e);
        }

    }

    private void updateMonitoring(AdminCommandContext context) {
        ActionReport report = context.getActionReport();

        //Add <container-monitoring name="jruby-container level="OFF"/> to <monitoring-service> if not already there
        ContainerMonitoring cm = monitoringService.getContainerMonitoring("jruby-container");
        if (cm == null) {
            try {
                ConfigSupport.apply(new SingleConfigCode<MonitoringService>() {
                    public Object run(MonitoringService param) throws PropertyVetoException, TransactionFailure {
                        ContainerMonitoring newItem = param.createChild(ContainerMonitoring.class);
                        newItem.setName("jruby-container");
                        newItem.setLevel("OFF");
                        param.getContainerMonitoring().add(newItem);
                        return newItem;
                    }
                }, monitoringService);

            } catch (TransactionFailure tf) {
                report.failure(context.logger, Messages.format(Messages.JRUBY_CONFIG_MONITORING_FAILED), tf);
            }
        } else if (monitoring != null) {
            try {
                ConfigSupport.apply(new SingleConfigCode<ContainerMonitoring>() {
                    public Object run(ContainerMonitoring param) throws PropertyVetoException, TransactionFailure {
                        String level;
                        if (monitoring) {
                            level = "HIGH";
                        } else {
                            level = "OFF";
                        }
                        param.setLevel(level);
                        return null;
                    }
                }, cm);

            } catch (TransactionFailure tf) {
                report.failure(context.logger, Messages.format(Messages.JRUBY_CONFIG_MONITORING_CHANGE_FAILED, monitoring), tf);
            }
        }
    }


    private int toInt(String value, String propName, AdminCommandContext context) {
        try {
            return Integer.valueOf(value);
        } catch (NumberFormatException ex) {
            context.getActionReport().failure(context.logger, Messages.format(Messages.ERR_INVALID_RUNTIME_PROPERTY, value, propName), ex);
            throw ex;
        }
    }


    private void updateJRubyRuntimePool(AdminCommandContext context) {
        if (jrubyRuntime == null && jrubyRuntimeMin == null && jrubyRuntimeMax == null) {
            return;
        }

        ActionReport report = context.getActionReport();

        JrubyRuntimePool pool = container.getJrubyRuntimePool();

        int existingNumRt = toInt(pool.getInitialPoolSize(), JrubyRuntimePool.INITIAL_POOL_SIZE, context);
        int existingMinNumRt = toInt(pool.getMinPoolSize(), JrubyRuntimePool.MIN_POOL_SIZE, context);
        int existingMaxNumRt = toInt(pool.getMaxPoolSize(), JrubyRuntimePool.MAX_POOL_SIZE, context);

        int numRt = (jrubyRuntime != null) ? toInt(jrubyRuntime, JrubyRuntimePool.INITIAL_POOL_SIZE, context) : existingNumRt;
        int numMinRt = (jrubyRuntimeMin != null) ? toInt(jrubyRuntimeMin, JrubyRuntimePool.MIN_POOL_SIZE, context) : existingMinNumRt;
        int numMaxRt = (jrubyRuntimeMax != null) ? toInt(jrubyRuntimeMax, JrubyRuntimePool.MAX_POOL_SIZE, context) : existingMaxNumRt;

        if (numMinRt > numRt || numMinRt > numMaxRt || numMaxRt < numRt) {
            report.failure(context.logger, Messages.format(Messages.JRUBY_RUNTIME_POOL_INCONSISTENT, numRt, numMinRt, numMaxRt));
            return;
        }


        try {
            ConfigSupport.apply(new SingleConfigCode<JrubyRuntimePool>() {
                @Override
                public Object run(JrubyRuntimePool param) throws PropertyVetoException, TransactionFailure {
                    param.setInitialPoolSize(jrubyRuntime);
                    param.setMinPoolSize(jrubyRuntimeMin);
                    param.setMaxPoolSize(jrubyRuntimeMax);
                    return null;
                }
            }, pool);
            report.getTopMessagePart().setMessage(Messages.format(Messages.JRUBY_RUNTIME_POOL_SUCCESS, pool.getInitialPoolSize(), pool.getMinPoolSize(), pool.getMaxPoolSize()));
        } catch (TransactionFailure e) {
            context.logger.log(Level.SEVERE, Messages.JRUBY_RUNTIME_POOL_FAILURE, e);
        }
    }
}
