/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.scripting.jruby.sniffer;

import com.sun.enterprise.config.serverbeans.MonitoringService;
import com.sun.enterprise.module.Module;
import com.sun.enterprise.module.ModuleDefinition;
import com.sun.enterprise.module.ModulesRegistry;
import com.sun.enterprise.module.ResolveError;
import org.glassfish.api.container.Sniffer;
import org.glassfish.api.deployment.archive.ReadableArchive;
import org.glassfish.api.monitoring.ContainerMonitoring;
import org.jvnet.hk2.annotations.Inject;
import org.jvnet.hk2.annotations.Scoped;
import org.jvnet.hk2.annotations.Service;
import org.jvnet.hk2.component.Habitat;
import org.jvnet.hk2.component.Singleton;
import org.jvnet.hk2.config.ConfigSupport;
import org.jvnet.hk2.config.SingleConfigCode;
import org.jvnet.hk2.config.TransactionFailure;

import java.beans.PropertyVetoException;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * JRuby sniffer
 *
 * @author Vivek Pandey
 */
@Service(name = "jruby")
@Scoped(Singleton.class)
public class JRubySniffer implements Sniffer {

    @Inject
    Habitat habitat;

    @Inject
    ModulesRegistry registry;

    @Inject
    MonitoringService monitoringService;

    private static final String containerName = "jruby";
    private static final String RAILS_APP = "app/controllers/application.rb";
    private static final String RUBY_SUFFIX = ".rb";
    private static final String RACKUP_SUFFIX = ".ru";
    private static final String RAKEFILE = "Rakefile";
    private final String[] containers = {"org.glassfish.scripting.jruby.JRubyContainer"};
    private static final String WEB_XML = "WEB-INF/web.xml";
    private static final String WEB_INF = "WEB-INF/";

    public boolean handles(ReadableArchive location, ClassLoader loader) {
        try {
            //if it is a WEB archive (a war file, we simply skip it
            if (location.exists(WEB_XML) || location.exists(WEB_INF))
                return false;

            if (location.exists(RAILS_APP)) {
                return true;
            }

            Enumeration<String> entries = location.entries();
            while (entries.hasMoreElements()) {
                String entryName = entries.nextElement();
                if (entryName.endsWith(RUBY_SUFFIX) || entryName.equalsIgnoreCase(RAKEFILE)||entryName.endsWith(RACKUP_SUFFIX)) {
                    return true;
                }
            }
        } catch (IOException e) {
            // ignore
        }
        return false;
    }

    public String[] getURLPatterns() {
        //Rails applications dont have any specific patterns
        return null;
    }

    public Class<? extends Annotation>[] getAnnotationTypes() {
        return new Class[0];
    }

    public String getModuleType() {
        return containerName;
    }

    private static final String GRIZZLY_JRUBY_MODULE = "org.glassfish.scripting.grizzly-jruby-module";

    /**
     * Sets up the container libraries so that any imported bundle from the
     * connector jar file will now be known to the module subsystem
     * <p/>
     * This method returns a {@link ModuleDefinition} for the module containing
     * the core implementation of the container. That means that this module
     * will be locked as long as there is at least one module loaded in the
     * associated container.
     *
     * @param containerHome is where the container implementation resides
     * @param logger        the logger to use
     * @return the module definition of the core container implementation.
     * @throws java.io.IOException exception if something goes sour
     */
    public Module[] setup(String containerHome, Logger logger) throws IOException {

        //Add <container-monitoring name="jruby-container level="OFF"/> to <monitoring-service> if not already there
        ContainerMonitoring cm = monitoringService.getContainerMonitoring("jruby-container");
        if (cm == null) {
            try {
                ConfigSupport.apply(new SingleConfigCode<MonitoringService>() {
                    public Object run(MonitoringService param) throws PropertyVetoException, TransactionFailure {
                        ContainerMonitoring newItem = param.createChild(ContainerMonitoring.class);
                        newItem.setName("jruby-container");
                        newItem.setLevel("OFF");
                        param.getContainerMonitoring().add(newItem);
                        return newItem;
                    }
                }, monitoringService);

            } catch (TransactionFailure tf) {
                logger.log(Level.SEVERE, Messages.format(Messages.SNIFFER_MONITORING_FAILED), tf);
            }
        }

        Collection<Module> modules = registry.getModules(GRIZZLY_JRUBY_MODULE);
        if (modules.size() == 0) {
            throw new ResolveError(Messages.format(Messages.JRUBY_MODULE_NOTFOUND, "grizzly-jruby-module.jar"));
        } else if (modules.size() > 1) {
            throw new ResolveError(Messages.format(Messages.JRUBY_MODULE_MORETHANONE, GRIZZLY_JRUBY_MODULE));
        }
        Module grizzlyRails = modules.iterator().next();
        return new Module[]{grizzlyRails};
    }

    public void tearDown() {

    }

    public String[] getContainersNames() {
        return containers;
    }


    /**
     * @return whether this sniffer should be visible to user
     */
    public boolean isUserVisible() {
        return true;
    }

    public Map<String, String> getDeploymentConfigurations(ReadableArchive readableArchive) throws IOException {
        return Collections.EMPTY_MAP;
    }

    @Override
    public String[] getIncompatibleSnifferTypes() {
        return new String[0];
    }

}
