/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.scripting.jython;

import org.glassfish.api.admin.ServerEnvironment;
import org.glassfish.api.container.EndpointRegistrationException;
import org.glassfish.api.container.RequestDispatcher;
import org.glassfish.api.deployment.ApplicationContainer;
import org.glassfish.api.deployment.ApplicationContext;
import org.glassfish.api.deployment.DeployCommandParameters;
import org.glassfish.api.deployment.DeploymentContext;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Vivek Pandey
 */
public class JythonApplication implements ApplicationContainer {
    private final Logger logger;
    private Object JythonRuntime;
    private com.sun.grizzly.tcp.Adapter adapter;
    private final RequestDispatcher dispatcher;
    private final ServerEnvironment serverEnvironment;
    private final String appRoot;
    private String jythonHome;
    private final DeploymentContext deploymentContext;

    private static final String JYTHON_HOME = "jython.home";
    private static final String V3_JYTHON_HOME=System.getProperty("com.sun.aas.installRoot") + File.separator+"modules"+File.separator+"jython";

    private final String contextRoot;

    public JythonApplication(DeploymentContext context, ServerEnvironment env, RequestDispatcher dispatcher) throws IOException {
        this.logger = context.getLogger();
        this.appRoot = context.getSourceDir().getCanonicalPath();
        this.dispatcher = dispatcher;
        this.serverEnvironment = env;
        this.deploymentContext = context;
        this.contextRoot = getContextRoot(context, env.getStartupContext().getArguments());
        String home = System.getProperty(JYTHON_HOME);


        //v3 container sets jruby.home system property to install-root/modules/jruby, which is unexpected and
        // problematic. Lets reset it
        if(home != null && home.equals(V3_JYTHON_HOME)){
            home=null;
        }


        //override the jython.home system property by deploy time property
        if (context.getAppProps().getProperty(JYTHON_HOME) != null) {
            home = context.getAppProps().getProperty(JYTHON_HOME);
        }

        String newJythonIPSLocation = System.getProperty("com.sun.aas.installRoot") + File.separator + "jython";
        if (home != null) {
            //GlassFish might be giving the jruby.home pointing to glassfish/modules/jruby which might not be right
            //check if it exists otherwise try alternate
            if (!new File(home).exists()) {
                logger.log(Level.SEVERE, Messages.format(Messages.ERR_JYTHONHOME_INVALID, home));
                throw new IllegalArgumentException(Messages.format(Messages.ERR_JYTHONHOME_INVALID, home));
            }
        } else if (new File(newJythonIPSLocation).exists()) {
            home = newJythonIPSLocation;
        } else {
            logger.log(Level.SEVERE, Messages.format(Messages.ERR_JYTHONHOME_INVALID, newJythonIPSLocation));
            throw new IllegalArgumentException(Messages.format(Messages.ERR_JYTHONHOME_INVALID, newJythonIPSLocation));
        }
        
        if(home != null){
            jythonHome =home;
        }
        if (jythonHome == null) {
            logger.severe(Messages.format(Messages.ERR_JYTHONHOME_NULL));
            throw new IllegalArgumentException(Messages.format(Messages.ERR_JYTHONHOME_NULL));
        }
        System.setProperty(JYTHON_HOME, jythonHome);
        context.getLogger().info(Messages.format(Messages.INFO_LOADING_APP,
                context.getCommandParameters(DeployCommandParameters.class).name,
                contextRoot));

    }

    public Object getDescriptor() {
        return null;
    }

    public boolean start(ApplicationContext startupContext) throws Exception {
        if(!startJythonGrizzlyAdapter())
            return false;

        try {
            String ctx=contextRoot;
            if(contextRoot.equals("/"))
                ctx = "";
            dispatcher.registerEndpoint(ctx, adapter, this);
        } catch (EndpointRegistrationException e) {
            logger.log(Level.SEVERE,
                    Messages.format(Messages.ERR_REGISTRY_JYTHONAPP,
                            startupContext.getAppProps().getProperty(deploymentContext.getCommandParameters(DeployCommandParameters.class).name)), e);
            return false;
        }
        return true;
    }

    public boolean stop(ApplicationContext stopContext) {
        try {
            JythonRuntime.getClass().getMethod("stopJythonRuntimePool").invoke(JythonRuntime);
        } catch (Exception e) {
            logger.log(Level.WARNING, Messages.format(Messages.ERR_STOPPING_JYTHONAPP, contextRoot, e.getMessage()), e);
            return false;
        }
        return true;

    }

    public boolean suspend() {
        return false;
    }

    public boolean resume() throws Exception {
        return false;
    }

    public ClassLoader getClassLoader() {
        return null;
    }

    /*package */ String getContextRoot(){
        return contextRoot;
    }

    private boolean startJythonGrizzlyAdapter(){
        try {
            /**
             * JRuby is not a OSGi bundle so it needs to be passed custom class loader that makes all the jruby jars availble and also
             * can load any of the java classes available inside the context classloader.
             *
             * So we have a jruby runtime specific wrapper class that weload using custom classloader and this classloader will eventually
             * be used to load everyting else by jruby.
             */
            URL[] urls = findJythonJars(jythonHome, serverEnvironment.getStartupContext().getRootDirectory());
            ClassLoader cl = new URLClassLoader(urls, JythonApplication.class.getClassLoader());

            Class<?> c = cl.loadClass("org.glassfish.scripting.jython.grizzly.JythonRuntime");
            this.JythonRuntime =  c.getConstructor(Properties.class, String.class, String.class).newInstance(deploymentContext.getAppProps(), appRoot, contextRoot);
            this.adapter = (com.sun.grizzly.tcp.Adapter) c.getMethod("getAdapter").invoke(JythonRuntime);
        } catch (Exception e) {
            logger.log(Level.SEVERE, Messages.format(Messages.ERR_CREATING_JYTHONADAPTOR, e.getMessage()), e);
            return false;
        }
        return true;
    }

    private URL[] findJythonJars(String jythHome, File grizzlyjython) throws MalformedURLException, ClassNotFoundException {
        List<URL> urls = new ArrayList<URL>();

        File jythonLibDir = new File(jythHome);
        if(!jythonLibDir.exists()){
            logger.severe(Messages.format(Messages.ERR_JYTHONHOME_INVALID, jythHome));
            throw new IllegalArgumentException(Messages.format(Messages.ERR_JYTHONHOME_INVALID, jythHome));
        }

        //jthon 2.5 rc3 does not contain javalib, the code below should be removed
        File jythonJavaLibDir = new File(jythHome + "/javalib");

        for (File lib : jythonLibDir.listFiles()) {
            if (lib.isFile() && lib.toString().endsWith(".jar")) {
                urls.add(lib.toURI().toURL());
            }
        }
        // Jython 2.5RC2 does not include a "javalib" directory, which causes a NPE here unless created or checked.
        if (jythonJavaLibDir.exists()) {
            for (File lib : jythonJavaLibDir.listFiles()) {
                if (lib.isFile() && lib.toString().endsWith(".jar")) {
                    urls.add(lib.toURI().toURL());
                }
            }
        }

        File[] grizzlyjythonJars = grizzlyjython.listFiles(new FilenameFilter(){
            public boolean accept(File dir, String name) {
                return name.startsWith("jython-grizzly");
            }
        });

        if(grizzlyjythonJars.length > 0)
            urls.add(grizzlyjythonJars[0].toURI().toURL());

        return urls.toArray(new URL[urls.size()]);
    }



    private static final String CONTEXT_ROOT_ARG ="--contextroot";
    private String getContextRoot(DeploymentContext context, Properties props) {
        String contextRoot = props.getProperty(CONTEXT_ROOT_ARG);
        DeployCommandParameters params = context.getCommandParameters(DeployCommandParameters.class);
        //For gem we get contextroot as part of startup context and in normal server deploy it comes from
        //DeploymentContext properties. We should override contextroot from startupcontext by the one
        // from DeploymentContext
        if(params.contextroot != null)
            contextRoot = params.contextroot;
        if (contextRoot == null || contextRoot.length() == 0) {
            contextRoot = "/" + params.name;
        }       
        return contextRoot;
    }
    
}
