/**
 * This file is part of Graylog.
 *
 * Graylog is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Graylog is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Graylog.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.graylog2.alarmcallbacks;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import org.bson.types.ObjectId;
import org.graylog2.database.MongoConnection;
import org.graylog2.database.PersistedServiceImpl;
import org.graylog2.plugin.Tools;
import org.graylog2.plugin.database.ValidationException;
import org.graylog2.plugin.streams.Stream;
import org.graylog2.rest.models.alarmcallbacks.requests.CreateAlarmCallbackRequest;

import javax.inject.Inject;
import java.util.List;
import java.util.Map;

public class AlarmCallbackConfigurationServiceImpl extends PersistedServiceImpl implements AlarmCallbackConfigurationService {
    @Inject
    public AlarmCallbackConfigurationServiceImpl(MongoConnection mongoConnection) {
        super(mongoConnection);
    }

    @Override
    public List<AlarmCallbackConfiguration> getForStreamId(String streamId) {
        final List<AlarmCallbackConfiguration> alarmCallbackConfigurations = Lists.newArrayList();
        final List<DBObject> respConfigurations = query(AlarmCallbackConfigurationImpl.class,
                new BasicDBObject("stream_id", streamId)
        );

        for (DBObject configuration : respConfigurations) {
            alarmCallbackConfigurations.add(new AlarmCallbackConfigurationImpl((ObjectId) configuration.get("_id"), configuration.toMap()));
        }

        return alarmCallbackConfigurations;
    }

    @Override
    public List<AlarmCallbackConfiguration> getForStream(Stream stream) {
        return getForStreamId(stream.getId());
    }

    @Override
    public AlarmCallbackConfiguration load(String alarmCallbackId) {
        DBObject rawModel = get(AlarmCallbackConfigurationImpl.class, alarmCallbackId);
        return (rawModel == null ? null : new AlarmCallbackConfigurationImpl((ObjectId) (rawModel.get("_id")), rawModel.toMap()));
    }

    @Override
    public AlarmCallbackConfiguration create(String streamId, CreateAlarmCallbackRequest request, String userId) {
        Map<String, Object> fields = Maps.newHashMap();
        fields.put("stream_id", new ObjectId(streamId));
        fields.put("type", request.type);
        fields.put("configuration", request.configuration);
        fields.put("created_at", Tools.iso8601());
        fields.put("creator_user_id", userId);

        return new AlarmCallbackConfigurationImpl(fields);
    }

    @Override
    public long count() {
        return count(AlarmCallbackConfigurationImpl.class, new BasicDBObject());
    }

    @Override
    public AlarmCallbackConfiguration update(String streamId, String alarmCallbackId, Map<String, Object> deltas) {
        return null;
    }

    @Override
    public String save(AlarmCallbackConfiguration model) throws ValidationException {

        return super.save(getImplOrFail(model));
    }

    @Override
    public int destroy(AlarmCallbackConfiguration model) {
        return super.destroy(getImplOrFail(model));
    }

    private AlarmCallbackConfigurationImpl getImplOrFail(AlarmCallbackConfiguration alarmCallback) {
        if (alarmCallback instanceof AlarmCallbackConfigurationImpl) {
            return (AlarmCallbackConfigurationImpl) alarmCallback;
        } else {
            throw new IllegalArgumentException("Argument must be of implementation class AlarmCallbackConfigurationImpl, not " + alarmCallback.getClass());
        }
    }
}
