package org.gridkit.quickrun;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.gridkit.nanoparser.NanoGrammar;
import org.gridkit.nanoparser.NanoGrammar.SyntaticScope;
import org.gridkit.nanoparser.NanoParser;
import org.gridkit.nanoparser.ParserException;
import org.gridkit.nanoparser.ReflectionActionSource;
import org.gridkit.nanoparser.Token;

public class PropsConf {


    private static final NanoParser<Void> KEY_PARSER = new NanoParser<>(KeyParsingHandler.SYNTAX, new KeyParsingHandler());

    private static final NanoParser<PropsConf> VALUE_PARSER = new NanoParser<>(InterpolationHandler.SYNTAX, new InterpolationHandler());

    private final Map<String, String> props = new LinkedHashMap<String, String>();

    private String wildMatchPattern = "";
    private Matcher wildMatcher = null;
    private List<String> wildValues = new ArrayList<>();

    public PropsConf() {
    }

    @SuppressWarnings({ "serial" })
    public void loadFromFile(String path) throws IOException {
        Properties pfile = new Properties() {
            @Override
            public synchronized Object put(Object key, Object value) {
                putProp((String) key, (String) value);
                return null;
            }
        };
        try (FileInputStream fileImageInputStream = new FileInputStream(new File(path))) {
            pfile.load(fileImageInputStream);
        }
    }

    public void load(Iterable<Map.Entry<String, String>> props) {
        for (Map.Entry<String, String> prop: props) {
            String key = prop.getKey();
            String val = prop.getValue();
            putProp(key, val);
        }
    }

    private void putProp(String key, String val) {
        this.props.put(key, val);
        try {
            Object pkey = KEY_PARSER.parse(null, Object.class, key);
            if (pkey instanceof WildKey) {
                if (wildMatchPattern.length() > 0) {
                    wildMatchPattern += "|";
                }
                wildMatchPattern += "(" + pkey + ")";
                wildMatcher = null;
                wildValues.add(val);
            }
        } catch (ParserException pe) {
            System.out.println(pe.formatVerboseErrorMessage());
            throw pe;
        }
    }

    public ConfBean conf() {
        return new CBean("");
    }

    private String[] split(String name) {
        return name.split("\\.");
    }

    private String postProcess(String key, String value) {
        if (value == null) {
            if (wildMatcher == null) {
                wildMatcher = Pattern.compile(wildMatchPattern).matcher("");
            }
            wildMatcher.reset(key);
            if (wildMatcher.matches()) {
                for (int n = 0; n < wildMatcher.groupCount(); ++n) {
                    if (wildMatcher.group(n + 1) != null) {
                        value = wildValues.get(n);
                        break;
                    }
                }
            }
        }
        if (value == null) {
            return null;
        }

        try {
            String ppValue = VALUE_PARSER.parse(this, String.class, value);
            return ppValue;
        } catch (ParserException pe) {
            System.out.println(pe.formatVerboseErrorMessage());
            throw pe;
        }
    }

    private static class KeyParsingHandler extends ReflectionActionSource<Void> {

        static final SyntaticScope SYNTAX = NanoGrammar.newParseTable()
                .token("**")
                .token("*")
                .glueOp("CC").rank(1)
                .infixOp(".", ".").rank(2)
                .term("~.+")
                .toScope();

        @Term("*")
        public WildKey asterisk() {
            return new WildKey("[^.]+");
        }

        @Term("**")
        public WildKey doubleAsterisk() {
            return new WildKey(".+");
        }

        @Binary("CC")
        public WildKey cc(WildKey wk, String lit) {
            return new WildKey(wk + Pattern.quote(lit));
        }

        @Binary("CC")
        public WildKey cc(String lit, WildKey wk) {
            return new WildKey(Pattern.quote(lit) + wk);
        }

        @Binary(".")
        public String dot(String a, String b) {
            return a + "." + b;
        }

        @Binary(".")
        public WildKey dot(WildKey a, WildKey b) {
            return new WildKey(a + "\\." + b);
        }

        @Binary(".")
        public WildKey dot(String a, WildKey b) {
            return new WildKey(Pattern.quote(a) + "\\." + b);
        }

        @Binary(".")
        public WildKey dot(WildKey a, String b) {
            return new WildKey(a + "\\." + Pattern.quote(b));
        }
    }

    private static class InterpolationHandler extends ReflectionActionSource<PropsConf> {

        static final SyntaticScope SYNTAX = NanoGrammar.newParseTable()
                .token("$$")
                .infixOrPrefixOp("$").rank(2)
                .term("EXPR", "~\\{.+?\\}")
                .term("ID", "~[_A-Za-z0-9.]+")
                .term("~[^${_.\\w]+?") // match longer string as long as no ambiguous characters
                .term("~.") // match any not matched character one by one
                .glueOp("glue").rank(1)
                .toScope();

        @Term("$$")
        public String doubleDollar() {
            return "$$";
        }

        @Unary("$")
        public String interpolate(@Context PropsConf conf, @Source Token tkn, Id id) {
            String key = id.toString();
            return readKey(tkn, conf, key);
        }

        @Binary("$")
        public String interpolate(@Context PropsConf conf, @Source Token tkn, String pref, Id id) {
            String key = id.toString();
            return pref + readKey(tkn, conf, key);
        }

        @Unary("$")
        public String interpolate(@Context PropsConf conf, @Source Token tkn, Expr expr) {
            String key = expr.toString();
            key = key.substring(1, key.length() - 1);
            return readKey(tkn, conf, key);
        }

        @Binary("$")
        public String interpolate(@Context PropsConf conf, @Source Token tkn, String pref, Expr expr) {
            String key = expr.toString();
            key = key.substring(1, key.length() - 1);
            return pref + readKey(tkn, conf, key);
        }

        private String readKey(Token token, PropsConf conf, String key) {
            String val = conf.conf().readProp(key);
            if (val == null) {
                throw new ParserException(token, "Unknown property '" + key + "'");
            }
            return val;
        }

        @Term("EXPR")
        public Expr expr(String text) {
            return new Expr(text);
        }

        @Term("EXPR")
        public String exprAsText(String text) {
            return text;
        }

        @Term("ID")
        public Id id(String text) {
            return new Id(text);
        }

        @Term("ID")
        public String idAsString(String text) {
            return text;
        }

        @Binary("glue")
        public String glue(String a, String b) {
            return a + b;
        }
    }

    private static class Id {

        final String id;

        public Id(String id) {
            this.id = id;
        }

        @Override
        public String toString() {
            return id;
        }
    }

    private static class Expr {

        final String expr;

        public Expr(String expr) {
            this.expr = expr;
        }

        @Override
        public String toString() {
            return expr;
        }
    }

    private static class WildKey {

        private final String regEx;

        public WildKey(String regEx) {
            this.regEx = regEx;
        }

        @Override
        public String toString() {
            return regEx;
        }
    }

    private class CBean implements ConfBean {

        private final String prefix;

        public CBean(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public String readProp(String key) {
            String val = readProp(key, null);
            if (val == null) {
                throw new RuntimeException("No value for '" + (prefix + key) + "'");
            } else {
                return val;
            }
        }

        @Override
        public String readProp(String key, String defaultValue) {
            String fpath = prefix + key;
            String val =  postProcess(fpath, props.get(fpath));
            return val != null ? val : defaultValue;
        }

        @Override
        public <T> T parseProp(String key, Function<String, T> parser) {
            String val = readProp(key);
            return parser.apply(val);
        }

        @Override
        public <T> T parseProp(String key, Function<String, T> parser, T defaultValue) {
            String val = readProp(key, (String)null);
            return val == null ? defaultValue : parser.apply(val);
        }

        @Override
        public Set<String> keySet() {
            Map<String, String> result = new LinkedHashMap<>();
            for (String kk: props.keySet()) {
                if (kk.startsWith(prefix) && !kk.equals(prefix)) {
                    String key = kk.substring(prefix.length());
                    result.put(key, null);
                }
            }
            return result.keySet();
        }

        @Override
        public ConfBean subConf(String key) {
            String fpath = prefix + key + ".";
            return new CBean(fpath);
        }

        private ConfBean subConf(String name, String key) {
            String fpath = prefix + key + ".";
            return new CBean(fpath);
        }

        @Override
        public Map<String, ConfBean> listSubConf(String key) {
            String fpath = prefix + key + ".";
            List<String> bnames = new ArrayList<String>();
            for (String kk: props.keySet()) {
                if (kk.startsWith(fpath) && !kk.equals(fpath)) {
                    String bname = split(kk.substring(fpath.length()))[0];
                    if (!bnames.contains(bname)) {
                        bnames.add(bname);
                    }
                }
            }
            Map<String, ConfBean> result = new LinkedHashMap<>();
            for (String bname: bnames) {
                ConfBean cb = subConf(bname, fpath + bname);
                result.put(bname, cb);
            }
            return result;
        }
    }
}
