package org.gridkit.quickrun;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import org.gridkit.quickrun.exec.SimpleTaskExecutor;
import org.gridkit.quickrun.exec.TaskExecutor;

public class QuickRunDriver {

    private Map<String, Module> modules = new HashMap<String, Module>();
    private List<Stage> executionPlan = new ArrayList<>();
    private Stage activeStage = null;
    private List<Module> activeModules = new ArrayList<>();
    private List<Module> activatedModules= new ArrayList<>();
    private Set<Module> blockingSet = new HashSet<>();
    private List<Supplier<String>> endOfRunRepoters = Collections.synchronizedList(new ArrayList<>());
    private String finalReport = "...";

    private long startTime;

    public QuickRunDriver(ConfBean conf) {
        loadConfiguration(conf);
    };

    private void loadConfiguration(ConfBean conf) {
        for (Map.Entry<String, ConfBean> e: conf.listSubConf("module").entrySet()) {
            String id = e.getKey();
            modules.put(id, loadModule(e.getValue()));
        }

        for (Map.Entry<String, ConfBean> e: conf.listSubConf("stage").entrySet()) {
            String id = e.getKey();
            Stage stage = loadStage(id, e.getValue());
            executionPlan.add(stage);
        }

        Collections.sort(executionPlan, Comparator.comparing(s -> s.order));
    }

    private Module loadModule(ConfBean m) {
        try {
            String implName = m.readProp("provider");
            BeanProvider bp = (BeanProvider) Class.forName(implName).newInstance();
            Module mm = (Module) bp.createInstace(m);

            return mm;
        } catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    private Stage loadStage(String id, ConfBean s) {
        Stage stage = new Stage();
        stage.stageId = id;
        stage.stageName = s.readProp("name", id);
        stage.order = s.parseProp("order", Integer::valueOf);

        ConfBean i = s.subConf("include-module");
        for (String m: i.keySet()) {
            if ("true".equalsIgnoreCase(i.readProp(m))) {
                Module mm = modules.get(m);
                if (mm == null) {
                    throw new RuntimeException("Unknown module '" + m + "'");
                }
                stage.modules.put(m, mm);
            }
        }

        return stage;
    }

    public synchronized void start() {
        startTime = System.nanoTime();
        Thread thread = new Thread(this::tick, "QuickRunDriver");
        activateNextStage();
        thread.start();
    }

    private void activateNextStage() {
        if (activeStage != null) {
            for (Module m: activeModules) {
                m.onStageCompleted(activeStage.stageId);
            }
        }

        // clean completed modules from active list
        Iterator<Module> it = activeModules.iterator();
        while(it.hasNext()) {
            Module m = it.next();
            if (m.isCompleted()) {
                it.remove();
            }
        }

        if (executionPlan.isEmpty()) {
            activeStage = null;
        } else {
            activeStage = executionPlan.remove(0);
            for (Map.Entry<String, Module> m: activeStage.modules.entrySet()) {
                activate(m.getKey(), m.getValue());
            }
        }
    }

    private void activate(String id, Module m) {
        Executor ex = Executors.newCachedThreadPool(new ThreadFactory() {

            private final AtomicInteger counter = new AtomicInteger();

            @Override
            public Thread newThread(Runnable r) {
                Thread t = new Thread(r, id + "-" + (counter.getAndIncrement()));
                t.setDaemon(true);
                return t;
            }
        });

        m.activate(new ExecCtx(ex));
        activeModules.add(m);
        activatedModules.add(m);
        if (!m.isBackground()) {
            blockingSet.add(m);
        }
    }

    private synchronized void tick() {
        while(true) {
            Iterator<Module> it = blockingSet.iterator();
            while(it.hasNext()) {
                Module m = it.next();
                if (m.isCompleted()) {
                    it.remove();
                }
            }
            if (blockingSet.isEmpty()) {
                activateNextStage();
            } else {
                for (Module m: activeModules) {
                    if (m.isAbortRequested()) {
                        activateNextStage();
                        break;
                    }
                }
            }
            if (activeStage == null) {
                processEndOfRun();
                break;
            }
            try {
                this.wait(500);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
    }

    private void processEndOfRun() {
        StringBuilder sb = new StringBuilder();
        for (Supplier<String> reporter: endOfRunRepoters) {
            sb.append(reporter.get()).append("\n");
        }
        finalReport = sb.toString();
    }

    public synchronized boolean isCompleted() {
        return executionPlan.isEmpty() && blockingSet.isEmpty();
    }

    public synchronized String statusReport() {
        StringBuilder sb = new StringBuilder();
        if (activeStage != null) {
            Duration time = Duration.ofSeconds(TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - startTime));
            sb.append(String.format("Simulation time [%02d:%02d:%02d]\n", time.toHours(), time.toMinutes() % 60, time.getSeconds() % 60));
            sb.append("Active stage: " + activeStage.stageName + "\n");
            for (Module m: activeModules) {
                sb.append(m.executionSummary());
                if (sb.length() > 0 && sb.charAt(sb.length() - 1) != '\n') {
                    sb.append('\n');
                }
            }
        } else {
            sb.append("Execution complete\n");
            sb.append(finalReport);
        }
        return sb.toString();
    }

    public void stop() {

    }

    private final class Stage {

        private String stageId;
        private String stageName;
        private int order;
        private Map<String, Module> modules = new LinkedHashMap<String, Module>();

    }

    private final class ExecCtx implements QuickRunContext {

        private final Executor executor;
        private final SimpleTaskExecutor manager;

        public ExecCtx(Executor executor) {
            this.executor = executor;
            this.manager = new SimpleTaskExecutor(executor);
        }

        @Override
        public long getCurrentTimeNanos() {
            return System.nanoTime();
        }

        @Override
        public Executor getExecutor() {
            return executor;
        }

        @Override
        public TaskExecutor getTaskExecutor() {
            return manager;
        }

        @Override
        public void addEndOfRunReport(Supplier<String> reporter) {
            endOfRunRepoters.add(reporter);
        }
    }
}
