package org.gridkit.quickrun.exec;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import org.gridkit.quickrun.exec.TaskSet.TaskWrapper;

/**
 * Simple wrapper tracking number of task in  flight.
 *
 * @author Alexey Ragozin (alexey.ragozin@gmail.com)
 */
public class ActiveTaskCounter implements TaskWrapper {

    private final AtomicLong completedTasks = new AtomicLong();
    private final AtomicInteger activeTasks = new AtomicInteger();
    private final Semaphore signal = new Semaphore(0);

    @Override
    public Task wrap(Task task) {

        return new ProxyTask(task) {

            @Override
            public CompletableFuture<Void> start(Executor executor) throws Exception {
                try {
                    activeTasks.incrementAndGet();
                    signal.drainPermits();
                    CompletableFuture<Void> fut = delegate.start(executor).handle(this::doHandle);
                    return fut;
                } catch (RuntimeException e) {
                    doHandle(null, e);
                    throw e;
                }
            }

            @Override
            public void skip() {
                super.skip();
            }

            private Void doHandle(Void v, Throwable e) {
                synchronized (this) {
                    releasePermit();
                }
                return null;
            }
        };
    }

    public int getActiveTaskCount() {
        return activeTasks.get();
    }

    public long getCompletedTaskCount() {
        return completedTasks.get();
    }

    public void joinWithTasks() throws InterruptedException {
        while (activeTasks.get() > 0) {
            signal.acquire();
        }
    }

    private void releasePermit() {
        activeTasks.decrementAndGet();
        completedTasks.incrementAndGet();
        signal.release(1);
    }
}
