package org.gridkit.quickrun.exec;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import org.gridkit.quickrun.exec.TaskSet.TaskWrapper;

/**
 * Task concurrency control utility class.
 * Controls number of concurrently active tasks.
 *
 * @author Alexey Ragozin (alexey.ragozin@gmail.com)
 */
public class ConcurrencyGate implements TaskWrapper {

    private final int concLimit;
    private int activeTasks;
    private Deque<LazyRunPermit> pendingPermits = new ArrayDeque<>();

    public ConcurrencyGate(int concLimit) {
        this.concLimit = concLimit;
    }

    @Override
    public Task wrap(Task task) {

        return new ProxyTask(task) {

            private LazyRunPermit permit = new LazyRunPermit(delegate.condition());

            @Override
            public synchronized SelectableLatch condition() {
                return permit;
            }

            @Override
            public CompletableFuture<Void> start(Executor executor) throws Exception {
                try {
                    permit.await();
                } catch (InterruptedException e) {
                    skip();
                    throw e;
                }
                try {
                    CompletableFuture<Void> fut = delegate.start(executor).handle(this::doHandle);
                    return fut;
                } catch (RuntimeException e) {
                    doHandle(null, e);
                    throw e;
                }
            }

            @Override
            public void skip() {
                super.skip();
                cleanPermit(permit);
            }

            private Void doHandle(Void v, Throwable e) {
                synchronized (this) {
                    releasePermit();
                }
                return null;
            }
        };
    }

    public synchronized int getActiveTaskCount() {
        return activeTasks;
    }

    private void cleanPermit(LazyRunPermit permit) {
        synchronized(this) {
            if (pendingPermits.remove(permit)) {
                return;
            } else if (permit.hasPermit()) {
                releasePermit();
            }
        }
    }

    private void releasePermit() {
        List<LazyRunPermit> toOpen = new ArrayList<>();
        synchronized (this) {
            --activeTasks;
            //System.err.println("Release permit");
            while(activeTasks < concLimit && pendingPermits.size() > 0) {
                LazyRunPermit lpr = pendingPermits.removeFirst();
                ++activeTasks;
                toOpen.add(lpr);
            }
        }
        // there is a race condition here, as permit could be discarded
        // TODO need to handle permit cancellation
        for (LazyRunPermit lrp: toOpen) {
            lrp.open();
        }
    }

    private void claimPermit(LazyRunPermit permit) {
        synchronized(this) {
            if (activeTasks < concLimit) {
                ++activeTasks;
            } else {
                pendingPermits.add(permit);
                //System.err.println("Claim permit, queue " + pendingPermits.size());
                return;
            }
        }
        permit.open();
    }

    private class LazyRunPermit extends SimpleLatch {

        private final SelectableLatch taskCond;
        private boolean active = false;

        public LazyRunPermit(SelectableLatch taskCond) {
            this.taskCond = taskCond;
        }

        @Override
        public synchronized long proposeWaitTimeNS() {
            if (!active && taskCond != null) {
                return taskCond.proposeWaitTimeNS();
            } else {
                return 0;
            }
        }

        private synchronized boolean hasPermit() {
            return open;
        }

        @Override
        public synchronized boolean isOpen() {
            if (open) {
                return true;
            } else {
                if (!active) {
                    if (taskCond != null && !taskCond.isOpen()) {
                        // waiting for nested condition
                        return false;
                    } else {
                        // claim exec permit
                        active = true;
                        claimPermit(this);
                        return open;
                    }
                } else {
                    return false;
                }
            }
        }
    }
}
